#!/usr/bin/env php
<?php
declare(strict_types=1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/functions.php';

ini_set('display_errors', '1');
error_reporting(E_ALL);

$missing = [];
if (!class_exists('mysqli')) {
    fwrite(STDERR, "PHP CLI is running without the mysqli extension enabled. Please enable it before running the tests.\n");
    exit(1);
}

$conn = get_db_connection();
if (!$conn) {
    fwrite(STDERR, "Unable to connect to the database. Please check configuration.\n");
    exit(1);
}

if (!$conn->begin_transaction()) {
    fwrite(STDERR, "Failed to start database transaction.\n");
    exit(1);
}

$tests = [
    'Region stock adjustments' => static function (mysqli $conn): void {
        $regionId = 1;
        $productId = 5;
        $delta = 8;

        $initialMap = mbh_get_region_stock_map($conn, $regionId);
        $initialValue = $initialMap[$productId] ?? 0;

        if (!mbh_increment_region_stock($conn, $regionId, $productId, $delta)) {
            throw new RuntimeException('Failed to increment region stock.');
        }

        $afterIncrement = mbh_get_region_stock_map($conn, $regionId)[$productId] ?? 0;
        if ($afterIncrement !== $initialValue + $delta) {
            throw new RuntimeException('Region stock value did not increase as expected.');
        }

        if (!mbh_decrement_region_stock($conn, $regionId, $productId, $delta)) {
            throw new RuntimeException('Failed to decrement region stock.');
        }

        $afterDecrement = mbh_get_region_stock_map($conn, $regionId)[$productId] ?? 0;
        if ($afterDecrement !== $initialValue) {
            throw new RuntimeException('Region stock value did not roll back to the original amount.');
        }
    },
    'Notification insertion' => static function (mysqli $conn): void {
        $userId = 21;
        $message = 'Transfer flow smoke ' . bin2hex(random_bytes(4));
        $link = '/tests/run_transfer_checks.php';

        if (!mbh_notify_user($conn, $userId, $message, $link)) {
            throw new RuntimeException('Failed to insert notification.');
        }

        $stmt = $conn->prepare('SELECT message, link FROM user_notification WHERE user_id = ? ORDER BY id DESC LIMIT 1');
        if (!$stmt) {
            throw new RuntimeException('Failed to prepare verification query.');
        }
        $stmt->bind_param('i', $userId);
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        if (!$row) {
            throw new RuntimeException('Notification record missing.');
        }

        if ($row['message'] !== $message || $row['link'] !== $link) {
            throw new RuntimeException('Last notification does not match the expected payload.');
        }
    },
];

$failed = false;
foreach ($tests as $name => $test) {
    try {
        $test($conn);
        echo "[PASS] $name\n";
    } catch (Throwable $ex) {
        $failed = true;
        echo "[FAIL] $name - {$ex->getMessage()}\n";
    }
}

$conn->rollback();
exit($failed ? 1 : 0);
