<?php
/**
 * Script to create a fresh production SQL dump
 * - Only includes admin users (DG, DGA, Admin, Board)
 * - All other tables empty
 * - AUTO_INCREMENT reset to 1
 * - No FK constraint errors
 */

$inputFile = __DIR__ . '/../database/masunzu_bar_hotel.sql';
$outputFile = __DIR__ . '/../database/masunzu_bar_hotel_production.sql';

$sql = file_get_contents($inputFile);
$lines = explode("\n", $sql);
$output = [];
$inCreateTable = false;
$inInsert = false;
$currentTable = '';
$keepData = false;
$adminRoleIds = [1, 2, 3, 10]; // DG, DGA, Admin, Board
$userData = [];

foreach ($lines as $line) {
    // Keep header
    if (strpos($line, '-- phpMyAdmin') !== false || 
        strpos($line, 'SET SQL_MODE') !== false ||
        strpos($line, 'START TRANSACTION') !== false ||
        strpos($line, 'SET time_zone') !== false ||
        strpos($line, 'CHARACTER_SET') !== false ||
        strpos($line, 'COLLATION') !== false ||
        strpos($line, 'SET NAMES') !== false ||
        strpos($line, 'Database:') !== false ||
        strpos($line, 'COMMIT') !== false) {
        $output[] = $line;
        continue;
    }
    
    // Handle CREATE TABLE - keep everything until closing parenthesis
    if (strpos($line, 'CREATE TABLE') !== false) {
        $inCreateTable = true;
        $output[] = $line;
        continue;
    }
    
    // Keep all lines inside CREATE TABLE until closing parenthesis
    if ($inCreateTable) {
        $output[] = $line;
        // Check if this is the end of CREATE TABLE (closing parenthesis with ENGINE)
        if (strpos($line, ') ENGINE=') !== false || strpos($line, ');') !== false) {
            $inCreateTable = false;
        }
        continue;
    }
    
    // Keep all ALTER TABLE statements (indexes, constraints, AUTO_INCREMENT)
    if (strpos($line, 'ALTER TABLE') !== false || 
        strpos($line, 'ADD PRIMARY KEY') !== false ||
        strpos($line, 'ADD KEY') !== false ||
        strpos($line, 'ADD CONSTRAINT') !== false ||
        strpos($line, 'MODIFY') !== false ||
        strpos($line, 'AUTO_INCREMENT') !== false) {
        // Reset AUTO_INCREMENT to 1
        if (strpos($line, 'AUTO_INCREMENT') !== false) {
            $line = preg_replace('/AUTO_INCREMENT=\d+/', 'AUTO_INCREMENT=1', $line);
        }
        $output[] = $line;
        continue;
    }
    
    // Handle INSERT statements
    if (strpos($line, 'INSERT INTO') !== false) {
        $inInsert = true;
        $currentTable = '';
        if (preg_match('/INSERT INTO `([^`]+)`/', $line, $matches)) {
            $currentTable = $matches[1];
        }
        
        // Keep role table data
        if ($currentTable === 'role') {
            $output[] = $line;
            $keepData = true;
        }
        // Keep admin users only
        else if ($currentTable === 'user') {
            $output[] = $line;
            $keepData = true;
        }
        // Skip all other INSERT statements
        else {
            $keepData = false;
            continue;
        }
        continue;
    }
    
    // Handle INSERT data lines
    if ($inInsert) {
        if ($currentTable === 'role') {
            $output[] = $line;
        }
        else if ($currentTable === 'user') {
            // Filter only admin users (role_id 1, 2, 3, 10)
            // Pattern: (id, 'username', 'password', role_id, ...
            if (preg_match('/\((\d+),/', $line, $matches)) {
                // Extract role_id from the line
                // Format: (id, 'username', 'password', role_id, ...
                if (preg_match("/\((\d+), '[^']+', '[^']+', (\d+)/", $line, $roleMatches)) {
                    $roleId = (int)$roleMatches[2];
                    if (in_array($roleId, $adminRoleIds)) {
                        $userData[] = $line;
                    }
                }
            } else {
                // Not a data line, keep it
                $output[] = $line;
            }
        }
        // End of INSERT
        if (strpos($line, ';') !== false && $keepData) {
            $inInsert = false;
            $keepData = false;
            // If we collected user data, output it now with sequential IDs
            if ($currentTable === 'user' && !empty($userData)) {
                $userId = 1;
                foreach ($userData as $userLine) {
                    $userLine = preg_replace('/^\((\d+),/', "($userId,", $userLine);
                    $output[] = $userLine;
                    $userId++;
                }
                $userData = [];
            }
        }
        continue;
    }
    
    // Keep empty lines and comments
    if (trim($line) === '' || strpos($line, '--') === 0) {
        $output[] = $line;
        continue;
    }
}

$finalOutput = implode("\n", $output);

// Add default DG user if not present
$hasDG = false;
$userInsertPos = strpos($finalOutput, "INSERT INTO `user`");
if ($userInsertPos !== false) {
    $userSection = substr($finalOutput, $userInsertPos, 500);
    if (strpos($userSection, ", 1,") !== false || strpos($userSection, ", 1, ") !== false) {
        $hasDG = true;
    }
}

if (!$hasDG) {
    // Find the INSERT INTO user line and add DG user
    $dgUser = "(1, 'dg', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 1, NULL, NULL, 'Directeur Gérant', 'DG', NULL, 'dg@masunzu.com', NOW(), NOW(), 1),\n";
    
    // Find the position after the VALUES line
    $valuesPos = strpos($finalOutput, "INSERT INTO `user`");
    if ($valuesPos !== false) {
        $nextLinePos = strpos($finalOutput, "\n", $valuesPos);
        if ($nextLinePos !== false) {
            // Insert DG user as first user and renumber others
            $before = substr($finalOutput, 0, $nextLinePos + 1);
            $after = substr($finalOutput, $nextLinePos + 1);
            // Renumber existing users
            $after = preg_replace('/^\((\d+),/', "(2,", $after, 1);
            $after = preg_replace('/^\((\d+),/', "(3,", $after, 1);
            $after = preg_replace('/^\((\d+),/', "(4,", $after, 1);
            $finalOutput = $before . $dgUser . $after;
        }
    }
}

// Update user AUTO_INCREMENT to 5
$finalOutput = preg_replace(
    '/ALTER TABLE `user`\s+MODIFY `id` int\(11\) NOT NULL AUTO_INCREMENT, AUTO_INCREMENT=\d+;/',
    'ALTER TABLE `user`' . "\n" . '  MODIFY `id` int(11) NOT NULL AUTO_INCREMENT, AUTO_INCREMENT=5;',
    $finalOutput
);

file_put_contents($outputFile, $finalOutput);
echo "Production SQL dump created successfully: $outputFile\n";
echo "File size: " . number_format(filesize($outputFile)) . " bytes\n";
