<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to admin roles (DG, Board, DGA, Admin)
$allowed_roles = ['DG', 'Board', 'DGA', 'Admin'];
if (!in_array($_SESSION['role_name'], $allowed_roles)) {
    error_log("Unauthorized access to manage.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php");
    exit;
}

if (!function_exists('mbh_bind_params')) {
    /**
     * Bind parameters to a mysqli prepared statement using a dynamic array.
     *
     * @param mysqli_stmt $stmt
     * @param string $types
     * @param array<int, mixed> $params
     */
    function mbh_bind_params(mysqli_stmt $stmt, string $types, array $params): void
    {
        if ($types === '' || empty($params)) {
            return;
        }

        $refs = [];
        foreach ($params as $key => $value) {
            $refs[$key] = &$params[$key];
        }
        array_unshift($refs, $types);
        call_user_func_array([$stmt, 'bind_param'], $refs);
    }
}

$canManageRoleUtilities = in_array($_SESSION['role_name'], ['DG', 'Board', 'DGA', 'Admin'], true);
$formErrors = [];
$formSuccess = [];

// Handle delete user
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_user_id'])) {
    $delete_id = (int)$_POST['delete_user_id'];
    if ($delete_id !== $_SESSION['user_id']) { // Prevent self-deletion
        $conn = get_db_connection();
        $stmt = $conn->prepare("DELETE FROM user WHERE id = ?");
        $stmt->bind_param("i", $delete_id);
        $stmt->execute();
        $stmt->close();
        $conn->close();
        log_action($_SESSION['user_id'], 'delete_user', "Deleted user ID $delete_id");
        header("Location: /masunzu_bar_hotel/modules/users/manage.php");
        exit;
    }
}

// Handle activate/deactivate user
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle_user_id'])) {
    $toggle_id = (int)$_POST['toggle_user_id'];
    $is_active = (int)$_POST['is_active'];
    if ($toggle_id !== $_SESSION['user_id']) { // Prevent self-deactivation
        $conn = get_db_connection();
        $stmt = $conn->prepare("UPDATE user SET is_active = ? WHERE id = ?");
        $stmt->bind_param("ii", $is_active, $toggle_id);
        $stmt->execute();
        $stmt->close();
        $conn->close();
        log_action($_SESSION['user_id'], $is_active ? 'activate_user' : 'deactivate_user', ($is_active ? "Activated" : "Deactivated") . " user ID $toggle_id");
        header("Location: /masunzu_bar_hotel/modules/users/manage.php");
        exit;
    }
}

$search = trim($_GET['search'] ?? '');
$filterRoleId = filter_input(INPUT_GET, 'role_id', FILTER_VALIDATE_INT, ['options' => ['default' => null]]);
$filterProvinceId = filter_input(INPUT_GET, 'province_id', FILTER_VALIDATE_INT, ['options' => ['default' => null]]);
$filterRegionId = filter_input(INPUT_GET, 'region_id', FILTER_VALIDATE_INT, ['options' => ['default' => null]]);
$filterPosition = trim($_GET['position'] ?? '');

$conn = get_db_connection();
if (!$conn) {
    die('Échec de la connexion à la base de données');
}
$hiddenRoles = ['Stock Manager Assistant'];
$escapedHiddenRoles = array_map([$conn, 'real_escape_string'], $hiddenRoles);
$hiddenRoleList = "'" . implode("','", $escapedHiddenRoles) . "'";

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($canManageRoleUtilities && isset($_POST['create_role'])) {
        $newRoleName = trim($_POST['role_name'] ?? '');
        $newRoleDescription = trim($_POST['role_description'] ?? '');

        if ($newRoleName === '') {
            $formErrors[] = 'Le nom du rôle est requis.';
        } else {
            $stmtRoleCheck = $conn->prepare('SELECT COUNT(*) AS total FROM role WHERE LOWER(name) = LOWER(?)');
            if ($stmtRoleCheck) {
                $stmtRoleCheck->bind_param('s', $newRoleName);
                $stmtRoleCheck->execute();
                $roleExists = (int)($stmtRoleCheck->get_result()->fetch_assoc()['total'] ?? 0);
                $stmtRoleCheck->close();

                if ($roleExists > 0) {
                    $formErrors[] = "Le rôle '{$newRoleName}' existe déjà.";
                } else {
                    $stmtRoleInsert = $conn->prepare('INSERT INTO role (name, description) VALUES (?, ?)');
                    if ($stmtRoleInsert) {
                        $stmtRoleInsert->bind_param('ss', $newRoleName, $newRoleDescription);
                        if ($stmtRoleInsert->execute()) {
                            $formSuccess[] = "Rôle '{$newRoleName}' créé avec succès.";
                            log_action($_SESSION['user_id'], 'create_role', "Created role '{$newRoleName}'");
                        } else {
                            $formErrors[] = 'Échec de la création du rôle: ' . $conn->error;
                        }
                        $stmtRoleInsert->close();
                    } else {
                        $formErrors[] = 'Échec de la préparation de la création du rôle. Veuillez réessayer.';
                    }
                }
            } else {
                $formErrors[] = 'Échec de la validation de la disponibilité du rôle. Veuillez réessayer.';
            }
        }
    }

    $depotRegionSelected = $_POST['province_region_id'] ?? '';
    if ($canManageRoleUtilities && isset($_POST['create_province'])) {
        $newProvinceName = trim($_POST['province_name'] ?? '');
        $newProvinceAddress = trim($_POST['province_address'] ?? '');
        $depotRegionId = (int)($_POST['province_region_id'] ?? 0);

        if ($newProvinceName === '') {
            $formErrors[] = 'Le nom de la province est requis.';
        } elseif ($depotRegionId <= 0) {
            $formErrors[] = 'Veuillez sélectionner une région pour le dépôt.';
        } else {
            $stmtProvinceCheck = $conn->prepare('SELECT COUNT(*) AS total FROM province WHERE LOWER(name) = LOWER(?)');
            if ($stmtProvinceCheck) {
                $stmtProvinceCheck->bind_param('s', $newProvinceName);
                $stmtProvinceCheck->execute();
                $provinceExists = (int)($stmtProvinceCheck->get_result()->fetch_assoc()['total'] ?? 0);
                $stmtProvinceCheck->close();

                if ($provinceExists > 0) {
                    $formErrors[] = "Le dépôt '{$newProvinceName}' existe déjà.";
                } else {
                    $conn->begin_transaction();
                    try {
                        $stmtRegionCheck = $conn->prepare('SELECT id FROM region WHERE id = ?');
                        if (!$stmtRegionCheck) {
                            throw new Exception('Impossible de valider la région sélectionnée.');
                        }
                        $stmtRegionCheck->bind_param('i', $depotRegionId);
                        $stmtRegionCheck->execute();
                        $regionRow = $stmtRegionCheck->get_result()->fetch_assoc();
                        $stmtRegionCheck->close();
                        if (!$regionRow) {
                            throw new Exception('Région invalide pour le dépôt.');
                        }

                        $nextIdResult = $conn->query('SELECT COALESCE(MAX(id), 0) AS max_id FROM province FOR UPDATE');
                        if (!$nextIdResult) {
                            throw new Exception('Impossible de déterminer le prochain identifiant de province.');
                        }
                        $newProvinceId = (int)($nextIdResult->fetch_assoc()['max_id'] ?? 0) + 1;
                        $nextIdResult->free();

                        $stmtProvinceInsert = $conn->prepare('INSERT INTO province (id, name, address, region_id) VALUES (?, ?, ?, ?)');
                        if (!$stmtProvinceInsert) {
                            throw new Exception('Échec de la préparation de la création de la province.');
                        }
                        $stmtProvinceInsert->bind_param('issi', $newProvinceId, $newProvinceName, $newProvinceAddress, $depotRegionId);
                        if (!$stmtProvinceInsert->execute()) {
                            $stmtProvinceInsert->close();
                            throw new Exception('Échec de l\'exécution de la création de la province.');
                        }
                        $stmtProvinceInsert->close();

                        $stmtProvinceAccount = $conn->prepare('INSERT INTO province_cash_account (province_id) VALUES (?) ON DUPLICATE KEY UPDATE updated_at = NOW()');
                        if (!$stmtProvinceAccount) {
                            throw new Exception('Échec de la préparation de l\'initialisation du compte de province.');
                        }
                        $stmtProvinceAccount->bind_param('i', $newProvinceId);
                        if (!$stmtProvinceAccount->execute()) {
                            $stmtProvinceAccount->close();
                            throw new Exception('Impossible d\'initialiser le compte de province.');
                        }
                        $stmtProvinceAccount->close();

                        $conn->commit();
                        $formSuccess[] = "Dépôt '{$newProvinceName}' créé avec succès.";
                        log_action($_SESSION['user_id'], 'create_province', "Created depot '{$newProvinceName}' (ID: $newProvinceId, Address: $newProvinceAddress)");
                    } catch (Throwable $e) {
                        $conn->rollback();
                        $formErrors[] = 'Échec de la création du dépôt. Veuillez réessayer.';
                        error_log('Dépôt creation failure: ' . $e->getMessage());
                    }
                }
            } else {
                $formErrors[] = 'Échec de la validation de la disponibilité de la province. Veuillez réessayer.';
            }
        }
    }
}

$roles = [];
$existingRoles = [];
$roleResult = $conn->query("SELECT id, name, description FROM role WHERE name NOT IN ($hiddenRoleList) ORDER BY name");
if ($roleResult instanceof mysqli_result) {
    $roles = $roleResult->fetch_all(MYSQLI_ASSOC);
    $existingRoles = $roles;
    $roleResult->free();
}

$provinces = [];
$existingProvinces = [];
$provinceResult = $conn->query('SELECT id, name FROM province ORDER BY name');
if ($provinceResult instanceof mysqli_result) {
    $provinces = $provinceResult->fetch_all(MYSQLI_ASSOC);
    $existingProvinces = $provinces;
    $provinceResult->free();
}

// Fetch detailed depot information with region and user count
$depotsDetailed = [];
$depotDetailQuery = $conn->query("
    SELECT 
        p.id,
        p.name,
        p.address,
        p.created_at,
        r.name AS region_name,
        COUNT(DISTINCT u.id) AS user_count
    FROM province p
    LEFT JOIN region r ON p.region_id = r.id
    LEFT JOIN user u ON u.province_id = p.id
    GROUP BY p.id, p.name, p.address, p.created_at, r.name
    ORDER BY p.name
");
if ($depotDetailQuery instanceof mysqli_result) {
    $depotsDetailed = $depotDetailQuery->fetch_all(MYSQLI_ASSOC);
    $depotDetailQuery->free();
}

// Fetch users per depot for the modal
$usersPerDepot = [];
$usersPerDepotQuery = $conn->query("
    SELECT 
        u.id,
        u.full_name,
        u.email,
        u.is_active,
        u.province_id,
        r.name AS role_name
    FROM user u
    LEFT JOIN role r ON u.role_id = r.id
    WHERE u.province_id IS NOT NULL
    ORDER BY u.full_name
");
if ($usersPerDepotQuery instanceof mysqli_result) {
    while ($row = $usersPerDepotQuery->fetch_assoc()) {
        $depotId = (int)$row['province_id'];
        if (!isset($usersPerDepot[$depotId])) {
            $usersPerDepot[$depotId] = [];
        }
        $usersPerDepot[$depotId][] = $row;
    }
    $usersPerDepotQuery->free();
}

$regions = [];
$regionResult = $conn->query('SELECT id, name FROM region ORDER BY name');
if ($regionResult instanceof mysqli_result) {
    $regions = $regionResult->fetch_all(MYSQLI_ASSOC);
    $regionResult->free();
}

$positions = [];
$positionResult = $conn->query("SELECT DISTINCT position FROM user WHERE position IS NOT NULL AND position <> '' ORDER BY position");
if ($positionResult instanceof mysqli_result) {
    while ($row = $positionResult->fetch_assoc()) {
        $positions[] = $row['position'];
    }
    $positionResult->free();
}

$conditions = [];
$types = '';
$params = [];

if ($search !== '') {
    $like = '%' . $search . '%';
    $conditions[] = '(u.username LIKE ? OR u.full_name LIKE ? OR u.email LIKE ? OR COALESCE(p.name, "") LIKE ? OR COALESCE(r.name, "") LIKE ? OR COALESCE(u.position, "") LIKE ?)';
    $types .= 'ssssss';
    $params = array_merge($params, [$like, $like, $like, $like, $like, $like]);
}

if (!empty($filterRoleId)) {
    $conditions[] = 'u.role_id = ?';
    $types .= 'i';
    $params[] = $filterRoleId;
}

if (!empty($filterProvinceId)) {
    $conditions[] = 'u.province_id = ?';
    $types .= 'i';
    $params[] = $filterProvinceId;
}

if (!empty($filterRegionId)) {
    $conditions[] = 'u.region_id = ?';
    $types .= 'i';
    $params[] = $filterRegionId;
}

if ($filterPosition !== '') {
    $conditions[] = 'u.position LIKE ?';
    $types .= 's';
    $params[] = '%' . $filterPosition . '%';
}

$conditions[] = "(r.name IS NULL OR r.name NOT IN ($hiddenRoleList))";

$whereClause = empty($conditions) ? '' : ' WHERE ' . implode(' AND ', $conditions);
$baseJoin = ' FROM user u LEFT JOIN role r ON u.role_id = r.id LEFT JOIN province p ON u.province_id = p.id LEFT JOIN region rg ON u.region_id = rg.id LEFT JOIN region prov_rg ON p.region_id = prov_rg.id';

$metrics = ['total_users' => 0, 'active_users' => 0, 'inactive_users' => 0];
$metricsStmt = $conn->prepare('SELECT COUNT(*) AS total_users, SUM(CASE WHEN u.is_active = 1 THEN 1 ELSE 0 END) AS active_users, SUM(CASE WHEN u.is_active = 0 THEN 1 ELSE 0 END) AS inactive_users' . $baseJoin . $whereClause);
if ($metricsStmt) {
    mbh_bind_params($metricsStmt, $types, $params);
    $metricsStmt->execute();
    $metricsResult = $metricsStmt->get_result();
    if ($metricsResult && $metricsRow = $metricsResult->fetch_assoc()) {
        $metrics = array_map('intval', $metricsRow);
    }
    $metricsStmt->close();
}

$roleBreakdown = [];
$roleBreakdownStmt = $conn->prepare('SELECT COALESCE(r.name, "Non assigné") AS role_name, COUNT(*) AS total' . $baseJoin . $whereClause . ' GROUP BY r.id, role_name ORDER BY total DESC');
if ($roleBreakdownStmt) {
    mbh_bind_params($roleBreakdownStmt, $types, $params);
    $roleBreakdownStmt->execute();
    $roleBreakdown = $roleBreakdownStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $roleBreakdownStmt->close();
}

$provinceBreakdown = [];
$provinceBreakdownStmt = $conn->prepare('SELECT COALESCE(p.name, "Non assigné") AS province_name, COUNT(*) AS total' . $baseJoin . $whereClause . ' GROUP BY p.id, province_name ORDER BY total DESC');
if ($provinceBreakdownStmt) {
    mbh_bind_params($provinceBreakdownStmt, $types, $params);
    $provinceBreakdownStmt->execute();
    $provinceBreakdown = $provinceBreakdownStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $provinceBreakdownStmt->close();
}

$regionBreakdown = [];
$regionBreakdownStmt = $conn->prepare('SELECT COALESCE(rg.name, prov_rg.name, "Non assigné") AS region_name, COUNT(*) AS total' . $baseJoin . $whereClause . ' GROUP BY COALESCE(rg.name, prov_rg.name, "Non assigné") ORDER BY total DESC');
if ($regionBreakdownStmt) {
    mbh_bind_params($regionBreakdownStmt, $types, $params);
    $regionBreakdownStmt->execute();
    $regionBreakdown = $regionBreakdownStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $regionBreakdownStmt->close();
}

$users = [];
$usersStmt = $conn->prepare('SELECT u.id, u.username, u.full_name, u.email, u.position, u.is_active, u.created_at, COALESCE(r.name, "Non assigné") AS role_name, COALESCE(p.name, "Non assigné") AS province_name, COALESCE(rg.name, prov_rg.name, "Non assigné") AS region_name' . $baseJoin . $whereClause . ' ORDER BY u.created_at DESC');
if ($usersStmt) {
    mbh_bind_params($usersStmt, $types, $params);
    $usersStmt->execute();
    $users = $usersStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $usersStmt->close();
}

$conn->close();
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<div class="dashboard-container" style="padding: 20px; background-color: #F4F0E4; min-height: 70vh;">
    <div class="dashboard-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
        <h1 style="color: #4B2F1F; font-size: 28px; font-weight: bold; margin: 0;">Gestion Utilisateurs</h1>
        <a href="/masunzu_bar_hotel/modules/users/add_user.php" style="background-color: #F4A261; color: #000000; padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: bold;">Ajouter Nouvel Utilisateur</a>
    </div>
    <div class="dashboard-content" style="background-color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.1);">
        <h2 style="color: #4B2F1F; margin-bottom: 20px;">Utilisateurs</h2>

        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(160px, 1fr)); gap: 15px; margin-bottom: 20px;">
            <div style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color: #F4F0E4; padding: 15px; border-radius: 10px;">
                <p style="margin: 0; font-size: 12px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.8;">Total Utilisateurs</p>
                <p style="margin: 5px 0 0; font-size: 28px; font-weight: 700;"><?php echo number_format($metrics['total_users']); ?></p>
            </div>
            <div style="background-color: #F4F0E4; padding: 15px; border-radius: 10px; border-left: 4px solid #2E7D32; color: #2C1810;">
                <p style="margin: 0; font-size: 12px; text-transform: uppercase; letter-spacing: 0.05em; color: rgba(44,24,16,0.65);">Actifs</p>
                <p style="margin: 5px 0 0; font-size: 24px; font-weight: 700; color: #2E7D32;"><?php echo number_format($metrics['active_users']); ?></p>
            </div>
            <div style="background-color: #F4F0E4; padding: 15px; border-radius: 10px; border-left: 4px solid #C62828; color: #2C1810;">
                <p style="margin: 0; font-size: 12px; text-transform: uppercase; letter-spacing: 0.05em; color: rgba(44,24,16,0.65);">Inactifs</p>
                <p style="margin: 5px 0 0; font-size: 24px; font-weight: 700; color: #C62828;"><?php echo number_format($metrics['inactive_users']); ?></p>
            </div>
        </div>

                <?php if (!empty($roleBreakdown) || !empty($provinceBreakdown) || !empty($regionBreakdown)): ?>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); gap: 15px; margin-bottom: 20px;">
                        <?php if (!empty($roleBreakdown)): ?>
                            <div style="background-color: #FDFBF7; padding: 15px; border-radius: 10px; border: 1px solid rgba(75,47,31,0.1);">
                                <h4 style="margin: 0 0 10px; color: #4B2F1F; font-size: 16px;">Utilisateurs par Rôle</h4>
                                <ul style="list-style: none; padding: 0; margin: 0;">
                            <?php foreach (array_slice($roleBreakdown, 0, 5) as $roleRow): ?>
                                <li style="display: flex; justify-content: space-between; padding: 6px 0; border-bottom: 1px dashed rgba(75,47,31,0.1); color: #2C1810;">
                                    <span><?php echo htmlspecialchars($roleRow['role_name'], ENT_QUOTES, 'UTF-8'); ?></span>
                                    <span style="font-weight: 600;"><?php echo number_format((int)$roleRow['total']); ?></span>
                                </li>
                            <?php endforeach; ?>
                            <?php if (empty($roleBreakdown)): ?>
                                <li style="color: rgba(44,24,16,0.65);">Aucune donnée de rôle.</li>
                            <?php endif; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                        <?php if (!empty($provinceBreakdown)): ?>
                            <div style="background-color: #FDFBF7; padding: 15px; border-radius: 10px; border: 1px solid rgba(75,47,31,0.1);">
                                <h4 style="margin: 0 0 10px; color: #4B2F1F; font-size: 16px;">Utilisateurs par Dépôt</h4>
                                <ul style="list-style: none; padding: 0; margin: 0;">
                            <?php foreach (array_slice($provinceBreakdown, 0, 5) as $provinceRow): ?>
                                <li style="display: flex; justify-content: space-between; padding: 6px 0; border-bottom: 1px dashed rgba(75,47,31,0.1); color: #2C1810;">
                                    <span><?php echo htmlspecialchars($provinceRow['province_name'], ENT_QUOTES, 'UTF-8'); ?></span>
                                    <span style="font-weight: 600;"><?php echo number_format((int)$provinceRow['total']); ?></span>
                                </li>
                            <?php endforeach; ?>
                                    <?php if (empty($provinceBreakdown)): ?>
                                        <li style="color: rgba(44,24,16,0.65);">Aucune donnée de dépôt.</li>
                                    <?php endif; ?>
                        </ul>
                    </div>
                        <?php endif; ?>
                        <?php if (!empty($regionBreakdown)): ?>
                            <div style="background-color: #FDFBF7; padding: 15px; border-radius: 10px; border: 1px solid rgba(75,47,31,0.1);">
                                <h4 style="margin: 0 0 10px; color: #4B2F1F; font-size: 16px;">Utilisateurs par Région</h4>
                                <ul style="list-style: none; padding: 0; margin: 0;">
                                    <?php foreach (array_slice($regionBreakdown, 0, 5) as $regionRow): ?>
                                        <li style="display: flex; justify-content: space-between; padding: 6px 0; border-bottom: 1px dashed rgba(75,47,31,0.1); color: #2C1810;">
                                            <span><?php echo htmlspecialchars($regionRow['region_name'], ENT_QUOTES, 'UTF-8'); ?></span>
                                            <span style="font-weight: 600;"><?php echo number_format((int)$regionRow['total']); ?></span>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

        <?php if ($canManageRoleUtilities): ?>
            <!-- Create Depot Card with Toggle -->
            <div style="background:#fff; border-radius:12px; box-shadow:0 2px 8px rgba(0,0,0,0.06); margin-bottom:20px; overflow:hidden;">
                <div style="display:flex; justify-content:space-between; align-items:center; padding:16px 20px; background:#FDFBF7; border-bottom:1px solid #F0F0F0;">
                    <div style="display:flex; align-items:center; gap:12px;">
                        <h4 style="color:#4B2F1F; font-size:16px; margin:0; font-weight:600;">Gestion des Dépôts</h4>
                        <span style="background:#e3f2fd; color:#1565C0; padding:4px 10px; border-radius:12px; font-size:12px; font-weight:600;"><?php echo count($depotsDetailed); ?> dépôt<?php echo count($depotsDetailed) !== 1 ? 's' : ''; ?></span>
                    </div>
                    <div style="display:flex; gap:10px;">
                        <button type="button" id="viewAllDepots" style="background:#2196F3; color:#fff; padding:10px 18px; border:none; border-radius:8px; cursor:pointer; font-size:14px; font-weight:600; display:flex; align-items:center; gap:8px; transition:all 0.2s;">
                            📋 Voir Tous les Dépôts
                        </button>
                        <button type="button" id="toggleCreateDepot" style="background:#4B2F1F; color:#F4F0E4; padding:10px 18px; border:none; border-radius:8px; cursor:pointer; font-size:14px; font-weight:600; display:flex; align-items:center; gap:8px; transition:all 0.2s;">
                            <span style="font-size:18px;">+</span> Créer Nouveau Dépôt
                        </button>
                    </div>
                </div>
                <div id="createDepotForm" style="display:none; padding:20px;">
                    <?php if (!empty($formErrors) || !empty($formSuccess)): ?>
                        <?php foreach ($formErrors as $err): ?>
                            <div style="background:#ffebee; color:#C62828; padding:12px; border-radius:6px; margin-bottom:12px; border-left:4px solid #C62828;"><?php echo htmlspecialchars($err); ?></div>
                        <?php endforeach; ?>
                        <?php foreach ($formSuccess as $s): ?>
                            <div style="background:#e8f5e9; color:#2E7D32; padding:12px; border-radius:6px; margin-bottom:12px; border-left:4px solid #2E7D32;">✓ <?php echo htmlspecialchars($s); ?></div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    <form method="POST" style="display:grid; grid-template-columns:repeat(3, 1fr); gap:15px; align-items:end;">
                        <div>
                            <label for="province_name" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Nom du dépôt <span style="color:red;">*</span></label>
                            <input type="text" id="province_name" name="province_name" placeholder="Ex: Bujumbura Centre" style="width:100%; padding:10px; border:1px solid #B08968; border-radius:6px; font-size:14px; box-sizing:border-box;" required>
                        </div>
                        <div>
                            <label for="province_region_id" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Région <span style="color:red;">*</span></label>
                            <select name="province_region_id" id="province_region_id" style="width:100%; padding:10px; border:1px solid #B08968; border-radius:6px; font-size:14px; box-sizing:border-box;" required>
                                <option value="">Sélectionnez une région</option>
                                <?php foreach ($regions as $regionOption): ?>
                                    <option value="<?php echo (int)$regionOption['id']; ?>" <?php echo ((int)$regionOption['id'] === (int)($depotRegionSelected ?? 0)) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($regionOption['name'], ENT_QUOTES, 'UTF-8'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <label for="province_address" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Adresse</label>
                            <input type="text" id="province_address" name="province_address" placeholder="Adresse du dépôt" style="width:100%; padding:10px; border:1px solid #B08968; border-radius:6px; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div style="grid-column:span 3; display:flex; gap:10px; justify-content:flex-end; margin-top:10px; padding-top:15px; border-top:1px solid #F0F0F0;">
                            <button type="button" onclick="document.getElementById('createDepotForm').style.display='none'; document.getElementById('toggleCreateDepot').innerHTML='<span style=font-size:18px>+</span> Créer Nouveau Dépôt'; document.getElementById('toggleCreateDepot').style.background='#4B2F1F';" style="background:#f0f0f0; color:#4B2F1F; padding:11px 24px; border:none; border-radius:6px; cursor:pointer; font-size:14px; font-weight:500;">Annuler</button>
                            <button type="submit" name="create_province" value="1" style="background:#4B2F1F; color:#F4F0E4; padding:11px 24px; border:none; border-radius:6px; cursor:pointer; font-size:14px; font-weight:600;">✓ Créer le Dépôt</button>
                        </div>
                    </form>
                </div>
            </div>
        <?php endif; ?>

        <form method="GET" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px; align-items: end;">
            <div>
                <label for="search" style="display:block; font-weight:600; color:#4B2F1F; margin-bottom:5px;">Rechercher</label>
                <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search, ENT_QUOTES, 'UTF-8'); ?>" placeholder="Nom, email, rôle..." style="width:100%; padding:9px; border-radius:8px; border:1px solid #4B2F1F; background-color:#F4F0E4; color:#4B2F1F;">
            </div>
            <div>
                <label for="role_id" style="display:block; font-weight:600; color:#4B2F1F; margin-bottom:5px;">Rôle</label>
                <select id="role_id" name="role_id" style="width:100%; padding:9px; border-radius:8px; border:1px solid #4B2F1F; background-color:#F4F0E4; color:#4B2F1F;">
                    <option value="">Tous les rôles</option>
                    <?php foreach ($roles as $roleOption): ?>
                        <option value="<?php echo (int)$roleOption['id']; ?>" <?php echo ((int)$roleOption['id'] === (int)($filterRoleId ?? 0)) ? 'selected' : ''; ?>><?php echo htmlspecialchars($roleOption['name'], ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label for="region_id" style="display:block; font-weight:600; color:#4B2F1F; margin-bottom:5px;">Région</label>
                <select id="region_id" name="region_id" style="width:100%; padding:9px; border-radius:8px; border:1px solid #4B2F1F; background-color:#F4F0E4; color:#4B2F1F;">
                    <option value="">Toutes les régions</option>
                    <?php foreach ($regions as $regionOption): ?>
                        <option value="<?php echo (int)$regionOption['id']; ?>" <?php echo ((int)$regionOption['id'] === (int)($filterRegionId ?? 0)) ? 'selected' : ''; ?>><?php echo htmlspecialchars($regionOption['name'], ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label for="province_id" style="display:block; font-weight:600; color:#4B2F1F; margin-bottom:5px;">Dépôt</label>
                <select id="province_id" name="province_id" style="width:100%; padding:9px; border-radius:8px; border:1px solid #4B2F1F; background-color:#F4F0E4; color:#4B2F1F;">
                    <option value="">Tous les dépôts</option>
                    <?php foreach ($provinces as $provinceOption): ?>
                        <option value="<?php echo (int)$provinceOption['id']; ?>" <?php echo ((int)$provinceOption['id'] === (int)($filterProvinceId ?? 0)) ? 'selected' : ''; ?>><?php echo htmlspecialchars($provinceOption['name'], ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label for="position" style="display:block; font-weight:600; color:#4B2F1F; margin-bottom:5px;">Poste</label>
                <select id="position" name="position" style="width:100%; padding:9px; border-radius:8px; border:1px solid #4B2F1F; background-color:#F4F0E4; color:#4B2F1F;">
                    <option value="">Tous les postes</option>
                    <?php foreach ($positions as $positionOption): ?>
                        <option value="<?php echo htmlspecialchars($positionOption, ENT_QUOTES, 'UTF-8'); ?>" <?php echo ($filterPosition !== '' && strcasecmp($filterPosition, $positionOption) === 0) ? 'selected' : ''; ?>><?php echo htmlspecialchars($positionOption, ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div style="display:flex; gap:10px;">
                <button type="submit" style="flex:1; background-color:#4B2F1F; color:#F4F0E4; padding:10px 15px; border:none; border-radius:8px; cursor:pointer; font-weight:600;">Appliquer Filtres</button>
                <a href="/masunzu_bar_hotel/modules/users/manage.php" style="flex:1; text-align:center; padding:10px 15px; border-radius:8px; border:1px solid #4B2F1F; color:#4B2F1F; text-decoration:none; font-weight:600;">Réinitialiser</a>
            </div>
        </form>

        <?php if (empty($users)): ?>
            <p style="color: #000000;">Aucun utilisateur ne correspond aux critères sélectionnés.</p>
        <?php else: ?>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 10px; text-align: left;">Nom d'utilisateur</th>
                            <th style="padding: 10px; text-align: left;">Nom Complet</th>
                            <th style="padding: 10px; text-align: left;">Email</th>
                            <th style="padding: 10px; text-align: left;">Poste</th>
                            <th style="padding: 10px; text-align: left;">Rôle</th>
                            <th style="padding: 10px; text-align: left;">Dépôt</th>
                            <th style="padding: 10px; text-align: left;">Région</th>
                            <th style="padding: 10px; text-align: left;">Créé</th>
                            <th style="padding: 10px; text-align: left;">Statut</th>
                            <th style="padding: 10px; text-align: left;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr style="border-bottom: 1px solid #F4F0E4;">
                                <td style="padding: 10px; color:#2C1810; font-weight:600;"><?php echo htmlspecialchars($user['username'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px;"><?php echo htmlspecialchars($user['full_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px;"><?php echo htmlspecialchars($user['email'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px;"><?php echo htmlspecialchars($user['position'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px;"><?php echo htmlspecialchars($user['role_name'] ?? 'Non assigné', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px;"><?php echo htmlspecialchars($user['province_name'] ?? 'Non assigné', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px;"><?php echo htmlspecialchars($user['region_name'] ?? 'Non assigné', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px;"><?php echo htmlspecialchars(date('Y-m-d', strtotime($user['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px; font-weight:600; color: <?php echo $user['is_active'] ? '#2E7D32' : '#C62828'; ?>;"><?php echo $user['is_active'] ? 'Actif' : 'Inactif'; ?></td>
                                <td style="padding: 10px;">
                                    <a href="/masunzu_bar_hotel/modules/users/view_user.php?id=<?php echo (int)$user['id']; ?>" style="color: #4B2F1F; text-decoration: none; margin-right: 10px; font-weight: 600;">Voir</a>
                                    <a href="/masunzu_bar_hotel/modules/users/edit_user.php?id=<?php echo (int)$user['id']; ?>" style="color: #F4A261; text-decoration: none; margin-right: 10px;">Modifier</a>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Êtes-vous sûr de vouloir supprimer cet utilisateur ?');">
                                        <input type="hidden" name="delete_user_id" value="<?php echo (int)$user['id']; ?>">
                                        <button type="submit" style="background: none; border: none; color: red; cursor: pointer;">Supprimer</button>
                                    </form>
                                    <button class="toggle-btn" data-user-id="<?php echo (int)$user['id']; ?>" data-is-active="<?php echo (int)$user['is_active']; ?>" style="background: none; border: none; color: <?php echo $user['is_active'] ? '#FFA500' : '#008000'; ?>; cursor: pointer;"><?php echo $user['is_active'] ? 'Désactiver' : 'Activer'; ?></button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>
<!-- Custom Modal for User Activation -->
<div id="confirmModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); justify-content: center; align-items: center; z-index: 1000;">
    <div style="background-color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 400px; width: 90%; text-align: center;">
        <h3 style="color: #4B2F1F; margin-bottom: 15px;">Confirmer Action</h3>
        <p id="modalMessage" style="color: #000000; margin-bottom: 20px;"></p>
        <form id="toggleForm" method="POST">
            <input type="hidden" name="toggle_user_id" id="modalUserId">
            <input type="hidden" name="is_active" id="modalIsActive">
            <button type="submit" style="background-color: #F4A261; color: #000000; padding: 10px 20px; border: none; border-radius: 8px; cursor: pointer; font-weight: bold; margin-right: 10px;">Confirmer</button>
            <button type="button" id="cancelBtn" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 20px; border: none; border-radius: 8px; cursor: pointer; font-weight: bold;">Annuler</button>
        </form>
    </div>
</div>

<!-- Modal for Viewing All Depots -->
<div id="depotsModal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.6); z-index:1001; overflow-y:auto;">
    <div style="background:#F4F0E4; margin:30px auto; padding:0; border-radius:12px; box-shadow:0 8px 32px rgba(0,0,0,0.25); max-width:1000px; width:95%; max-height:90vh; display:flex; flex-direction:column;">
        <!-- Modal Header -->
        <div style="background:linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color:#F4F0E4; padding:20px 24px; border-radius:12px 12px 0 0; display:flex; justify-content:space-between; align-items:center;">
            <div>
                <h3 style="margin:0; font-size:20px; font-weight:700;">Tous les Dépôts</h3>
                <p style="margin:5px 0 0; font-size:13px; opacity:0.9;"><?php echo count($depotsDetailed); ?> dépôt<?php echo count($depotsDetailed) !== 1 ? 's' : ''; ?> enregistré<?php echo count($depotsDetailed) !== 1 ? 's' : ''; ?></p>
            </div>
            <button type="button" onclick="closeDepotsModal()" style="background:none; border:none; color:#F4F0E4; font-size:28px; cursor:pointer; padding:0; line-height:1;">&times;</button>
        </div>
        
        <!-- Modal Body -->
        <div style="padding:20px 24px; overflow-y:auto; flex:1;">
            <!-- Search in modal -->
            <div style="margin-bottom:16px;">
                <input type="text" id="depotSearchInput" placeholder="Rechercher un dépôt..." style="width:100%; padding:12px 16px; border:1px solid #B08968; border-radius:8px; font-size:14px; box-sizing:border-box;">
            </div>
            
            <!-- Depots Grid -->
            <div id="depotsGrid" style="display:grid; grid-template-columns:repeat(auto-fill, minmax(300px, 1fr)); gap:16px;">
                <?php if (empty($depotsDetailed)): ?>
                    <div style="grid-column:1/-1; text-align:center; padding:40px; color:#6B4F2F;">
                        <p style="font-size:48px; margin:0 0 10px;">🏪</p>
                        <p style="font-weight:600; margin:0;">Aucun dépôt disponible</p>
                        <p style="font-size:13px; margin:5px 0 0;">Créez votre premier dépôt ci-dessus.</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($depotsDetailed as $depot): ?>
                        <?php $depotUsers = $usersPerDepot[(int)$depot['id']] ?? []; ?>
                        <div class="depot-card" data-name="<?php echo htmlspecialchars(strtolower($depot['name']), ENT_QUOTES, 'UTF-8'); ?>" style="background:#fff; border-radius:10px; box-shadow:0 2px 8px rgba(0,0,0,0.06); overflow:hidden; border:1px solid #F0F0F0;">
                            <!-- Depot Header -->
                            <div style="background:#FDFBF7; padding:14px 16px; border-bottom:1px solid #F0F0F0; display:flex; justify-content:space-between; align-items:center;">
                                <div>
                                    <h4 style="margin:0; color:#4B2F1F; font-size:16px; font-weight:600;"><?php echo htmlspecialchars($depot['name'], ENT_QUOTES, 'UTF-8'); ?></h4>
                                    <p style="margin:4px 0 0; color:#6B4F2F; font-size:12px;">
                                        📍 <?php echo htmlspecialchars($depot['region_name'] ?? 'Région non assignée', ENT_QUOTES, 'UTF-8'); ?>
                                    </p>
                                </div>
                                <span style="background:<?php echo (int)$depot['user_count'] > 0 ? '#e8f5e9' : '#FFF3E0'; ?>; color:<?php echo (int)$depot['user_count'] > 0 ? '#2E7D32' : '#E65100'; ?>; padding:4px 10px; border-radius:12px; font-size:12px; font-weight:600;">
                                    <?php echo (int)$depot['user_count']; ?> utilisateur<?php echo (int)$depot['user_count'] !== 1 ? 's' : ''; ?>
                                </span>
                            </div>
                            
                            <!-- Depot Details -->
                            <div style="padding:14px 16px;">
                                <?php if (!empty($depot['address'])): ?>
                                    <p style="margin:0 0 10px; color:#4B2F1F; font-size:13px;">
                                        <strong>Adresse:</strong> <?php echo htmlspecialchars($depot['address'], ENT_QUOTES, 'UTF-8'); ?>
                                    </p>
                                <?php endif; ?>
                                <p style="margin:0 0 10px; color:#6B4F2F; font-size:12px;">
                                    <strong>Créé le:</strong> <?php echo date('d/m/Y', strtotime($depot['created_at'] ?? 'now')); ?>
                                </p>
                                
                                <!-- Users List -->
                                <?php if (!empty($depotUsers)): ?>
                                    <div style="margin-top:12px; border-top:1px solid #F0F0F0; padding-top:12px;">
                                        <p style="margin:0 0 8px; color:#4B2F1F; font-size:12px; font-weight:600; text-transform:uppercase; letter-spacing:0.03em;">Utilisateurs assignés:</p>
                                        <div style="max-height:150px; overflow-y:auto;">
                                            <?php foreach ($depotUsers as $depotUser): ?>
                                                <div style="display:flex; justify-content:space-between; align-items:center; padding:6px 0; border-bottom:1px dashed #F0F0F0;">
                                                    <div>
                                                        <span style="color:#2C1810; font-size:13px; font-weight:500;"><?php echo htmlspecialchars($depotUser['full_name'], ENT_QUOTES, 'UTF-8'); ?></span>
                                                        <span style="color:#6B4F2F; font-size:11px; margin-left:6px;">(<?php echo htmlspecialchars($depotUser['role_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>)</span>
                                                    </div>
                                                    <span style="width:8px; height:8px; border-radius:50%; background:<?php echo $depotUser['is_active'] ? '#2E7D32' : '#C62828'; ?>;"></span>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div style="margin-top:12px; border-top:1px solid #F0F0F0; padding-top:12px; text-align:center;">
                                        <p style="margin:0; color:#999; font-size:12px; font-style:italic;">Aucun utilisateur assigné</p>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Modal Footer -->
        <div style="padding:16px 24px; border-top:1px solid #E8E8E8; background:#fff; border-radius:0 0 12px 12px; display:flex; justify-content:flex-end;">
            <button type="button" onclick="closeDepotsModal()" style="background:#4B2F1F; color:#F4F0E4; padding:10px 24px; border:none; border-radius:6px; cursor:pointer; font-size:14px; font-weight:600;">Fermer</button>
        </div>
    </div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Toggle user activation modal
    const toggleButtons = document.querySelectorAll('.toggle-btn');
    const modal = document.getElementById('confirmModal');
    const modalMessage = document.getElementById('modalMessage');
    const modalUserId = document.getElementById('modalUserId');
    const modalIsActive = document.getElementById('modalIsActive');
    const cancelBtn = document.getElementById('cancelBtn');

    toggleButtons.forEach(button => {
        button.addEventListener('click', function() {
            const userId = this.getAttribute('data-user-id');
            const isActive = this.getAttribute('data-is-active');
            modalMessage.textContent = `Êtes-vous sûr de vouloir ${isActive === '1' ? 'désactiver' : 'activer'} cet utilisateur ?`;
            modalUserId.value = userId;
            modalIsActive.value = isActive === '1' ? '0' : '1';
            modal.style.display = 'flex';
        });
    });

    cancelBtn.addEventListener('click', function() {
        modal.style.display = 'none';
    });

    // Toggle create depot form
    const toggleDepotBtn = document.getElementById('toggleCreateDepot');
    const createDepotForm = document.getElementById('createDepotForm');
    
    if (toggleDepotBtn && createDepotForm) {
        toggleDepotBtn.addEventListener('click', function() {
            const isHidden = createDepotForm.style.display === 'none';
            createDepotForm.style.display = isHidden ? 'block' : 'none';
            
            if (isHidden) {
                toggleDepotBtn.innerHTML = '<span style="font-size:18px">−</span> Masquer le Formulaire';
                toggleDepotBtn.style.background = '#6B4F2F';
                // Scroll to form smoothly
                createDepotForm.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                // Focus on first input
                const firstInput = document.getElementById('province_name');
                if (firstInput) {
                    setTimeout(() => firstInput.focus(), 300);
                }
            } else {
                toggleDepotBtn.innerHTML = '<span style="font-size:18px">+</span> Créer Nouveau Dépôt';
                toggleDepotBtn.style.background = '#4B2F1F';
            }
        });
        
        // Auto-show form if there are form errors or success messages
        <?php if (!empty($formErrors) || !empty($formSuccess)): ?>
        createDepotForm.style.display = 'block';
        toggleDepotBtn.innerHTML = '<span style="font-size:18px">−</span> Masquer le Formulaire';
        toggleDepotBtn.style.background = '#6B4F2F';
        <?php endif; ?>
    }

    // View All Depots Modal
    const viewAllDepotsBtn = document.getElementById('viewAllDepots');
    const depotsModal = document.getElementById('depotsModal');
    const depotSearchInput = document.getElementById('depotSearchInput');
    
    if (viewAllDepotsBtn && depotsModal) {
        viewAllDepotsBtn.addEventListener('click', function() {
            depotsModal.style.display = 'block';
            document.body.style.overflow = 'hidden';
            if (depotSearchInput) {
                setTimeout(() => depotSearchInput.focus(), 300);
            }
        });
        
        // Close modal when clicking outside
        depotsModal.addEventListener('click', function(e) {
            if (e.target === depotsModal) {
                closeDepotsModal();
            }
        });
    }
    
    // Search functionality in depots modal
    if (depotSearchInput) {
        depotSearchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase().trim();
            const depotCards = document.querySelectorAll('.depot-card');
            
            depotCards.forEach(card => {
                const depotName = card.getAttribute('data-name') || '';
                const cardText = card.textContent.toLowerCase();
                
                if (searchTerm === '' || depotName.includes(searchTerm) || cardText.includes(searchTerm)) {
                    card.style.display = '';
                } else {
                    card.style.display = 'none';
                }
            });
        });
    }
    
    // Close modal on Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && depotsModal && depotsModal.style.display === 'block') {
            closeDepotsModal();
        }
    });
});

function closeDepotsModal() {
    const depotsModal = document.getElementById('depotsModal');
    if (depotsModal) {
        depotsModal.style.display = 'none';
        document.body.style.overflow = 'auto';
    }
}
</script>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
