<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to admin roles
$allowed_roles = ['DG', 'Board', 'DGA', 'Admin'];
if (!in_array($_SESSION['role_name'], $allowed_roles)) {
    header("Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php");
    exit;
}

// Fetch user
$user_id = (int)($_GET['id'] ?? 0);
$conn = get_db_connection();
$stmt = $conn->prepare("SELECT u.id, u.username, u.full_name, u.email, u.mobile1, u.mobile2, u.position, u.role_id, u.is_active, u.address, u.province_id, u.region_id, r.name AS role_name FROM user u LEFT JOIN role r ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->num_rows === 1 ? $result->fetch_assoc() : null;
$stmt->close();

// Fetch roles
$roles_result = $conn->query("SELECT id, name FROM role");
$roles = $roles_result ? $roles_result->fetch_all(MYSQLI_ASSOC) : [];

$provinces_result = $conn->query("SELECT id, name FROM province ORDER BY name");
$provinces = $provinces_result ? $provinces_result->fetch_all(MYSQLI_ASSOC) : [];

$regions_result = $conn->query("SELECT id, name FROM region ORDER BY name");
$regions = $regions_result ? $regions_result->fetch_all(MYSQLI_ASSOC) : [];

$departments_result = $conn->query("SELECT id, code, label FROM department ORDER BY label");
$departments = $departments_result ? $departments_result->fetch_all(MYSQLI_ASSOC) : [];
$departmentCodesAvailable = $departments ? array_column($departments, 'code') : [];

$userDepartmentStmt = $conn->prepare("SELECT d.code FROM user_department ud JOIN department d ON ud.department_id = d.id WHERE ud.user_id = ?");
$userDepartmentStmt->bind_param("i", $user_id);
$userDepartmentStmt->execute();
$deptResult = $userDepartmentStmt->get_result();
$userDepartmentCodes = $deptResult ? array_column($deptResult->fetch_all(MYSQLI_ASSOC), 'code') : [];
$userDepartmentStmt->close();
$conn->close();

if (!$user) {
    header("Location: /masunzu_bar_hotel/modules/users/manage.php");
    exit;
}
$originalUser = $user;
$roleLabels = [];
foreach ($roles as $roleRow) {
    $roleLabels[$roleRow['id']] = $roleRow['name'];
}
$inputDepartmentCodes = $userDepartmentCodes;
$originalDepartments = $userDepartmentCodes;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $full_name = trim($_POST['full_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $mobile1 = trim($_POST['mobile1'] ?? '');
    $mobile2 = trim($_POST['mobile2'] ?? '');
    $position = trim($_POST['position'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $role_id = (int)($_POST['role_id'] ?? 0);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    $provinceValue = $_POST['province_id'] ?? '';
    $regionValue = $_POST['region_id'] ?? '';
    $province_id = $provinceValue === '' ? null : (int)$provinceValue;
    $region_id = $regionValue === '' ? null : (int)$regionValue;
    $departmentInput = array_map('strval', (array)($_POST['department_codes'] ?? []));
    $inputDepartmentCodes = array_values(array_unique($departmentInput));

    $role_name = $roleLabels[$role_id] ?? '';
    if ($role_name === '') {
        $error = "Rôle sélectionné invalide.";
    }
    $normalizedRole = mbh_normalize_role_name($role_name);
    $autoDepartments = mbh_get_role_departments($normalizedRole);
    if (!empty($autoDepartments)) {
        $inputDepartmentCodes = $autoDepartments;
    }
    $requiresProvince = mbh_role_requires_depot($role_name);
    $requiresRegion = mbh_role_requires_region($role_name);
    if (!$requiresProvince) {
        $province_id = null;
    }
    if (!$requiresRegion) {
        $region_id = null;
    }

    $departmentOptionalRoles = ['DG', 'DGA', 'Admin', 'Board'];

    if ($username && $full_name && $email && $role_id) {
        $conn = get_db_connection();
        $stmt = $conn->prepare("SELECT id FROM user WHERE (username = ? OR email = ?) AND id != ?");
        $stmt->bind_param("ssi", $username, $email, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $error = "Nom d'utilisateur ou email déjà existant.";
        } else {
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $error = "Adresse email invalide.";
            } elseif ($requiresProvince && $province_id === null) {
                $error = "Veuillez sélectionner un dépôt pour ce rôle.";
            } elseif ($province_id !== null && !in_array($province_id, array_map('intval', array_column($provinces, 'id')), true)) {
                $error = "Dépôt sélectionné invalide.";
            } elseif ($requiresRegion && $region_id === null) {
                $error = "Veuillez sélectionner une région pour ce rôle.";
            } elseif ($region_id !== null && !in_array($region_id, array_map('intval', array_column($regions, 'id')), true)) {
                $error = "Région sélectionnée invalide.";
            } elseif (!empty($inputDepartmentCodes)) {
                foreach ($inputDepartmentCodes as $code) {
                    if (!in_array($code, $departmentCodesAvailable, true)) {
                        $error = "Département sélectionné invalide.";
                        break;
                    }
                }
            }

            if (!$error && empty($inputDepartmentCodes) && !in_array($normalizedRole, $departmentOptionalRoles, true)) {
                $error = "Veuillez sélectionner au moins un département.";
            }

            if (!$error) {
                $stmt = $conn->prepare("UPDATE user SET username = ?, full_name = ?, email = ?, mobile1 = ?, mobile2 = ?, position = ?, role_id = ?, province_id = NULLIF(?, -1), region_id = NULLIF(?, -1), address = ?, is_active = ? WHERE id = ?");
                $provinceParam = $province_id ?? -1;
                $regionParam = $region_id ?? -1;
                $stmt->bind_param("ssssssiiisii", $username, $full_name, $email, $mobile1, $mobile2, $position, $role_id, $provinceParam, $regionParam, $address, $is_active, $user_id);
                $stmt->execute();
                $stmt->close();
                mbh_sync_user_departments($conn, $user_id, $inputDepartmentCodes);
                $conn->close();
                log_action($_SESSION['user_id'], 'edit_user', "Edited user ID $user_id (Active: " . ($is_active ? 'Yes' : 'No') . ")");

                $changes = [];
                $fieldsToTrack = [
                    'username' => ['label' => 'Nom utilisateur', 'old' => $originalUser['username'] ?? '', 'new' => $username],
                    'full_name' => ['label' => 'Nom complet', 'old' => $originalUser['full_name'] ?? '', 'new' => $full_name],
                    'email' => ['label' => 'Email', 'old' => $originalUser['email'] ?? '', 'new' => $email],
                    'mobile1' => ['label' => 'Téléphone principal', 'old' => $originalUser['mobile1'] ?? '', 'new' => $mobile1],
                    'mobile2' => ['label' => 'Téléphone secondaire', 'old' => $originalUser['mobile2'] ?? '', 'new' => $mobile2],
                    'position' => ['label' => 'Fonction', 'old' => $originalUser['position'] ?? '', 'new' => $position],
                    'address' => ['label' => 'Adresse', 'old' => $originalUser['address'] ?? '', 'new' => $address],
                ];
                foreach ($fieldsToTrack as $meta) {
                    if ((string)$meta['old'] !== (string)$meta['new']) {
                        $changes[] = [
                            'champ' => $meta['label'],
                            'ancien' => $meta['old'],
                            'nouveau' => $meta['new'],
                        ];
                    }
                }
                if ((int)($originalUser['role_id'] ?? 0) !== $role_id) {
                    $changes[] = [
                        'champ' => 'Rôle',
                        'ancien' => $roleLabels[$originalUser['role_id']] ?? ($originalUser['role_id'] ?? ''),
                        'nouveau' => $roleLabels[$role_id] ?? $role_id,
                    ];
                }
                if ((int)($originalUser['is_active'] ?? 0) !== $is_active) {
                    $changes[] = [
                        'champ' => 'Statut',
                        'ancien' => ($originalUser['is_active'] ?? 0) ? 'Actif' : 'Inactif',
                        'nouveau' => $is_active ? 'Actif' : 'Inactif',
                    ];
                }
                if ((int)($originalUser['province_id'] ?? -1) !== (int)($province_id ?? -1)) {
                    $changes[] = [
                        'champ' => 'Dépôt',
                        'ancien' => get_province_name($originalUser['province_id'] ?? null),
                        'nouveau' => get_province_name($province_id),
                    ];
                }
                if ((int)($originalUser['region_id'] ?? -1) !== (int)($region_id ?? -1)) {
                    $changes[] = [
                        'champ' => 'Région',
                        'ancien' => get_region_name($originalUser['region_id'] ?? null),
                        'nouveau' => get_region_name($region_id),
                    ];
                }
                if ($originalDepartments !== $inputDepartmentCodes) {
                    $changes[] = [
                        'champ' => 'Départements',
                        'ancien' => implode(', ', $originalDepartments),
                        'nouveau' => implode(', ', $inputDepartmentCodes),
                    ];
                }
                if (!empty($changes)) {
                    record_user_change($user_id, 'edit', (int)$_SESSION['user_id'], $changes);
                }
                header("Location: /masunzu_bar_hotel/modules/users/manage.php");
                exit;
            }
        }
        if (isset($stmt) && $stmt instanceof mysqli_stmt) {
            $stmt->close();
        }
        if (isset($conn) && $conn instanceof mysqli) {
            $conn->close();
        }
    } else {
        $error = "Tous les champs obligatoires doivent être remplis.";
    }

    if (isset($error)) {
        $user['username'] = $username;
        $user['full_name'] = $full_name;
        $user['email'] = $email;
        $user['mobile1'] = $mobile1;
        $user['mobile2'] = $mobile2;
        $user['position'] = $position;
        $user['address'] = $address;
        $user['role_id'] = $role_id;
        $user['is_active'] = $is_active;
        $user['province_id'] = $province_id;
        $user['region_id'] = $region_id;
    }
}

function get_province_name($id) {
    static $cache = [];
    if ($id === null) {
        return 'N/A';
    }
    if (isset($cache[$id])) {
        return $cache[$id];
    }
    $conn = get_db_connection();
    $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $name = $row['name'] ?? 'N/A';
    $cache[$id] = $name;
    $stmt->close();
    $conn->close();
    return $name;
}

function get_region_name($id) {
    static $cache = [];
    if ($id === null) {
        return 'N/A';
    }
    if (isset($cache[$id])) {
        return $cache[$id];
    }
    $conn = get_db_connection();
    $stmt = $conn->prepare("SELECT name FROM region WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $name = $row['name'] ?? 'N/A';
    $cache[$id] = $name;
    $stmt->close();
    $conn->close();
    return $name;
}
// Preserve the loaded user details because header.php fetches the current session user into $user.
$selectedUser = $user;
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
$user = $selectedUser;
?>
<div class="form-container" style="padding: 20px; background-color: #F4F0E4; min-height: 70vh;">
    <h1 style="color: #4B2F1F; font-size: 28px; font-weight: bold; margin-bottom: 20px;">Modifier l'utilisateur</h1>
    <div class="form-content" style="background-color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.1); max-width: 500px; margin: 0 auto;">
        <?php if (isset($error)): ?>
            <p style="color: red; font-size: 0.9em; margin-bottom: 10px;"><?php echo htmlspecialchars($error); ?></p>
        <?php endif; ?>
        <form method="POST">
            <div style="margin-bottom: 15px;">
                <label for="username" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Nom d'utilisateur <span style="color: red;">*</span></label>
                <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($user['username'] ?? ''); ?>" required style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="full_name" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Nom complet <span style="color: red;">*</span></label>
                <input type="text" id="full_name" name="full_name" value="<?php echo htmlspecialchars($user['full_name'] ?? ''); ?>" required style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="email" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Email <span style="color: red;">*</span></label>
                <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($user['email'] ?? ''); ?>" required style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="mobile1" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Téléphone principal</label>
                <input type="tel" id="mobile1" name="mobile1" value="<?php echo htmlspecialchars($user['mobile1'] ?? ''); ?>" placeholder="Ex: +257 79 123 456" style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="mobile2" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Téléphone secondaire</label>
                <input type="tel" id="mobile2" name="mobile2" value="<?php echo htmlspecialchars($user['mobile2'] ?? ''); ?>" placeholder="Ex: +257 79 654 321" style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="position" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Fonction</label>
                <input type="text" id="position" name="position" value="<?php echo htmlspecialchars($user['position'] ?? ''); ?>" style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="address" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Adresse</label>
                <input type="text" id="address" name="address" value="<?php echo htmlspecialchars($user['address'] ?? ''); ?>" placeholder="Adresse (optionnel)" style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="role_id" style="display: block; color: #4B2F1F; font-weight: bold; margin-bottom: 5px;">Rôle <span style="color: red;">*</span></label>
                <select name="role_id" id="role_id" required onchange="toggleAssignments()"
                        style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px; box-sizing: border-box;">
                    <option value=""><?php echo htmlspecialchars('Sélectionnez un rôle'); ?></option>
                <?php foreach ($roles as $role): ?>
                    <?php
                        $roleName = $role['name'];
                        $requiresProvinceAttr = mbh_role_requires_depot($roleName) ? '1' : '0';
                        $requiresRegionAttr = mbh_role_requires_region($roleName) ? '1' : '0';
                        $autoDepartmentsAttr = implode(',', mbh_get_role_departments(mbh_normalize_role_name($roleName)));
                    ?>
                    <option value="<?php echo $role['id']; ?>"
                            data-requires-province="<?php echo $requiresProvinceAttr; ?>"
                            data-requires-region="<?php echo $requiresRegionAttr; ?>"
                            data-auto-departments="<?php echo htmlspecialchars($autoDepartmentsAttr, ENT_QUOTES, 'UTF-8'); ?>"
                            <?php echo $role['id'] == ($user['role_id'] ?? 0) ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($roleName); ?>
                    </option>
                <?php endforeach; ?>
                </select>
            </div>

            <div id="province_section" style="display:none; margin:10px 0;">
                <label for="depot_search" style="display:block; margin-bottom:6px; font-weight:600; color:#4B2F1F;">Dépôt assigné</label>
                <?php if (empty($provinces)): ?>
                    <p style="color:#B00020; font-size:0.9em;">Aucun dépôt disponible.</p>
                <?php else: ?>
                    <input type="text" id="depot_search" placeholder="Tapez pour filtrer les dépôts"
                           style="width:100%; padding:10px; border:2px solid #4B2F1F; border-radius:8px; margin-bottom:8px;">
                    <select name="province_id" id="province_id"
                            style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">
                        <option value=""><?php echo htmlspecialchars('Sélectionnez un dépôt'); ?></option>
                        <?php foreach ($provinces as $prov): ?>
                            <option value="<?php echo $prov['id']; ?>" <?php echo (($user['province_id'] ?? '') == $prov['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($prov['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                <?php endif; ?>
            </div>

            <div id="region_section" style="display:none; margin:10px 0;">
                <label for="region_id" style="display:block; margin-bottom:6px; font-weight:600; color:#4B2F1F;">Région</label>
                <select name="region_id" id="region_id"
                        style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">
                    <option value=""><?php echo htmlspecialchars('Sélectionnez une région'); ?></option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo $region['id']; ?>" <?php echo (($user['region_id'] ?? '') == $region['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div id="department_section" style="margin:15px 0;">
                <p style="margin:0 0 8px; font-weight:600; color:#4B2F1F;">Départements assignés</p>
                <?php foreach ($departments as $department): ?>
                    <?php
                        $code = $department['code'];
                        $checked = in_array($code, $inputDepartmentCodes, true);
                    ?>
                    <label style="display:block; margin-bottom:6px; color:#4B2F1F;">
                        <input type="checkbox" class="department-checkbox" name="department_codes[]" value="<?php echo htmlspecialchars($code); ?>"
                               <?php echo $checked ? 'checked' : ''; ?>>
                        <?php echo htmlspecialchars($department['label']); ?>
                    </label>
                <?php endforeach; ?>
                <div id="department_hidden_inputs"></div>
            </div>

            <div style="margin-bottom: 20px;">
                <label style="display: flex; align-items: center; color: #4B2F1F; font-weight: bold; cursor: pointer;">
                    <input type="checkbox" name="is_active" <?php echo ($user['is_active'] ?? 0) ? 'checked' : ''; ?> style="margin-right: 10px; width: 18px; height: 18px;">
                    Utilisateur actif
                </label>
            </div>
            <button type="submit" style="background-color: #F4A261; color: #000000; padding: 12px; border: none; border-radius: 8px; cursor: pointer; width: 100%; font-size: 16px; font-weight: bold;">Enregistrer les modifications</button>
        </form>
</div>
</div>

<script>
function toggleAssignments() {
    const roleSelect = document.getElementById('role_id');
    const selectedOption = roleSelect.options[roleSelect.selectedIndex];
    const provinceSection = document.getElementById('province_section');
    const provinceSelect = document.getElementById('province_id');
    const regionSection = document.getElementById('region_section');
    const regionSelect = document.getElementById('region_id');

    const needsProvince = selectedOption && selectedOption.dataset.requiresProvince === '1';
    const needsRegion = selectedOption && selectedOption.dataset.requiresRegion === '1';

    if (provinceSection) {
        provinceSection.style.display = needsProvince ? 'block' : 'none';
        if (provinceSelect) {
            if (needsProvince) {
                provinceSelect.setAttribute('required', 'required');
            } else {
                provinceSelect.removeAttribute('required');
                provinceSelect.value = '';
            }
        }
    }

    if (regionSection) {
        regionSection.style.display = needsRegion ? 'block' : 'none';
        if (regionSelect) {
            if (needsRegion) {
                regionSelect.setAttribute('required', 'required');
            } else {
                regionSelect.removeAttribute('required');
                regionSelect.value = '';
            }
        }
    }

    const autoDepartments = selectedOption && selectedOption.dataset.autoDepartments
        ? selectedOption.dataset.autoDepartments.split(',').filter(Boolean)
        : [];
    updateDepartmentCheckboxes(autoDepartments);
}

function updateDepartmentCheckboxes(autoDepartments) {
    const checkboxes = document.querySelectorAll('.department-checkbox');
    const hiddenContainer = document.getElementById('department_hidden_inputs');
    if (hiddenContainer) {
        hiddenContainer.innerHTML = '';
    }
    if (checkboxes.length === 0) {
        return;
    }
    if (autoDepartments.length > 0) {
        checkboxes.forEach(cb => {
            const shouldCheck = autoDepartments.includes(cb.value);
            cb.checked = shouldCheck;
            cb.disabled = true;
        });
        if (hiddenContainer) {
            autoDepartments.forEach(code => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'department_codes[]';
                input.value = code;
                hiddenContainer.appendChild(input);
            });
        }
    } else {
        checkboxes.forEach(cb => {
            cb.disabled = false;
        });
    }
}

function filterDepots() {
    const searchInput = document.getElementById('depot_search');
    if (!searchInput) {
        return;
    }
    const filterValue = (searchInput.value || '').toLowerCase();
    const provinceSelect = document.getElementById('province_id');
    if (!provinceSelect) {
        return;
    }
    const options = provinceSelect.querySelectorAll('option');
    let firstVisibleValue = null;

    options.forEach(option => {
        if (option.value === '') {
            option.hidden = false;
            return;
        }
        const text = (option.text || '').toLowerCase();
        const isMatch = filterValue === '' || text.startsWith(filterValue);
        option.hidden = !isMatch;
        if (isMatch && firstVisibleValue === null) {
            firstVisibleValue = option.value;
        }
    });

    if (provinceSelect.value && provinceSelect.querySelector(`option[value="${provinceSelect.value}"]`)?.hidden) {
        provinceSelect.value = '';
    }

    if (filterValue !== '' && firstVisibleValue !== null) {
        provinceSelect.value = firstVisibleValue;
    }
}

document.addEventListener('DOMContentLoaded', () => {
    toggleAssignments();
    const depotSearch = document.getElementById('depot_search');
    if (depotSearch) {
        depotSearch.addEventListener('input', filterDepots);
    }
});
</script>

<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
