<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Stock Manager role
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in split_stock.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
if ($role !== 'Stock Manager') {
    error_log("Unauthorized access to split_stock.php by user {$_SESSION['full_name']} (Role: $role)");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}

// Fetch province_id
$province_id = $_SESSION['province_id'] ?? null;
if ($province_id === null || $province_id === '') {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $province_id = $stmt->get_result()->fetch_assoc()['province_id'] ?? null;
    $_SESSION['province_id'] = $province_id;
    $stmt->close();
}
$province_id = $_SESSION['province_id'] ?? null;
if ($province_id === null || $province_id === '') {
    error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: No province assigned. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$province_id = (int)$province_id;

// Fetch available products from province_stock
$stmt = $conn->prepare("
    SELECT ps.product_id, p.name, ps.quantity, p.crate_quantity
    FROM province_stock ps
    JOIN product p ON ps.product_id = p.id
    WHERE ps.province_id = ? AND ps.quantity > 0
    ORDER BY p.name
");
$stmt->bind_param("i", $province_id);
$stmt->execute();
$products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$errors = [];
$success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id = intval($_POST['product_id'] ?? 0);
    $wholesale_quantity = floatval($_POST['wholesale_quantity'] ?? 0);
    $retail_quantity = floatval($_POST['retail_quantity'] ?? 0);
    $unit_type = $_POST['unit_type'] ?? 'bottles';

    // Validation
    if ($product_id <= 0) {
        $errors[] = "Please select a product.";
    }
    if ($wholesale_quantity < 0 || $retail_quantity < 0) {
        $errors[] = "Quantities cannot be negative.";
    }
    if ($wholesale_quantity == 0 && $retail_quantity == 0) {
        $errors[] = "At least one quantity (wholesale or retail) must be specified.";
    }
    if (!in_array($unit_type, ['bottles', 'crates'])) {
        $errors[] = "Invalid unit type selected.";
    }

    // Fetch product details
    $stmt = $conn->prepare("
        SELECT ps.quantity, p.crate_quantity
        FROM province_stock ps
        JOIN product p ON ps.product_id = p.id
        WHERE ps.product_id = ? AND ps.province_id = ?
    ");
    $stmt->bind_param("ii", $product_id, $province_id);
    $stmt->execute();
    $product = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$product) {
        $errors[] = "Selected product is not available in this province.";
    } else {
        // Convert quantities to bottles for database operations
        $wholesale_bottles = $unit_type === 'crates' ? $wholesale_quantity * $product['crate_quantity'] : $wholesale_quantity;
        $retail_bottles = $unit_type === 'crates' ? $retail_quantity * $product['crate_quantity'] : $retail_quantity;
        $available_crates = floor($product['quantity'] / $product['crate_quantity']);

        if ($wholesale_bottles + $retail_bottles > $product['quantity']) {
            if ($unit_type === 'crates') {
                $total_crates = $wholesale_quantity + $retail_quantity;
                $errors[] = "Total allocated quantity ($total_crates crates, equivalent to " . ($wholesale_bottles + $retail_bottles) . " bottles) exceeds available stock ($available_crates crates, equivalent to {$product['quantity']} bottles).";
            } else {
                $errors[] = "Total allocated quantity ($wholesale_bottles + $retail_bottles bottles) exceeds available stock ({$product['quantity']} bottles, equivalent to $available_crates crates).";
            }
        } elseif ($unit_type === 'crates' && (
            ($wholesale_quantity != floor($wholesale_quantity)) ||
            ($retail_quantity != floor($retail_quantity))
        )) {
            $errors[] = "Quantities must be whole crates when splitting by crates.";
        } elseif ($unit_type === 'bottles' && $product['crate_quantity'] && (
            ($wholesale_bottles % ($product['crate_quantity'] / 2) != 0) ||
            ($retail_bottles % ($product['crate_quantity'] / 2) != 0)
        )) {
            $errors[] = "Bottle quantities must be in whole or half crates (e.g., multiples of {$product['crate_quantity']} or " . ($product['crate_quantity'] / 2) . " bottles).";
        }
    }

    if (empty($errors)) {
        $conn->begin_transaction();
        try {
            // Update province_stock
            $total_bottles = $wholesale_bottles + $retail_bottles;
            $stmt = $conn->prepare("
                UPDATE province_stock
                SET quantity = quantity - ?, last_updated = NOW()
                WHERE product_id = ? AND province_id = ?
            ");
            $stmt->bind_param("iii", $total_bottles, $product_id, $province_id);
            if (!$stmt->execute() || $stmt->affected_rows === 0) {
                throw new Exception("Failed to update province stock for product ID $product_id.");
            }
            $stmt->close();

            // Update or insert wholesale_stock
            if ($wholesale_bottles > 0) {
                $stmt = $conn->prepare("
                    INSERT INTO wholesale_stock (province_id, product_id, quantity, last_updated)
                    VALUES (?, ?, ?, NOW())
                    ON DUPLICATE KEY UPDATE quantity = quantity + ?, last_updated = NOW()
                ");
                $stmt->bind_param("iiii", $province_id, $product_id, $wholesale_bottles, $wholesale_bottles);
                if (!$stmt->execute()) {
                    throw new Exception("Failed to update wholesale stock: " . $stmt->error);
                }
                $stmt->close();
            }

            // Update or insert retail_stock
            if ($retail_bottles > 0) {
                $stmt = $conn->prepare("
                    INSERT INTO retail_stock (province_id, product_id, quantity, last_updated)
                    VALUES (?, ?, ?, NOW())
                    ON DUPLICATE KEY UPDATE quantity = quantity + ?, last_updated = NOW()
                ");
                $stmt->bind_param("iiii", $province_id, $product_id, $retail_bottles, $retail_bottles);
                if (!$stmt->execute()) {
                    throw new Exception("Failed to update retail stock: " . $stmt->error);
                }
                $stmt->close();
            }

            // Fetch product name for logging
            $stmt = $conn->prepare("SELECT name FROM product WHERE id = ?");
            $stmt->bind_param("i", $product_id);
            $stmt->execute();
            $product_name = $stmt->get_result()->fetch_assoc()['name'] ?? 'Unknown';
            $stmt->close();

            $log_unit = $unit_type === 'crates' ? "crates (converted to $wholesale_bottles bottles)" : "bottles";
            $retail_log_unit = $unit_type === 'crates' ? "crates (converted to $retail_bottles bottles)" : "bottles";
            log_action($_SESSION['user_id'], 'split_stock', "Split stock for product $product_name (ID $product_id): $wholesale_quantity $log_unit to wholesale, $retail_quantity $retail_log_unit to retail in province $province_id");
            $conn->commit();
            $success = "Stock split successfully for $product_name: $wholesale_quantity $log_unit to wholesale, $retail_quantity $retail_log_unit to retail.";
        } catch (Exception $e) {
            $conn->rollback();
            $errors[] = "Failed to split stock: " . $e->getMessage();
            error_log("Stock split error for product_id=$product_id, province_id=$province_id: " . $e->getMessage());
        }
    }
}

include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<div class="container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; display: flex; gap: 20px;">
    <!-- Stock Sidebar -->
    <div class="stock-sidebar" style="flex: 0.5; background-color: #F4F0E4; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-height: 80vh; overflow-y: auto; min-width: 200px;">
        <h3 style="color: #4B2F1F; font-size: 20px; margin-bottom: 15px;">Province Stock</h3>
        <input type="text" id="stock_search" placeholder="Search stock..." style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #FFFFFF; font-size: 14px; color: #4B2F1F; margin-bottom: 15px;">
        <table id="stock_table" style="width: 100%; border-collapse: collapse;">
            <thead>
                <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                    <th style="padding: 10px; text-align: left;">Product</th>
                    <th style="padding: 10px; text-align: left;">Bottles</th>
                    <th style="padding: 10px; text-align: left;">Crates</th>
                </tr>
            </thead>
            <tbody id="stock_body">
                <?php foreach ($products as $product): ?>
                    <?php
                        $total_bottles = $product['quantity'];
                        $number_of_crates = $total_bottles > 0 ? floor($total_bottles / $product['crate_quantity']) : 0;
                        $remaining_bottles = $total_bottles % $product['crate_quantity'];
                    ?>
                    <tr style="border-bottom: 1px solid #D3D3D3;">
                        <td style="padding: 10px; color: #4B2F1F; font-size: 14px;"><?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td style="padding: 10px; color: #4B2F1F; font-size: 14px;"><?php echo number_format($total_bottles, 0, ',', '.'); ?></td>
                        <td style="padding: 10px; color: #4B2F1F; font-size: 14px;"><?php echo number_format($number_of_crates, 0, ',', '.') . ($remaining_bottles > 0 ? ' + ' . $remaining_bottles . ' bottles' : ''); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <!-- Main Content -->
    <div class="content" style="flex: 2; background-color: #F4F0E4; padding: 30px; border-radius: 12px; box-shadow: 0 6px 12px rgba(0,0,0,0.15); max-width: 800px;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 20px; font-weight: 600; text-align: center;">Split Stock</h2>
        <p style="color: #4B2F1F; font-size: 16px; text-align: center;">Allocate stock from province stock to wholesale and retail stock for Masunzu Bar Hotel.</p>
        <?php if ($success): ?>
            <div style="background-color: #FFFFFF; color: #4B2F1F; padding: 15px; border-radius: 8px; border: 1px solid #F4A261; margin-bottom: 20px; text-align: center;">
                <p style="margin: 0 0 15px 0; font-size: 16px;"><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></p>
                <button type="button" onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 20px; border: none; border-radius: 8px; font-size: 14px; cursor: pointer;">Refresh Page</button>
            </div>
        <?php endif; ?>
        <?php if (!empty($errors)): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 15px; border-radius: 8px; border: 1px solid #FF0000; margin-bottom: 20px;">
                <ul style="margin: 0; padding-left: 20px;">
                    <?php foreach ($errors as $error): ?>
                        <li style="font-size: 14px;"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
        <?php if (empty($products)): ?>
            <p style="color: #FF0000; text-align: center; font-size: 16px;">No products available in province stock to split.</p>
        <?php else: ?>
            <form method="POST" style="display: flex; flex-direction: column; gap: 15px;">
                <div>
                    <label for="product_id" style="color: #4B2F1F; font-size: 16px; font-weight: 500;">Select Product</label>
                    <select id="product_id" name="product_id" required style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                        <option value="">Select a product</option>
                        <?php foreach ($products as $product): ?>
                            <?php
                                $available_crates = floor($product['quantity'] / $product['crate_quantity']);
                                $remaining_bottles = $product['quantity'] % $product['crate_quantity'];
                            ?>
                            <option value="<?php echo htmlspecialchars($product['product_id'], ENT_QUOTES, 'UTF-8'); ?>">
                                <?php echo htmlspecialchars($product['name'] . ' (' . $product['quantity'] . ' bottles, ' . $available_crates . ' crates' . ($remaining_bottles > 0 ? ' + ' . $remaining_bottles . ' bottles' : '') . ', ' . $product['crate_quantity'] . ' bottles/crate)', ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <label for="unit_type" style="color: #4B2F1F; font-size: 16px; font-weight: 500;">Unit Type</label>
                    <select id="unit_type" name="unit_type" required style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                        <option value="bottles">Bottles</option>
                        <option value="crates">Crates</option>
                    </select>
                </div>
                <div>
                    <label for="wholesale_quantity" style="color: #4B2F1F; font-size: 16px; font-weight: 500;">Wholesale Quantity</label>
                    <input type="number" id="wholesale_quantity" name="wholesale_quantity" min="0" step="1" value="0" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;" placeholder="Enter wholesale quantity">
                </div>
                <div>
                    <label for="retail_quantity" style="color: #4B2F1F; font-size: 16px; font-weight: 500;">Retail Quantity</label>
                    <input type="number" id="retail_quantity" name="retail_quantity" min="0" step="1" value="0" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;" placeholder="Enter retail quantity">
                </div>
                <button type="submit" style="background-color: #4B2F1F; color: #F4F0E4; padding: 15px; border: none; border-radius: 8px; font-size: 16px; font-weight: 500; cursor: pointer; transition: all 0.3s ease; box-shadow: 0 2px 4px rgba(0,0,0,0.2);">
                    Split Stock
                </button>
            </form>
        <?php endif; ?>
        <a href="/masunzu_bar_hotel/dashboards/stock_manager_dashboard.php" style="display: inline-block; color: #F4A261; text-decoration: none; font-size: 16px; margin-top: 15px;">Back to Dashboard</a>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Search functionality for stock table
        const stockSearch = document.getElementById('stock_search');
        const stockBody = document.getElementById('stock_body');
        if (stockSearch && stockBody) {
            stockSearch.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase();
                const rows = stockBody.getElementsByTagName('tr');
                for (let i = 0; i < rows.length; i++) {
                    const text = rows[i].textContent.toLowerCase();
                    rows[i].style.display = text.includes(searchTerm) ? '' : 'none';
                }
            });
        }

        // Update input step based on unit type
        const unitTypeSelect = document.getElementById('unit_type');
        const wholesaleInput = document.getElementById('wholesale_quantity');
        const retailInput = document.getElementById('retail_quantity');
        if (unitTypeSelect && wholesaleInput && retailInput) {
            unitTypeSelect.addEventListener('change', function() {
                if (this.value === 'crates') {
                    wholesaleInput.step = '1';
                    retailInput.step = '1';
                } else {
                    wholesaleInput.step = 'any';
                    retailInput.step = 'any';
                }
            });
        }
    });
</script>

<style>
    button:hover {
        background-color: #F4A261;
        color: #4B2F1F;
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.3);
    }
    input:focus, select:focus {
        outline: none;
        border-color: #F4A261;
        box-shadow: 0 0 5px rgba(244, 162, 97, 0.5);
    }
    @media (max-width: 768px) {
        .container { padding: 15px; flex-direction: column; }
        .stock-sidebar, .content { width: 100%; max-width: 100%; }
        .stock-sidebar { margin-bottom: 20px; }
        h2 { font-size: 20px; }
        label, select, input, button, a { font-size: 14px; }
    }
    @media (max-width: 480px) {
        h2 { font-size: 18px; }
        label, select, input, button, a { font-size: 12px; }
        .stock-sidebar { min-width: 100%; }
    }
</style>

<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
<?php
if ($conn && !$conn->connect_error) {
    $conn->close();
}
?>
