<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to DGA/Leadership roles
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in initiate_transfer.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
if (!in_array($role, ['DG', 'DGA', 'Board', 'Admin'], true)) {
    error_log("Unauthorized access to initiate_transfer.php by user {$_SESSION['full_name']} (Role: $role)");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}

ensure_region_stock_table($conn);

// Fetch products with available stock for dropdown and table (only crates, no bottles)
// Use SUM in case there are multiple rows (shouldn't happen but safety check)
$stmt = $conn->prepare("
    SELECT p.id, p.name, p.crate_quantity, p.price_per_crate, COALESCE(SUM(ms.quantity), 0) AS available_quantity
    FROM product p
    LEFT JOIN main_stock ms ON p.id = ms.product_id AND ms.unit_type = 'crates'
    GROUP BY p.id, p.name, p.crate_quantity, p.price_per_crate
    ORDER BY p.name
");
$stmt->execute();
$products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch available Operation Managers grouped by region
$stmt = $conn->prepare("
    SELECT u.id, u.full_name, u.region_id, COALESCE(r.name, 'Non assignée') AS region_name
    FROM user u
    JOIN role ro ON ro.id = u.role_id
    LEFT JOIN region r ON u.region_id = r.id
    WHERE ro.name = 'Operation Manager' AND u.is_active = 1
    ORDER BY region_name, u.full_name
");
$stmt->execute();
$operation_managers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch available Stock Managers grouped by province
$stmt = $conn->prepare("
    SELECT u.id, u.full_name, u.province_id, COALESCE(p.name, 'Province non assignée') AS province_name,
           p.region_id, COALESCE(r.name, 'Région non assignée') AS region_name
    FROM user u
    JOIN role ro ON ro.id = u.role_id
    LEFT JOIN province p ON u.province_id = p.id
    LEFT JOIN region r ON p.region_id = r.id
    WHERE ro.name = 'Stock Manager' AND u.is_active = 1
    ORDER BY province_name, u.full_name
");
$stmt->execute();
$stock_managers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$has_recipients = !empty($operation_managers) || !empty($stock_managers);

$errors = [];
$success = '';

if (isset($_SESSION['transfer_success'])) {
    $success = $_SESSION['transfer_success'];
    unset($_SESSION['transfer_success']);
}

$recipient_selection = '';
$selected_recipient = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_ids = $_POST['product_id'] ?? [];
    $quantities = $_POST['quantity'] ?? [];
    $recipient_selection = trim($_POST['recipient_id'] ?? '');
    $estimated_arrival_input = trim($_POST['estimated_arrival'] ?? '');
    $estimated_arrival = $estimated_arrival_input ?: date('Y-m-d\TH:i');
    $driver_name = trim($_POST['driver_name'] ?? '');
    $driver_mobile = trim($_POST['driver_mobile'] ?? '');
    $plate_number = trim($_POST['plate_number'] ?? '');

    // Validate recipient selection (Operation Manager or Stock Manager)
    if (empty($recipient_selection) || strpos($recipient_selection, '|') === false) {
        $errors[] = "Veuillez sélectionner un destinataire valide.";
    } else {
        [$recipient_type, $recipient_id_raw] = explode('|', $recipient_selection, 2);
        $recipient_type = trim($recipient_type);
        $recipient_user_id = (int)$recipient_id_raw;

        if ($recipient_user_id <= 0 || !in_array($recipient_type, ['operation_manager', 'stock_manager'], true)) {
            $errors[] = "Destinataire sélectionné invalide.";
        } else {
            if ($recipient_type === 'operation_manager') {
                $stmt = $conn->prepare("
                    SELECT u.id, u.region_id, u.full_name, COALESCE(r.name, 'Non assignée') AS region_name
                    FROM user u
                    JOIN role ro ON ro.id = u.role_id
                    LEFT JOIN region r ON u.region_id = r.id
                    WHERE u.id = ? AND ro.name = 'Operation Manager'
                ");
                $stmt->bind_param("i", $recipient_user_id);
                $stmt->execute();
                $recipient_data = $stmt->get_result()->fetch_assoc();
                $stmt->close();

                if (!$recipient_data) {
                    $errors[] = "Chargé des opérations sélectionné invalide.";
                } elseif (empty($recipient_data['region_id'])) {
                    $errors[] = "Le chargé des opérations sélectionné n'a pas de région assignée.";
                } else {
                    $selected_recipient = [
                        'type' => 'operation_manager',
                        'id' => $recipient_data['id'],
                        'full_name' => $recipient_data['full_name'],
                        'region_id' => (int)$recipient_data['region_id'],
                        'region_name' => $recipient_data['region_name'] ?? 'Non assignée',
                        'province_id' => null,
                        'province_name' => null
                    ];
                }
            } else {
                $stmt = $conn->prepare("
                    SELECT u.id, u.province_id, u.full_name, COALESCE(p.name, 'Province non assignée') AS province_name,
                           p.region_id, COALESCE(r.name, 'Région non assignée') AS region_name
                    FROM user u
                    JOIN role ro ON ro.id = u.role_id
                    LEFT JOIN province p ON u.province_id = p.id
                    LEFT JOIN region r ON p.region_id = r.id
                    WHERE u.id = ? AND ro.name = 'Stock Manager'
                ");
                $stmt->bind_param("i", $recipient_user_id);
                $stmt->execute();
                $recipient_data = $stmt->get_result()->fetch_assoc();
                $stmt->close();

                if (!$recipient_data) {
                    $errors[] = "Gestionnaire de stock sélectionné invalide.";
                } elseif (empty($recipient_data['province_id'])) {
                    $errors[] = "Le gestionnaire de stock sélectionné n'a pas de province assignée.";
                } else {
                    $selected_recipient = [
                        'type' => 'stock_manager',
                        'id' => $recipient_data['id'],
                        'full_name' => $recipient_data['full_name'],
                        'province_id' => (int)$recipient_data['province_id'],
                        'province_name' => $recipient_data['province_name'] ?? 'Province non assignée',
                        'region_id' => isset($recipient_data['region_id']) ? (int)$recipient_data['region_id'] : null,
                        'region_name' => $recipient_data['region_name'] ?? 'Région non assignée'
                    ];
                }
            }
        }
    }

    if (!$selected_recipient) {
        $errors[] = "Veuillez sélectionner un destinataire valide avant de poursuivre.";
    }



    // Validate products and quantities
    $total_amount = 0;
    $total_products = 0;
    $product_details = [];
    if (empty($product_ids) || count($product_ids) !== count($quantities)) {
        $errors[] = "Données de produit ou de quantité invalides.";
    } else {
        $seen_products = [];
        foreach ($product_ids as $index => $product_id) {
            $product_id = intval($product_id);
            $quantity = (int)($quantities[$index] ?? 0);

            if (in_array($product_id, $seen_products)) {
                $errors[] = "L'ID produit $product_id apparaît plus d'une fois dans le transfert. Veuillez combiner les quantités ou supprimer les doublons.";
                continue;
            }
            $seen_products[] = $product_id;

            // Fetch product details (only crates, no bottles)
            $stmt = $conn->prepare("
                SELECT p.name, p.crate_quantity, p.price_per_crate, COALESCE(ms.quantity, 0) AS available_quantity
                FROM product p
                LEFT JOIN main_stock ms ON p.id = ms.product_id AND ms.unit_type = 'crates'
                WHERE p.id = ?
            ");
            $stmt->bind_param("i", $product_id);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows === 0) {
                $errors[] = "Produit sélectionné invalide (ID: $product_id).";
                continue;
            }
            $product = $result->fetch_assoc();
            $stmt->close();

            $product_name = $product['name'];
            $crate_quantity = (int)$product['crate_quantity'];
            $price_per_crate = (int)$product['price_per_crate'];
            $available_quantity = (int)$product['available_quantity'];

            if ($quantity <= 0) {
                $errors[] = "La quantité pour $product_name doit être un nombre positif.";
                continue;
            }
            if ($quantity > $available_quantity) {
                $errors[] = "La quantité ($quantity caisses) pour $product_name dépasse le stock disponible ($available_quantity caisses).";
                continue;
            }

            $product_details[] = [
                'product_id' => $product_id,
                'quantity' => $quantity,
                'price_per_crate' => $price_per_crate,
                'product_name' => $product_name,
                'crate_quantity' => $crate_quantity
            ];
            $total_amount += $quantity * $price_per_crate;
            $total_products += $quantity;
        }
    }

    if (empty($product_details)) {
        $errors[] = "Aucun produit valide sélectionné pour le transfert.";
    }

    if (empty($errors)) {
            $shipment_id = mt_rand(100000, 999999); // Generate unique shipment_id
            $conn->begin_transaction();
            try {
                // Deduct from main_stock for each product (crates only, ensure unit_type is crates)
                foreach ($product_details as $detail) {
                    $stmt = $conn->prepare("
                    UPDATE main_stock
                    SET quantity = quantity - ?
                    WHERE product_id = ? AND unit_type = 'crates' AND quantity >= ?
                ");
                $stmt->bind_param("iii", $detail['quantity'], $detail['product_id'], $detail['quantity']);
                if (!$stmt->execute() || $stmt->affected_rows !== 1) {
                    throw new Exception("Échec de la mise à jour du stock principal pour l'ID produit {$detail['product_id']} ou stock insuffisant.");
                }
                $stmt->close();
            }

            // Insert into stock_transfer for each product
            $stmt = $conn->prepare("
                INSERT INTO stock_transfer (
                    product_id, province_id, region_id, quantity, unit_type,
                    operation_manager_id, operation_receiver_id, stock_manager_id, initiator_id,
                    status, sent_at, estimated_arrival, driver_name, driver_mobile, plate_number, shipment_id,
                    total_amount, total_products, main_stock_id, province_stock_id
                ) VALUES (?, ?, ?, ?, 'crates', ?, ?, ?, ?, 'pending', NOW(), ?, ?, ?, ?, ?, ?, ?, NULL, NULL)
            ");
            foreach ($product_details as $detail) {
                $province_id = ($selected_recipient['type'] === 'stock_manager') ? (int)$selected_recipient['province_id'] : null;
                $region_id = isset($selected_recipient['region_id']) ? (int)$selected_recipient['region_id'] : null;
                $operation_receiver_id = ($selected_recipient['type'] === 'operation_manager') ? (int)$selected_recipient['id'] : null;
                $destination_stock_manager = ($selected_recipient['type'] === 'stock_manager') ? (int)$selected_recipient['id'] : null;
                $product_id = $detail['product_id'];
                $quantity = $detail['quantity'];

                $paramTypes = str_repeat('i', 8) . 'ssssiii';
                $stmt->bind_param(
                    $paramTypes,
                    $product_id,
                    $province_id,
                    $region_id,
                    $quantity,
                    $_SESSION['user_id'],
                    $operation_receiver_id,
                    $destination_stock_manager,
                    $_SESSION['user_id'],
                    $estimated_arrival,
                    $driver_name,
                    $driver_mobile,
                    $plate_number,
                    $shipment_id,
                    $total_amount,
                    $total_products
                );
                if (!$stmt->execute()) {
                    throw new Exception("Échec de l'insertion du transfert de stock pour l'ID produit {$detail['product_id']}: " . $stmt->error);
                }
            }
            $stmt->close();

            // Log action with details
            if ($selected_recipient) {
                if ($selected_recipient['type'] === 'stock_manager') {
                    $destination_label = "gestionnaire de stock " . $selected_recipient['full_name'] . " (" . ($selected_recipient['province_name'] ?? 'Province inconnue') . ")";
                } else {
                    $destination_label = "chargé des opérations " . $selected_recipient['full_name'] . " (" . ($selected_recipient['region_name'] ?? 'Région inconnue') . ")";
                }
            } else {
                $destination_label = 'destinataire';
            }
            $log_description = "Transfert de stock initié avec l'ID d'expédition $shipment_id pour $total_products caisses vers $destination_label : ";
            foreach ($product_details as $detail) {
                $formatted_quantity = format_quantity($detail['quantity']);
                $log_description .= "{$detail['product_name']} ($formatted_quantity), ";
            }
            $log_description = rtrim($log_description, ', ');
            log_action($_SESSION['user_id'], 'initiate_transfer', $log_description);
            $conn->commit();

            // Notify the selected recipient
            if ($selected_recipient) {
                if ($selected_recipient['type'] === 'stock_manager') {
                    $message = sprintf(
                        "Nouveau transfert #%d vers votre province %s. Merci de confirmer la réception.",
                        $shipment_id,
                        $selected_recipient['province_name'] ?? ''
                    );
                    $link = "/masunzu_bar_hotel/modules/operation/receive_stock.php";
                } else {
                    $message = sprintf(
                        "Nouveau transfert #%d vers votre région %s. Merci de confirmer ou rejeter la réception.",
                        $shipment_id,
                        $selected_recipient['region_name'] ?? ''
                    );
                    $link = "/masunzu_bar_hotel/modules/operation/receive_stock.php";
                }
                mbh_notify_user($conn, (int)$selected_recipient['id'], $message, $link);
            }

            $_SESSION['transfer_success'] = "Transfert de stock initié avec succès pour $total_products caisses.";
            $conn->close();
            header("Location: /masunzu_bar_hotel/modules/operation/initiate_transfer.php");
            exit;
        } catch (Exception $e) {
            $conn->rollback();
            $errors[] = "Échec de l'initiation du transfert de stock : " . $e->getMessage();
            error_log("Stock transfer error: " . $e->getMessage());
        }
    }
}

$conn->close();
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
</head>
<body>
    <div class="container" style="padding: 0; margin: 0; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; box-sizing: border-box; width: 100vw; max-width: 100vw; overflow-x: hidden; position: relative;">
        <div style="display: grid; grid-template-columns: 455px 1fr; gap: 15px; max-width: 100vw; margin: 0; padding: 20px 20px 20px 20px; box-sizing: border-box;">
            <!-- Stock Table -->
            <div style="grid-column: 1 / 2;">
                <h3 style="color: #F4F0E4; font-size: 18px; margin-bottom: 12px; font-weight: 500;">Stock Principal</h3>
                <div style="margin-bottom: 12px;">
                    <input type="text" id="stockSearch" placeholder="Rechercher des produits..." style="width: 100%; padding: 8px; border: 1px solid #F4A261; border-radius: 6px; font-size: 13px; background-color: #FFFFFF;">
                </div>
                <div style="overflow-x: auto;">
                    <table id="stockTable" style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 6px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.15);">
                        <thead>
                            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                                <th style="padding: 12px; text-align: left; font-size: 14px; font-weight: 500;">Produit</th>
                                <th style="padding: 12px; text-align: left; font-size: 14px; font-weight: 500;">Quantité (caisses)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($products as $product): ?>
                                <?php 
                                $available_crates = (int)$product['available_quantity'];
                                ?>
                                <tr style="border-bottom: 1px solid #F0F0F0; transition: background-color 0.2s;">
                                    <td style="padding: 10px 12px; text-align: left; font-size: 13px;" data-product-name="<?php echo htmlspecialchars(strtolower($product['name']), ENT_QUOTES, 'UTF-8'); ?>">
                                        <?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 10px 12px; text-align: left; font-size: 13px;">
                                        <?php echo number_format($available_crates, 0); ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <!-- Form -->
            <div style="grid-column: 2 / -1; margin-right: 0; padding-right: 0; position: relative;">
                <div class="content" style="background-color: #F4F0E4; padding: 25px; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); margin-right: 0; width: 100%; box-sizing: border-box; position: relative; right: 0;">
                    <h2 style="color: #4B2F1F; font-size: 24px; margin-bottom: 25px; font-weight: 600;">Transfert de Stock</h2>
                    <?php if ($success): ?>
                        <div style="background-color: #FFFFFF; color: #4B2F1F; padding: 15px; border-radius: 8px; border: 1px solid #F4A261; margin-bottom: 20px; text-align: center;">
                            <p style="margin: 0; font-size: 16px;"><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></p>
                        </div>
                    <?php endif; ?>
                    <div id="notification" style="display: none; background-color: #FFFFFF; color: #FF0000; padding: 15px; border-radius: 8px; border: 1px solid #FF0000; margin-bottom: 20px; text-align: center; font-size: 14px;"></div>
                    <?php if (!empty($errors)): ?>
                        <div style="background-color: #FFFFFF; color: #FF0000; padding: 15px; border-radius: 8px; border: 1px solid #FF0000; margin-bottom: 20px;">
                            <ul style="margin: 0; padding-left: 20px;">
                                <?php foreach ($errors as $error): ?>
                                    <li style="font-size: 14px;"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>
                    <?php if (!$has_recipients): ?>
                        <div style="background-color:#FFF3E0;color:#A34B00;padding:15px;border-radius:8px;border:1px solid #F4A261;margin-bottom:20px;">
                            Aucun chargé des opérations ou gestionnaire de stock actif n'est disponible. Créez-en au moins un et assignez-lui une région ou une province avant de lancer un transfert.
                        </div>
                    <?php else: ?>
                    <form method="POST" style="display: flex; flex-direction: column; gap: 18px;">
                        <div>
                            <label for="recipient_id" style="color: #4B2F1F; font-size: 13px; font-weight: 600; display: block; margin-bottom: 6px;">Destinataire (Chargé / Gestionnaire)</label>
                            <select id="recipient_id" name="recipient_id" required style="width: 100%; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px; background-color: #fff;">
                                <option value="">Sélectionner un destinataire</option>
                                <?php if (!empty($operation_managers)): ?>
                                    <optgroup label="Chargés des opérations">
                                        <?php foreach ($operation_managers as $manager): ?>
                                            <?php $option_value = 'operation_manager|' . $manager['id']; ?>
                                            <option value="<?php echo $option_value; ?>" data-region="<?php echo htmlspecialchars($manager['region_name'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" <?php echo ($recipient_selection === $option_value) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($manager['full_name'] . ' • ' . ($manager['region_name'] ?? 'Région non assignée'), ENT_QUOTES, 'UTF-8'); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php endif; ?>
                                <?php if (!empty($stock_managers)): ?>
                                    <optgroup label="Gestionnaires de stock">
                                        <?php foreach ($stock_managers as $manager): ?>
                                            <?php $option_value = 'stock_manager|' . $manager['id']; ?>
                                            <option value="<?php echo $option_value; ?>" data-province="<?php echo htmlspecialchars($manager['province_name'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" <?php echo ($recipient_selection === $option_value) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($manager['full_name'] . ' • ' . ($manager['province_name'] ?? 'Province non assignée'), ENT_QUOTES, 'UTF-8'); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php endif; ?>
                            </select>
                        </div>
                        <div>
                            <label style="color: #4B2F1F; font-size: 13px; font-weight: 600; display: block; margin-bottom: 6px;">Produits</label>
                            <div id="product-list">
                                <div class="product-entry" style="display: flex; gap: 8px; margin-bottom: 8px; flex-wrap: wrap;">
                                    <select name="product_id[]" class="product-select" required style="flex: 2; min-width: 150px; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px; background-color: #fff;">
                                        <option value="">Sélectionner un produit</option>
                                        <?php foreach ($products as $product): ?>
                                            <?php $available_crates = (int)$product['available_quantity']; ?>
                                            <option value="<?php echo $product['id']; ?>" data-available="<?php echo $available_crates; ?>">
                                                <?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?> (<?php echo $available_crates; ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <input type="number" name="quantity[]" min="1" step="1" inputmode="numeric" pattern="\d+" required style="flex: 1; min-width: 80px; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px;" placeholder="Caisses">
                                    <button type="button" class="remove-product" style="background-color: #ddd; color: #4B2F1F; padding: 9px 12px; border: none; border-radius: 6px; cursor: pointer; font-size: 13px; font-weight: 500; transition: all 0.2s;">Supprimer</button>
                                </div>
                            </div>
                            <button type="button" id="add-product" style="background-color: transparent; color: #4B2F1F; padding: 8px 0; border: 1px solid #4B2F1F; border-radius: 6px; font-size: 12px; cursor: pointer; margin-top: 8px; transition: all 0.2s; font-weight: 500; width: 100%;">+ Ajouter un Produit</button>
                        </div>
                        <div style="border-top: 1px solid #e0e0e0; padding-top: 16px;">
                            <h4 style="color: #4B2F1F; font-size: 13px; font-weight: 600; margin-bottom: 12px; margin-top: 0;">Informations du Conducteur</h4>
                            <div style="display: flex; flex-direction: column; gap: 12px;">
                                <div>
                                    <label for="estimated_arrival" style="color: #4B2F1F; font-size: 12px; font-weight: 600; display: block; margin-bottom: 4px;">Date et Heure d'Arrivée (optionnel)</label>
                                    <input type="datetime-local" id="estimated_arrival" name="estimated_arrival" style="width: 100%; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px;">
                                </div>
                                <div>
                                    <label for="driver_name" style="color: #4B2F1F; font-size: 12px; font-weight: 600; display: block; margin-bottom: 4px;">Nom du Conducteur</label>
                                    <input type="text" id="driver_name" name="driver_name" style="width: 100%; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px;">
                                </div>
                                <div>
                                    <label for="driver_mobile" style="color: #4B2F1F; font-size: 12px; font-weight: 600; display: block; margin-bottom: 4px;">Mobile (8 chiffres)</label>
                                    <input type="text" id="driver_mobile" name="driver_mobile" style="width: 100%; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px;" pattern="\d{8}">
                                </div>
                                <div>
                                    <label for="plate_number" style="color: #4B2F1F; font-size: 12px; font-weight: 600; display: block; margin-bottom: 4px;">Numéro de Plaque</label>
                                    <input type="text" id="plate_number" name="plate_number" style="width: 100%; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px;">
                                </div>
                            </div>
                        </div>
                        <button type="submit" style="background-color: #4B2F1F; color: #F4F0E4; padding: 12px; border: none; border-radius: 6px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.2s ease; margin-top: 4px;">
                            Soumettre le Transfert
                        </button>
                    </form>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <script>
            $(document).ready(function() {
                $('.product-select').select2({
                    placeholder: "Sélectionner un produit",
                    allowClear: true,
                    width: '100%'
                });

                // Helper function to show notifications
                function showNotification(message) {
                    const notif = document.getElementById('notification');
                    notif.textContent = message;
                    notif.style.display = 'block';
                    setTimeout(() => {
                        notif.style.display = 'none';
                    }, 4000);
                }

                // Re-initialize Select2 for new product entries
                document.getElementById('add-product').addEventListener('click', function() {
                    const existingSelects = document.querySelectorAll('.product-select');
                    const selectedProducts = Array.from(existingSelects).map(s => s.value).filter(v => v);
                    if (selectedProducts.length > 0 && selectedProducts.length === new Set(selectedProducts).size) {
                        // Check if any product is unselected (empty)
                        const hasUnselected = Array.from(existingSelects).some(s => !s.value);
                        if (hasUnselected) {
                            showNotification('Veuillez sélectionner des produits pour toutes les lignes avant d\'en ajouter une nouvelle.');
                            return;
                        }
                    }
                    
                    const productList = document.getElementById('product-list');
                    const newEntry = document.createElement('div');
                    newEntry.className = 'product-entry';
                    newEntry.style.cssText = 'display: flex; gap: 8px; margin-bottom: 8px; flex-wrap: wrap;';
                    newEntry.innerHTML = `
                        <select name="product_id[]" class="product-select" required style="flex: 2; min-width: 150px; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px; background-color: #fff;">
                            <option value="">Sélectionner un produit</option>
                            <?php foreach ($products as $product): ?>
                                <?php $available_crates = (int)$product['available_quantity']; ?>
                                <option value="<?php echo $product['id']; ?>" data-available="<?php echo $available_crates; ?>">
                                    <?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?> (<?php echo $available_crates; ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <input type="number" name="quantity[]" min="1" step="1" inputmode="numeric" pattern="\d+" required style="flex: 1; min-width: 80px; padding: 9px 10px; border: 1px solid #ddd; border-radius: 6px; font-size: 13px;" placeholder="Caisses">
                        <button type="button" class="remove-product" style="background-color: #ddd; color: #4B2F1F; padding: 9px 12px; border: none; border-radius: 6px; cursor: pointer; font-size: 13px; font-weight: 500; transition: all 0.2s;">Supprimer</button>
                    `;
                    productList.appendChild(newEntry);
                    $(newEntry).find('.product-select').select2({
                        placeholder: "Sélectionner un produit",
                        allowClear: true,
                        width: '100%'
                    }).on('change', function() {
                        validateNoDuplicateProducts();
                    });
                });

                // Validate no duplicate products when a product is selected
                function validateNoDuplicateProducts() {
                    const selects = document.querySelectorAll('.product-select');
                    const selected = [];
                    selects.forEach(s => {
                        if (s.value) selected.push(s.value);
                    });
                    const duplicates = selected.filter((val, idx) => selected.indexOf(val) !== idx);
                    if (duplicates.length > 0) {
                        showNotification('Le produit est déjà sélectionné. Veuillez choisir un autre produit ou supprimer le doublon.');
                        // Reset the last changed select
                        Array.from(selects).pop().value = '';
                        $(Array.from(selects).pop()).trigger('change');
                    }
                }
                
                // Attach validation to existing selects
                document.querySelectorAll('.product-select').forEach(select => {
                    $(select).on('change', validateNoDuplicateProducts);
                });

                // Remove product row
                document.addEventListener('click', function(e) {
                    if (e.target.classList.contains('remove-product')) {
                        const productEntries = document.querySelectorAll('.product-entry');
                        if (productEntries.length > 1) {
                            $(e.target.parentElement).find('.product-select').select2('destroy');
                            e.target.parentElement.remove();
                        }
                    }
                });

                // Search functionality for stock table
                document.getElementById('stockSearch').addEventListener('input', function(e) {
                    const searchTerm = e.target.value.toLowerCase();
                    const rows = document.querySelectorAll('#stockTable tbody tr');
                    rows.forEach(row => {
                        const productName = row.querySelector('td[data-product-name]').getAttribute('data-product-name');
                        row.style.display = productName.includes(searchTerm) ? '' : 'none';
                    });
                });
            });
        </script>
        <style>
            input[type="number"]::-webkit-outer-spin-button,
            input[type="number"]::-webkit-inner-spin-button {
                -webkit-appearance: none;
                margin: 0;
            }
            input[type="number"] {
                -moz-appearance: textfield;
            }
            input[type="number"]::-webkit-outer-spin-button,
            input[type="number"]::-webkit-inner-spin-button {
                -webkit-appearance: none;
                margin: 0;
            }
            input[type="number"] {
                -moz-appearance: textfield;
            }
            html, body {
                margin: 0;
                padding: 0;
                width: 100%;
                overflow-x: hidden;
            }
            body {
                background-color: #F4F0E4 !important;
            }
            .container {
                margin-right: 0 !important;
                padding-right: 0 !important;
                width: 100vw !important;
                max-width: 100vw !important;
            }
            .container > div {
                padding-right: 0 !important;
                max-width: 100vw !important;
            }
            .container > div > div[style*="grid-column: 2"] {
                margin-right: 0 !important;
                padding-right: 0 !important;
                width: 100% !important;
            }
            .content {
                margin-right: 0 !important;
                width: 100% !important;
                max-width: 100% !important;
            }
            button:hover {
                background-color: #F4A261;
                color: #4B2F1F;
                transform: translateY(-1px);
                box-shadow: 0 2px 6px rgba(0,0,0,0.2);
            }
            #add-product:hover {
                background-color: #4B2F1F;
                color: #F4F0E4;
            }
            .remove-product:hover {
                background-color: #c0c0c0 !important;
            }
            table tr:hover {
                background-color: #f9f9f9;
            }
            #stockSearch:focus, select:focus, input:focus {
                outline: none;
                border-color: #F4A261 !important;
                box-shadow: 0 0 3px rgba(244, 162, 97, 0.4);
            }
            .select2-container {
                width: 100% !important;
            }
            .select2-selection--single {
                border: 1px solid #ddd !important;
                border-radius: 6px !important;
                padding: 7px 10px !important;
                font-size: 13px !important;
                height: auto !important;
            }
            .select2-selection__rendered {
                color: #4B2F1F !important;
            }
            .select2-selection__arrow {
                height: 32px !important;
            }
            @media (max-width: 1024px) {
                .container > div {
                    grid-template-columns: 390px 1fr;
                    gap: 12px;
                }
            }
            @media (max-width: 768px) {
                .container > div {
                    grid-template-columns: 1fr;
                    gap: 0;
                    padding: 15px;
                }
                .content {
                    border-radius: 8px;
                    padding: 20px;
                }
                h2 {
                    font-size: 20px;
                    margin-bottom: 18px;
                }
                h3 {
                    font-size: 16px;
                }
                h4 {
                    font-size: 13px;
                }
                .product-entry {
                    flex-direction: column;
                }
                .remove-product {
                    width: 100%;
                }
                #add-product {
                    width: 100%;
                }
            }
            @media (max-width: 480px) {
                .container > div {
                    padding: 12px;
                }
                .content {
                    padding: 15px;
                }
                h2 {
                    font-size: 18px;
                }
                h3 {
                    font-size: 15px;
                }
            }
        </style>
    </div>
</body>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
