<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// === SECURITY: Only Operation Manager ===
$conn = get_db_connection();
if (!$conn) {
    die("<p style='color:red;text-align:center;padding:20px;'>Database connection failed.</p>");
}

// Ensure user_notification table exists
$createNotificationTableSql = "
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read),
        CONSTRAINT fk_notification_user_edit_product FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createNotificationTableSql)) {
    error_log("Failed to ensure user_notification table: " . $conn->error);
}

$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();

$allowed_roles = ['DG', 'DGA', 'Board', 'Admin'];
if (!in_array($role, $allowed_roles, true)) {
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}

// === GET PRODUCT ID ===
$product_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($product_id <= 0) {
    die("<p style='color:red;text-align:center;padding:20px;'>Invalid product ID.</p>");
}

// === FETCH PRODUCT ===
$stmt = $conn->prepare("SELECT * FROM product WHERE id = ?");
$stmt->bind_param("i", $product_id);
$stmt->execute();
$product = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$product) {
    die("<p style='color:red;text-align:center;padding:20px;'>Product not found.</p>");
}

// === HANDLE FORM SUBMISSION ===
$errors = [];
$success = '';

if (isset($_SESSION['edit_product_success'])) {
    $success = $_SESSION['edit_product_success'];
    unset($_SESSION['edit_product_success']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $volume_cl = (int)($_POST['volume_cl'] ?? 0);
    $crate_quantity = (int)($_POST['crate_quantity'] ?? 0);
    $price_per_crate = (int)($_POST['price_per_crate'] ?? 0);
    $low_stock_alert_crates = $_POST['low_stock_alert_crates'] !== '' ? (int)$_POST['low_stock_alert_crates'] : null;
    $low_stock_alert_units = $_POST['low_stock_alert_units'] !== '' ? (int)$_POST['low_stock_alert_units'] : null;
    $expiration_date = trim($_POST['expiration_date'] ?? '') ?: null;

    if ($name === '' || $volume_cl <= 0 || $crate_quantity <= 0 || $price_per_crate <= 0) {
        $errors[] = 'Please fill all required fields.';
    } else {
        $old_price_per_crate = $product['price_per_crate'] ?? 0;
        $prices_changed = ($old_price_per_crate != $price_per_crate);

        $stmt = $conn->prepare("UPDATE product SET name=?, volume_cl=?, crate_quantity=?, low_stock_alert_crates=?, low_stock_alert_units=?, expiration_date=?, price_per_crate=? WHERE id=?");
        $stmt->bind_param("siiiisii", $name, $volume_cl, $crate_quantity, $low_stock_alert_crates, $low_stock_alert_units, $expiration_date, $price_per_crate, $product_id);
        if ($stmt->execute()) {
            // Log price change if prices were changed
            if ($prices_changed) {
                // Ensure product_price_history table exists with correct structure
                $tableCheck = $conn->query("SHOW TABLES LIKE 'product_price_history'");
                if ($tableCheck && $tableCheck->num_rows > 0) {
                    // Check columns and migrate if needed
                    $columnCheck = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'old_price'");
                    if (!$columnCheck || $columnCheck->num_rows === 0) {
                        // Check if old columns exist
                        $hasOldPricePerCrate = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'old_price_per_crate'");
                        if ($hasOldPricePerCrate && $hasOldPricePerCrate->num_rows > 0) {
                            $conn->query("ALTER TABLE product_price_history CHANGE COLUMN old_price_per_crate old_price INT NOT NULL");
                        }
                        $hasNewPricePerCrate = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'new_price_per_crate'");
                        if ($hasNewPricePerCrate && $hasNewPricePerCrate->num_rows > 0) {
                            $conn->query("ALTER TABLE product_price_history CHANGE COLUMN new_price_per_crate new_price INT NOT NULL");
                        }
                        if ($columnCheck) $columnCheck->free();
                        if ($hasOldPricePerCrate) $hasOldPricePerCrate->free();
                        if ($hasNewPricePerCrate) $hasNewPricePerCrate->free();
                    } else {
                        if ($columnCheck) $columnCheck->free();
                    }
                    $tableCheck->free();
                }

                // Create table with standard structure (old_price, new_price)
                $createTableSql = "
                    CREATE TABLE IF NOT EXISTS product_price_history (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        product_id INT NOT NULL,
                        old_price INT NOT NULL,
                        new_price INT NOT NULL,
                        changed_by INT NOT NULL,
                        changed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                        INDEX idx_product (product_id),
                        INDEX idx_changed_at (changed_at),
                        FOREIGN KEY (product_id) REFERENCES product(id) ON DELETE CASCADE,
                        FOREIGN KEY (changed_by) REFERENCES user(id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                ";
                if (!$conn->query($createTableSql)) {
                    // If foreign key constraint fails, try without foreign keys
                    $createTableSqlSimple = "
                        CREATE TABLE IF NOT EXISTS product_price_history (
                            id INT AUTO_INCREMENT PRIMARY KEY,
                            product_id INT NOT NULL,
                            old_price INT NOT NULL,
                            new_price INT NOT NULL,
                            changed_by INT NOT NULL,
                            changed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                            INDEX idx_product (product_id),
                            INDEX idx_changed_at (changed_at)
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                    ";
                    $conn->query($createTableSqlSimple);
                }

                // Insert price change record
                $changed_by = (int)$_SESSION['user_id'];
                $stmt_history = $conn->prepare("INSERT INTO product_price_history (product_id, old_price, new_price, changed_by) VALUES (?, ?, ?, ?)");
                if ($stmt_history) {
                    $stmt_history->bind_param("iiii", $product_id, $old_price_per_crate, $price_per_crate, $changed_by);
                    $stmt_history->execute();
                    $stmt_history->close();

                    // Notify all users about the price change
                    try {
                        $users_stmt = $conn->query("SELECT id FROM user WHERE id IS NOT NULL");
                        if ($users_stmt === false) {
                            error_log("Failed to query users for notification: " . $conn->error);
                        } else {
                            $notification_message = sprintf(
                                'Le prix du produit "%s" a changé: %s BIF → %s BIF par caisse',
                                $name,
                                number_format($old_price_per_crate, 0, ',', '.'),
                                number_format($price_per_crate, 0, ',', '.')
                            );
                            $notification_link = '/masunzu_bar_hotel/modules/inventory/product_catalog.php';
                            
                            $notif_stmt = $conn->prepare('INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())');
                            if ($notif_stmt === false) {
                                error_log("Failed to prepare notification statement: " . $conn->error);
                                if ($users_stmt) $users_stmt->free();
                            } else {
                                $notification_count = 0;
                                $users_fetched = 0;
                                while ($user = $users_stmt->fetch_assoc()) {
                                    $users_fetched++;
                                    $user_id = (int)$user['id'];
                                    if ($user_id > 0) {
                                        $notif_stmt->bind_param('iss', $user_id, $notification_message, $notification_link);
                                        if ($notif_stmt->execute()) {
                                            $notification_count++;
                                        } else {
                                            error_log("Failed to send notification to user {$user_id}: " . $notif_stmt->error);
                                        }
                                    }
                                }
                                $notif_stmt->close();
                                if ($users_stmt) $users_stmt->free();
                                
                                if ($notification_count > 0) {
                                    error_log("✓ Price change notifications sent to {$notification_count} users for product: {$name}");
                                } else {
                                    error_log("⚠ No notifications sent. Users fetched: {$users_fetched}");
                                }
                            }
                        }
                    } catch (Exception $e) {
                        error_log("Exception while sending notifications: " . $e->getMessage());
                    }
                }
            }

            $_SESSION['edit_product_success'] = 'Product updated successfully!';
            $conn->close();
            header("Location: /masunzu_bar_hotel/modules/operation/edit_product.php?id=$product_id");
            exit;
        } else {
            $errors[] = 'Failed to update product.';
            $stmt->close();
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Product</title>
    <link rel="stylesheet" href="/masunzu_bar_hotel/assets/css/style.css">
    <style>
        :root {
            --primary: #4B2F1F;
            --accent: #F4A261;
            --light: #F4F0E4;
            --dark: #2E1A12;
            --gray: #6c757d;
        }
        body {
            font-family: 'Segoe UI', Arial, sans-serif;
            margin: 0;
            padding: 0;
            background: linear-gradient(135deg, var(--primary) 0%, var(--accent) 100%);
            min-height: 100vh;
        }
        .container {
            max-width: 600px;
            margin: 40px auto;
            padding: 0 15px;
        }
        .card {
            background: var(--light);
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 6px 15px rgba(0,0,0,0.2);
        }
        .card h2 {
            color: var(--primary);
            margin: 0 0 25px;
            font-size: 1.5rem;
            border-bottom: 2px solid var(--accent);
            padding-bottom: 12px;
        }
        .label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--primary);
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .required::after {
            content: ' *';
            color: #d9534f;
        }
        .input-group {
            margin-bottom: 20px;
        }
        input, textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid rgba(75,47,31,0.3);
            border-radius: 8px;
            font-size: 14px;
            box-sizing: border-box;
            font-family: 'Segoe UI', Arial, sans-serif;
            background: rgba(255,255,255,0.7);
            transition: all 0.2s;
        }
        input:focus, textarea:focus {
            outline: none;
            background: white;
            border-color: var(--accent);
            box-shadow: 0 0 0 3px rgba(244,162,97,0.1);
        }
        .grid-2 {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 8px;
        }
        .alert-danger {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .alert-danger ul {
            margin: 10px 0 0 0;
            padding-left: 20px;
        }
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .btn-group {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid rgba(75,47,31,0.1);
        }
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            text-decoration: none;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.2s;
        }
        .btn-primary {
            background: var(--primary);
            color: var(--light);
        }
        .btn-primary:hover {
            background: var(--accent);
            color: var(--primary);
            transform: translateY(-1px);
        }
        .btn-secondary {
            background: var(--gray);
            color: white;
        }
        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-1px);
        }
        @media (max-width: 768px) {
            .grid-2 {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <?php require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php'; ?>
    
    <div class="container">
        <div class="card">
            <h2>Edit Product</h2>

            <?php if (!empty($errors)): ?>
                <div class="alert alert-danger">
                    <strong>❌ Please fix the following:</strong>
                    <ul>
                        <?php foreach ($errors as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <?php if (!empty($success)): ?>
                <div class="alert alert-success">
                    ✓ <?= htmlspecialchars($success) ?>
                </div>
            <?php endif; ?>

            <form method="POST" action="">
                <div class="input-group">
                    <label for="name" class="required">Product Name</label>
                    <input type="text" id="name" name="name" value="<?= htmlspecialchars($product['name']) ?>" required>
                </div>

                <div class="grid-2">
                    <div class="input-group">
                        <label for="volume_cl" class="required">Volume (CL)</label>
                        <input type="number" id="volume_cl" name="volume_cl" value="<?= htmlspecialchars($product['volume_cl']) ?>" required min="1">
                    </div>

                    <div class="input-group">
                        <label for="crate_quantity" class="required">Bottles/Crate</label>
                        <input type="number" id="crate_quantity" name="crate_quantity" value="<?= htmlspecialchars($product['crate_quantity']) ?>" required min="1">
                    </div>
                </div>

                <div class="input-group">
                    <label for="price_per_crate" class="required">Price / Crate (BIF)</label>
                    <input type="number" id="price_per_crate" name="price_per_crate" value="<?= htmlspecialchars($product['price_per_crate']) ?>" required min="1">
                </div>

                <div class="grid-2">
                    <div class="input-group">
                        <label for="low_stock_alert_crates">Low Alert (Crates)</label>
                        <input type="number" id="low_stock_alert_crates" name="low_stock_alert_crates" value="<?= $product['low_stock_alert_crates'] ?? '' ?>" min="0">
                    </div>

                    <div class="input-group">
                        <label for="low_stock_alert_units">Low Alert (Units)</label>
                        <input type="number" id="low_stock_alert_units" name="low_stock_alert_units" value="<?= $product['low_stock_alert_units'] ?? '' ?>" min="0">
                    </div>
                </div>

                <div class="input-group">
                    <label for="expiration_date">Expiration Date</label>
                    <input type="date" id="expiration_date" name="expiration_date" value="<?= $product['expiration_date'] ?? '' ?>">
                </div>

                <div class="btn-group">
                    <a href="view_product.php?id=<?= $product['id'] ?>" class="btn btn-secondary">Cancel</a>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>

    <?php require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
</body>
</html>
