<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// === SECURITY: DGA & Leadership Only ===
$conn = get_db_connection();
if (!$conn) {
    die("<p style='color:red;text-align:center;padding:20px;'>Échec de la connexion à la base de données.</p>");
}

$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();

$allowed_roles = ['DG', 'DGA', 'Board', 'Admin'];
if (!in_array($role, $allowed_roles, true)) {
    error_log("Unauthorized access attempt to create_product.php by {$_SESSION['full_name']} (Role: $role)");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}

// === ENSURE AUDIT COLUMNS EXIST ===
// === SCHEMA SAFEGUARDS ===
$hasPricePerCrate = $conn->query("SHOW COLUMNS FROM product LIKE 'price_per_crate'");
if ($hasPricePerCrate && $hasPricePerCrate->num_rows === 0) {
    $hasWholesale = $conn->query("SHOW COLUMNS FROM product LIKE 'wholesale_price'");
    if ($hasWholesale && $hasWholesale->num_rows > 0) {
        $conn->query("ALTER TABLE product CHANGE COLUMN wholesale_price price_per_crate INT NOT NULL");
    } else {
        $conn->query("ALTER TABLE product ADD COLUMN price_per_crate INT NOT NULL DEFAULT 0 AFTER crate_quantity");
    }
}
$conn->query("ALTER TABLE product DROP COLUMN IF EXISTS wholesale_price");
$conn->query("ALTER TABLE product DROP COLUMN IF EXISTS retail_price");
$conn->query("ALTER TABLE product ADD COLUMN IF NOT EXISTS created_by INT NULL AFTER price_per_crate");
$conn->query("ALTER TABLE product ADD COLUMN IF NOT EXISTS created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP AFTER created_by");

// Ensure user_notification table exists
$createNotificationTableSql = "
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read),
        CONSTRAINT fk_notification_user_create_product FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createNotificationTableSql)) {
    error_log("Failed to ensure user_notification table: " . $conn->error);
}

// Ensure product_price_history table exists with correct structure
$tableCheck = $conn->query("SHOW TABLES LIKE 'product_price_history'");
if ($tableCheck && $tableCheck->num_rows > 0) {
    // Table exists, check columns and migrate if needed
    $columnCheck = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'old_price'");
    if (!$columnCheck || $columnCheck->num_rows === 0) {
        // Check if old columns exist
        $hasOldPricePerCrate = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'old_price_per_crate'");
        if ($hasOldPricePerCrate && $hasOldPricePerCrate->num_rows > 0) {
            // Rename columns to match standard structure
            $conn->query("ALTER TABLE product_price_history CHANGE COLUMN old_price_per_crate old_price INT NOT NULL");
        }
        $hasNewPricePerCrate = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'new_price_per_crate'");
        if ($hasNewPricePerCrate && $hasNewPricePerCrate->num_rows > 0) {
            $conn->query("ALTER TABLE product_price_history CHANGE COLUMN new_price_per_crate new_price INT NOT NULL");
        }
        // Handle legacy columns
        $hasOldWholesale = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'old_wholesale_price'");
        if ($hasOldWholesale && $hasOldWholesale->num_rows > 0) {
            $conn->query("ALTER TABLE product_price_history CHANGE COLUMN old_wholesale_price old_price INT NOT NULL");
        }
        $hasNewWholesale = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'new_wholesale_price'");
        if ($hasNewWholesale && $hasNewWholesale->num_rows > 0) {
            $conn->query("ALTER TABLE product_price_history CHANGE COLUMN new_wholesale_price new_price INT NOT NULL");
        }
        $conn->query("ALTER TABLE product_price_history DROP COLUMN IF EXISTS old_retail_price");
        $conn->query("ALTER TABLE product_price_history DROP COLUMN IF EXISTS new_retail_price");
        if ($columnCheck) $columnCheck->free();
        if ($hasOldPricePerCrate) $hasOldPricePerCrate->free();
        if ($hasNewPricePerCrate) $hasNewPricePerCrate->free();
        if ($hasOldWholesale) $hasOldWholesale->free();
        if ($hasNewWholesale) $hasNewWholesale->free();
    } else {
        if ($columnCheck) $columnCheck->free();
    }
    $tableCheck->free();
}

// Create table with standard structure (old_price, new_price)
$createTableSql = "
    CREATE TABLE IF NOT EXISTS product_price_history (
        id INT AUTO_INCREMENT PRIMARY KEY,
        product_id INT NOT NULL,
        old_price INT NOT NULL,
        new_price INT NOT NULL,
        changed_by INT NOT NULL,
        changed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_product (product_id),
        INDEX idx_changed_at (changed_at),
        FOREIGN KEY (product_id) REFERENCES product(id) ON DELETE CASCADE,
        FOREIGN KEY (changed_by) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createTableSql)) {
    error_log("Failed to ensure product_price_history table: " . $conn->error);
    // If foreign key constraint fails, try without foreign keys
    $createTableSqlSimple = "
        CREATE TABLE IF NOT EXISTS product_price_history (
            id INT AUTO_INCREMENT PRIMARY KEY,
            product_id INT NOT NULL,
            old_price INT NOT NULL,
            new_price INT NOT NULL,
            changed_by INT NOT NULL,
            changed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_product (product_id),
            INDEX idx_changed_at (changed_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    if (!$conn->query($createTableSqlSimple)) {
        error_log("Failed to create product_price_history table even without foreign keys: " . $conn->error);
    }
}

// === HANDLE FORM SUBMISSIONS ===
$errors = [];
$success = '';

if (isset($_SESSION['product_success'])) {
    $success = $_SESSION['product_success'];
    unset($_SESSION['product_success']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $name = trim($_POST['name'] ?? '');
        $volume_cl = (int)($_POST['volume_cl'] ?? 0);
        $crate_quantity = (int)($_POST['crate_quantity'] ?? 0);
    $price_per_crate = (int)($_POST['price_per_crate'] ?? 0);
        $low_stock_alert_crates = $_POST['low_stock_alert_crates'] !== '' ? (int)$_POST['low_stock_alert_crates'] : null;
    $low_stock_alert_units = null;
        $expiration_date = trim($_POST['expiration_date'] ?? '') ?: null;

    if ($name && $volume_cl > 0 && $crate_quantity > 0 && $price_per_crate > 0) {
            $stmt = $conn->prepare("SELECT id FROM product WHERE name = ?");
            $stmt->bind_param("s", $name);
            $stmt->execute();
            if ($stmt->get_result()->num_rows > 0) {
                $errors[] = 'Un produit avec ce nom existe déjà.';
            } else {
            $stmt = $conn->prepare("INSERT INTO product (name, volume_cl, crate_quantity, low_stock_alert_crates, low_stock_alert_units, expiration_date, price_per_crate, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                $created_by = $_SESSION['user_id'];
            $stmt->bind_param("siiiisii", $name, $volume_cl, $crate_quantity, $low_stock_alert_crates, $low_stock_alert_units, $expiration_date, $price_per_crate, $created_by);
                if ($stmt->execute()) {
                $new_product_id = $conn->insert_id;

                // Log initial product creation as the first price history entry
                // Use old_price = 0 to indicate initial creation
                $stmt_history = $conn->prepare("INSERT INTO product_price_history (product_id, old_price, new_price, changed_by) VALUES (?, 0, ?, ?)");
                if ($stmt_history) {
                    $stmt_history->bind_param("iii", $new_product_id, $price_per_crate, $created_by);
                    $stmt_history->execute();
                    $stmt_history->close();
                }
                
                // Notify all users about the new product
                try {
                    $users_stmt = $conn->query("SELECT id FROM user WHERE id IS NOT NULL");
                    if ($users_stmt === false) {
                        error_log("Failed to query users for notification: " . $conn->error);
                    } else {
                        $notification_message = sprintf(
                            'Nouveau produit créé: "%s" - Prix: %s BIF par caisse',
                            $name,
                            number_format($price_per_crate, 0, ',', '.')
                        );
                        $notification_link = '/masunzu_bar_hotel/modules/inventory/product_catalog.php';
                        
                        $notif_stmt = $conn->prepare('INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())');
                        if ($notif_stmt === false) {
                            error_log("Failed to prepare notification statement: " . $conn->error);
                            if ($users_stmt) $users_stmt->free();
                        } else {
                            $notification_count = 0;
                            $users_fetched = 0;
                            while ($user = $users_stmt->fetch_assoc()) {
                                $users_fetched++;
                                $user_id = (int)$user['id'];
                                if ($user_id > 0) {
                                    $notif_stmt->bind_param('iss', $user_id, $notification_message, $notification_link);
                                    if ($notif_stmt->execute()) {
                                        $notification_count++;
                                    } else {
                                        error_log("Failed to send notification to user {$user_id}: " . $notif_stmt->error);
                                    }
                                }
                            }
                            $notif_stmt->close();
                            if ($users_stmt) $users_stmt->free();
                            
                            if ($notification_count > 0) {
                                error_log("✓ New product notifications sent to {$notification_count} users for product: {$name}");
                            } else {
                                error_log("⚠ No notifications sent. Users fetched: {$users_fetched}");
                            }
                        }
                    }
                } catch (Exception $e) {
                    error_log("Exception while sending notifications: " . $e->getMessage());
                }
                
                $_SESSION['product_success'] = 'Produit créé avec succès !';
                $conn->close();
                header("Location: /masunzu_bar_hotel/modules/operation/create_product.php");
                exit;
                } else {
                    $errors[] = 'Échec de la création du produit.';
                }
            }
            $stmt->close();
        } else {
            $errors[] = 'Veuillez remplir tous les champs requis correctement.';
    }
}

// === SERVER-SIDE FILTERING ===
$search = trim($_GET['search'] ?? '');
$filter_volume = !empty($_GET['volume_cl']) && is_numeric($_GET['volume_cl']) 
    ? (int)$_GET['volume_cl'] 
    : null;
$filter_crate = !empty($_GET['crate_quantity']) && is_numeric($_GET['crate_quantity']) 
    ? (int)$_GET['crate_quantity'] 
    : null;
$filter_has_low_alert = isset($_GET['has_low_alert']) ? 1 : 0;

$conditions = [];
$types = '';
$params = [];

if ($search !== '') {
    $conditions[] = 'name LIKE ?';
    $types .= 's';
    $params[] = "%$search%";
}
if ($filter_volume !== null) {
    $conditions[] = 'volume_cl = ?';
    $types .= 'i';
    $params[] = $filter_volume;
}
if ($filter_crate !== null) {
    $conditions[] = 'crate_quantity = ?';
    $types .= 'i';
    $params[] = $filter_crate;
}
if ($filter_has_low_alert) {
    $conditions[] = 'low_stock_alert_crates IS NOT NULL';
}

$where = $conditions ? ' WHERE ' . implode(' AND ', $conditions) : '';
$sql = "SELECT id, name, volume_cl, crate_quantity, low_stock_alert_crates, expiration_date, price_per_crate, created_at 
        FROM product $where 
        ORDER BY created_at DESC, id DESC";

$products = [];
$stmt = $conn->prepare($sql);
if ($stmt) {
    if ($types) {
    $refs = [];
    foreach ($params as $k => $v) $refs[$k] = &$params[$k];
    array_unshift($refs, $types);
    call_user_func_array([$stmt, 'bind_param'], $refs);
}
    $stmt->execute();
    $result = $stmt->get_result();
    $products = $result->fetch_all(MYSQLI_ASSOC) ?? [];
    $stmt->close();
}
$conn->close();

include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gérer les Produits - Masunzu Bar & Hotel</title>
    <style>
        :root {
            --primary: #4B2F1F;
            --accent: #F4A261;
            --light: #F4F0E4;
            --dark: #2E1A12;
            --gray: #6c757d;
        }
        body { font-family: 'Segoe UI', Arial, sans-serif; margin: 0; padding: 0; background: #F4F0E4; min-height: 100vh; }
        .container { padding: 20px; }
        .card { background: #fff; border-radius: 12px; padding: 20px; margin-bottom: 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.08); }
        .card h2 { color: var(--primary); margin: 0 0 10px; font-size: 1.5rem; display: flex; justify-content: space-between; align-items: center; }
        .btn { background: var(--primary); color: var(--light); border: none; padding: 10px 16px; border-radius: 8px; cursor: pointer; font-size: 14px; transition: all 0.2s; text-decoration: none; display: inline-block; }
        .btn:hover { background: var(--accent); color: var(--primary); transform: translateY(-1px); }
        .btn-sm { padding: 6px 10px; font-size: 12px; }
        .btn-view { background: #17a2b8; color: white; }
        .btn-view:hover { background: #138496; }
        .btn-edit { background: #ffc107; color: #212529; }
        .btn-edit:hover { background: #e0a800; }
        .btn-history { background: #6c63ff; color: white; }
        .btn-history:hover { background: #5a52d5; }
        .btn-reset { background: var(--gray); }
        .btn-reset:hover { background: #5a6268; }
        .input, .select { width: 100%; padding: 10px; border: 1px solid #B08968; border-radius: 8px; font-size: 14px; background: #fff; box-sizing: border-box; }
        .grid { display: grid; gap: 14px; }
        .grid-3 { grid-template-columns: repeat(3, 1fr); }
        .grid-2 { grid-template-columns: repeat(2, 1fr); }
        .label { color: var(--primary); font-weight: 600; font-size: 14px; margin-bottom: 6px; display: block; }
        .table { width: 100%; border-collapse: collapse; margin-top: 10px; }
        .table th, .table td { padding: 12px 14px; text-align: left; border-bottom: 1px solid #F0F0F0; font-size: 14px; color: var(--primary); }
        .table th { background: var(--primary); color: #F4F0E4; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em; }
        .toolbar { display: flex; flex-wrap: wrap; gap: 10px; align-items: center; margin-bottom: 10px; }
        .no-wrap { white-space: nowrap; }
        .alert-success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; padding: 12px; border-radius: 8px; margin: 12px 0; }
        .alert-error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; padding: 12px; border-radius: 8px; margin: 12px 0; }
        .action-buttons { display: flex; gap: 5px; white-space: nowrap; }
        @media (max-width: 768px) {
            .grid-3, .grid-2 { grid-template-columns: 1fr; }
            .toolbar { flex-direction: column; align-items: stretch; }
            .input, .select { max-width: 100%; }
        }
    </style>
</head>
<body>
<div class="container">

    <!-- CREATE FORM -->
    <div class="card">
        <h2>Nouveau Produit <button id="toggleCreate" class="btn">Créer un Produit</button></h2>

        <?php if ($success): ?>
            <div class="alert-success"><?= htmlspecialchars($success, ENT_QUOTES, 'UTF-8') ?></div>
        <?php endif; ?>

        <?php if ($errors): ?>
            <div class="alert-error">
                <ul style="margin:0; padding-left:20px;">
                    <?php foreach ($errors as $e): ?>
                        <li><?= htmlspecialchars($e, ENT_QUOTES, 'UTF-8') ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <form id="createForm" method="POST" style="display:none; margin-top:15px;">
            <div class="grid grid-3">
                <div><label class="label">Nom</label><input class="input" type="text" name="name" required></div>
                <div><label class="label">Volume (cl)</label><input class="input" type="number" name="volume_cl" min="1" required></div>
                <div><label class="label">Bouteilles/Caisse</label><input class="input" type="number" name="crate_quantity" min="1" required></div>
                <div><label class="label">Alerte Stock Faible (Caisses)</label><input class="input" type="number" name="low_stock_alert_crates" min="0" placeholder="ex: 2"></div>
                <div><label class="label">Date d'Expiration</label><input class="input" type="date" name="expiration_date"></div>
                <div><label class="label">Prix / Caisse (BIF)</label><input class="input" type="number" name="price_per_crate" min="1" required></div>
            </div>
            <div style="margin-top:15px;">
                <button type="submit" class="btn">Créer le Produit</button>
            </div>
        </form>
    </div>

    <!-- PRODUCTS LIST -->
    <div class="card">
        <h2>Produits (<?= count($products) ?>)</h2>
        
        <!-- FILTERS -->
        <form method="GET" id="filterForm">
        <div class="toolbar">
                <input type="text" name="search" class="input" placeholder="Rechercher un nom..." value="<?= htmlspecialchars($search) ?>" style="max-width:200px;">
                <input type="number" name="volume_cl" class="input" placeholder="Volume (cl)" min="1" value="<?= $filter_volume ?? '' ?>" style="max-width:120px;">
                <input type="number" name="crate_quantity" class="input" placeholder="Bouteilles/Caisse" min="1" value="<?= $filter_crate ?? '' ?>" style="max-width:140px;">
                <?php if ($filter_has_low_alert): ?>
                    <input type="hidden" name="has_low_alert" value="1">
                <?php endif; ?>
                <button type="submit" class="btn">Filtrer</button>
                <a href="create_product.php" class="btn btn-reset">Réinitialiser</a>
        </div>
        </form>

        <!-- TABLE -->
        <div style="overflow-x:auto;">
            <table class="table">
                <thead>
                    <tr>
                        <th>#</th>
                        <th>Nom</th>
                        <th>Volume</th>
                        <th>Caisse</th>
                        <th>Alerte Stock Faible</th>
                        <th>Prix / Caisse</th>
                        <th>Expire</th>
                        <th>Créé</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($products)): ?>
                        <tr>
                            <td colspan="9" style="text-align: center; padding: 20px; color: #999;">
                                Aucun produit trouvé. <?php if ($search || $filter_volume || $filter_crate): ?>Effacez les filtres et réessayez.<?php endif; ?>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($products as $i => $p): ?>
                            <tr>
                                <td><?= $i + 1 ?></td>
                                <td><?= htmlspecialchars($p['name']) ?></td>
                                <td><?= htmlspecialchars($p['volume_cl']) ?> cl</td>
                                <td><?= htmlspecialchars($p['crate_quantity']) ?></td>
                                <td><?= $p['low_stock_alert_crates'] ? number_format($p['low_stock_alert_crates'], 0, ',', '.') . ' caisses' : '—' ?></td>
                                <td><?= number_format($p['price_per_crate'], 0, ',', '.') ?> BIF</td>
                                <td><?= $p['expiration_date'] ? htmlspecialchars($p['expiration_date']) : '—' ?></td>
                                <td><?= htmlspecialchars(substr($p['created_at'], 0, 10)) ?></td>
                                <td class="no-wrap">
                                    <div class="action-buttons">
                                        <a href="view_product.php?id=<?= $p['id'] ?>" class="btn btn-sm btn-view">Voir</a>
                                        <a href="edit_product.php?id=<?= $p['id'] ?>" class="btn btn-sm btn-edit">Modifier</a>
                                        <a href="product_history.php?id=<?= $p['id'] ?>" class="btn btn-sm btn-history">Historique</a>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
    // Toggle create form
    document.getElementById('toggleCreate')?.addEventListener('click', function() {
        const form = document.getElementById('createForm');
        if (form) {
            form.style.display = form.style.display === 'none' ? 'block' : 'none';
            this.textContent = form.style.display === 'none' ? 'Créer un Produit' : 'Masquer le Formulaire';
        }
    });
</script>

</body>
</html>

<?php require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
