<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Operation Manager, DG, DGA, Board, Admin roles
$allowed_roles = ['Operation Manager', 'DG', 'DGA', 'Board', 'Admin'];
if (!in_array($_SESSION['role_name'], $allowed_roles)) {
    error_log("Unauthorized access to add_main_stock_history.php by user {$_SESSION['full_name']} (Role: {$_SESSION['role_name']})");
    // Use dashboard mapping to redirect correctly
    $dashboard_map = [
        'Board' => 'admin_dashboard.php',
        'DG' => 'admin_dashboard.php',
        'DGA' => 'admin_dashboard.php',
        'Admin' => 'admin_dashboard.php',
        'Operation Manager' => 'operation_dashboard.php',
        'Stock Manager' => 'stock_manager_dashboard.php',
        'Accountant' => 'accountant_dashboard.php',
        'Caissier_Comptable' => 'cashier_dashboard.php',
    ];
    $dashboard = $dashboard_map[$_SESSION['role_name']] ?? 'admin_dashboard.php';
    header("Location: /masunzu_bar_hotel/dashboards/" . $dashboard);
    exit;
}

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in add_main_stock_history.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}

// Fetch products for product name mapping and crate_quantity
$products_query = $conn->query("SELECT id, name, crate_quantity FROM product ORDER BY name");
$products = $products_query->fetch_all(MYSQLI_ASSOC);
$product_map = array_column($products, 'crate_quantity', 'id');

// Handle filters
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';
$product_name = $_GET['product_name'] ?? '';

// Build WHERE clauses for both queries
$where_transfer = [];
$where_log = [];
$params = [];
$param_types = '';

if ($start_date) {
    $where_transfer[] = "DATE(st.sent_at) >= ?";
    $where_log[] = "DATE(l.created_at) >= ?";
    $params[] = $start_date;
    $param_types .= 's';
}
if ($end_date) {
    $where_transfer[] = "DATE(st.sent_at) <= ?";
    $where_log[] = "DATE(l.created_at) <= ?";
    $params[] = $end_date;
    $param_types .= 's';
}
if ($product_name) {
    $where_transfer[] = "p.name LIKE ?";
    $where_log[] = "p.name LIKE ?";
    $params[] = "%$product_name%";
    $param_types .= 's';
}

// Query 1: Stock transfers (confirmed status)
// Must have same number and order of columns as log_query
$transfer_query = "SELECT st.id, 
                          st.product_id, 
                          st.quantity, 
                          st.unit_type, 
                          st.sent_at as created_at, 
                          st.received_at,
                          st.province_id,
                          st.shipment_id,
                          st.operation_manager_id as user_id, 
                          u.full_name AS user_name, 
                          p.name AS product_name, 
                          COALESCE(p.crate_quantity, 12) as crate_quantity,
                          NULL as log_description,
                          pr.name as province_name,
                          'transfer' as source_type
                  FROM stock_transfer st
                  LEFT JOIN user u ON st.operation_manager_id = u.id
                  LEFT JOIN product p ON st.product_id = p.id
                  LEFT JOIN province pr ON st.province_id = pr.id
                  WHERE st.status = 'confirmed'";
if (!empty($where_transfer)) {
    $transfer_query .= " AND " . implode(' AND ', $where_transfer);
}

// Query 2: Direct stock additions from log table (action = 'add_stock')
// Format: "Added {formatted_quantity} of {product_name} to main stock ({formatted_quantity})"
// Must have same number and order of columns as transfer_query
$log_query = "SELECT l.id, 
                     p.id as product_id,
                     0 as quantity,  -- Will be extracted in PHP from description
                     'crates' as unit_type,
                     l.created_at,
                     NULL as received_at,
                     NULL as province_id,
                     NULL as shipment_id,
                     l.user_id,
                     u.full_name AS user_name,
                     p.name AS product_name,
                     COALESCE(p.crate_quantity, 12) as crate_quantity,
                     l.description as log_description,
                     NULL as province_name,
                     'direct_add' as source_type
              FROM log l
              LEFT JOIN user u ON l.user_id = u.id
              LEFT JOIN product p ON l.description LIKE CONCAT('%', p.name, '%')
              WHERE l.action = 'add_stock' 
                AND (
                    l.description LIKE '%to main stock%' 
                    OR l.description LIKE '%stock principal%'
                )";
if (!empty($where_log)) {
    $log_query .= " AND " . implode(' AND ', $where_log);
}

// Combine both queries with UNION ALL
$combined_query = "($transfer_query) UNION ALL ($log_query) ORDER BY created_at DESC";

$stmt = $conn->prepare($combined_query);
if ($param_types && $stmt) {
    // We need to bind params twice (once for each query in UNION)
    $all_params = array_merge($params, $params); // Duplicate params for both queries
    $all_param_types = $param_types . $param_types;
    $stmt->bind_param($all_param_types, ...$all_params);
} elseif (!$param_types && $stmt) {
    // No params needed
} else {
    error_log("Query preparation failed in add_main_stock_history.php: " . ($conn->error ?? 'Unknown error'));
    $stock_additions = [];
}

if ($stmt) {
    $stmt->execute();
    $stock_additions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    
    // Extract quantity from log_description for direct_add entries
    foreach ($stock_additions as &$addition) {
        if (isset($addition['source_type']) && $addition['source_type'] === 'direct_add' && isset($addition['log_description'])) {
            // Parse quantity from description like "Added 10 crates..." or "Ajouté 10 caisses..."
            // Try to extract number before the unit keyword or inside parentheses
            $desc = $addition['log_description'];
            if (preg_match('/Added\s+([\d,\.]+)\s+(?:crates?|bottles?)/i', $desc, $matches)) {
                $addition['quantity'] = (int)str_replace([',', '.'], '', $matches[1]);
            } elseif (preg_match('/\(([\d,\.]+)\s*(?:crates?|bottles?)\)/i', $desc, $matches)) {
                $addition['quantity'] = (int)str_replace([',', '.'], '', $matches[1]);
            } elseif (preg_match('/Ajout[ée]\s+([\d\s,\.]+)\s+(?:caisses?|bouteilles?)/i', $desc, $matches)) {
                $addition['quantity'] = (int)str_replace([',', '.', ' '], '', $matches[1]);
            } elseif (preg_match('/\(([\d\s,\.]+)\s*(?:caisses?|bouteilles?)\)/i', $desc, $matches)) {
                $addition['quantity'] = (int)str_replace([',', '.', ' '], '', $matches[1]);
            }
            // Remove log_description from result as it's not needed in display
            unset($addition['log_description']);
        }
    }
    unset($addition);
} else {
    $stock_additions = [];
}

if (empty($stock_additions)) {
    error_log("No stock additions found in add_main_stock_history.php");
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="main_stock_addition_history_' . date('Y-m-d') . '.csv"');
    
    // Create output stream
    $output = fopen('php://output', 'w');
    
    // Add BOM for UTF-8 Excel compatibility
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // CSV headers
    fputcsv($output, [
        'ID',
        'Added By',
        'Product Name',
        'Quantity (Crates)',
        'Unit Type',
        'Date Added'
    ]);
    
    // CSV data rows
    foreach ($stock_additions as $addition) {
        $qty = (int)($addition['quantity'] ?? 0);
        $unit_type = $addition['unit_type'] ?? 'crates';
        $created_at = $addition['created_at'] ?? ($addition['sent_at'] ?? '');
        fputcsv($output, [
            $addition['id'],
            $addition['user_name'] ?: 'Unknown',
            $addition['product_name'] ?: 'N/A',
            $qty, // Raw number for CSV (no formatting)
            $unit_type ?: 'N/A',
            $created_at ? date('Y-m-d H:i:s', strtotime($created_at)) : 'N/A'
        ]);
    }
    
    fclose($output);
    $conn->close();
    exit;
}

$conn->close();
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
</head>
<body>
    <div class="container" style="padding: 20px 20px 20px 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; box-sizing: border-box; width: 100%; margin: 0; position: relative; left: 0; overflow-x: hidden;">
        <div style="width: 100%; max-width: 100%; margin: 0; padding: 0;">
            <h1 style="color: #4B2F1F; font-size: 28px; font-weight: bold; text-align: center; margin-bottom: 20px;">Historique des Ajouts au Stock Principal</h1>
            
            <!-- Filters -->
            <div style="background-color: #FFFFFF; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); margin-bottom: 30px; border: none;">
                <form method="GET" style="display: flex; flex-wrap: wrap; gap: 15px;">
                    <div style="flex: 1; min-width: 150px;">
                        <label for="start_date" style="color: #4B2F1F; font-size: 14px; font-weight: 500;">Date de Début</label>
                        <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                    </div>
                    <div style="flex: 1; min-width: 150px;">
                        <label for="end_date" style="color: #4B2F1F; font-size: 14px; font-weight: 500;">Date de Fin</label>
                        <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                    </div>
                    <div style="flex: 1; min-width: 150px;">
                        <label for="product_name" style="color: #4B2F1F; font-size: 14px; font-weight: 500;">Produit</label>
                        <input type="text" id="product_name" name="product_name" value="<?php echo htmlspecialchars($product_name); ?>" placeholder="Entrer le nom du produit" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                    </div>
                    <div style="flex: 1; min-width: 150px; display: flex; align-items: flex-end;">
                        <button type="submit" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px; border: none; border-radius: 8px; font-size: 14px; cursor: pointer; width: 100%;">Appliquer les Filtres</button>
                    </div>
                </form>
            </div>

            <!-- Main Stock Addition History -->
            <div style="background-color: #FFFFFF; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); border: none;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                    <h2 style="color: #4B2F1F; font-size: 22px; margin: 0;">Historique des Ajouts au Stock Principal</h2>
                    <?php if (!empty($stock_additions)): ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'csv'])); ?>" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 20px; border-radius: 8px; text-decoration: none; font-size: 14px; display: inline-block;">Exporter en CSV</a>
                    <?php endif; ?>
                </div>
                <div style="margin-bottom: 15px;">
                    <input type="text" id="stockSearch" placeholder="Rechercher les ajouts de stock..." style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                </div>
                <?php if (empty($stock_additions)): ?>
                    <p style="color: #4B2F1F;">Aucun ajout de stock trouvé pour les filtres sélectionnés.</p>
                <?php else: ?>
                    <div style="overflow-x: auto;">
                        <table id="stockAddTable" style="width: 100%; border-collapse: collapse;">
                            <thead>
                                <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                                    <th style="padding: 10px; text-align: left; border: none;">ID</th>
                                    <th style="padding: 10px; text-align: left; border: none;">Ajouté Par</th>
                                    <th style="padding: 10px; text-align: left; border: none;">Produit</th>
                                    <th style="padding: 10px; text-align: left; border: none;">Quantité</th>
                                    <th style="padding: 10px; text-align: left; border: none;">Type</th>
                                    <th style="padding: 10px; text-align: left; border: none;">Dépôt</th>
                                    <th style="padding: 10px; text-align: left; border: none;">Date de Création</th>
                                    <th style="padding: 10px; text-align: left; border: none;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($stock_additions as $addition): ?>
                                    <?php
                                        $qty = (int)($addition['quantity'] ?? 0);
                                        $unit_type = $addition['unit_type'] ?? 'crates';
                                        $crate_qty = (int)($addition['crate_quantity'] ?? 12);
                                        $created_at = $addition['created_at'] ?? ($addition['sent_at'] ?? '');
                                        $source_type = $addition['source_type'] ?? 'transfer';
                                    ?>
                                    <tr style="border-bottom: 1px solid #E0E0E0;" data-search="<?php echo htmlspecialchars(strtolower(($addition['product_name'] ?? '') . ' ' . ($addition['user_name'] ?? '') . ' ' . ($addition['province_name'] ?? '')), ENT_QUOTES, 'UTF-8'); ?>">
                                        <td style="padding: 10px; border: none;"><?php echo htmlspecialchars($addition['id'] ?? 'N/A'); ?></td>
                                        <td style="padding: 10px; border: none;"><?php echo htmlspecialchars($addition['user_name'] ?: 'Inconnu'); ?></td>
                                        <td style="padding: 10px; border: none;"><?php echo htmlspecialchars($addition['product_name'] ?: 'N/A'); ?></td>
                                        <td style="padding: 10px; border: none;">
                                            <?php 
                                            if ($qty > 0) {
                                                echo format_quantity($qty, $unit_type, $crate_qty);
                                            } else {
                                                echo 'N/A';
                                            }
                                            ?>
                                        </td>
                                        <td style="padding: 10px; border: none;">
                                            <?php if ($source_type === 'transfer'): ?>
                                                <span style="color: #FF9800; font-weight: 600;">📦 Transfert</span>
                                            <?php else: ?>
                                                <span style="color: #4CAF50; font-weight: 600;">➕ Ajout Direct</span>
                                            <?php endif; ?>
                                        </td>
                                        <td style="padding: 10px; border: none;">
                                            <?php if ($source_type === 'transfer' && isset($addition['province_name']) && $addition['province_name']): ?>
                                                <span style="color: #FF9800; font-weight: 500;">📍 <?php echo htmlspecialchars($addition['province_name']); ?></span>
                                            <?php else: ?>
                                                <span style="color: #999;">—</span>
                                            <?php endif; ?>
                                        </td>
                                        <td style="padding: 10px; border: none;"><?php echo $created_at ? date('Y-m-d H:i', strtotime($created_at)) : 'N/A'; ?></td>
                                        <td style="padding: 10px; border: none;">
                                            <?php if ($source_type === 'transfer'): ?>
                                                <a href="add_main_stock_details.php?transfer_id=<?php echo $addition['id']; ?>&source_type=transfer" style="background-color: #4B2F1F; color: #F4F0E4; padding: 8px 12px; border-radius: 8px; text-decoration: none; font-size: 14px;">Voir</a>
                                            <?php else: ?>
                                                <a href="add_main_stock_details.php?log_id=<?php echo $addition['id']; ?>&source_type=direct_add" style="background-color: #4CAF50; color: #FFFFFF; padding: 8px 12px; border-radius: 8px; text-decoration: none; font-size: 14px;" title="Ajout Direct: Stock ajouté directement au stock principal (non transféré vers un dépôt)">Voir</a>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <script>
            $(document).ready(function() {
                document.getElementById('stockSearch').addEventListener('input', function(e) {
                    const searchTerm = e.target.value.toLowerCase();
                    const rows = document.querySelectorAll('#stockAddTable tbody tr');
                    rows.forEach(row => {
                        const searchData = row.getAttribute('data-search');
                        row.style.display = searchData.includes(searchTerm) ? '' : 'none';
                    });
                });
            });
        </script>
        <style>
            table tr:hover {
                background-color: #F4A26120;
            }
            input:focus, select:focus {
                outline: none;
                border-color: #F4A261;
                box-shadow: 0 0 5px rgba(244, 162, 97, 0.5);
            }
            button:hover, a:hover {
                background-color: #F4A261;
                color: #4B2F1F;
                transform: translateY(-2px);
                box-shadow: 0 4px 8px rgba(0,0,0,0.3);
            }
            table {
                border-collapse: collapse;
            }
            table th, table td {
                border: none;
            }
            body {
                margin: 0;
                padding: 0;
                overflow-x: hidden;
            }
            .container {
                width: 100% !important;
                max-width: 100% !important;
                margin: 0 !important;
                left: 0 !important;
            }
            .container > div {
                width: 100% !important;
                max-width: 100% !important;
                margin: 0 !important;
            }
            #stockAddTable {
                width: 100%;
            }
            @media (max-width: 768px) {
                .container {
                    padding: 10px;
                    width: 100% !important;
                    max-width: 100% !important;
                }
                .container > div {
                    max-width: 100% !important;
                }
                h1, h2 {
                    font-size: 20px;
                }
                h3 {
                    font-size: 16px;
                }
                label, select, input, button, table th, table td, a {
                    font-size: 13px;
                }
            }
            @media (max-width: 480px) {
                .container {
                    padding: 5px;
                    width: 100% !important;
                    max-width: 100% !important;
                }
                .container > div {
                    max-width: 100% !important;
                }
                h1, h2 {
                    font-size: 18px;
                }
                h3 {
                    font-size: 14px;
                }
                label, select, input, button, table th, table td, a {
                    font-size: 12px;
                }
            }
        </style>
    </div>
</body>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
