<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to DGA/Leadership roles
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in add_main_stock.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
if (!$stmt) {
    error_log("Role query preparation failed in add_main_stock.php: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur interne du serveur. Contactez l'administrateur.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
if (!in_array($role, ['DG', 'DGA', 'Board', 'Admin'], true)) {
    error_log("Unauthorized access to add_main_stock.php by user {$_SESSION['full_name']} (Role: $role)");
    // Use dashboard mapping to redirect correctly
    $dashboard_map = [
        'Board' => 'admin_dashboard.php',
        'DG' => 'admin_dashboard.php',
        'DGA' => 'admin_dashboard.php',
        'Admin' => 'admin_dashboard.php',
        'Operation Manager' => 'operation_dashboard.php',
        'Stock Manager' => 'stock_manager_dashboard.php',
        'Accountant' => 'accountant_dashboard.php',
        'Caissier_Comptable' => 'cashier_dashboard.php',
    ];
    $dashboard = $dashboard_map[$role] ?? 'admin_dashboard.php';
    header("Location: /masunzu_bar_hotel/dashboards/" . $dashboard);
    exit;
}

// Fetch products for dropdown
$stmt = $conn->prepare("SELECT id, name, crate_quantity FROM product ORDER BY name");
if (!$stmt) {
    error_log("Product query preparation failed in add_main_stock.php: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur interne du serveur. Contactez l'administrateur.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt->execute();
$products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch main stock overview
$stock_items = [];
$stock_stmt = $conn->prepare("
    SELECT 
        p.id,
        p.name,
        p.volume_cl,
        p.crate_quantity,
        COALESCE(ms.quantity, 0) AS quantity,
        ms.last_updated
    FROM product p
    LEFT JOIN main_stock ms ON p.id = ms.product_id
    WHERE ms.quantity IS NOT NULL
    ORDER BY p.name
");
if ($stock_stmt) {
    $stock_stmt->execute();
    $stock_items = $stock_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stock_stmt->close();
}

$errors = [];
$success = '';

if (isset($_SESSION['stock_success'])) {
    $success = $_SESSION['stock_success'];
    unset($_SESSION['stock_success']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id = intval($_POST['product_id'] ?? 0);
    $quantity = (int)($_POST['quantity'] ?? 0);

    // Fetch product details
    $stmt = $conn->prepare("SELECT name, crate_quantity FROM product WHERE id = ?");
    if (!$stmt) {
        error_log("Product fetch query preparation failed in add_main_stock.php: " . $conn->error);
        $errors[] = "Erreur interne du serveur. Contactez l'administrateur.";
    } else {
        $stmt->bind_param("i", $product_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows === 0) {
            $errors[] = "Produit sélectionné invalide.";
        } else {
            $product = $result->fetch_assoc();
            $product_name = $product['name'];
            $crate_quantity = $product['crate_quantity'] ?? 12;
        }
        $stmt->close();
    }

    // Validation
    if ($quantity <= 0) {
        $errors[] = "La quantité doit être un nombre positif.";
    }
    if (empty($errors)) {
        $quantity_in_crates = $quantity;
        // Check if product exists in main_stock
        $stmt = $conn->prepare("SELECT quantity FROM main_stock WHERE product_id = ?");
        if (!$stmt) {
            error_log("Main stock check query preparation failed in add_main_stock.php: " . $conn->error);
            $errors[] = "Erreur interne du serveur. Contactez l'administrateur.";
        } else {
            $stmt->bind_param("i", $product_id);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                // Update existing stock
                $current_quantity = $result->fetch_assoc()['quantity'];
                $new_quantity = $current_quantity + $quantity_in_crates;
                $stmt = $conn->prepare("UPDATE main_stock SET quantity = ?, unit_type = 'crates' WHERE product_id = ?");
                $stmt->bind_param("ii", $new_quantity, $product_id);
            } else {
                // Insert new stock
                $stmt = $conn->prepare("INSERT INTO main_stock (product_id, quantity, unit_type) VALUES (?, ?, 'crates')");
                $stmt->bind_param("ii", $product_id, $quantity_in_crates);
            }
            if ($stmt->execute()) {
                $formatted_quantity = format_quantity($quantity_in_crates);
                $success_msg = "Ajouté $formatted_quantity de " . htmlspecialchars($product_name, ENT_QUOTES, 'UTF-8') . " au stock principal.";
                log_action($_SESSION['user_id'], 'add_stock', "Ajouté $formatted_quantity de $product_name au stock principal", $product_id, $quantity_in_crates);
                $_SESSION['stock_success'] = $success_msg;
                $conn->close();
                header("Location: /masunzu_bar_hotel/modules/operation/add_main_stock.php");
                exit;
            } else {
                error_log("Failed to add stock in add_main_stock.php: " . $stmt->error);
                $errors[] = "Échec de l'ajout au stock. Veuillez réessayer.";
            }
            $stmt->close();
        }
    }
}

$conn->close();
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        :root {
            --bg: #F9F6F1;
            --panel: #ffffff;
            --text: #3A2415;
            --muted: #7A5D46;
            --border: #D9C6B0;
            --accent: #F4A261;
            --accent-dark: #3A2415;
        }
        body {
            background: var(--bg);
        }
        .page-shell {
            min-height: calc(100vh - 70px);
            padding: 20px;
            box-sizing: border-box;
        }
        .layout {
            display: grid;
            gap: 24px;
            grid-template-columns: 350px 1fr;
        }
        .panel {
            background: var(--panel);
            border-radius: 14px;
            padding: 20px;
            box-shadow: 0 12px 30px rgba(58, 36, 21, 0.08);
        }
        .stock-panel {
            position: sticky;
            top: 90px;
            align-self: start;
            max-height: calc(100vh - 120px);
        }
        .panel h2,
        .panel h3 {
            margin: 0 0 14px;
            color: var(--text);
        }
        .stock-search {
            width: 100%;
            padding: 10px 12px;
            border: 1px solid var(--border);
            border-radius: 8px;
            margin-bottom: 12px;
            font-size: 14px;
            box-sizing: border-box;
        }
        .stock-list {
            list-style: none;
            margin: 0;
            padding: 0;
            max-height: calc(100vh - 240px);
            overflow-y: auto;
        }
        .stock-item {
            padding: 10px 0;
            border-bottom: 1px solid rgba(58, 36, 21, 0.08);
        }
        .stock-item h4 {
            margin: 0;
            font-size: 15px;
            color: var(--text);
        }
        .stock-meta {
            font-size: 12px;
            color: var(--muted);
            margin-top: 4px;
            line-height: 1.4;
        }
        .form-panel form {
            display: flex;
            flex-direction: column;
            gap: 16px;
        }
        .form-panel label {
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.03em;
            color: var(--muted);
            margin-bottom: 4px;
        }
        .form-panel select,
        .form-panel input {
            width: 100%;
            padding: 12px;
            border-radius: 10px;
            border: 1px solid var(--border);
            font-size: 14px;
            background: var(--panel);
            box-sizing: border-box;
        }
        .product-search-container {
            position: relative;
        }
        .product-search-input {
            width: 100%;
            padding: 12px;
            border-radius: 10px;
            border: 1px solid var(--border);
            font-size: 14px;
            background: var(--panel);
            box-sizing: border-box;
            cursor: text;
        }
        .product-search-input:focus {
            outline: none;
            border-color: var(--accent);
            box-shadow: 0 0 0 3px rgba(244, 162, 97, 0.1);
        }
        .product-dropdown {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: var(--panel);
            border: 1px solid var(--border);
            border-radius: 10px;
            margin-top: 4px;
            max-height: 300px;
            overflow-y: auto;
            z-index: 100;
            box-shadow: 0 8px 24px rgba(58, 36, 21, 0.15);
            display: none;
            pointer-events: none;
        }
        .product-dropdown.show {
            display: block;
            pointer-events: auto;
        }
        .product-option {
            padding: 12px;
            cursor: pointer;
            border-bottom: 1px solid rgba(58, 36, 21, 0.05);
            transition: background 0.15s ease;
            display: block;
            text-decoration: none;
            color: var(--text);
        }
        .product-option:last-child {
            border-bottom: none;
        }
        .product-option:hover,
        .product-option.selected {
            background: rgba(244, 162, 97, 0.1);
        }
        .product-option-name {
            font-weight: 500;
            font-size: 14px;
            color: var(--text);
        }
        .product-option-meta {
            font-size: 12px;
            color: var(--muted);
            margin-top: 2px;
        }
        .product-option.hidden {
            display: none;
        }
        .selected-product-display {
            display: none;
            margin-top: 8px;
            padding: 10px;
            background: rgba(244, 162, 97, 0.1);
            border-radius: 8px;
            font-size: 14px;
        }
        .selected-product-display.show {
            display: block;
        }
        .selected-product-display strong {
            color: var(--accent-dark);
        }
        .form-panel button {
            background: var(--accent-dark);
            color: #fff;
            border: none;
            padding: 15px;
            border-radius: 12px;
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.2s ease;
        }
        .form-panel button:hover {
            background: var(--accent);
            color: var(--accent-dark);
        }
        .alert {
            border-radius: 10px;
            padding: 14px;
            margin-bottom: 16px;
            font-size: 14px;
        }
        .alert.success {
            background: #E8F7EF;
            border: 1px solid #8ED6A2;
            color: #1F6C3A;
        }
        .alert.error {
            background: #FFEAEA;
            border: 1px solid #F2A5A5;
            color: #8B1E2D;
        }
        @media (max-width: 980px) {
            .layout {
                grid-template-columns: 1fr;
            }
            .stock-panel {
                position: static;
            }
            .stock-list {
                max-height: 300px;
            }
        }
        @media (max-width: 520px) {
            .page-shell {
                padding: 16px;
            }
            .panel {
                padding: 16px;
            }
            .stock-item h4 {
                font-size: 14px;
            }
        }
    </style>
</head>
<body>
    <div class="page-shell">
        <div class="layout">
            <aside class="panel stock-panel">
                <h3>Stock Principal Actuel</h3>
                <input type="text" id="stockSearch" class="stock-search" placeholder="Rechercher un produit...">
                <ul class="stock-list" id="stockList">
                    <?php if (empty($stock_items)): ?>
                        <li style="color: var(--muted); font-size: 13px;">Aucun stock enregistré pour le moment.</li>
                    <?php else: ?>
                        <?php foreach ($stock_items as $stock): ?>
                            <li class="stock-item" data-name="<?php echo htmlspecialchars(strtolower($stock['name']), ENT_QUOTES, 'UTF-8'); ?>">
                                <h4><?php echo htmlspecialchars($stock['name'], ENT_QUOTES, 'UTF-8'); ?></h4>
                                <div class="stock-meta">
                                    <?php echo number_format((int)$stock['quantity']); ?> caisses · <?php echo (int)$stock['crate_quantity']; ?> bouteilles/caisse
                                    <?php if (!empty($stock['last_updated'])): ?>
                                        <br>Mis à jour le <?php echo htmlspecialchars(date('d M Y', strtotime($stock['last_updated'])), ENT_QUOTES, 'UTF-8'); ?>
                                    <?php endif; ?>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </ul>
            </aside>

            <section class="panel form-panel">
                <h2>Ajouter au Stock Principal</h2>
                <?php if ($success): ?>
                    <div class="alert success"><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></div>
                <?php endif; ?>
                <?php if (!empty($errors)): ?>
                    <div class="alert error">
                        <ul style="margin:0; padding-left:18px;">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                <form method="POST" id="stockForm">
                    <div>
                        <label for="product_search">Sélectionner un Produit</label>
                        <div class="product-search-container">
                            <input 
                                type="text" 
                                id="product_search" 
                                class="product-search-input" 
                                placeholder="Tapez pour rechercher un produit..." 
                                autocomplete="off"
                            >
                            <input type="hidden" id="product_id" name="product_id" required>
                            <div class="product-dropdown" id="productDropdown">
                                <?php foreach ($products as $product): ?>
                                    <div 
                                        class="product-option" 
                                        data-id="<?php echo $product['id']; ?>"
                                        data-name="<?php echo htmlspecialchars(strtolower($product['name']), ENT_QUOTES, 'UTF-8'); ?>"
                                    >
                                        <div class="product-option-name"><?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?></div>
                                        <div class="product-option-meta"><?php echo $product['crate_quantity']; ?> bouteilles/caisse</div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            <div class="selected-product-display" id="selectedProductDisplay">
                                <strong>Produit sélectionné :</strong> <span id="selectedProductName"></span>
                            </div>
                        </div>
                    </div>
                    <div>
                        <label for="quantity">Quantité (caisses)</label>
                        <input type="number" id="quantity" name="quantity" min="1" step="1" required>
                    </div>
                    <button type="submit">Ajouter au Stock</button>
                </form>
            </section>
        </div>
    </div>
    <script>
        // Stock search functionality
        const searchInput = document.getElementById('stockSearch');
        const stockItems = document.querySelectorAll('#stockList .stock-item');
        if (searchInput && stockItems.length) {
            searchInput.addEventListener('input', function () {
                const query = this.value.trim().toLowerCase();
                stockItems.forEach(item => {
                    const name = item.dataset.name || '';
                    item.style.display = name.includes(query) ? 'block' : 'none';
                });
            });
        }

        // Product search functionality
        const productSearchInput = document.getElementById('product_search');
        const productDropdown = document.getElementById('productDropdown');
        const productIdInput = document.getElementById('product_id');
        const selectedProductDisplay = document.getElementById('selectedProductDisplay');
        const selectedProductName = document.getElementById('selectedProductName');
        const productOptions = document.querySelectorAll('.product-option');
        const stockForm = document.getElementById('stockForm');

        if (productSearchInput && productDropdown) {
            let selectedProduct = null;

            // Show/hide dropdown based on focus and input
            productSearchInput.addEventListener('focus', function() {
                if (!productIdInput.value) { // Only show if no product selected
                    filterProducts();
                    productDropdown.classList.add('show');
                    productDropdown.style.display = 'block';
                    productDropdown.style.pointerEvents = 'auto';
                }
            });

            productSearchInput.addEventListener('input', function() {
                // If user starts typing, clear selection and show dropdown
                if (productIdInput.value) {
                    productIdInput.value = '';
                    selectedProductDisplay.classList.remove('show');
                }
                filterProducts();
                productDropdown.classList.add('show');
                productDropdown.style.display = 'block';
                productDropdown.style.pointerEvents = 'auto';
            });

            // Hide dropdown when clicking outside
            document.addEventListener('click', function(e) {
                const container = productSearchInput.closest('.product-search-container');
                if (container && !container.contains(e.target)) {
                    productDropdown.classList.remove('show');
                    productDropdown.style.display = 'none';
                    productDropdown.style.pointerEvents = 'none';
                }
            });

            // Filter products based on search input
            function filterProducts() {
                const query = productSearchInput.value.trim().toLowerCase();
                let hasVisibleOptions = false;

                productOptions.forEach(option => {
                    const productName = option.dataset.name || '';
                    const productId = option.dataset.id || '';
                    const optionText = option.textContent.toLowerCase();

                    if (optionText.includes(query) || productName.includes(query)) {
                        option.classList.remove('hidden');
                        hasVisibleOptions = true;
                    } else {
                        option.classList.add('hidden');
                    }
                });

                // If no visible options, hide dropdown
                if (!hasVisibleOptions && query !== '') {
                    productDropdown.style.display = 'none';
                    productDropdown.style.pointerEvents = 'none';
                } else if (productDropdown.classList.contains('show')) {
                    productDropdown.style.display = 'block';
                    productDropdown.style.pointerEvents = 'auto';
                }
            }

            // Handle product selection
            productOptions.forEach(option => {
                option.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    
                    const productId = this.dataset.id;
                    const productText = this.querySelector('.product-option-name').textContent;
                    const productMeta = this.querySelector('.product-option-meta').textContent;

                    // Update hidden input
                    productIdInput.value = productId;

                    // Update display
                    selectedProductName.textContent = productText + ' (' + productMeta + ')';
                    selectedProductDisplay.classList.add('show');

                    // Update search input with selected product
                    productSearchInput.value = productText;

                    // Hide dropdown immediately and forcefully
                    productDropdown.classList.remove('show');
                    productDropdown.style.display = 'none';
                    productDropdown.style.pointerEvents = 'none';

                    // Remove previous selection highlight
                    productOptions.forEach(opt => opt.classList.remove('selected'));

                    // Highlight selected option
                    this.classList.add('selected');
                    selectedProduct = {
                        id: productId,
                        name: productText,
                        meta: productMeta
                    };

                    // Small delay before focusing to ensure dropdown is closed
                    setTimeout(function() {
                        document.getElementById('quantity').focus();
                    }, 100);
                });
            });

            // Form validation
            stockForm.addEventListener('submit', function(e) {
                if (!productIdInput.value || productIdInput.value === '') {
                    e.preventDefault();
                    alert('Veuillez sélectionner un produit.');
                    productSearchInput.focus();
                    productDropdown.classList.add('show');
                    return false;
                }
            });

            // Keyboard navigation
            let selectedIndex = -1;
            productSearchInput.addEventListener('keydown', function(e) {
                const visibleOptions = Array.from(productOptions).filter(opt => !opt.classList.contains('hidden'));

                if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    selectedIndex = Math.min(selectedIndex + 1, visibleOptions.length - 1);
                    visibleOptions.forEach((opt, idx) => {
                        opt.classList.toggle('selected', idx === selectedIndex);
                    });
                    if (visibleOptions[selectedIndex]) {
                        visibleOptions[selectedIndex].scrollIntoView({ block: 'nearest' });
                    }
                } else if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    selectedIndex = Math.max(selectedIndex - 1, -1);
                    visibleOptions.forEach((opt, idx) => {
                        opt.classList.toggle('selected', idx === selectedIndex);
                    });
                    if (visibleOptions[selectedIndex]) {
                        visibleOptions[selectedIndex].scrollIntoView({ block: 'nearest' });
                    }
                } else if (e.key === 'Enter' && selectedIndex >= 0 && visibleOptions[selectedIndex]) {
                    e.preventDefault();
                    visibleOptions[selectedIndex].click();
                } else if (e.key === 'Escape') {
                    productDropdown.classList.remove('show');
                    productDropdown.style.display = 'none';
                    productDropdown.style.pointerEvents = 'none';
                    selectedIndex = -1;
                }
            });
        }
    </script>
</body>

<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
