<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in manage_customers.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Restrict to authorized roles for creation (Marketing Agent, Stock Manager, DG, DGA, Admin)
// Restrict deletion to (Stock Manager, DG, DGA, Admin)
$stmt = $conn->prepare("SELECT u.id, u.full_name, r.name AS role_name, r.id AS role_id FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role_result = $stmt->get_result();
$user = $role_result->fetch_assoc();
$stmt->close();

$roleName = $user['role_name'] ?? '';
$can_create = in_array($roleName, ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant'], true);
$can_delete = in_array($roleName, ['DG', 'DGA', 'Admin', 'Stock Manager'], true);
$can_edit = $can_create;
// Cashiers can view but not edit/create/delete
$can_view = in_array($roleName, ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant', 'Caissier_Comptable'], true);
$loan_admin_roles = ['DG', 'DGA', 'Admin'];
$can_authorize_loan = in_array($roleName, $loan_admin_roles, true);
$allowed_user_types = ['moral', 'physical'];
$allowed_loan_statuses = ['inactive', 'active', 'suspended'];
if (!$can_view) {
    error_log("Unauthorized access to manage_customers.php by user {$user['full_name']} (Role ID: {$user['role_id']})");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $user['name'])) . "_dashboard.php");
    exit;
}

// Fetch province_id if not already set (handled by header for non-province roles)
$rolesWithoutDepot = ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant'];
if (!isset($_SESSION['province_id'])) {
    if (in_array($roleName, $rolesWithoutDepot, true)) {
        $_SESSION['province_id'] = null;
    } else {
        // Cashiers need province_id
        $stmt_province = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        $stmt_province->bind_param("i", $_SESSION['user_id']);
        $stmt_province->execute();
        $result = $stmt_province->get_result();
        $_SESSION['province_id'] = $result->fetch_assoc()['province_id'] ?? null;
        $stmt_province->close();
        // For cashiers, province_id is required but we don't block access if missing
        // They just won't see province-filtered data
    }
} elseif (in_array($roleName, $rolesWithoutDepot, true)) {
    $_SESSION['province_id'] = null;
}
$province_id = $_SESSION['province_id'] ?? null;

function normalize_customer_loan_balances(array &$customers): void {
    // Use ledger_outstanding as source of truth, fallback to loan_balance_bif
    // Don't use amount_due from unpaid_invoices as it may be stale
    foreach ($customers as &$customer) {
        $ledgerOutstanding = (int)($customer['ledger_outstanding'] ?? 0);
        $loanBalance = (int)($customer['loan_balance_bif'] ?? 0);
        // Use ledger if available, otherwise use loan_balance_bif
        $customer['loan_balance_bif'] = $ledgerOutstanding > 0 ? $ledgerOutstanding : $loanBalance;
        // Also update amount_due to match ledger_outstanding for consistency
        $customer['amount_due'] = $ledgerOutstanding > 0 ? $ledgerOutstanding : $loanBalance;
    }
    unset($customer);
}

$loanAmountDueExpr = mbh_get_loan_outstanding_sql('c', 'ui');

// Ensure customer table has is_active field
try {
    $check_is_active = $conn->query("SHOW COLUMNS FROM customer WHERE Field = 'is_active'");
    if ($check_is_active->num_rows == 0) {
        $conn->query("ALTER TABLE customer ADD COLUMN is_active TINYINT(1) NOT NULL DEFAULT 1 AFTER loan_last_review_at");
        error_log("Added is_active column to customer table");
    }
} catch (Exception $e) {
    error_log("Error checking/adding is_active column: " . $e->getMessage());
}

// Fetch all customers with amount due (sum of unpaid loan invoices)
// Use customer_loan_ledger as source of truth for loan balance
// Customer summary counts by type
$customers = [];
$stmt_customers = $conn->prepare("
    SELECT 
        c.id, 
        c.full_name, 
        c.nif, 
        c.tel, 
        c.driver_name, 
        c.driver_contacts, 
        c.plate_numbers, 
        c.additional_contacts, 
        c.address, 
        c.user_type, 
        c.can_pay_by_loan, 
        c.loan_limit_bif, 
        c.loan_balance_bif, 
        c.loan_terms_days, 
        c.loan_status, 
        c.loan_last_review_at, 
        COALESCE(c.is_active, 1) as is_active,
        {$loanAmountDueExpr} AS amount_due,
        COALESCE(GREATEST(
            COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
            COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
            0
        ), c.loan_balance_bif, 0) AS ledger_outstanding,
        creator.full_name AS created_by_name,
        c.created_at
    FROM customer c
    LEFT JOIN customer_loan_ledger cll ON (
        cll.customer_id = c.id
        AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
    )
    LEFT JOIN unpaid_invoices ui ON (ui.customer_id = c.id OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel))
    LEFT JOIN user creator ON c.created_by = creator.id
    GROUP BY c.id, c.full_name, c.nif, c.tel, c.driver_name, c.driver_contacts, c.plate_numbers, c.additional_contacts, c.address, c.user_type, c.can_pay_by_loan, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status, c.loan_last_review_at, c.is_active, creator.full_name, c.created_at
    ORDER BY c.created_at DESC
");
$stmt_customers->execute();
$customers = $stmt_customers->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt_customers->close();

normalize_customer_loan_balances($customers);

$customer_counts = [
    'total' => count($customers),
    'types' => [
        'physical' => 0,
        'moral' => 0
    ]
];
foreach ($customers as $cust) {
    $typeKey = strtolower($cust['user_type'] ?? 'physical');
    if (!isset($customer_counts['types'][$typeKey])) {
        $customer_counts['types'][$typeKey] = 0;
    }
    $customer_counts['types'][$typeKey]++;
}

$errors = [];
$successMessages = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_customer'])) {
    // Safety check: If an ID is present, this might be an edit form submission
    $id_check = intval($_POST['id'] ?? 0);
    if ($id_check > 0) {
        error_log("WARNING: Create customer form submitted with ID=$id_check. This might be an edit form. Ignoring create request.");
        $errors[] = "Erreur: Impossible de créer le client - champ ID détecté. Si vous essayiez de modifier, veuillez utiliser le bouton Modifier.";
    } else {
        $full_name = trim($_POST['full_name'] ?? '');
        $nif = trim($_POST['nif'] ?? '');
        $tel = trim($_POST['tel'] ?? '');
        $driver_name = trim($_POST['driver_name'] ?? '');
        $driver_contacts = trim($_POST['driver_contacts'] ?? '');
        $plate_numbers = trim($_POST['plate_numbers'] ?? '');
        $additional_contacts = trim($_POST['additional_contacts'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $requested_user_type = strtolower(trim($_POST['user_type'] ?? 'physical'));
        $user_type = in_array($requested_user_type, $allowed_user_types, true) ? $requested_user_type : 'physical';

        if (empty($full_name)) $errors[] = "Le nom complet est requis.";
        if (!empty($tel) && !preg_match('/^\d{8}$/', $tel)) $errors[] = "Format de numéro de téléphone invalide. Veuillez utiliser exactement 8 chiffres (ex: 12345678).";
        if (!empty($nif) && !preg_match('/^\d{9,}$/', $nif)) $errors[] = "Format NIF invalide (chiffres uniquement, au moins 9 chiffres).";
        if (!empty($nif)) {
            $stmt_check_nif = $conn->prepare("SELECT id FROM customer WHERE nif = ? AND id != 0");
            $stmt_check_nif->bind_param("s", $nif);
            $stmt_check_nif->execute();
            if ($stmt_check_nif->get_result()->num_rows > 0) {
                $errors[] = "Le NIF '$nif' est déjà utilisé par un autre client.";
            }
            $stmt_check_nif->close();
        }

        if (empty($errors)) {
            $stmt_insert = $conn->prepare("INSERT INTO customer (full_name, nif, tel, driver_name, driver_contacts, plate_numbers, additional_contacts, address, user_type, can_pay_by_loan, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 0, ?)");
            $stmt_insert->bind_param("sssssssssi", $full_name, $nif, $tel, $driver_name, $driver_contacts, $plate_numbers, $additional_contacts, $address, $user_type, $_SESSION['user_id']);
            if ($stmt_insert->execute()) {
                $successMessages[] = "Client '{$full_name}' créé avec succès.";
                $stmt_refresh = $conn->prepare("
                    SELECT 
                        c.id, 
                        c.full_name, 
                        c.nif, 
                        c.tel, 
                        c.driver_name, 
                        c.driver_contacts, 
                        c.plate_numbers, 
                        c.additional_contacts, 
                        c.address, 
                        c.user_type, 
                        c.can_pay_by_loan, 
                        c.loan_limit_bif, 
                        c.loan_balance_bif, 
                        c.loan_terms_days, 
                        c.loan_status, 
                        c.loan_last_review_at, 
                        COALESCE(c.is_active, 1) as is_active,
                        {$loanAmountDueExpr} AS amount_due
                    FROM customer c
                    LEFT JOIN unpaid_invoices ui ON (ui.customer_id = c.id OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel))
                    GROUP BY c.id, c.full_name, c.nif, c.tel, c.driver_name, c.driver_contacts, c.plate_numbers, c.additional_contacts, c.address, c.user_type, c.can_pay_by_loan, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status, c.loan_last_review_at, c.is_active
                    ORDER BY c.full_name
                ");
                $stmt_refresh->execute();
                $customers = $stmt_refresh->get_result()->fetch_all(MYSQLI_ASSOC);
                normalize_customer_loan_balances($customers);
                normalize_customer_loan_balances($customers);
                $stmt_refresh->close();

                $stmt_log = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'create_customer', ?, NOW())");
                $description = "Created customer '$full_name' by user {$user['full_name']} (ID: {$_SESSION['user_id']})";
                $stmt_log->bind_param("is", $_SESSION['user_id'], $description);
                $stmt_log->execute();
                $stmt_log->close();
            } else {
                $errors[] = "Failed to create customer: " . $conn->error;
            }
            $stmt_insert->close();
        }
    }
}

// Handle activate/deactivate
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle_customer_status']) && $can_delete) {
    $customer_id = intval($_POST['customer_id'] ?? 0);
    $new_status = intval($_POST['new_status'] ?? 1);
    
    if ($customer_id > 0) {
        $stmt_select = $conn->prepare("SELECT full_name, COALESCE(is_active, 1) as current_status FROM customer WHERE id = ?");
        $stmt_select->bind_param("i", $customer_id);
        $stmt_select->execute();
        $customer = $stmt_select->get_result()->fetch_assoc();
        $stmt_select->close();
        
        if ($customer) {
            $stmt_update = $conn->prepare("UPDATE customer SET is_active = ? WHERE id = ?");
            $stmt_update->bind_param("ii", $new_status, $customer_id);
            if ($stmt_update->execute()) {
                $status_text = $new_status == 1 ? 'activated' : 'deactivated';
                $status_text_fr = $new_status == 1 ? 'activé' : 'désactivé';
                $successMessages[] = "Le client '{$customer['full_name']}' a été {$status_text_fr}.";
                
                // Refresh customer list
                $stmt_refresh = $conn->prepare("
                    SELECT 
                        c.id, 
                        c.full_name, 
                        c.nif, 
                        c.tel, 
                        c.driver_name, 
                        c.driver_contacts, 
                        c.plate_numbers, 
                        c.additional_contacts, 
                        c.address, 
                        c.user_type, 
                        c.can_pay_by_loan, 
                        c.loan_limit_bif, 
                        c.loan_balance_bif, 
                        c.loan_terms_days, 
                        c.loan_status, 
                        c.loan_last_review_at, 
                        COALESCE(c.is_active, 1) as is_active,
                        {$loanAmountDueExpr} AS amount_due
                    FROM customer c
                    LEFT JOIN unpaid_invoices ui ON (ui.customer_id = c.id OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel))
                    GROUP BY c.id, c.full_name, c.nif, c.tel, c.driver_name, c.driver_contacts, c.plate_numbers, c.additional_contacts, c.address, c.user_type, c.can_pay_by_loan, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status, c.loan_last_review_at, c.is_active
                    ORDER BY c.full_name
                ");
                $stmt_refresh->execute();
                $customers = $stmt_refresh->get_result()->fetch_all(MYSQLI_ASSOC);
                $stmt_refresh->close();
                
                $stmt_log = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'toggle_customer_status', ?, NOW())");
                $description = "Customer '{$customer['full_name']}' {$status_text} by user {$user['full_name']} (ID: {$_SESSION['user_id']})";
                $stmt_log->bind_param("is", $_SESSION['user_id'], $description);
                $stmt_log->execute();
                $stmt_log->close();
            } else {
                $errors[] = "Failed to update customer status: " . $conn->error;
            }
            $stmt_update->close();
        } else {
            $errors[] = "Client non trouvé.";
        }
    } else {
        $errors[] = "ID client invalide.";
    }
}

// Handle deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_customer']) && $can_delete) {
    $customer_id = intval($_POST['customer_id'] ?? 0);
    if ($customer_id > 0) {
        $stmt_select = $conn->prepare("SELECT full_name FROM customer WHERE id = ?");
        $stmt_select->bind_param("i", $customer_id);
        $stmt_select->execute();
        $customer = $stmt_select->get_result()->fetch_assoc();
        $stmt_select->close();

        $stmt_delete = $conn->prepare("DELETE FROM customer WHERE id = ?");
        $stmt_delete->bind_param("i", $customer_id);
        if ($stmt_delete->execute()) {
            $successMessages[] = "Client supprimé avec succès.";
            $stmt_refresh = $conn->prepare("SELECT id, full_name, nif, tel, driver_name, driver_contacts, plate_numbers, additional_contacts, address, user_type, can_pay_by_loan, loan_limit_bif, loan_balance_bif, loan_terms_days, loan_status, loan_last_review_at, COALESCE(is_active, 1) as is_active FROM customer ORDER BY full_name");
            $stmt_refresh->execute();
            $customers = $stmt_refresh->get_result()->fetch_all(MYSQLI_ASSOC);
            normalize_customer_loan_balances($customers);
            $stmt_refresh->close();

            $stmt_log = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'delete_customer', ?, NOW())");
            $description = "Deleted customer '{$customer['full_name']}' by user {$user['full_name']} (ID: {$_SESSION['user_id']})";
            $stmt_log->bind_param("is", $_SESSION['user_id'], $description);
            $stmt_log->execute();
            $stmt_log->close();
        } else {
            $errors[] = "Failed to delete customer: " . $conn->error;
        }
        $stmt_delete->close();
    }
}

// Handle edit form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['edit_customer']) && $can_edit) {
    // Ensure we have a valid customer ID - this is critical to prevent creating new customers
    $id = intval($_POST['id'] ?? 0);
    
    // Log for debugging (remove in production if needed)
    error_log("Edit customer attempt - ID from POST: " . ($_POST['id'] ?? 'NOT SET') . ", Parsed ID: $id");
    
    // If ID is missing or invalid, show error and stop processing immediately
    if ($id <= 0) {
        $errors[] = "ID client invalide (reçu: " . htmlspecialchars($_POST['id'] ?? 'vide', ENT_QUOTES, 'UTF-8') . "). Impossible de mettre à jour le client sans un ID valide. Veuillez fermer le formulaire d'édition et réessayer.";
    } else {
        // First verify the customer exists before processing
        $stmt_verify = $conn->prepare("SELECT id, full_name FROM customer WHERE id = ?");
        $stmt_verify->bind_param("i", $id);
        $stmt_verify->execute();
        $customer_exists = $stmt_verify->get_result()->fetch_assoc();
        $stmt_verify->close();
        
        if (!$customer_exists) {
            $errors[] = "Le client avec l'ID $id n'existe pas. Impossible de mettre à jour un client inexistant.";
        } else {
            $full_name = trim($_POST['full_name'] ?? '');
            $nif = trim($_POST['nif'] ?? '');
            $tel = trim($_POST['tel'] ?? '');
            $driver_name = trim($_POST['driver_name'] ?? '');
            $driver_contacts = trim($_POST['driver_contacts'] ?? '');
            $plate_numbers = trim($_POST['plate_numbers'] ?? '');
            $additional_contacts = trim($_POST['additional_contacts'] ?? '');
            $address = trim($_POST['address'] ?? '');
            $requested_user_type_edit = strtolower(trim($_POST['user_type'] ?? 'physical'));
            $loan_toggle_requested = isset($_POST['can_pay_by_loan']) ? 1 : 0;
            $loan_limit_raw = preg_replace('/[^\d]/', '', $_POST['loan_limit_bif'] ?? '');
            $loan_limit_bif = $loan_limit_raw === '' ? 0 : (int)$loan_limit_raw;
            $loan_terms_raw = trim($_POST['loan_terms_days'] ?? '');
            $loan_terms_days = $loan_terms_raw === '' ? 0 : max(0, (int)$loan_terms_raw);
            $loan_status = strtolower(trim($_POST['loan_status'] ?? 'inactive'));

            // Validate required fields
            if (empty($full_name)) {
                $errors[] = "Le nom complet est requis.";
            }
            
            // Validate phone number format if provided
            if (!empty($tel) && !preg_match('/^\d{8}$/', $tel)) {
                $errors[] = "Format de numéro de téléphone invalide. Veuillez utiliser exactement 8 chiffres (ex: 12345678).";
            }
            
            // Validate NIF format if provided
            if (!empty($nif) && !preg_match('/^\d{9,}$/', $nif)) {
                $errors[] = "Format NIF invalide (chiffres uniquement, au moins 9 chiffres).";
            }

            // Get existing customer data (including NIF to check if it changed)
            $stmt_existing = $conn->prepare("SELECT nif, can_pay_by_loan, loan_limit_bif, loan_balance_bif, loan_terms_days, loan_status, loan_last_review_at, address, user_type FROM customer WHERE id = ?");
            $stmt_existing->bind_param("i", $id);
            $stmt_existing->execute();
            $existing = $stmt_existing->get_result()->fetch_assoc();
            $stmt_existing->close();

            if (!$existing) {
                $errors[] = "L'enregistrement client sélectionné n'a pas pu être trouvé dans la base de données.";
            }
            
            // Re-check NIF duplicate only if NIF has changed
            if (!empty($nif) && empty($errors) && $id > 0 && $existing) {
                $existing_nif = trim($existing['nif'] ?? '');
                $new_nif = trim($nif);
                
                // Only check for duplicates if the NIF has actually changed
                if ($existing_nif !== $new_nif) {
                    $stmt_check_nif = $conn->prepare("SELECT id, full_name FROM customer WHERE nif = ? AND id != ?");
                    $stmt_check_nif->bind_param("si", $nif, $id);
                    $stmt_check_nif->execute();
                    $nif_result = $stmt_check_nif->get_result();
                    if ($nif_result->num_rows > 0) {
                        $duplicate = $nif_result->fetch_assoc();
                        $errors[] = "Le NIF '$nif' est déjà utilisé par un autre client (ID: {$duplicate['id']}, Nom: {$duplicate['full_name']}).";
                    }
                    $stmt_check_nif->close();
                }
            }

            // Set user type
            if ($existing) {
                $user_type = in_array($requested_user_type_edit, $allowed_user_types, true)
                    ? $requested_user_type_edit
                    : ($existing['user_type'] ?? 'physical');
            } else {
                $user_type = in_array($requested_user_type_edit, $allowed_user_types, true) ? $requested_user_type_edit : 'physical';
            }

            // Validate and set loan status
            if (empty($loan_status) || !in_array($loan_status, $allowed_loan_statuses, true)) {
                $loan_status = $existing['loan_status'] ?? 'inactive';
            }

            $current_outstanding = (int)($existing['loan_balance_bif'] ?? 0);

            // Process loan settings
            if (empty($errors) && $existing) {
                if (!$can_authorize_loan) {
                    // Non-authorized users cannot change loan settings
                    $can_pay_by_loan = (int)$existing['can_pay_by_loan'];
                    $loan_limit_bif = (int)$existing['loan_limit_bif'];
                    $loan_terms_days = (int)($existing['loan_terms_days'] ?? 0);
                    $loan_status = $existing['loan_status'] ?? 'inactive';
                } else {
                    // Authorized users can modify loan settings
                    $can_pay_by_loan = $loan_toggle_requested;
                    if ($can_pay_by_loan && $loan_limit_bif <= 0) {
                        $errors[] = "La limite de crédit doit être supérieure à zéro lorsque l'accès au crédit est activé.";
                    }
                    if ($loan_limit_bif < $current_outstanding) {
                        $errors[] = "La limite de crédit ne peut pas être inférieure au solde impayé actuel de " . number_format($current_outstanding, 0, ',', '.') . " BIF.";
                    }
                    if (!in_array($loan_status, $allowed_loan_statuses, true)) {
                        $errors[] = "Sélection de statut de crédit invalide.";
                    }
                    if (!$can_pay_by_loan) {
                        $loan_status = 'inactive';
                    } elseif ($loan_status === 'inactive') {
                        $loan_status = 'active';
                    }
                }
            }

            // Perform the update if no errors
            if (empty($errors) && $existing) {
                $loan_profile_changed = false;
                if ($can_authorize_loan) {
                    $loan_profile_changed = (
                        (int)$existing['can_pay_by_loan'] !== $can_pay_by_loan ||
                        (int)$existing['loan_limit_bif'] !== $loan_limit_bif ||
                        (int)($existing['loan_terms_days'] ?? 0) !== $loan_terms_days ||
                        ($existing['loan_status'] ?? 'inactive') !== $loan_status
                    );
                }
                $loan_last_review_at = $loan_profile_changed ? date('Y-m-d H:i:s') : ($existing['loan_last_review_at'] ?? null);

                $stmt_update = $conn->prepare("UPDATE customer SET full_name = ?, nif = ?, tel = ?, driver_name = ?, driver_contacts = ?, plate_numbers = ?, additional_contacts = ?, address = ?, user_type = ?, can_pay_by_loan = ?, loan_limit_bif = ?, loan_terms_days = ?, loan_status = ?, loan_last_review_at = ? WHERE id = ?");
                if ($stmt_update === false) {
                    $errors[] = "Failed to prepare customer update: " . $conn->error;
                } else {
                    $stmt_update->bind_param(
                        "sssssssssiiissi",
                        $full_name,
                        $nif,
                        $tel,
                        $driver_name,
                        $driver_contacts,
                        $plate_numbers,
                        $additional_contacts,
                        $address,
                        $user_type,
                        $can_pay_by_loan,
                        $loan_limit_bif,
                        $loan_terms_days,
                        $loan_status,
                        $loan_last_review_at,
                        $id
                    );
                    if ($stmt_update->execute()) {
                        // Verify the update affected exactly one row
                        if ($stmt_update->affected_rows === 0) {
                            $errors[] = "Aucune modification n'a été effectuée. Le client a peut-être été supprimé ou les données sont identiques.";
                        } else {
                            $successMessages[] = "Client '{$full_name}' mis à jour avec succès.";
                            $stmt_refresh = $conn->prepare("
                    SELECT 
                        c.id, 
                        c.full_name, 
                        c.nif, 
                        c.tel, 
                        c.driver_name, 
                        c.driver_contacts, 
                        c.plate_numbers, 
                        c.additional_contacts, 
                        c.address, 
                        c.user_type, 
                        c.can_pay_by_loan, 
                        c.loan_limit_bif, 
                        c.loan_balance_bif, 
                        c.loan_terms_days, 
                        c.loan_status, 
                        c.loan_last_review_at, 
                        COALESCE(c.is_active, 1) as is_active,
                        {$loanAmountDueExpr} AS amount_due
                    FROM customer c
                    LEFT JOIN unpaid_invoices ui ON (ui.customer_id = c.id OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel))
                    GROUP BY c.id, c.full_name, c.nif, c.tel, c.driver_name, c.driver_contacts, c.plate_numbers, c.additional_contacts, c.address, c.user_type, c.can_pay_by_loan, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status, c.loan_last_review_at, c.is_active
                    ORDER BY c.full_name
                ");
                            $stmt_refresh->execute();
                            $customers = $stmt_refresh->get_result()->fetch_all(MYSQLI_ASSOC);
                            normalize_customer_loan_balances($customers);
                            $stmt_refresh->close();

                            $stmt_log = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'edit_customer', ?, NOW())");
                            $description = "Edited customer '$full_name' (ID: $id) by user {$user['full_name']} (ID: {$_SESSION['user_id']})";
                            $stmt_log->bind_param("is", $_SESSION['user_id'], $description);
                            $stmt_log->execute();
                            $stmt_log->close();
                        }
                    } else {
                        $errors[] = "Failed to update customer: " . $stmt_update->error;
                    }
                    $stmt_update->close();
                }
            }
        }
    }
}

include __DIR__ . '/../../includes/header.php';
?>
<style>
body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; }
.customer-cards { grid-template-columns: repeat(auto-fill, minmax(320px, 1fr)); }
.customer-card { transition: transform 0.2s; }
.customer-card:hover { transform: translateY(-2px); box-shadow: 0 4px 8px rgba(0,0,0,0.15); }
@media (max-width: 768px) {
    .customers-content {
        max-width: 100% !important;
        padding: 15px !important;
    }
    .manage-customers-container {
        padding: 10px !important;
    }
    .customer-cards {
        grid-template-columns: 1fr !important;
    }
    form {
        grid-template-columns: 1fr !important;
    }
    .modal-content {
        width: 95% !important;
        max-width: none !important;
        margin: 10% auto !important;
        padding: 15px !important;
    }
    .modal-content form, .modal-content > div {
        grid-template-columns: 1fr !important;
    }
}
</style>
<div class="manage-customers-container" style="padding: 20px; min-height: 70vh;">
    <div class="customers-content" style="background-color: #FFFFFF; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); max-width: 100%; margin: 0;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px;">Gérer Clients</h2>
        <p style="color: #000000; font-size: 16px;">Ajouter ou supprimer des clients pour Masunzu Bar Hotel.</p>
        <?php if (!empty($errors)): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 10px; border-radius: 5px; border: 1px solid #FF0000; margin-bottom: 15px;">
                <?php foreach ($errors as $error): ?>
                    <p><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <?php if (!empty($successMessages)): ?>
            <div style="background-color: #4CAF50; color: #FFFFFF; padding: 10px; border-radius: 5px; margin-bottom: 15px;">
                <?php foreach ($successMessages as $message): ?>
                    <p style="margin: 0;"><?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <div style="display: flex; flex-wrap: wrap; gap: 12px; margin-bottom: 20px;">
            <div style="flex: 1; min-width: 180px; background: #fdfbf7; padding: 16px; border-radius: 10px; border: 1px solid #e0e0e0;">
                <strong style="text-transform: uppercase; letter-spacing: 0.2em; font-size: 12px; color: #4B2F1F;">Total clients</strong>
                <p style="margin: 6px 0 0; font-size: 26px; color: #2C1810;"><?php echo number_format($customer_counts['total'], 0, ',', '.'); ?></p>
            </div>
            <?php foreach ($customer_counts['types'] as $type => $count): ?>
                <div style="flex: 1; min-width: 180px; background: #fff; padding: 16px; border-radius: 10px; border: 1px solid #e0e0e0;">
                    <strong style="text-transform: uppercase; letter-spacing: 0.2em; font-size: 12px; color: #4B2F1F;"><?php echo htmlspecialchars(ucfirst($type), ENT_QUOTES, 'UTF-8'); ?> clients</strong>
                    <p style="margin: 6px 0 0; font-size: 26px; color: #2C1810;"><?php echo number_format($count, 0, ',', '.'); ?></p>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Customer Creation Card with Toggle -->
        <?php if ($can_create): ?>
            <div style="background:#fff; border-radius:12px; box-shadow:0 2px 8px rgba(0,0,0,0.06); margin-bottom:20px; overflow:hidden;">
                <div style="display:flex; justify-content:space-between; align-items:center; padding:16px 20px; background:#FDFBF7; border-bottom:1px solid #F0F0F0;">
                    <h3 style="color:#4B2F1F; font-size:18px; margin:0; font-weight:600;">Nouveau Client</h3>
                    <button type="button" id="toggleCreateCustomer" style="background:#4B2F1F; color:#F4F0E4; padding:10px 18px; border:none; border-radius:8px; cursor:pointer; font-size:14px; font-weight:600; display:flex; align-items:center; gap:8px; transition:all 0.2s;">
                        <span style="font-size:18px;">+</span> Ajouter un Client
                    </button>
                </div>
                <div id="createCustomerForm" style="display:none; padding:20px;">
                    <p style="color:#6B4F2F; font-size:13px; margin:0 0 15px; padding:10px; background:#FFF8E1; border-radius:6px; border-left:3px solid #F4A261;">
                        💡 Le numéro de téléphone doit contenir exactement 8 chiffres (ex: 12345678).
                    </p>
                    <form method="POST" style="display:grid; grid-template-columns:repeat(3, 1fr); gap:15px;">
                        <div>
                            <label for="full_name" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Nom Complet <span style="color:red;">*</span></label>
                            <input type="text" id="full_name" name="full_name" required placeholder="Nom complet du client" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="nif" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">NIF</label>
                            <input type="text" id="nif" name="nif" placeholder="Numéro d'identification fiscale" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="tel" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Téléphone</label>
                            <input type="text" id="tel" name="tel" placeholder="8 chiffres (ex: 12345678)" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="driver_name" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Nom Chauffeur</label>
                            <input type="text" id="driver_name" name="driver_name" placeholder="Nom du chauffeur" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="driver_contacts" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Contacts Chauffeur</label>
                            <input type="text" id="driver_contacts" name="driver_contacts" placeholder="Téléphone du chauffeur" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="plate_numbers" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Numéros Plaque</label>
                            <input type="text" id="plate_numbers" name="plate_numbers" placeholder="Plaques d'immatriculation" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="additional_contacts" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Contacts Additionnels</label>
                            <input type="text" id="additional_contacts" name="additional_contacts" placeholder="Autres contacts" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="address" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Adresse</label>
                            <input type="text" id="address" name="address" placeholder="Adresse complète" style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                        </div>
                        <div>
                            <label for="user_type" style="display:block; font-size:12px; font-weight:600; color:#4B2F1F; margin-bottom:6px;">Type Client <span style="color:red;">*</span></label>
                            <select id="user_type" name="user_type" required style="width:100%; padding:10px; border-radius:6px; border:1px solid #B08968; font-size:14px; box-sizing:border-box;">
                                <option value="physical" selected>Personne Physique</option>
                                <option value="moral">Personne Morale</option>
                            </select>
                        </div>
                        <div style="grid-column:span 3; display:flex; gap:10px; justify-content:flex-end; margin-top:10px; padding-top:15px; border-top:1px solid #F0F0F0;">
                            <button type="button" onclick="document.getElementById('createCustomerForm').style.display='none'; document.getElementById('toggleCreateCustomer').innerHTML='<span style=font-size:18px>+</span> Ajouter un Client';" style="background:#f0f0f0; color:#4B2F1F; padding:11px 24px; border:none; border-radius:6px; cursor:pointer; font-size:14px; font-weight:500;">Annuler</button>
                            <button type="submit" name="create_customer" style="background:#4B2F1F; color:#F4F0E4; padding:11px 24px; border:none; border-radius:6px; cursor:pointer; font-size:14px; font-weight:600;">✓ Créer le Client</button>
                        </div>
                    </form>
                </div>
            </div>
        <?php endif; ?>

        <!-- Search Input -->
        <div style="margin-top: 20px; margin-bottom: 15px;">
            <input type="text" id="searchInput" placeholder="Rechercher clients..." style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
        </div>

        <!-- Customer List -->
        <?php if (!empty($customers)): ?>
            <h3 style="color: #4B2F1F; font-size: 20px; margin-top: 30px;">Liste Clients</h3>

            <!-- Filters and Export -->
            <div style="margin-bottom: 15px; display: flex; gap: 15px; align-items: center; flex-wrap: wrap;">
                <div>
                    <label for="filterType" style="color: #4B2F1F; font-weight: 600; font-size: 14px;">Type Client:</label>
                    <select id="filterType" style="padding: 5px 10px; border-radius: 4px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 14px; color: #4B2F1F;">
                        <option value="">Tous Types</option>
                        <option value="physical">Personne Physique</option>
                        <option value="moral">Personne Morale</option>
                    </select>
                </div>
                <div>
                    <label for="filterLoan" style="color: #4B2F1F; font-weight: 600; font-size: 14px;">Éligible Crédit:</label>
                    <select id="filterLoan" style="padding: 5px 10px; border-radius: 4px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 14px; color: #4B2F1F;">
                        <option value="">Tous</option>
                        <option value="1">Oui</option>
                        <option value="0">Non</option>
                    </select>
                </div>
                <div>
                    <label for="filterUnpaid" style="color: #4B2F1F; font-weight: 600; font-size: 14px;">Crédits Impayés:</label>
                    <select id="filterUnpaid" style="padding: 5px 10px; border-radius: 4px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 14px; color: #4B2F1F;">
                        <option value="">Tous</option>
                        <option value="1">A des Impayés</option>
                        <option value="0">Aucun Impayé</option>
                    </select>
                </div>
                <button id="applyFiltersBtn" style="background-color: #4CAF50; color: #FFFFFF; padding: 8px 15px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 600;">Appliquer Filtres</button>
                <button id="resetFiltersBtn" style="background-color: #FF9800; color: #FFFFFF; padding: 8px 15px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 600;">Réinitialiser Filtres</button>
                <button id="exportCsvBtn" style="background-color: #4B2F1F; color: #F4F0E4; padding: 8px 15px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px;">Exporter en CSV</button>
            </div>

            <div style="overflow-x: auto; margin-top: 15px;">
                <div class="customer-table-wrapper" style="max-height: 720px; overflow-y: auto; border: 1px solid #ddd; border-radius: 8px; background-color: #FFFFFF; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
                    <table id="customerTable" style="width: 100%; border-collapse: collapse; background-color: transparent;">
                        <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">#</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Nom Complet</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">NIF</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Tél</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Type</th>
                        <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Statut</th>
                        <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Crédit Actif</th>
                        <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Éligible Crédit</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Statut Crédit</th>
                            <th style="padding: 12px; text-align: right; border: 1px solid #ddd; font-size: 14px;">Montant Dû</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($customers as $index => $customer): ?>
                            <?php 
                            // Use ledger_outstanding as source of truth (same as loan_management.php)
                            $ledgerOutstanding = (int)($customer['ledger_outstanding'] ?? 0);
                            $loanBalance = (int)($customer['loan_balance_bif'] ?? 0);
                            $outstanding = $ledgerOutstanding > 0 ? $ledgerOutstanding : $loanBalance;
                            $has_unpaid = $outstanding > 0 ? '1' : '0';
                            ?>
                            <tr class="customer-row" data-customer-id="<?php echo $customer['id']; ?>" data-user-type="<?php echo $customer['user_type'] ?? 'physical'; ?>" data-loan-eligible="<?php echo $customer['can_pay_by_loan'] ?? 0; ?>" data-is-active="<?php echo isset($customer['is_active']) ? (int)$customer['is_active'] : 1; ?>" data-has-unpaid="<?php echo $has_unpaid; ?>" data-outstanding="<?php echo $outstanding; ?>" style="border: 1px solid #ddd;">
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;"><?php echo $index + 1; ?></td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;"><?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;"><?php echo htmlspecialchars($customer['nif'] ?: '-', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;"><?php echo htmlspecialchars($customer['tel'] ?: '-', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;"><?php echo ($customer['user_type'] ?? 'physical') === 'moral' ? 'Personne Morale' : 'Personne Physique'; ?></td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;">
                                    <?php 
                                    $is_active = isset($customer['is_active']) ? (int)$customer['is_active'] : 1;
                                    $status_color = $is_active ? '#4CAF50' : '#FF0000';
                                    $status_text = $is_active ? 'Actif' : 'Désactivé';
                                    ?>
                                    <span style="color: <?php echo $status_color; ?>; font-weight: 600;"><?php echo $status_text; ?></span>
                                </td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;">
                                    <?php $has_active_credit = $outstanding > 0; ?>
                                    <span style="color: <?php echo $has_active_credit ? '#C62828' : '#2E7D32'; ?>; font-weight: 600;"><?php echo $has_active_credit ? 'Oui' : 'Non'; ?></span>
                                </td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;"><?php echo ($customer['can_pay_by_loan'] ?? 0) == 1 ? 'Oui' : 'Non'; ?></td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;">
                                    <?php
                                    $loan_status = strtolower($customer['loan_status'] ?? 'inactive');
                                    $loan_status_fr = ($loan_status === 'active') ? 'Actif' : (($loan_status === 'inactive') ? 'Inactif' : (($loan_status === 'suspended') ? 'Suspendu' : ucfirst($loan_status)));
                                    echo htmlspecialchars($loan_status_fr, ENT_QUOTES, 'UTF-8');
                                    ?>
                                </td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px; text-align: right; font-weight: <?php echo $outstanding > 0 ? '600' : '400'; ?>; color: <?php echo $outstanding > 0 ? '#DC3545' : '#28a745'; ?>;">
                                    <?php echo number_format($outstanding, 0, ',', '.'); ?> BIF
                                </td>
                                <td style="padding: 12px; border: 1px solid #ddd; font-size: 14px;">
                                    <div style="display: flex; gap: 5px; flex-wrap: wrap;">
                                        <button type="button" onclick="openViewModal(<?php echo $customer['id']; ?>)" style="background-color: #4B2F1F; color: #F4F0E4; padding: 6px 10px; border: none; border-radius: 3px; cursor: pointer; font-size: 12px;">Voir</button>
                                        <a href="/masunzu_bar_hotel/modules/customers/customer_transaction_history.php?customer_id=<?php echo $customer['id']; ?>" style="background-color: #2196F3; color: #FFFFFF; padding: 6px 10px; border: none; border-radius: 3px; cursor: pointer; font-size: 12px; text-decoration: none; display: inline-block;">Historique</a>
                                        <?php if ($can_edit): ?>
                                            <button type="button" onclick="openEditModal(<?php echo $customer['id']; ?>)" style="background-color: #4CAF50; color: #FFFFFF; padding: 6px 10px; border: none; border-radius: 3px; cursor: pointer; font-size: 12px;">Modifier</button>
                                        <?php endif; ?>
                                        <?php if ($can_delete): ?>
                                            <?php $is_active = isset($customer['is_active']) ? (int)$customer['is_active'] : 1; ?>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('Êtes-vous sûr de vouloir <?php echo $is_active ? 'désactiver' : 'activer'; ?> ce client ?');">
                                                <input type="hidden" name="customer_id" value="<?php echo $customer['id']; ?>">
                                                <input type="hidden" name="new_status" value="<?php echo $is_active ? 0 : 1; ?>">
                                                <button type="submit" name="toggle_customer_status" style="background-color: <?php echo $is_active ? '#FF9800' : '#4CAF50'; ?>; color: #FFFFFF; padding: 6px 10px; border: none; border-radius: 3px; cursor: pointer; font-size: 12px;">
                                                    <?php echo $is_active ? 'Désactiver' : 'Activer'; ?>
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                    </table>
                </div>
            </div>
        <?php else: ?>
            <p style="color: #666; margin-top: 20px; text-align: center;">Aucun client trouvé.</p>
        <?php endif; ?>
    </div>
</div>

<!-- View Modal -->
<div id="viewModal" class="modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); z-index: 1000;">
    <div class="modal-content" style="background-color: #F4F0E4; margin: 5% auto; padding: 20px; border-radius: 10px; width: 90%; max-width: 600px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); position: relative;">
        <span class="close" onclick="closeViewModal()" style="position: absolute; top: 10px; right: 15px; font-size: 24px; color: #FF0000; cursor: pointer;">&times;</span>
        <h3 style="color: #4B2F1F; font-size: 20px; margin-bottom: 15px;">Détails Client</h3>
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
            <div><strong>Nom Complet:</strong> <span id="view_full_name"></span></div>
            <div><strong>NIF:</strong> <span id="view_nif"></span></div>
            <div><strong>Tél:</strong> <span id="view_tel"></span></div>
            <div><strong>Nom Chauffeur:</strong> <span id="view_driver_name"></span></div>
            <div><strong>Contacts Chauffeur:</strong> <span id="view_driver_contacts"></span></div>
            <div><strong>Numéros Plaque:</strong> <span id="view_plate_numbers"></span></div>
            <div style="grid-column: span 2;"><strong>Contacts Additionnels:</strong> <span id="view_additional_contacts"></span></div>
            <div style="grid-column: span 2;"><strong>Adresse:</strong> <span id="view_address"></span></div>
            <div><strong>Type Client:</strong> <span id="view_user_type"></span></div>
            <div><strong>Éligible Crédit:</strong> <span id="view_can_pay_by_loan"></span></div>
            <div><strong>Créé par:</strong> <span id="view_created_by"></span></div>
            <div><strong>Limite Crédit (BIF):</strong> <span id="view_loan_limit_bif"></span></div>
            <div><strong>Crédit Impayé (BIF):</strong> <span id="view_loan_balance_bif"></span></div>
            <div><strong>Conditions Crédit (jours):</strong> <span id="view_loan_terms_days"></span></div>
            <div><strong>Statut Crédit:</strong> <span id="view_loan_status"></span></div>
            <div><strong>Dernière Révision Crédit:</strong> <span id="view_loan_last_review"></span></div>
        </div>
    </div>
</div>

<!-- Edit Modal -->
<div id="editModal" class="modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); z-index: 1000;">
    <div class="modal-content" style="background-color: #F4F0E4; margin: 5% auto; padding: 20px; border-radius: 10px; width: 90%; max-width: 600px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); position: relative;">
        <span class="close" onclick="closeEditModal()" style="position: absolute; top: 10px; right: 15px; font-size: 24px; color: #FF0000; cursor: pointer;">&times;</span>
        <h3 style="color: #4B2F1F; font-size: 20px; margin-bottom: 15px;">Modifier Client</h3>
        <form method="POST" id="editForm" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;" onsubmit="return validateEditForm();">
            <input type="hidden" name="id" id="edit_id" required>
            <div>
                <label for="edit_full_name" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Nom Complet *</label>
                <input type="text" id="edit_full_name" name="full_name" required style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div>
                <label for="edit_nif" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">NIF</label>
                <input type="text" id="edit_nif" name="nif" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div>
                <label for="edit_tel" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Tel</label>
                <input type="text" id="edit_tel" name="tel" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div>
                <label for="edit_driver_name" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Nom Chauffeur</label>
                <input type="text" id="edit_driver_name" name="driver_name" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div>
                <label for="edit_driver_contacts" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Contacts Chauffeur</label>
                <input type="text" id="edit_driver_contacts" name="driver_contacts" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div>
                <label for="edit_plate_numbers" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Numéros Plaque</label>
                <input type="text" id="edit_plate_numbers" name="plate_numbers" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div>
                <label for="edit_additional_contacts" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Contacts Additionnels</label>
                <textarea id="edit_additional_contacts" name="additional_contacts" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px; resize: vertical;"></textarea>
            </div>
            <div>
                <label for="edit_address" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Adresse</label>
                <textarea id="edit_address" name="address" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px; resize: vertical;"></textarea>
            </div>
            <div>
                <label for="edit_user_type" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Type Client</label>
                <select id="edit_user_type" name="user_type" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;">
                    <option value="physical">Personne Physique</option>
                    <option value="moral">Personne Morale</option>
                </select>
            </div>
            <div>
                <label for="edit_can_pay_by_loan" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Autoriser Paiement par Crédit</label>
                <input type="checkbox" id="edit_can_pay_by_loan" name="can_pay_by_loan" style="margin-left: 10px; transform: scale(1.5);" <?php echo $can_authorize_loan ? '' : ' disabled'; ?>>
                <?php if (!$can_authorize_loan): ?>
                    <input type="hidden" name="can_pay_by_loan" id="edit_can_pay_by_loan_hidden">
                <?php endif; ?>
            </div>
            <div>
                <label for="edit_loan_limit_bif" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Limite Crédit (BIF)</label>
                <input type="number" id="edit_loan_limit_bif" name="loan_limit_bif" min="0" step="1" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;" <?php echo $can_authorize_loan ? '' : ' readonly'; ?>>
            </div>
            <div>
                <label for="edit_loan_balance_bif" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Crédit Impayé (BIF)</label>
                <input type="text" id="edit_loan_balance_bif" readonly style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #ECE7DA; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div>
                <label for="edit_loan_terms_days" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Conditions Crédit (jours)</label>
                <input type="number" id="edit_loan_terms_days" name="loan_terms_days" min="0" step="1" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;" <?php echo $can_authorize_loan ? '' : ' readonly'; ?>>
            </div>
            <div>
                <label for="edit_loan_status" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Statut Crédit</label>
                <select id="edit_loan_status" name="loan_status" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; height: 40px;" <?php echo $can_authorize_loan ? '' : ' disabled'; ?>>
                    <?php foreach ($allowed_loan_statuses as $statusOption): ?>
                        <option value="<?php echo htmlspecialchars($statusOption, ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars(ucfirst($statusOption), ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
                <?php if (!$can_authorize_loan): ?>
                    <input type="hidden" name="loan_status" id="edit_loan_status_hidden">
                <?php endif; ?>
            </div>
            <div>
                <label for="edit_loan_last_review" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Dernière Révision Crédit</label>
                <input type="text" id="edit_loan_last_review" readonly style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #ECE7DA; font-size: 16px; color: #4B2F1F; height: 40px;">
            </div>
            <div style="grid-column: span 2; text-align: center; margin-top: 20px;">
                <button type="submit" name="edit_customer" style="background-color: #4B2F1F; color: #F4F0E4; padding: 12px 20px; border: none; border-radius: 8px; cursor: pointer; font-size: 16px; width: 100%; max-width: 200px;">Enregistrer Modifications</button>
            </div>
        </form>
    </div>
</div>

<script>
const customersData = <?php echo json_encode($customers); ?>;
const canAuthorizeLoan = <?php echo $can_authorize_loan ? 'true' : 'false'; ?>;
const numberFormatter = new Intl.NumberFormat('fr-FR');

// Toggle create customer form
document.addEventListener('DOMContentLoaded', function() {
    const toggleBtn = document.getElementById('toggleCreateCustomer');
    const createForm = document.getElementById('createCustomerForm');
    
    if (toggleBtn && createForm) {
        toggleBtn.addEventListener('click', function() {
            const isHidden = createForm.style.display === 'none';
            createForm.style.display = isHidden ? 'block' : 'none';
            
            if (isHidden) {
                toggleBtn.innerHTML = '<span style="font-size:18px">−</span> Masquer le Formulaire';
                toggleBtn.style.background = '#6B4F2F';
                // Scroll to form smoothly
                createForm.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                // Focus on first input
                const firstInput = createForm.querySelector('input[type="text"]');
                if (firstInput) {
                    setTimeout(() => firstInput.focus(), 300);
                }
            } else {
                toggleBtn.innerHTML = '<span style="font-size:18px">+</span> Ajouter un Client';
                toggleBtn.style.background = '#4B2F1F';
            }
        });
    }
});

function formatBif(value) {
    const numericValue = Number(value || 0);
    return numberFormatter.format(Number.isFinite(numericValue) ? numericValue : 0);
}

function openViewModal(customerId) {
    const id = parseInt(customerId, 10);
    const customer = customersData.find(c => parseInt(c.id, 10) === id);
    if (!customer) {
        return;
    }
    document.getElementById('view_full_name').textContent = customer.full_name || '';
    document.getElementById('view_nif').textContent = customer.nif || '';
    document.getElementById('view_tel').textContent = customer.tel || '';
    document.getElementById('view_driver_name').textContent = customer.driver_name || '';
    document.getElementById('view_driver_contacts').textContent = customer.driver_contacts || '';
    document.getElementById('view_plate_numbers').textContent = customer.plate_numbers || '';
    document.getElementById('view_additional_contacts').textContent = customer.additional_contacts || '';
    document.getElementById('view_address').textContent = customer.address || '';
    document.getElementById('view_user_type').textContent = (customer.user_type || 'physical') === 'moral' ? 'Personne Morale' : 'Personne Physique';
    document.getElementById('view_can_pay_by_loan').textContent = Number(customer.can_pay_by_loan || 0) === 1 ? 'Oui' : 'Non';
    document.getElementById('view_created_by').textContent = customer.created_by_name || 'Inconnu';
    document.getElementById('view_loan_limit_bif').textContent = formatBif(customer.loan_limit_bif || 0);
    document.getElementById('view_loan_balance_bif').textContent = formatBif(customer.loan_balance_bif || 0);
    document.getElementById('view_loan_terms_days').textContent = Number(customer.loan_terms_days || 0) > 0 ? customer.loan_terms_days : '-';
    const loanStatus = (customer.loan_status || 'inactive').toLowerCase();
    document.getElementById('view_loan_status').textContent = loanStatus.charAt(0).toUpperCase() + loanStatus.slice(1);
    document.getElementById('view_loan_last_review').textContent = customer.loan_last_review_at ? customer.loan_last_review_at : '—';

    document.getElementById('viewModal').style.display = 'block';
}

function closeViewModal() {
    document.getElementById('viewModal').style.display = 'none';
}

function openEditModal(customerId) {
    const id = parseInt(customerId, 10);
    
    // Validate customer ID parameter
    if (!id || id <= 0 || isNaN(id)) {
        alert('Erreur: ID client invalide. Impossible d\'ouvrir le formulaire d\'édition.');
        return;
    }
    
    // Find customer in data
    const customer = customersData.find(c => parseInt(c.id, 10) === id);
    if (!customer || !customer.id || customer.id <= 0) {
        alert('Erreur: Client non trouvé. Impossible d\'ouvrir le formulaire d\'édition.');
        return;
    }
    
    // Ensure ID field is set and make it required
    const idField = document.getElementById('edit_id');
    if (!idField) {
        alert('Erreur: Champ ID du formulaire d\'édition non trouvé.');
        return;
    }
    
    // Set the ID value and ensure it's required
    idField.value = customer.id;
    idField.setAttribute('required', 'required');
    idField.setAttribute('readonly', 'readonly'); // Prevent accidental modification
    document.getElementById('edit_full_name').value = customer.full_name || '';
    document.getElementById('edit_nif').value = customer.nif || '';
    document.getElementById('edit_tel').value = customer.tel || '';
    document.getElementById('edit_driver_name').value = customer.driver_name || '';
    document.getElementById('edit_driver_contacts').value = customer.driver_contacts || '';
    document.getElementById('edit_plate_numbers').value = customer.plate_numbers || '';
    document.getElementById('edit_additional_contacts').value = customer.additional_contacts || '';
    document.getElementById('edit_address').value = customer.address || '';
    const editUserType = document.getElementById('edit_user_type');
    if (editUserType) {
        const currentType = (customer.user_type || 'physical').toLowerCase();
        editUserType.value = ['moral', 'physical'].includes(currentType) ? currentType : 'physical';
    }

    const loanEnabled = Number(customer.can_pay_by_loan || 0) === 1;
    const loanCheckbox = document.getElementById('edit_can_pay_by_loan');
    if (loanCheckbox) {
        loanCheckbox.checked = loanEnabled;
    }
    if (!canAuthorizeLoan) {
        const hiddenToggle = document.getElementById('edit_can_pay_by_loan_hidden');
        if (hiddenToggle) {
            hiddenToggle.value = loanEnabled ? 1 : 0;
        }
    }

    const loanLimitInput = document.getElementById('edit_loan_limit_bif');
    if (loanLimitInput) {
        loanLimitInput.value = Number(customer.loan_limit_bif || 0);
    }

    const loanBalanceInput = document.getElementById('edit_loan_balance_bif');
    if (loanBalanceInput) {
        loanBalanceInput.value = formatBif(customer.loan_balance_bif || 0);
    }

    const loanTermsInput = document.getElementById('edit_loan_terms_days');
    if (loanTermsInput) {
        loanTermsInput.value = Number(customer.loan_terms_days || 0);
    }

    const loanStatusValue = (customer.loan_status || 'inactive').toLowerCase();
    const loanStatusSelect = document.getElementById('edit_loan_status');
    if (loanStatusSelect) {
        loanStatusSelect.value = loanStatusValue;
    }
    if (!canAuthorizeLoan) {
        const hiddenStatus = document.getElementById('edit_loan_status_hidden');
        if (hiddenStatus) {
            hiddenStatus.value = loanStatusValue;
        }
    }

    const loanReviewInput = document.getElementById('edit_loan_last_review');
    if (loanReviewInput) {
        loanReviewInput.value = customer.loan_last_review_at ? customer.loan_last_review_at : '—';
    }

    document.getElementById('editModal').style.display = 'block';
}

function closeEditModal() {
    document.getElementById('editModal').style.display = 'none';
}

function validateEditForm() {
    const idField = document.getElementById('edit_id');
    const idValue = idField ? parseInt(idField.value, 10) : 0;
    const fullNameField = document.getElementById('edit_full_name');
    const fullName = fullNameField ? fullNameField.value.trim() : '';
    
    // Validate ID
    if (!idField || !idValue || idValue <= 0) {
        alert('Erreur: ID client invalide. Impossible d\'enregistrer les modifications. Veuillez fermer et rouvrir le formulaire d\'édition.');
        return false;
    }
    
    // Validate required fields
    if (!fullName || fullName === '') {
        alert('Erreur: Le nom complet est requis.');
        if (fullNameField) {
            fullNameField.focus();
        }
        return false;
    }
    
    // Validate phone number format if provided
    const telField = document.getElementById('edit_tel');
    if (telField && telField.value.trim() !== '') {
        const tel = telField.value.trim();
        if (!/^\d{8}$/.test(tel)) {
            alert('Erreur: Le numéro de téléphone doit contenir exactement 8 chiffres (ex: 12345678).');
            telField.focus();
            return false;
        }
    }
    
    // Validate NIF format if provided
    const nifField = document.getElementById('edit_nif');
    if (nifField && nifField.value.trim() !== '') {
        const nif = nifField.value.trim();
        if (!/^\d{9,}$/.test(nif)) {
            alert('Erreur: Le NIF doit contenir uniquement des chiffres et avoir au moins 9 chiffres.');
            nifField.focus();
            return false;
        }
    }
    
    return true;
}

function closeViewModal() {
    document.getElementById('viewModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const editModal = document.getElementById('editModal');
    const viewModal = document.getElementById('viewModal');
    if (event.target === editModal) {
        editModal.style.display = 'none';
    }
    if (event.target === viewModal) {
        viewModal.style.display = 'none';
    }
};

// Keep hidden loan fields synced for users without authorization
document.getElementById('editForm').addEventListener('change', function() {
    if (!canAuthorizeLoan) {
        const loanCheckbox = document.getElementById('edit_can_pay_by_loan');
        const hiddenToggle = document.getElementById('edit_can_pay_by_loan_hidden');
        if (loanCheckbox && hiddenToggle) {
            hiddenToggle.value = loanCheckbox.checked ? 1 : 0;
        }
        const loanStatusSelect = document.getElementById('edit_loan_status');
        const hiddenStatus = document.getElementById('edit_loan_status_hidden');
        if (loanStatusSelect && hiddenStatus) {
            hiddenStatus.value = loanStatusSelect.value;
        }
    }
});

// Filter and search functionality
function applyFilters() {
    const searchTerm = (document.getElementById('searchInput').value || '').toLowerCase();
    const filterType = document.getElementById('filterType').value || '';
    const filterLoan = document.getElementById('filterLoan').value || '';
    const filterUnpaid = document.getElementById('filterUnpaid').value || '';
    const rows = document.querySelectorAll('.customer-row');
    
    let visibleCount = 0;
    
    rows.forEach(row => {
        const rowText = row.textContent.toLowerCase();
        const userType = row.getAttribute('data-user-type') || '';
        const loanEligible = row.getAttribute('data-loan-eligible') || '0';
        const hasUnpaid = row.getAttribute('data-has-unpaid') || '0';
        
        // Check search term
        const matchesSearch = !searchTerm || rowText.includes(searchTerm);
        
        // Check type filter
        const matchesType = !filterType || userType === filterType;
        
        // Check loan filter
        const matchesLoan = !filterLoan || loanEligible === filterLoan;
        
        // Check unpaid filter
        const matchesUnpaid = !filterUnpaid || hasUnpaid === filterUnpaid;
        
        // Show row if all conditions match
        if (matchesSearch && matchesType && matchesLoan && matchesUnpaid) {
            row.style.display = '';
            visibleCount++;
        } else {
            row.style.display = 'none';
        }
    });
    
    // Update row numbers
    let rowNumber = 1;
    rows.forEach(row => {
        if (row.style.display !== 'none') {
            const firstCell = row.querySelector('td:first-child');
            if (firstCell) {
                firstCell.textContent = rowNumber++;
            }
        }
    });
}

// Search functionality
document.getElementById('searchInput').addEventListener('input', function() {
    applyFilters();
});

// Apply filters button
document.getElementById('applyFiltersBtn').addEventListener('click', function() {
    applyFilters();
});

// Reset filters button
document.getElementById('resetFiltersBtn').addEventListener('click', function() {
    document.getElementById('searchInput').value = '';
    document.getElementById('filterType').value = '';
    document.getElementById('filterLoan').value = '';
    document.getElementById('filterUnpaid').value = '';
    applyFilters();
});

// Export to CSV functionality
document.getElementById('exportCsvBtn').addEventListener('click', function() {
    const rows = document.querySelectorAll('.customer-row');
    const visibleRows = Array.from(rows).filter(row => row.style.display !== 'none');
    
    if (visibleRows.length === 0) {
        alert('Aucun client à exporter. Veuillez ajuster vos filtres.');
        return;
    }
    
    // CSV headers
    const headers = ['#', 'Nom Complet', 'NIF', 'Tél', 'Type Client', 'Éligible Crédit', 'Statut Crédit', 'Limite Crédit (BIF)', 'Crédit Impayé (BIF)', 'Montant Dû (BIF)', 'Conditions Crédit (jours)', 'Nom Chauffeur', 'Contacts Chauffeur', 'Numéros Plaque', 'Contacts Additionnels', 'Adresse'];
    
    // Build CSV content
    let csvContent = headers.join(',') + '\n';
    
    visibleRows.forEach(row => {
        const cells = row.querySelectorAll('td');
        const rowData = [];
        
        // Get basic info from table cells
        rowData.push(cells[0].textContent.trim()); // #
        rowData.push('"' + (cells[1].textContent.trim() || '').replace(/"/g, '""') + '"'); // Full Name
        rowData.push('"' + (cells[2].textContent.trim() || '').replace(/"/g, '""') + '"'); // NIF
        rowData.push('"' + (cells[3].textContent.trim() || '').replace(/"/g, '""') + '"'); // Tel
        rowData.push('"' + (cells[4].textContent.trim() || '').replace(/"/g, '""') + '"'); // Type
        rowData.push('"' + (cells[5].textContent.trim() || '').replace(/"/g, '""') + '"'); // Loan Eligible
        rowData.push('"' + (cells[6].textContent.trim() || '').replace(/"/g, '""') + '"'); // Loan Status
        
        // Get additional data from customersData array
        const customerId = parseInt(row.getAttribute('data-customer-id'), 10);
        const customer = customersData.find(c => parseInt(c.id, 10) === customerId);
        
        if (customer) {
            rowData.push(customer.loan_limit_bif ? formatBif(customer.loan_limit_bif) : '0'); // Loan Limit
            // Use ledger_outstanding as source of truth (same as loan_management.php)
            const ledgerOutstanding = parseInt(customer.ledger_outstanding || customer.loan_balance_bif || 0);
            const loanBalance = parseInt(customer.loan_balance_bif || 0);
            const outstanding = ledgerOutstanding > 0 ? ledgerOutstanding : loanBalance;
            rowData.push(outstanding ? formatBif(outstanding) : '0'); // Outstanding Loan (from ledger)
            rowData.push(outstanding ? formatBif(outstanding) : '0'); // Amount Due (same as outstanding for consistency)
            rowData.push(customer.loan_terms_days || '0'); // Loan Terms
            rowData.push('"' + (customer.driver_name || '').replace(/"/g, '""') + '"'); // Driver Name
            rowData.push('"' + (customer.driver_contacts || '').replace(/"/g, '""') + '"'); // Driver Contacts
            rowData.push('"' + (customer.plate_numbers || '').replace(/"/g, '""') + '"'); // Plate Numbers
            rowData.push('"' + (customer.additional_contacts || '').replace(/"/g, '""') + '"'); // Additional Contacts
            rowData.push('"' + (customer.address || '').replace(/"/g, '""') + '"'); // Address
        } else {
            // Fallback if customer data not found
            rowData.push('0', '0', '0', '0', '""', '""', '""', '""', '""');
        }
        
        csvContent += rowData.join(',') + '\n';
    });
    
    // Create filename with distinctive title and timestamp
    const now = new Date();
    const dateStr = now.toISOString().split('T')[0]; // YYYY-MM-DD
    const timeStr = now.toTimeString().split(' ')[0].replace(/:/g, '-'); // HH-MM-SS
    const filterType = document.getElementById('filterType').value;
    const filterLoan = document.getElementById('filterLoan').value;
    const filterUnpaid = document.getElementById('filterUnpaid').value;
    const searchTerm = document.getElementById('searchInput').value;
    
    let filenamePrefix = 'Masunzu_Bar_Hotel_Customers';
    if (filterType || filterLoan || filterUnpaid || searchTerm) {
        filenamePrefix += '_Filtered';
    }
    
    const filename = `${filenamePrefix}_${dateStr}_${timeStr}.csv`;
    
    // Create download link
    const blob = new Blob(['\ufeff' + csvContent], { type: 'text/csv;charset=utf-8;' }); // BOM for Excel compatibility
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
});
</script>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
