<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in customer_transaction_history.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$customer_id = isset($_GET['customer_id']) ? (int)$_GET['customer_id'] : 0;
if ($customer_id <= 0) {
    header("Location: /masunzu_bar_hotel/modules/customers/manage_customers.php");
    exit;
}

// Get customer info
$customer = null;
$stmt = $conn->prepare("SELECT id, full_name, nif, tel FROM customer WHERE id = ?");
if ($stmt) {
    $stmt->bind_param("i", $customer_id);
    $stmt->execute();
    $customer = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

if (!$customer) {
    header("Location: /masunzu_bar_hotel/modules/customers/manage_customers.php");
    exit;
}

// Date filters
$date_from = isset($_GET['date_from']) ? trim($_GET['date_from']) : date('Y-m-01'); // First day of current month
$date_to = isset($_GET['date_to']) ? trim($_GET['date_to']) : date('Y-m-d'); // Today

// Validate dates
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_from) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $date_to)) {
    $date_from = date('Y-m-01');
    $date_to = date('Y-m-d');
}

$date_from_datetime = $date_from . ' 00:00:00';
$date_to_datetime = $date_to . ' 23:59:59';

// Fetch paid invoices (sales)
$paid_invoices = [];
$stmt = $conn->prepare("
    SELECT 
        pi.id,
        pi.invoice_number,
        pi.created_at,
        pi.paid_at,
        pi.paid_amount,
        pi.subtotal_amount,
        pi.tax_amount,
        pi.payment_method,
        'paid' as transaction_type,
        'cash' as sale_type
    FROM paid_invoices pi
    WHERE (pi.customer_id = ? OR (pi.customer_id IS NULL AND pi.customer_name = ? AND pi.phone_number = ?))
    AND DATE(pi.created_at) BETWEEN ? AND ?
    ORDER BY pi.created_at DESC
");
if ($stmt) {
    $customer_name = $customer['full_name'];
    $customer_tel = $customer['tel'] ?? '';
    $stmt->bind_param("issss", $customer_id, $customer_name, $customer_tel, $date_from, $date_to);
    $stmt->execute();
    $paid_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Fetch unpaid invoices (loan sales)
$unpaid_invoices = [];
$stmt = $conn->prepare("
    SELECT 
        ui.id,
        ui.invoice_number,
        ui.created_at,
        ui.paid_amount,
        ui.subtotal_amount,
        ui.tax_amount,
        ui.is_loan_sale,
        ui.loan_outstanding_bif,
        ui.approval_status,
        'unpaid' as transaction_type,
        CASE WHEN ui.is_loan_sale = 1 THEN 'loan' ELSE 'pending' END as sale_type
    FROM unpaid_invoices ui
    WHERE (ui.customer_id = ? OR (ui.customer_id IS NULL AND ui.customer_name = ? AND ui.phone_number = ?))
    AND DATE(ui.created_at) BETWEEN ? AND ?
    ORDER BY ui.created_at DESC
");
if ($stmt) {
    $customer_name = $customer['full_name'];
    $customer_tel = $customer['tel'] ?? '';
    $stmt->bind_param("issss", $customer_id, $customer_name, $customer_tel, $date_from, $date_to);
    $stmt->execute();
    $unpaid_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Fetch loan repayments
$loan_repayments = [];
$stmt = $conn->prepare("
    SELECT 
        lp.id,
        lp.payment_amount,
        lp.payment_method,
        lp.payment_date,
        lp.notes,
        'repayment' as transaction_type
    FROM loan_payment lp
    WHERE lp.customer_id = ?
    AND DATE(lp.payment_date) BETWEEN ? AND ?
    ORDER BY lp.payment_date DESC
");
if ($stmt) {
    $stmt->bind_param("iss", $customer_id, $date_from, $date_to);
    $stmt->execute();
    $loan_repayments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Calculate totals
$total_sales = 0;
$total_loan_sales = 0;
$total_repayments = 0;

foreach ($paid_invoices as $inv) {
    $total_sales += (int)($inv['paid_amount'] ?? 0);
}

foreach ($unpaid_invoices as $inv) {
    if ($inv['is_loan_sale'] == 1 && ($inv['approval_status'] != 'rejected')) {
        $total_loan_sales += (int)($inv['paid_amount'] ?? 0);
    }
}

foreach ($loan_repayments as $repayment) {
    $total_repayments += (int)($repayment['payment_amount'] ?? 0);
}

// Combine all transactions for display
$all_transactions = [];
foreach ($paid_invoices as $inv) {
    $all_transactions[] = [
        'date' => $inv['created_at'],
        'type' => 'Vente (Payée)',
        'invoice_number' => $inv['invoice_number'],
        'amount' => (int)($inv['paid_amount'] ?? 0),
        'subtotal' => (int)($inv['subtotal_amount'] ?? 0),
        'tax' => (int)($inv['tax_amount'] ?? 0),
        'payment_method' => $inv['payment_method'] ?? 'N/A',
        'is_loan' => false
    ];
}

foreach ($unpaid_invoices as $inv) {
    if ($inv['is_loan_sale'] == 1 && ($inv['approval_status'] != 'rejected')) {
        $all_transactions[] = [
            'date' => $inv['created_at'],
            'type' => 'Vente (Crédit)',
            'invoice_number' => $inv['invoice_number'],
            'amount' => (int)($inv['paid_amount'] ?? 0),
            'subtotal' => (int)($inv['subtotal_amount'] ?? 0),
            'tax' => (int)($inv['tax_amount'] ?? 0),
            'payment_method' => 'Crédit',
            'is_loan' => true,
            'outstanding' => (int)($inv['loan_outstanding_bif'] ?? 0)
        ];
    }
}

foreach ($loan_repayments as $repayment) {
    $all_transactions[] = [
        'date' => $repayment['payment_date'],
        'type' => 'Remboursement Crédit',
        'invoice_number' => 'N/A',
        'amount' => (int)($repayment['payment_amount'] ?? 0),
        'subtotal' => (int)($repayment['payment_amount'] ?? 0),
        'tax' => 0,
        'payment_method' => $repayment['payment_method'] ?? 'N/A',
        'is_loan' => true,
        'notes' => $repayment['notes'] ?? ''
    ];
}

// Sort by date descending
usort($all_transactions, function($a, $b) {
    return strtotime($b['date']) - strtotime($a['date']);
});

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="customer_transactions_' . $customer_id . '_' . $date_from . '_to_' . $date_to . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // BOM for Excel compatibility
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Headers
    fputcsv($output, ['Date', 'Type', 'Numéro Commande', 'Montant (BIF)', 'Sous-Total (BIF)', 'Taxe (BIF)', 'Méthode Paiement', 'Impayé (BIF)', 'Notes']);
    
    // Data
    foreach ($all_transactions as $transaction) {
        fputcsv($output, [
            $transaction['date'],
            $transaction['type'],
            $transaction['invoice_number'],
            $transaction['amount'],
            $transaction['subtotal'],
            $transaction['tax'],
            $transaction['payment_method'],
            isset($transaction['outstanding']) ? $transaction['outstanding'] : '',
            isset($transaction['notes']) ? $transaction['notes'] : ''
        ]);
    }
    
    // Summary row
    fputcsv($output, []);
    fputcsv($output, ['Résumé']);
    fputcsv($output, ['Total Ventes (Payées)', '', '', $total_sales]);
    fputcsv($output, ['Total Ventes Crédit', '', '', $total_loan_sales]);
    fputcsv($output, ['Total Remboursements', '', '', $total_repayments]);
    fputcsv($output, ['Ventes Nettes', '', '', $total_sales + $total_loan_sales - $total_repayments]);
    
    fclose($output);
    exit;
}

include __DIR__ . '/../../includes/header.php';
?>
<style>
body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; }
</style>
<div style="padding: 20px; background-color: #f9f9f9; min-height: 70vh;">
    <div style="background-color: #FFFFFF; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); max-width: 1400px; margin: 0 auto;">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
            <div>
                <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 5px;">Historique Transactions</h2>
                <p style="color: #666; font-size: 16px; margin: 0;">Client: <strong><?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?></strong></p>
            </div>
            <a href="/masunzu_bar_hotel/modules/customers/manage_customers.php" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 20px; border-radius: 5px; text-decoration: none; font-weight: 600;">← Retour aux Clients</a>
        </div>

        <!-- Date Filter Form -->
        <form method="GET" style="background-color: #F4F0E4; padding: 20px; border-radius: 8px; margin-bottom: 20px; display: flex; gap: 15px; align-items: end; flex-wrap: wrap;">
            <input type="hidden" name="customer_id" value="<?php echo $customer_id; ?>">
            <div>
                <label for="date_from" style="color: #4B2F1F; font-weight: 600; font-size: 14px; display: block; margin-bottom: 5px;">Date Début</label>
                <input type="date" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from, ENT_QUOTES, 'UTF-8'); ?>" required style="padding: 8px; border-radius: 5px; border: 1px solid #4B2F1F; font-size: 14px;">
            </div>
            <div>
                <label for="date_to" style="color: #4B2F1F; font-weight: 600; font-size: 14px; display: block; margin-bottom: 5px;">Date Fin</label>
                <input type="date" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to, ENT_QUOTES, 'UTF-8'); ?>" required style="padding: 8px; border-radius: 5px; border: 1px solid #4B2F1F; font-size: 14px;">
            </div>
            <button type="submit" style="background-color: #4B2F1F; color: #F4F0E4; padding: 8px 20px; border: none; border-radius: 5px; cursor: pointer; font-size: 14px; font-weight: 600; height: 38px;">Filtrer</button>
            <a href="?customer_id=<?php echo $customer_id; ?>&date_from=<?php echo $date_from; ?>&date_to=<?php echo $date_to; ?>&export=csv" style="background-color: #4CAF50; color: #FFFFFF; padding: 8px 20px; border-radius: 5px; text-decoration: none; font-size: 14px; font-weight: 600; height: 38px; display: inline-flex; align-items: center;">Exporter CSV</a>
        </form>

        <!-- Summary Cards -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 30px;">
            <div style="background-color: #E3F2FD; padding: 20px; border-radius: 8px; border-left: 4px solid #2196F3;">
                <p style="margin: 0; color: #666; font-size: 12px; text-transform: uppercase;">Total Ventes (Payées)</p>
                <p style="margin: 5px 0 0; color: #2196F3; font-size: 24px; font-weight: 700;"><?php echo number_format($total_sales, 0, ',', '.'); ?> BIF</p>
            </div>
            <div style="background-color: #FFF3E0; padding: 20px; border-radius: 8px; border-left: 4px solid #FF9800;">
                <p style="margin: 0; color: #666; font-size: 12px; text-transform: uppercase;">Total Ventes Crédit</p>
                <p style="margin: 5px 0 0; color: #FF9800; font-size: 24px; font-weight: 700;"><?php echo number_format($total_loan_sales, 0, ',', '.'); ?> BIF</p>
            </div>
            <div style="background-color: #E8F5E9; padding: 20px; border-radius: 8px; border-left: 4px solid #4CAF50;">
                <p style="margin: 0; color: #666; font-size: 12px; text-transform: uppercase;">Total Remboursements</p>
                <p style="margin: 5px 0 0; color: #4CAF50; font-size: 24px; font-weight: 700;"><?php echo number_format($total_repayments, 0, ',', '.'); ?> BIF</p>
            </div>
            <div style="background-color: #F3E5F5; padding: 20px; border-radius: 8px; border-left: 4px solid #9C27B0;">
                <p style="margin: 0; color: #666; font-size: 12px; text-transform: uppercase;">Ventes Nettes</p>
                <p style="margin: 5px 0 0; color: #9C27B0; font-size: 24px; font-weight: 700;"><?php echo number_format($total_sales + $total_loan_sales - $total_repayments, 0, ',', '.'); ?> BIF</p>
            </div>
        </div>

        <!-- Transactions Table -->
        <h3 style="color: #4B2F1F; font-size: 20px; margin-bottom: 15px;">Transactions (<?php echo count($all_transactions); ?>)</h3>
        <?php if (!empty($all_transactions)): ?>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Date</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Type</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Numéro Commande</th>
                            <th style="padding: 12px; text-align: right; border: 1px solid #ddd; font-size: 14px;">Montant (BIF)</th>
                            <th style="padding: 12px; text-align: right; border: 1px solid #ddd; font-size: 14px;">Sous-Total (BIF)</th>
                            <th style="padding: 12px; text-align: right; border: 1px solid #ddd; font-size: 14px;">Taxe (BIF)</th>
                            <th style="padding: 12px; text-align: left; border: 1px solid #ddd; font-size: 14px;">Méthode Paiement</th>
                            <th style="padding: 12px; text-align: right; border: 1px solid #ddd; font-size: 14px;">Impayé (BIF)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($all_transactions as $transaction): ?>
                            <tr style="border-bottom: 1px solid #ddd;">
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px;"><?php echo date('Y-m-d H:i', strtotime($transaction['date'])); ?></td>
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px;">
                                    <span style="padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: 600; 
                                        background-color: <?php 
                                            echo ($transaction['type'] === 'Sale (Paid)' || $transaction['type'] === 'Vente (Payée)') ? '#E3F2FD' : 
                                                (($transaction['type'] === 'Sale (Loan)' || $transaction['type'] === 'Vente (Crédit)') ? '#FFF3E0' : '#E8F5E9'); 
                                        ?>; 
                                        color: <?php 
                                            echo ($transaction['type'] === 'Sale (Paid)' || $transaction['type'] === 'Vente (Payée)') ? '#2196F3' : 
                                                (($transaction['type'] === 'Sale (Loan)' || $transaction['type'] === 'Vente (Crédit)') ? '#FF9800' : '#4CAF50'); 
                                        ?>;">
                                        <?php
                                        $type_fr = ($transaction['type'] === 'Sale (Paid)') ? 'Vente (Payée)' : (($transaction['type'] === 'Sale (Loan)') ? 'Vente (Crédit)' : (($transaction['type'] === 'Loan Repayment') ? 'Remboursement Crédit' : $transaction['type']));
                                        echo htmlspecialchars($type_fr, ENT_QUOTES, 'UTF-8');
                                        ?>
                                    </span>
                                </td>
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px;"><?php echo htmlspecialchars($transaction['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px; text-align: right; font-weight: 600;"><?php echo number_format($transaction['amount'], 0, ',', '.'); ?></td>
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px; text-align: right;"><?php echo number_format($transaction['subtotal'], 0, ',', '.'); ?></td>
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px; text-align: right;"><?php echo number_format($transaction['tax'], 0, ',', '.'); ?></td>
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px; text-transform: capitalize;">
                                    <?php
                                    $payment_method = $transaction['payment_method'];
                                    $payment_method_fr = ($payment_method === 'Loan') ? 'Crédit' : (($payment_method === 'Cash') ? 'Espèces' : (($payment_method === 'Mobile') ? 'Mobile' : (($payment_method === 'Bank') ? 'Banque' : str_replace('_', ' ', $payment_method))));
                                    echo htmlspecialchars($payment_method_fr, ENT_QUOTES, 'UTF-8');
                                    ?>
                                </td>
                                <td style="padding: 10px; border: 1px solid #ddd; font-size: 14px; text-align: right; <?php echo isset($transaction['outstanding']) && $transaction['outstanding'] > 0 ? 'color: #DC3545; font-weight: 600;' : ''; ?>">
                                    <?php echo isset($transaction['outstanding']) ? number_format($transaction['outstanding'], 0, ',', '.') : '-'; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="color: #666; text-align: center; padding: 40px;">Aucune transaction trouvée pour la période sélectionnée.</p>
        <?php endif; ?>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
<?php $conn->close(); ?>

