<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

// Initialize database connection with error logging
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in view_manager_invoice.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    exit;
}

// Restrict to DGA or Admin
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
if ($stmt === false) {
    error_log("Prepare failed: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la préparation de la base de données. Vérifiez les logs.</p>";
    $conn->close();
    exit;
}
$stmt->bind_param("i", $_SESSION['user_id']);
if (!$stmt->execute()) {
    error_log("Execute failed: " . $stmt->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de l'exécution de la base de données. Vérifiez les logs.</p>";
    $stmt->close();
    $conn->close();
    exit;
}
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
if ($role !== 'DGA' && $role !== 'Admin') {
    error_log("Unauthorized access to view_manager_invoice.php by user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    $conn->close();
    exit;
}

// Fetch invoice details
$invoice = null;
if (isset($_GET['id'])) {
    $invoice_id = intval($_GET['id']);
    $stmt = $conn->prepare("
        SELECT ui.id, ui.invoice_number, ui.customer_name, ui.phone_number, ui.created_at, ui.approval_status, ui.created_by, u.full_name AS created_by_name, ui.approved_by, ua.full_name AS approved_by_name 
        FROM unpaid_invoices ui 
        LEFT JOIN user u ON ui.created_by = u.id 
        LEFT JOIN user ua ON ui.approved_by = ua.id 
        WHERE ui.id = ?
        UNION
        SELECT pi.id, pi.invoice_number, pi.customer_name, pi.phone_number, pi.created_at, pi.approval_status, pi.created_by, u.full_name AS created_by_name, pi.approved_by, ua.full_name AS approved_by_name 
        FROM paid_invoices pi 
        LEFT JOIN user u ON pi.created_by = u.id 
        LEFT JOIN user ua ON pi.approved_by = ua.id 
        WHERE pi.id = ?
    ");
    if ($stmt === false) {
        error_log("Prepare failed: " . $conn->error);
        http_response_code(500);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la préparation de la base de données. Vérifiez les logs.</p>";
        $conn->close();
        exit;
    }
    $stmt->bind_param("ii", $invoice_id, $invoice_id);
    if ($stmt->execute()) {
        $result = $stmt->get_result();
        $invoice = $result->fetch_assoc();
    } else {
        error_log("Execute failed: " . $stmt->error);
    }
    $stmt->close();
}

if (!$invoice) {
    http_response_code(404);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Commande introuvable.</p>";
    $conn->close();
    exit;
}

// Fetch invoice items
$items = [];
$stmt = $conn->prepare("SELECT ii.quantity, ii.unit_price, p.name, p.crate_quantity FROM invoice_items ii JOIN product p ON ii.product_id = p.id WHERE ii.invoice_id = ?");
if ($stmt === false) {
    error_log("Prepare failed: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la préparation de la base de données. Vérifiez les logs.</p>";
    $conn->close();
    exit;
}
$stmt->bind_param("i", $invoice_id);
if ($stmt->execute()) {
    $result = $stmt->get_result();
    $items = $result->fetch_all(MYSQLI_ASSOC);
} else {
    error_log("Execute failed: " . $stmt->error);
}
$stmt->close();

include __DIR__ . '/../../includes/header.php';
?>
<div style="padding: 20px; background-color: #F4F0E4; margin: 20px 0; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 800px; margin-left: auto; margin-right: auto;">
    <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px;">Détails Commande</h2>
    <p style="color: #000000; font-size: 16px;">Commande #: <?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Client: <?php echo htmlspecialchars($invoice['customer_name'], ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Téléphone: <?php echo htmlspecialchars($invoice['phone_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Date Création: <?php echo date('Y-m-d H:i', strtotime($invoice['created_at'])); ?></p>
    <p style="color: #000000; font-size: 16px;">Créé Par: <?php echo htmlspecialchars($invoice['created_by_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Approuvé Par: <?php echo htmlspecialchars($invoice['approved_by_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Statut Approbation: <?php 
    $status = $invoice['approval_status'];
    $status_fr = ($status === 'approved') ? 'Approuvé' : (($status === 'pending') ? 'En Attente' : (($status === 'rejected') ? 'Rejeté' : $status));
    echo htmlspecialchars($status_fr, ENT_QUOTES, 'UTF-8'); 
    ?></p>

    <h3 style="color: #4B2F1F; font-size: 20px; margin-top: 20px; margin-bottom: 15px;">Produits</h3>
    <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
        <thead>
            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                <th style="padding: 10px; text-align: left;">Nom Produit</th>
                <th style="padding: 10px; text-align: left;">Quantité</th>
                <th style="padding: 10px; text-align: left;">Prix Unitaire (BIF)</th>
                <th style="padding: 10px; text-align: left;">Total (BIF)</th>
            </tr>
        </thead>
        <tbody>
            <?php
            $total_amount = 0;
            foreach ($items as $item) {
                // quantity is stored in crates, unit_price is per crate
                $quantity_crates = (int)$item['quantity'];
                $total = $item['unit_price'] * $quantity_crates;
                $total_amount += $total;
                echo "<tr style='border-bottom: 1px solid #D3D3D3;'>";
                echo "<td style='padding: 10px;'>" . htmlspecialchars($item['name'], ENT_QUOTES, 'UTF-8') . "</td>";
                echo "<td style='padding: 10px;'>" . number_format($quantity_crates, 0, ',', '.') . " caisses</td>";
                echo "<td style='padding: 10px;'>" . number_format($item['unit_price'], 0, ',', '.') . "</td>";
                echo "<td style='padding: 10px;'>" . number_format($total, 0, ',', '.') . "</td>";
                echo "</tr>";
            }
            ?>
            <tr style="background-color: #F4F0E4; font-weight: bold;">
                <td colspan="3" style="padding: 10px; text-align: right;">Montant Total:</td>
                <td style="padding: 10px;"><?php echo number_format($total_amount, 2, ',', '.'); ?> BIF</td>
            </tr>
        </tbody>
    </table>
    <a href="/masunzu_bar_hotel/dashboards/admin_dashboard.php" style="color: #4B2F1F; text-decoration: underline;">Retour au Tableau de Bord</a>
</div>

<?php
include __DIR__ . '/../../includes/footer.php';
if ($conn && !$conn->connect_error) {
    $conn->close();
}
?>
