<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

// Initialize database connection with error logging
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in view_invoice.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    exit;
}

// Fetch province_id (optional, allow NULL for flexibility)
if (!isset($_SESSION['province_id'])) {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    if ($stmt === false) {
        error_log("Prepare failed: " . $conn->error);
        http_response_code(500);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la préparation de la base de données. Vérifiez les logs.</p>";
        $conn->close();
        exit;
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
        http_response_code(500);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de l'exécution de la base de données. Vérifiez les logs.</p>";
        $stmt->close();
        $conn->close();
        exit;
    }
    $result = $stmt->get_result();
    $province_row = $result->fetch_assoc();
    $_SESSION['province_id'] = isset($province_row['province_id']) ? $province_row['province_id'] : null;
    $stmt->close();
}
$province_id = isset($_SESSION['province_id']) ? $_SESSION['province_id'] : null;

// Restrict to Caissier_Comptable (role_id = 11) and Marketing Agent (role_id = 10)
$stmt = $conn->prepare("SELECT u.id, u.full_name, r.name, r.id AS role_id FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
if ($stmt === false) {
    error_log("Prepare failed: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la préparation de la base de données. Vérifiez les logs.</p>";
        $conn->close();
        exit;
    }
$stmt->bind_param("i", $_SESSION['user_id']);
if (!$stmt->execute()) {
    error_log("Execute failed: " . $stmt->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de l'exécution de la base de données. Vérifiez les logs.</p>";
        $stmt->close();
        $conn->close();
        exit;
    }
$role_result = $stmt->get_result();
$role = $role_result->fetch_assoc();
$stmt->close();
if (!$role || !in_array($role['name'], ['Caissier_Comptable', 'Marketing Agent'])) {
    error_log("Unauthorized access to view_invoice.php by user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    header("Location: /masunzu_bar_hotel/dashboards/" . ($role && $role['name'] === 'Caissier_Comptable' ? 'cashier_dashboard.php' : 'marketing_dashboard.php'));
    $conn->close();
    exit;
}

// Fetch invoice details from both unpaid and paid invoices, and order table
$invoice = null;
if (isset($_GET['id'])) {
    $invoice_id = intval($_GET['id']);
    
    // First, try to find invoice in unpaid_invoices (all statuses, not just pending)
    $stmt = $conn->prepare("
        SELECT ui.id, ui.invoice_number, ui.customer_name, ui.phone_number, ui.created_at, ui.approval_status, ui.created_by, u.full_name AS created_by_name, ui.approved_by, ua.full_name AS approved_by_name, 'unpaid' AS invoice_stage, ui.has_custom_prices, ui.custom_price_summary
        FROM unpaid_invoices ui
        LEFT JOIN user u ON ui.created_by = u.id
        LEFT JOIN user ua ON ui.approved_by = ua.id
        WHERE ui.id = ? AND (ui.province_id = ? OR ui.province_id IS NULL OR ? IS NULL)
    ");
    if ($stmt !== false) {
        $stmt->bind_param("iii", $invoice_id, $province_id, $province_id);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $invoice = $result->fetch_assoc();
        } else {
            error_log("Execute failed (unpaid_invoices): " . $stmt->error);
        }
        $stmt->close();
    }
    
    // If not found, try paid_invoices
    if (!$invoice) {
        $stmt = $conn->prepare("
            SELECT pi.id, pi.invoice_number, pi.customer_name, pi.phone_number, pi.created_at, pi.approval_status, pi.created_by, u.full_name AS created_by_name, pi.approved_by, ua.full_name AS approved_by_name, 'paid' AS invoice_stage, 0 AS has_custom_prices, NULL AS custom_price_summary
            FROM paid_invoices pi
            LEFT JOIN user u ON pi.created_by = u.id
            LEFT JOIN user ua ON pi.approved_by = ua.id
            WHERE pi.id = ? AND (pi.province_id = ? OR pi.province_id IS NULL OR ? IS NULL)
        ");
        if ($stmt !== false) {
            $stmt->bind_param("iii", $invoice_id, $province_id, $province_id);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $invoice = $result->fetch_assoc();
            } else {
                error_log("Execute failed (paid_invoices): " . $stmt->error);
            }
            $stmt->close();
        }
    }
    
    // If still not found, try order table (for invoices created via order system)
    if (!$invoice) {
        $stmt = $conn->prepare("
            SELECT o.id, o.invoice_number, o.customer_name, NULL AS phone_number, o.created_at, 
                   CASE WHEN o.paid_status = 'Paid' THEN 'approved' ELSE 'pending' END AS approval_status,
                   o.created_by, u.full_name AS created_by_name, NULL AS approved_by, NULL AS approved_by_name,
                   CASE WHEN o.paid_status = 'Paid' THEN 'paid' ELSE 'order' END AS invoice_stage,
                   0 AS has_custom_prices, NULL AS custom_price_summary
            FROM `order` o
            LEFT JOIN user u ON o.created_by = u.id
            WHERE o.id = ? AND (o.province_id = ? OR o.province_id IS NULL OR ? IS NULL)
        ");
        if ($stmt !== false) {
            $stmt->bind_param("iii", $invoice_id, $province_id, $province_id);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $invoice = $result->fetch_assoc();
            } else {
                error_log("Execute failed (order): " . $stmt->error);
            }
            $stmt->close();
        }
    }
}

if (!$invoice) {
    // Log more details for debugging
    $role_name = isset($_SESSION['role_name']) ? $_SESSION['role_name'] : 'unknown';
    error_log("Invoice not found - ID: {$invoice_id}, User: {$_SESSION['user_id']}, Province: {$province_id}, Role: {$role_name}");
    
    http_response_code(404);
    require_once __DIR__ . '/../../includes/header.php';
    echo "<div style='padding: 40px; text-align: center; background-color: #F4F0E4; min-height: 60vh; display: flex; align-items: center; justify-content: center;'>";
    echo "<div style='background-color: #FFFFFF; padding: 30px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 500px;'>";
    echo "<h2 style='color: #D9534F; margin-bottom: 15px;'>Commande Introuvable</h2>";
    echo "<p style='color: #666; margin-bottom: 20px;'>La commande ID #{$invoice_id} n'a pas été trouvée ou vous n'avez pas la permission de la consulter.</p>";
    echo "<p style='color: #999; font-size: 14px; margin-bottom: 20px;'>Cela peut arriver si:<br>";
    echo "• La commande n'existe pas<br>";
    echo "• La commande appartient à une autre province<br>";
    echo "• La commande a été supprimée ou archivée</p>";
    echo "<a href='/masunzu_bar_hotel/dashboards/cashier_dashboard.php' style='background-color: #4B2F1F; color: #F4F0E4; padding: 12px 24px; border-radius: 8px; text-decoration: none; font-weight: 600; display: inline-block;'>← Retour au Tableau de Bord</a>";
    echo "</div></div>";
    include __DIR__ . '/../../includes/footer.php';
    $conn->close();
    exit;
}

// Fetch invoice items
$invoice_stage = isset($invoice['invoice_stage']) ? $invoice['invoice_stage'] : 'unpaid';
$items = [];

if ($invoice_stage === 'paid') {
    // Items from paid_invoice_items
    $itemSql = "SELECT pii.quantity, pii.unit_price, p.name, p.crate_quantity, pii.price_source, pii.system_unit_price, pii.custom_unit_price, pii.custom_price_basis, pii.custom_price_input, pii.custom_price_note FROM paid_invoice_items pii JOIN product p ON pii.product_id = p.id WHERE pii.invoice_id = ?";
    $stmt = $conn->prepare($itemSql);
    if ($stmt !== false) {
        $stmt->bind_param("i", $invoice_id);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $items = $result->fetch_all(MYSQLI_ASSOC);
        } else {
            error_log("Execute failed (paid_invoice_items): " . $stmt->error);
        }
        $stmt->close();
    }
} elseif ($invoice_stage === 'order') {
    // Items from order table (single product per order)
    // Note: order.paid_amount is total amount, unit_price = paid_amount / quantity
    $itemSql = "SELECT o.quantity, 
                       CASE WHEN o.quantity > 0 THEN (o.paid_amount / o.quantity) ELSE o.paid_amount END AS unit_price,
                       p.name, p.crate_quantity, 
                       'system' AS price_source, 
                       CASE WHEN o.quantity > 0 THEN (o.paid_amount / o.quantity) ELSE o.paid_amount END AS system_unit_price,
                       NULL AS custom_unit_price, NULL AS custom_price_basis, NULL AS custom_price_input, NULL AS custom_price_note 
                FROM `order` o 
                JOIN product p ON o.product_id = p.id 
                WHERE o.id = ?";
    $stmt = $conn->prepare($itemSql);
    if ($stmt !== false) {
        $stmt->bind_param("i", $invoice_id);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $items = $result->fetch_all(MYSQLI_ASSOC);
        } else {
            error_log("Execute failed (order items): " . $stmt->error);
        }
        $stmt->close();
    }
} else {
    // Items from invoice_items (unpaid invoices)
    $itemSql = "SELECT ii.quantity, ii.unit_price, p.name, p.crate_quantity, ii.price_source, ii.system_unit_price, ii.custom_unit_price, ii.custom_price_basis, ii.custom_price_input, ii.custom_price_note FROM invoice_items ii JOIN product p ON ii.product_id = p.id WHERE ii.invoice_id = ?";
    $stmt = $conn->prepare($itemSql);
    if ($stmt !== false) {
        $stmt->bind_param("i", $invoice_id);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $items = $result->fetch_all(MYSQLI_ASSOC);
        } else {
            error_log("Execute failed (invoice_items): " . $stmt->error);
        }
        $stmt->close();
    }
}

include __DIR__ . '/../../includes/header.php';
?>
<div style="padding: 20px; background-color: #F4F0E4; margin: 20px 0; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 800px; margin-left: auto; margin-right: auto;">
    <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px;">Détails Commande</h2>
    <p style="color: #000000; font-size: 16px;">Commande #: <?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Client: <?php echo htmlspecialchars($invoice['customer_name'], ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Téléphone: <?php echo htmlspecialchars(isset($invoice['phone_number']) ? $invoice['phone_number'] : 'N/A', ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Date Création: <?php echo date('Y-m-d H:i', strtotime($invoice['created_at'])); ?></p>
    <p style="color: #000000; font-size: 16px;">Créé Par: <?php echo htmlspecialchars(isset($invoice['created_by_name']) ? $invoice['created_by_name'] : 'N/A', ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Approuvé Par: <?php echo htmlspecialchars(isset($invoice['approved_by_name']) ? $invoice['approved_by_name'] : 'N/A', ENT_QUOTES, 'UTF-8'); ?></p>
    <p style="color: #000000; font-size: 16px;">Statut Approbation: <?php 
        $status = isset($invoice['approval_status']) ? $invoice['approval_status'] : 'N/A';
        $status_fr = ($status === 'approved') ? 'Approuvé' : (($status === 'pending') ? 'En Attente' : (($status === 'rejected') ? 'Rejeté' : $status));
        echo htmlspecialchars($status_fr, ENT_QUOTES, 'UTF-8'); 
    ?></p>

    <h3 style="color: #4B2F1F; font-size: 20px; margin-top: 20px; margin-bottom: 15px;">Produits (Caisses)</h3>
    <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
        <thead>
            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                <th style="padding: 10px; text-align: left;">Nom Produit</th>
                <th style="padding: 10px; text-align: left;">Quantité (caisses)</th>
                <th style="padding: 10px; text-align: left;">Prix Unitaire / caisse (BIF)</th>
                <th style="padding: 10px; text-align: left;">Total (BIF)</th>
            </tr>
        </thead>
        <tbody>
            <?php
            $total_amount = 0;
            foreach ($items as $item) {
                $quantity_in_crates = (int)$item['quantity'];
                $unit_price_per_crate = (int)$item['unit_price'];
                $line_total = $unit_price_per_crate * $quantity_in_crates;
                $total_amount += $line_total;

                echo "<tr style='border-bottom: 1px solid #D3D3D3;'>";
                echo "<td style='padding: 10px;'>" . htmlspecialchars($item['name'], ENT_QUOTES, 'UTF-8') . "</td>";
                echo "<td style='padding: 10px;'>" . number_format($quantity_in_crates, 0, ',', '.') . "</td>";
                echo "<td style='padding: 10px;'>" . number_format($unit_price_per_crate, 0, ',', '.') . "</td>";
                echo "<td style='padding: 10px;'>" . number_format($line_total, 0, ',', '.') . "</td>";
                echo "</tr>";
            }
            ?>
            <tr style="background-color: #F4F0E4; font-weight: bold;">
                <td colspan="3" style="padding: 10px; text-align: right;">Montant Total:</td>
                <td style="padding: 10px;"><?php echo number_format($total_amount, 2, ',', '.'); ?> BIF</td>
            </tr>
        </tbody>
    </table>
    <a href="/masunzu_bar_hotel/dashboards/cashier_dashboard.php" style="color: #4B2F1F; text-decoration: underline;">Retour au Tableau de Bord</a>
</div>

<?php
include __DIR__ . '/../../includes/footer.php';
if ($conn && !$conn->connect_error) {
    $conn->close();
}
?>
