<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

if ($_SESSION['role_name'] !== 'Caissier_Comptable') {
    error_log("Unauthorized access to transfer_cash.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/index.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in transfer_cash.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$userId = (int)($_SESSION['user_id'] ?? 0);

// Ensure province id available
if (!isset($_SESSION['province_id'])) {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $province_row = $stmt->get_result()->fetch_assoc();
        $_SESSION['province_id'] = $province_row['province_id'] ?? null;
        $stmt->close();
    }
}
$province_id = (int)($_SESSION['province_id'] ?? 0);

register_shutdown_function(static function () use ($conn) {
    if ($conn instanceof mysqli) {
        @$conn->close();
    }
});

function fetchActiveShift(mysqli $conn, int $cashierId): ?array {
    $stmt = $conn->prepare("
        SELECT id, shift_start, beginning_balance, deposits
        FROM cashier_balance
        WHERE cashier_id = ? AND shift_end IS NULL
        ORDER BY shift_start DESC
        LIMIT 1
    ");
    if (!$stmt) {
        return null;
    }
    $stmt->bind_param("i", $cashierId);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc() ?: null;
    $stmt->close();
    return $row ?: null;
}

function computeShiftAmount(array $shift): float {
    // Always calculate from beginning_balance + deposits - withdrawals
    // This ensures we get the correct amount even if end_balance is 0 or null
    $begin = (float)($shift['beginning_balance'] ?? 0);
    $deposits = (float)($shift['deposits'] ?? 0);
    $withdrawals = (float)($shift['withdrawals'] ?? 0);
    $calculated = $begin + $deposits - $withdrawals;

    // If end_balance exists and is not null, use it as a reference but prefer calculated value
    // Only use end_balance if calculated is 0 or negative and end_balance is positive
    if (array_key_exists('end_balance', $shift) && $shift['end_balance'] !== null) {
        $endBalance = (float)$shift['end_balance'];
        // Use the maximum of calculated and end_balance to ensure we don't miss any amount
        $calculated = max($calculated, $endBalance);
    }

    return $calculated > 0 ? $calculated : 0.0;
}

function ensureShiftTransferSchema(mysqli $conn): array {
    $errors = [];
    $createSql = "
        CREATE TABLE IF NOT EXISTS cashier_shift_transfer (
            id INT AUTO_INCREMENT PRIMARY KEY,
            shift_id INT NOT NULL,
            transfer_id INT NOT NULL,
            amount DECIMAL(15,2) NOT NULL,
            created_by INT NOT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY uniq_shift (shift_id),
            KEY idx_transfer (transfer_id),
            CONSTRAINT fk_shift_transfer_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE CASCADE,
            CONSTRAINT fk_shift_transfer_transfer FOREIGN KEY (transfer_id) REFERENCES cash_transfer(id) ON DELETE CASCADE,
            CONSTRAINT fk_shift_transfer_user FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    if (!$conn->query($createSql)) {
        $errors[] = 'Impossible d\'initialiser le suivi des transferts de session. Contactez l\'administrateur.';
    }
    return $errors;
}

function ensureCashTransferSchema(mysqli $conn): array {
    $errors = [];
    $createSql = "
        CREATE TABLE IF NOT EXISTS cash_transfer (
            id INT(11) NOT NULL AUTO_INCREMENT,
            cashier_id INT(11) NOT NULL,
            accountant_id INT(11) NOT NULL,
            amount BIGINT(20) NOT NULL,
            status ENUM('pending','received','rejected') NOT NULL DEFAULT 'pending',
            transfer_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            received_at DATETIME DEFAULT NULL,
            PRIMARY KEY (id),
            KEY idx_cash_transfer_cashier (cashier_id),
            KEY idx_cash_transfer_accountant (accountant_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    if (!$conn->query($createSql)) {
        error_log('Failed to ensure cash_transfer table exists: ' . $conn->error);
        $errors[] = 'Impossible d\'initialiser la table des transferts d\'espèces. Contactez l\'administrateur.';
        return $errors;
    }

    $columnResult = $conn->query("SHOW COLUMNS FROM cash_transfer LIKE 'id'");
    if (!$columnResult) {
        error_log('Failed to inspect cash_transfer.id column: ' . $conn->error);
        $errors[] = 'Impossible de vérifier la structure des transferts d\'espèces. Contactez l\'administrateur.';
        return $errors;
    }

    $column = $columnResult->fetch_assoc();
    $columnResult->free();
    if (!$column) {
        $errors[] = 'La colonne identifiant manque dans la table des transferts d\'espèces.';
        return $errors;
    }

    $extra = strtolower((string)($column['Extra'] ?? ''));
    if (strpos($extra, 'auto_increment') === false) {
        $alterSql = "ALTER TABLE cash_transfer MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT";
        if (!$conn->query($alterSql)) {
            error_log('Failed to set AUTO_INCREMENT on cash_transfer.id: ' . $conn->error);
            $errors[] = 'Impossible d\'initialiser le compteur automatique des transferts d\'espèces. Contactez l\'administrateur.';
        }
    }

    return $errors;
}

function getRoleIdByName(mysqli $conn, string $roleName): ?int {
    static $roleCache = [];
    $key = strtoupper($roleName);
    if (array_key_exists($key, $roleCache)) {
        return $roleCache[$key];
    }
    $stmt = $conn->prepare("SELECT id FROM role WHERE UPPER(name) = ? LIMIT 1");
    if ($stmt) {
        $stmt->bind_param('s', $key);
        $stmt->execute();
        $roleCache[$key] = $stmt->get_result()->fetch_assoc()['id'] ?? null;
        $stmt->close();
    } else {
        $roleCache[$key] = null;
    }
    return $roleCache[$key];
}

$schemaBlockingErrors = ensureShiftTransferSchema($conn);
$schemaBlockingErrors = array_merge($schemaBlockingErrors, ensureCashTransferSchema($conn));
$schemaWarnings = [];

$active_shift = fetchActiveShift($conn, $userId);

if ($province_id <= 0) {
    $schemaBlockingErrors[] = 'Aucune province n\'est associée à votre compte. Contactez l\'administrateur.';
}

$availableShifts = [];
$availableShiftIndex = [];
$allShifts = []; // For diagnostic display
if (empty($schemaBlockingErrors)) {
    // Debug: Check all closed shifts first - more detailed
    $debugStmt = $conn->prepare("
        SELECT
            cb.id,
            cb.cashier_id,
            cb.shift_start,
            cb.shift_end,
            cb.beginning_balance,
            cb.end_balance,
            cb.deposits,
            cb.withdrawals,
            u.province_id AS user_province_id,
            COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
            COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
            csc.counted_amount,
            (cb.beginning_balance + cb.deposits - cb.withdrawals) AS calculated_amount
        FROM cashier_balance cb
        INNER JOIN user u ON cb.cashier_id = u.id
        LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
        WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
        ORDER BY cb.shift_end DESC
        LIMIT 10
    ");
    if ($debugStmt) {
        $debugStmt->bind_param("i", $userId);
        $debugStmt->execute();
        $debugResult = $debugStmt->get_result();
        $allShifts = [];
        while ($debugRow = $debugResult->fetch_assoc()) {
            $allShifts[] = $debugRow;
        }
        error_log("DEBUG transfer_cash.php: Found " . count($allShifts) . " closed shifts for cashier_id=$userId");
        foreach ($allShifts as $shift) {
            $matchesProvince = ($shift['user_province_id'] == $province_id);
            $isDeposited = strtolower($shift['deposited_to_bank'] ?? 'no') === 'yes';
            $isTransferred = strtolower($shift['transferred_to_accountant'] ?? 'no') === 'yes';
            
            // Use counted_amount if available, otherwise use calculated
            $amount = 0;
            if (isset($shift['counted_amount']) && $shift['counted_amount'] !== null && $shift['counted_amount'] > 0) {
                $amount = (float)$shift['counted_amount'];
            } else {
                $amount = (float)$shift['calculated_amount'];
            }
            
            // Will appear if province matches, not deposited, not transferred, and has amount > 0
            // Shift appears in BOTH pages only when BOTH flags are 'no'
            $willAppear = $matchesProvince && !$isDeposited && !$isTransferred && $amount > 0;
            error_log(sprintf(
                "  Shift ID=%d, province_match=%s, end=%s, counted_amount=%s, calculated=%s, deposited=%s, transferred=%s, WILL_APPEAR=%s",
                $shift['id'],
                $matchesProvince ? 'YES' : 'NO',
                $shift['shift_end'] ?? 'NULL',
                $shift['counted_amount'] ?? 'NULL',
                number_format((float)$shift['calculated_amount'], 0, ',', '.'),
                $isDeposited ? 'YES' : 'NO',
                $isTransferred ? 'YES' : 'NO',
                $willAppear ? 'YES' : 'NO'
            ));
        }
        $debugStmt->close();
    }
    
    $stmtShifts = $conn->prepare("
        SELECT
            cb.id,
            cb.shift_start,
            cb.shift_end,
            cb.beginning_balance,
            cb.end_balance,
            cb.deposits,
            cb.withdrawals,
            COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
            COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
            csc.counted_amount
        FROM cashier_balance cb
        INNER JOIN user u ON cb.cashier_id = u.id
        LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
        WHERE cb.cashier_id = ? AND u.province_id = ? AND cb.shift_end IS NOT NULL 
            AND (COALESCE(csc.deposited_to_bank, 'no') = 'no' OR csc.deposited_to_bank IS NULL)
            AND (COALESCE(csc.transferred_to_accountant, 'no') = 'no' OR csc.transferred_to_accountant IS NULL)
        ORDER BY cb.shift_end DESC
    ");
    if ($stmtShifts) {
        $stmtShifts->bind_param("ii", $userId, $province_id);
        $stmtShifts->execute();
        $resultShifts = $stmtShifts->get_result();
        $totalRows = $resultShifts->num_rows;
        error_log("DEBUG transfer_cash.php: Found $totalRows closed shifts for cashier_id=$userId, province_id=$province_id");
        $filteredCount = 0;
        while ($row = $resultShifts->fetch_assoc()) {
            // Use counted_amount if available, otherwise calculate from end_balance or beginning+deposits-withdrawals
            $shiftAmount = 0;
            if (isset($row['counted_amount']) && $row['counted_amount'] !== null && $row['counted_amount'] > 0) {
                $shiftAmount = (float)$row['counted_amount'];
            } else {
                $shiftAmount = computeShiftAmount($row);
            }
            error_log("DEBUG transfer_cash.php: Shift ID={$row['id']}, counted_amount={$row['counted_amount']}, end_balance={$row['end_balance']}, beginning_balance={$row['beginning_balance']}, deposits={$row['deposits']}, withdrawals={$row['withdrawals']}, final_amount=$shiftAmount");
            if ($shiftAmount <= 0) {
                $filteredCount++;
                error_log("DEBUG transfer_cash.php: Shift ID={$row['id']} filtered out (amount <= 0)");
                continue;
            }
            $startLabel = $row['shift_start'] ? date('d M Y H:i', strtotime($row['shift_start'])) : 'Début inconnu';
            $endLabel = $row['shift_end'] ? date('d M Y H:i', strtotime($row['shift_end'])) : 'Fin inconnue';
            $amountLabel = number_format($shiftAmount, 0, ',', '.');
            $shiftData = [
                'id' => (int)$row['id'],
                'shift_start' => $row['shift_start'],
                'shift_end' => $row['shift_end'],
                'beginning_balance' => (float)$row['beginning_balance'],
                'deposits' => (float)$row['deposits'],
                'withdrawals' => (float)$row['withdrawals'],
                'amount' => $shiftAmount,
                'label' => sprintf('%s - %s - %s BIF', $startLabel, $endLabel, $amountLabel),
            ];
            $availableShifts[] = $shiftData;
            $availableShiftIndex[$shiftData['id']] = $shiftData;
        }
        error_log("DEBUG transfer_cash.php: Total shifts found: $totalRows, filtered out (amount <= 0): $filteredCount, available: " . count($availableShifts));
        $stmtShifts->close();
    } else {
        $schemaBlockingErrors[] = 'Échec du chargement des sessions fermées pour transfert.';
    }
}

$selectedShiftId = isset($_POST['shift_id']) ? (int)$_POST['shift_id'] : 0;
if ($selectedShiftId === 0 && !empty($availableShifts)) {
    $selectedShiftId = $availableShifts[0]['id'];
}
$selectedAccountantId = isset($_POST['accountant_id']) ? (int)$_POST['accountant_id'] : 0;

// Fetch accountants (same province preferred, fallback)
$accountants = [];
$accountantRoleId = getRoleIdByName($conn, 'Accountant') ?? 4;
$accountantSql = "
    SELECT id, full_name
    FROM user
    WHERE role_id = ?
";
if ($province_id) {
    $accountantSql .= " AND (province_id = ? OR province_id IS NULL)";
}
$accountantSql .= " ORDER BY province_id IS NULL, full_name";
$stmt = $conn->prepare($accountantSql);
if ($stmt) {
    if ($province_id) {
        $stmt->bind_param('ii', $accountantRoleId, $province_id);
    } else {
        $stmt->bind_param('i', $accountantRoleId);
    }
    $stmt->execute();
    $accountants = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accountant_id = isset($_POST['accountant_id']) ? (int)$_POST['accountant_id'] : 0;
    $posted_shift_id = isset($_POST['shift_id']) ? (int)$_POST['shift_id'] : 0;
    $selectedAccountantId = $accountant_id;

    if (!empty($schemaBlockingErrors)) {
        $errors[] = implode(' ', $schemaBlockingErrors);
    }
    if ($active_shift) {
        $errors[] = 'Fermez votre session active avant d\'enregistrer un transfert d\'espèces.';
    }
    if ($posted_shift_id <= 0 || !isset($availableShiftIndex[$posted_shift_id])) {
        $errors[] = 'Sélectionnez une session fermée valide à transférer.';
    } else {
        $selectedShiftId = $posted_shift_id;
        $selectedShift = $availableShiftIndex[$selectedShiftId];
    }

    $transferAmount = $selectedShift ? (int)round($selectedShift['amount']) : 0;
    if ($selectedShift && $transferAmount <= 0) {
        $errors[] = 'La session sélectionnée n\'a plus d\'espèces restantes à transférer.';
    }

    $accountant_ids = array_column($accountants, 'id');
    if ($accountant_id <= 0 || !in_array($accountant_id, $accountant_ids, true)) {
        $errors[] = 'Sélectionnez un comptable valide.';
    }

    if (empty($errors)) {
        $conn->begin_transaction();
        try {
            $stmtShift = $conn->prepare("
                SELECT 
                    cb.id, 
                    cb.shift_end, 
                    cb.beginning_balance, 
                    cb.end_balance, 
                    cb.deposits, 
                    cb.withdrawals,
                    csc.counted_amount
                FROM cashier_balance cb
                LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
                WHERE cb.id = ? AND cb.cashier_id = ? AND cb.shift_end IS NOT NULL
                FOR UPDATE
            ");
            if (!$stmtShift) {
                throw new Exception('Échec de la préparation de la recherche de session.');
            }
            $stmtShift->bind_param("ii", $selectedShiftId, $userId);
            $stmtShift->execute();
            $lockedShift = $stmtShift->get_result()->fetch_assoc() ?: null;
            $stmtShift->close();

            if (!$lockedShift) {
                throw new Exception('La session sélectionnée n\'est plus disponible pour transfert.');
            }

            // Check if shift has been deposited or transferred using YES/NO flags
            $stmtClosureCheck = $conn->prepare("
                SELECT deposited_to_bank, transferred_to_accountant
                FROM cashier_shift_closure
                WHERE shift_id = ?
                LIMIT 1
                FOR UPDATE
            ");
            if ($stmtClosureCheck) {
                $stmtClosureCheck->bind_param("i", $selectedShiftId);
                $stmtClosureCheck->execute();
                $closureRow = $stmtClosureCheck->get_result()->fetch_assoc() ?: null;
                $stmtClosureCheck->close();
                
                if ($closureRow) {
                    $is_deposited = strtolower($closureRow['deposited_to_bank'] ?? 'no') === 'yes';
                    $is_transferred = strtolower($closureRow['transferred_to_accountant'] ?? 'no') === 'yes';
                    
                    if ($is_deposited) {
                        throw new Exception('Cette session a déjà été déposée à la banque. Vous ne pouvez pas déposer à la banque et transférer au comptable en même temps.');
                    }
                    if ($is_transferred) {
                        throw new Exception('Cette session a déjà été transférée à un comptable.');
                    }
                }
            }
            
            // Fallback: Check old junction tables for backward compatibility
            $stmtDepositCheck = $conn->prepare("
                SELECT deposit_id
                FROM cashier_shift_deposit
                WHERE shift_id = ?
                LIMIT 1
                FOR UPDATE
            ");
            if ($stmtDepositCheck) {
                $stmtDepositCheck->bind_param("i", $selectedShiftId);
                $stmtDepositCheck->execute();
                $depositRow = $stmtDepositCheck->get_result()->fetch_assoc() ?: null;
                $stmtDepositCheck->close();
                if ($depositRow) {
                    throw new Exception('Cette session a déjà été déposée à la banque.');
                }
            }

            // Vérifier si le shift a déjà été transféré (old method)
            $stmtTransferCheck = $conn->prepare("
                SELECT id
                FROM cashier_shift_transfer
                WHERE shift_id = ?
                LIMIT 1
                FOR UPDATE
            ");
            if ($stmtTransferCheck) {
                $stmtTransferCheck->bind_param("i", $selectedShiftId);
                $stmtTransferCheck->execute();
                $transferRow = $stmtTransferCheck->get_result()->fetch_assoc() ?: null;
                $stmtTransferCheck->close();
                if ($transferRow) {
                    throw new Exception('Cette session a déjà été transférée à un comptable.');
                }
            }

            // Use counted_amount if available, otherwise calculate from end_balance or beginning+deposits-withdrawals
            $lockedAmount = 0;
            if (isset($lockedShift['counted_amount']) && $lockedShift['counted_amount'] !== null && $lockedShift['counted_amount'] > 0) {
                $lockedAmount = (int)round((float)$lockedShift['counted_amount']);
            } else {
                $lockedAmount = (int)round(computeShiftAmount($lockedShift));
            }
            if ($lockedAmount <= 0) {
                throw new Exception('La session sélectionnée n\'a plus d\'espèces restantes à transférer.');
            }

            $stmtInsertTransfer = $conn->prepare("
                INSERT INTO cash_transfer (cashier_id, accountant_id, amount, status, transfer_date)
                VALUES (?, ?, ?, 'pending', NOW())
            ");
            if (!$stmtInsertTransfer) {
                throw new Exception('Échec de la préparation de l\'insertion de transfert d\'espèces.');
            }
            $stmtInsertTransfer->bind_param("iii", $userId, $accountant_id, $lockedAmount);
            if (!$stmtInsertTransfer->execute()) {
                $errorMsg = $stmtInsertTransfer->error ?: 'erreur inconnue';
                $stmtInsertTransfer->close();
                throw new Exception('Échec de l\'enregistrement du transfert: ' . $errorMsg);
            }
            $transferId = (int)$conn->insert_id;
            $stmtInsertTransfer->close();

            $stmtLink = $conn->prepare("
                INSERT INTO cashier_shift_transfer (shift_id, transfer_id, amount, created_by)
                VALUES (?, ?, ?, ?)
            ");
            if (!$stmtLink) {
                throw new Exception('Échec de la préparation de la liaison transfert-session.');
            }
            $amountDecimal = (float)round($lockedAmount, 2);
            $stmtLink->bind_param("iidi", $selectedShiftId, $transferId, $amountDecimal, $userId);
            if (!$stmtLink->execute()) {
                $errorMsg = $stmtLink->error ?: 'erreur inconnue';
                $stmtLink->close();
                throw new Exception('Échec de la liaison du transfert à la session: ' . $errorMsg);
            }
            $stmtLink->close();
            
            // Update cashier_shift_closure: set transferred_to_accountant='yes'
            // First check that deposited_to_bank is not 'yes' (both cannot be 'yes' at the same time)
            $stmtCheckDeposit = $conn->prepare("
                SELECT deposited_to_bank 
                FROM cashier_shift_closure 
                WHERE shift_id = ?
            ");
            if ($stmtCheckDeposit) {
                $stmtCheckDeposit->bind_param("i", $selectedShiftId);
                $stmtCheckDeposit->execute();
                $depositCheckResult = $stmtCheckDeposit->get_result()->fetch_assoc();
                $stmtCheckDeposit->close();
                
                if ($depositCheckResult) {
                    $is_deposited = strtolower($depositCheckResult['deposited_to_bank'] ?? 'no') === 'yes';
                    if ($is_deposited) {
                        throw new Exception('Cette session a déjà été déposée à la banque. Vous ne pouvez pas déposer à la banque et transférer au comptable en même temps.');
                    }
                }
            }
            
            $stmtUpdateClosure = $conn->prepare("
                UPDATE cashier_shift_closure 
                SET transferred_to_accountant = 'yes' 
                WHERE shift_id = ? AND (deposited_to_bank = 'no' OR deposited_to_bank IS NULL)
            ");
            if ($stmtUpdateClosure) {
                $stmtUpdateClosure->bind_param("i", $selectedShiftId);
                if (!$stmtUpdateClosure->execute()) {
                    error_log("Failed to update transferred_to_accountant flag: " . $stmtUpdateClosure->error);
                    throw new Exception('Impossible de mettre à jour le statut de transfert. La session a peut-être déjà été déposée à la banque.');
                }
                if ($stmtUpdateClosure->affected_rows === 0) {
                    throw new Exception('Impossible de mettre à jour le statut de transfert. La session a peut-être déjà été déposée à la banque.');
                }
                $stmtUpdateClosure->close();
            }
            
            // Enregistrer dans cashier_account
            // Ensure cashier_account table exists
            $createCashierAccountSql = "
                CREATE TABLE IF NOT EXISTS cashier_account (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    cashier_id INT NOT NULL,
                    shift_id INT NULL,
                    transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                    direction ENUM('CREDIT','DEBIT') NOT NULL,
                    amount BIGINT(20) NOT NULL,
                    reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                    reference_id INT NULL,
                    description VARCHAR(255) DEFAULT NULL,
                    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    created_by INT NULL,
                    INDEX idx_cashier_created (cashier_id, created_at),
                    INDEX idx_shift (shift_id),
                    INDEX idx_reference (reference_type, reference_id),
                    CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                    CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                    CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
            ";
            @$conn->query($createCashierAccountSql);
            
            $shiftEnd = $lockedShift['shift_end'] ?? '';
            $shiftEndLabel = $shiftEnd ? date('Y-m-d H:i', strtotime($shiftEnd)) : 'unknown';
            $description = sprintf('Transfert vers comptable: Session %d (terminée %s)', $selectedShiftId, $shiftEndLabel);
            $stmtAccount = $conn->prepare("
                INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                VALUES (?, ?, 'CASH_TRANSFER', 'DEBIT', ?, 'CASH_TRANSFER', ?, ?, ?)
            ");
            if ($stmtAccount) {
                $stmtAccount->bind_param("iiiiis", $userId, $selectedShiftId, $lockedAmount, $transferId, $userId, $description);
                if (!$stmtAccount->execute()) {
                    error_log("Failed to record CASH_TRANSFER in cashier_account: " . $stmtAccount->error);
                }
                $stmtAccount->close();
            }

            if (function_exists('log_action')) {
                log_action(
                    $userId,
                    'cash_transfer',
                    "Cash transfer of {$lockedAmount} BIF recorded for shift {$selectedShiftId} (ended {$shiftEndLabel}) to accountant ID {$accountant_id}"
                );
            }

            $conn->commit();
            $message = sprintf(
                'Nouveau transfert d\'espèces #%d de %s (%s) pour %s BIF.',
                $transferId,
                $_SESSION['full_name'] ?? 'Caissier',
                date('d/m/Y H:i'),
                number_format($lockedAmount, 0, ',', '.')
            );
            mbh_notify_user($conn, $accountant_id, $message, '/masunzu_bar_hotel/modules/accounting/receive_cash.php');
            header("Location: /masunzu_bar_hotel/modules/cashiers/transfer_cash.php?success=1");
            exit;
        } catch (Throwable $th) {
            $conn->rollback();
            $errors[] = $th->getMessage();
        }
    }
}

$selectedShift = $availableShiftIndex[$selectedShiftId] ?? null;
$selectedShiftAmount = $selectedShift ? number_format((int)round($selectedShift['amount']), 0, '.', '') : '';
$selectedBeginning = $selectedShift ? (int)round($selectedShift['beginning_balance']) : 0;
$selectedDeposits = $selectedShift ? (int)round($selectedShift['deposits']) : 0;
$selectedWithdrawals = $selectedShift ? (int)round($selectedShift['withdrawals']) : 0;
$selectedTransferred = 0;

// Calculate total amount available for transfer (sum of all available shifts)
$totalAvailableForTransfer = 0;
foreach ($availableShifts as $shift) {
    $totalAvailableForTransfer += (float)$shift['amount'];
}
$activeShiftStartLabel = $active_shift && !empty($active_shift['shift_start'])
    ? date('Y-m-d H:i', strtotime($active_shift['shift_start']))
    : 'Début inconnu';
$selectedShiftStartLabel = $selectedShift && !empty($selectedShift['shift_start'])
    ? date('Y-m-d H:i', strtotime($selectedShift['shift_start']))
    : 'Début inconnu';
$selectedShiftEndLabel = $selectedShift && !empty($selectedShift['shift_end'])
    ? date('Y-m-d H:i', strtotime($selectedShift['shift_end']))
    : null;

if (isset($_GET['success'])) {
    $success = 'Transfert d\'espèces enregistré avec succès et en attente d\'accusé de réception.';
}

$formDisabled = $active_shift || empty($availableShifts) || empty($accountants) || !empty($schemaBlockingErrors);

// Fetch latest transfers for display
$transfer_history = [];
$stmt = $conn->prepare("
    SELECT 
        ct.id,
        ct.amount,
        ct.status,
        ct.transfer_date,
        ct.received_at,
        u.full_name AS accountant_name
    FROM cash_transfer ct
    LEFT JOIN user u ON ct.accountant_id = u.id
    WHERE ct.cashier_id = ?
    ORDER BY ct.transfer_date DESC
    LIMIT 15
");
if ($stmt) {
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $transfer_history = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

require_once __DIR__ . '/../../includes/header.php';
?>
<div class="container-fluid py-4" style="background-color: #F4F0E4; min-height: 100vh;">
    <div class="row justify-content-center">
        <div class="col-lg-10 col-xl-8">
            <div class="card shadow-sm border" style="border-radius: 8px;">
                <div class="card-header border-bottom text-center py-4">
                    <div class="d-flex align-items-center justify-content-center">
                        <i class="bi bi-cash-coin fs-2 me-3 text-primary"></i>
                        <h2 class="mb-0 fw-bold h4">Transférer Espèces vers Comptable</h2>
                    </div>
                </div>
                </div>
                <div class="card-body p-4">

                    <!-- Status Information Card -->
                    <div class="card status-card mb-4 border shadow-sm">
                        <div class="card-body p-4">
                            <?php if ($active_shift): ?>
                                <div class="alert alert-warning border-0 d-flex align-items-center" role="alert">
                                    <i class="bi bi-exclamation-triangle-fill fs-4 me-3"></i>
                                    <div>
                                        <strong>Session Active Détectée</strong><br>
                                        Votre session a commencé le <strong><?php echo htmlspecialchars($activeShiftStartLabel, ENT_QUOTES, 'UTF-8'); ?></strong>.
                                        Veuillez fermer la session dans <a href="/masunzu_bar_hotel/modules/cashiers/balance_management.php" class="alert-link">Gestion des Soldes</a> avant d'enregistrer un transfert d'espèces.
                                    </div>
                                </div>
                            <?php elseif ($selectedShift): ?>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="card border shadow-sm h-100">
                                            <div class="card-body text-center p-3">
                                                <i class="bi bi-calendar-check text-success fs-1 mb-2"></i>
                                                <h6 class="card-title text-muted mb-2">Période Session</h6>
                                                <p class="mb-0 fw-semibold">
                                                    <?php echo htmlspecialchars($selectedShiftStartLabel, ENT_QUOTES, 'UTF-8'); ?>
                                                    <?php if ($selectedShiftEndLabel): ?>
                                                        <br><small class="text-muted">jusqu'à <?php echo htmlspecialchars($selectedShiftEndLabel, ENT_QUOTES, 'UTF-8'); ?></small>
                                                    <?php endif; ?>
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card border shadow-sm h-100">
                                            <div class="card-body text-center p-3">
                                                <i class="bi bi-cash-stack text-success fs-1 mb-2"></i>
                                                <h6 class="card-title text-muted mb-2">Disponible pour Transfert</h6>
                                                <h5 class="mb-0 fw-bold text-success"><?php echo number_format((int)round($totalAvailableForTransfer), 0, ',', '.'); ?> BIF</h5>
                                                <?php if (count($availableShifts) > 1): ?>
                                                    <small class="text-muted">(<?php echo count($availableShifts); ?> sessions)</small>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="row g-3 mt-3">
                                    <div class="col-6 col-md-3">
                                        <div class="text-center p-2">
                                            <i class="bi bi-piggy-bank text-primary fs-4 mb-1"></i>
                                            <small class="text-muted d-block">Solde Initial</small>
                                            <strong class="text-primary"><?php echo number_format($selectedBeginning, 0, ',', '.'); ?> BIF</strong>
                                        </div>
                                    </div>
                                    <div class="col-6 col-md-3">
                                        <div class="text-center p-2">
                                            <i class="bi bi-graph-up-arrow text-success fs-4 mb-1"></i>
                                            <small class="text-muted d-block">Ventes Créditées</small>
                                            <strong class="text-success"><?php echo number_format($selectedDeposits, 0, ',', '.'); ?> BIF</strong>
                                        </div>
                                    </div>
                                    <div class="col-6 col-md-3">
                                        <div class="text-center p-2">
                                            <i class="bi bi-cash-coin text-danger fs-4 mb-1"></i>
                                            <small class="text-muted d-block">Dépenses Payées</small>
                                            <strong class="text-danger"><?php echo number_format($selectedWithdrawals, 0, ',', '.'); ?> BIF</strong>
                                        </div>
                                    </div>
                                    <div class="col-6 col-md-3">
                                        <div class="text-center p-2">
                                            <i class="bi bi-arrow-right-circle text-warning fs-4 mb-1"></i>
                                            <small class="text-muted d-block">Déjà Transféré</small>
                                            <strong class="text-warning"><?php echo number_format($selectedTransferred, 0, ',', '.'); ?> BIF</strong>
                                        </div>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="alert alert-info border-0 d-flex align-items-center" role="alert">
                                    <i class="bi bi-info-circle-fill fs-4 me-3"></i>
                                    <div>
                                        <strong>Aucune Session Fermée Disponible</strong><br>
                                        Aucune session fermée n'est en attente de transfert. Une fois que vous fermez une session (et avant de la déposer à la banque), elle apparaîtra ici.
                                    </div>
                                </div>
                                
                                <!-- Debug Information -->
                                <?php if (isset($allShifts) && !empty($allShifts)): ?>
                                    <div class="alert alert-warning border-0 mt-3" role="alert">
                                        <strong>🔍 Diagnostic - Sessions Fermées Trouvées:</strong><br>
                                        <small>
                                            <?php 
                                            foreach ($allShifts as $shift): 
                                                $matchesProvince = ($shift['user_province_id'] == $province_id);
                                                $isDeposited = strtolower($shift['deposited_to_bank'] ?? 'no') === 'yes';
                                                $isTransferred = strtolower($shift['transferred_to_accountant'] ?? 'no') === 'yes';
                                                
                                                // Use counted_amount if available, otherwise use calculated
                                                $amount = 0;
                                                if (isset($shift['counted_amount']) && $shift['counted_amount'] !== null && $shift['counted_amount'] > 0) {
                                                    $amount = (float)$shift['counted_amount'];
                                                } else {
                                                    $amount = (float)$shift['calculated_amount'];
                                                }
                                                
                                                $willAppear = $matchesProvince && !$isDeposited && !$isTransferred && $amount > 0;
                                                $status = [];
                                                if (!$matchesProvince) $status[] = "Province différente (user: {$shift['user_province_id']}, attendu: $province_id)";
                                                if ($isDeposited) $status[] = "Déjà déposée à la banque";
                                                if ($isTransferred) $status[] = "Déjà transférée au comptable";
                                                if ($amount <= 0) $status[] = "Montant = 0";
                                                $statusText = !empty($status) ? implode(', ', $status) : 'Devrait apparaître';
                                            ?>
                                            <br>• Session #<?php echo $shift['id']; ?> (<?php echo $shift['shift_end'] ?? 'N/A'; ?>): 
                                            <?php echo $willAppear ? '<span style="color:green;">✓ Disponible</span>' : '<span style="color:red;">✗ ' . htmlspecialchars($statusText, ENT_QUOTES, 'UTF-8') . '</span>'; ?>
                                            (Montant: <?php echo number_format($amount, 0, ',', '.'); ?> BIF<?php if (isset($shift['counted_amount']) && $shift['counted_amount'] !== null): ?> - compté<?php else: ?> - calculé<?php endif; ?>)
                                            <?php endforeach; ?>
                                        </small>
                                    </div>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- System Messages -->
                    <?php if (!empty($schemaBlockingErrors)): ?>
                        <?php foreach ($schemaBlockingErrors as $blocking): ?>
                            <div class="alert alert-danger border-0 shadow-sm d-flex align-items-center mb-3" role="alert">
                                <i class="bi bi-exclamation-triangle-fill fs-5 me-3"></i>
                                <div><?php echo htmlspecialchars($blocking, ENT_QUOTES, 'UTF-8'); ?></div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>

                    <?php if (!empty($schemaWarnings)): ?>
                        <?php foreach ($schemaWarnings as $warning): ?>
                            <div class="alert alert-warning border-0 shadow-sm d-flex align-items-center mb-3" role="alert">
                                <i class="bi bi-exclamation-circle-fill fs-5 me-3"></i>
                                <div><?php echo htmlspecialchars($warning, ENT_QUOTES, 'UTF-8'); ?></div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>

                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger border-0 shadow-sm mb-3" role="alert">
                            <div class="d-flex align-items-center mb-2">
                                <i class="bi bi-x-circle-fill fs-5 me-3"></i>
                                <strong>Échec du Transfert</strong>
                            </div>
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if ($success): ?>
                        <div class="alert alert-success border-0 shadow-sm d-flex align-items-center mb-3" role="alert">
                            <i class="bi bi-check-circle-fill fs-5 me-3"></i>
                            <div><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></div>
                        </div>
                    <?php endif; ?>

                    <!-- Transfer Form -->
                    <div class="card form-card mb-4 border shadow-sm">
                        <div class="card-header">
                            <h5 class="mb-0 d-flex align-items-center">
                                <i class="bi bi-pencil-square me-2 text-primary"></i>
                                Détails Transfert
                            </h5>
                        </div>
                        <div class="card-body p-4 position-relative">
                            <!-- Loading Overlay -->
                            <div id="form-loading-overlay" class="position-absolute top-0 start-0 w-100 h-100 d-none" style="background: rgba(255, 255, 255, 0.9); backdrop-filter: blur(5px); z-index: 10; border-radius: 15px;">
                                <div class="d-flex align-items-center justify-content-center h-100">
                                    <div class="text-center">
                                        <div class="spinner-border text-primary mb-3" role="status" style="width: 3rem; height: 3rem;">
                                            <span class="visually-hidden">Loading...</span>
                                        </div>
                                        <h6 class="text-primary fw-bold">Traitement du Transfert...</h6>
                                        <p class="text-muted small mb-0">Veuillez patienter pendant l'enregistrement de votre transfert d'espèces</p>
                                    </div>
                                </div>
                            </div>

                            <form method="POST">
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <label for="shift_id" class="form-label">
                                            <i class="bi bi-calendar-event me-1"></i>Session Fermée
                                        </label>
                                        <select
                                            class="form-select"
                                            id="shift_id"
                                            name="shift_id"
                                            <?php echo $formDisabled ? 'disabled' : ''; ?>
                                            required
                                        >
                                                <option value="">Sélectionner session fermée</option>
                                                <?php foreach ($availableShifts as $shift): ?>
                                                    <option
                                                        value="<?php echo (int)$shift['id']; ?>"
                                                        data-amount="<?php echo number_format((int)round($shift['amount']), 0, '.', ''); ?>"
                                                        <?php echo $selectedShiftId === $shift['id'] ? 'selected' : ''; ?>
                                                    >
                                                        <?php echo htmlspecialchars($shift['label'], ENT_QUOTES, 'UTF-8'); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-6">
                                        <label for="amount" class="form-label">
                                            <i class="bi bi-cash me-1"></i>Montant à Transférer (BIF)
                                        </label>
                                        <input
                                            type="number"
                                            class="form-control"
                                            id="amount"
                                            value="<?php echo htmlspecialchars($selectedShiftAmount, ENT_QUOTES, 'UTF-8'); ?>"
                                            placeholder="Sélectionner une session fermée"
                                            readonly
                                        >
                                    </div>
                                    <div class="col-md-6">
                                        <label for="accountant_id" class="form-label">
                                            <i class="bi bi-person me-1"></i>Comptable Receveur
                                        </label>
                                        <select
                                            class="form-select"
                                            id="accountant_id"
                                            name="accountant_id"
                                            <?php echo $formDisabled ? 'disabled' : ''; ?>
                                            required
                                        >
                                            <option value="">Sélectionner comptable</option>
                                            <?php foreach ($accountants as $accountant): ?>
                                                <option
                                                    value="<?php echo (int)$accountant['id']; ?>"
                                                    <?php echo $selectedAccountantId === (int)$accountant['id'] ? 'selected' : ''; ?>
                                                >
                                                    <?php echo htmlspecialchars($accountant['full_name'], ENT_QUOTES, 'UTF-8'); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-6 d-flex align-items-end">
                                        <button
                                            type="submit"
                                            class="btn btn-primary w-100"
                                            <?php echo $formDisabled ? 'disabled' : ''; ?>
                                        >
                                            <i class="bi bi-send me-2"></i>
                                            Soumettre Transfert
                                        </button>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Confirmation Modal -->
                    <div class="modal fade" id="confirmTransferModal" tabindex="-1" aria-labelledby="confirmTransferModalLabel" aria-hidden="true">
                        <div class="modal-dialog modal-dialog-centered">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title" id="confirmTransferModalLabel">
                                        <i class="bi bi-exclamation-triangle me-2 text-warning"></i>
                                        Confirmer Transfert d'Espèces
                                    </h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fermer"></button>
                                </div>
                                <div class="modal-body">
                                    <div class="mb-3">
                                        <p class="mb-3">Veuillez confirmer les détails du transfert suivants :</p>
                                        <ul class="list-group list-group-flush">
                                            <li class="list-group-item px-0">
                                                <strong>Session:</strong> <span id="confirm-shift-label">Non sélectionnée</span>
                                            </li>
                                            <li class="list-group-item px-0">
                                                <strong>Montant:</strong> <span id="confirm-amount" class="text-success fw-bold">0 BIF</span>
                                            </li>
                                            <li class="list-group-item px-0">
                                                <strong>Comptable:</strong> <span id="confirm-accountant">Non sélectionné</span>
                                            </li>
                                        </ul>
                                    </div>
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle me-2"></i>
                                        Cette action enregistrera le transfert d'espèces et notifiera le comptable.
                                    </div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                        <i class="bi bi-x me-1"></i>Annuler
                                    </button>
                                    <button type="button" class="btn btn-success" id="confirm-transfer-btn">
                                        <i class="bi bi-check me-1"></i>Confirmer Transfert
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <script>
                    document.addEventListener('DOMContentLoaded', function () {
                        var shiftSelect = document.getElementById('shift_id');
                        var amountInput = document.getElementById('amount');
                        var accountantSelect = document.getElementById('accountant_id');
                        var form = document.querySelector('form');
                        var submitBtn = document.querySelector('button[type="submit"]');
                        var confirmModal = new bootstrap.Modal(document.getElementById('confirmTransferModal'));
                        var confirmBtn = document.getElementById('confirm-transfer-btn');

                        if (!shiftSelect || !amountInput || !accountantSelect || !form) {
                            return;
                        }

                        function syncTransferAmount() {
                            var option = shiftSelect.options[shiftSelect.selectedIndex];
                            if (option && option.dataset.amount) {
                                var amount = parseFloat(option.dataset.amount);
                                amountInput.value = isNaN(amount) ? '' : Math.round(amount).toString();
                            } else {
                                amountInput.value = '';
                            }
                        }

                        function updateConfirmModal() {
                            var selectedShiftOption = shiftSelect.options[shiftSelect.selectedIndex];
                            var selectedAccountantOption = accountantSelect.options[accountantSelect.selectedIndex];

                            document.getElementById('confirm-shift-label').textContent =
                                selectedShiftOption && selectedShiftOption.value ? selectedShiftOption.text : 'Non sélectionné';

                            document.getElementById('confirm-amount').textContent =
                                amountInput.value ? amountInput.value + ' BIF' : '0 BIF';

                            document.getElementById('confirm-accountant').textContent =
                                selectedAccountantOption && selectedAccountantOption.value ? selectedAccountantOption.text : 'Non sélectionné';
                        }

                        function showConfirmationModal(e) {
                            e.preventDefault();

                            // Validate form
                            if (!shiftSelect.value || !accountantSelect.value) {
                                alert('Veuillez sélectionner une session et un comptable avant de continuer.');
                                return;
                            }

                            updateConfirmModal();
                            confirmModal.show();
                        }

                        function setLoadingState(isLoading) {
                            var loadingOverlay = document.getElementById('form-loading-overlay');
                            submitBtn.disabled = isLoading;
                            confirmBtn.disabled = isLoading;

                            if (isLoading) {
                                loadingOverlay.classList.remove('d-none');
                                submitBtn.innerHTML = '<i class="bi bi-hourglass-split me-2"></i><span>Traitement...</span>';
                                confirmBtn.innerHTML = '<i class="bi bi-hourglass-split me-2"></i><span>Traitement...</span>';
                                submitBtn.style.opacity = '0.7';
                                confirmBtn.style.opacity = '0.7';
                            } else {
                                loadingOverlay.classList.add('d-none');
                                submitBtn.innerHTML = '<i class="bi bi-send-check-fill me-2 fs-5"></i><span>Soumettre Transfert</span>';
                                confirmBtn.innerHTML = '<i class="bi bi-check-circle-fill me-2"></i>Confirmer Transfert';
                                submitBtn.style.opacity = '1';
                                confirmBtn.style.opacity = '1';
                            }
                        }

                        function submitForm() {
                            setLoadingState(true);

                            // Create a hidden submit button to trigger actual form submission
                            var hiddenSubmit = document.createElement('input');
                            hiddenSubmit.type = 'submit';
                            hiddenSubmit.style.display = 'none';
                            form.appendChild(hiddenSubmit);
                            hiddenSubmit.click();
                            form.removeChild(hiddenSubmit);

                            // Reset loading state after a delay (in case of client-side validation failure)
                            setTimeout(function() {
                                setLoadingState(false);
                            }, 3000);
                        }

                        shiftSelect.addEventListener('change', syncTransferAmount);
                        submitBtn.addEventListener('click', showConfirmationModal);
                        confirmBtn.addEventListener('click', function() {
                            confirmModal.hide();
                            submitForm();
                        });

                        // Reset loading state when modal is hidden
                        document.getElementById('confirmTransferModal').addEventListener('hidden.bs.modal', function() {
                            setLoadingState(false);
                        });

                        syncTransferAmount();
                    });
                    </script>

                    <!-- Recent Transfers Section -->
                    <div class="card table-card border shadow-sm">
                        <div class="card-header">
                            <h5 class="mb-0 d-flex align-items-center">
                                <i class="bi bi-clock-history me-2 text-primary"></i>
                                Transferts Récents
                            </h5>
                        </div>
                        <div class="card-body p-0">
                            <?php if (empty($transfer_history)): ?>
                                <div class="text-center py-5 px-4">
                                    <div class="bg-light bg-opacity-50 rounded-circle d-inline-flex align-items-center justify-content-center mb-4" style="width: 100px; height: 100px;">
                                        <i class="bi bi-inbox fs-1 text-muted"></i>
                                    </div>
                                    <h6 class="text-muted fw-semibold mb-2">Aucun transfert enregistré</h6>
                                    <p class="text-muted small mb-0">Votre historique de transferts apparaîtra ici une fois que vous effectuerez des transferts.</p>
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead class="table-light">
                                            <tr>
                                                <th><i class="bi bi-hash me-1"></i>ID</th>
                                                <th><i class="bi bi-cash me-1"></i>Montant (BIF)</th>
                                                <th class="d-none d-md-table-cell"><i class="bi bi-person me-1"></i>Comptable</th>
                                                <th><i class="bi bi-flag me-1"></i>Statut</th>
                                                <th class="d-none d-lg-table-cell"><i class="bi bi-send me-1"></i>Envoyé Le</th>
                                                <th class="d-none d-lg-table-cell"><i class="bi bi-check-circle me-1"></i>Reçu</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($transfer_history as $transfer): ?>
                                                <tr>
                                                    <td class="text-primary">#<?php echo (int)$transfer['id']; ?></td>
                                                    <td class="text-success fw-bold"><?php echo number_format((int)$transfer['amount'], 0, ',', '.'); ?> BIF</td>
                                                    <td class="d-none d-md-table-cell"><?php echo htmlspecialchars($transfer['accountant_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></td>
                                                    <td>
                                                        <span class="badge bg-<?php echo match($transfer['status']) {
                                                            'pending' => 'warning',
                                                            'received' => 'success',
                                                            'rejected' => 'danger',
                                                            default => 'secondary'
                                                        }; ?>">
                                                            <?php 
                                                            $status_fr = match($transfer['status']) {
                                                                'pending' => 'En Attente',
                                                                'received' => 'Reçu',
                                                                'rejected' => 'Rejeté',
                                                                default => $transfer['status']
                                                            };
                                                            echo $status_fr; 
                                                            ?>
                                                        </span>
                                                    </td>
                                                    <td class="d-none d-lg-table-cell text-muted">
                                                        <?php echo htmlspecialchars(date('d M Y H:i', strtotime($transfer['transfer_date'])), ENT_QUOTES, 'UTF-8'); ?>
                                                    </td>
                                                    <td class="d-none d-lg-table-cell">
                                                        <?php if ($transfer['received_at']): ?>
                                                            <span class="text-success">
                                                                <i class="bi bi-check-circle me-1"></i>
                                                                <?php echo htmlspecialchars(date('d M Y H:i', strtotime($transfer['received_at'])), ENT_QUOTES, 'UTF-8'); ?>
                                                            </span>
                                                        <?php else: ?>
                                                            <span class="text-muted">En Attente</span>
                                                        <?php endif; ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>

                                <!-- Mobile Cards View for Small Screens -->
                                <div class="d-md-none p-3">
                                    <?php foreach ($transfer_history as $transfer): ?>
                                        <div class="card border-0 shadow-sm mb-3" style="border-radius: 10px; background: #F4F0E4;">
                                            <div class="card-body p-3">
                                                <div class="d-flex justify-content-between align-items-start mb-2">
                                                    <span class="badge bg-primary">#<?php echo (int)$transfer['id']; ?></span>
                                                    <span class="fw-bold text-success"><?php echo number_format((int)$transfer['amount'], 0, ',', '.'); ?> BIF</span>
                                                </div>
                                                <div class="row g-2">
                                                    <div class="col-6">
                                                        <small class="text-muted d-block">Comptable</small>
                                                        <span class="fw-medium"><?php echo htmlspecialchars($transfer['accountant_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></span>
                                                    </div>
                                                    <div class="col-6">
                                                        <small class="text-muted d-block">Statut</small>
                                                        <?php
                                                        $status = $transfer['status'];
                                                        $statusConfig = match($status) {
                                                            'pending' => ['class' => 'warning', 'icon' => 'clock', 'text' => 'En Attente'],
                                                            'received' => ['class' => 'success', 'icon' => 'check-circle-fill', 'text' => 'Reçu'],
                                                            'rejected' => ['class' => 'danger', 'icon' => 'x-circle-fill', 'text' => 'Rejeté'],
                                                            default => ['class' => 'secondary', 'icon' => 'question-circle', 'text' => 'Inconnu']
                                                        };
                                                        ?>
                                                        <span class="badge bg-<?php echo $statusConfig['class']; ?>">
                                                            <i class="bi bi-<?php echo $statusConfig['icon']; ?> me-1"></i>
                                                            <?php echo $statusConfig['text']; ?>
                                                        </span>
                                                    </div>
                                                    <div class="col-12">
                                                        <small class="text-muted d-block">Envoyé Le</small>
                                                        <span><?php echo htmlspecialchars(date('d M Y H:i', strtotime($transfer['transfer_date'])), ENT_QUOTES, 'UTF-8'); ?></span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* System Color Scheme */
:root {
    --dark-brown: #4B2F1F;
    --bright-orange: #F4A261;
    --off-white: #F4F0E4;
    --black: #000000;
}

/* Main container background */
.container-fluid {
    background-color: var(--off-white) !important;
}

/* Card headers */
.card-header {
    background-color: var(--dark-brown) !important;
    color: white !important;
    border-color: var(--black) !important;
}

.card-header.bg-light {
    background-color: var(--dark-brown) !important;
}

/* Primary buttons */
.btn-primary {
    background-color: var(--bright-orange) !important;
    border-color: var(--bright-orange) !important;
    color: var(--black) !important;
}

.btn-primary:hover {
    background-color: #e6934f !important; /* Slightly darker orange */
    border-color: #e6934f !important;
}

/* Icons */
.text-primary {
    color: var(--bright-orange) !important;
}

.text-success {
    color: #28a745 !important; /* Keep green for success */
}

.text-danger {
    color: #dc3545 !important; /* Keep red for danger */
}

.text-warning {
    color: #ffc107 !important; /* Keep yellow for warning */
}

.text-muted {
    color: #6c757d !important; /* Keep muted for secondary text */
}

/* Borders */
.card, .alert, .modal-content {
    border-color: var(--black) !important;
}

/* Form elements */
.form-control, .form-select {
    border-color: var(--black) !important;
}

.form-control:focus, .form-select:focus {
    border-color: var(--bright-orange) !important;
    box-shadow: 0 0 0 0.2rem rgba(244, 162, 97, 0.25) !important;
}

/* Table */
.table th {
    background-color: var(--dark-brown) !important;
    color: white !important;
    border-color: var(--black) !important;
}

.table td {
    border-color: var(--black) !important;
}

/* Badges */
.badge.bg-warning {
    background-color: #ffc107 !important;
}

.badge.bg-success {
    background-color: #28a745 !important;
}

.badge.bg-danger {
    background-color: #dc3545 !important;
}

/* Minimal responsive adjustments */
@media (max-width: 768px) {
    .container-fluid {
        padding-left: 15px;
        padding-right: 15px;
    }

    .card-body {
        padding: 1rem;
    }

    .table th,
    .table td {
        padding: 0.5rem;
        font-size: 0.875rem;
    }
}

@media (max-width: 576px) {
    .card-header h2 {
        font-size: 1.25rem;
    }

    .btn {
        padding: 0.5rem 1rem;
        font-size: 0.9rem;
    }

    .form-label {
        font-size: 0.9rem;
        margin-bottom: 0.25rem;
    }

    .modal-dialog {
        margin: 0.5rem;
    }
}
</style>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
