<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

// Check session variables
if (!isset($_SESSION['user_id']) || !isset($_SESSION['full_name'])) {
    error_log("Session variables missing for user_id or full_name in reprint_receipt.php");
    header("Location: /masunzu_bar_hotel/auth.php");
    exit;
}
require_login();

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in reprint_receipt.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Check for receipt_id
$receipt_id = $_GET['receipt_id'] ?? '';
$is_reprint = isset($_GET['reprint']) && $_GET['reprint'] == '1';
$auto_download = isset($_GET['auto_download']) && $_GET['auto_download'] == '1';
if (empty($receipt_id)) {
    error_log("No receipt_id provided in reprint_receipt.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Aucun reçu spécifié.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Check TCPDF availability
$tcpdf_path = __DIR__ . '/../../libs/tcpdf/tcpdf.php';
$composer_autoload = __DIR__ . '/../../vendor/autoload.php';
$tcpdf_available = file_exists($tcpdf_path) || file_exists($composer_autoload);
if ($tcpdf_available) {
    if (file_exists($composer_autoload)) {
        require_once $composer_autoload;
    } elseif (file_exists($tcpdf_path)) {
        require_once $tcpdf_path;
    }
} else {
    error_log("TCPDF not found at $tcpdf_path or $composer_autoload");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>La génération de PDF est désactivée. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Fetch receipt details
// First, get the receipt header info
$stmt = $conn->prepare("
    SELECT r.receipt_id, r.transaction_type, r.transaction_id, r.receipt_number,
           r.customer_name, r.phone_number, r.total_amount, r.subtotal_amount, r.tax_amount, r.tax_rate,
           r.created_at,
           CASE 
               WHEN r.transaction_type = 'direct_sale' THEN o.payment_method
               WHEN r.transaction_type = 'invoice' THEN pi.payment_method
           END AS payment_method,
           CASE 
               WHEN r.transaction_type = 'invoice' THEN pi.nif
               ELSE NULL
           END AS nif,
           CASE 
               WHEN r.transaction_type = 'invoice' THEN pi.rc
               ELSE NULL
           END AS rc,
           CASE 
               WHEN r.transaction_type = 'invoice' THEN pi.payment_details
               ELSE NULL
           END AS payment_details,
           CASE 
               WHEN r.transaction_type = 'invoice' THEN pi.is_loan_sale
               ELSE 0
           END AS is_loan_sale
    FROM receipts r
    LEFT JOIN `order` o ON r.transaction_id = o.id AND r.transaction_type = 'direct_sale'
    LEFT JOIN paid_invoices pi ON r.transaction_id = pi.id AND r.transaction_type = 'invoice'
    WHERE r.receipt_id = ? AND r.created_by = ?
    LIMIT 1
");
$stmt->bind_param("si", $receipt_id, $_SESSION['user_id']);
$stmt->execute();
$receipt_header = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$receipt_header) {
    error_log("Receipt ID $receipt_id not found or unauthorized access by user {$_SESSION['user_id']}");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Reçu introuvable ou vous n'êtes pas autorisé à le réimprimer.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Fetch receipt items based on transaction type
$receipts = [];
if ($receipt_header['transaction_type'] === 'direct_sale') {
    // For direct sales, get from order table
    $stmt = $conn->prepare("
        SELECT p.name AS product_name, 
               o.quantity,
               CASE 
                   WHEN o.quantity > 0 THEN o.paid_amount / o.quantity
                   ELSE 0
               END AS price_per_unit
        FROM `order` o
        JOIN product p ON o.product_id = p.id
        WHERE o.id = ?
    ");
    $stmt->bind_param("i", $receipt_header['transaction_id']);
    $stmt->execute();
    $receipts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    // For invoices, get from paid_invoice_items (or invoice_items as fallback for loan sales)
    $transaction_id = (int)($receipt_header['transaction_id'] ?? 0);
    error_log("DEBUG reprint_receipt.php: Looking for items in paid_invoice_items with invoice_id={$transaction_id} for receipt_id={$receipt_id}");
    
    // First try paid_invoice_items
    $checkStmt = $conn->prepare("SELECT COUNT(*) as count FROM paid_invoice_items WHERE invoice_id = ?");
    if ($checkStmt) {
        $checkStmt->bind_param("i", $transaction_id);
        $checkStmt->execute();
        $checkResult = $checkStmt->get_result()->fetch_assoc();
        $itemCount = (int)($checkResult['count'] ?? 0);
        $checkStmt->close();
        error_log("DEBUG reprint_receipt.php: Found {$itemCount} items in paid_invoice_items for invoice_id={$transaction_id}");
    }
    
    $stmt = $conn->prepare("
        SELECT p.name AS product_name,
               pii.quantity,
               CASE 
                   WHEN pii.quantity > 0 THEN pii.unit_price
                   ELSE 0
               END AS price_per_unit
        FROM paid_invoice_items pii
        JOIN product p ON pii.product_id = p.id
        WHERE pii.invoice_id = ?
        ORDER BY pii.id
    ");
    if (!$stmt) {
        error_log("ERROR reprint_receipt.php: Failed to prepare query for paid_invoice_items: " . $conn->error);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur lors de la récupération des articles: " . htmlspecialchars($conn->error, ENT_QUOTES, 'UTF-8') . "</p>";
        include __DIR__ . '/../../includes/footer.php';
        exit;
    }
    $stmt->bind_param("i", $transaction_id);
    if (!$stmt->execute()) {
        $errorMsg = $stmt->error ?: 'unknown error';
        error_log("ERROR reprint_receipt.php: Failed to execute query for paid_invoice_items: {$errorMsg}");
        $stmt->close();
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur lors de la récupération des articles: " . htmlspecialchars($errorMsg, ENT_QUOTES, 'UTF-8') . "</p>";
        include __DIR__ . '/../../includes/footer.php';
        exit;
    }
    $receipts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    
    // If no items found in paid_invoice_items, try invoice_items (for loan sales that may not have been copied yet)
    if (empty($receipts)) {
        error_log("DEBUG reprint_receipt.php: No items in paid_invoice_items, trying invoice_items for invoice_id={$transaction_id}");
        $stmt = $conn->prepare("
            SELECT p.name AS product_name,
                   ii.quantity,
                   CASE 
                       WHEN ii.quantity > 0 THEN ii.unit_price
                       ELSE 0
                   END AS price_per_unit
            FROM invoice_items ii
            JOIN product p ON ii.product_id = p.id
            WHERE ii.invoice_id = ?
            ORDER BY ii.id
        ");
        if ($stmt) {
            $stmt->bind_param("i", $transaction_id);
            if ($stmt->execute()) {
                $receipts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                error_log("DEBUG reprint_receipt.php: Retrieved " . count($receipts) . " items from invoice_items for invoice_id={$transaction_id}");
            }
            $stmt->close();
        }
    } else {
        error_log("DEBUG reprint_receipt.php: Retrieved " . count($receipts) . " items from paid_invoice_items for invoice_id={$transaction_id}");
    }
}

if (empty($receipts)) {
    $transaction_id = (int)($receipt_header['transaction_id'] ?? 0);
    error_log("ERROR reprint_receipt.php: No items found for receipt ID $receipt_id (transaction_id={$transaction_id}, transaction_type={$receipt_header['transaction_type']})");
    
    // Try to find items in invoice_items as fallback (in case they weren't copied yet)
    if ($receipt_header['transaction_type'] === 'invoice') {
        // Get original invoice_id from paid_invoices
        $fallbackStmt = $conn->prepare("SELECT id FROM paid_invoices WHERE id = ? LIMIT 1");
        if ($fallbackStmt) {
            $fallbackStmt->bind_param("i", $transaction_id);
            $fallbackStmt->execute();
            $paidInvoice = $fallbackStmt->get_result()->fetch_assoc();
            $fallbackStmt->close();
            
            if ($paidInvoice) {
                error_log("DEBUG reprint_receipt.php: Paid invoice exists with id={$transaction_id}, but no items found in paid_invoice_items");
            }
        }
    }
    
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Aucun article trouvé pour ce reçu (transaction_id={$transaction_id}). Veuillez contacter l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Generate PDF - Half A4 size (A4 width, half A4 height)
$pdf = new TCPDF('P', 'mm', array(210, 148.5)); // Half A4: 210mm x 148.5mm
$pdf->SetCreator('Masunzu Bar Hotel');
$pdf->SetAuthor($_SESSION['full_name']);
$title = $receipt_header['transaction_type'] === 'direct_sale' ? 'Reçu ' . $receipt_header['receipt_number'] : 'Commande ' . $receipt_header['receipt_number'];
if ($is_reprint) {
    $title .= ' (RÉIMPRIMÉ)';
}
$pdf->SetTitle($title);
$pdf->SetMargins(8, 8, 8);
$pdf->SetAutoPageBreak(false); // Disable auto page break for half page
$pdf->AddPage();
$pdf->SetFont('helvetica', '', 8);

// Add watermark if reprint - add it AFTER all content is written
// We'll add it at the end using StartTransform/StopTransform to avoid affecting content

$transactionLabel = $receipt_header['transaction_type'] === 'direct_sale' ? 'Reçu Détail' : 'Commande Gros';
$receiptDate = date('Y-m-d H:i', strtotime($receipt_header['created_at']));
$total_amount = 0;

$html = '<style>
            .receipt-header { color: #4B2F1F; font-size: 12px; margin-bottom: 4px; }
            .reprint-badge {
                background-color: #FF0000;
                color: #FFFFFF;
                padding: 2px 8px;
                border-radius: 3px;
                font-size: 7px;
                font-weight: bold;
                display: inline-block;
                margin-left: 8px;
            }
            .info-table, .items-table, .summary-table { width: 100%; border-collapse: collapse; }
            .info-table td { font-size: 7px; padding: 1px 3px; color: #000000; }
            .items-table th { font-size: 7px; padding: 2px 3px; background-color: #4B2F1F; color: #F4F0E4; }
            .items-table td { font-size: 7px; padding: 2px 3px; color: #000000; }
            .summary-table td { font-size: 7px; padding: 2px 3px; color: #000000; }
            .summary-table tr:nth-child(even) { background-color: #F4F0E4; }
            .invoice-type-badge {
                background-color: #FFA500;
                color: #000000;
                padding: 2px 6px;
                border-radius: 3px;
                font-size: 7px;
                font-weight: bold;
                display: inline-block;
            }
            .invoice-type-badge.loan {
                background-color: #FFD700;
            }
        </style>';

// Fetch company information for the receipt (shared company info + depot name)
$province_id = null;
// Try to get province from user session if available
if (isset($_SESSION['province_id'])) {
    $province_id = $_SESSION['province_id'];
} else {
    // Get province from the user who created the receipt
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $user_result = $stmt->get_result()->fetch_assoc();
        if ($user_result) {
            $province_id = $user_result['province_id'];
        }
        $stmt->close();
    }
}

// Get shared company info with depot name
$company_info = mbh_get_company_info_for_receipt($conn, $province_id);

$logo_html = '';
if (!empty($company_info['logo_path'])) {
    $logo_path = __DIR__ . '/../../' . $company_info['logo_path'];
    if (file_exists($logo_path)) {
        $logo_html = '<img src="' . $logo_path . '" alt="Company Logo" style="width: 35px; height: 35px;" />';
    }
}

// Build company name with depot
$companyNameDisplay = htmlspecialchars($company_info['company_name'] ?? 'Masunzu Bar Hotel', ENT_QUOTES, 'UTF-8');
if (!empty($company_info['depot_name'])) {
    $companyNameDisplay .= ' - ' . htmlspecialchars($company_info['depot_name'], ENT_QUOTES, 'UTF-8');
}

$html .= '<table style="width: 100%; border-collapse: collapse; margin-bottom: 6px; border-bottom: 1px solid #4B2F1F; padding-bottom: 4px;">
            <tr>
                <td style="width: 40px; vertical-align: top; padding-right: 6px;">' . $logo_html . '</td>
                <td style="vertical-align: top; text-align: left;">
                    <h2 style="color: #4B2F1F; font-size: 10px; margin: 0 0 1px 0; font-weight: bold;">' . $companyNameDisplay . '</h2>
                    <p style="color: #666666; font-size: 7px; margin: 0 0 2px 0; font-weight: 600;">'. ucfirst($transactionLabel);
if ($is_reprint) {
    $html .= ' &mdash; <span style="color: #FF0000; font-weight: bold;">RÉIMPRIMÉ</span>';
}
$html .= '</p>
                    <table style="width: 100%; border-collapse: collapse;">
                        <tr>
                            <td style="font-size: 6px; color: #444444; padding: 0.5px 0;"><strong>Adresse:</strong> ' . htmlspecialchars($company_info['address'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>
                        </tr>
                        <tr>
                            <td style="font-size: 6px; color: #444444; padding: 0.5px 0;"><strong>Tél:</strong> ' . htmlspecialchars($company_info['phone1'] ?? '', ENT_QUOTES, 'UTF-8');
if (!empty($company_info['phone2'])) {
    $html .= ' | ' . htmlspecialchars($company_info['phone2'], ENT_QUOTES, 'UTF-8');
}
$html .= '</td>
                        </tr>
                        <tr>
                            <td style="font-size: 6px; color: #444444; padding: 0.5px 0;"><strong>NIF:</strong> ' . htmlspecialchars($company_info['nif'] ?? '', ENT_QUOTES, 'UTF-8') . ' <strong>RC:</strong> ' . htmlspecialchars($company_info['rc'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>
                        </tr>
                    </table>
                </td>
            </tr>
        </table>';

// Determine invoice type and payment method
$is_loan_invoice = !empty($receipt_header['is_loan_sale']);
$invoice_type_label = $is_loan_invoice ? 'VENTE À CRÉDIT' : 'PAIEMENT NORMAL';
$invoice_type_class = $is_loan_invoice ? 'loan' : '';

$payment_method_display = $receipt_header['payment_method'] ?? 'N/A';
if ($receipt_header['transaction_type'] === 'invoice') {
    // Check if it's a loan sale or if payment_details mentions credit
    if ($is_loan_invoice || strpos($receipt_header['payment_details'] ?? '', 'Crédit') !== false || strpos($receipt_header['payment_details'] ?? '', 'Type Paiement: Crédit') !== false) {
        $payment_method_display = 'Crédit';
    } else {
        $payment_method_display = ucfirst(str_replace('_', ' ', $payment_method_display));
        $payment_method_display = str_replace(['Cash', 'Mobile', 'Bank'], ['Espèces', 'Mobile', 'Banque'], $payment_method_display);
    }
} else {
    $payment_method_display = ucfirst($payment_method_display);
    $payment_method_display = str_replace(['Cash', 'Mobile', 'Bank'], ['Espèces', 'Mobile', 'Banque'], $payment_method_display);
}

$html .= '<table class="info-table" cellspacing="0" cellpadding="0">
            <tr>
                <td><strong>Numéro:</strong> ' . htmlspecialchars($receipt_header['receipt_number'], ENT_QUOTES, 'UTF-8') . '</td>
                <td><strong>Date:</strong> ' . $receiptDate . '</td>
            </tr>';
if ($receipt_header['transaction_type'] === 'invoice') {
    $html .= '<tr>
                <td colspan="2"><strong>Type Facture:</strong> <span class="invoice-type-badge ' . $invoice_type_class . '">' . $invoice_type_label . '</span></td>
             </tr>';
}
$html .= '<tr>
                <td><strong>Client:</strong> ' . htmlspecialchars($receipt_header['customer_name'], ENT_QUOTES, 'UTF-8') . '</td>
                <td><strong>Téléphone:</strong> ' . htmlspecialchars($receipt_header['phone_number'], ENT_QUOTES, 'UTF-8') . '</td>
            </tr>';
if ($receipt_header['transaction_type'] === 'invoice') {
    $html .= '<tr>
                <td><strong>NIF:</strong> ' . htmlspecialchars($receipt_header['nif'] ?? 'N/A', ENT_QUOTES, 'UTF-8') . '</td>
                <td><strong>RC:</strong> ' . htmlspecialchars($receipt_header['rc'] ?? 'N/A', ENT_QUOTES, 'UTF-8') . '</td>
             </tr>';
}
$html .= '<tr>
                <td><strong>Méthode de Paiement:</strong> ' . htmlspecialchars($payment_method_display, ENT_QUOTES, 'UTF-8') . '</td>
                <td></td>
              </tr>';
if ($is_loan_invoice) {
    $html .= '<tr>
                <td colspan="2" style="color: #FF6600; font-weight: bold; font-size: 7px;">⚠️ Aucun argent reçu - Crédit à rembourser ultérieurement</td>
              </tr>';
}
$html .= '</table>';

$html .= '<h3 style="color: #4B2F1F; font-size: 9px; margin-top: 4px; margin-bottom: 2px;">Articles</h3>
          <table class="items-table" border="1" cellspacing="0" cellpadding="0">
            <tr>
                <th>Produit</th>
                <th>Qté</th>
                <th>Prix/Unité</th>
                <th>Total</th>
            </tr>';

foreach ($receipts as $receipt) {
    $lineTotal = $receipt['price_per_unit'] * $receipt['quantity'];
    $total_amount += $lineTotal;
    $html .= '<tr>
                <td>' . htmlspecialchars($receipt['product_name'], ENT_QUOTES, 'UTF-8') . '</td>
                <td>' . htmlspecialchars($receipt['quantity'], ENT_QUOTES, 'UTF-8') . '</td>
                <td>' . number_format($receipt['price_per_unit'], 2, ',', '.') . ' BIF</td>
                <td>' . number_format($lineTotal, 2, ',', '.') . ' BIF</td>
              </tr>';
}

$html .= '</table>';

$html .= '<h3 style="color: #4B2F1F; font-size: 9px; margin-top: 4px; margin-bottom: 2px;">Totaux</h3>
          <table class="summary-table" border="1" cellspacing="0" cellpadding="0">';
if ($receipt_header['transaction_type'] === 'invoice' && isset($receipt_header['subtotal_amount'])) {
    $html .= '<tr>
                <td><strong>Sous-Total</strong></td>
                <td>' . number_format($receipt_header['subtotal_amount'], 2, ',', '.') . ' BIF</td>
            </tr>';
    if (isset($receipt_header['tax_amount']) && $receipt_header['tax_amount'] > 0) {
        $html .= '<tr>
                    <td><strong>Taxe (' . number_format($receipt_header['tax_rate'] ?? 0, 2, ',', '.') . '%)</strong></td>
                    <td>' . number_format($receipt_header['tax_amount'], 2, ',', '.') . ' BIF</td>
                </tr>';
    }
}
$html .= '<tr>
            <td><strong>Montant Total</strong></td>
            <td>' . number_format($receipt_header['total_amount'] ?? $total_amount, 2, ',', '.') . ' BIF</td>
        </tr>';
if ($receipt_header['transaction_type'] === 'direct_sale') {
    $payment_method_display = ucfirst($receipt_header['payment_method'] ?? 'N/A');
    $payment_method_display = str_replace(['Cash', 'Mobile', 'Bank'], ['Espèces', 'Mobile', 'Banque'], $payment_method_display);
    $html .= '<tr>
                <td><strong>Méthode de Paiement</strong></td>
                <td>' . htmlspecialchars($payment_method_display, ENT_QUOTES, 'UTF-8') . '</td>
             </tr>';
} else {
    $payment_method_display = $receipt_header['payment_method'] ?? 'N/A';
    if (strpos($receipt_header['payment_details'] ?? '', 'Payment Type: Loan') !== false) {
        $payment_method_display = 'Crédit';
    } else {
        $payment_method_display = ucfirst(str_replace('_', ' ', $payment_method_display));
        $payment_method_display = str_replace(['Cash', 'Mobile', 'Bank'], ['Espèces', 'Mobile', 'Banque'], $payment_method_display);
    }
    $html .= '<tr>
                <td><strong>Méthode de Paiement</strong></td>
                <td>' . htmlspecialchars($payment_method_display, ENT_QUOTES, 'UTF-8') . '</td>
             </tr>';
}
$html .= '</table>';

$pdf->writeHTML($html, true, false, true, false, '');

// Add watermark overlay after content is written (if reprint)
if ($is_reprint) {
    // Get page dimensions
    $pageWidth = $pdf->getPageWidth();
    $pageHeight = $pdf->getPageHeight();
    $centerX = $pageWidth / 2;
    $centerY = $pageHeight / 2;
    
    // Save current state and add watermark
    $pdf->StartTransform();
    
    // Set transparency and color for watermark
    $pdf->SetAlpha(0.15);
    $pdf->SetTextColor(255, 0, 0);
    $pdf->SetFont('helvetica', 'B', 72);
    
    // Rotate around center point
    $pdf->Rotate(45, $centerX, $centerY);
    
    // Write watermark text centered
    $pdf->Text($centerX - 80, $centerY, 'RÉIMPRIMÉ');
    
    // Restore state (this stops the rotation)
    $pdf->StopTransform();
    
    // Reset to normal state
    $pdf->SetAlpha(1);
    $pdf->SetTextColor(0, 0, 0);
    $pdf->SetFont('helvetica', '', 10);
}

$pdf->Output($receipt_header['transaction_type'] === 'direct_sale' ? 'receipt_' . $receipt_header['receipt_number'] . '.pdf' : 'invoice_' . $receipt_header['receipt_number'] . '.pdf', 'D');

// Log the reprint action
$stmt = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'reprint_receipt', ?, NOW())");
$description = "Reprinted receipt/invoice {$receipt_header['receipt_number']} (ID: $receipt_id)";
$stmt->bind_param("is", $_SESSION['user_id'], $description);
$stmt->execute();
$stmt->close();

$conn->close();
exit;
?>
