<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

/**
 * Helps locate a customer when an invoice lacks a direct customer ID.
 */
function find_customer_by_name_and_phone(mysqli $conn, string $name, string $phone): ?int
{
    $name = trim($name);
    $phone = trim($phone);
    if ($name === '' || $phone === '') {
        return null;
    }
    $stmt = $conn->prepare("SELECT id FROM customer WHERE full_name = ? AND tel = ? LIMIT 1");
    if (!$stmt) {
        error_log("find_customer_by_name_and_phone: prepare failed - " . $conn->error);
        return null;
    }
    $stmt->bind_param('ss', $name, $phone);
    if (!$stmt->execute()) {
        error_log("find_customer_by_name_and_phone: execute failed - " . $stmt->error);
        $stmt->close();
        return null;
    }
    $row = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    return $row ? (int)$row['id'] : null;
}

$loanPaymentAutoIncrement = false;
$paidInvoiceItemsAutoIncrement = false;
$paidInvoicesAutoIncrement = false;
$bankDepositAutoIncrement = false;
$bankAccountTxnAutoIncrement = false;

// Generate UUID for receipt_id
function generate_uuid() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        mt_rand(0, 0xffff), mt_rand(0, 0xffff),
        mt_rand(0, 0xffff),
        mt_rand(0, 0x0fff) | 0x4000,
        mt_rand(0, 0x3fff) | 0x8000,
        mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
    );
}

/**
 * Ensure a table column is AUTO_INCREMENT. Returns true if column already was or was altered successfully.
 */
function ensure_auto_increment($conn, $table, $column, $alterSql, $logLabel) {
    try {
        $checkSql = sprintf("SHOW COLUMNS FROM `%s` LIKE '%s'", $conn->real_escape_string($table), $conn->real_escape_string($column));
        $result = $conn->query($checkSql);
        if ($result && $result->num_rows > 0) {
            $col = $result->fetch_assoc();
            $result->free();
            $extra = strtolower($col['Extra'] ?? '');
            if (strpos($extra, 'auto_increment') === false) {
                if ($conn->query($alterSql)) {
                    error_log("✅ {$logLabel}: Added AUTO_INCREMENT.");
                    return true;
                }
                error_log("❌ {$logLabel}: Failed to add AUTO_INCREMENT - " . $conn->error);
                return false;
            }
            error_log("✅ {$logLabel}: Already AUTO_INCREMENT.");
            return true;
        } elseif ($result) {
            $result->free();
        }
        error_log("⚠️ {$logLabel}: Column or table not found while checking AUTO_INCREMENT.");
    } catch (Exception $e) {
        error_log("⚠️ {$logLabel}: Error while ensuring AUTO_INCREMENT - " . $e->getMessage());
    }
    return false;
}

/**
 * Fetch the next numeric ID for a table that lacks AUTO_INCREMENT.
 * Uses SELECT ... FOR UPDATE to avoid race conditions inside the current transaction.
 *
 * @throws Exception when the query fails.
 */
function get_next_manual_id($conn, $table) {
    $tableEscaped = $conn->real_escape_string($table);
    $sql = "SELECT id FROM `{$tableEscaped}` ORDER BY id DESC LIMIT 1 FOR UPDATE";
    $result = $conn->query($sql);
    if (!$result) {
        throw new Exception("Failed to fetch next ID for {$table}: " . $conn->error);
    }
    $row = $result->fetch_assoc();
    $result->free();
    return $row ? ((int)$row['id'] + 1) : 1;
}

// Restrict to Caissier_Comptable role
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in process_payment.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Ensure loan ledger entry_origin column exists
mbh_ensure_loan_ledger_entry_origin($conn);

$loanPaymentAutoIncrement = ensure_auto_increment(
    $conn,
    'loan_payment',
    'id',
    "ALTER TABLE loan_payment MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT",
    'loan_payment.id'
);
$paidInvoiceItemsAutoIncrement = ensure_auto_increment(
    $conn,
    'paid_invoice_items',
    'id',
    "ALTER TABLE paid_invoice_items MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT",
    'paid_invoice_items.id'
);
$paidInvoicesAutoIncrement = ensure_auto_increment(
    $conn,
    'paid_invoices',
    'id',
    "ALTER TABLE paid_invoices MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT",
    'paid_invoices.id'
);
$bankDepositAutoIncrement = ensure_auto_increment(
    $conn,
    'bank_deposit',
    'id',
    "ALTER TABLE bank_deposit MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT",
    'bank_deposit.id'
);
$bankAccountTxnAutoIncrement = ensure_auto_increment(
    $conn,
    'bank_account_transaction',
    'id',
    "ALTER TABLE bank_account_transaction MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT",
    'bank_account_transaction.id'
);

try {
    // Check user role
    $stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
    if (!$stmt) {
        throw new Exception("Prepare failed for role check: " . $conn->error);
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
    $stmt->close();
    if ($role !== 'Caissier_Comptable') {
        error_log("Unauthorized access to process_payment.php by user {$_SESSION['full_name']} (Role: $role)");
        header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
        exit;
    }

    // Check and initialize active shift
    $stmt = $conn->prepare("SELECT id, deposits, end_balance FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL");
    if (!$stmt) {
        throw new Exception("Prepare failed for shift check: " . $conn->error);
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
   $shift_result = $stmt->get_result();
   $active_shift = $shift_result->fetch_assoc();
   $stmt->close();
    
    if (!$active_shift) {
        error_log("No active shift for cashier_id {$_SESSION['user_id']}. Payment processing aborted.");
        
        include __DIR__ . '/../../includes/header.php';
        ?>
        <div style="padding: 40px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; display: flex; align-items: center; justify-content: center;">
            <div style="max-width: 600px; background: #FDFBF7; padding: 40px; border-radius: 12px; box-shadow: 0 8px 16px rgba(0,0,0,0.2); text-align: center;">
                <div style="font-size: 48px; margin-bottom: 20px;">⚠️</div>
                <h1 style="color: #D9534F; margin: 0 0 15px; font-size: 24px;">Aucune Session Active</h1>
                <p style="color: #666; font-size: 16px; margin: 0 0 30px; line-height: 1.6;">
                    Vous devez démarrer une session caissier avant de traiter les paiements. Veuillez aller à la Gestion des Soldes pour initier votre session.
                </p>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                    <a href="/masunzu_bar_hotel/dashboards/cashier_dashboard.php" 
                       style="background-color: #4B2F1F; color: #FDFBF7; padding: 12px 20px; border-radius: 8px; text-decoration: none; font-weight: 600; display: inline-block; transition: all 0.3s;">
                        ← Retour au Tableau de Bord
                    </a>
                    <a href="/masunzu_bar_hotel/modules/cashiers/balance_management.php" 
                       style="background-color: #2A9D8F; color: #FDFBF7; padding: 12px 20px; border-radius: 8px; text-decoration: none; font-weight: 600; display: inline-block; transition: all 0.3s;">
                        ⏱️ Démarrer Session
                    </a>
                </div>
            </div>
        </div>
        <?php
        include __DIR__ . '/../../includes/footer.php';
        exit;
    } elseif ($active_shift['end_balance'] === null) {
        error_log("Active shift has NULL end_balance for cashier_id {$_SESSION['user_id']}, shift_id={$active_shift['id']}. Initializing to 0.00.");
        $stmt = $conn->prepare("UPDATE cashier_balance SET end_balance = 0.00 WHERE id = ? AND cashier_id = ? AND shift_end IS NULL");
        $stmt->bind_param("ii", $active_shift['id'], $_SESSION['user_id']);
        $stmt->execute();
        $active_shift['end_balance'] = 0.00;
        $stmt->close();
    }
    error_log("Active shift for cashier_id {$_SESSION['user_id']}: shift_id={$active_shift['id']}, deposits={$active_shift['deposits']}, end_balance={$active_shift['end_balance']}");

    // Fetch province_id
    $province_id = $_SESSION['province_id'] ?? null;
    if ($province_id === null || $province_id === '') {
        $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if (!$stmt) {
            throw new Exception("Prepare failed for province check: " . $conn->error);
        }
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $province_id = $stmt->get_result()->fetch_assoc()['province_id'] ?? null;
        $_SESSION['province_id'] = $province_id;
        $stmt->close();
    }
    $province_id = $_SESSION['province_id'] ?? null;
    if ($province_id === null || $province_id === '') {
        error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: No province assigned. Contact administrator.</p>";
        include __DIR__ . '/../../includes/footer.php';
        exit;
    }
    $province_id = (int)$province_id;

    // Check for TCPDF availability
    $tcpdf_path = __DIR__ . '/../../libs/tcpdf/tcpdf.php';
    $composer_autoload = __DIR__ . '/../../vendor/autoload.php';
    $tcpdf_available = file_exists($tcpdf_path) || file_exists($composer_autoload);
    if ($tcpdf_available) {
        if (file_exists($composer_autoload)) {
            require_once $composer_autoload;
        } elseif (file_exists($tcpdf_path)) {
            require_once $tcpdf_path;
        }
    } else {
        error_log("TCPDF not found at $tcpdf_path or $composer_autoload");
    }

    // Check if paid_invoice_items table exists
    $paid_invoice_items_exists = false;
    $result = $conn->query("SHOW TABLES LIKE 'paid_invoice_items'");
    if ($result && $result->num_rows > 0) {
        $paid_invoice_items_exists = true;

        $paid_unit = $conn->query("SHOW COLUMNS FROM paid_invoice_items LIKE 'unit_price'");
        if ($paid_unit && $paid_unit->num_rows == 0) {
            $legacy_paid = $conn->query("SHOW COLUMNS FROM paid_invoice_items LIKE 'wholesale_price'");
            if ($legacy_paid && $legacy_paid->num_rows > 0) {
                $conn->query("ALTER TABLE paid_invoice_items CHANGE COLUMN wholesale_price unit_price BIGINT(20) NOT NULL");
            } else {
                $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN unit_price BIGINT(20) NOT NULL AFTER quantity");
            }
        }
        $conn->query("ALTER TABLE paid_invoice_items DROP COLUMN IF EXISTS wholesale_price");
        $conn->query("ALTER TABLE paid_invoice_items DROP COLUMN IF EXISTS retail_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS price_source VARCHAR(20) DEFAULT 'system' AFTER unit_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS system_unit_price BIGINT(20) DEFAULT NULL AFTER price_source");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_unit_price BIGINT(20) DEFAULT NULL AFTER system_unit_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_price_basis VARCHAR(50) DEFAULT NULL AFTER custom_unit_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_price_input BIGINT(20) DEFAULT NULL AFTER custom_price_basis");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_price_note TEXT DEFAULT NULL AFTER custom_price_input");
    }

    // Fetch company information (shared company info + depot name)
    $company_info = mbh_get_company_info_for_receipt($conn, $province_id);

    // Fetch unpaid and approved invoices (exclude loan invoices that have been processed, and invoices that have been paid)
    $invoices = [];
    $stmt = $conn->prepare("
        SELECT ui.id,
               ui.invoice_number,
               ui.customer_name,
               ui.phone_number,
               ui.nif,
               ui.rc,
               ui.is_loan_sale,
               COALESCE(SUM(ii.quantity), 0) AS total_quantity,
               ui.paid_amount,
               ui.subtotal_amount,
               ui.tax_amount,
               ui.tax_rate,
               ui.tax_mode,
               ui.created_at,
               ui.approval_status,
               ui.approved_by
        FROM unpaid_invoices ui
        LEFT JOIN invoice_items ii ON ui.id = ii.invoice_id
        WHERE ui.province_id = ?
          AND ui.approval_status = 'approved'
          AND (ui.status IS NULL OR ui.status = '' OR ui.status != 'processed')
          AND NOT EXISTS (
              SELECT 1 FROM paid_invoices pi 
              WHERE pi.invoice_number = ui.invoice_number 
              AND pi.customer_name = ui.customer_name
              AND pi.created_at >= DATE_SUB(ui.created_at, INTERVAL 1 DAY)
          )
        GROUP BY ui.id, ui.invoice_number, ui.customer_name, ui.phone_number, ui.nif, ui.rc, ui.is_loan_sale, ui.paid_amount, ui.subtotal_amount, ui.tax_amount, ui.tax_rate, ui.tax_mode, ui.created_at
        ORDER BY ui.created_at DESC
    ");
    if (!$stmt) {
        throw new Exception("Query preparation failed for invoice fetch: " . $conn->error);
    }
    $stmt->bind_param("i", $province_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $invoices = $result->fetch_all(MYSQLI_ASSOC);
    error_log("Fetched " . count($invoices) . " unpaid and approved invoices for province_id=$province_id at " . date('Y-m-d H:i:s'));
    error_log("Sample invoice data: " . json_encode(array_slice($invoices, 0, 1)));

    // Fetch invoice items for each invoice (for PDF generation)
    $invoice_items = [];
    foreach ($invoices as $invoice) {
        $stmt = $conn->prepare("
            SELECT ii.product_id, p.name AS product_name, ii.quantity, ii.unit_price
            FROM invoice_items ii
            JOIN product p ON ii.product_id = p.id
            WHERE ii.invoice_id = ?
        ");
        $stmt->bind_param("i", $invoice['id']);
        $stmt->execute();
        $invoice_items[$invoice['id']] = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
    error_log("Invoice items: " . json_encode($invoice_items));

    // Fetch available bank accounts (limited to current province or global)
    $bank_accounts = [];
    $bankAccountTableExists = $conn->query("SHOW TABLES LIKE 'bank_account'");
    if ($bankAccountTableExists && $bankAccountTableExists->num_rows > 0) {
        $stmtBank = $conn->prepare("
            SELECT id, account_number, currency, is_main, province_id
            FROM bank_account
            WHERE province_id IS NULL OR province_id = ?
            ORDER BY is_main DESC, account_number ASC
        ");
        if ($stmtBank) {
            $stmtBank->bind_param("i", $province_id);
            $stmtBank->execute();
            $bank_accounts = $stmtBank->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtBank->close();
        }
    }

    // Debug: Check all unpaid_invoices and paid_invoices
    if (empty($invoices)) {
            $stmt_debug = $conn->prepare("
            SELECT COUNT(*) as total, 
                   COUNT(CASE WHEN approval_status = 'approved' THEN 1 END) as approved_count,
                   GROUP_CONCAT(id) as invoice_ids
            FROM unpaid_invoices 
            WHERE province_id = ?
        ");
        $stmt_debug->bind_param("i", $province_id);
        $stmt_debug->execute();
        $unpaid_debug = $stmt_debug->get_result()->fetch_assoc();
        error_log("Unpaid invoices debug for province_id=$province_id: Total={$unpaid_debug['total']}, Approved={$unpaid_debug['approved_count']}, IDs={$unpaid_debug['invoice_ids']}");

        $stmt_debug = $conn->prepare("
            SELECT COUNT(*) as total, 
                   COUNT(CASE WHEN approval_status = 'approved' THEN 1 END) as approved_count,
                   GROUP_CONCAT(id) as invoice_ids
            FROM paid_invoices 
            WHERE province_id = ?
        ");
        $stmt_debug->bind_param("i", $province_id);
        $stmt_debug->execute();
        $paid_debug = $stmt_debug->get_result()->fetch_assoc();
        error_log("Paid invoices debug for province_id=$province_id: Total={$paid_debug['total']}, Approved={$paid_debug['approved_count']}, IDs={$paid_debug['invoice_ids']}");
        $stmt_debug->close();
    }

    $allowed_tax_modes = ['HTVA', 'TVAC'];
    $selected_tax_mode = 'HTVA';

    // Handle form submission
    $errors = [];
    $success = '';
    // Get errors from session (persisted after redirect)
    if (!empty($_SESSION['cashier_payment_errors'])) {
        $errors = $_SESSION['cashier_payment_errors'];
        error_log("Retrieved errors from session: " . print_r($errors, true));
        unset($_SESSION['cashier_payment_errors']);
    }
    if (!empty($_SESSION['cashier_payment_success'])) {
        $success = $_SESSION['cashier_payment_success'];
        error_log("Retrieved success message from session: " . substr($success, 0, 100));
        unset($_SESSION['cashier_payment_success']);
    }
    $receipt_id_for_print = $_SESSION['cashier_payment_receipt_id'] ?? null;
    $is_loan_payment_display = !empty($_SESSION['cashier_payment_is_loan']);
    if ($is_loan_payment_display) {
        unset($_SESSION['cashier_payment_is_loan']);
    }
    // Check for auto-print from session or URL parameter
    $auto_print_receipt = !empty($_SESSION['cashier_payment_auto_print']) || (isset($_GET['print_receipt']) && $_GET['print_receipt'] == '1');
    if (!empty($_SESSION['cashier_payment_auto_print'])) {
        unset($_SESSION['cashier_payment_auto_print']);
    }
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit'])) {
        error_log("=== PAYMENT PROCESSING STARTED ===");
        error_log("POST data: " . print_r($_POST, true));
        
        // Generate and validate CSRF token to prevent double submission
        if (empty($_SESSION['payment_token'])) {
            $_SESSION['payment_token'] = bin2hex(random_bytes(32));
            error_log("Generated new payment_token");
        }
        $submitted_token = $_POST['payment_token'] ?? '';
        error_log("Token check: submitted=" . substr($submitted_token, 0, 10) . "... session=" . substr($_SESSION['payment_token'] ?? '', 0, 10) . "...");
        if (empty($submitted_token) || $submitted_token !== $_SESSION['payment_token']) {
            $errors[] = "Token de sécurité invalide. Veuillez réessayer.";
            error_log("Token validation failed!");
        } else {
            error_log("Token validation passed");
        }
        
        $invoice_id = intval($_POST['invoice_id'] ?? 0);
        // Get payment method from POST, but default to 'cash'
        // For loan sales (is_loan_sale=1), payment_method should be 'loan'
        $payment_method = trim($_POST['payment_method'] ?? 'cash');
        $payment_details = trim($_POST['payment_details'] ?? '');
        $bank_account_id = isset($_POST['bank_account_id']) ? (int)$_POST['bank_account_id'] : null;
        $bank_slip_number = trim($_POST['bank_slip_number'] ?? '');
        $bank_slip_public_path = null;
        $bank_account_number_value = null;
        $bank_account_currency = null;
        $submitted_tax_mode = strtoupper(trim($_POST['tax_mode'] ?? ''));
        if (!in_array($submitted_tax_mode, $allowed_tax_modes, true)) {
            $errors[] = "Un mode de taxe valide (HTVA ou TVAC) est requis.";
        } else {
            $selected_tax_mode = $submitted_tax_mode;
        }

        if ($invoice_id <= 0) {
            $errors[] = "Invoice is required.";
        }
        
        // Validate payment method
        $allowed_payment_methods = ['cash', 'loan', 'bank_slip'];
        if (!in_array($payment_method, $allowed_payment_methods, true)) {
            $errors[] = "Méthode de paiement invalide. Choisissez Espèces, Crédit ou Bordereau Bancaire.";
        }
        
        // Check if invoice has already been paid (prevent double payment)
        if ($invoice_id > 0 && empty($errors)) {
            $stmtCheckPaid = $conn->prepare("SELECT id FROM paid_invoices WHERE id = ? LIMIT 1");
            if ($stmtCheckPaid) {
                $stmtCheckPaid->bind_param("i", $invoice_id);
                $stmtCheckPaid->execute();
                $alreadyPaid = $stmtCheckPaid->get_result()->fetch_assoc();
                $stmtCheckPaid->close();
                if ($alreadyPaid) {
                    $errors[] = "Cette commande a déjà été payée. Veuillez rafraîchir la page.";
                }
            }
        }

        $invoice_subtotal = 0;
        $tax_rate_value = 18.00;
        $computed_tax_amount = 0;
        $total_paid = 0;
        $effective_tax_mode = $selected_tax_mode;

        // Verify invoice (including is_loan_sale flag)
        $stmt = $conn->prepare("
            SELECT ui.*, c.id AS customer_id
            FROM unpaid_invoices ui
            LEFT JOIN customer c ON c.full_name = ui.customer_name AND c.tel = ui.phone_number
            WHERE ui.id = ?
              AND ui.province_id = ?
              AND ui.approval_status = 'approved'
        ");
        if (!$stmt) {
            throw new Exception("Query preparation failed for invoice check: " . $conn->error);
        }
        $stmt->bind_param("ii", $invoice_id, $province_id);
        $stmt->execute();
        $invoice = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        if (empty($invoice)) {
            $errors[] = "Commande invalide, déjà payée ou non approuvée sélectionnée.";
        } elseif (!is_numeric($invoice['paid_amount']) || $invoice['paid_amount'] <= 0) {
            $errors[] = "Montant de commande invalide ou manquant.";
            error_log("Invalid paid_amount for invoice_id=$invoice_id: " . var_export($invoice['paid_amount'], true));
        } elseif (!empty($invoice['is_loan_sale']) && $payment_method !== 'loan') {
            // For loan sales, payment_method must be 'loan'
            $errors[] = "Cette commande a été enregistrée comme vente à crédit. Veuillez sélectionner la méthode de paiement 'Crédit'.";
        } elseif ($payment_method === 'loan' && empty($invoice['is_loan_sale'])) {
            // Cannot use 'loan' payment method for non-loan sales
            $errors[] = "La méthode de paiement 'Crédit' n'est disponible que pour les commandes créées comme ventes à crédit.";
        } else {
            /**
             * NOTE:
             * Stock reservation and deduction for wholesale invoices is handled when:
             *  - the invoice is created (reservation in province_stock, in crates)
             *  - the invoice is approved or rejected in modules/admin/approve_invoices.php
             *
             * By the time we reach process_payment.php, the invoice has already passed
             * those checks and stock has been reserved/deducted accordingly.
             *
             * Running an additional blocking stock validation here against wholesale_stock
             * was causing valid payments to fail (e.g. showing 0 crates even though
             * province_stock already handled the stock).
             *
             * Therefore we skip the legacy stock check here and trust the earlier pipeline.
             */
            error_log("process_payment.php: Skipping legacy stock check for invoice_id={$invoice_id}; stock already validated at approval time.");
        }

        if (empty($errors)) {
            $invoice_existing_tax_mode = strtoupper((string)($invoice['tax_mode'] ?? 'HTVA'));
            if (!in_array($effective_tax_mode, $allowed_tax_modes, true)) {
                $effective_tax_mode = in_array($invoice_existing_tax_mode, $allowed_tax_modes, true)
                    ? $invoice_existing_tax_mode
                    : 'HTVA';
            }

            $invoice_subtotal = (int)($invoice['subtotal_amount'] ?? 0);
            if ($invoice_subtotal <= 0) {
                $invoice_subtotal = (int)$invoice['paid_amount'];
            }

            if ($invoice_subtotal <= 0) {
                $errors[] = "Calculated subtotal is invalid for the selected invoice.";
            }

            $tax_rate_value = isset($invoice['tax_rate']) && (float)$invoice['tax_rate'] > 0
                ? (float)$invoice['tax_rate']
                : 18.00;

            if (in_array($effective_tax_mode, $allowed_tax_modes, true) && $effective_tax_mode === 'TVAC') {
                $existing_tax_amount = (int)($invoice['tax_amount'] ?? 0);
                $computed_tax_amount = ($existing_tax_amount > 0 && $invoice_existing_tax_mode === 'TVAC')
                    ? $existing_tax_amount
                    : (int)round($invoice_subtotal * $tax_rate_value / 100);
            } else {
                $computed_tax_amount = 0;
                $effective_tax_mode = 'HTVA';
            }

            $selected_tax_mode = $effective_tax_mode;

            $total_paid = $invoice_subtotal + $computed_tax_amount;
            if ($total_paid <= 0) {
                $errors[] = "Calculated total payment is invalid.";
            }
        }

        $is_bank_slip_payment = ($payment_method === 'bank_slip');

        if ($is_bank_slip_payment) {
            if ($bank_account_id === null || $bank_account_id <= 0) {
                $errors[] = "Veuillez sélectionner le compte bancaire où le paiement a été reçu.";
            } else {
                $stmtBankCheck = $conn->prepare("SELECT account_number, currency FROM bank_account WHERE id = ? AND (province_id IS NULL OR province_id = ?)");
                if ($stmtBankCheck) {
                    $stmtBankCheck->bind_param("ii", $bank_account_id, $province_id);
                    $stmtBankCheck->execute();
                    $bankAccountRow = $stmtBankCheck->get_result()->fetch_assoc();
                    $stmtBankCheck->close();
                    if (empty($bankAccountRow)) {
                        $errors[] = "Compte bancaire invalide ou non autorisé pour votre province.";
                    } else {
                        $bank_account_number_value = $bankAccountRow['account_number'] ?? null;
                        $bank_account_currency = $bankAccountRow['currency'] ?? null;
                    }
                } else {
                    $errors[] = "Impossible de vérifier le compte bancaire sélectionné.";
                }
            }

            if ($bank_slip_number === '') {
                $errors[] = "Le numéro de bordereau bancaire est requis pour un paiement via banque.";
            }

            if (!isset($_FILES['bank_slip_file']) || $_FILES['bank_slip_file']['error'] === UPLOAD_ERR_NO_FILE) {
                $errors[] = "Veuillez télécharger le fichier du bordereau bancaire.";
            } elseif ($_FILES['bank_slip_file']['error'] !== UPLOAD_ERR_OK) {
                $errors[] = "Le fichier du bordereau n'a pas pu être téléchargé (code: " . $_FILES['bank_slip_file']['error'] . ").";
            } else {
                $file = $_FILES['bank_slip_file'];
                $maxSize = 5 * 1024 * 1024; // 5MB
                $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png'];
                $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

                if ($file['size'] > $maxSize) {
                    $errors[] = "Le fichier du bordereau est trop volumineux (max 5MB).";
                }
                if (!in_array($extension, $allowedExtensions, true)) {
                    $errors[] = "Formats autorisés pour le bordereau: PDF, JPG ou PNG.";
                }

                if (empty($errors)) {
                    $uploadDir = dirname(__DIR__, 2) . '/uploads/bank_slips';
                    if (!is_dir($uploadDir) && !mkdir($uploadDir, 0775, true)) {
                        $errors[] = "Impossible de créer le dossier d'upload pour les bordereaux.";
                    } else {
                        $sanitizedName = preg_replace('/[^a-zA-Z0-9_\-]/', '_', pathinfo($file['name'], PATHINFO_FILENAME));
                        $randomSuffix = null;
                        try {
                            $randomSuffix = bin2hex(random_bytes(3));
                        } catch (Exception $e) {
                            $errors[] = "Impossible de préparer le fichier du bordereau (génération aléatoire).";
                        }
                        if (empty($errors)) {
                            // Ensure directory writable
                            if (!is_writable($uploadDir)) {
                                @chmod($uploadDir, 0775);
                            }
                            if (!is_writable($uploadDir)) {
                                $errors[] = "Le dossier d'upload n'est pas accessible en écriture: {$uploadDir}";
                                error_log("Bank slip upload: directory not writable {$uploadDir}");
                            }
                        }
                        if (empty($errors)) {
                            $filename = 'bankslip_' . ($_SESSION['user_id'] ?? 'user') . '_' . time() . '_' . $randomSuffix . '_' . $sanitizedName . '.' . $extension;
                            $fullPath = rtrim($uploadDir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $filename;
                            $moved = move_uploaded_file($file['tmp_name'], $fullPath);
                            if (!$moved) {
                                // Fallback to copy in case move_uploaded_file is blocked by permissions
                                if (is_uploaded_file($file['tmp_name']) && @copy($file['tmp_name'], $fullPath)) {
                                    @unlink($file['tmp_name']);
                                    $moved = true;
                                    error_log("Bank slip upload used copy() fallback to {$fullPath}");
                                }
                            }
                            if (!$moved) {
                                $errors[] = "Le fichier du bordereau n'a pas pu être enregistré sur le serveur.";
                                error_log("Bank slip upload failed: tmp=" . $file['tmp_name'] . " target=" . $fullPath . " err=" . json_encode(error_get_last()));
                            } else {
                                @chmod($fullPath, 0664);
                                $bank_slip_public_path = '/masunzu_bar_hotel/uploads/bank_slips/' . $filename;
                            }
                        }
                    }
                }
            }
        }

        // Check if this is a loan payment
        $is_loan_payment = ($payment_method === 'loan');
        
        // If there are validation errors, store them in session and redirect
        if (!empty($errors)) {
            error_log("Validation errors found: " . print_r($errors, true));
            $_SESSION['cashier_payment_errors'] = $errors;
            header("Location: /masunzu_bar_hotel/modules/cashiers/process_payment.php");
            exit;
        }
        
        error_log("No validation errors, starting transaction for invoice_id=$invoice_id, payment_method=$payment_method");
        if (empty($errors)) {
            try {
                $conn->begin_transaction();
                error_log("Transaction started");
                // Re-verify active shift (required for all payments, but loan payments don't credit balance)
                $stmt = $conn->prepare("SELECT id, deposits, end_balance FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL");
                $stmt->bind_param("i", $_SESSION['user_id']);
                $stmt->execute();
                $shift_check = $stmt->get_result()->fetch_assoc();
                $stmt->close();
                if (!$shift_check) {
                    throw new Exception("No active shift found during payment processing for cashier_id {$_SESSION['user_id']}.");
                }
                if ($shift_check['end_balance'] === null) {
                    $stmt = $conn->prepare("UPDATE cashier_balance SET end_balance = 0.00 WHERE id = ? AND cashier_id = ? AND shift_end IS NULL");
                    $stmt->bind_param("ii", $shift_check['id'], $_SESSION['user_id']);
                    $stmt->execute();
                    $shift_check['end_balance'] = 0.00;
                    $stmt->close();
                    error_log("Fixed NULL end_balance for shift_id={$shift_check['id']}, cashier_id={$_SESSION['user_id']}");
                }
                error_log("Re-verified active shift: shift_id={$shift_check['id']}, deposits={$shift_check['deposits']}, end_balance={$shift_check['end_balance']}");

                // For loan payments (payment_method='loan'), we still create a paid_invoice record
                // to satisfy the foreign key constraint in paid_invoice_items, but the original
                // invoice stays in unpaid_invoices until the loan is repaid
                $paid_invoice_id = null;
                // Create paid_invoice record for both loan and non-loan payments
                if (true) { // Always create paid_invoice record
                    // Move invoice to paid_invoices with TVA breakdown
                    $stockManagerId = (int)($invoice['stock_manager_id'] ?? 0);
                $invoiceCreatedAt = $invoice['created_at'] ?? date('Y-m-d H:i:s');
                $customerName = $invoice['customer_name'] ?? '';
                $phoneNumber = $invoice['phone_number'] ?? '';
                $invoiceProvinceId = (int)($invoice['province_id'] ?? $province_id);
                $invoiceNif = $invoice['nif'] ?? null;
                $invoiceRc = $invoice['rc'] ?? null;
                $invoiceNumber = $invoice['invoice_number'] ?? ('Commande #' . $invoice_id);
                $approvedBy = $invoice['approved_by'] ?? null;
                $approvalStatus = $invoice['approval_status'] ?? 'approved';
                $approvedAt = $invoice['approved_at'] ?? null;
                $invoiceCreatedBy = (int)($invoice['created_by'] ?? $_SESSION['user_id']);
                $sanitizedPaymentDetails = $payment_details !== '' ? $payment_details : null;
                
                // For loan payments, mark in payment_details
                if ($payment_method === 'loan') {
                    $sanitizedPaymentDetails = ($sanitizedPaymentDetails ? $sanitizedPaymentDetails . ' | ' : '') . 'Type Paiement: Crédit';
                } elseif ($is_bank_slip_payment) {
                    $bankParts = [];
                    if ($bank_slip_number !== '') {
                        $bankParts[] = 'Bordereau: ' . $bank_slip_number;
                    }
                    if ($bank_account_number_value) {
                        $bankParts[] = 'Compte: ' . $bank_account_number_value . ($bank_account_currency ? ' ' . $bank_account_currency : '');
                    }
                    if ($bank_slip_public_path) {
                        $bankParts[] = 'Fichier: ' . $bank_slip_public_path;
                    }
                    $bankSummary = implode(' | ', $bankParts);
                    $sanitizedPaymentDetails = ($sanitizedPaymentDetails ? $sanitizedPaymentDetails . ' | ' : '') . 'Type Paiement: Bordereau Bancaire' . ($bankSummary ? ' | ' . $bankSummary : '');
                }
                // For loan payments, store payment_method as 'cash' in DB (enum limitation)
                $payment_method_for_db = ($payment_method === 'loan') ? 'cash' : $payment_method;
                if ($payment_method_for_db === 'bank_slip') {
                    // Map bank slip to existing enum value
                    $payment_method_for_db = 'bank_transfer';
                }

                $approvedByParam = is_numeric($approvedBy) ? (int)$approvedBy : null;
                $invoiceCreatedByParam = (int)$invoiceCreatedBy;
                $provinceIdParam = (int)$invoiceProvinceId;
                $totalPaidParam = (int)$total_paid; // For paid_invoices
                $totalPaidParamDouble = (float)$total_paid; // For loan_payment (bigint)
                $subtotalParam = (int)$invoice_subtotal;
                $taxAmountParam = (int)$computed_tax_amount;
                $taxRateParam = (float)$tax_rate_value;
                $isLoanSaleFlag = !empty($invoice['is_loan_sale']) ? 1 : 0;
                $loanOutstandingValue = (int)($invoice['loan_outstanding_bif'] ?? 0);

                $manualPaidInvoiceId = null;
                if (!$paidInvoicesAutoIncrement) {
                    $manualPaidInvoiceId = get_next_manual_id($conn, 'paid_invoices');
                }

                if ($paidInvoicesAutoIncrement) {
                    $stmt = $conn->prepare("
                    INSERT INTO paid_invoices (
                        stock_manager_id,
                        status,
                        created_at,
                        customer_name,
                        phone_number,
                        province_id,
                        nif,
                        rc,
                        invoice_number,
                        serve_status,
                        paid_amount,
                        subtotal_amount,
                        tax_amount,
                        tax_rate,
                        tax_mode,
                        approved_by,
                        approval_status,
                        approved_at,
                        created_by,
                        paid_at,
                        payment_method,
                        payment_details,
                        is_loan_sale,
                        loan_amount_bif,
                        served_by,
                        sale_type,
                        served_at
                    ) VALUES (
                        ?, 'paid', ?, ?, ?, ?, ?, ?, ?, 'Pending', ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, NULL, 'wholesale', NULL
                        )
                    ");
                    if (!$stmt) {
                        throw new Exception("Failed to prepare paid_invoices insert: " . $conn->error);
                    }
                    $stmt->bind_param(
                        'isssisssiiidsississii',
                        $stockManagerId,
                        $invoiceCreatedAt,
                        $customerName,
                        $phoneNumber,
                        $provinceIdParam,
                        $invoiceNif,
                        $invoiceRc,
                        $invoiceNumber,
                        $totalPaidParam,
                        $subtotalParam,
                        $taxAmountParam,
                        $taxRateParam,
                        $effective_tax_mode,
                        $approvedByParam,
                        $approvalStatus,
                        $approvedAt,
                        $invoiceCreatedByParam,
                        $payment_method_for_db,
                        $sanitizedPaymentDetails,
                        $isLoanSaleFlag,
                        $loanOutstandingValue
                    );
                    if (!$stmt->execute() || $stmt->affected_rows === 0) {
                        throw new Exception("Failed to move invoice to paid_invoices: " . $stmt->error);
                    }
                    $paid_invoice_id = $conn->insert_id;
                    $stmt->close();
                } else {
                    $stmt = $conn->prepare("
                        INSERT INTO paid_invoices (
                            id,
                            stock_manager_id,
                            status,
                            created_at,
                            customer_name,
                            phone_number,
                            province_id,
                            nif,
                            rc,
                            invoice_number,
                            serve_status,
                            paid_amount,
                            subtotal_amount,
                            tax_amount,
                            tax_rate,
                            tax_mode,
                            approved_by,
                            approval_status,
                            approved_at,
                            created_by,
                            paid_at,
                            payment_method,
                            payment_details,
                            is_loan_sale,
                            loan_amount_bif,
                            served_by,
                            sale_type,
                            served_at
                        ) VALUES (
                            ?, ?, 'paid', ?, ?, ?, ?, ?, ?, ?, 'Pending', ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, NULL, 'wholesale', NULL
                        )
                    ");
                    if (!$stmt) {
                        throw new Exception("Failed to prepare manual paid_invoices insert: " . $conn->error);
                    }
                    $stmt->bind_param(
                        'iisssisssiiidsississii',
                        $manualPaidInvoiceId,
                        $stockManagerId,
                        $invoiceCreatedAt,
                        $customerName,
                        $phoneNumber,
                        $provinceIdParam,
                        $invoiceNif,
                        $invoiceRc,
                        $invoiceNumber,
                        $totalPaidParam,
                        $subtotalParam,
                        $taxAmountParam,
                        $taxRateParam,
                        $effective_tax_mode,
                        $approvedByParam,
                        $approvalStatus,
                        $approvedAt,
                        $invoiceCreatedByParam,
                        $payment_method_for_db,
                        $sanitizedPaymentDetails,
                        $isLoanSaleFlag,
                        $loanOutstandingValue
                    );
                    if (!$stmt->execute() || $stmt->affected_rows === 0) {
                        throw new Exception("Failed to insert paid_invoices record with manual ID: " . $stmt->error);
                    }
                    $paid_invoice_id = $manualPaidInvoiceId;
                    $stmt->close();
                }
                
                if ($is_loan_payment) {
                    // For loan payments, invoice stays in unpaid_invoices - the paid_invoice record
                    // is created only to satisfy foreign key constraints for paid_invoice_items
                    error_log("Loan payment - paid_invoice record created (paid_invoice_id=$paid_invoice_id) for foreign key constraint, but original invoice (invoice_id=$invoice_id) stays in unpaid_invoices until repaid via loan_repayments.php");
                } else {
                    error_log("Non-loan payment - invoice moved to paid_invoices (paid_invoice_id=$paid_invoice_id, original invoice_id=$invoice_id)");
                }
                } // End of if (true) - Always create paid_invoice record

                // Copy invoice_items to paid_invoice_items for both loan and non-loan payments
                // Always use paid_invoice_id (which is now created for both loan and non-loan payments)
                if ($paid_invoice_items_exists && $paid_invoice_id) {
                    $target_invoice_id = $paid_invoice_id;
                    // First check if there are any items to copy
                    $checkItemsStmt = $conn->prepare("SELECT COUNT(*) as item_count FROM invoice_items WHERE invoice_id = ?");
                    if (!$checkItemsStmt) {
                        error_log("ERROR: Failed to prepare check for invoice items: " . $conn->error);
                        throw new Exception("Failed to check invoice items: " . $conn->error);
                    }
                    $checkItemsStmt->bind_param("i", $invoice_id);
                    if (!$checkItemsStmt->execute()) {
                        $checkError = $checkItemsStmt->error ?: 'unknown error';
                        $checkItemsStmt->close();
                        error_log("ERROR: Failed to execute check for invoice items: {$checkError}");
                        throw new Exception("Failed to check invoice items: {$checkError}");
                    }
                    $itemCountResult = $checkItemsStmt->get_result()->fetch_assoc();
                    $itemCount = (int)($itemCountResult['item_count'] ?? 0);
                    $checkItemsStmt->close();
                    
                    error_log("Found {$itemCount} items in invoice_items for invoice_id={$invoice_id}, will copy to paid_invoice_items with paid_invoice_id={$paid_invoice_id}");
                    
                    if ($itemCount === 0) {
                        error_log("Warning: No invoice items found for invoice_id={$invoice_id}. Skipping item copy to paid_invoice_items.");
                    } else {
                        if ($paidInvoiceItemsAutoIncrement) {
                                $stmt = $conn->prepare("
                                    INSERT INTO paid_invoice_items (
                                        invoice_id, product_id, quantity, unit_price, price_source,
                                        system_unit_price, custom_unit_price, custom_price_basis,
                                        custom_price_input, custom_price_note
                                    )
                                    SELECT ?, product_id, quantity, unit_price, price_source,
                                           system_unit_price, custom_unit_price, custom_price_basis,
                                           custom_price_input, custom_price_note
                                    FROM invoice_items 
                                    WHERE invoice_id = ?
                                ");
                                if (!$stmt) {
                                    throw new Exception("Failed to prepare INSERT for paid_invoice_items: " . $conn->error);
                                }
                                $stmt->bind_param("ii", $target_invoice_id, $invoice_id);
                                if (!$stmt->execute()) {
                                    $errorMsg = $stmt->error ?: 'unknown error';
                                    $stmt->close();
                                    throw new Exception("Failed to copy invoice items to paid_invoice_items: {$errorMsg} (invoice_id={$invoice_id}, target_invoice_id={$target_invoice_id}, items_count={$itemCount})");
                                }
                                $affectedRows = $stmt->affected_rows;
                                $stmt->close();
                                
                                if ($affectedRows === 0) {
                                    error_log("Warning: No rows inserted into paid_invoice_items for invoice_id={$invoice_id} (expected {$itemCount} items). This may be normal if items were already copied.");
                                } else {
                                    error_log("Successfully copied {$affectedRows} items from invoice_items to paid_invoice_items (invoice_id={$invoice_id}, target_invoice_id={$target_invoice_id})");
                                }
                        } else {
                            // Manual copy when AUTO_INCREMENT is unavailable
                                $stmtItems = $conn->prepare("
                                    SELECT product_id, quantity, unit_price, price_source,
                                           system_unit_price, custom_unit_price, custom_price_basis,
                                           custom_price_input, custom_price_note
                                    FROM invoice_items
                                    WHERE invoice_id = ?
                                ");
                                if (!$stmtItems) {
                                    throw new Exception("Failed to prepare invoice_items fetch for manual copy: " . $conn->error);
                                }
                                $stmtItems->bind_param("i", $invoice_id);
                                $stmtItems->execute();
                                $itemsResult = $stmtItems->get_result();
                                $invoiceItemsData = $itemsResult ? $itemsResult->fetch_all(MYSQLI_ASSOC) : [];
                                $stmtItems->close();

                                if (empty($invoiceItemsData)) {
                                    throw new Exception("No invoice items found to copy for invoice_id {$invoice_id}.");
                                }

                                $nextPaidInvoiceItemId = get_next_manual_id($conn, 'paid_invoice_items');
                                $insertManualItemStmt = $conn->prepare("
                                    INSERT INTO paid_invoice_items (
                                        id,
                                        invoice_id,
                                        product_id,
                                        quantity,
                                        unit_price,
                                        price_source,
                                        system_unit_price,
                                        custom_unit_price,
                                        custom_price_basis,
                                        custom_price_input,
                                        custom_price_note
                                    ) VALUES (
                                        ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?
                                    )
                                ");
                                if (!$insertManualItemStmt) {
                                    throw new Exception("Failed to prepare manual paid_invoice_items insert statement: " . $conn->error);
                                }
                                $manualItemId = 0;
                                $manualPaidInvoiceIdParam = $target_invoice_id;
                                $manualProductId = 0;
                                $manualQuantity = 0;
                                $manualUnitPrice = 0;
                                $manualPriceSource = null;
                                $manualSystemUnitPrice = null;
                                $manualCustomUnitPrice = null;
                                $manualCustomPriceBasis = null;
                                $manualCustomPriceInput = null;
                                $manualCustomPriceNote = null;
                                $insertManualItemStmt->bind_param(
                                    "iiiiisiisis",
                                    $manualItemId,
                                    $manualPaidInvoiceIdParam,
                                    $manualProductId,
                                    $manualQuantity,
                                    $manualUnitPrice,
                                    $manualPriceSource,
                                    $manualSystemUnitPrice,
                                    $manualCustomUnitPrice,
                                    $manualCustomPriceBasis,
                                    $manualCustomPriceInput,
                                    $manualCustomPriceNote
                                );

                                foreach ($invoiceItemsData as $itemRow) {
                                    $manualItemId = $nextPaidInvoiceItemId++;
                                    $manualProductId = (int)($itemRow['product_id'] ?? 0);
                                    $manualQuantity = (int)($itemRow['quantity'] ?? 0);
                                    $manualUnitPrice = (int)($itemRow['unit_price'] ?? 0);
                                    $manualPriceSource = $itemRow['price_source'] ?? 'system';
                                    $manualSystemUnitPrice = isset($itemRow['system_unit_price']) ? (int)$itemRow['system_unit_price'] : null;
                                    $manualCustomUnitPrice = isset($itemRow['custom_unit_price']) ? (int)$itemRow['custom_unit_price'] : null;
                                    $manualCustomPriceBasis = $itemRow['custom_price_basis'] ?? null;
                                    $manualCustomPriceInput = isset($itemRow['custom_price_input']) ? (int)$itemRow['custom_price_input'] : null;
                                    $manualCustomPriceNote = $itemRow['custom_price_note'] ?? null;

                                    if (!$insertManualItemStmt->execute()) {
                                        throw new Exception("Failed to insert paid_invoice_items row with manual ID {$manualItemId}: " . $insertManualItemStmt->error);
                                    }
                                }
                                $insertManualItemStmt->close();
                        }
                    }

                    // Delete from invoice_items only if invoice was moved to paid_invoices (non-loan payments)
                    if (!$is_loan_payment) {
                        $stmt = $conn->prepare("DELETE FROM invoice_items WHERE invoice_id = ?");
                        $stmt->bind_param("i", $invoice_id);
                        if (!$stmt->execute()) {
                            throw new Exception("Failed to delete from invoice_items: " . $stmt->error);
                        }
                        $stmt->close();
                    } else {
                        // For loan payments, invoice_items stay with unpaid_invoice (items are copied but not deleted)
                        error_log("Loan payment - invoice_items NOT deleted (invoice_id=$invoice_id). Items stay with unpaid_invoice but are also copied to paid_invoice_items for receipt generation.");
                    }
                } else {
                    throw new Exception("paid_invoice_items table does not exist. Cannot process payment.");
                }

                // Record bank ledger for bank slip payments (money goes straight to bank account)
                $bank_deposit_id = null;
                if ($is_bank_slip_payment) {
                    $stmtDeposit = $conn->prepare("
                        INSERT INTO bank_deposit (user_id, amount, slip_number, slip_file_path, deposit_date, province_id, bank_account_id)
                        VALUES (?, ?, ?, ?, NOW(), ?, ?)
                    ");
                    if (!$stmtDeposit) {
                        throw new Exception("Failed to prepare bank_deposit insert: " . $conn->error);
                    }
                    $stmtDeposit->bind_param(
                        "idssii",
                        $_SESSION['user_id'],
                        $total_paid,
                        $bank_slip_number,
                        $bank_slip_public_path,
                        $invoiceProvinceId,
                        $bank_account_id
                    );
                    if (!$stmtDeposit->execute()) {
                        $err = $stmtDeposit->error ?: 'unknown error';
                        $stmtDeposit->close();
                        throw new Exception("Échec de l'enregistrement du bordereau bancaire: {$err}");
                    }
                    $bank_deposit_id = $conn->insert_id;
                    $stmtDeposit->close();

                    $stmtBankUpdate = $conn->prepare("UPDATE bank_account SET balance = balance + ? WHERE id = ?");
                    if ($stmtBankUpdate) {
                        $stmtBankUpdate->bind_param("di", $total_paid, $bank_account_id);
                        if (!$stmtBankUpdate->execute() || $stmtBankUpdate->affected_rows === 0) {
                            $err = $stmtBankUpdate->error ?: 'unknown error';
                            $stmtBankUpdate->close();
                            throw new Exception("Impossible de mettre à jour le solde du compte bancaire: {$err}");
                        }
                        $stmtBankUpdate->close();
                    } else {
                        throw new Exception("Failed to préparer la mise à jour du compte bancaire: " . $conn->error);
                    }

                    $stmtBankLedger = $conn->prepare("
                        INSERT INTO bank_account_transaction (bank_account_id, province_id, transaction_type, direction, amount, reference_type, reference_id, description, created_by)
                        VALUES (?, ?, 'BANK_DEPOSIT', 'CREDIT', ?, 'CASH_DEPOSIT', ?, ?, ?)
                    ");
                    if ($stmtBankLedger) {
                        $ledgerDescription = sprintf('Paiement facture %s via bordereau %s', $invoiceNumber, $bank_slip_number);
                        $stmtBankLedger->bind_param(
                            "iidisi",
                            $bank_account_id,
                            $invoiceProvinceId,
                            $total_paid,
                            $bank_deposit_id,
                            $ledgerDescription,
                            $_SESSION['user_id']
                        );
                        if (!$stmtBankLedger->execute()) {
                            $err = $stmtBankLedger->error ?: 'unknown error';
                            $stmtBankLedger->close();
                            throw new Exception("Impossible d'enregistrer la transaction bancaire: {$err}");
                        }
                        $stmtBankLedger->close();
                    }
                }

                // Update cashier_balance for NON-LOAN payments only
                // Loan payments should NOT credit cashier balance (only loan_repayments.php credits cashier)
                if (!$is_loan_payment && !$is_bank_slip_payment) {
                    // Ensure cashier_account table exists
                    $createCashierAccountSql = "
                        CREATE TABLE IF NOT EXISTS cashier_account (
                            id INT AUTO_INCREMENT PRIMARY KEY,
                            cashier_id INT NOT NULL,
                            shift_id INT NULL,
                            transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                            direction ENUM('CREDIT','DEBIT') NOT NULL,
                            amount BIGINT(20) NOT NULL,
                            reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                            reference_id INT NULL,
                            description VARCHAR(255) DEFAULT NULL,
                            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                            created_by INT NULL,
                            INDEX idx_cashier_created (cashier_id, created_at),
                            INDEX idx_shift (shift_id),
                            INDEX idx_reference (reference_type, reference_id),
                            CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                            CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                            CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                    ";
                    @$conn->query($createCashierAccountSql);
                    
                    // Get shift_id first
                    $stmtShift = $conn->prepare("SELECT id FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL LIMIT 1");
                    $stmtShift->bind_param("i", $_SESSION['user_id']);
                    $stmtShift->execute();
                    $shiftResult = $stmtShift->get_result();
                    $shiftRow = $shiftResult ? $shiftResult->fetch_assoc() : null;
                    $shift_id = $shiftRow ? (int)$shiftRow['id'] : null;
                    $stmtShift->close();
                    
                    if (!$shift_id) {
                        throw new Exception("No active shift found for cashier_id {$_SESSION['user_id']}.");
                    }
                    
                    $stmt = $conn->prepare("
                        UPDATE cashier_balance
                        SET deposits = deposits + ?, end_balance = end_balance + ?
                        WHERE cashier_id = ? AND shift_end IS NULL
                    ");
                    $stmt->bind_param("ddi", $total_paid, $total_paid, $_SESSION['user_id']);
                    if (!$stmt->execute()) {
                        throw new Exception("Failed to execute cashier_balance update: " . $stmt->error);
                    }
                    if ($stmt->affected_rows === 0) {
                        throw new Exception("No rows updated in cashier_balance for cashier_id {$_SESSION['user_id']}. Shift may have ended or no matching record found.");
                    }
                    error_log("Updated cashier_balance for cashier_id {$_SESSION['user_id']}: added $total_paid to deposits and end_balance");
                    $stmt->close();
                    
                    // Enregistrer dans cashier_account
                    $invoiceNumber = $invoice['invoice_number'] ?? ('Invoice #' . $invoice_id);
                    $description = "Paiement facture: {$invoiceNumber}";
                    $totalPaidForAccount = (int)$total_paid; // Convert to int for BIGINT column
                    $stmtAccount = $conn->prepare("
                        INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                        VALUES (?, ?, 'INVOICE_PAYMENT', 'CREDIT', ?, 'INVOICE', ?, ?, ?)
                    ");
                    if ($stmtAccount) {
                        $stmtAccount->bind_param("iiiiss", $_SESSION['user_id'], $shift_id, $totalPaidForAccount, $invoice_id, $_SESSION['user_id'], $description);
                        if (!$stmtAccount->execute()) {
                            error_log("Failed to record INVOICE_PAYMENT in cashier_account: " . $stmtAccount->error);
                        }
                        $stmtAccount->close();
                    }
                } elseif ($is_bank_slip_payment) {
                    // Ensure cashier_account table exists to keep a trace for the cashier
                    $createCashierAccountSql = "
                        CREATE TABLE IF NOT EXISTS cashier_account (
                            id INT AUTO_INCREMENT PRIMARY KEY,
                            cashier_id INT NOT NULL,
                            shift_id INT NULL,
                            transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                            direction ENUM('CREDIT','DEBIT') NOT NULL,
                            amount BIGINT(20) NOT NULL,
                            reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                            reference_id INT NULL,
                            description VARCHAR(255) DEFAULT NULL,
                            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                            created_by INT NULL,
                            INDEX idx_cashier_created (cashier_id, created_at),
                            INDEX idx_shift (shift_id),
                            INDEX idx_reference (reference_type, reference_id),
                            CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                            CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                            CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                    ";
                    @$conn->query($createCashierAccountSql);

                    $stmtShift = $conn->prepare("SELECT id FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL LIMIT 1");
                    $shift_id_bank = null;
                    if ($stmtShift) {
                        $stmtShift->bind_param("i", $_SESSION['user_id']);
                        $stmtShift->execute();
                        $shiftResult = $stmtShift->get_result();
                        $shiftRow = $shiftResult ? $shiftResult->fetch_assoc() : null;
                        $shift_id_bank = $shiftRow ? (int)$shiftRow['id'] : null;
                        $stmtShift->close();
                    }

                    $invoiceNumber = $invoice['invoice_number'] ?? ('Invoice #' . $invoice_id);
                    $description = "Paiement bordereau bancaire: {$invoiceNumber}";
                    $totalPaidForAccount = (int)$total_paid;
                    $stmtAccountBank = $conn->prepare("
                        INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                        VALUES (?, ?, 'INVOICE_PAYMENT', 'CREDIT', ?, 'INVOICE', ?, ?, ?)
                    ");
                    if ($stmtAccountBank) {
                        $stmtAccountBank->bind_param("iiiiss", $_SESSION['user_id'], $shift_id_bank, $totalPaidForAccount, $invoice_id, $_SESSION['user_id'], $description);
                        if (!$stmtAccountBank->execute()) {
                            error_log("Failed to record BANK SLIP payment in cashier_account: " . $stmtAccountBank->error);
                        }
                        $stmtAccountBank->close();
                    }
                } else {
                    // For loan payments, record in cashier_account but with a note that no cash was received
                    // This is for tracking purposes only - no cash balance is credited
                    $invoiceNumber = $invoice['invoice_number'] ?? ('Invoice #' . $invoice_id);
                    $description = "Vente à crédit (facture: {$invoiceNumber}) - Aucun argent reçu";
                    $totalPaidForAccount = (int)$total_paid;
                    
                    // Get shift_id for loan payments too (for consistency)
                    $stmtShift = $conn->prepare("SELECT id FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL LIMIT 1");
                    $shift_id_loan = null;
                    if ($stmtShift) {
                        $stmtShift->bind_param("i", $_SESSION['user_id']);
                        $stmtShift->execute();
                        $shiftResult = $stmtShift->get_result();
                        $shiftRow = $shiftResult ? $shiftResult->fetch_assoc() : null;
                        $shift_id_loan = $shiftRow ? (int)$shiftRow['id'] : null;
                        $stmtShift->close();
                    }
                    
                    // Record loan sale in cashier_account with 0 amount (or use a special marker)
                    // We'll use amount=0 to indicate no cash was received, but still track the transaction
                    $stmtAccountLoan = $conn->prepare("
                        INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                        VALUES (?, ?, 'INVOICE_PAYMENT', 'CREDIT', 0, 'INVOICE', ?, ?, ?)
                    ");
                    if ($stmtAccountLoan) {
                        $stmtAccountLoan->bind_param("iiiis", $_SESSION['user_id'], $shift_id_loan, $invoice_id, $_SESSION['user_id'], $description);
                        if (!$stmtAccountLoan->execute()) {
                            error_log("Failed to record LOAN_SALE in cashier_account: " . $stmtAccountLoan->error);
                        } else {
                            error_log("Recorded loan sale in cashier_account (amount=0, no cash received) for invoice_id=$invoice_id");
                        }
                        $stmtAccountLoan->close();
                    }
                    error_log("Loan payment processed - cashier balance NOT credited (amount: $total_paid, invoice_id: $invoice_id), but transaction recorded in cashier_account");
                }
                
                if ($is_loan_payment) {
                    // When payment_method='loan', the customer is NOT paying immediately
                    // The debt remains recorded in customer_loan_ledger (already created when invoice was created)
                    // We just need to ensure the customer_id is set and the invoice is properly linked
                    $customer_id = (int)($invoice['customer_id'] ?? 0);
                    $customer_name_for_lookup = trim($invoice['customer_name'] ?? '');
                    $customer_phone_for_lookup = trim($invoice['phone_number'] ?? '');
                    if ($customer_id <= 0) {
                        $fallbackCustomerId = find_customer_by_name_and_phone($conn, $customer_name_for_lookup, $customer_phone_for_lookup);
                        if ($fallbackCustomerId !== null) {
                            $customer_id = $fallbackCustomerId;
                        }
                    }
                    if ($customer_id > 0) {
                        // Verify that a loan_sale entry exists in the ledger for this invoice
                        // If not, create it (should already exist from invoice creation, but ensure it's there)
                        $stmtCheckLedger = $conn->prepare("
                            SELECT id FROM customer_loan_ledger 
                            WHERE customer_id = ? 
                            AND entry_type = 'loan_sale' 
                            AND direction = 'debit'
                            AND reference_table = 'unpaid_invoices' 
                            AND reference_id = ?
                            LIMIT 1
                        ");
                        if ($stmtCheckLedger) {
                            $stmtCheckLedger->bind_param("ii", $customer_id, $invoice_id);
                            $stmtCheckLedger->execute();
                            $existingLedger = $stmtCheckLedger->get_result()->fetch_assoc();
                            $stmtCheckLedger->close();
                            
                            if (!$existingLedger) {
                                // Create the loan_sale entry if it doesn't exist (shouldn't happen, but safety check)
                                $ledgerNotes = 'Commande de vente à crédit traitée';
                                $stmtLedger = $conn->prepare("
                                    INSERT INTO customer_loan_ledger (
                                        customer_id, entry_type, direction, entry_origin, amount_bif, 
                                        reference_table, reference_id, notes, created_by
                                    ) VALUES (?, 'loan_sale', 'debit', 'process_payment', ?, 'unpaid_invoices', ?, ?, ?)
                                ");
                                if ($stmtLedger) {
                                    $stmtLedger->bind_param("isisi", $customer_id, $total_paid, $invoice_id, $ledgerNotes, $_SESSION['user_id']);
                                    if ($stmtLedger->execute()) {
                                        error_log("Created missing loan_sale ledger entry for invoice_id=$invoice_id, customer_id=$customer_id, amount=$total_paid");
                                    }
                                    $stmtLedger->close();
                                }
                            }
                        }
                        
                        // DO NOT decrement loan_balance_bif - the debt remains until paid via loan_repayments.php
                        // The loan_balance_bif was already incremented when the invoice was created
                        error_log("Loan payment processed - debt remains recorded in ledger (customer_id=$customer_id, amount=$total_paid, invoice_id=$invoice_id). Customer will pay later via loan_repayments.php");
                    }
                }
                if (!empty($invoice['is_loan_sale'])) {
                    // For loan sales processed with payment_method='loan', the customer hasn't paid yet
                    // The debt remains recorded in customer_loan_ledger and the invoice stays in unpaid_invoices
                    if ($is_loan_payment) {
                        // Customer hasn't paid - debt remains, invoice stays in unpaid_invoices
                        // Mark invoice as 'processed' so it doesn't appear in the payment list anymore
                        // The invoice will remain in unpaid_invoices until paid via loan_repayments.php
                        $stmtUpdateLoanInvoice = $conn->prepare("UPDATE unpaid_invoices SET status = 'processed' WHERE id = ?");
                        if ($stmtUpdateLoanInvoice) {
                            $stmtUpdateLoanInvoice->bind_param("i", $invoice_id);
                            if ($stmtUpdateLoanInvoice->execute()) {
                                error_log("Loan sale invoice marked as 'processed' (invoice_id=$invoice_id). Invoice stays in unpaid_invoices but won't appear in payment list.");
                            } else {
                                error_log("Failed to update invoice status to 'processed': " . $stmtUpdateLoanInvoice->error);
                            }
                            $stmtUpdateLoanInvoice->close();
                        }
                        error_log("Loan sale invoice processed with payment_method='loan' - debt remains, invoice stays in unpaid_invoices (invoice_id=$invoice_id)");
                    } else {
                        // Customer paid with cash/bank/mobile - reduce outstanding and delete from unpaid_invoices
                        $newInvoiceOutstanding = max(0, (int)($invoice['loan_outstanding_bif'] ?? 0) - $total_paid);
                        $newLoanStatus = $newInvoiceOutstanding === 0 ? 'settled' : 'pending';
                        $invoiceStatus = $newInvoiceOutstanding === 0 ? 'paid' : 'pending';
                        $stmtUpdateInvoice = $conn->prepare("UPDATE unpaid_invoices SET loan_outstanding_bif = ?, loan_status = ?, status = ? WHERE id = ?");
                        if (!$stmtUpdateInvoice) {
                            throw new Exception("Failed to prepare invoice update: " . $conn->error);
                        }
                        $stmtUpdateInvoice->bind_param("issi", $newInvoiceOutstanding, $newLoanStatus, $invoiceStatus, $invoice_id);
                        if (!$stmtUpdateInvoice->execute()) {
                            throw new Exception("Failed to update invoice: " . $stmtUpdateInvoice->error);
                        }
                        $stmtUpdateInvoice->close();
                    }
                }

                // Delete from unpaid_invoices only if NOT a loan payment
                // For loan payments (payment_method='loan'), the invoice stays in unpaid_invoices until paid via loan_repayments.php
                if (!$is_loan_payment) {
                    $stmt = $conn->prepare("DELETE FROM unpaid_invoices WHERE id = ? AND province_id = ?");
                    if (!$stmt) {
                        throw new Exception("Failed to prepare DELETE statement: " . $conn->error);
                    }
                    $stmt->bind_param("ii", $invoice_id, $invoiceProvinceId);
                    if (!$stmt->execute()) {
                        throw new Exception("Failed to delete from unpaid_invoices: " . $stmt->error);
                    }
                    $deletedRows = $stmt->affected_rows;
                    $stmt->close();
                    if ($deletedRows === 0) {
                        // Invoice may have already been deleted - this is a critical error
                        error_log("CRITICAL: DELETE from unpaid_invoices affected 0 rows for invoice_id=$invoice_id, province_id=$invoiceProvinceId. Invoice may have already been deleted or province mismatch.");
                        throw new Exception("La commande n'a pas pu être supprimée de la liste des commandes impayées. Elle a peut-être déjà été traitée. Veuillez rafraîchir la page.");
                    } else {
                        error_log("Successfully deleted invoice_id=$invoice_id from unpaid_invoices (affected_rows=$deletedRows)");
                    }
                } else {
                    error_log("Loan payment - invoice NOT deleted from unpaid_invoices (invoice_id=$invoice_id). Customer will pay later via loan_repayments.php");
                }

                // Define variables needed for success message and logging (for both loan and non-loan payments)
                $invoiceNumber = $invoice['invoice_number'] ?? ('Commande #' . $invoice_id);
                $customerName = trim($invoice['customer_name'] ?? '');
                $formattedTotal = number_format((float)$total_paid, 0, '', '.');
                $formattedSubtotal = number_format((float)$invoice_subtotal, 0, '', '.');
                $formattedTax = number_format((float)$computed_tax_amount, 0, '', '.');
                $modeLabel = $effective_tax_mode === 'TVAC' ? 'TVAC (avec 18% TVA)' : 'HTVA';
                
                // Create success message and log message based on payment type
                if (!$is_loan_payment) {
                    // Non-loan payment success message
                    $methodLabel = $payment_method === 'bank_slip'
                        ? 'Bordereau Bancaire'
                        : ucfirst(str_replace('_', ' ', $payment_method));
                    $amountSummary = $effective_tax_mode === 'TVAC'
                        ? sprintf(' (Sous-total %s BIF + TVA %s BIF = %s BIF)', $formattedSubtotal, $formattedTax, $formattedTotal)
                        : sprintf(' (Total %s BIF)', $formattedTotal);
                    if ($is_bank_slip_payment) {
                        $baseSuccessMessage = sprintf(
                            'Paiement pour %s enregistré via bordereau bancaire%s%s.',
                            $invoiceNumber,
                            $amountSummary,
                            $customerName !== '' ? " pour le client {$customerName}" : ''
                        );
                        $logMessage = sprintf(
                            'Processed bank slip payment for invoice %s (%s) total %s BIF on account %s',
                            $invoiceNumber,
                            $effective_tax_mode,
                            $formattedTotal,
                            $bank_account_number_value ?: 'n/a'
                        );
                    } else {
                        $baseSuccessMessage = sprintf(
                            'Paiement pour %s traité en mode %s%s via %s%s.',
                            $invoiceNumber,
                            $modeLabel,
                            $amountSummary,
                            $methodLabel,
                            $customerName !== '' ? " pour le client {$customerName}" : ''
                        );
                        $logMessage = sprintf(
                            'Processed payment for invoice %s (%s) with subtotal %s BIF, TVA %s BIF, total %s BIF',
                            $invoiceNumber,
                            $effective_tax_mode,
                            $formattedSubtotal,
                            $formattedTax,
                            $formattedTotal
                        );
                    }
                } else {
                    // Loan payment success message (will be overridden later, but define it here for consistency)
                    $baseSuccessMessage = sprintf(
                        'Commande de vente à crédit %s enregistrée. Le montant de %s BIF a été enregistré comme dette que le client devra payer plus tard via la page de remboursement de crédit.',
                        $invoiceNumber,
                        $formattedTotal
                    );
                    $logMessage = sprintf(
                        'Processed loan payment for invoice %s - debt recorded: %s BIF',
                        $invoiceNumber,
                        $formattedTotal
                    );
                }

                // Generate receipt for both loan and non-loan payments
                $receipt_id = null;
                if ($is_loan_payment) {
                    // For loan sales, generate receipt using paid_invoice_id (same as non-loan payments)
                    // This ensures the receipt can be found in "Paiements d'Aujourd'hui"
                    $receipt_id = generate_uuid();
                    $receipt_number = $invoiceNumber; // Use invoice number as receipt number
                    
                    $stmtReceipt = $conn->prepare("
                        INSERT INTO receipts (
                            receipt_id, transaction_type, transaction_id, receipt_number,
                            customer_name, phone_number, total_amount, subtotal_amount, tax_amount, tax_rate,
                            tax_mode, created_at, created_by
                        ) VALUES (?, 'invoice', ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)
                    ");
                    if ($stmtReceipt) {
                        // Use paid_invoice_id so the receipt can be found in "Paiements d'Aujourd'hui"
                        $stmtReceipt->bind_param(
                            "sisssiiidsi",
                            $receipt_id,
                            $paid_invoice_id,
                            $receipt_number,
                            $customerName,
                            $phoneNumber,
                            $totalPaidParam,
                            $subtotalParam,
                            $taxAmountParam,
                            $taxRateParam,
                            $effective_tax_mode,
                            $_SESSION['user_id']
                        );
                        if (!$stmtReceipt->execute()) {
                            error_log("Failed to insert receipt for loan sale: " . $stmtReceipt->error);
                            $receipt_id = null;
                        } else {
                            error_log("Receipt created successfully for loan sale: receipt_id=$receipt_id for paid_invoice_id=$paid_invoice_id");
                        }
                        $stmtReceipt->close();
                    } else {
                        error_log("Failed to prepare receipt insert statement for loan sale: " . $conn->error);
                    }
                } elseif ($paid_invoice_id) {
                    // For non-loan payments, generate receipt using paid_invoice_id
                    $receipt_id = generate_uuid();
                    $receipt_number = $invoiceNumber; // Use invoice number as receipt number
                    
                    $stmtReceipt = $conn->prepare("
                        INSERT INTO receipts (
                            receipt_id, transaction_type, transaction_id, receipt_number,
                            customer_name, phone_number, total_amount, subtotal_amount, tax_amount, tax_rate,
                            tax_mode, created_at, created_by
                        ) VALUES (?, 'invoice', ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)
                    ");
                    if ($stmtReceipt) {
                        $stmtReceipt->bind_param(
                            "sisssiiidsi",
                            $receipt_id,
                            $paid_invoice_id,
                            $receipt_number,
                            $customerName,
                            $phoneNumber,
                            $totalPaidParam,
                            $subtotalParam,
                            $taxAmountParam,
                            $taxRateParam,
                            $effective_tax_mode,
                            $_SESSION['user_id']
                        );
                        if (!$stmtReceipt->execute()) {
                            error_log("Failed to insert receipt: " . $stmtReceipt->error);
                            $receipt_id = null;
                        } else {
                            error_log("Receipt created successfully: receipt_id=$receipt_id for paid_invoice_id=$paid_invoice_id");
                        }
                        $stmtReceipt->close();
                    } else {
                        error_log("Failed to prepare receipt insert statement: " . $conn->error);
                    }
                } else {
                    error_log("WARNING: Paid invoice not created (paid_invoice_id is null) for non-loan payment invoice_id=$invoice_id");
                }
                
                // Commit transaction - verify it succeeds
                error_log("About to commit transaction for invoice_id=$invoice_id, is_loan_payment=" . ($is_loan_payment ? 'true' : 'false'));
                if (!$conn->commit()) {
                    error_log("CRITICAL: Commit failed for invoice_id=$invoice_id: " . $conn->error);
                    throw new Exception("Failed to commit transaction: " . $conn->error);
                }
                
                error_log("Payment transaction committed successfully for invoice_id=$invoice_id, paid_invoice_id=" . ($paid_invoice_id ?? 'null') . ", receipt_id=" . ($receipt_id ?? 'null'));
                
                // Log the action (logMessage is defined earlier in the code for both loan and non-loan payments)
                if (isset($logMessage)) {
                    log_action($_SESSION['user_id'], 'process_payment', $logMessage);
                } else {
                    error_log("WARNING: logMessage not defined for invoice_id=$invoice_id");
                }
                
                // Ensure baseSuccessMessage is defined (should already be defined above for both loan and non-loan payments)
                if (!isset($baseSuccessMessage)) {
                    error_log("CRITICAL: baseSuccessMessage not defined for invoice_id=$invoice_id, is_loan_payment=" . ($is_loan_payment ? 'true' : 'false'));
                    $baseSuccessMessage = "Paiement traité avec succès pour la commande #$invoice_id";
                }
                
                // Invalidate the payment token to prevent double submission
                unset($_SESSION['payment_token']);
                
                error_log("Setting success message in session: " . substr($baseSuccessMessage, 0, 100));
                $_SESSION['cashier_payment_success'] = $baseSuccessMessage;
                $_SESSION['cashier_payment_receipt_id'] = $receipt_id;
                $_SESSION['cashier_payment_is_loan'] = $is_loan_payment;
                
                // Force session write to ensure data is saved
                if (session_status() === PHP_SESSION_ACTIVE) {
                    session_write_close();
                    session_start();
                }

                error_log("Redirecting after successful payment: is_loan_payment=" . ($is_loan_payment ? 'true' : 'false') . ", receipt_id=" . ($receipt_id ?? 'null') . ", paid_invoice_id=" . ($paid_invoice_id ?? 'null'));
                
                // Redirect based on payment type - Always trigger auto-print if receipt exists
                if ($receipt_id) {
                    // All payments with receipt - trigger auto-print (including loan payments)
                    $_SESSION['cashier_payment_auto_print'] = true;
                    error_log("Redirecting to print_receipt=1 for receipt_id=$receipt_id (is_loan_payment=" . ($is_loan_payment ? 'true' : 'false') . ")");
                    header("Location: /masunzu_bar_hotel/modules/cashiers/process_payment.php?print_receipt=1");
                    exit;
                } else {
                    // Payment without receipt - normal redirect
                    error_log("Redirecting to normal page (no receipt generated)");
                    header("Location: /masunzu_bar_hotel/modules/cashiers/process_payment.php");
                    exit;
                }
            } catch (Exception $e) {
                if ($conn->in_transaction) {
                    $conn->rollback();
                }
                $errorMessage = "Échec du traitement du paiement: " . $e->getMessage();
                $errors[] = $errorMessage;
                error_log("Payment processing error for invoice_id=$invoice_id: " . $e->getMessage());
                error_log("Stack trace: " . $e->getTraceAsString());
                // Store errors in session so they persist after redirect
                $_SESSION['cashier_payment_errors'] = $errors;
                // Redirect to show errors
                header("Location: /masunzu_bar_hotel/modules/cashiers/process_payment.php");
                exit;
            } catch (mysqli_sql_exception $e) {
                if ($conn->in_transaction) {
                    $conn->rollback();
                }
                $errorMessage = "Erreur de base de données lors du traitement du paiement: " . $e->getMessage();
                $errors[] = $errorMessage;
                error_log("Database error during payment processing for invoice_id=$invoice_id: " . $e->getMessage());
                error_log("SQL Error Code: " . $e->getCode());
                // Store errors in session so they persist after redirect
                $_SESSION['cashier_payment_errors'] = $errors;
                // Redirect to show errors
                header("Location: /masunzu_bar_hotel/modules/cashiers/process_payment.php");
                exit;
            } // End of try-catch block
        } // End of if (empty($errors))
    } // End of if ($_SERVER['REQUEST_METHOD'] === 'POST')

    // Fetch today's payments for this cashier
    $today_payments = [];
    $conn2 = get_db_connection();
    if ($conn2) {
        // Get payments where receipts were created by this cashier today
        // This includes both loan and non-loan payments (both create paid_invoice records)
        $stmtToday = $conn2->prepare("
            SELECT 
                pi.id,
                pi.invoice_number,
                pi.customer_name,
                pi.phone_number,
                pi.paid_amount,
                pi.subtotal_amount,
                pi.tax_amount,
                pi.tax_rate,
                pi.tax_mode,
                pi.payment_method,
                pi.payment_details,
                pi.paid_at,
                pi.nif,
                pi.rc,
                pi.is_loan_sale,
                r.receipt_id,
                r.created_by AS receipt_created_by
            FROM receipts r
            INNER JOIN paid_invoices pi ON r.transaction_type = 'invoice' AND r.transaction_id = pi.id
            WHERE r.created_by = ? AND DATE(r.created_at) = CURDATE()
            ORDER BY r.created_at DESC
        ");
        if ($stmtToday) {
            $stmtToday->bind_param("i", $_SESSION['user_id']);
            $stmtToday->execute();
            $today_payments = $stmtToday->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtToday->close();
        }
        $conn2->close();
    }

    $conn->close();
    include __DIR__ . '/../../includes/header.php';
?>
    <style>
        html, body {
            background-color: #F5F5F5 !important;
            min-height: 100vh;
            margin: 0;
            padding: 0;
            width: 100%;
            overflow-x: hidden;
        }
        .payment-page-container {
            width: 100%;
            min-height: 100vh;
            background: transparent;
            margin: 0;
            padding: 0;
        }
        .payment-page-content {
            width: 100%;
            max-width: 100%;
            min-height: 100vh;
            padding: 20px;
            border-radius: 0;
            box-shadow: none;
            background-color: #F5F5F5;
            box-sizing: border-box;
            margin: 0;
        }
        @media (min-width: 1200px) {
            .payment-page-content {
                padding: 30px 40px;
            }
        }
    </style>
    <div class="payment-page-container">
    <div class="payment-page-content">
        <div style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4E3D 100%); padding: 30px; border-radius: 12px; margin-bottom: 30px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
            <h2 style="color: #F4F0E4; font-size: 32px; margin: 0 0 10px 0; font-weight: 700; text-align: center;">Traiter Paiement</h2>
            <p style="color: #F4F0E4; font-size: 16px; text-align: center; margin: 0; opacity: 0.95;">Traiter le paiement pour une commande existante chez Masunzu Bar Hotel</p>
        </div>
        <?php if (!empty($errors)): ?>
            <div style="background-color: #FFEBEE; color: #C62828; padding: 20px; border-radius: 10px; border-left: 5px solid #C62828; margin-bottom: 25px; box-shadow: 0 2px 8px rgba(198, 40, 40, 0.1);">
                <div style="display: flex; align-items: center; margin-bottom: 10px;">
                    <span style="font-size: 24px; margin-right: 10px;">⚠️</span>
                    <strong style="font-size: 16px;">Erreurs détectées</strong>
                </div>
                <ul style="margin: 0; padding-left: 25px; font-size: 14px; line-height: 1.8;">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
        <?php if (!empty($success)): ?>
            <div id="custom-notification" style="position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); background-color: #4B2F1F; color: #F4F0E4; padding: 20px 30px; border-radius: 8px; box-shadow: 0 4px 8px rgba(0,0,0,0.3); z-index: 10000; opacity: 0; transition: opacity 0.5s; text-align: center; min-width: 300px; max-width: 500px;">
                <?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?>
        <?php if (!empty($receipt_id_for_print) && $tcpdf_available): ?>
                    <br><br>
                    <a href="/masunzu_bar_hotel/modules/cashiers/reprint_receipt.php?receipt_id=<?php echo htmlspecialchars($receipt_id_for_print, ENT_QUOTES, 'UTF-8'); ?>" target="_blank" style="display: inline-block; background-color: #F4A261; color: #4B2F1F; padding: 10px 20px; border-radius: 6px; text-decoration: none; font-weight: 600; margin-top: 10px;">Imprimer Reçu</a>
                <?php endif; ?>
            </div>
            <div id="notification-backdrop" style="position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); z-index: 9999; opacity: 0; transition: opacity 0.5s;"></div>
            <script>
                document.addEventListener('DOMContentLoaded', function() {
                    const notification = document.getElementById('custom-notification');
                    const backdrop = document.getElementById('notification-backdrop');
                    if (notification && backdrop) {
                        // Animate in
                        setTimeout(function() {
                            backdrop.style.opacity = '1';
                            notification.style.opacity = '1';
                        }, 100);

                        // Animate out and remove after 5 seconds
                        setTimeout(function() {
                            backdrop.style.opacity = '0';
                            notification.style.opacity = '0';
                            setTimeout(function() {
                                notification.remove();
                                backdrop.remove();
                            }, 500);
                        }, 5000);
                        
                        // Close on backdrop click
                        backdrop.addEventListener('click', function() {
                            backdrop.style.opacity = '0';
                            notification.style.opacity = '0';
                            setTimeout(function() {
                                notification.remove();
                                backdrop.remove();
                            }, 500);
                        });
                    }
                });
            </script>
        <?php endif; ?>
        <?php 
        // Check if we should auto-print (either from session or from URL parameter)
        $should_auto_print = $auto_print_receipt || (isset($_GET['print_receipt']) && $_GET['print_receipt'] == '1');
        if ($should_auto_print && !empty($receipt_id_for_print) && $tcpdf_available): ?>
            <script>
                document.addEventListener('DOMContentLoaded', function() {
                    // Small delay to ensure page is fully loaded
                    setTimeout(function() {
                        if (!document.getElementById('auto-print-receipt-frame')) {
                            const iframe = document.createElement('iframe');
                            iframe.style.display = 'none';
                            iframe.id = 'auto-print-receipt-frame';
                            iframe.src = '/masunzu_bar_hotel/modules/cashiers/reprint_receipt.php?receipt_id=<?php echo htmlspecialchars($receipt_id_for_print, ENT_QUOTES, 'UTF-8'); ?>&auto_download=1';
                            document.body.appendChild(iframe);
                            console.log('Auto-printing receipt: <?php echo htmlspecialchars($receipt_id_for_print, ENT_QUOTES, 'UTF-8'); ?>');
                        }
                    }, 500);
                });
            </script>
        <?php endif; ?>
        <?php if (!empty($receipt_id_for_print)): ?>
            <?php unset($_SESSION['cashier_payment_receipt_id']); ?>
        <?php endif; ?>
        <?php if (!$tcpdf_available): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 15px; border-radius: 8px; border: 1px solid #FF0000; margin-bottom: 20px;">
                <p style="font-size: 14px;">La génération de reçus PDF est désactivée. Veuillez installer la bibliothèque TCPDF à libs/tcpdf/tcpdf.php ou via Composer.</p>
            </div>
        <?php endif; ?>
        <?php if (!$paid_invoice_items_exists): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 15px; border-radius: 8px; border: 1px solid #FF0000; margin-bottom: 20px;">
                <p style="font-size: 14px;">Erreur: La table paid_invoice_items est manquante. Contactez l'administrateur pour la créer.</p>
            </div>
        <?php endif; ?>
        <?php if (empty($invoices)): ?>
            <div style="background-color: #FFFFFF; padding: 40px; border-radius: 12px; text-align: center; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                <div style="font-size: 48px; margin-bottom: 15px;">📋</div>
                <p style="color: #6B5B52; font-size: 18px; margin: 0; font-weight: 500;">Aucune commande non payée et approuvée disponible dans votre province.</p>
            </div>
        <?php else: ?>
            <div style="background-color: #FFFFFF; padding: 30px; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin-bottom: 30px;">
            <?php
            // Generate CSRF token for form submission
            if (empty($_SESSION['payment_token'])) {
                $_SESSION['payment_token'] = bin2hex(random_bytes(32));
            }
            ?>
            <form method="POST" id="payment-form" style="display: flex; flex-direction: column; gap: 15px;" onsubmit="return handlePaymentSubmit(event);" novalidate enctype="multipart/form-data">
                <input type="hidden" name="payment_token" value="<?php echo htmlspecialchars($_SESSION['payment_token'], ENT_QUOTES, 'UTF-8'); ?>">
                <input type="hidden" name="submit" value="1">
                <div style="margin-bottom: 20px;">
                    <label for="invoice_id" style="color: #4B2F1F; font-size: 16px; font-weight: 600; margin-bottom: 8px; display: block;">Sélectionner Commande</label>
                    <select id="invoice_id" name="invoice_id" required style="width: 100%; padding: 12px; border: 2px solid #D3C2B6; border-radius: 8px; font-size: 15px; background-color: #FFFFFF; color: #4B2F1F; transition: all 0.3s;">
                        <option value="">Sélectionner une commande</option>
                        <?php foreach ($invoices as $invoice): ?>
                            <?php
                                $optionSubtotal = (int)($invoice['subtotal_amount'] ?? 0);
                                if ($optionSubtotal <= 0) {
                                    $optionSubtotal = (int)($invoice['paid_amount'] ?? 0);
                                }
                                $optionTaxAmount = (int)($invoice['tax_amount'] ?? 0);
                                $optionTaxMode = strtoupper((string)($invoice['tax_mode'] ?? 'HTVA'));
                                if (!in_array($optionTaxMode, $allowed_tax_modes, true)) {
                                    $optionTaxMode = 'HTVA';
                                }
                                $optionDisplayTotal = $optionTaxMode === 'TVAC'
                                    ? $optionSubtotal + $optionTaxAmount
                                    : $optionSubtotal;
                                $optionAmountLabel = $optionTaxMode === 'TVAC'
                                    ? sprintf('Sous-total %s BIF / TVAC %s BIF', number_format($optionSubtotal, 0, '', '.'), number_format($optionDisplayTotal, 0, '', '.'))
                                    : sprintf('Sous-total %s BIF', number_format($optionSubtotal, 0, '', '.'));
                                $optionCreatedAt = date('Y-m-d H:i', strtotime($invoice['created_at']));
                            ?>
                            <option value="<?php echo htmlspecialchars($invoice['id'], ENT_QUOTES, 'UTF-8'); ?>" data-is-loan-sale="<?php echo !empty($invoice['is_loan_sale']) ? '1' : '0'; ?>">
                                <?php echo htmlspecialchars($invoice['invoice_number'] . ' - ' . $invoice['customer_name'] . ' (' . $optionAmountLabel . ', Créée: ' . $optionCreatedAt . ')' . (!empty($invoice['is_loan_sale']) ? ' [Vente Crédit]' : ''), ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div style="margin-bottom: 20px;">
                    <label for="payment_method" style="color: #4B2F1F; font-size: 16px; font-weight: 600; margin-bottom: 8px; display: block;">Méthode de Paiement</label>
                    <select id="payment_method" name="payment_method" required style="width: 100%; padding: 12px; border: 2px solid #D3C2B6; border-radius: 8px; font-size: 15px; background-color: #FFFFFF; color: #4B2F1F; transition: all 0.3s;">
                        <option value="">Sélectionner méthode de paiement</option>
                        <option value="cash">Espèces</option>
                        <option value="bank_slip">Bordereau Bancaire (virement)</option>
                        <option value="loan" id="loan_option" style="display: none;">Crédit</option>
                    </select>
                    <p id="payment_method_note" style="color: #4B2F1F; font-size: 12px; margin-top: 6px;">Sélectionnez d'abord une commande pour voir les méthodes de paiement disponibles.</p>
                    <div id="loan_info_message" style="display: none; background-color: #E3F2FD; color: #1565C0; padding: 10px; border-radius: 6px; margin-top: 8px; font-size: 13px; border-left: 4px solid #2196F3;">
                        <strong>ℹ️ Note:</strong> Cette commande est une vente à crédit. En sélectionnant "Crédit", aucun argent ne sera reçu maintenant, mais un reçu sera généré indiquant que le client doit rembourser plus tard.
                    </div>
                    <div id="bank_slip_fields" style="display: none; margin-top: 15px; background: #F9F5F0; border: 1px dashed #D3C2B6; padding: 12px; border-radius: 8px;">
                        <div style="margin-bottom: 12px;">
                            <label for="bank_account_id" style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 6px; display: block;">Compte Bancaire</label>
                            <select id="bank_account_id" name="bank_account_id" style="width: 100%; padding: 10px; border: 1px solid #D3C2B6; border-radius: 8px; font-size: 14px; background-color: #FFFFFF; color: #4B2F1F;">
                                <option value="">Sélectionner un compte</option>
                                <?php foreach ($bank_accounts as $bankAccount): ?>
                                    <?php
                                        $accountLabel = $bankAccount['account_number'] ?? ('Compte #' . $bankAccount['id']);
                                        if (!empty($bankAccount['currency'])) {
                                            $accountLabel .= ' (' . $bankAccount['currency'] . ')';
                                        }
                                        if (!empty($bankAccount['is_main'])) {
                                            $accountLabel .= ' - Principal';
                                        }
                                    ?>
                                    <option value="<?php echo (int)$bankAccount['id']; ?>">
                                        <?php echo htmlspecialchars($accountLabel, ENT_QUOTES, 'UTF-8'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div style="margin-bottom: 12px;">
                            <label for="bank_slip_number" style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 6px; display: block;">Numéro de Bordereau / Référence</label>
                            <input type="text" id="bank_slip_number" name="bank_slip_number" style="width: 100%; padding: 10px; border: 1px solid #D3C2B6; border-radius: 8px; font-size: 14px; background-color: #FFFFFF; color: #4B2F1F;" placeholder="ex: 123456789">
                        </div>
                        <div>
                            <label for="bank_slip_file" style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 6px; display: block;">Fichier Bordereau (PDF, JPG, PNG, max 5MB)</label>
                            <input type="file" id="bank_slip_file" name="bank_slip_file" accept="application/pdf,image/jpeg,image/png" style="width: 100%; padding: 8px; border: 1px solid #D3C2B6; border-radius: 8px; font-size: 14px; background-color: #FFFFFF; color: #4B2F1F;">
                        </div>
                    </div>
                </div>
                <div style="margin-bottom: 20px;">
                    <label for="tax_mode" style="color: #4B2F1F; font-size: 16px; font-weight: 600; margin-bottom: 8px; display: block;">Mode de Taxe</label>
                    <select id="tax_mode" name="tax_mode" required style="width: 100%; padding: 12px; border: 2px solid #D3C2B6; border-radius: 8px; font-size: 15px; background-color: #FFFFFF; color: #4B2F1F; transition: all 0.3s;">
                        <option value="HTVA" <?php echo $selected_tax_mode === 'HTVA' ? 'selected' : ''; ?>>HTVA (sans TVA)</option>
                        <option value="TVAC" <?php echo $selected_tax_mode === 'TVAC' ? 'selected' : ''; ?>>TVAC (avec 18% TVA)</option>
                    </select>
                    <p style="color: #4B2F1F; font-size: 12px; margin-top: 6px;">Choisissez HTVA pour garder le total de la commande tel quel ou TVAC pour ajouter 18% TVA pour le client.</p>
                </div>
                <div style="margin-bottom: 25px;">
                    <label for="payment_details" style="color: #4B2F1F; font-size: 16px; font-weight: 600; margin-bottom: 8px; display: block;">Détails de Paiement (Optionnel)</label>
                    <input type="text" id="payment_details" name="payment_details" style="width: 100%; padding: 12px; border: 2px solid #D3C2B6; border-radius: 8px; font-size: 15px; background-color: #FFFFFF; color: #4B2F1F; transition: all 0.3s;" placeholder="ex: ID Transaction ou notes">
                </div>
                <button type="submit" name="submit" value="1" id="submit-payment-btn" style="width: 100%; background: linear-gradient(135deg, #4B2F1F 0%, #6B4E3D 100%); color: #F4F0E4; padding: 16px; border: none; border-radius: 10px; font-size: 18px; font-weight: 700; cursor: pointer; transition: all 0.3s ease; box-shadow: 0 4px 12px rgba(75, 47, 31, 0.4); text-transform: uppercase; letter-spacing: 0.5px;">
                    💳 Traiter Paiement
                </button>
            </form>
            </div>
            <script>
                function handlePaymentSubmit(event) {
                    const form = event.target;
                    const invoiceSelect = document.getElementById('invoice_id');
                    const paymentMethodSelect = document.getElementById('payment_method');
                    const taxModeSelect = document.getElementById('tax_mode');
                    const submitBtn = document.getElementById('submit-payment-btn');
                    const bankSlipFields = document.getElementById('bank_slip_fields');
                    const bankAccountSelect = document.getElementById('bank_account_id');
                    const bankSlipNumberInput = document.getElementById('bank_slip_number');
                    const bankSlipFileInput = document.getElementById('bank_slip_file');
                    
                    // Validate invoice is selected
                    if (!invoiceSelect.value) {
                        alert('Veuillez sélectionner une commande.');
                        event.preventDefault();
                        return false;
                    }
                    
                    // Validate payment method is selected (check both visible and hidden inputs)
                    const hiddenPaymentMethod = document.getElementById('hidden_loan_payment_method');
                    const effectivePaymentMethod = paymentMethodSelect.value || (hiddenPaymentMethod ? hiddenPaymentMethod.value : '');
                    if (!effectivePaymentMethod) {
                        alert('Veuillez sélectionner une méthode de paiement.');
                        event.preventDefault();
                        return false;
                    }

                    // Validate bank slip details
                    if (effectivePaymentMethod === 'bank_slip') {
                        if (!bankAccountSelect || !bankAccountSelect.value) {
                            alert('Veuillez sélectionner le compte bancaire où le paiement a été reçu.');
                            event.preventDefault();
                            return false;
                        }
                        if (!bankSlipNumberInput || !bankSlipNumberInput.value.trim()) {
                            alert('Veuillez saisir le numéro de bordereau bancaire.');
                            event.preventDefault();
                            return false;
                        }
                        if (!bankSlipFileInput || !bankSlipFileInput.files || bankSlipFileInput.files.length === 0) {
                            alert('Veuillez joindre le fichier du bordereau bancaire.');
                            event.preventDefault();
                            return false;
                        }
                    }
                    
                    // Validate tax mode is selected
                    if (!taxModeSelect.value) {
                        alert('Veuillez sélectionner un mode de paiement (HTVA ou TVAC).');
                        event.preventDefault();
                        return false;
                    }
                    
                    // Disable button and change text to prevent double submission
                    if (submitBtn) {
                        submitBtn.disabled = true;
                        submitBtn.style.opacity = '0.6';
                        submitBtn.style.cursor = 'not-allowed';
                        submitBtn.textContent = 'Traitement en cours...';
                    }
                    
                    // Allow form to submit normally
                    return true;
                }
                
                // Also prevent double-click on the button
                document.addEventListener('DOMContentLoaded', function() {
                    const submitBtn = document.getElementById('submit-payment-btn');
                    if (submitBtn) {
                        submitBtn.addEventListener('click', function(e) {
                            if (this.disabled) {
                                e.preventDefault();
                                return false;
                            }
                        });
                    }
                });
            </script>
        <?php endif; ?>
        
        <!-- Today's Payments List -->
        <div style="background-color: #FFFFFF; border-radius: 12px; padding: 25px; margin-top: 30px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; padding-bottom: 15px; border-bottom: 2px solid #E0E0E0;">
                <h3 style="color: #4B2F1F; font-size: 24px; margin: 0; font-weight: 700;">Paiements d'Aujourd'hui</h3>
                <span style="background-color: #4B2F1F; color: #F4F0E4; padding: 6px 14px; border-radius: 20px; font-size: 14px; font-weight: 600;"><?php echo count($today_payments); ?> paiement(s)</span>
            </div>
            <div style="margin-bottom: 20px; display: flex; gap: 12px; flex-wrap: wrap; align-items: flex-end; background-color: #F9F9F9; padding: 15px; border-radius: 8px;">
                <div style="flex: 1; min-width: 200px;">
                    <label for="payment-search" style="display: block; color: #4B2F1F; font-size: 14px; font-weight: 500; margin-bottom: 5px;">Rechercher</label>
                    <input type="text" id="payment-search" placeholder="Numéro commande, nom client..." style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                </div>
                <div style="flex: 1; min-width: 150px;">
                    <label for="payment-date-from" style="display: block; color: #4B2F1F; font-size: 14px; font-weight: 500; margin-bottom: 5px;">Date Début</label>
                    <input type="date" id="payment-date-from" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                </div>
                <div style="flex: 1; min-width: 150px;">
                    <label for="payment-date-to" style="display: block; color: #4B2F1F; font-size: 14px; font-weight: 500; margin-bottom: 5px;">Date Fin</label>
                    <input type="date" id="payment-date-to" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                </div>
                <button id="export-csv-btn" style="background-color: #4B2F1F; color: #F4F0E4; padding: 12px 20px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.3s ease; box-shadow: 0 2px 6px rgba(75, 47, 31, 0.3); min-width: 140px; height: fit-content;">📥 Exporter CSV</button>
            </div>
            <?php if (!empty($today_payments)): ?>
                <div style="overflow-x: auto; border-radius: 8px; border: 1px solid #E0E0E0;">
                    <table id="payments-table" style="width: 100%; border-collapse: collapse; background-color: #FFFFFF;">
                        <thead>
                            <tr style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4E3D 100%); color: #F4F0E4;">
                                <th style="padding: 14px 12px; text-align: left; font-weight: 600; font-size: 14px;">#</th>
                                <th style="padding: 14px 12px; text-align: left; font-weight: 600; font-size: 14px;">Numéro Commande</th>
                                <th style="padding: 14px 12px; text-align: left; font-weight: 600; font-size: 14px;">Client</th>
                                <th style="padding: 14px 12px; text-align: right; font-weight: 600; font-size: 14px;">Montant</th>
                                <th style="padding: 14px 12px; text-align: left; font-weight: 600; font-size: 14px;">Méthode</th>
                                <th style="padding: 14px 12px; text-align: left; font-weight: 600; font-size: 14px;">Heure</th>
                                <th style="padding: 14px 12px; text-align: center; font-weight: 600; font-size: 14px;">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $row_num = 1;
                            foreach ($today_payments as $payment): 
                                $payment_method_display = $payment['payment_method'] ?? 'N/A';
                                $is_loan = !empty($payment['is_loan_sale']) || strpos($payment['payment_details'] ?? '', 'Crédit') !== false || strpos($payment['payment_details'] ?? '', 'Type Paiement: Crédit') !== false;
                                if ($is_loan) {
                                    $payment_method_display = 'Crédit';
                                } else {
                                    if (($payment['payment_method'] ?? '') === 'bank_transfer' && strpos($payment['payment_details'] ?? '', 'Bordereau') !== false) {
                                        $payment_method_display = 'Bordereau Bancaire';
                                    }
                                    $payment_method_display = ucfirst(str_replace('_', ' ', $payment_method_display));
                                    $payment_method_display = str_replace(['Cash', 'Mobile', 'Bank'], ['Espèces', 'Mobile', 'Banque'], $payment_method_display);
                                }
                            ?>
                                <tr class="payment-row" style="border-bottom: 1px solid #F0F0F0; transition: background-color 0.2s;">
                                    <td style="padding: 12px; color: #4B2F1F; font-weight: 500;"><?php echo $row_num++; ?></td>
                                    <td style="padding: 12px; color: #4B2F1F; font-weight: 600;"><?php echo htmlspecialchars($payment['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #4B2F1F;"><?php echo htmlspecialchars($payment['customer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; text-align: right; color: #4B2F1F; font-weight: 600; font-size: 15px;"><?php echo number_format((int)$payment['paid_amount'], 0, ',', '.'); ?> BIF</td>
                                    <td style="padding: 12px;">
                                        <?php if ($is_loan): ?>
                                            <span style="background-color: #FFF3CD; color: #856404; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 600;"><?php echo htmlspecialchars($payment_method_display, ENT_QUOTES, 'UTF-8'); ?></span>
                                        <?php else: ?>
                                            <span style="background-color: #D1ECF1; color: #0C5460; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 600;"><?php echo htmlspecialchars($payment_method_display, ENT_QUOTES, 'UTF-8'); ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td style="padding: 12px; color: #6B5B52; font-weight: 500;"><?php echo date('H:i', strtotime($payment['paid_at'])); ?></td>
                                    <td style="padding: 12px; text-align: center;">
                                        <div style="display: flex; gap: 6px; justify-content: center; flex-wrap: wrap;">
                                            <?php if (!empty($payment['receipt_id'])): ?>
                                                <a href="/masunzu_bar_hotel/modules/cashiers/reprint_receipt.php?receipt_id=<?php echo htmlspecialchars($payment['receipt_id'], ENT_QUOTES, 'UTF-8'); ?>&reprint=1" target="_blank" style="display: inline-block; background-color: #4B2F1F; color: #F4F0E4; padding: 8px 14px; border-radius: 6px; text-decoration: none; font-size: 12px; font-weight: 600; transition: all 0.2s; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">🖨️ Réimprimer</a>
                                            <?php endif; ?>
                                            <button onclick="viewPaymentDetails(<?php echo (int)$payment['id']; ?>)" style="background-color: #F4A261; color: #4B2F1F; padding: 8px 14px; border: none; border-radius: 6px; font-size: 12px; font-weight: 600; cursor: pointer; transition: all 0.2s; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">👁️ Voir</button>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div style="text-align: center; padding: 40px; background-color: #F9F9F9; border-radius: 8px;">
                    <div style="font-size: 48px; margin-bottom: 15px;">💳</div>
                    <p style="color: #6B5B52; font-size: 16px; margin: 0; font-weight: 500;">Aucun paiement traité aujourd'hui.</p>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Payment Details Modal -->
        <div id="payment-details-modal" class="payment-modal" style="display: none;">
            <div class="payment-modal-content">
                <div id="payment-details-content"></div>
            </div>
        </div>
        
        <div style="text-align: center; margin-top: 30px; padding-top: 20px; border-top: 2px solid #E0E0E0;">
            <a href="/masunzu_bar_hotel/dashboards/cashier_dashboard.php" style="display: inline-block; background-color: #4B2F1F; color: #F4F0E4; padding: 12px 30px; border-radius: 8px; text-decoration: none; font-size: 16px; font-weight: 600; transition: all 0.3s; box-shadow: 0 2px 6px rgba(75, 47, 31, 0.3);">← Retour au Tableau de Bord</a>
        </div>
    </div>
    <style>
        /* Payment Modal Styles */
        .payment-modal {
            display: none;
            position: fixed;
            z-index: 10000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.5);
            backdrop-filter: blur(2px);
            align-items: center;
            justify-content: center;
        }
        .payment-modal-content {
            background-color: #F4F0E4;
            margin: auto;
            border-radius: 12px;
            width: 90%;
            max-width: 900px;
            max-height: 90vh;
            overflow: hidden;
            box-shadow: 0 8px 32px rgba(0,0,0,0.3);
            display: flex;
            flex-direction: column;
        }
        .payment-modal-header {
            background: #4B2F1F;
            color: #F4F0E4;
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .payment-modal-close {
            background: none;
            border: none;
            color: #F4F0E4;
            font-size: 32px;
            cursor: pointer;
            line-height: 1;
            padding: 0;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: background 0.2s;
        }
        .payment-modal-close:hover {
            background: rgba(255,255,255,0.2);
        }
        .payment-modal-body {
            padding: 30px;
            overflow-y: auto;
            flex: 1;
        }
        .payment-info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
            padding: 20px;
            background: #FFFFFF;
            border-radius: 8px;
        }
        .payment-info-item {
            display: flex;
            flex-direction: column;
        }
        .payment-info-label {
            color: #6B5B52;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 4px;
        }
        .payment-info-value {
            color: #4B2F1F;
            font-size: 16px;
            font-weight: 600;
        }
        .badge-loan {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            background: #D1ECF1;
            color: #0C5460;
            border: 1px solid #BEE5EB;
        }
        .payment-items-table {
            width: 100%;
            border-collapse: collapse;
            background: #FFFFFF;
            border-radius: 8px;
            overflow: hidden;
            margin-bottom: 20px;
        }
        .payment-items-table thead {
            background: #F4A261;
            color: #4B2F1F;
        }
        .payment-items-table th {
            padding: 12px;
            text-align: left;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
        }
        .payment-items-table td {
            padding: 12px;
            border-bottom: 1px solid #F4F0E4;
            color: #4B2F1F;
            font-size: 14px;
        }
        .payment-items-table tbody tr:last-child td {
            border-bottom: none;
        }
        .payment-summary {
            background: #FFFFFF;
            border-radius: 8px;
            padding: 20px;
            margin-top: 20px;
        }
        .payment-summary-row {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid #F4F0E4;
        }
        .payment-summary-row:last-child {
            border-bottom: none;
        }
        .payment-summary-row.total {
            border-top: 2px solid #4B2F1F;
            margin-top: 10px;
            padding-top: 15px;
            font-weight: 700;
            font-size: 18px;
        }
        .payment-summary-label {
            color: #4B2F1F;
            font-weight: 600;
        }
        .payment-summary-value {
            color: #4B2F1F;
            font-weight: 600;
        }
        .payment-summary-row.total .payment-summary-value {
            color: #4B2F1F;
            font-size: 20px;
        }
        .payment-notes {
            background: #FFF3CD;
            color: #856404;
            border: 1px solid #FFE69C;
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
            font-size: 14px;
        }
        .payment-modal-footer {
            padding: 20px 30px;
            background: #FFFFFF;
            border-top: 2px solid #D3C2B6;
            display: flex;
            justify-content: flex-end;
            gap: 12px;
        }
        .btn-reprint-modal, .btn-close-modal {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
            text-decoration: none;
            display: inline-block;
        }
        .btn-reprint-modal {
            background: #4B2F1F;
            color: #F4F0E4;
        }
        .btn-reprint-modal:hover {
            background: #3A2418;
        }
        .btn-close-modal {
            background: #6B5B52;
            color: #F4F0E4;
        }
        .btn-close-modal:hover {
            background: #5A4B42;
        }
        button:hover {
            background-color: #F4A261;
            color: #4B2F1F;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.3);
        }
        input:focus, select:focus {
            outline: none;
            border-color: #4B2F1F !important;
            box-shadow: 0 0 0 3px rgba(75, 47, 31, 0.1) !important;
            transform: translateY(-1px);
        }
        .payment-row:hover {
            background-color: #F9F9F9 !important;
        }
        select:disabled {
            background-color: #E0E0E0;
            color: #4B2F1F;
            cursor: not-allowed;
            opacity: 0.7;
        }
        @media (max-width: 768px) {
            .payment-page-content { padding: 15px; }
            h2 { font-size: 24px; }
            label, select, input, button, a { font-size: 14px; }
            .payment-modal-content {
                width: 95%;
                max-height: 95vh;
            }
            .payment-modal-body {
                padding: 20px;
            }
            .payment-info-grid {
                grid-template-columns: 1fr;
            }
            #payments-table {
                font-size: 12px;
            }
            #payments-table th,
            #payments-table td {
                padding: 8px 6px;
            }
        }
        @media (max-width: 480px) {
            .payment-page-content { padding: 10px; }
            h2 { font-size: 20px; }
            label, select, input, button, a { font-size: 12px; }
        }
    </style>
    <script>
        // Search and filter functionality
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.getElementById('payment-search');
            const dateFromInput = document.getElementById('payment-date-from');
            const dateToInput = document.getElementById('payment-date-to');
            const exportBtn = document.getElementById('export-csv-btn');
            const paymentRows = document.querySelectorAll('.payment-row');
            
            // Get payment data from table
            const getPaymentData = () => {
                const data = [];
                paymentRows.forEach(function(row) {
                    const cells = row.querySelectorAll('td');
                    if (cells.length >= 7) {
                        data.push({
                            number: cells[1].textContent.trim(),
                            customer: cells[2].textContent.trim(),
                            amount: cells[3].textContent.trim(),
                            method: cells[4].textContent.trim(),
                            time: cells[5].textContent.trim(),
                            element: row
                        });
                    }
                });
                return data;
            };
            
            // Parse date from payment row time (HH:MM format - same day)
            const getPaymentDate = (timeStr, baseDate) => {
                const date = new Date(baseDate);
                const [hours, minutes] = timeStr.split(':').map(Number);
                date.setHours(hours, minutes, 0, 0);
                return date;
            };
            
            // Filter payments based on search and dates
            const filterPayments = () => {
                const searchTerm = (searchInput ? searchInput.value.toLowerCase().trim() : '');
                const dateFrom = dateFromInput ? dateFromInput.value : '';
                const dateTo = dateToInput ? dateToInput.value : '';
                const today = new Date().toISOString().split('T')[0];
                
                const dateFromObj = dateFrom ? new Date(dateFrom + 'T00:00:00') : null;
                const dateToObj = dateTo ? new Date(dateTo + 'T23:59:59') : null;
                
                let visibleCount = 0;
                paymentRows.forEach(function(row) {
                    const cells = row.querySelectorAll('td');
                    if (cells.length < 7) return;
                    
                    const text = row.textContent.toLowerCase();
                    const timeStr = cells[5].textContent.trim();
                    const paymentDate = getPaymentDate(timeStr, today);
                    
                    // Check search term
                    const matchesSearch = !searchTerm || text.includes(searchTerm);
                    
                    // Check date range
                    const matchesDateFrom = !dateFromObj || paymentDate >= dateFromObj;
                    const matchesDateTo = !dateToObj || paymentDate <= dateToObj;
                    
                    const shouldShow = matchesSearch && matchesDateFrom && matchesDateTo;
                    row.style.display = shouldShow ? '' : 'none';
                    if (shouldShow) visibleCount++;
                });
                
                // Show/hide empty message
                const tbody = document.querySelector('#payments-table tbody');
                let emptyMsg = document.getElementById('payments-empty-msg');
                if (visibleCount === 0 && tbody) {
                    if (!emptyMsg) {
                        emptyMsg = document.createElement('tr');
                        emptyMsg.id = 'payments-empty-msg';
                        emptyMsg.innerHTML = '<td colspan="7" style="padding: 20px; text-align: center; color: #6B5B52;">No payments match your filters.</td>';
                        tbody.appendChild(emptyMsg);
                    }
                    emptyMsg.style.display = '';
                } else if (emptyMsg) {
                    emptyMsg.style.display = 'none';
                }
            };
            
            // Export to CSV
            const exportToCSV = () => {
                const rows = Array.from(paymentRows).filter(row => row.style.display !== 'none');
                
                if (rows.length === 0) {
                    alert('No payments to export. Please adjust your filters.');
                    return;
                }
                
                const headers = ['#', 'Invoice Number', 'Customer', 'Amount', 'Payment Method', 'Time'];
                const csvContent = [];
                csvContent.push(headers.join(','));
                
                rows.forEach((row, index) => {
                    const cells = row.querySelectorAll('td');
                    const rowData = [
                        index + 1,
                        `"${cells[1].textContent.trim().replace(/"/g, '""')}"`,
                        `"${cells[2].textContent.trim().replace(/"/g, '""')}"`,
                        cells[3].textContent.trim().replace(/"/g, '""'),
                        `"${cells[4].textContent.trim().replace(/"/g, '""')}"`,
                        cells[5].textContent.trim()
                    ];
                    csvContent.push(rowData.join(','));
                });
                
                const csv = csvContent.join('\n');
                const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
                const link = document.createElement('a');
                const url = URL.createObjectURL(blob);
                
                const dateStr = new Date().toISOString().split('T')[0];
                link.setAttribute('href', url);
                link.setAttribute('download', `payments_${dateStr}.csv`);
                link.style.visibility = 'hidden';
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
            };
            
            // Event listeners
            if (searchInput) {
                searchInput.addEventListener('input', filterPayments);
            }
            if (dateFromInput) {
                dateFromInput.addEventListener('change', filterPayments);
            }
            if (dateToInput) {
                dateToInput.addEventListener('change', filterPayments);
            }
            if (exportBtn) {
                exportBtn.addEventListener('click', exportToCSV);
            }
            
            // Handle invoice selection and payment method validation
            const invoiceSelect = document.getElementById('invoice_id');
            const paymentMethodSelect = document.getElementById('payment_method');
            const loanOption = document.getElementById('loan_option');
            const paymentMethodNote = document.getElementById('payment_method_note');
            const loanErrorMessage = document.getElementById('loan_error_message');
            const bankSlipFields = document.getElementById('bank_slip_fields');
            const bankAccountSelect = document.getElementById('bank_account_id');
            const bankSlipNumberInput = document.getElementById('bank_slip_number');
            const bankSlipFileInput = document.getElementById('bank_slip_file');
            
            if (invoiceSelect && paymentMethodSelect && loanOption) {
                invoiceSelect.addEventListener('change', function() {
                    const selectedOption = this.options[this.selectedIndex];
                    const isLoanSale = selectedOption ? selectedOption.getAttribute('data-is-loan-sale') === '1' : false;
                    
                    if (isLoanSale) {
                        // Loan sale: Force "Credit" method and disable the dropdown
                        loanOption.style.display = '';
                        loanOption.removeAttribute('disabled');
                        paymentMethodSelect.value = 'loan';
                        paymentMethodSelect.disabled = true;
                        paymentMethodSelect.removeAttribute('required');
                        if (bankSlipFields) {
                            bankSlipFields.style.display = 'none';
                        }
                        if (bankAccountSelect) bankAccountSelect.value = '';
                        if (bankSlipNumberInput) bankSlipNumberInput.value = '';
                        if (bankSlipFileInput) bankSlipFileInput.value = '';
                        // Rename the select so it won't be submitted, and create a hidden input
                        paymentMethodSelect.name = 'payment_method_display';
                        // Show info message
                        const loanInfoMessage = document.getElementById('loan_info_message');
                        if (loanInfoMessage) {
                            loanInfoMessage.style.display = 'block';
                        }
                        paymentMethodNote.innerHTML = '<strong>Note:</strong> Cette commande est une vente à crédit. La méthode de paiement est automatiquement définie sur "Crédit" et ne peut pas être modifiée. Aucun argent ne sera reçu maintenant, mais un reçu sera généré.';
                        paymentMethodNote.style.color = '#4B2F1F';
                        if (loanErrorMessage) {
                            loanErrorMessage.style.display = 'none';
                        }
                        
                        // Add a hidden input to ensure the value is submitted even when disabled
                        let hiddenLoanInput = document.getElementById('hidden_loan_payment_method');
                        if (!hiddenLoanInput) {
                            hiddenLoanInput = document.createElement('input');
                            hiddenLoanInput.type = 'hidden';
                            hiddenLoanInput.id = 'hidden_loan_payment_method';
                            hiddenLoanInput.name = 'payment_method';
                            hiddenLoanInput.value = 'loan';
                            paymentMethodSelect.parentNode.appendChild(hiddenLoanInput);
                        } else {
                            hiddenLoanInput.value = 'loan';
                        }
                    } else {
                        // Non-loan sale: Hide "Loan" option, enable dropdown, clear selection
                        paymentMethodSelect.value = '';
                        paymentMethodSelect.disabled = false;
                        paymentMethodSelect.setAttribute('required', 'required');
                        paymentMethodSelect.name = 'payment_method';
                        loanOption.style.display = 'none';
                        loanOption.setAttribute('disabled', 'disabled');
                        // Hide loan info message
                        const loanInfoMessage = document.getElementById('loan_info_message');
                        if (loanInfoMessage) {
                            loanInfoMessage.style.display = 'none';
                        }
                        if (bankSlipFields) {
                            bankSlipFields.style.display = 'none';
                        }
                        paymentMethodNote.innerHTML = 'Sélectionnez d\'abord une commande pour voir les méthodes de paiement disponibles.';
                        paymentMethodNote.style.color = '#4B2F1F';
                        if (loanErrorMessage) {
                            loanErrorMessage.style.display = 'none';
                        }
                        
                        // Remove hidden input if it exists
                        const hiddenLoanInput = document.getElementById('hidden_loan_payment_method');
                        if (hiddenLoanInput) {
                            hiddenLoanInput.remove();
                        }
                    }
                });
                
                // Prevent manual changes when dropdown is disabled (for loan sales)
                paymentMethodSelect.addEventListener('change', function() {
                    const selectedInvoiceOption = invoiceSelect.options[invoiceSelect.selectedIndex];
                    const isLoanSale = selectedInvoiceOption ? selectedInvoiceOption.getAttribute('data-is-loan-sale') === '1' : false;
                    
                    if (this.disabled || isLoanSale) {
                        // Force back to 'loan' if it's a loan sale
                        this.value = 'loan';
                        const hiddenLoanInput = document.getElementById('hidden_loan_payment_method');
                        if (hiddenLoanInput) {
                            hiddenLoanInput.value = 'loan';
                        }
                        return false;
                    }
                    
                    if (this.value === 'loan' && !isLoanSale) {
                        this.value = '';
                        if (loanErrorMessage) {
                            loanErrorMessage.style.display = 'block';
                        }
                        alert('La méthode de paiement "Crédit" n\'est disponible que pour les commandes créées comme ventes à crédit. Cette commande n\'a pas été créée comme vente à crédit. Veuillez sélectionner Espèces à la place.');
                        return false;
                    } else {
                        if (loanErrorMessage) {
                            loanErrorMessage.style.display = 'none';
                        }
                        if (this.value === 'bank_slip') {
                            if (bankSlipFields) {
                                bankSlipFields.style.display = 'block';
                            }
                        } else if (bankSlipFields) {
                            bankSlipFields.style.display = 'none';
                            if (bankAccountSelect) bankAccountSelect.value = '';
                            if (bankSlipNumberInput) bankSlipNumberInput.value = '';
                            if (bankSlipFileInput) bankSlipFileInput.value = '';
                        }
                    }
                });
                
                // Also prevent any attempts to enable the select for loan sales
                paymentMethodSelect.addEventListener('mousedown', function(e) {
                    const selectedInvoiceOption = invoiceSelect.options[invoiceSelect.selectedIndex];
                    const isLoanSale = selectedInvoiceOption ? selectedInvoiceOption.getAttribute('data-is-loan-sale') === '1' : false;
                    if (isLoanSale && this.disabled) {
                        e.preventDefault();
                        return false;
                    }
                });
                
                paymentMethodSelect.addEventListener('keydown', function(e) {
                    const selectedInvoiceOption = invoiceSelect.options[invoiceSelect.selectedIndex];
                    const isLoanSale = selectedInvoiceOption ? selectedInvoiceOption.getAttribute('data-is-loan-sale') === '1' : false;
                    if (isLoanSale && this.disabled) {
                        e.preventDefault();
                        return false;
                    }
                });
            }
        });
        
        function viewPaymentDetails(paymentId) {
            // Fetch payment details via AJAX
            fetch('/masunzu_bar_hotel/modules/cashiers/get_payment_details.php?payment_id=' + paymentId)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        openPaymentModal(data.payment);
                    } else {
                        alert('Erreur lors du chargement des détails du paiement: ' + (data.error || 'Erreur inconnue'));
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Échec du chargement des détails du paiement. Veuillez réessayer.');
                });
        }

        function openPaymentModal(payment) {
            const modal = document.getElementById('payment-details-modal');
            const modalContent = document.getElementById('payment-details-content');
            
            if (!modal || !modalContent) {
                console.error('Modal elements not found');
                return;
            }

            // Format payment method
            let paymentMethodDisplay = payment.payment_method || 'N/A';
            if (payment.payment_details && (payment.payment_details.includes('Payment Type: Loan') || payment.payment_details.includes('Type Paiement: Crédit'))) {
                paymentMethodDisplay = 'Crédit';
            } else {
                if ((payment.payment_method === 'bank_transfer') && payment.payment_details && payment.payment_details.includes('Bordereau')) {
                    paymentMethodDisplay = 'Bordereau Bancaire';
                }
                paymentMethodDisplay = paymentMethodDisplay.charAt(0).toUpperCase() + paymentMethodDisplay.slice(1).replace('_', ' ');
                if (paymentMethodDisplay === 'Cash') paymentMethodDisplay = 'Espèces';
                if (paymentMethodDisplay === 'Bank transfer') paymentMethodDisplay = 'Virement Bancaire';
                if (paymentMethodDisplay === 'Mobile') paymentMethodDisplay = 'Paiement Mobile';
            }

            // Build modal content
            let itemsHtml = '';
            if (payment.items && payment.items.length > 0) {
                payment.items.forEach((item, index) => {
                    const lineTotal = item.quantity * item.price_per_unit;
                    itemsHtml += `
                        <tr style="border-bottom: 1px solid #F4F0E4;">
                            <td style="padding: 12px; color: #4B2F1F;">${index + 1}</td>
                            <td style="padding: 12px; color: #4B2F1F; font-weight: 600;">${escapeHtml(item.product_name)}</td>
                            <td style="padding: 12px; text-align: right; color: #4B2F1F;">${formatNumber(item.quantity)}</td>
                            <td style="padding: 12px; text-align: right; color: #4B2F1F;">${formatNumber(item.price_per_unit)} BIF</td>
                            <td style="padding: 12px; text-align: right; color: #4B2F1F; font-weight: 600;">${formatNumber(lineTotal)} BIF</td>
                        </tr>
                    `;
                });
            } else {
                itemsHtml = '<tr><td colspan="5" style="padding: 20px; text-align: center; color: #6B5B52;">Aucun article trouvé</td></tr>';
            }

            modalContent.innerHTML = `
                <div class="payment-modal-header">
                    <h2 style="margin: 0; color: #F4F0E4; font-size: 24px;">Détails du Paiement</h2>
                    <button class="payment-modal-close" onclick="closePaymentModal()">&times;</button>
                </div>
                <div class="payment-modal-body">
                    <div class="payment-info-grid">
                        <div class="payment-info-item">
                            <div class="payment-info-label">Numéro Commande</div>
                            <div class="payment-info-value">${escapeHtml(payment.invoice_number || 'N/A')}</div>
                        </div>
                        <div class="payment-info-item">
                            <div class="payment-info-label">Client</div>
                            <div class="payment-info-value">${escapeHtml(payment.customer_name || 'N/A')}</div>
                        </div>
                        <div class="payment-info-item">
                            <div class="payment-info-label">Téléphone</div>
                            <div class="payment-info-value">${escapeHtml(payment.phone_number || 'N/A')}</div>
                        </div>
                        <div class="payment-info-item">
                            <div class="payment-info-label">Date Paiement</div>
                            <div class="payment-info-value">${formatDateTime(payment.paid_at)}</div>
                        </div>
                        <div class="payment-info-item">
                            <div class="payment-info-label">Méthode Paiement</div>
                            <div class="payment-info-value">${escapeHtml(paymentMethodDisplay)}</div>
                        </div>
                        ${payment.is_loan_sale ? `
                        <div class="payment-info-item">
                            <div class="payment-info-label">Type</div>
                            <div class="payment-info-value"><span class="badge-loan">Vente Crédit</span></div>
                        </div>
                        ` : ''}
                        ${payment.nif ? `
                        <div class="payment-info-item">
                            <div class="payment-info-label">NIF</div>
                            <div class="payment-info-value">${escapeHtml(payment.nif)}</div>
                        </div>
                        ` : ''}
                        ${payment.rc ? `
                        <div class="payment-info-item">
                            <div class="payment-info-label">RC</div>
                            <div class="payment-info-value">${escapeHtml(payment.rc)}</div>
                        </div>
                        ` : ''}
                    </div>

                    <h3 style="color: #4B2F1F; font-size: 18px; margin-top: 25px; margin-bottom: 15px;">Articles</h3>
                    <div style="overflow-x: auto;">
                        <table class="payment-items-table">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Produit</th>
                                    <th style="text-align: right;">Quantité</th>
                                    <th style="text-align: right;">Prix Unitaire</th>
                                    <th style="text-align: right;">Total Ligne</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${itemsHtml}
                            </tbody>
                        </table>
                    </div>

                    <div class="payment-summary">
                        ${payment.subtotal_amount ? `
                        <div class="payment-summary-row">
                            <span class="payment-summary-label">Sous-total:</span>
                            <span class="payment-summary-value">${formatNumber(payment.subtotal_amount)} BIF</span>
                        </div>
                        ` : ''}
                        ${payment.tax_amount && payment.tax_amount > 0 ? `
                        <div class="payment-summary-row">
                            <span class="payment-summary-label">TVA (${payment.tax_rate || 0}%):</span>
                            <span class="payment-summary-value">${formatNumber(payment.tax_amount)} BIF</span>
                        </div>
                        ` : ''}
                        <div class="payment-summary-row total">
                            <span class="payment-summary-label">Montant Total:</span>
                            <span class="payment-summary-value">${formatNumber(payment.paid_amount)} BIF</span>
                        </div>
                    </div>

                    ${payment.payment_details ? `
                    <div class="payment-notes">
                        <strong>Détails Paiement:</strong> ${escapeHtml(payment.payment_details)}
                    </div>
                    ` : ''}
                </div>
                <div class="payment-modal-footer">
                    ${payment.receipt_id ? `
                    <a href="/masunzu_bar_hotel/modules/cashiers/reprint_receipt.php?receipt_id=${payment.receipt_id}&reprint=1" target="_blank" class="btn-reprint-modal">Réimprimer Reçu</a>
                    ` : ''}
                    <button onclick="closePaymentModal()" class="btn-close-modal">Fermer</button>
                </div>
            `;

            modal.style.display = 'flex';
            document.body.style.overflow = 'hidden';
        }

        function closePaymentModal() {
            const modal = document.getElementById('payment-details-modal');
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = '';
            }
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        function formatNumber(num) {
            return new Intl.NumberFormat('fr-FR').format(num || 0);
        }

        function formatDateTime(dateString) {
            if (!dateString) return 'N/A';
            const date = new Date(dateString);
            return date.toLocaleString('fr-FR', {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit'
            });
        }

        // Close modal when clicking outside
        document.addEventListener('click', function(event) {
            const modal = document.getElementById('payment-details-modal');
            if (event.target === modal) {
                closePaymentModal();
            }
        });

        // Close modal with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closePaymentModal();
            }
        });
    </script>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
<?php
} catch (Exception $e) {
    error_log("Critical error in process_payment.php: " . $e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>An unexpected error occurred. Check logs for details. Contact administrator.</p>";
    if ($conn) $conn->close();
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
?>
