<?php
// Start output buffering to prevent any output before PDF generation
ob_start();

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/cash_helpers.php';

try {
    require_login();

    $role_name = $_SESSION['role_name'] ?? '';
    
    // Restrict to Caissier_Comptable role only
    if ($role_name !== 'Caissier_Comptable') {
        $dashboard = strtolower(str_replace(' ', '_', $role_name)) . '_dashboard.php';
        header('Location: /masunzu_bar_hotel/dashboards/' . $dashboard);
        exit;
    }

    // Don't include header.php yet - we need to process POST requests first (especially PDF generation)
    // Header will be included after POST processing if not generating PDF
    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception('Database connection failed: ' . mysqli_connect_error());
    }

    // Ensure expense.payment_method column exists
    $column_check = $conn->query("SHOW COLUMNS FROM expense LIKE 'payment_method'");
    if (!$column_check) {
        throw new Exception('Failed to inspect expense table: ' . $conn->error);
    }
    if ($column_check->num_rows === 0) {
        if (!$conn->query("ALTER TABLE expense ADD COLUMN payment_method ENUM('cash','bank_transfer','mobile') DEFAULT 'cash'")) {
            throw new Exception('Failed to add payment_method column: ' . $conn->error);
        }
    }
    if ($column_check instanceof mysqli_result) {
        $column_check->free();
    }

    // Check if requisition.department exists (optional feature)
    $department_exists = false;
    $department_check = $conn->query("SHOW COLUMNS FROM requisition LIKE 'department'");
    if (!$department_check) {
        throw new Exception('Failed to inspect requisition table: ' . $conn->error);
    }
    if ($department_check->num_rows > 0) {
        $department_exists = true;
    }
    if ($department_check instanceof mysqli_result) {
        $department_check->free();
    }

    // Ensure we have province in session
    if (!isset($_SESSION['province_id']) || empty($_SESSION['province_id'])) {
        $stmt = $conn->prepare('SELECT province_id FROM user WHERE id = ?');
        $stmt->bind_param('i', $_SESSION['user_id']);
        $stmt->execute();
        $province_row = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        $_SESSION['province_id'] = $province_row['province_id'] ?? null;
    }
    $province_id = $_SESSION['province_id'];
    if (!$province_id) {
        throw new Exception('No province assigned to this cashier.');
    }

    // Check for active shift
    $stmt = $conn->prepare("SELECT id, shift_start FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
    if (!$stmt) {
        throw new Exception("Prepare failed for shift check: " . $conn->error);
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $shift_result = $stmt->get_result();
    $active_shift = $shift_result->fetch_assoc();
    $stmt->close();
    
    if (!$active_shift) {
        include __DIR__ . '/../../includes/header.php';
        ?>
        <div style="padding: 40px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; display: flex; align-items: center; justify-content: center;">
            <div style="max-width: 600px; background: #FDFBF7; padding: 40px; border-radius: 12px; box-shadow: 0 8px 16px rgba(0,0,0,0.2); text-align: center;">
                <div style="font-size: 48px; margin-bottom: 20px;">⚠️</div>
                <h1 style="color: #D9534F; margin: 0 0 15px; font-size: 24px;">Aucune Session Active</h1>
                <p style="color: #666; font-size: 16px; margin: 0 0 30px; line-height: 1.6;">
                    Vous devez démarrer une session de caissier avant de payer les réquisitions. Veuillez aller à Gestion des Soldes pour initier votre session.
                </p>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                    <a href="/masunzu_bar_hotel/dashboards/cashier_dashboard.php" 
                       style="background-color: #4B2F1F; color: #FDFBF7; padding: 12px 20px; border-radius: 8px; text-decoration: none; font-weight: 600; display: inline-block; transition: all 0.3s;">
                        ← Retour au Tableau de Bord
                    </a>
                    <a href="/masunzu_bar_hotel/modules/cashiers/balance_management.php" 
                       style="background-color: #2A9D8F; color: #FDFBF7; padding: 12px 20px; border-radius: 8px; text-decoration: none; font-weight: 600; display: inline-block; transition: all 0.3s;">
                        ⏱️ Démarrer Session
                    </a>
                </div>
            </div>
        </div>
        <?php
        include __DIR__ . '/../../includes/footer.php';
        exit;
    }

    $active_shift_id = (int)$active_shift['id'];
    $shift_start = $active_shift['shift_start'] ?? null;
    $shift_deposits = (float)($active_shift['deposits'] ?? 0);
    $shift_expenses = 0.0;
    $shift_transfers = 0.0;
    $current_balance_active = 0.0;

    if ($shift_start) {
        // Expenses in current shift
        $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
        $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
        $stmt->execute();
        $shift_expenses = (float)($stmt->get_result()->fetch_assoc()['total_expenses'] ?? 0);
        $stmt->close();

        // Transfers in current shift
        $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transfers FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
        $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
        $stmt->execute();
        $shift_transfers = (float)($stmt->get_result()->fetch_assoc()['total_transfers'] ?? 0);
        $stmt->close();

        $beginning_balance = (float)($active_shift['beginning_balance'] ?? 0);
        $current_balance_active = $beginning_balance + $shift_deposits - $shift_expenses - $shift_transfers;
        if ($current_balance_active < 0) {
            $current_balance_active = 0;
        }
    }

    // Money from closed shifts not yet deposited/transferred (same logic as cashier_dashboard)
    $available_from_closed_shifts = 0.0;
    $has_yesno_columns = false;
    $checkColumns = $conn->query("
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'cashier_shift_closure' 
        AND COLUMN_NAME = 'deposited_to_bank'
    ");
    if ($checkColumns && $checkColumns->num_rows > 0) {
        $has_yesno_columns = true;
        $checkColumns->close();
    }

    if ($has_yesno_columns) {
        $stmt = $conn->prepare("
            SELECT 
                cb.id,
                cb.shift_end,
                cb.beginning_balance,
                cb.end_balance,
                cb.deposits,
                cb.withdrawals,
                COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
                COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
                csc.counted_amount
            FROM cashier_balance cb
            LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
            WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
            ORDER BY cb.shift_end DESC
        ");
    } else {
        $stmt = $conn->prepare("
            SELECT 
                cb.id,
                cb.shift_end,
                cb.beginning_balance,
                cb.end_balance,
                cb.deposits,
                cb.withdrawals
            FROM cashier_balance cb
            WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
            ORDER BY cb.shift_end DESC
        ");
    }

    if ($stmt) {
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $closed_shifts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        foreach ($closed_shifts as $shift) {
            $shift_id = (int)$shift['id'];
            $end_balance = isset($shift['end_balance']) && $shift['end_balance'] !== null
                ? (float)$shift['end_balance']
                : (float)($shift['beginning_balance'] ?? 0) + (float)($shift['deposits'] ?? 0) - (float)($shift['withdrawals'] ?? 0);

            $is_deposited = false;
            $is_transferred = false;

            if ($has_yesno_columns) {
                $is_deposited = strtolower($shift['deposited_to_bank'] ?? 'no') === 'yes';
                $is_transferred = strtolower($shift['transferred_to_accountant'] ?? 'no') === 'yes';
            } else {
                $stmtDeposit = $conn->prepare("
                    SELECT COUNT(*) AS has_deposit
                    FROM cashier_shift_deposit csd
                    INNER JOIN bank_deposit bd ON csd.deposit_id = bd.id
                    WHERE csd.shift_id = ?
                ");
                if ($stmtDeposit) {
                    $stmtDeposit->bind_param("i", $shift_id);
                    $stmtDeposit->execute();
                    $depositResult = $stmtDeposit->get_result()->fetch_assoc();
                    $is_deposited = (int)($depositResult['has_deposit'] ?? 0) > 0;
                    $stmtDeposit->close();
                }

                $stmtTransfer = $conn->prepare("
                    SELECT COUNT(*) AS has_transfer
                    FROM cashier_shift_transfer cst
                    INNER JOIN cash_transfer ct ON cst.transfer_id = ct.id
                    WHERE cst.shift_id = ? AND ct.status = 'received'
                ");
                if ($stmtTransfer) {
                    $stmtTransfer->bind_param("i", $shift_id);
                    $stmtTransfer->execute();
                    $transferResult = $stmtTransfer->get_result()->fetch_assoc();
                    $is_transferred = (int)($transferResult['has_transfer'] ?? 0) > 0;
                    $stmtTransfer->close();
                }
            }

            $available_amount = $end_balance;
            if ($has_yesno_columns && isset($shift['counted_amount']) && $shift['counted_amount'] !== null) {
                $available_amount = (float)$shift['counted_amount'];
            }

            if ($available_amount > 0 && !$is_deposited && !$is_transferred) {
                $available_from_closed_shifts += $available_amount;
            }
        }
    }

    $current_balance = $available_from_closed_shifts + $current_balance_active;
    if ($current_balance < 0) {
        $current_balance = 0;
    }

    // TCPDF availability check
    $tcpdf_path = __DIR__ . '/../../libs/tcpdf/tcpdf.php';
    $composer_autoload = __DIR__ . '/../../vendor/autoload.php';
    $tcpdf_available = false;
    if (file_exists($composer_autoload)) {
        require_once $composer_autoload;
        if (class_exists('TCPDF')) $tcpdf_available = true;
    } elseif (file_exists($tcpdf_path)) {
        require_once $tcpdf_path;
        if (class_exists('TCPDF')) $tcpdf_available = true;
    }

    $errors = [];
    $success = '';

    // =========================
    // Payment submission handle
    // =========================
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pay_requisition'])) {
        $requisition_id   = filter_input(INPUT_POST, 'requisition_id', FILTER_VALIDATE_INT);
        // Payment method is always cash for requisition payments
        $payment_method   = 'cash';
        $payment_comments = trim(filter_input(INPUT_POST, 'payment_comments', FILTER_UNSAFE_RAW) ?? '');

        if (!$requisition_id)   { $errors[] = 'Réquisition invalide sélectionnée.'; }
        if ($payment_comments === '') { $errors[] = 'Veuillez fournir des commentaires de paiement.'; }

        // Re-verify active shift
        $stmt = $conn->prepare("SELECT id, shift_start, beginning_balance, deposits, end_balance FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $shift_check = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if (!$shift_check) {
            $errors[] = 'Aucune session active trouvée. Veuillez démarrer une session avant de payer les réquisitions.';
        }

        if (empty($errors)) {
            // Fetch requisition for this province and ensure approved
            $stmt = $conn->prepare('
                SELECT r.id, r.total_amount, r.status, r.province_id, r.recipient_id,
                       r.recipient_type, r.recipient_external_name, r.recipient_external_phone,
                       r.created_at, u.full_name AS initiator_name, recip.full_name AS recipient_name, r.comments
                FROM requisition r
                LEFT JOIN user u ON r.user_id = u.id
                LEFT JOIN user recip ON r.recipient_id = recip.id
                WHERE r.id = ? AND r.status = "approved"
            ');
            $stmt->bind_param('i', $requisition_id);
            $stmt->execute();
            $requisition = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            if (!$requisition) {
                $errors[] = 'Réquisition introuvable ou non approuvée.';
            } else {
                $total_amount = (float)($requisition['total_amount'] ?? 0);
                
                // Recalculate current balance to ensure we have enough
                $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
                $stmt->bind_param("is", $_SESSION['user_id'], $shift_check['shift_start']);
                $stmt->execute();
                $expenses_check = (float)($stmt->get_result()->fetch_assoc()['total_expenses'] ?? 0);
                $stmt->close();
                
                $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transfers FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
                $stmt->bind_param("is", $_SESSION['user_id'], $shift_check['shift_start']);
                $stmt->execute();
                $transfers_check = (float)($stmt->get_result()->fetch_assoc()['total_transfers'] ?? 0);
                $stmt->close();
                
                $available_balance = $current_balance;
                if ($available_balance < $total_amount) {
                    $errors[] = sprintf('Solde caissier insuffisant. Disponible: %s BIF, Requis: %s BIF', 
                        number_format($available_balance, 0, '.', ','), 
                        number_format($total_amount, 0, '.', ','));
                } else {
                    // Requisition items
                    $stmt = $conn->prepare('SELECT ri.category, ri.description, ri.quantity, ri.unit_price 
                                            FROM requisition_item ri WHERE ri.requisition_id = ?');
                    $stmt->bind_param('i', $requisition_id);
                    $stmt->execute();
                    $items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                    $stmt->close();

                    // Generate unique receipt number
                    $receipt_number = '';
                    do {
                        $receipt_number = 'REQ-' . date('Ymd') . '-' . mt_rand(1000, 9999);
                        $check_stmt = $conn->prepare('SELECT id FROM expense WHERE receipt_number = ? LIMIT 1');
                        $check_stmt->bind_param('s', $receipt_number);
                        $check_stmt->execute();
                        $exists = $check_stmt->get_result()->num_rows > 0;
                        $check_stmt->close();
                    } while ($exists);

                    // Ensure AUTO_INCREMENT is set on expense.id and province_cash_transaction.id
                    // Using a direct approach: ALTER TABLE is idempotent, so we can try it directly
                    // This must be done BEFORE the transaction since ALTER TABLE causes implicit commits
                    $tables_to_fix = ['expense', 'province_cash_transaction'];
                    foreach ($tables_to_fix as $table_name) {
                        $alter_sql = "ALTER TABLE `{$table_name}` MODIFY COLUMN `id` INT(11) NOT NULL AUTO_INCREMENT";
                        if (!$conn->query($alter_sql)) {
                            // If it fails, it might already have AUTO_INCREMENT or there's a real error
                            // Log it but don't fail - the INSERT will fail with a clearer error if needed
                            error_log("WARNING: Could not set AUTO_INCREMENT on {$table_name}.id: " . $conn->error . " (This may be OK if already set)");
                        } else {
                            error_log("Successfully ensured AUTO_INCREMENT on {$table_name}.id");
                        }
                    }
                    
                    $conn->begin_transaction();
                    try {
                        // 1) Insert expense (proof of payment) - this automatically affects cashier balance
                        $receiver_user_id = null;
                        $receiver_external_name = null;
                        $receiver_external_phone = null;
                        if (($requisition['recipient_type'] ?? 'internal') === 'external') {
                            $receiver_external_name = $requisition['recipient_external_name'] ?? '';
                            $receiver_external_phone = $requisition['recipient_external_phone'] ?? '';
                        } else {
                            $receiver_user_id = (int)($requisition['recipient_id'] ?? 0) ?: null;
                        }

                        $stmt = $conn->prepare('
                            INSERT INTO expense 
                                (requisition_id, cashier_id, amount, receipt_number, recipient_signature, accountant_id, approval_status, payment_status, payment_method, receiver_user_id, receiver_external_name, receiver_external_phone, created_at) 
                            VALUES (?, ?, ?, ?, ?, ?, "approved", "processed", ?, ?, ?, ?, NOW())
                        ');
                        if (!$stmt) {
                            throw new Exception("Query preparation failed for expense insert: " . $conn->error);
                        }
                        $stmt->bind_param('iidssisiss',
                            $requisition_id,
                            $_SESSION['user_id'], // cashier_id
                            $total_amount,
                            $receipt_number,
                            $payment_comments,
                            $_SESSION['user_id'], // accountant_id (same as cashier for requisition payments)
                            $payment_method,
                            $receiver_user_id,
                            $receiver_external_name,
                            $receiver_external_phone
                        );
                        if (!$stmt->execute()) {
                            error_log("ERROR: Failed to execute expense insert: " . $stmt->error);
                            error_log("ERROR: SQL State: " . $stmt->sqlstate);
                            throw new Exception("Failed to insert expense record: " . $stmt->error);
                        }
                        $expense_id = $stmt->insert_id;
                        if (!$expense_id) {
                            error_log("ERROR: No insert_id returned after expense insert");
                            throw new Exception("Failed to get expense ID after insert.");
                        }
                        $stmt->close();
                        
                        // Enregistrer dans cashier_account
                        // Ensure cashier_account table exists
                        $createCashierAccountSql = "
                            CREATE TABLE IF NOT EXISTS cashier_account (
                                id INT AUTO_INCREMENT PRIMARY KEY,
                                cashier_id INT NOT NULL,
                                shift_id INT NULL,
                                transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                                direction ENUM('CREDIT','DEBIT') NOT NULL,
                                amount BIGINT(20) NOT NULL,
                                reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                                reference_id INT NULL,
                                description VARCHAR(255) DEFAULT NULL,
                                created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                                created_by INT NULL,
                                INDEX idx_cashier_created (cashier_id, created_at),
                                INDEX idx_shift (shift_id),
                                INDEX idx_reference (reference_type, reference_id),
                                CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                                CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                                CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                        ";
                        @$conn->query($createCashierAccountSql);
                        
                        // Get shift_id
                        $stmtShift = $conn->prepare("SELECT id FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL LIMIT 1");
                        $stmtShift->bind_param("i", $_SESSION['user_id']);
                        $stmtShift->execute();
                        $shiftResult = $stmtShift->get_result();
                        $shiftRow = $shiftResult ? $shiftResult->fetch_assoc() : null;
                        $shift_id = $shiftRow ? (int)$shiftRow['id'] : null;
                        $stmtShift->close();
                        
                        if ($shift_id) {
                            $description = "Paiement réquisition #{$requisition_id}";
                            $stmtAccount = $conn->prepare("
                                INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                                VALUES (?, ?, 'REQUISITION_PAYMENT', 'DEBIT', ?, 'REQUISITION', ?, ?, ?)
                            ");
                            if ($stmtAccount) {
                                $stmtAccount->bind_param("iidiss", $_SESSION['user_id'], $shift_id, $total_amount, $requisition_id, $_SESSION['user_id'], $description);
                                if (!$stmtAccount->execute()) {
                                    error_log("Failed to record REQUISITION_PAYMENT in cashier_account: " . $stmtAccount->error);
                                }
                                $stmtAccount->close();
                            }
                        }

                        // 2) Mark requisition as paid
                        $stmt = $conn->prepare('UPDATE requisition SET status = "paid" WHERE id = ?');
                        $stmt->bind_param('i', $requisition_id);
                        $stmt->execute();
                        if ($stmt->affected_rows === 0) {
                            throw new Exception('Failed to mark requisition as paid.');
                        }
                        $stmt->close();

                        // 3) Province cash: DEBIT & ledger write (for accounting purposes)
                        adjustProvinceCash(
                            $conn,
                            $province_id,
                            $total_amount,
                            'REQUISITION_PAYMENT',
                            'REQUISITION',
                            $requisition_id,
                            intval($_SESSION['user_id']),
                            'Requisition Payment ID ' . $requisition_id . ' (Receipt ' . $receipt_number . ')'
                        );

                        // 4) Log action
                        if (($requisition['recipient_type'] ?? 'internal') === 'external') {
                            $externalName = trim((string)($requisition['recipient_external_name'] ?? ''));
                            $externalPhone = trim((string)($requisition['recipient_external_phone'] ?? ''));
                            $recipientSummary = 'external recipient ' . ($externalName !== '' ? $externalName : 'unknown');
                            if ($externalPhone !== '') {
                                $recipientSummary .= ' (' . $externalPhone . ')';
                            }
                        } else {
                            $internalName = trim((string)($requisition['recipient_name'] ?? ''));
                            $recipientSummary = 'internal recipient ' . ($internalName !== '' ? $internalName : 'unknown');
                        }
                        log_action(
                            $_SESSION['user_id'],
                            'pay_requisition',
                            'Paid requisition ID ' . $requisition_id . ' for ' . $total_amount . ' BIF with receipt ' . $receipt_number . ' to ' . $recipientSummary
                        );

                        // NOTE: No notifications to accountants - they perform the action themselves
                        // Notifications are only for amount adjustments on approval

                        // Store receipt number in session for success message after PDF download
                        $_SESSION['last_payment_receipt'] = $receipt_number;
                        $_SESSION['last_payment_requisition_id'] = $requisition_id;
                        
                        // 5) PDF receipt (optional)
                        if ($tcpdf_available) {
                            // Fetch shared company info with depot name
                            $company_info = mbh_get_company_info_for_receipt($conn, $province_id);

                            $pdf = new TCPDF();
                            $pdf->SetCreator('Masunzu Bar Hotel');
                            $pdf->SetAuthor($_SESSION['full_name'] ?? 'Caissier_Comptable');
                            $pdf->SetTitle('Requisition Payment Receipt ' . $receipt_number);
                            $pdf->AddPage();
                            $pdf->SetFont('helvetica', '', 12);

                            $html = '<h1 style="color:#4B2F1F;">Reçu Paiement Réquisition</h1>';
                            $html .= '<p><strong>Numéro Reçu:</strong> ' . htmlspecialchars($receipt_number, ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Date:</strong> ' . date('Y-m-d H:i') . '</p>';

                            $html .= '<h3 style="color:#4B2F1F;">Informations Entreprise</h3>';
                            $companyNameWithDepot = htmlspecialchars($company_info['company_name'] ?? 'Masunzu Bar Hotel', ENT_QUOTES, 'UTF-8');
                            if (!empty($company_info['depot_name'])) {
                                $companyNameWithDepot .= ' - ' . htmlspecialchars($company_info['depot_name'], ENT_QUOTES, 'UTF-8');
                            }
                            $html .= '<p><strong>Nom:</strong> ' . $companyNameWithDepot . '</p>';
                            $html .= '<p><strong>Adresse:</strong> ' . htmlspecialchars($company_info['address'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Téléphone 1:</strong> ' . htmlspecialchars($company_info['phone1'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Téléphone 2:</strong> ' . htmlspecialchars($company_info['phone2'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Email:</strong> ' . htmlspecialchars($company_info['email'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>NIF:</strong> ' . htmlspecialchars($company_info['nif'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>RC:</strong> ' . htmlspecialchars($company_info['rc'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';

                            $html .= '<h3 style="color:#4B2F1F;">Détails Réquisition</h3>';
                            $html .= '<p><strong>ID Réquisition:</strong> ' . $requisition['id'] . '</p>';
                            $html .= '<p><strong>Initiateur:</strong> ' . htmlspecialchars($requisition['initiator_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Créé Le:</strong> ' . htmlspecialchars($requisition['created_at'], ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Commentaires:</strong> ' . htmlspecialchars($requisition['comments'] ?? 'Aucun', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Caissier:</strong> ' . htmlspecialchars($_SESSION['full_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Méthode Paiement:</strong> ' . htmlspecialchars(ucfirst($payment_method), ENT_QUOTES, 'UTF-8') . '</p>';
                            $html .= '<p><strong>Commentaires Paiement:</strong> ' . htmlspecialchars($payment_comments, ENT_QUOTES, 'UTF-8') . '</p>';

                            $html .= '<h3 style="color:#4B2F1F;">Articles Réquisition</h3>';
                            $html .= '<table border="1" cellpadding="5"><tr style="background-color:#4B2F1F;color:#F4F0E4;"><th>Catégorie</th><th>Description</th><th>Quantité</th><th>Prix Unitaire (BIF)</th><th>Total (BIF)</th></tr>';
                            foreach ($items as $item) {
                                $item_total = (float)$item['quantity'] * (float)$item['unit_price'];
                                $html .= '<tr><td>' . htmlspecialchars(ucwords(str_replace('_', ' ', $item['category'])), ENT_QUOTES, 'UTF-8') . '</td><td>' . htmlspecialchars($item['description'], ENT_QUOTES, 'UTF-8') . '</td><td>' . htmlspecialchars($item['quantity'], ENT_QUOTES, 'UTF-8') . '</td><td>' . number_format((float)$item['unit_price'], 2, '.', ',') . '</td><td>' . number_format($item_total, 2, '.', ',') . '</td></tr>';
                            }
                            $html .= '<tr><td colspan="4" style="text-align:right;"><strong>Total Payé</strong></td><td>' . number_format($total_amount, 2, '.', ',') . '</td></tr>';
                            $html .= '</table>';

                            $pdf->writeHTML($html);
                            $conn->commit(); // commit before output
                            
                            // Clear any output that might have been buffered
                            ob_end_clean();
                            
                            // Output PDF for download
                            // Note: After PDF output, we can't redirect server-side (headers already sent)
                            // JavaScript will handle page reload after download
                            $pdf->Output('requisition_receipt_' . $receipt_number . '.pdf', 'D');
                            exit;
                        }

                        // If no PDF or after rendering completes
                        $conn->commit();
                        $success = 'Réquisition payée avec succès. Numéro de Reçu: ' . $receipt_number . ($tcpdf_available ? ' (Téléchargement démarré)' : ' (PDF indisponible)');

                    } catch (Exception $e) {
                        $conn->rollback();
                        $errors[] = $e->getMessage();
                    }
                }
            }
        }
    }

    // =====================
    // List approved items
    // =====================
    $requisitions = [];
    $department_select = $department_exists ? "COALESCE(r.department, 'N/A') AS department" : "'N/A' AS department";
    $requisition_query = "
        SELECT r.id, r.total_amount, r.created_at, r.recipient_id, r.recipient_type,
               r.recipient_external_name, r.recipient_external_phone, r.comments, {$department_select}, 
               r.approved_by, u.full_name AS initiator_name, u2.full_name AS recipient_name, 
               approver.full_name AS approver_name
        FROM requisition r
        LEFT JOIN user u ON r.user_id = u.id
        LEFT JOIN user u2 ON r.recipient_id = u2.id
        LEFT JOIN user approver ON r.approved_by = approver.id
        LEFT JOIN expense e ON r.id = e.requisition_id
        WHERE r.status = 'approved' AND e.id IS NULL
        ORDER BY r.created_at DESC";
    $stmt = $conn->prepare($requisition_query);
    if (!$stmt) {
        error_log("Failed to prepare requisition query: " . $conn->error);
        throw new Exception("Failed to prepare requisition query: " . $conn->error);
    }
    $stmt->execute();
    $requisitions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    
    // Debug logging
    error_log("Caissier_Comptable pay_requisitions.php - Province ID: $province_id, Found requisitions: " . count($requisitions));
    
    // Additional debug: Check all approved requisitions (any province) to verify approval is working
    $debug_stmt = $conn->prepare("SELECT id, status, province_id, approved_by FROM requisition WHERE status = 'approved' ORDER BY id DESC LIMIT 5");
    if ($debug_stmt) {
        $debug_stmt->execute();
        $debug_results = $debug_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        error_log("Debug - All approved requisitions: " . json_encode($debug_results));
        $debug_stmt->close();
    }

    // Requisition items for displayed list
    $requisition_items = [];
    if (!empty($requisitions)) {
        $ids = array_column($requisitions, 'id');
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $types = str_repeat('i', count($ids));
        $item_stmt = $conn->prepare('SELECT requisition_id, category, description, quantity, unit_price FROM requisition_item WHERE requisition_id IN (' . $placeholders . ')');
        $item_stmt->bind_param($types, ...$ids);
        $item_stmt->execute();
        $item_results = $item_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $item_stmt->close();
        foreach ($item_results as $row) {
            $requisition_items[$row['requisition_id']][] = $row;
        }
    }

    // Fetch recent paid requisitions (latest 10, searchable)
    $paid_search = isset($_GET['paid_search']) ? trim($_GET['paid_search']) : '';
    $paid_requisitions = [];
    $paid_query = "
        SELECT r.id, r.total_amount, r.created_at, r.recipient_id, r.recipient_type,
               r.recipient_external_name, r.recipient_external_phone, r.comments,
               u.full_name AS initiator_name, u2.full_name AS recipient_name,
               e.receipt_number, e.created_at AS paid_at, e.cashier_id,
               cashier.full_name AS cashier_name
        FROM requisition r
        LEFT JOIN user u ON r.user_id = u.id
        LEFT JOIN user u2 ON r.recipient_id = u2.id
        LEFT JOIN expense e ON r.id = e.requisition_id
        LEFT JOIN user cashier ON e.cashier_id = cashier.id
        WHERE r.status = 'paid' AND e.id IS NOT NULL
    ";
    $paid_params = [];
    $paid_param_types = '';
    
    if ($paid_search !== '') {
        $paid_query .= " AND (r.id LIKE ? OR u.full_name LIKE ? OR u2.full_name LIKE ? OR e.receipt_number LIKE ? OR r.recipient_external_name LIKE ?)";
        $search_term = '%' . $paid_search . '%';
        $paid_params[] = $search_term;
        $paid_params[] = $search_term;
        $paid_params[] = $search_term;
        $paid_params[] = $search_term;
        $paid_params[] = $search_term;
        $paid_param_types .= 'sssss';
    }
    
    $paid_query .= " ORDER BY e.created_at DESC LIMIT 10";
    
    $paid_stmt = $conn->prepare($paid_query);
    if ($paid_stmt) {
        if ($paid_param_types !== '') {
            $paid_stmt->bind_param($paid_param_types, ...$paid_params);
        }
        $paid_stmt->execute();
        $paid_requisitions = $paid_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $paid_stmt->close();
    }

    $conn->close();
    
    // Include header.php after POST processing (PDF generation exits before this point)
    // Clear output buffer before including header to ensure no accidental output
    if (ob_get_level() > 0) {
        ob_end_clean();
    }
    require_once __DIR__ . '/../../includes/header.php';

} catch (Exception $e) {
    error_log('ERROR in cashiers/pay_requisitions.php: ' . $e->getMessage());
    // Don't include header if we're in the middle of PDF generation
    if (!isset($_POST['pay_requisition'])) {
        // Clear output buffer and include header for error display
        if (ob_get_level() > 0) {
            ob_end_clean();
        }
        require_once __DIR__ . '/../../includes/header.php';
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
        include __DIR__ . '/../../includes/footer.php';
    } else {
        // For PDF generation errors, just output error (no HTML)
        if (ob_get_level() > 0) {
            ob_end_clean();
        }
        echo "Error: " . htmlspecialchars($e->getMessage());
    }
    exit;
}

?>
<style>
    .cashier-page {
        background: linear-gradient(180deg, #f5efe6 0%, #fdf4e8 40%, #f7f2eb 100%);
        padding: 60px 20px 80px;
        min-height: calc(100vh - 70px);
    }
    .cashier-shell {
        background: #fffdf9;
        max-width: 1400px;
        margin: 0 auto;
        border-radius: 32px;
        padding: 40px 45px;
        box-shadow: 0 30px 50px rgba(33, 25, 16, 0.15);
        border: 1px solid rgba(75, 47, 31, 0.08);
    }
    .cashier-shell h2 {
        margin: 0;
        font-size: 30px;
        color: #331e10;
    }
    .cashier-shell p {
        color: #4b2f1f;
        line-height: 1.6;
    }
    .info-card {
        background: #fff;
        border-radius: 16px;
        border-left: 4px solid #4b2f1f;
        padding: 18px 20px;
        margin: 24px 0;
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.05);
    }
    .info-card strong {
        font-size: 16px;
        color: #4b2f1f;
    }
    .balance-amount {
        font-size: 24px;
        font-weight: 700;
        color: #2e7d32;
        margin: 4px 0 0;
        display: block;
    }
    .balance-subtext {
        margin: 6px 0 0;
        color: #6b4f2f;
        font-size: 13px;
    }
    .alert {
        padding: 14px 18px;
        border-radius: 12px;
        font-size: 14px;
        margin-bottom: 20px;
    }
    .alert-success {
        background: #d4edda;
        border: 1px solid #c3e6cb;
        color: #155724;
    }
    .alert-error {
        background: #f8d7da;
        border: 1px solid #f5c6cb;
        color: #721c24;
    }
    .empty-state {
        text-align: center;
        color: #4b2f1f;
        font-size: 16px;
        padding: 20px 0;
        border: 1px dashed rgba(75, 47, 31, 0.24);
        border-radius: 12px;
        margin-bottom: 20px;
    }
    .table-responsive {
        overflow-x: auto;
        margin-bottom: 16px;
    }
    .main-table {
        width: 100%;
        border-collapse: collapse;
        font-size: 14px;
        min-width: 900px;
    }
    .main-table th,
    .main-table td {
        padding: 12px;
        border-bottom: 1px solid #eee3d6;
        text-align: left;
    }
    .main-table th {
        background: #4b2f1f;
        color: #f4f0e4;
        font-weight: 600;
        position: sticky;
        top: 0;
        z-index: 1;
    }
    .main-table tbody tr:nth-child(odd) {
        background: #fff;
    }
    .action-group {
        display: inline-flex;
        gap: 8px;
        flex-wrap: wrap;
    }
    .action-group button {
        border-radius: 6px;
        padding: 10px 16px;
        font-weight: 600;
        border: none;
        cursor: pointer;
        transition: transform 0.2s ease;
    }
    .action-group button.view-details {
        background: #ffffff;
        color: #4b2f1f;
        border: 1px solid #4b2f1f;
    }
    .action-group button.open-payment {
        background: linear-gradient(135deg, #4b2f1f 0%, #6b4f2f 100%);
        color: #f4f0e4;
    }
    .action-group button:disabled {
        cursor: not-allowed;
        opacity: 0.55;
    }
    .recent-paid {
        margin-top: 40px;
        padding-top: 30px;
        border-top: 2px solid #d3c2b6;
    }
    .search-bar {
        display: flex;
        gap: 12px;
        margin-bottom: 20px;
        flex-wrap: wrap;
    }
    .search-bar input {
        flex: 1;
        padding: 10px 14px;
        border-radius: 10px;
        border: 1px solid #4b2f1f;
        font-size: 14px;
    }
    .search-bar button,
    .search-bar a {
        border-radius: 10px;
        padding: 10px 18px;
        font-weight: 600;
        text-decoration: none;
        border: none;
        display: inline-flex;
        align-items: center;
        justify-content: center;
    }
    .search-bar button {
        background: linear-gradient(135deg, #4b2f1f 0%, #5a3f2f 100%);
        color: #f4f0e4;
        cursor: pointer;
    }
    .search-bar a {
        background: #6c757d;
        color: #ffffff;
    }
    .paid-table {
        width: 100%;
        border-collapse: collapse;
        font-size: 14px;
    }
    .paid-table th,
    .paid-table td {
        padding: 12px;
        border-bottom: 1px solid #eee3d6;
    }
    .paid-table th {
        background: #4b2f1f;
        color: #f4f0e4;
    }
    .paid-table tbody tr:nth-child(odd) {
        background: #fff;
    }
    .empty-paid {
        text-align: center;
        padding: 20px 0;
        color: #666;
        font-size: 14px;
    }
</style>
<div class="cashier-page">
    <div class="cashier-shell">
        <div class="page-title">
            <h2>Payer Réquisitions Approuvées</h2>
            <p>Payer les réquisitions approuvées en utilisant le solde de votre session de caissier.</p>
        </div>

        <div class="info-card">
            <p><strong>Solde Caissier Actuel</strong></p>
            <span class="balance-amount"><?php echo number_format($current_balance, 0, '.', ','); ?> BIF</span>
            <p class="balance-subtext">Session démarrée : <?php echo date('Y-m-d H:i', strtotime($shift_start)); ?></p>
        </div>

        <?php 
        $display_success = $success;
        if (isset($_GET['payment_success']) && isset($_SESSION['last_payment_receipt'])) {
            $display_success = 'Réquisition payée avec succès. Numéro de Reçu : ' . htmlspecialchars($_SESSION['last_payment_receipt'], ENT_QUOTES, 'UTF-8') . ' (PDF téléchargé)';
            unset($_SESSION['last_payment_receipt'], $_SESSION['last_payment_requisition_id']);
        }
        ?>
        <?php if (!empty($display_success)): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($display_success, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div class="alert alert-error">
                <?php foreach ($errors as $error): ?>
                    <p style="margin: 0 0 6px;"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if (empty($requisitions)): ?>
            <div class="empty-state">Aucune réquisition approuvée disponible.</div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="main-table">
                    <thead>
                        <tr>
                            <th>Réquisition</th>
                            <th>Initiateur</th>
                            <th>Bénéficiaire</th>
                            <th>Approbateur</th>
                            <th>Total</th>
                            <th>Créé</th>
                            <th>Articles</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($requisitions as $requisition): ?>
                            <?php
                                $items_for_requisition = $requisition_items[$requisition['id']] ?? [];
                                $detail_items = [];
                                foreach ($items_for_requisition as $item) {
                                    $detail_items[] = [
                                        'category'   => $item['category'] ?? '',
                                        'description'=> $item['description'] ?? '',
                                        'quantity'   => $item['quantity'] ?? 0,
                                        'unit_price' => $item['unit_price'] ?? 0,
                                        'total'      => (float)($item['quantity'] ?? 0) * (float)($item['unit_price'] ?? 0)
                                    ];
                                }
                                $recipient_display = ($requisition['recipient_type'] ?? 'internal') === 'external'
                                    ? trim(($requisition['recipient_external_name'] ?? '') . (($requisition['recipient_external_phone'] ?? '') !== '' ? ' (' . $requisition['recipient_external_phone'] . ')' : ''))
                                    : ($requisition['recipient_name'] ?? 'Inconnu');
                                $detail_payload = [
                                    'id'           => (int)($requisition['id'] ?? 0),
                                    'initiator'    => $requisition['initiator_name'] ?? 'Inconnu',
                                    'recipient'    => $recipient_display,
                                    'recipient_type' => $requisition['recipient_type'] ?? 'internal',
                                    'recipient_external_name' => $requisition['recipient_external_name'] ?? '',
                                    'recipient_external_phone' => $requisition['recipient_external_phone'] ?? '',
                                    'recipient_user_id' => (int)($requisition['recipient_id'] ?? 0),
                                    'approver'     => $requisition['approver_name'] ?? 'En Attente',
                                    'department'   => $requisition['department'] ?? 'N/A',
                                    'total_amount' => (float)($requisition['total_amount'] ?? 0),
                                    'created_at'   => $requisition['created_at'] ?? '',
                                    'comments'     => $requisition['comments'] ?? '',
                                    'items'        => $detail_items
                                ];
                                $detail_json = htmlspecialchars(json_encode($detail_payload, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP), ENT_QUOTES, 'UTF-8');
                                $payment_payload = [
                                    'id' => (int)($requisition['id'] ?? 0),
                                    'total_amount' => (float)($requisition['total_amount'] ?? 0),
                                    'recipient_type' => $requisition['recipient_type'] ?? 'internal',
                                    'recipient_id' => (int)($requisition['recipient_id'] ?? 0),
                                    'recipient_display' => $recipient_display,
                                    'recipient_external_name' => $requisition['recipient_external_name'] ?? '',
                                    'recipient_external_phone' => $requisition['recipient_external_phone'] ?? ''
                                ];
                                $payment_json = htmlspecialchars(json_encode($payment_payload, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP), ENT_QUOTES, 'UTF-8');
                                $can_afford = $current_balance >= (float)$requisition['total_amount'];
                            ?>
                            <tr>
                                <td>#<?php echo htmlspecialchars($requisition['id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($requisition['initiator_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($recipient_display, ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($requisition['approver_name'] ?? 'En Attente', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="font-weight: bold; color: #4b2f1f;"><?php echo number_format((float)$requisition['total_amount'], 0, '.', ','); ?> BIF</td>
                                <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($requisition['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                                <td>
                                    <?php if (!empty($items_for_requisition)): ?>
                                        <ul style="margin: 0; padding-left: 18px; color: #4b2f1f;">
                                            <?php foreach ($items_for_requisition as $item): ?>
                                                <li><?php echo htmlspecialchars(ucwords(str_replace('_', ' ', $item['category'])), ENT_QUOTES, 'UTF-8'); ?> - <?php echo htmlspecialchars($item['description'], ENT_QUOTES, 'UTF-8'); ?> (<?php echo htmlspecialchars($item['quantity'], ENT_QUOTES, 'UTF-8'); ?> x <?php echo number_format((float)$item['unit_price'], 0, '.', ','); ?>)</li>
                                            <?php endforeach; ?>
                                        </ul>
                                    <?php else: ?>
                                        <span style="color: #999;">Aucun article</span>
                                    <?php endif; ?>
                                </td>
                                <td style="text-align: right;">
                                    <div class="action-group">
                                        <button data-details="<?php echo $detail_json; ?>" class="view-details">Voir</button>
                                        <button data-payment="<?php echo $payment_json; ?>" class="open-payment" <?php echo !$can_afford ? 'disabled title="Solde insuffisant"' : ''; ?>>Payer</button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

        <section class="recent-paid">
            <h3>Réquisitions Payées Récentes</h3>
            <form method="GET" class="search-bar">
                <input type="text" name="paid_search" placeholder="Rechercher par ID, nom ou numéro de reçu..." value="<?php echo htmlspecialchars($paid_search, ENT_QUOTES, 'UTF-8'); ?>">
                <button type="submit">Rechercher</button>
                <?php if ($paid_search !== ''): ?>
                    <a href="/masunzu_bar_hotel/modules/cashiers/pay_requisitions.php">Effacer</a>
                <?php endif; ?>
            </form>
            <?php if (empty($paid_requisitions)): ?>
                <div class="empty-paid">Aucune réquisition payée trouvée<?php echo $paid_search !== '' ? ' correspondant à votre recherche.' : '.'; ?></div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="paid-table">
                        <thead>
                            <tr>
                                <th>ID Réquisition</th>
                                <th>Numéro Reçu</th>
                                <th>Initiateur</th>
                                <th>Bénéficiaire</th>
                                <th>Montant</th>
                                <th>Payé Par</th>
                                <th>Payé Le</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($paid_requisitions as $paid_req): ?>
                                <?php
                                    $paid_recipient_display = ($paid_req['recipient_type'] ?? 'internal') === 'external'
                                        ? trim(($paid_req['recipient_external_name'] ?? '') . (($paid_req['recipient_external_phone'] ?? '') !== '' ? ' (' . $paid_req['recipient_external_phone'] . ')' : ''))
                                        : ($paid_req['recipient_name'] ?? 'Inconnu');
                                ?>
                                <tr>
                                    <td>#<?php echo htmlspecialchars($paid_req['id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="font-weight: 600; color: #2e7d32;"><?php echo htmlspecialchars($paid_req['receipt_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td><?php echo htmlspecialchars($paid_req['initiator_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td><?php echo htmlspecialchars($paid_recipient_display, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="font-weight: bold; color: #4b2f1f;"><?php echo number_format((float)($paid_req['total_amount'] ?? 0), 0, '.', ','); ?> BIF</td>
                                    <td><?php echo htmlspecialchars($paid_req['cashier_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($paid_req['paid_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php if (count($paid_requisitions) >= 10): ?>
                    <p style="color: #666; font-size: 12px; margin-top: 10px; text-align: center;">Affichage des 10 dernières réquisitions payées. Utilisez la recherche pour en trouver plus.</p>
                <?php endif; ?>
            <?php endif; ?>
        </section>
    </div>
</div>

<!-- Payment Modal -->
<div id="payment-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); backdrop-filter: blur(4px); z-index: 1000;">
    <div style="background: #F4F0E4; border-radius: 12px; max-width: 500px; width: 90%; margin: 60px auto; padding: 20px; position: relative; box-shadow: 0 20px 60px rgba(0,0,0,0.3);">
        <button id="close-modal" style="position: absolute; top: 10px; right: 10px; background: none; border: none; font-size: 24px; color: #4B2F1F; cursor: pointer;">×</button>
        <h3 style="color: #4B2F1F; margin-bottom: 15px;">Payer Réquisition</h3>
        <form method="POST" id="payment-form">
            <input type="hidden" name="pay_requisition" value="1">
            <input type="hidden" name="requisition_id" id="modal-requisition-id" value="">
            <div style="background-color:#FFFFFF; border-radius:8px; border:1px solid rgba(75,47,31,0.15); padding:12px; margin-bottom:15px; color:#2C1810;">
                <p style="margin:0 0 6px;"><strong>Réquisition:</strong> <span id="modal-summary-id">#-</span></p>
                <p style="margin:0 0 6px;"><strong>Montant Total:</strong> <span id="modal-total-amount">-</span></p>
                <p style="margin:0;"><strong>Bénéficiaire Désigné:</strong> <span id="modal-recipient-display">-</span></p>
            </div>
            <div style="margin-bottom: 12px;">
                <label style="color: #4B2F1F; font-weight: 600;">Méthode de Paiement</label>
                <div style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F; background-color: #F4F0E4; color: #4B2F1F; font-weight: 600;">
                    Espèces (Seule méthode pour réquisitions)
                </div>
                <input type="hidden" name="payment_method" value="cash">
            </div>
            <div style="margin-bottom: 12px;">
                <label for="payment_comments" style="color: #4B2F1F; font-weight: 600;">Commentaires de Paiement</label>
                <textarea name="payment_comments" id="payment_comments" rows="3" required style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F; resize: vertical;"></textarea>
            </div>
            <button type="submit" style="background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%); color: #FFFFFF; border: none; padding: 12px 20px; border-radius: 6px; cursor: pointer; font-weight: 600; width: 100%;">Confirmer Paiement</button>
        </form>
    </div>
</div>

<!-- Detail Modal -->
<div id="detail-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.55); backdrop-filter: blur(4px); z-index: 1000;">
    <div style="background: #F4F0E4; border-radius: 12px; max-width: 650px; width: 92%; margin: 60px auto; padding: 24px; position: relative; box-shadow: 0 20px 60px rgba(0,0,0,0.35);">
        <button id="close-detail" style="position: absolute; top: 12px; right: 12px; background: none; border: none; font-size: 24px; color: #4B2F1F; cursor: pointer;">×</button>
        <h3 style="color: #4B2F1F; margin-bottom: 18px;">Détails Réquisition</h3>
        <div id="detail-meta" style="display: flex; flex-direction: column; gap: 6px; margin-bottom: 18px; color: #4B2F1F;"></div>
        <div style="max-height: 260px; overflow-y: auto; border: 1px solid rgba(75,47,31,0.15); border-radius: 10px;">
            <table style="width: 100%; border-collapse: collapse;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 10px; text-align: left;">Catégorie</th>
                        <th style="padding: 10px; text-align: left;">Description</th>
                        <th style="padding: 10px; text-align: left;">Qté</th>
                        <th style="padding: 10px; text-align: left;">Prix Unitaire</th>
                        <th style="padding: 10px; text-align: left;">Total</th>
                    </tr>
                </thead>
                <tbody id="detail-items-body">
                    <tr>
                        <td colspan="5" style="padding: 12px; text-align: center; color: #4B2F1F;">Aucun article à afficher.</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
const modal = document.getElementById('payment-modal');
const closeModal = document.getElementById('close-modal');
const detailModal = document.getElementById('detail-modal');
const closeDetail = document.getElementById('close-detail');
const requisitionInput = document.getElementById('modal-requisition-id');
const form = document.getElementById('payment-form');
const detailMeta = document.getElementById('detail-meta');
const detailItemsBody = document.getElementById('detail-items-body');
const modalSummaryId = document.getElementById('modal-summary-id');
const modalTotalAmount = document.getElementById('modal-total-amount');
const modalRecipientDisplay = document.getElementById('modal-recipient-display');

// Handle form submission - reload page after PDF download
if (form) {
    let formSubmitted = false;
    form.addEventListener('submit', function(e) {
        formSubmitted = true;
        // After form submits, the PDF will download
        // Reload page after a delay to allow PDF download to start
        setTimeout(function() {
            window.location.href = '/masunzu_bar_hotel/modules/cashiers/pay_requisitions.php?payment_success=1';
        }, 2000);
    });
    
    // Also check on page load if we just submitted (fallback)
    if (sessionStorage.getItem('formJustSubmitted') === 'true') {
        sessionStorage.removeItem('formJustSubmitted');
        // Page was reloaded after form submission
        if (!window.location.search.includes('payment_success')) {
            window.location.href = '/masunzu_bar_hotel/modules/cashiers/pay_requisitions.php?payment_success=1';
        }
    }
}

function formatAmount(value) {
    const amount = Number(value || 0);
    return amount.toLocaleString(undefined, { minimumFractionDigits: 0 }) + ' BIF';
}

document.querySelectorAll('.open-payment').forEach(button => {
    button.addEventListener('click', () => {
        if (button.disabled) {
            alert('Insufficient cashier balance to pay this requisition.');
            return;
        }
        const payloadRaw = button.getAttribute('data-payment');
        try {
            const payload = JSON.parse(payloadRaw);
            form.reset();
            requisitionInput.value = payload.id || '';
            modalSummaryId.textContent = payload.id ? `#${payload.id}` : '#-';
            modalTotalAmount.textContent = formatAmount(payload.total_amount);
            modalRecipientDisplay.textContent = payload.recipient_display || 'N/A';
            modal.style.display = 'block';
        } catch (error) {
            console.error('Failed to parse payment payload', error);
            alert('Unable to open payment modal. Please refresh and try again.');
        }
    });
});

document.querySelectorAll('.view-details').forEach(button => {
    button.addEventListener('click', () => {
        const payload = button.getAttribute('data-details');
        try {
            const details = JSON.parse(payload);
            detailMeta.innerHTML = `
                <div><strong>Requisition:</strong> #${details.id}</div>
                <div><strong>Initiator:</strong> ${details.initiator}</div>
                <div><strong>Recipient:</strong> ${details.recipient}</div>
                <div><strong>Approver:</strong> ${details.approver}</div>
                <div><strong>Department:</strong> ${details.department}</div>
                <div><strong>Total:</strong> ${Number(details.total_amount).toLocaleString(undefined, { minimumFractionDigits: 0 })} BIF</div>
                <div><strong>Created:</strong> ${details.created_at}</div>
                <div><strong>Comments:</strong> ${details.comments || 'None'}</div>
            `;
            if (Array.isArray(details.items) && details.items.length > 0) {
                detailItemsBody.innerHTML = details.items.map(item => {
                    return `
                        <tr>
                            <td style="padding: 10px;">${(item.category || '').replace(/_/g, ' ')}</td>
                            <td style="padding: 10px;">${item.description || ''}</td>
                            <td style="padding: 10px;">${item.quantity ?? 0}</td>
                            <td style="padding: 10px;">${Number(item.unit_price || 0).toLocaleString(undefined, { minimumFractionDigits: 0 })} BIF</td>
                            <td style="padding: 10px;">${Number(item.total || 0).toLocaleString(undefined, { minimumFractionDigits: 0 })} BIF</td>
                        </tr>
                    `;
                }).join('');
            } else {
                detailItemsBody.innerHTML = '<tr><td colspan="5" style="padding: 12px; text-align: center; color: #4B2F1F;">No items available.</td></tr>';
            }
            detailModal.style.display = 'block';
        } catch (error) {
            console.error('Failed to parse requisition details', error);
        }
    });
});

closeModal?.addEventListener('click', () => { modal.style.display = 'none'; });
closeDetail?.addEventListener('click', () => { detailModal.style.display = 'none'; });

modal.addEventListener('click', e => { if (e.target === modal) modal.style.display = 'none'; });
detailModal.addEventListener('click', e => { if (e.target === detailModal) detailModal.style.display = 'none'; });
</script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
