<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

if (!in_array($_SESSION['role_name'], ['Caissier_Comptable', 'Accountant'], true)) {
    error_log("Unauthorized access to loan_repayments.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/index.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in loan_repayments.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

if (!isset($_SESSION['province_id'])) {
    $stmtProvince = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    if ($stmtProvince) {
        $stmtProvince->bind_param("i", $_SESSION['user_id']);
        $stmtProvince->execute();
        $provinceRow = $stmtProvince->get_result()->fetch_assoc();
        $_SESSION['province_id'] = $provinceRow['province_id'] ?? null;
        $stmtProvince->close();
    }
}
$province_id = $_SESSION['province_id'] ?? null;

$conn->query("ALTER TABLE loan_payment MODIFY payment_method enum('cash','bank_transfer','mobile','bank_deposit') DEFAULT NULL");

$active_shift_id = null;
$has_active_shift = true;
$activeShiftRow = null;
$cashier_closed_cash = 0.0;
$cashier_active_cash = 0.0;
$cashier_available_total = 0.0;
$shift_start = null;

if ($_SESSION['role_name'] === 'Caissier_Comptable') {
    $stmtActiveShiftStatus = $conn->prepare("SELECT id, shift_start, beginning_balance, deposits, end_balance FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
    if ($stmtActiveShiftStatus) {
        $stmtActiveShiftStatus->bind_param("i", $_SESSION['user_id']);
        $stmtActiveShiftStatus->execute();
        $activeShiftRow = $stmtActiveShiftStatus->get_result()->fetch_assoc();
        $stmtActiveShiftStatus->close();
        if (!empty($activeShiftRow['id'])) {
            $active_shift_id = (int)$activeShiftRow['id'];
            $shift_start = $activeShiftRow['shift_start'] ?? null;
        }
    }
    $has_active_shift = $active_shift_id !== null;

    // Cash from active shift (same logic as cashier_dashboard)
    if ($shift_start) {
        $beginning_balance = (float)($activeShiftRow['beginning_balance'] ?? 0);
        $shift_deposits = (float)($activeShiftRow['deposits'] ?? 0);

        $shift_expenses = 0.0;
        $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
        if ($stmt) {
            $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
            $stmt->execute();
            $shift_expenses = (float)($stmt->get_result()->fetch_assoc()['total_expenses'] ?? 0);
            $stmt->close();
        }

        $shift_transfers = 0.0;
        $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transfers FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
        if ($stmt) {
            $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
            $stmt->execute();
            $shift_transfers = (float)($stmt->get_result()->fetch_assoc()['total_transfers'] ?? 0);
            $stmt->close();
        }

        $cashier_active_cash = $beginning_balance + $shift_deposits - $shift_expenses - $shift_transfers;
        if ($cashier_active_cash < 0) {
            $cashier_active_cash = 0;
        }
    }

    // Cash from closed shifts not deposited/transferred
    $has_yesno_columns = false;
    $checkColumns = $conn->query("
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
          AND TABLE_NAME = 'cashier_shift_closure' 
          AND COLUMN_NAME = 'deposited_to_bank'
    ");
    if ($checkColumns && $checkColumns->num_rows > 0) {
        $has_yesno_columns = true;
        $checkColumns->close();
    }

    if ($has_yesno_columns) {
        $stmtClosed = $conn->prepare("
            SELECT 
                cb.id,
                cb.shift_end,
                cb.beginning_balance,
                cb.end_balance,
                cb.deposits,
                cb.withdrawals,
                COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
                COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
                csc.counted_amount
            FROM cashier_balance cb
            LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
            WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
            ORDER BY cb.shift_end DESC
        ");
    } else {
        $stmtClosed = $conn->prepare("
            SELECT 
                cb.id,
                cb.shift_end,
                cb.beginning_balance,
                cb.end_balance,
                cb.deposits,
                cb.withdrawals
            FROM cashier_balance cb
            WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
            ORDER BY cb.shift_end DESC
        ");
    }

    if ($stmtClosed) {
        $stmtClosed->bind_param("i", $_SESSION['user_id']);
        $stmtClosed->execute();
        $closed_shifts = $stmtClosed->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtClosed->close();

        foreach ($closed_shifts as $shiftRow) {
            $shift_id = (int)$shiftRow['id'];
            $end_balance = isset($shiftRow['end_balance']) && $shiftRow['end_balance'] !== null
                ? (float)$shiftRow['end_balance']
                : (float)($shiftRow['beginning_balance'] ?? 0) + (float)($shiftRow['deposits'] ?? 0) - (float)($shiftRow['withdrawals'] ?? 0);

            $is_deposited = false;
            $is_transferred = false;

            if ($has_yesno_columns) {
                $is_deposited = strtolower($shiftRow['deposited_to_bank'] ?? 'no') === 'yes';
                $is_transferred = strtolower($shiftRow['transferred_to_accountant'] ?? 'no') === 'yes';
            } else {
                $stmtDeposit = $conn->prepare("
                    SELECT COUNT(*) AS has_deposit
                    FROM cashier_shift_deposit csd
                    INNER JOIN bank_deposit bd ON csd.deposit_id = bd.id
                    WHERE csd.shift_id = ?
                ");
                if ($stmtDeposit) {
                    $stmtDeposit->bind_param("i", $shift_id);
                    $stmtDeposit->execute();
                    $depositResult = $stmtDeposit->get_result()->fetch_assoc();
                    $is_deposited = (int)($depositResult['has_deposit'] ?? 0) > 0;
                    $stmtDeposit->close();
                }

                $stmtTransfer = $conn->prepare("
                    SELECT COUNT(*) AS has_transfer
                    FROM cashier_shift_transfer cst
                    INNER JOIN cash_transfer ct ON cst.transfer_id = ct.id
                    WHERE cst.shift_id = ? AND ct.status = 'received'
                ");
                if ($stmtTransfer) {
                    $stmtTransfer->bind_param("i", $shift_id);
                    $stmtTransfer->execute();
                    $transferResult = $stmtTransfer->get_result()->fetch_assoc();
                    $is_transferred = (int)($transferResult['has_transfer'] ?? 0) > 0;
                    $stmtTransfer->close();
                }
            }

            $available_amount = $end_balance;
            if ($has_yesno_columns && isset($shiftRow['counted_amount']) && $shiftRow['counted_amount'] !== null) {
                $available_amount = (float)$shiftRow['counted_amount'];
            }

            if ($available_amount > 0 && !$is_deposited && !$is_transferred) {
                $cashier_closed_cash += $available_amount;
            }
        }
    }

    $cashier_available_total = $cashier_closed_cash + $cashier_active_cash;
    if ($cashier_available_total < 0) {
        $cashier_available_total = 0;
    }
}

/**
 * Compute outstanding loan balance using both the ledger and cached customer.loan_balance_bif.
 */
function get_customer_outstanding_balance(mysqli $conn, int $customerId): int {
    $stmt = $conn->prepare("
        SELECT 
            GREATEST(
                COALESCE(c.loan_balance_bif, 0),
                COALESCE((
                    SELECT 
                        COALESCE(SUM(CASE WHEN direction = 'debit' THEN amount_bif ELSE 0 END), 0) -
                        COALESCE(SUM(CASE WHEN direction = 'credit' THEN amount_bif ELSE 0 END), 0)
                    FROM customer_loan_ledger
                    WHERE customer_id = c.id
                ), 0),
                COALESCE((
                    SELECT 
                        SUM(loan_outstanding_bif)
                    FROM unpaid_invoices
                    WHERE customer_id = c.id
                      AND is_loan_sale = 1
                      AND loan_outstanding_bif > 0
                      AND (approval_status IS NULL OR approval_status != 'rejected')
                ), 0)
            ) AS outstanding
        FROM customer c
        WHERE c.id = ?
        LIMIT 1
    ");
    if ($stmt) {
        $stmt->bind_param("i", $customerId);
        $stmt->execute();
        $outstanding = (int)($stmt->get_result()->fetch_assoc()['outstanding'] ?? 0);
        $stmt->close();
        return max($outstanding, 0);
    }
    error_log('Failed to compute outstanding balance for customer ' . $customerId . ': ' . $conn->error);
    return 0;
}

mbh_ensure_loan_ledger_entry_origin($conn);

if (!function_exists('generate_uuid')) {
    function generate_uuid(): string {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff)
        );
    }
}

$loan_customers = [];
// Calculate effective balance from customer_loan_ledger (source of truth)
$loanCustomersSql = "
    SELECT 
        c.id,
        c.full_name,
        c.loan_limit_bif,
        c.loan_balance_bif,
        c.loan_terms_days,
        c.loan_status,
        COALESCE(GREATEST(
            COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
            COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
            0
        ), c.loan_balance_bif, 0) AS effective_balance
    FROM customer c
    LEFT JOIN customer_loan_ledger cll ON (
        cll.customer_id = c.id
        AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
    )
    WHERE (c.can_pay_by_loan = 1 OR c.loan_balance_bif > 0)
    GROUP BY c.id, c.full_name, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status
    HAVING effective_balance > 0
    ORDER BY c.full_name
";
$stmtCustomers = $conn->prepare($loanCustomersSql);
if ($stmtCustomers) {
    $stmtCustomers->execute();
    $loan_customers = $stmtCustomers->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtCustomers->close();
}

// Bank accounts for deposit-based repayments
$bank_accounts = [];
$stmtAccounts = $conn->prepare("SELECT id, account_number, currency, is_main FROM bank_account ORDER BY is_main DESC, id ASC");
if ($stmtAccounts) {
    $stmtAccounts->execute();
    $bank_accounts = $stmtAccounts->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtAccounts->close();
}

$selected_customer_id = isset($_GET['customer_id']) ? (int)$_GET['customer_id'] : 0;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $selected_customer_id = isset($_POST['customer_id']) ? (int)$_POST['customer_id'] : 0;
}

$errors = [];
$success = '';
$available_invoices = [];
$customer_profile = null;
// Calculate effective balance from customer_loan_ledger (source of truth)
$customerProfileSql = "
    SELECT 
        c.id,
        c.full_name,
        c.tel,
        c.loan_limit_bif,
        c.loan_balance_bif,
        c.loan_terms_days,
        c.loan_status,
        COALESCE(GREATEST(
            COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
            COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
            0
        ), c.loan_balance_bif, 0) AS effective_balance
    FROM customer c
    LEFT JOIN customer_loan_ledger cll ON (
        cll.customer_id = c.id
        AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
    )
    WHERE c.id = ?
    GROUP BY c.id, c.full_name, c.tel, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status
    LIMIT 1
";

if ($selected_customer_id > 0) {
    // Always fetch fresh customer data, especially after a payment
    $stmtProfile = $conn->prepare($customerProfileSql);
    if ($stmtProfile) {
        $stmtProfile->bind_param("i", $selected_customer_id);
        $stmtProfile->execute();
        $customer_profile = $stmtProfile->get_result()->fetch_assoc();
        $stmtProfile->close();
        
        // Ensure loan_balance_bif is properly cast to integer
        if ($customer_profile) {
            $customer_profile['loan_balance_bif'] = (int)($customer_profile['loan_balance_bif'] ?? 0);
            $effectiveBalance = (int)($customer_profile['effective_balance'] ?? 0);
            $customerOutstanding = $effectiveBalance > 0 ? $effectiveBalance : $customer_profile['loan_balance_bif'];
            $customer_profile['loan_balance_bif'] = $customerOutstanding;
            $customer_profile['outstanding_balance'] = $customerOutstanding;
            $customer_profile['customer_phone'] = $customer_profile['tel'] ?? '';
            $customer_profile['loan_limit_bif'] = (int)($customer_profile['loan_limit_bif'] ?? 0);
            $customer_profile['loan_terms_days'] = (int)($customer_profile['loan_terms_days'] ?? 0);
        }
    }
    if ($customer_profile) {
        // Exclude rejected invoices from available invoices for repayment
        $stmtInvoices = $conn->prepare("SELECT id, invoice_number, loan_outstanding_bif, loan_due_date, created_at FROM unpaid_invoices WHERE customer_id = ? AND is_loan_sale = 1 AND loan_outstanding_bif IS NOT NULL AND loan_outstanding_bif > 0 AND (approval_status IS NULL OR approval_status != 'rejected') ORDER BY created_at");
        if ($stmtInvoices) {
            $stmtInvoices->bind_param("i", $selected_customer_id);
            $stmtInvoices->execute();
            $available_invoices = $stmtInvoices->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtInvoices->close();
        }
        if (!empty($available_invoices)) {
            $invoiceOutstandingTotal = 0;
            foreach ($available_invoices as $idx => $invoice) {
                $amountOutstanding = (int)($invoice['loan_outstanding_bif'] ?? 0);
                $available_invoices[$idx]['loan_outstanding_bif'] = $amountOutstanding;
                $invoiceOutstandingTotal += $amountOutstanding;
            }
            $customer_profile['invoice_outstanding_total'] = $invoiceOutstandingTotal;
        } else {
            $customer_profile['invoice_outstanding_total'] = 0;
        }
        $customer_profile['outstanding_balance'] = max(
            (int)($customer_profile['outstanding_balance'] ?? 0),
            (int)($customer_profile['invoice_outstanding_total'] ?? 0)
        );
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['record_loan_payment'])) {
    $amount = (int)($_POST['amount'] ?? 0);
    $payment_method = trim($_POST['payment_method'] ?? '');
    $invoice_id = isset($_POST['invoice_id']) ? (int)$_POST['invoice_id'] : 0;
    $payment_notes = trim($_POST['notes'] ?? '');
    $activeCaissier_ComptableShiftId = $active_shift_id;

    $bank_account_id = isset($_POST['bank_account_id']) ? (int)$_POST['bank_account_id'] : 0;
    $slip_number = trim($_POST['slip_number'] ?? '');
    $slip_file_path = null;

    if ($_SESSION['role_name'] === 'Caissier_Comptable' && $activeCaissier_ComptableShiftId === null && $payment_method === 'cash') {
        $errors[] = "Aucune session active trouvée. Veuillez ouvrir une session avant d'enregistrer un remboursement en espèces.";
    }

    if ($selected_customer_id <= 0 || !$customer_profile) {
        $errors[] = "Un client valide est requis.";
    }
    if ($amount <= 0) {
        $errors[] = "Le montant du remboursement doit être supérieur à zéro.";
    }
    $valid_methods = ['cash', 'bank_transfer', 'mobile', 'bank_deposit'];
    if (!in_array($payment_method, $valid_methods, true)) {
        $errors[] = "Sélectionnez une méthode de paiement valide.";
    }
    if ($payment_method === 'bank_deposit') {
        if ($bank_account_id <= 0) {
            $errors[] = "Sélectionnez le compte bancaire de destination.";
        }
        if ($slip_number === '') {
            $errors[] = "Le numéro du bordereau bancaire est requis.";
        }
        if (!isset($_FILES['slip_file']) || $_FILES['slip_file']['error'] !== UPLOAD_ERR_OK) {
            $errors[] = "Veuillez téléverser une image du bordereau bancaire.";
        }
    }
    if (mb_strlen($payment_notes) > 500) {
        $errors[] = "Les notes doivent contenir 500 caractères ou moins.";
    }

    $outstanding_balance = $customer_profile ? (int)($customer_profile['outstanding_balance'] ?? $customer_profile['loan_balance_bif']) : 0;
    if ($outstanding_balance <= 0) {
        $errors[] = "Le client sélectionné n'a pas de solde crédit en attente.";
    }
    if ($amount > $outstanding_balance) {
        $errors[] = "Le remboursement ne peut pas dépasser le solde crédit en attente (" . number_format($outstanding_balance, 0, ',', '.') . " BIF).";
    }
    if ($payment_method === 'cash' && $_SESSION['role_name'] === 'Caissier_Comptable') {
        if (!$has_active_shift) {
            $errors[] = "Aucune session active trouvée pour le caissier.";
        }
        if ($amount > $cashier_available_total) {
            $errors[] = "Solde caisse insuffisant. Disponible: " . number_format($cashier_available_total, 0, ',', '.') . " BIF.";
        }
    }

    $invoice_row = null;
    if ($invoice_id > 0) {
        foreach ($available_invoices as $invoice) {
            if ((int)$invoice['id'] === $invoice_id) {
                $invoice_row = $invoice;
                break;
            }
        }
        if (!$invoice_row) {
            $errors[] = "La commande sélectionnée n'est pas éligible pour le remboursement.";
        } elseif ($amount > (int)$invoice_row['loan_outstanding_bif']) {
            $errors[] = "Le remboursement dépasse le montant en attente pour la commande sélectionnée (" . number_format($invoice_row['loan_outstanding_bif'], 0, ',', '.') . " BIF).";
        }
    }

    if ($payment_method === 'bank_deposit' && empty($errors)) {
        $uploadBase = realpath(__DIR__ . '/../../');
        $slipDir = $uploadBase . '/uploads/bank_slips';
        if (!is_dir($slipDir)) {
            mkdir($slipDir, 0755, true);
        }
        $fileInfo = pathinfo($_FILES['slip_file']['name'] ?? '');
        $ext = strtolower($fileInfo['extension'] ?? '');
        $allowedExt = ['jpg','jpeg','png','pdf'];
        if (!in_array($ext, $allowedExt, true)) {
            $errors[] = "Format de bordereau non supporté. Formats autorisés: jpg, jpeg, png, pdf.";
        } else {
            $safeName = 'loan_deposit_' . $_SESSION['user_id'] . '_' . uniqid() . '.' . $ext;
            $destPath = $slipDir . '/' . $safeName;
            if (!move_uploaded_file($_FILES['slip_file']['tmp_name'], $destPath)) {
                $errors[] = "Échec du téléversement du bordereau bancaire.";
            } else {
                $slip_file_path = '/masunzu_bar_hotel/uploads/bank_slips/' . $safeName;
            }
        }
    }

    if (empty($errors)) {
        $conn->begin_transaction();
        try {
            // Calculate effective balance from customer_loan_ledger (source of truth)
            $stmtLockCustomer = $conn->prepare("
                SELECT 
                    c.loan_balance_bif,
                    COALESCE(GREATEST(
                        COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
                        COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
                        0
                    ), c.loan_balance_bif, 0) AS effective_balance
                FROM customer c
                LEFT JOIN customer_loan_ledger cll ON (
                    cll.customer_id = c.id
                    AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
                )
                WHERE c.id = ?
                GROUP BY c.loan_balance_bif
                FOR UPDATE
            ");
            if (!$stmtLockCustomer) {
                throw new Exception("Failed to prepare customer lock statement: " . $conn->error);
            }
            $stmtLockCustomer->bind_param("i", $selected_customer_id);
            if (!$stmtLockCustomer->execute()) {
                throw new Exception("Failed to lock customer row: " . $stmtLockCustomer->error);
            }
            $lockedCustomer = $stmtLockCustomer->get_result()->fetch_assoc();
            $stmtLockCustomer->close();
            if (!$lockedCustomer) {
                throw new Exception("Profil client introuvable.");
            }
            $lockedBalance = (int)($lockedCustomer['loan_balance_bif'] ?? 0);
            $lockedOutstanding = (int)($lockedCustomer['effective_balance'] ?? $lockedBalance);
            if ($lockedOutstanding <= 0) {
                throw new Exception("Le client n'a pas de solde crédit en attente.");
            }
            $amountDueBeforePayment = $lockedOutstanding;
            
            // Validate amount against invoice outstanding or total balance
            if ($invoice_row) {
                // If invoice is selected, check against invoice outstanding
                $invoiceOutstanding = (int)$invoice_row['loan_outstanding_bif'];
                if ($amount > $invoiceOutstanding) {
                    throw new Exception("Le montant du remboursement (" . number_format($amount, 0, ',', '.') . " BIF) dépasse le montant en attente pour la commande " . $invoice_row['invoice_number'] . " (" . number_format($invoiceOutstanding, 0, ',', '.') . " BIF). Maximum autorisé: " . number_format($invoiceOutstanding, 0, ',', '.') . " BIF.");
                }
            } else {
                // If no invoice selected, check against total customer balance
                if ($amount > $lockedOutstanding) {
                    throw new Exception("Le montant du remboursement (" . number_format($amount, 0, ',', '.') . " BIF) dépasse le solde crédit total en attente (" . number_format($lockedOutstanding, 0, ',', '.') . " BIF). Maximum autorisé: " . number_format($lockedOutstanding, 0, ',', '.') . " BIF.");
                }
            }

            $stmtLedger = $conn->prepare("INSERT INTO customer_loan_ledger (customer_id, entry_type, direction, entry_origin, amount_bif, reference_table, reference_id, notes, created_by) VALUES (?, 'loan_repayment', 'credit', 'loan_repayment', ?, ?, ?, ?, ?)");
            if (!$stmtLedger) {
                throw new Exception("Failed to prepare loan ledger insertion: " . $conn->error);
            }
            $referenceTable = $invoice_row ? 'unpaid_invoices' : 'manual';
            $referenceId = $invoice_row ? (int)$invoice_row['id'] : null;
            $amountParam = (string)$amount;
            $notesParam = $payment_notes === '' ? null : mb_substr($payment_notes, 0, 500);
            $stmtLedger->bind_param("issisi", $selected_customer_id, $amountParam, $referenceTable, $referenceId, $notesParam, $_SESSION['user_id']);
            if (!$stmtLedger->execute()) {
                throw new Exception("Failed to insert loan ledger: " . $stmtLedger->error);
            }
            $ledger_id = $conn->insert_id;
            $stmtLedger->close();

            $stmtPayment = $conn->prepare("INSERT INTO loan_payment (customer_id, order_id, payment_amount, payment_date, created_by, province_id, invoice_id, ledger_entry_id, notes, payment_method) VALUES (?, NULL, ?, NOW(), ?, ?, ?, ?, ?, ?)");
            if (!$stmtPayment) {
                throw new Exception("Failed to prepare loan payment insertion: " . $conn->error);
            }
            $provinceIdParam = $province_id ?: null;
            // invoice_id in loan_payment references paid_invoices.id, but we're dealing with unpaid_invoices
            // So we set it to NULL and use ledger_entry_id to track the relationship
            $invoiceIdParam = null; // Always NULL for loan repayments since they reference unpaid_invoices, not paid_invoices
            $methodParam = $payment_method;
            $stmtPayment->bind_param("iiiiiiss", $selected_customer_id, $amountParam, $_SESSION['user_id'], $provinceIdParam, $invoiceIdParam, $ledger_id, $notesParam, $methodParam);
            if (!$stmtPayment->execute()) {
                throw new Exception("Failed to insert loan payment: " . $stmtPayment->error);
            }
            $loanPaymentId = (int)$conn->insert_id;
            $stmtPayment->close();

            $bankDepositId = null;
            $bankTransactionId = null;

            if ($payment_method === 'bank_deposit') {
                $stmtDeposit = $conn->prepare("
                    INSERT INTO bank_deposit (user_id, amount, slip_number, slip_file_path, deposit_date, province_id, bank_account_id)
                    VALUES (?, ?, ?, ?, NOW(), ?, ?)
                ");
                if (!$stmtDeposit) {
                    throw new Exception("Échec de préparation du dépôt bancaire: " . $conn->error);
                }
                $provinceParam = $province_id ?: null;
                $stmtDeposit->bind_param("iissii", $_SESSION['user_id'], $amount, $slip_number, $slip_file_path, $provinceParam, $bank_account_id);
                if (!$stmtDeposit->execute()) {
                    throw new Exception("Échec d'enregistrement du dépôt bancaire: " . $stmtDeposit->error);
                }
                $bankDepositId = (int)$conn->insert_id;
                $stmtDeposit->close();

                $description = "Remboursement crédit (dépôt) - Client ID {$selected_customer_id}";
                $stmtBankTxn = $conn->prepare("
                    INSERT INTO bank_account_transaction (bank_account_id, province_id, transaction_type, direction, amount, reference_type, reference_id, description, created_by)
                    VALUES (?, ?, 'BANK_DEPOSIT', 'CREDIT', ?, 'CASH_DEPOSIT', ?, ?, ?)
                ");
                if (!$stmtBankTxn) {
                    throw new Exception("Échec de préparation de la transaction bancaire: " . $conn->error);
                }
                $stmtBankTxn->bind_param("iiisis", $bank_account_id, $provinceParam, $amount, $bankDepositId, $description, $_SESSION['user_id']);
                if (!$stmtBankTxn->execute()) {
                    throw new Exception("Échec d'enregistrement de la transaction bancaire: " . $stmtBankTxn->error);
                }
                $bankTransactionId = (int)$conn->insert_id;
                $stmtBankTxn->close();
            }

            $newInvoiceOutstanding = null;
            if ($invoice_row) {
                $newInvoiceOutstanding = (int)$invoice_row['loan_outstanding_bif'] - $amount;
                $newInvoiceOutstanding = max(0, $newInvoiceOutstanding);
                $newLoanStatus = $newInvoiceOutstanding === 0 ? 'settled' : 'pending';
                $invoiceStatus = $newInvoiceOutstanding === 0 ? 'paid' : 'pending';

                $stmtUpdateInvoice = $conn->prepare("UPDATE unpaid_invoices SET loan_outstanding_bif = ?, loan_status = ?, status = ? WHERE id = ?");
                if (!$stmtUpdateInvoice) {
                    throw new Exception("Failed to prepare invoice update: " . $conn->error);
                }
                $stmtUpdateInvoice->bind_param("issi", $newInvoiceOutstanding, $newLoanStatus, $invoiceStatus, $invoice_row['id']);
                if (!$stmtUpdateInvoice->execute()) {
                    throw new Exception("Failed to update invoice: " . $stmtUpdateInvoice->error);
                }
                $stmtUpdateInvoice->close();
            }

            $newCustomerOutstanding = max(0, $lockedOutstanding - $amount);
            $stmtUpdateCustomer = $conn->prepare("UPDATE customer SET loan_balance_bif = ?, loan_last_review_at = NOW() WHERE id = ?");
            if (!$stmtUpdateCustomer) {
                throw new Exception("Failed to prepare customer update: " . $conn->error);
            }
            $stmtUpdateCustomer->bind_param("ii", $newCustomerOutstanding, $selected_customer_id);
            if (!$stmtUpdateCustomer->execute()) {
                throw new Exception("Failed to update customer loan balance: " . $stmtUpdateCustomer->error);
            }
            $stmtUpdateCustomer->close();
            
            // Credit cashier balance for loan repayment (cashier receives cash from client)
            $cashierShiftIdForUpdate = null;
            if ($_SESSION['role_name'] === 'Caissier_Comptable' && $activeCaissier_ComptableShiftId !== null) {
                $stmtCaissier_ComptableBalance = $conn->prepare("SELECT id FROM cashier_balance WHERE id = ? AND cashier_id = ? AND shift_end IS NULL FOR UPDATE");
                if ($stmtCaissier_ComptableBalance) {
                    $stmtCaissier_ComptableBalance->bind_param("ii", $activeCaissier_ComptableShiftId, $_SESSION['user_id']);
                    $stmtCaissier_ComptableBalance->execute();
                    $cashierShift = $stmtCaissier_ComptableBalance->get_result()->fetch_assoc();
                    $stmtCaissier_ComptableBalance->close();
                    if ($cashierShift) {
                        $cashierShiftIdForUpdate = (int)$cashierShift['id'];
                    } else {
                        throw new Exception("Votre session active a été clôturée. Veuillez rouvrir une session avant d'enregistrer un remboursement.");
                    }
                }
            }
            
            if ($payment_method === 'cash' && $cashierShiftIdForUpdate !== null) {
                // Ensure cashier_account table exists
                $createCashierAccountSql = "
                    CREATE TABLE IF NOT EXISTS cashier_account (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        cashier_id INT NOT NULL,
                        shift_id INT NULL,
                        transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                        direction ENUM('CREDIT','DEBIT') NOT NULL,
                        amount BIGINT(20) NOT NULL,
                        reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                        reference_id INT NULL,
                        description VARCHAR(255) DEFAULT NULL,
                        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                        created_by INT NULL,
                        INDEX idx_cashier_created (cashier_id, created_at),
                        INDEX idx_shift (shift_id),
                        INDEX idx_reference (reference_type, reference_id),
                        CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                        CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                        CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                ";
                @$conn->query($createCashierAccountSql);
                
                $stmtUpdateCaissier_Comptable = $conn->prepare("UPDATE cashier_balance SET deposits = deposits + ?, end_balance = end_balance + ? WHERE id = ? AND cashier_id = ?");
                if ($stmtUpdateCaissier_Comptable) {
                    $stmtUpdateCaissier_Comptable->bind_param("ddii", $amount, $amount, $cashierShiftIdForUpdate, $_SESSION['user_id']);
                    if ($stmtUpdateCaissier_Comptable->execute()) {
                        error_log("Loan repayment credited to cashier balance: shift_id={$cashierShiftIdForUpdate}, amount=$amount");
                        
                        // Enregistrer dans cashier_account
                        $customerName = $customer_profile['full_name'] ?? 'Client';
                        $description = "Remboursement crédit: {$customerName}";
                        $stmtAccount = $conn->prepare("
                            INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                            VALUES (?, ?, 'LOAN_REPAYMENT', 'CREDIT', ?, 'LOAN_PAYMENT', ?, ?, ?)
                        ");
                        if ($stmtAccount) {
                            $stmtAccount->bind_param("iiiiis", $_SESSION['user_id'], $cashierShiftIdForUpdate, $amount, $loanPaymentId, $_SESSION['user_id'], $description);
                            if (!$stmtAccount->execute()) {
                                error_log("Failed to record LOAN_REPAYMENT in cashier_account: " . $stmtAccount->error);
                            }
                            $stmtAccount->close();
                        }
                    } else {
                        error_log("Failed to update cashier balance for loan repayment: " . $stmtUpdateCaissier_Comptable->error);
                    }
                    $stmtUpdateCaissier_Comptable->close();
                }
            }

            $receipt_id = null;
            $receipt_number = null;
            if (!empty($loanPaymentId)) {
                $receipt_id = generate_uuid();
                $receipt_number = sprintf('LRP-%s-%04d', date('YmdHis'), $loanPaymentId);
                $customerName = $customer_profile['full_name'] ?? 'Client';
                $customerPhone = $customer_profile['customer_phone'] ?? '';
                $receiptDetails = [
                    'customer_id' => $selected_customer_id,
                    'invoice_number' => $invoice_row['invoice_number'] ?? null,
                    'amount_due_before' => $amountDueBeforePayment,
                    'amount_paid' => $amount,
                    'remaining_balance' => $newCustomerOutstanding,
                    'payment_method' => $payment_method,
                    'applied_invoice_id' => $invoice_row['id'] ?? null,
                    'invoice_amount_remaining' => $newInvoiceOutstanding,
                    'full_settlement' => $invoice_row ? ($newInvoiceOutstanding === 0) : ($newCustomerOutstanding === 0)
                ];
                $receiptPayload = json_encode($receiptDetails, JSON_UNESCAPED_UNICODE);
                $stmtReceipt = $conn->prepare("
                    INSERT INTO receipts (
                        receipt_id, transaction_type, transaction_id, receipt_number,
                        customer_name, phone_number, total_amount, subtotal_amount, tax_amount, tax_rate, tax_mode,
                        created_at, created_by, pdf_content
                    ) VALUES (?, 'loan_repayment', ?, ?, ?, ?, ?, ?, 0, 0.00, 'HTVA', NOW(), ?, ?)
                ");
                if ($stmtReceipt) {
                    $stmtReceipt->bind_param(
                        "sisssddis",
                        $receipt_id,
                        $loanPaymentId,
                        $receipt_number,
                        $customerName,
                        $customerPhone,
                        $amount,
                        $amount,
                        $_SESSION['user_id'],
                        $receiptPayload
                    );
                    if (!$stmtReceipt->execute()) {
                        error_log("Failed to insert loan repayment receipt: " . $stmtReceipt->error);
                        $receipt_id = null;
                        $receipt_number = null;
                    }
                    $stmtReceipt->close();
                }
            }
            
            // Notify accountants about the loan repayment (but don't credit their balance)
            if ($province_id) {
                $stmtAccountants = $conn->prepare("SELECT id FROM user WHERE role_id = 9 AND province_id = ?");
                if ($stmtAccountants) {
                    $stmtAccountants->bind_param("i", $province_id);
                    $stmtAccountants->execute();
                    $accountantsResult = $stmtAccountants->get_result();
                    
                    $customerName = $customer_profile ? trim($customer_profile['full_name'] ?? 'Customer') : 'Customer';
                    $invoiceRef = $invoice_row ? ('Commande: ' . $invoice_row['invoice_number']) : 'Remboursement général';
                    $notificationMessage = sprintf(
                        'Loan repayment received: %s BIF from %s (%s)',
                        number_format($amount, 0, ',', '.'),
                        $customerName,
                        $invoiceRef
                    );
                    $notificationLink = '/masunzu_bar_hotel/modules/accounting/loan_management.php?customer_id=' . $selected_customer_id;
                    
                    while ($accountantRow = $accountantsResult->fetch_assoc()) {
                        $accountantId = (int)$accountantRow['id'];
                        $stmtNotification = $conn->prepare('INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())');
                        if ($stmtNotification) {
                            $stmtNotification->bind_param('iss', $accountantId, $notificationMessage, $notificationLink);
                            if (!$stmtNotification->execute()) {
                                error_log('Failed to send loan repayment notification to accountant: ' . $stmtNotification->error);
                            }
                            $stmtNotification->close();
                        }
                    }
                    $stmtAccountants->close();
                }
                
                // Also notify admins (DG, DGA, Admin roles)
                $stmtAdmins = $conn->prepare("SELECT u.id FROM user u JOIN role r ON u.role_id = r.id WHERE r.name IN ('Admin', 'DG', 'DGA')");
                if ($stmtAdmins) {
                    $stmtAdmins->execute();
                    $adminsResult = $stmtAdmins->get_result();
                    
                    $customerName = $customer_profile ? trim($customer_profile['full_name'] ?? 'Customer') : 'Customer';
                    $invoiceRef = $invoice_row ? ('Commande: ' . $invoice_row['invoice_number']) : 'Remboursement général';
                    $notificationMessage = sprintf(
                        'Loan repayment received: %s BIF from %s (%s)',
                        number_format($amount, 0, ',', '.'),
                        $customerName,
                        $invoiceRef
                    );
                    $notificationLink = '/masunzu_bar_hotel/modules/accounting/loan_management.php?customer_id=' . $selected_customer_id;
                    
                    while ($adminRow = $adminsResult->fetch_assoc()) {
                        $adminId = (int)$adminRow['id'];
                        $stmtNotification = $conn->prepare('INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())');
                        if ($stmtNotification) {
                            $stmtNotification->bind_param('iss', $adminId, $notificationMessage, $notificationLink);
                            if (!$stmtNotification->execute()) {
                                error_log('Failed to send loan repayment notification to admin: ' . $stmtNotification->error);
                            }
                            $stmtNotification->close();
                        }
                    }
                    $stmtAdmins->close();
                }
            }

            $logDescription = "Recorded loan repayment of " . number_format($amount, 0, ',', '.') . " BIF for customer ID $selected_customer_id";
            if ($invoice_row) {
                $logDescription .= " against invoice " . $invoice_row['invoice_number'];
            }
            $stmtLog = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'loan_repayment', ?, NOW())");
            if ($stmtLog) {
                $stmtLog->bind_param("is", $_SESSION['user_id'], $logDescription);
                $stmtLog->execute();
                $stmtLog->close();
            }

            $conn->commit();
            $summaryParts = [];
            if ($invoice_row) {
                $summaryParts[] = 'Commande ' . $invoice_row['invoice_number'];
            }
            $summaryParts[] = 'Montant dû: ' . number_format($amountDueBeforePayment, 0, ',', '.') . ' BIF';
            $summaryParts[] = 'Payé: ' . number_format($amount, 0, ',', '.') . ' BIF';
            $summaryParts[] = 'Solde restant: ' . number_format($newCustomerOutstanding, 0, ',', '.') . ' BIF';
            if ($invoice_row && $newInvoiceOutstanding === 0) {
                $summaryParts[] = 'Paiement total de la commande';
            } elseif (!$invoice_row && $newCustomerOutstanding === 0) {
                $summaryParts[] = 'Solde crédit intégralement réglé';
            }
            $successMessage = "Remboursement crédit enregistré (" . implode(' • ', $summaryParts) . ")";
            if ($receipt_number) {
                $successMessage .= " • Reçu #{$receipt_number}";
            }
            $_SESSION['loan_repayment_success'] = $successMessage;
            if ($receipt_id) {
                $_SESSION['loan_repayment_receipt_id'] = $receipt_id;
            }
            header("Location: /masunzu_bar_hotel/modules/cashiers/loan_repayments.php?customer_id=" . $selected_customer_id . "&success=1");
            exit;
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Loan repayment failed: " . $e->getMessage());
            $errors[] = "Échec de l'enregistrement du remboursement: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
        }
    }
}

if (isset($_GET['success'])) {
    if (!empty($_SESSION['loan_repayment_success'])) {
        $success = $_SESSION['loan_repayment_success'];
        unset($_SESSION['loan_repayment_success'], $_SESSION['loan_repayment_receipt_id']);
    } else {
        $success = "Remboursement crédit enregistré avec succès.";
    }
}

$formDisabled = ($_SESSION['role_name'] === 'Caissier_Comptable' && !$has_active_shift);

$recent_payments = [];
if ($selected_customer_id > 0) {
    $stmtRecent = $conn->prepare("
        SELECT lp.payment_amount, lp.payment_method, lp.payment_date, c.full_name
        FROM loan_payment lp
        LEFT JOIN customer c ON lp.customer_id = c.id
        WHERE lp.customer_id = ?
        ORDER BY lp.payment_date DESC
        LIMIT 10
    ");
    if ($stmtRecent) {
        $stmtRecent->bind_param("i", $selected_customer_id);
        $stmtRecent->execute();
        $recent_payments = $stmtRecent->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtRecent->close();
    }
}

// Clients with outstanding loan invoices (overview)
$loan_clients_overview = [];
$loan_invoice_details = [];
$loan_invoice_items = [];

$loan_clients_overview = [];
$loan_invoice_details = [];
$loan_invoice_items = [];

// Fetch all unpaid loan invoices (no province restriction so cashiers see all)
$invoiceSql = "
    SELECT 
        ui.id,
        ui.customer_id,
        ui.customer_name,
        ui.phone_number,
        ui.invoice_number,
        ui.loan_outstanding_bif,
        ui.created_at,
        ui.approval_status,
        ui.status
    FROM unpaid_invoices ui
    WHERE ui.is_loan_sale = 1
      AND ui.loan_outstanding_bif > 0
      AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
      AND (ui.approval_status != 'rejected')
    ORDER BY ui.created_at DESC
";
$stmtInvoices = $conn->prepare($invoiceSql);
if ($stmtInvoices) {
    $stmtInvoices->execute();
    $loan_invoice_details = $stmtInvoices->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtInvoices->close();
}

// Build customer overview from invoices
$overviewMap = [];
foreach ($loan_invoice_details as $inv) {
    $cid = (int)($inv['customer_id'] ?? 0);
    $cname = trim($inv['customer_name'] ?? '');
    $cphone = trim($inv['phone_number'] ?? '');
    $key = $cid > 0 ? "id:$cid" : "name:" . strtolower($cname) . "|phone:" . $cphone;
    if (!isset($overviewMap[$key])) {
        $overviewMap[$key] = [
            'id' => $cid,
            'full_name' => $cname ?: 'Client',
            'user_type' => 'physical',
            'phone_number' => $cphone,
            'total_due' => 0,
            'invoice_count' => 0,
        ];
    }
    $overviewMap[$key]['total_due'] += (int)($inv['loan_outstanding_bif'] ?? 0);
    $overviewMap[$key]['invoice_count'] += 1;
}
$loan_clients_overview = array_values($overviewMap);
usort($loan_clients_overview, function($a, $b) {
    return strcasecmp($a['full_name'], $b['full_name']);
});

// Items for these invoices
$invoiceIds = array_column($loan_invoice_details, 'id');
if (!empty($invoiceIds)) {
    $placeholders = implode(',', array_fill(0, count($invoiceIds), '?'));
    $typesItems = str_repeat('i', count($invoiceIds));
    $itemSql = "
        SELECT 
            ii.invoice_id,
            p.name AS product_name,
            ii.quantity,
            ii.unit_price
        FROM invoice_items ii
        JOIN product p ON ii.product_id = p.id
        WHERE ii.invoice_id IN ($placeholders)
    ";
    $stmtItems = $conn->prepare($itemSql);
    if ($stmtItems) {
        $stmtItems->bind_param($typesItems, ...$invoiceIds);
        $stmtItems->execute();
        $loan_invoice_items = $stmtItems->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtItems->close();
    }
}

include __DIR__ . '/../../includes/header.php';
?>
<div class="loan-repayments-container" style="padding: 0; background: #F4F0E4; min-height: calc(100vh - 70px);">
    <div class="loan-repayments-content" style="background-color: #FFFFFF; padding: 30px; border-radius: 0; box-shadow: none; width: 100%; margin: 0;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px;">Enregistrer Remboursement Crédit</h2>
        <p style="color: #000000; font-size: 16px;">Sélectionnez un client pour examiner son crédit en attente et enregistrer un remboursement.</p>
        <?php if (!empty($errors)): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 10px; border-radius: 5px; border: 1px solid #FF0000; margin-bottom: 15px;">
                <?php foreach ($errors as $error): ?>
                    <p><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div style="background-color: #4CAF50; color: #FFFFFF; padding: 10px; border-radius: 5px; margin-bottom: 15px;">
                <p><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></p>
            </div>
        <?php endif; ?>
        <?php if ($_SESSION['role_name'] === 'Caissier_Comptable'): ?>
            <div style="background-color: #F7F1E8; border: 1px solid #D3C2B6; border-radius: 10px; padding: 14px 16px; margin-bottom: 15px; display: grid; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); gap: 12px;">
                <div>
                    <p style="margin: 0; color: #6b4f2f; font-size: 12px; text-transform: uppercase;">Argent Disponible</p>
                    <p style="margin: 2px 0 0; font-size: 22px; font-weight: 700; color: #2e7d32;">
                        <?php echo number_format($cashier_available_total, 0, ',', '.'); ?> BIF
                    </p>
                </div>
                <div>
                    <p style="margin: 0; color: #6b4f2f; font-size: 12px; text-transform: uppercase;">Sessions fermées</p>
                    <p style="margin: 2px 0 0; font-size: 16px; font-weight: 700; color: #4B2F1F;">
                        <?php echo number_format($cashier_closed_cash, 0, ',', '.'); ?> BIF
                    </p>
                </div>
                <div>
                    <p style="margin: 0; color: #6b4f2f; font-size: 12px; text-transform: uppercase;">Session active</p>
                    <p style="margin: 2px 0 0; font-size: 16px; font-weight: 700; color: #4B2F1F;">
                        <?php echo number_format($cashier_active_cash, 0, ',', '.'); ?> BIF
                    </p>
                </div>
            </div>
        <?php endif; ?>
        <?php if ($formDisabled): ?>
            <div style="background-color: #FEE2E2; color: #991B1B; border: 1px solid #FCA5A5; padding: 12px 16px; border-radius: 8px; margin-bottom: 20px;">
                Aucune session active détectée. Veuillez démarrer une session de caisse avant d'enregistrer un remboursement.
            </div>
        <?php endif; ?>
        <form method="get" id="customer-select-form" style="display: flex; flex-wrap: wrap; gap: 15px; margin-bottom: 20px; align-items: flex-end;">
            <div style="flex: 1 1 300px; position: relative;">
                <label for="customer_search" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Rechercher Client</label>
                <div style="position: relative; width: 100%;">
                    <input type="text" id="customer_search" name="customer_search" placeholder="Commencez à taper le nom du client..." 
                           value="<?php 
                               if ($selected_customer_id > 0) {
                                   foreach ($loan_customers as $customer) {
                                       if ((int)$customer['id'] === $selected_customer_id) {
                                           echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8');
                                           break;
                                       }
                                   }
                               }
                           ?>"
                           style="width: 100%; padding: 12px 40px 12px 12px; border-radius: 8px; border: 2px solid #4B2F1F; background-color: #FFFFFF; font-size: 16px; color: #4B2F1F; height: 45px; box-sizing: border-box;" autocomplete="off">
                    <span style="position: absolute; right: 12px; top: 50%; transform: translateY(-50%); color: #4B2F1F; font-size: 18px; pointer-events: none;">🔍</span>
                    <div id="customer_dropdown" style="display: none; position: absolute; top: calc(100% + 5px); left: 0; right: 0; background-color: #FFFFFF; border: 2px solid #4B2F1F; border-radius: 8px; max-height: 300px; overflow-y: auto; z-index: 9999; box-shadow: 0 6px 12px rgba(0,0,0,0.15);">
                        <!-- Dropdown items will be populated by JavaScript -->
                    </div>
                </div>
                <input type="hidden" id="customer_id" name="customer_id" value="<?php echo $selected_customer_id; ?>">
                <div id="selected_customer_display" style="margin-top: 10px; padding: 10px; background-color: #F4F0E4; border-radius: 5px; display: <?php echo $selected_customer_id > 0 ? 'block' : 'none'; ?>;">
                    <strong>Sélectionné:</strong> <span id="selected_customer_name">
                        <?php 
                            if ($selected_customer_id > 0) {
                                foreach ($loan_customers as $customer) {
                                    if ((int)$customer['id'] === $selected_customer_id) {
                                        echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8');
                                        break;
                                    }
                                }
                            }
                        ?>
                    </span>
                </div>
            </div>
            <button type="submit" style="height: 45px; padding: 0 20px; background-color: #4B2F1F; color: #F4F0E4; border: none; border-radius: 8px; cursor: pointer; font-size: 16px; font-weight: 600;">Charger Aperçu</button>
        </form>

        <?php if ($customer_profile): ?>
            <div style="background-color: #FFFFFF; border: 1px solid #D3C2B6; border-radius: 10px; padding: 20px; margin-bottom: 20px;">
                <h3 style="color: #4B2F1F; font-size: 22px; margin-bottom: 10px;">Résumé Crédit</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 12px;">
                    <div>
                        <p style="margin: 0; color: rgba(75, 47, 31, 0.75); font-size: 12px; text-transform: uppercase;">Crédit En Attente</p>
                        <p style="margin: 4px 0 0; color: #2C1810; font-size: 18px; font-weight: 700;">
                            <?php echo number_format((int)($customer_profile['outstanding_balance'] ?? $customer_profile['loan_balance_bif'] ?? 0), 0, ',', '.'); ?> BIF
                        </p>
                        <?php if (($customer_profile['invoice_outstanding_total'] ?? 0) > 0): ?>
                            <p style="margin: 2px 0 0; font-size: 12px; color: #8b5e3c;">Dont commandes: <?php echo number_format((int)$customer_profile['invoice_outstanding_total'], 0, ',', '.'); ?> BIF</p>
                        <?php endif; ?>
                    </div>
                    <div>
                        <p style="margin: 0; color: rgba(75, 47, 31, 0.75); font-size: 12px; text-transform: uppercase;">Limite Crédit</p>
                        <p style="margin: 4px 0 0; color: #2C1810; font-size: 18px; font-weight: 700;">
                            <?php echo number_format((int)($customer_profile['loan_limit_bif'] ?? 0), 0, ',', '.'); ?> BIF
                        </p>
                    </div>
                    <div>
                        <p style="margin: 0; color: rgba(75, 47, 31, 0.75); font-size: 12px; text-transform: uppercase;">Limite Restante</p>
                        <p style="margin: 4px 0 0; color: #2C1810; font-size: 18px; font-weight: 700;">
                            <?php
                                $remainingLimit = max(0, (int)($customer_profile['loan_limit_bif'] ?? 0) - (int)($customer_profile['loan_balance_bif'] ?? 0));
                                echo number_format($remainingLimit, 0, ',', '.');
                            ?> BIF
                        </p>
                    </div>
                    <div>
                        <p style="margin: 0; color: rgba(75, 47, 31, 0.75); font-size: 12px; text-transform: uppercase;">Conditions Crédit</p>
                        <p style="margin: 4px 0 0; color: #2C1810; font-size: 18px; font-weight: 700;">
                            <?php echo (int)($customer_profile['loan_terms_days'] ?? 0) > 0 ? intval($customer_profile['loan_terms_days']) . ' jours' : 'Non définie'; ?>
                        </p>
                    </div>
                    <div>
                        <p style="margin: 0; color: rgba(75, 47, 31, 0.75); font-size: 12px; text-transform: uppercase;">Statut Crédit</p>
                        <p style="margin: 4px 0 0; color: #2C1810; font-size: 18px; font-weight: 700;">
                            <?php 
                                $status = $customer_profile['loan_status'] ?? 'inactive';
                                $status_fr = ($status === 'active') ? 'Actif' : (($status === 'inactive') ? 'Inactif' : ucfirst($status));
                                echo htmlspecialchars($status_fr, ENT_QUOTES, 'UTF-8'); 
                            ?>
                        </p>
        </div>
    </div>
</div>
<!-- Modal -->
<div id="loan-details-modal" style="display:none; position:fixed; inset:0; background:rgba(0,0,0,0.5); align-items:center; justify-content:center; z-index:2000;">
    <div style="background:#FFFFFF; border-radius:10px; padding:20px; width:95%; max-width:900px; max-height:90vh; overflow:auto; position:relative;">
        <button onclick="closeLoanModal()" style="position:absolute; top:10px; right:10px; background:#4B2F1F; color:#fff; border:none; border-radius:4px; padding:6px 10px; cursor:pointer;">Fermer</button>
        <h4 id="modal-customer-name" style="margin-top:0; color:#4B2F1F;">Détails Client</h4>
        <div id="modal-invoice-list"></div>
    </div>
</div>

<script>
    const loanInvoices = <?php echo json_encode($loan_invoice_details, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP); ?>;
    const loanItems = <?php echo json_encode($loan_invoice_items, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP); ?>;

    function closeLoanModal() {
        const modal = document.getElementById('loan-details-modal');
        if (modal) modal.style.display = 'none';
    }

    function makeKey(inv) {
        const cid = Number(inv.customer_id || 0);
        if (cid > 0) return 'id:' + cid;
        const cname = (inv.customer_name || '').trim().toLowerCase();
        const cphone = (inv.phone_number || '').trim();
        return 'name:' + cname + '|phone:' + cphone;
    }

    function renderLoanDetails(customerKey, customerName = '', customerPhone = '') {
        const modal = document.getElementById('loan-details-modal');
        const listEl = document.getElementById('modal-invoice-list');
        const nameEl = document.getElementById('modal-customer-name');
        if (!modal || !listEl || !nameEl) return;

        const invoices = loanInvoices.filter(inv => makeKey(inv) === customerKey);
        if (invoices.length === 0) {
            listEl.innerHTML = '<p style="color:#000;">Aucune facture impayée.</p>';
        } else {
            const rows = invoices.map(inv => {
                const items = loanItems.filter(it => Number(it.invoice_id) === Number(inv.id));
                const itemsHtml = items.map(it => {
                    const lineTotal = (Number(it.quantity) || 0) * (Number(it.unit_price) || 0);
                    return `<li>${it.product_name} — ${Number(it.quantity).toLocaleString('fr-FR')} × ${Number(it.unit_price).toLocaleString('fr-FR')} BIF = ${lineTotal.toLocaleString('fr-FR')} BIF</li>`;
                }).join('');
                const created = inv.created_at ? new Date(inv.created_at.replace(' ', 'T')).toLocaleString('fr-FR') : 'N/A';
                return `
                    <div style="border:1px solid #eee; border-radius:8px; padding:10px; margin-bottom:10px;">
                        <strong>Facture:</strong> ${inv.invoice_number || 'N/A'}<br>
                        <strong>Impayé:</strong> ${Number(inv.loan_outstanding_bif || 0).toLocaleString('fr-FR')} BIF<br>
                        <strong>Statut:</strong> ${inv.approval_status || 'N/A'} / ${inv.status || 'N/A'}<br>
                        <strong>Créée le:</strong> ${created}
                        <div style="margin-top:8px;">
                            <strong>Produits:</strong>
                            <ul style="margin:6px 0 0 18px; padding:0;">
                                ${itemsHtml || '<li>Aucun article trouvé</li>'}
                            </ul>
                        </div>
                    </div>
                `;
            }).join('');
            listEl.innerHTML = rows;
        }
        if (nameEl) {
            const title = customerName ? `Détails Client : ${customerName}` : 'Détails Client';
            nameEl.textContent = title;
        }
        modal.style.display = 'flex';
    }

    document.addEventListener('DOMContentLoaded', () => {
        document.querySelectorAll('.view-loan-details').forEach(btn => {
            btn.addEventListener('click', () => {
                const ckey = btn.getAttribute('data-customer-key');
                const cname = btn.getAttribute('data-customer-name') || '';
                const cphone = btn.getAttribute('data-customer-phone') || '';
                renderLoanDetails(ckey, cname, cphone);
            });
        });
    });
</script>

            <div style="background-color: #fff4e6; border: 1px dashed #d97706; border-radius: 10px; padding: 12px 16px; margin-bottom: 20px;">
                <strong style="color: #92400e;">Montant total dû :</strong>
                <span style="font-weight: 700; color: #7c2d12;"><?php echo number_format((int)($customer_profile['outstanding_balance'] ?? 0), 0, ',', '.'); ?> BIF</span>
                <?php if (($customer_profile['invoice_outstanding_total'] ?? 0) > 0): ?>
                    <span style="color: #a16207; margin-left: 8px;">(Commandes en attente : <?php echo number_format((int)$customer_profile['invoice_outstanding_total'], 0, ',', '.'); ?> BIF)</span>
                <?php endif; ?>
            </div>

            <?php if (!empty($available_invoices)): ?>
                <div style="background-color: #FFFFFF; border: 1px solid #D3C2B6; border-radius: 10px; padding: 20px; margin-bottom: 20px;">
                    <h3 style="color: #4B2F1F; font-size: 22px; margin-bottom: 10px;">Commandes Crédit En Attente</h3>
                    <table style="width: 100%; border-collapse: collapse;">
                        <thead>
                            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                                <th style="padding: 8px; text-align: left;">Commande</th>
                                <th style="padding: 8px; text-align: right;">En Attente (BIF)</th>
                                <th style="padding: 8px; text-align: left;">Date Échéance</th>
                                <th style="padding: 8px; text-align: left;">Créée</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($available_invoices as $invoice): ?>
                                <tr style="background-color: #FDFBF7;">
                                    <td style="padding: 8px;"><?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 8px; text-align: right;"><?php echo number_format((int)$invoice['loan_outstanding_bif'], 0, ',', '.'); ?></td>
                                    <td style="padding: 8px;"><?php echo $invoice['loan_due_date'] ? htmlspecialchars($invoice['loan_due_date'], ENT_QUOTES, 'UTF-8') : 'N/A'; ?></td>
                                    <td style="padding: 8px;"><?php echo htmlspecialchars($invoice['created_at'], ENT_QUOTES, 'UTF-8'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>

            <form method="post" enctype="multipart/form-data" style="background-color: #FFFFFF; border: 1px solid #D3C2B6; border-radius: 10px; padding: 20px;">
                <input type="hidden" name="customer_id" value="<?php echo htmlspecialchars($selected_customer_id, ENT_QUOTES, 'UTF-8'); ?>">
                <h3 style="color: #4B2F1F; font-size: 20px; margin-bottom: 15px;">Détails Remboursement</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                    <div>
                        <label for="amount" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Montant (BIF)</label>
                        <input type="text" inputmode="numeric" pattern="\d*" id="amount" name="amount" required <?php echo $formDisabled ? 'disabled' : ''; ?> style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                        <div id="max_amount_info" style="margin-top: 5px; font-size: 12px; color: #4B2F1F;">
                            <span id="max_amount_text">Maximum: <?php echo number_format((int)($customer_profile['outstanding_balance'] ?? $customer_profile['loan_balance_bif'] ?? 0), 0, ',', '.'); ?> BIF (Total En Attente)</span>
                        </div>
                    </div>
                    <div>
                        <label for="payment_method" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Méthode de Paiement</label>
                        <select id="payment_method" name="payment_method" required <?php echo $formDisabled ? 'disabled' : ''; ?> style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                            <option value="cash">Espèces</option>
                            <option value="bank_deposit">Dépôt bancaire</option>
                        </select>
                        <div style="margin-top: 5px; font-size: 12px; color: #666; font-style: italic;">Choisir "Dépôt bancaire" pour verser directement sur un compte.</div>
                    </div>
                    <div>
                        <label for="invoice_id" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Appliquer à la Commande</label>
                        <select id="invoice_id" name="invoice_id" <?php echo $formDisabled ? 'disabled' : ''; ?> style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                            <option value="0" data-max-amount="<?php echo (int)($customer_profile['outstanding_balance'] ?? $customer_profile['loan_balance_bif'] ?? 0); ?>">Remboursement général</option>
                            <?php foreach ($available_invoices as $invoice): ?>
                                <option value="<?php echo htmlspecialchars($invoice['id'], ENT_QUOTES, 'UTF-8'); ?>" data-max-amount="<?php echo (int)$invoice['loan_outstanding_bif']; ?>">
                                    <?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?> - <?php echo number_format((int)$invoice['loan_outstanding_bif'], 0, ',', '.'); ?> BIF
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div id="bank_fields" style="grid-column: span 2; display: none;">
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 12px;">
                            <div>
                                <label for="bank_account_id" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Compte bancaire</label>
                                <select id="bank_account_id" name="bank_account_id" <?php echo $formDisabled ? 'disabled' : ''; ?> style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                                    <option value="0">Sélectionner un compte</option>
                                    <?php foreach ($bank_accounts as $acct): ?>
                                        <option value="<?php echo (int)$acct['id']; ?>">
                                            <?php echo htmlspecialchars($acct['account_number'], ENT_QUOTES, 'UTF-8'); ?> (<?php echo htmlspecialchars($acct['currency'], ENT_QUOTES, 'UTF-8'); ?><?php echo !empty($acct['is_main']) ? ', principal' : ''; ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div>
                                <label for="slip_number" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">N° Bordereau</label>
                                <input type="text" id="slip_number" name="slip_number" <?php echo $formDisabled ? 'disabled' : ''; ?> style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                            </div>
                            <div>
                                <label for="slip_file" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Justificatif (image/pdf)</label>
                                <input type="file" id="slip_file" name="slip_file" accept=".jpg,.jpeg,.png,.pdf" <?php echo $formDisabled ? 'disabled' : ''; ?> style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 14px; color: #4B2F1F;">
                            </div>
                        </div>
                    </div>
                    <div style="grid-column: span 2;">
                        <label for="notes" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 5px;">Notes (optionnel)</label>
                        <textarea id="notes" name="notes" rows="3" <?php echo $formDisabled ? 'disabled' : ''; ?> style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; resize: vertical;"></textarea>
                    </div>
                </div>
                <button type="submit" name="record_loan_payment" <?php echo $formDisabled ? 'disabled' : ''; ?> style="margin-top: 20px; background-color: #4B2F1F; color: #F4F0E4; padding: 15px 25px; border: none; border-radius: 8px; cursor: pointer; font-size: 18px; font-weight: 600;">Enregistrer Remboursement</button>
            </form>

            <?php if (!empty($recent_payments)): ?>
                <div style="background-color: #FFFFFF; border: 1px solid #D3C2B6; border-radius: 10px; padding: 20px; margin-top: 20px;">
                    <h3 style="color: #4B2F1F; font-size: 22px; margin-bottom: 10px;">Remboursements Crédit Récents</h3>
                    <table style="width: 100%; border-collapse: collapse;">
                        <thead>
                            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                                <th style="padding: 8px; text-align: left;">Date</th>
                                <th style="padding: 8px; text-align: left;">Client</th>
                                <th style="padding: 8px; text-align: left;">Méthode</th>
                                <th style="padding: 8px; text-align: right;">Montant (BIF)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_payments as $payment): ?>
                                <tr style="background-color: #FDFBF7;">
                                    <td style="padding: 8px;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($payment['payment_date'])), ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 8px;"><?php echo htmlspecialchars($payment['full_name'] ?? 'Client', ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 8px; text-transform: capitalize;"><?php 
                                        $paymentMethod = $payment['payment_method'] ?? 'cash';
                                        $method = str_replace('_', ' ', (string)$paymentMethod);
                                        if ($paymentMethod === 'bank_deposit') {
                                            $method_fr = 'Dépôt bancaire';
                                        } elseif ($method === 'cash') {
                                            $method_fr = 'Espèces';
                                        } elseif ($method === 'bank transfer') {
                                            $method_fr = 'Virement Bancaire';
                                        } elseif ($method === 'mobile') {
                                            $method_fr = 'Paiement Mobile';
                                        } else {
                                            $method_fr = $method;
                                        }
                                        echo htmlspecialchars($method_fr, ENT_QUOTES, 'UTF-8'); 
                                    ?></td>
                                    <td style="padding: 8px; text-align: right;"><?php echo number_format((int)$payment['payment_amount'], 0, ',', '.'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        <?php elseif ($selected_customer_id > 0): ?>
            <p style="color: #FF0000; margin-top: 10px;">Impossible de charger le profil crédit du client. Veuillez réessayer.</p>
        <?php endif; ?>
    </div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Customer search functionality - interactive search with instant results
    const customerData = <?php echo json_encode(array_map(function($c) { 
        return [
            'id' => (int)$c['id'],
            'full_name' => trim($c['full_name']),
            'loan_balance_bif' => (int)($c['effective_balance'] ?? $c['loan_balance_bif'] ?? 0),
            'loan_limit_bif' => (int)($c['loan_limit_bif'] ?? 0),
            'loan_status' => $c['loan_status'] ?? 'inactive'
        ]; 
    }, $loan_customers ?? [])); ?>;
    
    const customerSearchEl = document.getElementById('customer_search');
    const customerDropdownEl = document.getElementById('customer_dropdown');
    const customerIdEl = document.getElementById('customer_id');
    const selectedCustomerDisplayEl = document.getElementById('selected_customer_display');
    const selectedCustomerNameEl = document.getElementById('selected_customer_name');
    const customerSelectForm = document.getElementById('customer-select-form');
    
    let isDropdownOpen = false;
    let searchDebounceTimer = null;
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function showDropdownMessage(message) {
        if (!customerDropdownEl) return;
        customerDropdownEl.innerHTML = '';
        const info = document.createElement('div');
        info.textContent = message;
        info.style.cssText = 'padding: 12px; color: #666; text-align: center; font-style: italic;';
        customerDropdownEl.appendChild(info);
    }
    
    function renderCustomerResults(results) {
        if (!customerDropdownEl) return;
        customerDropdownEl.innerHTML = '';
        
        if (results.length === 0) {
            showDropdownMessage('Aucun client trouvé');
            return;
        }
        
        results.forEach(customer => {
            const item = document.createElement('div');
            item.className = 'customer-item';
            item.style.cssText = 'padding: 12px; cursor: pointer; border-bottom: 1px solid #E0E0E0; transition: all 0.2s;';
            
            const balance = customer.loan_balance_bif || 0;
            const limit = customer.loan_limit_bif || 0;
            const status = customer.loan_status || 'inactive';
            
            item.innerHTML = `
                <div style="font-weight: 600; color: #4B2F1F;">${escapeHtml(customer.full_name || '')}</div>
                <div style="font-size: 12px; color: #666; margin-top: 4px;">
                    Solde: ${balance.toLocaleString('fr-FR')} BIF | 
                    Limite: ${limit.toLocaleString('fr-FR')} BIF | 
                    Statut: ${escapeHtml(status === 'active' ? 'Actif' : (status === 'inactive' ? 'Inactif' : status))}
                </div>
            `;
            
            item.addEventListener('mouseenter', function() {
                this.style.backgroundColor = '#F4F0E4';
                this.style.paddingLeft = '16px';
            });
            
            item.addEventListener('mouseleave', function() {
                this.style.backgroundColor = '';
                this.style.paddingLeft = '12px';
            });
            
            item.addEventListener('click', function(e) {
                e.stopPropagation();
                selectCustomer(customer);
            });
            
            customerDropdownEl.appendChild(item);
        });
    }
    
    function filterCustomers(searchTerm) {
        if (!searchTerm || searchTerm.trim() === '') {
            // Show first 20 customers when no search term
            const results = customerData.slice(0, 20);
            renderCustomerResults(results);
            customerDropdownEl.style.display = results.length > 0 ? 'block' : 'none';
            isDropdownOpen = results.length > 0;
            return;
        }
        
        const normalizedTerm = searchTerm.trim().toLowerCase();
        const results = customerData.filter(c => {
            const nameMatch = c.full_name && c.full_name.toLowerCase().includes(normalizedTerm);
            return nameMatch;
        }).slice(0, 50); // Limit to 50 results for performance
        
        renderCustomerResults(results);
        customerDropdownEl.style.display = 'block';
        isDropdownOpen = true;
    }
    
    function selectCustomer(customer) {
        if (customerSearchEl) {
            customerSearchEl.value = customer.full_name || '';
        }
        if (customerIdEl) {
            customerIdEl.value = customer.id || '';
        }
        if (selectedCustomerNameEl) {
            selectedCustomerNameEl.textContent = customer.full_name || '';
        }
        if (selectedCustomerDisplayEl) {
            selectedCustomerDisplayEl.style.display = 'block';
        }
        if (customerDropdownEl) {
            customerDropdownEl.style.display = 'none';
            isDropdownOpen = false;
        }
    }
    
    // Set up customer search event listeners
    if (customerSearchEl && customerDropdownEl && customerIdEl) {
        // Show dropdown on focus if there's data
        customerSearchEl.addEventListener('focus', function() {
            if (customerData.length > 0) {
                clearTimeout(searchDebounceTimer);
                const term = this.value.trim();
                if (term === '') {
                    filterCustomers('');
                } else {
                    filterCustomers(term);
                }
            }
        });
        
        // Real-time search as user types
        customerSearchEl.addEventListener('input', function() {
            clearTimeout(searchDebounceTimer);
            const term = this.value;
            
            // Debounce search for better performance
            searchDebounceTimer = setTimeout(() => {
                filterCustomers(term);
            }, 150); // 150ms delay
        });
        
        // Handle form submission
        if (customerSelectForm) {
            customerSelectForm.addEventListener('submit', function(e) {
                const typedName = customerSearchEl.value.trim().toLowerCase();
                if (!typedName) {
                    if (customerIdEl) {
                        customerIdEl.value = '';
                    }
                    e.preventDefault();
                    alert('Veuillez sélectionner un client');
                    return false;
                }
                
                // Try to find exact match
                const exact = customerData.find(c => 
                    c.full_name && c.full_name.toLowerCase() === typedName
                );
                
                if (exact) {
                    if (customerIdEl) {
                        customerIdEl.value = exact.id;
                    }
                } else {
                    // If no exact match, find first partial match
                    const partial = customerData.find(c => 
                        c.full_name && c.full_name.toLowerCase().includes(typedName)
                    );
                    
                    if (partial) {
                        if (customerIdEl) {
                            customerIdEl.value = partial.id;
                        }
                        customerSearchEl.value = partial.full_name;
                    } else {
                        e.preventDefault();
                        alert('Veuillez sélectionner un client valide dans la liste déroulante');
                        customerSearchEl.focus();
                        return false;
                    }
                }
            });
        }
        
        // Close dropdown when clicking outside
        document.addEventListener('click', function(e) {
            if (isDropdownOpen && 
                customerDropdownEl && 
                !customerDropdownEl.contains(e.target) && 
                customerSearchEl && 
                !customerSearchEl.contains(e.target)) {
                customerDropdownEl.style.display = 'none';
                isDropdownOpen = false;
            }
        });
        
        // Handle keyboard navigation
        customerSearchEl.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                if (customerDropdownEl) {
                    customerDropdownEl.style.display = 'none';
                    isDropdownOpen = false;
                }
            }
        });
    }
    
    // Toggle bank deposit specific fields
    const paymentMethodSelect = document.getElementById('payment_method');
    const bankFields = document.getElementById('bank_fields');
    function toggleBankFields() {
        if (!paymentMethodSelect || !bankFields) return;
        bankFields.style.display = paymentMethodSelect.value === 'bank_deposit' ? 'block' : 'none';
    }
    if (paymentMethodSelect) {
        paymentMethodSelect.addEventListener('change', toggleBankFields);
        toggleBankFields();
    }

    // Existing invoice amount validation code
    const invoiceSelect = document.getElementById('invoice_id');
    const amountInput = document.getElementById('amount');
    const maxAmountText = document.getElementById('max_amount_text');
    
    if (invoiceSelect && amountInput) {
        invoiceSelect.addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const maxAmount = selectedOption ? parseInt(selectedOption.getAttribute('data-max-amount') || '0') : 0;
            
            // Update input max attribute
            amountInput.setAttribute('max', maxAmount);
            
            // If "Remboursement général" is selected (value="0"), auto-fill with total amount due and disable input
            if (this.value === '0') {
                amountInput.value = maxAmount; // Auto-fill with total amount due
                amountInput.readOnly = true; // Disable typing
                amountInput.style.backgroundColor = '#E0E0E0'; // Visual indicator that field is disabled
                amountInput.style.cursor = 'not-allowed';
            } else {
                // When a specific invoice is selected, auto-fill with invoice amount
                amountInput.value = maxAmount; // Auto-fill with invoice outstanding amount
                amountInput.readOnly = false; // Allow editing for specific invoice (in case of partial payment)
                amountInput.style.backgroundColor = '#F4F0E4'; // Restore normal background
                amountInput.style.cursor = 'text';
            }
            
            // Update max amount display if element exists
            if (maxAmountText) {
                if (this.value === '0') {
                    maxAmountText.textContent = 'Maximum: ' + new Intl.NumberFormat('fr-FR').format(maxAmount) + ' BIF (Total En Attente)';
                } else {
                    const invoiceNumber = selectedOption.textContent.split(' - ')[0];
                    maxAmountText.textContent = 'Maximum: ' + new Intl.NumberFormat('fr-FR').format(maxAmount) + ' BIF (Commande: ' + invoiceNumber + ')';
                }
            }
            
            // Validate current amount if it exceeds new max (only for specific invoices)
            if (this.value !== '0') {
                const currentAmount = parseInt(amountInput.value) || 0;
                if (currentAmount > maxAmount) {
                    amountInput.value = '';
                    alert('Le montant saisi dépasse le maximum autorisé. Veuillez saisir un montant ne dépassant pas ' + new Intl.NumberFormat('fr-FR').format(maxAmount) + ' BIF.');
                }
            }
        });
        
        // Initialize amount field on page load if "Remboursement général" is selected
        if (invoiceSelect.value === '0') {
            const selectedOption = invoiceSelect.options[invoiceSelect.selectedIndex];
            const maxAmount = selectedOption ? parseInt(selectedOption.getAttribute('data-max-amount') || '0') : 0;
            if (maxAmount > 0) {
                amountInput.value = maxAmount;
                amountInput.readOnly = true;
                amountInput.style.backgroundColor = '#E0E0E0';
                amountInput.style.cursor = 'not-allowed';
            }
        }
        
        // Validate on amount input
        amountInput.addEventListener('input', function() {
            this.value = this.value.replace(/\D/g, '');
            const selectedOption = invoiceSelect.options[invoiceSelect.selectedIndex];
            const maxAmount = selectedOption ? parseInt(selectedOption.getAttribute('data-max-amount') || '0') : 0;
            const enteredAmount = parseInt(this.value) || 0;
            
            if (enteredAmount > maxAmount) {
                this.style.borderColor = '#FF0000';
                this.style.backgroundColor = '#FFE0E0';
                if (maxAmountText) {
                    maxAmountText.style.color = '#FF0000';
                    maxAmountText.style.fontWeight = 'bold';
                }
            } else {
                this.style.borderColor = '#4B2F1F';
                this.style.backgroundColor = '#F4F0E4';
                if (maxAmountText) {
                    maxAmountText.style.color = '#4B2F1F';
                    maxAmountText.style.fontWeight = 'normal';
                }
            }
        });
        
        // Validate on form submit
        const form = amountInput.closest('form');
        if (form) {
            form.addEventListener('submit', function(e) {
                const selectedOption = invoiceSelect.options[invoiceSelect.selectedIndex];
                const maxAmount = selectedOption ? parseInt(selectedOption.getAttribute('data-max-amount') || '0') : 0;
                const enteredAmount = parseInt(amountInput.value) || 0;
                
                if (enteredAmount > maxAmount) {
                    e.preventDefault();
                    alert('Le montant du remboursement ne peut pas dépasser le montant maximum autorisé de ' + new Intl.NumberFormat('fr-FR').format(maxAmount) + ' BIF.');
                    amountInput.focus();
                    return false;
                }
            });
        }
    }
});
</script>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
<?php $conn->close(); ?>
