<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

// Log script start
error_log("Script started at " . date('Y-m-d H:i:s') . " for create_invoice.php");

require_login();

// Initialize database connection with error logging
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in create_invoice.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    exit;
}

// Restrict to matching roles
$stmt = $conn->prepare("SELECT u.id, u.full_name, r.name, r.id AS role_id FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
if ($stmt === false) {
    error_log("Prepare failed: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database prepare failed. Check logs.</p>";
    $conn->close();
    exit;
}
$stmt->bind_param("i", $_SESSION['user_id']);
if (!$stmt->execute()) {
    error_log("Execute failed: " . $stmt->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database execute failed. Check logs.</p>";
    $stmt->close();
    $conn->close();
    exit;
}
$role_result = $stmt->get_result();
$role = $role_result->fetch_assoc();
$stmt->close();
$allowed_roles = ['Caissier_Comptable', 'Marketing Agent', 'Agent Marketing', 'Chef Marketing', 'DG', 'DGA', 'Operation Manager'];
if (!$role || !in_array($role['name'], $allowed_roles, true)) {
    error_log("Unauthorized access to create_invoice.php by user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    header("Location: /masunzu_bar_hotel/dashboards/" . ($role && $role['name'] === 'Caissier_Comptable' ? 'cashier_dashboard.php' : 'marketing_dashboard.php'));
    $conn->close();
    exit;
}

// Load available cashiers
$cashiers = [];
$cashierStmt = $conn->prepare("SELECT id, full_name FROM user WHERE role_id = (SELECT id FROM role WHERE name = 'Caissier_Comptable') ORDER BY full_name ASC");
if ($cashierStmt) {
    $cashierStmt->execute();
    $cashierResult = $cashierStmt->get_result();
    while ($cashierRow = $cashierResult->fetch_assoc()) {
        $cashiers[] = $cashierRow;
    }
    $cashierStmt->close();
}
$cashier_ids = array_map('intval', array_column($cashiers, 'id'));
$requires_cashier_selection = in_array($role['name'], ['Marketing Agent', 'Agent Marketing', 'Chef Marketing', 'DG', 'DGA', 'Operation Manager'], true);

// Load province options
$province_options = [];
$provinceStmt = $conn->prepare("SELECT id, name FROM province ORDER BY name ASC");
if ($provinceStmt) {
    $provinceStmt->execute();
    $province_options = $provinceStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $provinceStmt->close();
}
$province_ids = array_map('intval', array_column($province_options, 'id'));

// Check for assigned_cashier_id column existence
$has_assigned_cashier_column = false;
$columnStmt = $conn->prepare("SHOW COLUMNS FROM unpaid_invoices LIKE 'assigned_cashier_id'");
if ($columnStmt) {
    $columnStmt->execute();
    $columnResult = $columnStmt->get_result();
    $has_assigned_cashier_column = $columnResult && $columnResult->num_rows > 0;
    $columnStmt->close();
}

// Load cashiers and their provinces
$cashiers = [];
$cashierStmt = $conn->prepare("
    SELECT u.id, u.full_name, u.province_id
    FROM user u
    JOIN role r ON u.role_id = r.id
    WHERE r.name = 'Caissier_Comptable'
    ORDER BY u.full_name ASC
");
if ($cashierStmt) {
    $cashierStmt->execute();
    $cashierRows = $cashierStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    foreach ($cashierRows as $row) {
        $cashiers[] = $row;
    }
    $cashierStmt->close();
}

$selected_cashier_id = filter_input(INPUT_GET, 'cashier_id', FILTER_VALIDATE_INT);
if ($selected_cashier_id) {
    $_SESSION['selected_cashier_id'] = $selected_cashier_id;
}
if (!isset($_SESSION['selected_cashier_id'])) {
    if ($role['name'] === 'Caissier_Comptable') {
        $_SESSION['selected_cashier_id'] = $_SESSION['user_id'];
    } elseif (!empty($cashiers)) {
        $_SESSION['selected_cashier_id'] = $cashiers[0]['id'];
    }
}
$selected_cashier_id = $_SESSION['selected_cashier_id'] ?? null;
$selected_cashier = null;
$cashier_lookup = array_column($cashiers, null, 'id');
if ($selected_cashier_id && isset($cashier_lookup[$selected_cashier_id])) {
    $selected_cashier = $cashier_lookup[$selected_cashier_id];
}
if (!$selected_cashier && $role['name'] === 'Caissier_Comptable') {
    $selected_cashier = [
        'id' => $_SESSION['user_id'],
        'full_name' => $_SESSION['full_name'],
        'province_id' => $province_id,
    ];
}
if ($selected_cashier) {
    $province_id = (int)($selected_cashier['province_id'] ?? $province_id);
    $_SESSION['province_id'] = $province_id;
}
// Ensure required columns exist in database tables
try {
    // Check and add missing columns to unpaid_invoices
    $check_col = $conn->query("SHOW COLUMNS FROM unpaid_invoices LIKE 'has_custom_prices'");
    if ($check_col->num_rows == 0) {
        $conn->query("ALTER TABLE unpaid_invoices ADD COLUMN has_custom_prices TINYINT(1) NOT NULL DEFAULT 0 AFTER is_loan_sale");
    }
    $check_col = $conn->query("SHOW COLUMNS FROM unpaid_invoices LIKE 'custom_price_summary'");
    if ($check_col->num_rows == 0) {
        $conn->query("ALTER TABLE unpaid_invoices ADD COLUMN custom_price_summary TEXT DEFAULT NULL AFTER has_custom_prices");
    }
    
    // Normalize invoice_items schema to a single unit_price column
    $col_unit_price = $conn->query("SHOW COLUMNS FROM invoice_items LIKE 'unit_price'");
    if ($col_unit_price->num_rows == 0) {
        $legacy_wholesale = $conn->query("SHOW COLUMNS FROM invoice_items LIKE 'wholesale_price'");
        if ($legacy_wholesale && $legacy_wholesale->num_rows > 0) {
            $conn->query("ALTER TABLE invoice_items CHANGE COLUMN wholesale_price unit_price BIGINT(20) NOT NULL");
        } else {
            $conn->query("ALTER TABLE invoice_items ADD COLUMN unit_price BIGINT(20) NOT NULL AFTER quantity");
        }
    }
    $conn->query("ALTER TABLE invoice_items DROP COLUMN IF EXISTS wholesale_price");
    $conn->query("ALTER TABLE invoice_items DROP COLUMN IF EXISTS retail_price");
    $conn->query("ALTER TABLE invoice_items ADD COLUMN IF NOT EXISTS price_source VARCHAR(20) DEFAULT 'system' AFTER unit_price");
    $conn->query("ALTER TABLE invoice_items ADD COLUMN IF NOT EXISTS system_unit_price BIGINT(20) DEFAULT NULL AFTER price_source");
    $conn->query("ALTER TABLE invoice_items ADD COLUMN IF NOT EXISTS custom_unit_price BIGINT(20) DEFAULT NULL AFTER system_unit_price");
    $conn->query("ALTER TABLE invoice_items DROP COLUMN IF EXISTS system_wholesale_price");
    $conn->query("ALTER TABLE invoice_items DROP COLUMN IF EXISTS system_retail_price");
    $conn->query("ALTER TABLE invoice_items DROP COLUMN IF EXISTS custom_wholesale_price");
    $conn->query("ALTER TABLE invoice_items DROP COLUMN IF EXISTS custom_retail_price");
    $conn->query("ALTER TABLE invoice_items ADD COLUMN IF NOT EXISTS custom_price_basis VARCHAR(50) DEFAULT NULL AFTER custom_unit_price");
    $conn->query("ALTER TABLE invoice_items ADD COLUMN IF NOT EXISTS custom_price_input BIGINT(20) DEFAULT NULL AFTER custom_price_basis");
    $conn->query("ALTER TABLE invoice_items ADD COLUMN IF NOT EXISTS custom_price_note TEXT DEFAULT NULL AFTER custom_price_input");
    
    // Ensure invoice_items.id has AUTO_INCREMENT
    try {
        $check_id_col = $conn->query("SHOW COLUMNS FROM invoice_items LIKE 'id'");
        if ($check_id_col && $check_id_col->num_rows > 0) {
            $id_col = $check_id_col->fetch_assoc();
            $has_auto_increment = isset($id_col['Extra']) && strpos(strtolower($id_col['Extra']), 'auto_increment') !== false;
            if (!$has_auto_increment) {
                // Get the current column definition
                $col_type = $id_col['Type'] ?? 'INT(11)';
                $col_null = $id_col['Null'] ?? 'NO';
                $null_def = $col_null === 'YES' ? 'NULL' : 'NOT NULL';
                
                // Check if id is a primary key
                $check_primary = $conn->query("SHOW KEYS FROM invoice_items WHERE Key_name = 'PRIMARY' AND Column_name = 'id'");
                $is_primary = $check_primary && $check_primary->num_rows > 0;
                
                if ($is_primary) {
                    $alter_query = "ALTER TABLE invoice_items MODIFY COLUMN id {$col_type} {$null_def} AUTO_INCREMENT";
                    if ($conn->query($alter_query)) {
                        error_log("✅ Added AUTO_INCREMENT to invoice_items.id column");
                    } else {
                        error_log("❌ Failed to add AUTO_INCREMENT to invoice_items.id: " . $conn->error);
                    }
                } else {
                    // If not primary key, make it one first
                    $alter_query = "ALTER TABLE invoice_items MODIFY COLUMN id {$col_type} {$null_def} AUTO_INCREMENT PRIMARY KEY";
                    if ($conn->query($alter_query)) {
                        error_log("✅ Added AUTO_INCREMENT and PRIMARY KEY to invoice_items.id column");
                    } else {
                        error_log("❌ Failed to add AUTO_INCREMENT and PRIMARY KEY to invoice_items.id: " . $conn->error);
                    }
                }
            } else {
                error_log("✅ invoice_items.id already has AUTO_INCREMENT");
            }
        } else {
            error_log("⚠️ invoice_items table or id column not found");
        }
    } catch (Exception $e) {
        error_log("❌ Error checking invoice_items.id AUTO_INCREMENT: " . $e->getMessage());
    }
    
    // Align paid_invoice_items schema for archival copies
    $paid_items_table = $conn->query("SHOW TABLES LIKE 'paid_invoice_items'");
    if ($paid_items_table && $paid_items_table->num_rows > 0) {
        $paid_unit = $conn->query("SHOW COLUMNS FROM paid_invoice_items LIKE 'unit_price'");
        if ($paid_unit->num_rows == 0) {
            $paid_legacy = $conn->query("SHOW COLUMNS FROM paid_invoice_items LIKE 'wholesale_price'");
            if ($paid_legacy && $paid_legacy->num_rows > 0) {
                $conn->query("ALTER TABLE paid_invoice_items CHANGE COLUMN wholesale_price unit_price BIGINT(20) NOT NULL");
            } else {
                $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN unit_price BIGINT(20) NOT NULL AFTER quantity");
            }
        }
        $conn->query("ALTER TABLE paid_invoice_items DROP COLUMN IF EXISTS wholesale_price");
        $conn->query("ALTER TABLE paid_invoice_items DROP COLUMN IF EXISTS retail_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS price_source VARCHAR(20) DEFAULT 'system' AFTER unit_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS system_unit_price BIGINT(20) DEFAULT NULL AFTER price_source");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_unit_price BIGINT(20) DEFAULT NULL AFTER system_unit_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_price_basis VARCHAR(50) DEFAULT NULL AFTER custom_unit_price");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_price_input BIGINT(20) DEFAULT NULL AFTER custom_price_basis");
        $conn->query("ALTER TABLE paid_invoice_items ADD COLUMN IF NOT EXISTS custom_price_note TEXT DEFAULT NULL AFTER custom_price_input");
    }
    
    // Check and create invoice_price_adjustment_history table if it doesn't exist
    $check_table = $conn->query("SHOW TABLES LIKE 'invoice_price_adjustment_history'");
    if ($check_table->num_rows == 0) {
        $conn->query("CREATE TABLE invoice_price_adjustment_history (
            id INT(11) NOT NULL AUTO_INCREMENT,
            unpaid_invoice_id INT(11) NOT NULL,
            invoice_number VARCHAR(20) NOT NULL,
            product_id INT(11) NOT NULL,
            price_source VARCHAR(20) DEFAULT 'custom',
            original_unit_price BIGINT(20) NOT NULL DEFAULT 0,
            new_unit_price BIGINT(20) NOT NULL DEFAULT 0,
            custom_price_basis VARCHAR(50) DEFAULT NULL,
            custom_price_input BIGINT(20) DEFAULT NULL,
            quantity_units INT(11) DEFAULT NULL,
            changed_by INT(11) NOT NULL,
            change_note TEXT DEFAULT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_unpaid_invoice (unpaid_invoice_id),
            KEY idx_product (product_id),
            KEY idx_changed_by (changed_by),
            KEY idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
    }
    
    // Ensure customer table id has AUTO_INCREMENT
    $check_customer_ai = $conn->query("SHOW COLUMNS FROM customer WHERE Field = 'id' AND Extra LIKE '%auto_increment%'");
    if ($check_customer_ai->num_rows == 0) {
        $conn->query("ALTER TABLE customer MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT");
        error_log("Added AUTO_INCREMENT to customer.id column");
    }
} catch (Exception $e) {
    error_log("Error checking/adding columns or tables: " . $e->getMessage());
}

// Fetch province_id
if (!isset($_SESSION['province_id'])) {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    if ($stmt === false) {
        error_log("Prepare failed: " . $conn->error);
        http_response_code(500);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database prepare failed. Check logs.</p>";
        $conn->close();
        exit;
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
        http_response_code(500);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database execute failed. Check logs.</p>";
        $stmt->close();
        $conn->close();
        exit;
    }
    $result = $stmt->get_result();
    $province_row = $result->fetch_assoc();
    $_SESSION['province_id'] = isset($province_row['province_id']) ? (int)$province_row['province_id'] : null;
    error_log("DEBUG create_invoice.php: Fetched province_id from DB: " . ($_SESSION['province_id'] ?? 'NULL') . " for user_id={$_SESSION['user_id']}");
    $stmt->close();
}
    $province_id = isset($_SESSION['province_id']) ? (int)$_SESSION['province_id'] : null;
    $marketing_roles_without_province = ['Marketing Head', 'Chef Marketing', 'Marketing Agent', 'Agent Marketing'];
    if (!$province_id) {
        if (in_array($_SESSION['role_name'], $marketing_roles_without_province, true)) {
            $fallback_stmt = $conn->prepare("SELECT id FROM province ORDER BY name LIMIT 1");
            if ($fallback_stmt) {
                $fallback_stmt->execute();
                $fallback_row = $fallback_stmt->get_result()->fetch_assoc();
                $province_id = isset($fallback_row['id']) ? (int)$fallback_row['id'] : null;
                $fallback_stmt->close();
            }
            if (!$province_id) {
                error_log("No fallback province available for marketing user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
                http_response_code(500);
                echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur: Aucun dépôt disponible pour votre profil. Contactez l'administrateur.</p>";
                $conn->close();
                exit;
            }
            $_SESSION['province_id'] = $province_id;
        } else {
            error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
            http_response_code(500);
            echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur: Aucune province assignée. Contactez l'administrateur.</p>";
            $conn->close();
            exit;
        }
    }

    $postProvinceOverride = filter_input(INPUT_POST, 'province_id', FILTER_VALIDATE_INT);
    if ($postProvinceOverride && in_array($postProvinceOverride, $province_ids, true)) {
        $province_id = $postProvinceOverride;
        $_SESSION['province_id'] = $province_id;
    }
error_log("DEBUG create_invoice.php: User {$_SESSION['full_name']} (ID: {$_SESSION['user_id']}) has province_id=$province_id (type: " . gettype($province_id) . ")");

// Fetch stock_manager_id for the province (role_id = 5 is Stock Manager)
$stock_manager_id = null;
$stmt = $conn->prepare("SELECT id FROM user WHERE role_id = 5 AND province_id = ? LIMIT 1");
if ($stmt === false) {
    error_log("Prepare failed: " . $conn->error);
    // Don't exit - stock_manager_id is optional for invoice creation
} else {
$stmt->bind_param("i", $province_id);
    if ($stmt->execute()) {
$result = $stmt->get_result();
$stock_manager = $result->fetch_assoc();
if ($stock_manager) {
    $stock_manager_id = $stock_manager['id'];
            error_log("DEBUG create_invoice.php: Found Stock Manager ID=$stock_manager_id for province_id=$province_id");
} else {
            error_log("DEBUG create_invoice.php: No Stock Manager found for province_id=$province_id (this is non-blocking)");
        }
    } else {
        error_log("Execute failed: " . $stmt->error);
    }
    $stmt->close();
}

    // Ensure customer table has is_active field
try {
    $check_is_active = $conn->query("SHOW COLUMNS FROM customer WHERE Field = 'is_active'");
    if ($check_is_active->num_rows == 0) {
        $conn->query("ALTER TABLE customer ADD COLUMN is_active TINYINT(1) NOT NULL DEFAULT 1 AFTER loan_last_review_at");
        error_log("Added is_active column to customer table");
    }
    
    // Add reserved_quantity to province_stock for stock reservation
    $check_reserved = $conn->query("SHOW COLUMNS FROM province_stock WHERE Field = 'reserved_quantity'");
    if ($check_reserved->num_rows == 0) {
        $conn->query("ALTER TABLE province_stock ADD COLUMN reserved_quantity INT(11) NOT NULL DEFAULT 0 AFTER quantity");
        error_log("Added reserved_quantity column to province_stock table");
    }
    
    // Add expires_at to unpaid_invoices for expiration tracking
    $check_expires = $conn->query("SHOW COLUMNS FROM unpaid_invoices WHERE Field = 'expires_at'");
    if ($check_expires->num_rows == 0) {
        $conn->query("ALTER TABLE unpaid_invoices ADD COLUMN expires_at DATETIME DEFAULT NULL AFTER approved_at");
        error_log("Added expires_at column to unpaid_invoices table");
    }
    
    // Ensure customer_loan_ledger.id has AUTO_INCREMENT
    try {
        $check_ledger_id = $conn->query("SHOW COLUMNS FROM customer_loan_ledger LIKE 'id'");
        if ($check_ledger_id && $check_ledger_id->num_rows > 0) {
            $ledger_id_col = $check_ledger_id->fetch_assoc();
            $ledger_has_auto_increment = isset($ledger_id_col['Extra']) && strpos(strtolower($ledger_id_col['Extra']), 'auto_increment') !== false;
            if (!$ledger_has_auto_increment) {
                $col_type = $ledger_id_col['Type'] ?? 'INT(11)';
                $col_null = $ledger_id_col['Null'] ?? 'NO';
                $null_def = $col_null === 'YES' ? 'NULL' : 'NOT NULL';
                
                $check_ledger_primary = $conn->query("SHOW KEYS FROM customer_loan_ledger WHERE Key_name = 'PRIMARY' AND Column_name = 'id'");
                $ledger_is_primary = $check_ledger_primary && $check_ledger_primary->num_rows > 0;
                if ($ledger_is_primary) {
                    $alter_query = "ALTER TABLE customer_loan_ledger MODIFY COLUMN id {$col_type} {$null_def} AUTO_INCREMENT";
                    if ($conn->query($alter_query)) {
                        error_log("✅ Added AUTO_INCREMENT to customer_loan_ledger.id column");
                    } else {
                        error_log("❌ Failed to add AUTO_INCREMENT to customer_loan_ledger.id: " . $conn->error);
                    }
                } else {
                    $alter_query = "ALTER TABLE customer_loan_ledger MODIFY COLUMN id {$col_type} {$null_def} AUTO_INCREMENT PRIMARY KEY";
                    if ($conn->query($alter_query)) {
                        error_log("✅ Added AUTO_INCREMENT and PRIMARY KEY to customer_loan_ledger.id column");
                    } else {
                        error_log("❌ Failed to add AUTO_INCREMENT and PRIMARY KEY to customer_loan_ledger.id: " . $conn->error);
                    }
                }
            } else {
                error_log("✅ customer_loan_ledger.id already has AUTO_INCREMENT");
            }
        }
    } catch (Exception $e) {
        error_log("❌ Error checking customer_loan_ledger.id AUTO_INCREMENT: " . $e->getMessage());
    }
    
    // Ensure log.id has AUTO_INCREMENT (should already be set, but check to be safe)
    try {
        $check_log_id = $conn->query("SHOW COLUMNS FROM log LIKE 'id'");
        if ($check_log_id && $check_log_id->num_rows > 0) {
            $log_id_col = $check_log_id->fetch_assoc();
            $log_has_auto_increment = isset($log_id_col['Extra']) && strpos(strtolower($log_id_col['Extra']), 'auto_increment') !== false;
            if (!$log_has_auto_increment) {
                $col_type = $log_id_col['Type'] ?? 'INT(11)';
                $col_null = $log_id_col['Null'] ?? 'NO';
                $null_def = $col_null === 'YES' ? 'NULL' : 'NOT NULL';
                
                $check_log_primary = $conn->query("SHOW KEYS FROM log WHERE Key_name = 'PRIMARY' AND Column_name = 'id'");
                $log_is_primary = $check_log_primary && $check_log_primary->num_rows > 0;
                if ($log_is_primary) {
                    $alter_query = "ALTER TABLE log MODIFY COLUMN id {$col_type} {$null_def} AUTO_INCREMENT";
                    if ($conn->query($alter_query)) {
                        error_log("✅ Added AUTO_INCREMENT to log.id column");
                    } else {
                        error_log("❌ Failed to add AUTO_INCREMENT to log.id: " . $conn->error);
                    }
                } else {
                    $alter_query = "ALTER TABLE log MODIFY COLUMN id {$col_type} {$null_def} AUTO_INCREMENT PRIMARY KEY";
                    if ($conn->query($alter_query)) {
                        error_log("✅ Added AUTO_INCREMENT and PRIMARY KEY to log.id column");
                    } else {
                        error_log("❌ Failed to add AUTO_INCREMENT and PRIMARY KEY to log.id: " . $conn->error);
                    }
                }
            } else {
                error_log("✅ log.id already has AUTO_INCREMENT");
            }
        }
    } catch (Exception $e) {
        error_log("❌ Error checking log.id AUTO_INCREMENT: " . $e->getMessage());
    }
} catch (Exception $e) {
    error_log("Error checking/adding columns: " . $e->getMessage());
}

// Note: Invoice expiration check runs automatically in approve_invoices.php
// When invoices expire (>3 hours), stock is automatically returned to available stock

// Fetch customers for dropdown, including NIF (only active customers)
$customers = [];
$stmt = $conn->prepare("SELECT id, full_name, nif, tel, can_pay_by_loan, loan_limit_bif, loan_balance_bif, loan_terms_days, loan_status, COALESCE(is_active, 1) as is_active FROM customer WHERE COALESCE(is_active, 1) = 1 ORDER BY full_name");
if ($stmt === false) {
    error_log("Prepare failed: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database prepare failed. Check logs.</p>";
    $conn->close();
    exit;
}
$stmt->execute();
$customers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

if (empty($customers)) {
    error_log("create_invoice.php: No active customers found, auto-seeding a default record.");
    $seed_stmt = $conn->prepare("
        INSERT INTO customer (full_name, user_type, is_active, created_at, can_pay_by_loan, loan_limit_bif, loan_balance_bif, loan_status)
        VALUES ('Default Walk-In Customer', 'physical', 1, NOW(), 0, 0, 0, 'inactive')
    ");
    if ($seed_stmt && $seed_stmt->execute()) {
        $seed_stmt->close();
        $stmt = $conn->prepare("SELECT id, full_name, nif, tel, can_pay_by_loan, loan_limit_bif, loan_balance_bif, loan_terms_days, loan_status, COALESCE(is_active, 1) as is_active FROM customer WHERE COALESCE(is_active, 1) = 1 ORDER BY full_name");
        if ($stmt) {
            $stmt->execute();
            $customers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
        }
    } else {
        error_log("create_invoice.php: Failed to seed default customer - " . ($seed_stmt ? $seed_stmt->error : $conn->error));
        if ($seed_stmt) {
            $seed_stmt->close();
        }
    }
}

$customer_js_struct = array_map(function ($c) {
    return [
        'id' => (int)$c['id'],
        'full_name' => trim($c['full_name']),
        'nif' => $c['nif'] ? trim($c['nif']) : null,
        'tel' => $c['tel'] ? trim($c['tel']) : null,
        'display' => trim($c['full_name']) . ($c['nif'] ? ' - NIF: ' . trim($c['nif']) : '')
    ];
}, $customers ?? []);
$customer_js_data = json_encode(
    $customer_js_struct,
    JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE | JSON_INVALID_UTF8_SUBSTITUTE
);
if ($customer_js_data === false) {
    error_log('Failed to encode customer data for create_invoice.php: ' . json_last_error_msg());
    $customer_js_data = '[]';
}

// Fetch products from province_stock for the sidebar and form
$products = [];
error_log("DEBUG create_invoice.php: Fetching products for province_id=$province_id");

// First, let's verify the data exists with a simple query (include reserved-only rows)
$verify_stmt = $conn->prepare("
    SELECT COUNT(*) as cnt 
    FROM province_stock 
    WHERE (? IS NULL OR province_id = ?)
      AND (COALESCE(quantity,0) > 0 OR COALESCE(reserved_quantity,0) > 0)
      AND (unit_type = 'crates' OR unit_type IS NULL OR unit_type = '')
");
if ($verify_stmt) {
    $verify_stmt->bind_param("ii", $province_id, $province_id);
    $verify_stmt->execute();
    $verify_result = $verify_stmt->get_result();
    $verify_row = $verify_result->fetch_assoc();
    error_log("DEBUG create_invoice.php: Direct count query for province_id=$province_id: " . $verify_row['cnt'] . " rows");
    $verify_stmt->close();
}

$stmt = $conn->prepare("
    SELECT 
        p.id, 
        p.name, 
        p.volume_cl, 
        p.price_per_crate, 
        p.crate_quantity,
        COALESCE(CEIL(p.price_per_crate / NULLIF(p.crate_quantity, 0)), 0) AS price,
        COALESCE(s.quantity, 0) AS stock_quantity, 
        COALESCE(s.reserved_quantity, 0) AS reserved_quantity,
        COALESCE(pending.pending_reserved_crates, 0) AS pending_reserved_crates,
        (COALESCE(s.quantity, 0) - COALESCE(s.reserved_quantity, 0)) AS available_quantity,
        s.unit_type AS stock_unit_type
    FROM product p
    INNER JOIN province_stock s ON p.id = s.product_id
    LEFT JOIN (
        SELECT 
            ii.product_id,
            ui.province_id,
            SUM(
                CASE 
                    WHEN p2.crate_quantity IS NULL OR p2.crate_quantity <= 0 THEN ii.quantity
                    ELSE CEIL(ii.quantity / p2.crate_quantity)
                END
            ) AS pending_reserved_crates
        FROM invoice_items ii
        INNER JOIN unpaid_invoices ui ON ui.id = ii.invoice_id
        INNER JOIN product p2 ON p2.id = ii.product_id
        WHERE (? IS NULL OR ui.province_id = ?)
          AND ui.status <> 'paid'
          AND ui.approval_status = 'pending'
        GROUP BY ii.product_id, ui.province_id
        ) pending ON pending.product_id = p.id AND pending.province_id = s.province_id
    WHERE (? IS NULL OR s.province_id = ?)
      AND (s.unit_type = 'crates' OR s.unit_type IS NULL OR s.unit_type = '')
      AND (
            COALESCE(s.quantity,0) > 0 
         OR COALESCE(s.reserved_quantity, 0) > 0
         OR COALESCE(pending.pending_reserved_crates, 0) > 0
      )
    ORDER BY p.name
");
if ($stmt === false) {
    error_log("Prepare failed: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database prepare failed. Check logs.</p>";
    $conn->close();
    exit;
}
$stmt->bind_param("iiii", $province_id, $province_id, $province_id, $province_id);
if (!$stmt->execute()) {
    error_log("Execute failed: " . $stmt->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database execute failed. Check logs.</p>";
    $stmt->close();
    $conn->close();
    exit;
}
$result = $stmt->get_result();
$products = [];
while ($row = $result->fetch_assoc()) {
    $products[] = $row;
}
$stmt->close();
error_log("DEBUG create_invoice.php: Found " . count($products) . " products for province_id=$province_id");
if (count($products) > 0) {
    error_log("DEBUG create_invoice.php: First product: " . json_encode($products[0]));
} else {
    error_log("DEBUG create_invoice.php: No products found - checking why...");
}
// If no products are found, we no longer hard-stop the page; the sidebar will simply be empty
if (empty($products)) {
    // Debug: Check what's actually in province_stock for this province
    $debug_stmt = $conn->prepare("SELECT id, province_id, product_id, quantity, reserved_quantity, unit_type FROM province_stock WHERE province_id = ?");
    if ($debug_stmt) {
        $debug_stmt->bind_param("i", $province_id);
        $debug_stmt->execute();
        $debug_result = $debug_stmt->get_result();
        $debug_rows = $debug_result->fetch_all(MYSQLI_ASSOC);
        error_log("DEBUG create_invoice.php: Raw province_stock data for province_id=$province_id (for sidebar products): " . json_encode($debug_rows));
        $debug_stmt->close();
    }
    error_log("DEBUG create_invoice.php: No products to display in sidebar for province_id=$province_id (no rows with quantity/reserved_quantity > 0).");
}

// Handle customer selection
$errors = [];
$selected_customer = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['select_customer'])) {
    $customer_id = intval($_POST['customer_id'] ?? 0);
    $new_customer = $_POST['new_customer'] ?? 0;
    $full_name = trim($_POST['full_name'] ?? '');
    $nif = trim($_POST['nif'] ?? '');
    $tel = trim($_POST['tel'] ?? '');
    $driver_name = trim($_POST['driver_name'] ?? '');
    $driver_contacts = trim($_POST['driver_contacts'] ?? '');
    $plate_numbers = trim($_POST['plate_numbers'] ?? '');
    $additional_contacts = trim($_POST['additional_contacts'] ?? '');
    $requested_user_type = strtolower(trim($_POST['user_type'] ?? 'physical'));
    $user_type = in_array($requested_user_type, ['moral', 'physical'], true) ? $requested_user_type : 'physical';

    if ($new_customer) {
        if (empty($full_name)) {
            $errors[] = "Le nom complet est requis pour un nouveau client.";
        } else {
            // Avoid double insert if this request already created the customer (id carried in hidden field)
            $existing_new_id = isset($_POST['new_customer_id']) ? (int)$_POST['new_customer_id'] : 0;
            if ($existing_new_id > 0) {
                $customer_id = $existing_new_id;
            } else {
                $stmt = $conn->prepare("INSERT INTO customer (full_name, nif, tel, driver_name, driver_contacts, plate_numbers, additional_contacts, user_type, is_active) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1)");
                if ($stmt === false) {
                    error_log("Prepare failed: " . $conn->error);
                    $errors[] = "Échec de la préparation de la base de données.";
                } else {
                    $stmt->bind_param("ssssssss", $full_name, $nif, $tel, $driver_name, $driver_contacts, $plate_numbers, $additional_contacts, $user_type);
                    if ($stmt->execute()) {
                        $customer_id = $conn->insert_id;
                        $_POST['new_customer_id'] = $customer_id; // prevent duplicate on accidental resubmit
                    } else {
                        error_log("Execute failed: " . $stmt->error);
                        $errors[] = "Échec de la création du nouveau client.";
                    }
                    $stmt->close();
                }
            }
        }
    }

    if (!$new_customer && $customer_id <= 0) {
        $search_name = trim($_POST['customer_search'] ?? '');
        if ($search_name !== '') {
            $stmt = $conn->prepare("SELECT id FROM customer WHERE full_name = ? AND COALESCE(is_active,1)=1 ORDER BY id ASC LIMIT 1");
            if ($stmt) {
                $stmt->bind_param("s", $search_name);
                if ($stmt->execute()) {
                    $row = $stmt->get_result()->fetch_assoc();
                    if ($row) {
                        $customer_id = (int)$row['id'];
                    }
                }
                $stmt->close();
            }
        }
        if ($customer_id <= 0) {
        $errors[] = "Veuillez sélectionner ou ajouter un client.";
        }
    }

    if (empty($errors) && $customer_id > 0) {
        $stmt = $conn->prepare("
            SELECT id, full_name, nif, tel, driver_name, driver_contacts, plate_numbers, additional_contacts,
                   can_pay_by_loan, loan_limit_bif, loan_balance_bif, loan_terms_days, loan_status,
                   COALESCE(is_active, 1) as is_active
            FROM customer
            WHERE id = ?
            LIMIT 1
        ");
        if ($stmt === false) {
            error_log("Prepare failed: " . $conn->error);
            $errors[] = "Échec de la préparation de la base de données.";
        } else {
            $stmt->bind_param("i", $customer_id);
            if ($stmt->execute()) {
                $selected_customer = $stmt->get_result()->fetch_assoc();
                if (!$selected_customer) {
                    $errors[] = "Client sélectionné introuvable.";
                } else {
                    // Check if customer is active
                    $is_active = isset($selected_customer['is_active']) ? (int)$selected_customer['is_active'] : 1;
                    if ($is_active == 0) {
                        $errors[] = "Ce client a été désactivé par le DGA. Vous ne pouvez pas créer de commandes pour des clients désactivés. Veuillez contacter l'administrateur.";
                        $selected_customer = null; // Prevent setting in session
                    } else {
                        $_SESSION['selected_customer'] = $selected_customer;
                    }
                }
            } else {
                error_log("Execute failed: " . $stmt->error);
                $errors[] = "Échec de la récupération des détails du client.";
            }
            $stmt->close();
        }
    }

    if (empty($errors) && $selected_customer) {
        $_SESSION['selected_customer'] = $selected_customer;
    }
}

// Handle invoice creation (only if customer is selected)
$invoice_errors = [];
$success = '';

if (isset($_SESSION['invoice_success'])) {
    $success = $_SESSION['invoice_success'];
    unset($_SESSION['invoice_success']);
}

$full_name = '';
$tel = '';
$ledger_outstanding = 0;
if (isset($_SESSION['selected_customer'])) {
    $full_name = $_SESSION['selected_customer']['full_name'];
    $tel = $_SESSION['selected_customer']['tel'] ?? '';
    $customer_id = isset($_SESSION['selected_customer']['id']) ? (int)$_SESSION['selected_customer']['id'] : 0;
    
    // Calculate actual outstanding balance from customer_loan_ledger (source of truth)
    if ($customer_id > 0) {
        $ledgerStmt = $conn->prepare("
            SELECT 
                COALESCE(GREATEST(
                    COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
                    COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
                    0
                ), 0) AS ledger_outstanding
            FROM customer c
            LEFT JOIN customer_loan_ledger cll ON (
                cll.customer_id = c.id
                AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
            )
            WHERE c.id = ?
            GROUP BY c.id
        ");
        if ($ledgerStmt) {
            $ledgerStmt->bind_param("i", $customer_id);
            if ($ledgerStmt->execute()) {
                $ledgerResult = $ledgerStmt->get_result();
                if ($ledgerRow = $ledgerResult->fetch_assoc()) {
                    $ledger_outstanding = (int)($ledgerRow['ledger_outstanding'] ?? 0);
                }
            }
            $ledgerStmt->close();
        }
    }
}
$loan_profile = [
    'can_pay_by_loan' => isset($_SESSION['selected_customer']['can_pay_by_loan']) ? (int)$_SESSION['selected_customer']['can_pay_by_loan'] : 0,
    'loan_limit_bif' => isset($_SESSION['selected_customer']['loan_limit_bif']) ? (int)$_SESSION['selected_customer']['loan_limit_bif'] : 0,
    'loan_balance_bif' => isset($_SESSION['selected_customer']['loan_balance_bif']) ? (int)$_SESSION['selected_customer']['loan_balance_bif'] : 0,
    'loan_terms_days' => isset($_SESSION['selected_customer']['loan_terms_days']) ? (int)$_SESSION['selected_customer']['loan_terms_days'] : 0,
    'loan_status' => $_SESSION['selected_customer']['loan_status'] ?? 'inactive',
    'ledger_outstanding' => $ledger_outstanding
];
$loan_remaining_bif = max(0, $loan_profile['loan_limit_bif'] - $loan_profile['ledger_outstanding']);
if (isset($_SESSION['selected_customer']) && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_submit'])) {
    error_log("Form submitted at " . date('Y-m-d H:i:s'));
    error_log("Full POST data: " . print_r($_POST, true));
    $customer_id = $_SESSION['selected_customer']['id'];
    $nif = $_SESSION['selected_customer']['nif'] ?? '';
    $products_selected = $_POST['products'] ?? [];
    $quantities = $_POST['quantities'] ?? [];

    // Validation
    if (empty($products_selected) || empty($quantities)) {
        $invoice_errors[] = "Au moins un produit et une quantité doivent être sélectionnés.";
    }
    if (count($products_selected) !== count($quantities)) {
        $invoice_errors[] = "Désaccord entre les produits et les quantités.";
    }
    
    // Check for duplicate products
    $unique_products = array_unique($products_selected);
    if (count($unique_products) !== count($products_selected)) {
        $duplicate_products = array_diff_assoc($products_selected, $unique_products);
        foreach ($duplicate_products as $dup) {
            $invoice_errors[] = "Le produit '$dup' apparaît plus d'une fois dans la commande. Veuillez combiner les quantités ou supprimer les doublons.";
        }
    }

    $valid_products = [];
    $total_amount = 0;
    for ($i = 0; $i < count($products_selected); $i++) {
        $product_name = $products_selected[$i];
        $quantity = isset($quantities[$i]) ? (int)$quantities[$i] : 0;
        // Always use crates
        $unit_type = 'crates';
        error_log("Processing item $i: product_name=$product_name, quantity=$quantity crates");

        $product = array_filter($products, fn($p) => $p['name'] === $product_name);
        $product = array_values($product)[0] ?? null;
        if (!$product) {
            $invoice_errors[] = "Product $product_name not found in available stock.";
            continue;
        }
        $crate_quantity = max(1, (int)$product['crate_quantity']);

        if ($quantity <= 0) {
            $invoice_errors[] = "Quantity for product $product_name must be greater than 0. Requested: $quantity crates";
            continue;
        }

        $product_id = $product['id'];
        error_log("Mapped: product_name=$product_name, product_id=$product_id, quantity=$quantity crates");

        try {
            $stmt = $conn->prepare("
                SELECT p.id, p.name, p.price_per_crate, p.crate_quantity,
                       COALESCE(CEIL(p.price_per_crate / NULLIF(p.crate_quantity, 0)), 0) AS price,
                       s.quantity AS stock_quantity,
                       COALESCE(s.reserved_quantity, 0) AS reserved_quantity,
                       (s.quantity - COALESCE(s.reserved_quantity, 0)) AS available_quantity,
                       s.unit_type AS stock_unit_type,
                       s.id AS province_stock_id
                FROM product p
                INNER JOIN province_stock s ON p.id = s.product_id
                WHERE p.id = ? AND s.province_id = ? AND s.unit_type = 'crates'
            ");
            if ($stmt === false) {
                throw new Exception("Prepare failed: " . $conn->error);
            }
            $stmt->bind_param("ii", $product_id, $province_id);
            if (!$stmt->execute()) {
                throw new Exception("Execute failed: " . $stmt->error);
            }
            $validated_product = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            if (!$validated_product) {
                $invoice_errors[] = "Product ID $product_id ($product_name) not found or out of stock in province stock.";
            } else {
                // Quantity is already in crates
                $quantity_in_crates = $quantity;
                $available_quantity = (int)$validated_product['available_quantity'];
                
                if ($validated_product['stock_unit_type'] !== 'crates') {
                    $invoice_errors[] = "Stock for {$validated_product['name']} is not in crates.";
                } elseif ($available_quantity < $quantity_in_crates) {
                    $invoice_errors[] = "Stock insuffisant pour {$validated_product['name']}. Disponible: $available_quantity caisses, Demandé: $quantity_in_crates caisses.";
                } else {
                    $system_price_per_crate = (int)$validated_product['price_per_crate'];

                    $valid_products[] = [
                        'id' => $validated_product['id'],
                        'name' => $validated_product['name'],
                        'price' => $system_price_per_crate,
                        'quantity' => $quantity_in_crates,
                        'quantity_in_crates' => $quantity_in_crates,
                        'crate_quantity' => $validated_product['crate_quantity'],
                        'province_stock_id' => (int)$validated_product['province_stock_id'],
                        'price_source' => 'system',
                        'system_price' => $system_price_per_crate
                    ];
                    $total_amount += $system_price_per_crate * $quantity_in_crates;
                }
            }
        } catch (Exception $e) {
            error_log("Validation error for product_id $product_id: " . $e->getMessage());
            $invoice_errors[] = "Database error validating product $product_name: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
        }
    }

    $is_loan_sale = isset($_POST['is_loan_sale']) && $_POST['is_loan_sale'] === '1';
    $loan_note = trim($_POST['loan_note'] ?? '');
    $loan_due_date_input = trim($_POST['loan_due_date'] ?? '');
    $loan_due_date_string = null;
    if ($loan_due_date_input !== '') {
        $loan_due_date_obj = DateTime::createFromFormat('Y-m-d', $loan_due_date_input);
        if ($loan_due_date_obj === false) {
            $invoice_errors[] = "Invalid loan due date format. Use YYYY-MM-DD.";
        } else {
            $loan_due_date_string = $loan_due_date_obj->format('Y-m-d');
        }
    }
    if (mb_strlen($loan_note) > 500) {
        $invoice_errors[] = "Loan note must be 500 characters or less.";
    }


    $assigned_cashier_id = null;
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $assigned_cashier_id_input = filter_input(INPUT_POST, 'assigned_cashier_id', FILTER_VALIDATE_INT);
        if ($requires_cashier_selection) {
            if (!$assigned_cashier_id_input || !in_array($assigned_cashier_id_input, $cashier_ids, true)) {
                $invoice_errors[] = "Veuillez sélectionner un caissier valide pour traiter la commande.";
            } else {
                $assigned_cashier_id = $assigned_cashier_id_input;
            }
        } else {
            $assigned_cashier_id = $_SESSION['user_id'];
        }
    }

    if (empty($invoice_errors)) {
        $invoice_number = 'INV-' . date('Ymd') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        $paid_amount = $total_amount; // Set paid_amount to calculated total
        error_log("DEBUG create_invoice.php: Calculated total_amount = " . number_format($total_amount, 0, ',', '.') . " BIF, paid_amount = " . number_format($paid_amount, 0, ',', '.') . " BIF");

        $customer_name = $full_name;
        $phone_number = $tel;
        $status = 'pending';
        $rc = '';
        $approved_by = null;
        $approval_status = 'pending';
        $approved_at = null;
        $created_by = $_SESSION['user_id'];
        $loan_outstanding_bif = $is_loan_sale ? $total_amount : 0;
        $loan_status_value = $is_loan_sale ? 'pending' : null;
        $loan_note_trimmed = $loan_note !== '' ? mb_substr($loan_note, 0, 500) : null;

    mbh_ensure_loan_ledger_entry_origin($conn);
    $conn->begin_transaction();
        try {
            if ($is_loan_sale) {
                $stmtLoanProfile = $conn->prepare("SELECT can_pay_by_loan, loan_limit_bif, loan_balance_bif, loan_terms_days FROM customer WHERE id = ? FOR UPDATE");
                if ($stmtLoanProfile === false) {
                    throw new Exception("Failed to prepare loan profile lookup: " . $conn->error);
                }
                $stmtLoanProfile->bind_param("i", $customer_id);
                if (!$stmtLoanProfile->execute()) {
                    throw new Exception("Failed to fetch loan profile: " . $stmtLoanProfile->error);
                }
                $loanProfileRow = $stmtLoanProfile->get_result()->fetch_assoc();
                $stmtLoanProfile->close();
                if (!$loanProfileRow || (int)$loanProfileRow['can_pay_by_loan'] !== 1) {
                    throw new Exception("Customer is not authorized for loan sales.");
                }
                $loan_limit = (int)($loanProfileRow['loan_limit_bif'] ?? 0);
                $loan_balance = (int)($loanProfileRow['loan_balance_bif'] ?? 0);
                $loan_terms_days_db = (int)($loanProfileRow['loan_terms_days'] ?? 0);
                $loan_remaining_limit = $loan_limit > 0 ? $loan_limit - $loan_balance : PHP_INT_MAX;
                if ($loan_limit > 0 && $loan_outstanding_bif > $loan_remaining_limit) {
                    throw new Exception("Loan sale exceeds the customer's remaining loan limit. Remaining: " . number_format($loan_remaining_limit, 0, ',', '.') . " BIF");
                }
                if ($loan_due_date_string === null && $loan_terms_days_db > 0) {
                    $loan_due_date_string = date('Y-m-d', strtotime('+' . $loan_terms_days_db . ' days'));
                }
            }

            // Set expiration time (3 hours from now)
            $expires_at = date('Y-m-d H:i:s', strtotime('+3 hours'));

            // Insert into unpaid_invoices
            $assigned_cashier_id = $selected_assigned_cashier_id ?? $assigned_cashier_to_save;
            $invoice_columns = [
                'customer_id',
                'is_loan_sale',
                'loan_due_date',
                'loan_outstanding_bif',
                'loan_status',
                'stock_manager_id',
                'status',
                'customer_name',
                'phone_number',
                'province_id',
                'nif',
                'rc',
                'invoice_number',
                'paid_amount',
                'approved_by',
                'approval_status',
                'approved_at',
                'expires_at',
                'created_by'
            ];
            $placeholders = array_fill(0, count($invoice_columns), '?');
            $types = 'iisisisssisssissssi';
            $values = [
                $customer_id,
                $is_loan_sale,
                $loan_due_date_string,
                $loan_outstanding_bif,
                $loan_status_value,
                $stock_manager_id,
                $status,
                $customer_name,
                $phone_number,
                $province_id,
                $nif,
                $rc,
                $invoice_number,
                $paid_amount,
                $approved_by,
                $approval_status,
                $approved_at,
                $expires_at,
                $created_by
            ];
            if ($has_assigned_cashier_column && $assigned_cashier_id !== null) {
                $invoice_columns[] = 'assigned_cashier_id';
                $placeholders[] = '?';
                $types .= 'i';
                $values[] = $assigned_cashier_id;
            }
            $sql = "INSERT INTO unpaid_invoices (" . implode(', ', $invoice_columns) . ") VALUES (" . implode(', ', $placeholders) . ")";
            $stmt = $conn->prepare($sql);
            if ($stmt === false) {
                throw new Exception("Prepare failed: " . $conn->error);
            }
            $bindParams = [$types];
            foreach ($values as $k => $v) {
                $bindParams[] = &$values[$k];
            }
            call_user_func_array([$stmt, 'bind_param'], $bindParams);
            if (!$stmt->execute()) {
                throw new Exception("Execute failed: " . $stmt->error);
            }
            $invoice_id = $conn->insert_id;
            error_log("Inserted unpaid invoice with ID: $invoice_id, paid_amount: $paid_amount");
            $stmt->close();

            // Ensure invoice_items.id has AUTO_INCREMENT before inserting
            try {
                // Use LIKE syntax which is supported in MySQL
                $check_items_id = $conn->query("SHOW COLUMNS FROM invoice_items LIKE 'id'");
                if ($check_items_id && $check_items_id->num_rows > 0) {
                    $items_id_col = $check_items_id->fetch_assoc();
                    $items_has_auto_increment = isset($items_id_col['Extra']) && stripos($items_id_col['Extra'], 'auto_increment') !== false;
                    if (!$items_has_auto_increment) {
                        error_log("⚠️ invoice_items.id does not have AUTO_INCREMENT, attempting to fix...");
                        $items_col_type = $items_id_col['Type'] ?? 'INT(11)';
                        $items_col_null = ($items_id_col['Null'] ?? 'NO') === 'YES' ? 'NULL' : 'NOT NULL';
                        
                        // Check if it's a primary key
                        $check_pk = $conn->query("SHOW KEYS FROM invoice_items WHERE Key_name = 'PRIMARY' AND Column_name = 'id'");
                        $has_pk = $check_pk && $check_pk->num_rows > 0;
                        
                        if ($has_pk) {
                            // It's already a primary key, just add AUTO_INCREMENT
                            $alter_items_query = "ALTER TABLE invoice_items MODIFY COLUMN id {$items_col_type} {$items_col_null} AUTO_INCREMENT";
                            if ($conn->query($alter_items_query)) {
                                error_log("✅ Successfully added AUTO_INCREMENT to invoice_items.id");
                            } else {
                                error_log("❌ Failed to add AUTO_INCREMENT to invoice_items.id: " . $conn->error);
                            }
                        } else {
                            // Not a primary key, make it one first, then add AUTO_INCREMENT
                            if ($conn->query("ALTER TABLE invoice_items ADD PRIMARY KEY (id)")) {
                                error_log("✅ Added PRIMARY KEY to invoice_items.id");
                                $alter_items_query = "ALTER TABLE invoice_items MODIFY COLUMN id {$items_col_type} {$items_col_null} AUTO_INCREMENT";
                                if ($conn->query($alter_items_query)) {
                                    error_log("✅ Successfully added AUTO_INCREMENT to invoice_items.id");
                                } else {
                                    error_log("❌ Failed to add AUTO_INCREMENT to invoice_items.id: " . $conn->error);
                                }
                            } else {
                                error_log("❌ Failed to add PRIMARY KEY to invoice_items.id: " . $conn->error);
                            }
                        }
                    } else {
                        error_log("✅ invoice_items.id already has AUTO_INCREMENT");
                    }
                } else {
                    error_log("⚠️ invoice_items table or id column not found");
                }
            } catch (Exception $e) {
                error_log("⚠️ Error checking invoice_items.id AUTO_INCREMENT: " . $e->getMessage());
            }

            // Insert into invoice_items
            $stmt = $conn->prepare("
            INSERT INTO invoice_items (
                    invoice_id, product_id, quantity, unit_price, price_source
                ) VALUES (?, ?, ?, ?, ?)
            ");
            if ($stmt === false) {
                throw new Exception("Prepare failed for invoice_items: " . $conn->error);
            }
            $price_source = 'system'; // Must be a variable for bind_param
            foreach ($valid_products as $product) {
                $unitPriceApplied = (int)$product['price'];

                $stmt->bind_param(
                    "iiiis",
                    $invoice_id,
                    $product['id'],
                    $product['quantity'],
                    $unitPriceApplied,
                    $price_source
                );
                if (!$stmt->execute()) {
                    $error_msg = "Execute failed for invoice_items: " . $stmt->error . " (Product ID: {$product['id']}, Invoice ID: $invoice_id)";
                    error_log("❌ " . $error_msg);
                    throw new Exception($error_msg);
                }
            }
            $stmt->close();
            error_log("✅ Successfully inserted " . count($valid_products) . " items into invoice_items");


            // Reserve stock (temporarily remove from available stock)
            foreach ($valid_products as $product) {
                $quantity_to_reserve = (int)$product['quantity_in_crates']; // Always in crates
                $province_stock_id = (int)$product['province_stock_id'];
                $product_id = (int)$product['id'];
                
                error_log("Attempting to reserve stock: product_id=$product_id, province_stock_id=$province_stock_id, quantity_to_reserve=$quantity_to_reserve crates, province_id=$province_id");
                
                // First, verify the current stock state
                $check_stmt = $conn->prepare("
                    SELECT quantity, reserved_quantity, (quantity - COALESCE(reserved_quantity, 0)) AS available_quantity
                    FROM province_stock
                    WHERE id = ? AND product_id = ? AND province_id = ? AND unit_type = 'crates'
                ");
                if ($check_stmt) {
                    $check_stmt->bind_param("iii", $province_stock_id, $product_id, $province_id);
                    $check_stmt->execute();
                    $stock_check = $check_stmt->get_result()->fetch_assoc();
                    $check_stmt->close();
                    if ($stock_check) {
                        error_log("Current stock state: quantity={$stock_check['quantity']}, reserved_quantity={$stock_check['reserved_quantity']}, available_quantity={$stock_check['available_quantity']}");
                    }
                }
                
                // Reserve stock by increasing reserved_quantity and checking available quantity
                // Use COALESCE to handle NULL reserved_quantity values
                $stmt = $conn->prepare("
                    UPDATE province_stock 
                    SET reserved_quantity = COALESCE(reserved_quantity, 0) + ?,
                        last_updated = NOW()
                    WHERE id = ? 
                      AND product_id = ? 
                      AND province_id = ? 
                      AND unit_type = 'crates'
                      AND (quantity - COALESCE(reserved_quantity, 0)) >= ?
                    ");
                if ($stmt === false) {
                    $error_msg = "Prepare failed for stock reservation: " . $conn->error;
                    error_log("❌ " . $error_msg);
                    throw new Exception($error_msg);
                }
                $stmt->bind_param("iiiii", $quantity_to_reserve, $province_stock_id, $product_id, $province_id, $quantity_to_reserve);
                if (!$stmt->execute()) {
                    $error_msg = "Execute failed for stock reservation: " . $stmt->error;
                    error_log("❌ " . $error_msg);
                    $stmt->close();
                    throw new Exception($error_msg);
                }
                
                $affected_rows = $stmt->affected_rows;
                $stmt->close();
                
                if ($affected_rows === 0) {
                    // Verify stock state after failed reservation
                    $verify_stmt = $conn->prepare("
                        SELECT quantity, COALESCE(reserved_quantity, 0) AS reserved_quantity, 
                               (quantity - COALESCE(reserved_quantity, 0)) AS available_quantity
                        FROM province_stock
                        WHERE id = ? AND product_id = ? AND province_id = ? AND unit_type = 'crates'
                    ");
                    if ($verify_stmt) {
                        $verify_stmt->bind_param("iii", $province_stock_id, $product_id, $province_id);
                        $verify_stmt->execute();
                        $verify_result = $verify_stmt->get_result()->fetch_assoc();
                        $verify_stmt->close();
                        if ($verify_result) {
                            $error_msg = "Failed to reserve stock: No rows updated. Product ID: $product_id, Province Stock ID: $province_stock_id, Quantity to reserve: $quantity_to_reserve crates. Current state: quantity={$verify_result['quantity']}, reserved_quantity={$verify_result['reserved_quantity']}, available_quantity={$verify_result['available_quantity']}. Available stock may have changed or the WHERE conditions did not match.";
                        } else {
                            $error_msg = "Failed to reserve stock: Product ID: $product_id, Province Stock ID: $province_stock_id not found in province_stock.";
                        }
                    } else {
                        $error_msg = "Failed to reserve stock: No rows updated. Product ID: $product_id, Province Stock ID: $province_stock_id, Quantity to reserve: $quantity_to_reserve crates.";
                    }
                    error_log("❌ " . $error_msg);
                    throw new Exception($error_msg);
            }

                // Verify the reservation was successful
                $verify_after = $conn->prepare("
                    SELECT quantity, COALESCE(reserved_quantity, 0) AS reserved_quantity, 
                           (quantity - COALESCE(reserved_quantity, 0)) AS available_quantity
                    FROM province_stock
                    WHERE id = ? AND product_id = ? AND province_id = ? AND unit_type = 'crates'
                ");
                if ($verify_after) {
                    $verify_after->bind_param("iii", $province_stock_id, $product_id, $province_id);
                    $verify_after->execute();
                    $verify_after_result = $verify_after->get_result()->fetch_assoc();
                    $verify_after->close();
                    if ($verify_after_result) {
                        error_log("✅ Stock reservation verified: product_id=$product_id, quantity={$verify_after_result['quantity']}, reserved_quantity={$verify_after_result['reserved_quantity']}, available_quantity={$verify_after_result['available_quantity']}");
                    }
                }
                
                error_log("✅ Successfully reserved {$quantity_to_reserve} crates of product {$product_id} (province_stock_id: $province_stock_id) for invoice {$invoice_id} (affected_rows: $affected_rows)");
            }

            if ($is_loan_sale) {
            $stmtLedger = $conn->prepare("
                    INSERT INTO customer_loan_ledger (
                        customer_id, entry_type, direction, entry_origin, amount_bif, reference_table, reference_id, notes, created_by
                    ) VALUES (?, 'loan_sale', 'debit', 'loan_creation', ?, 'unpaid_invoices', ?, ?, ?)
                ");
                if ($stmtLedger === false) {
                    throw new Exception("Failed to prepare loan ledger entry: " . $conn->error);
                }
                $stmtLedger->bind_param(
                    "isisi",
                    $customer_id,
                    $loan_outstanding_bif,
                    $invoice_id,
                    $loan_note_trimmed,
                    $created_by
                );
                if (!$stmtLedger->execute()) {
                    throw new Exception("Failed to record loan ledger entry: " . $stmtLedger->error);
                }
                $ledger_entry_id = $conn->insert_id;
                $stmtLedger->close();

                $stmtUpdateCustomer = $conn->prepare("
                    UPDATE customer
                    SET loan_balance_bif = loan_balance_bif + ?, loan_last_review_at = NOW(),
                        loan_status = CASE WHEN loan_status = 'suspended' THEN loan_status ELSE 'active' END
                    WHERE id = ?
                ");
                if ($stmtUpdateCustomer === false) {
                    throw new Exception("Failed to prepare customer loan balance update: " . $conn->error);
                }
                $stmtUpdateCustomer->bind_param("si", $loan_outstanding_bif, $customer_id);
                if (!$stmtUpdateCustomer->execute()) {
                    throw new Exception("Failed to update customer loan balance: " . $stmtUpdateCustomer->error);
                }
                $stmtUpdateCustomer->close();
            }

            // Log entry
            $stmt = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'create_invoice', ?, NOW())");
            if ($stmt === false) {
                throw new Exception("Prepare failed: " . $conn->error);
            }
            $description = "Created unpaid invoice with number $invoice_number for customer $customer_name, total amount " . number_format($total_amount, 2) . " BIF";
            if ($is_loan_sale) {
                $description .= " [Loan sale]";
            }
            $stmt->bind_param("is", $_SESSION['user_id'], $description);
            if (!$stmt->execute()) {
                throw new Exception("Execute failed: " . $stmt->error);
            }
            $stmt->close();

            $conn->commit();
            $success_msg = "Unpaid invoice <a href='/masunzu_bar_hotel/modules/cashiers/view_invoice.php?id=$invoice_id' style='color: #FFFFFF; text-decoration: underline;'>$invoice_number</a> created successfully and is pending payment processing. Click to view.";
            if ($is_loan_sale) {
                $success_msg = "Loan invoice <a href='/masunzu_bar_hotel/modules/cashiers/view_invoice.php?id=$invoice_id' style='color: #FFFFFF; text-decoration: underline;'>$invoice_number</a> created successfully. The loan balance has been updated. Click to view.";
            }
            error_log("Success message set: $success_msg");
            $_SESSION['invoice_success'] = $success_msg;
            unset($_SESSION['selected_customer']);
            $conn->close();
            header("Location: /masunzu_bar_hotel/modules/cashiers/create_invoice.php");
            exit;
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Unpaid invoice creation failed: " . $e->getMessage());
            $invoice_errors[] = "Échec de la création de la commande: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
        }
    } else {
        $invoice_errors[] = "Échec de la validation: Veuillez vérifier les détails des produits.";
    }
}

include __DIR__ . '/../../includes/header.php';
?>
<?php if (in_array($role['name'], $marketing_roles_without_province, true)): ?>
    <div style="padding:0 40px 10px;">
        <form method="post" style="display:flex; align-items:center; gap:10px; flex-wrap:wrap;">
            <label for="province_id_select" style="font-weight:600; color:#4B2F1F;">Dépot à charger :</label>
            <select id="province_id_select" name="province_id" required style="padding:8px 12px; border:1px solid #4B2F1F; border-radius:8px;">
                <option value="">Choisissez un dépôt</option>
                <?php foreach ($province_options as $province_option): ?>
                    <option value="<?php echo (int)$province_option['id']; ?>" <?php echo ((int)$province_id === (int)$province_option['id']) ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($province_option['name'], ENT_QUOTES, 'UTF-8'); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <button type="submit" style="background:#4B2F1F;color:#F4F0E4;border:none;padding:8px 18px;border-radius:8px;cursor:pointer;">Actualiser dépôt</button>
        </form>
    </div>
<?php endif; ?>
<div class="create-invoice-container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 70vh; display: flex; gap: 20px;">
    <div class="product-sidebar" style="flex: 0.5; background-color: #F4F0E4; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-height: 80vh; overflow-y: auto; min-width: 200px;">
        <h3 style="color: #4B2F1F; font-size: 20px; margin-bottom: 15px;">Produits (Disponibles & Réservés)</h3>
        <input type="text" id="product_search_sidebar" placeholder="Rechercher produits..." style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #FFFFFF; font-size: 14px; color: #4B2F1F; margin-bottom: 15px;">
        <ul id="product_list" style="list-style: none; padding: 0; margin: 0;">
            <?php foreach ($products as $product): ?>
                <?php
                    // stock_quantity is already in crates (from province_stock with unit_type='crates')
                    $total_crates = (int)$product['stock_quantity'];
                    $reserved_crates = (int)($product['reserved_quantity'] ?? 0);
                    $pending_reserved_crates = (int)($product['pending_reserved_crates'] ?? 0);
                    $display_reserved_crates = max($reserved_crates, $pending_reserved_crates);
                    $available_crates = (int)($product['available_quantity'] ?? 0);
                    $crate_quantity = max((int)$product['crate_quantity'], 1);
                    $price_per_crate = isset($product['price_per_crate']) ? (int)$product['price_per_crate'] : ($product['price'] * $crate_quantity);

                    $total_crates_label = number_format($total_crates, 0, ',', '.');
                    $available_crates_label = number_format($available_crates, 0, ',', '.');
                    $crate_quantity_label = number_format($crate_quantity, 0, ',', '.');

                    $volume_label = isset($product['volume_cl']) && $product['volume_cl'] !== null
                        ? trim(number_format((int)$product['volume_cl'], 0, ',', '.') . ' cl')
                        : '';

                    $title_segments = [];
                    $title_segments[] = $product['name'];
                    if ($volume_label !== '') {
                        $title_segments[] = $volume_label;
                    }
                    $title_text = implode(' ', array_filter($title_segments));

                    // Build stock display with highlighted "Available" text
                    $stock_html_parts = [];
                    if ($available_crates > 0) {
                        $stock_html_parts[] = '<span style="color: #2E7D32; font-weight: 700; font-size: 15px;">Disponible: <span style="color: #1B5E20; font-weight: 800; font-size: 16px;">' . $available_crates_label . '</span> caisses</span>';
                    }
                    if ($display_reserved_crates > 0) {
                        $reserved_label = number_format($display_reserved_crates, 0, ',', '.');
                        $reserved_suffix = $pending_reserved_crates > $reserved_crates
                            ? ' (en attente d\'approbation)'
                            : '';
                        $stock_html_parts[] = '<span style="color: #856404;">Réservé: ' . $reserved_label . ' caisses' . $reserved_suffix . '</span>';
                    }
                    if (empty($stock_html_parts)) {
                        $stock_html_parts[] = '<span>Total: ' . $total_crates_label . ' caisses</span>';
                    } else {
                        $stock_html_parts[] = '<span style="color: #666;">Total: ' . $total_crates_label . ' caisses</span>';
                    }
                    $stock_html = implode(' <span style="color: #999;">|</span> ', $stock_html_parts);
                    
                    // Determine if product is fully reserved
                    $is_fully_reserved = $available_crates == 0 && $display_reserved_crates > 0;
                ?>
                <li style="padding: 12px; border-bottom: 1px solid #D3D3D3; color: #4B2F1F; font-size: 14px; <?php echo $is_fully_reserved ? 'background-color: #FFF3CD; border-left: 3px solid #FFC107;' : ''; ?>">
                    <div style="display: flex; justify-content: space-between; gap: 10px; flex-wrap: wrap; align-items: baseline;">
                        <strong><?php echo htmlspecialchars($title_text, ENT_QUOTES, 'UTF-8'); ?></strong>
                        <span style="color: #2F4F4F;"><?php echo htmlspecialchars($crate_quantity_label . ' unités par caisse', ENT_QUOTES, 'UTF-8'); ?></span>
                    </div>
                    <div><?php echo $stock_html; ?></div>
                    <?php if ($is_fully_reserved): ?>
                        <br><span style="color: #856404; font-weight: 600; font-size: 12px;">⚠️ Entièrement Réservé (dans commandes non payées/non approuvées)</span>
                    <?php endif; ?>
                    <br><span>Prix par caisse: <?php echo number_format($price_per_crate, 2, ',', '.'); ?> BIF</span><br>
                </li>
            <?php endforeach; ?>
        </ul>
    </div>
    <div class="invoice-content" style="flex: 2; background-color: #F4F0E4; padding: 30px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 1000px; margin: 0 auto;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px;">Créer Commande Gros</h2>
        <p style="color: #000000; font-size: 16px;">Créer une nouvelle commande gros pour Masunzu Bar Hotel.</p>
        <?php if ($success): ?>
            <div style="background: linear-gradient(135deg, #2E7D32 0%, #66BB6A 100%); color: #FFFFFF; padding: 20px; border-radius: 12px; box-shadow: 0 6px 18px rgba(46, 125, 50, 0.35); margin-bottom: 20px; display: flex; gap: 16px; align-items: center;">
                <div style="font-size: 36px; line-height: 1;">✅</div>
                <div style="flex: 1;">
                    <p style="margin: 0; font-size: 18px; font-weight: 600;">Commande Créée avec Succès</p>
                    <p style="margin: 6px 0 0; font-size: 16px;"><?php echo $success; ?></p>
                </div>
                <div style="display: flex; flex-direction: column; gap: 8px;">
                    <a href="/masunzu_bar_hotel/dashboards/cashier_dashboard.php" style="background-color: #FFFFFF; color: #2E7D32; padding: 10px 16px; border-radius: 8px; text-decoration: none; font-weight: 600; text-align: center;">Aller au Tableau de Bord</a>
                    <a href="/masunzu_bar_hotel/modules/cashiers/create_invoice.php" style="background-color: rgba(255,255,255,0.2); color: #FFFFFF; padding: 10px 16px; border-radius: 8px; text-decoration: none; font-weight: 600; text-align: center;">Créer une Autre</a>
                </div>
            </div>
        <?php endif; ?>

        <?php if (!isset($_SESSION['selected_customer'])): ?>
            <!-- Step 1: Customer Selection -->
            <?php if (!empty($errors)): ?>
                <div style="background-color: #FFFFFF; color: #FF0000; padding: 10px; border-radius: 5px; border: 1px solid #FF0000; margin-bottom: 15px;">
                    <?php foreach ($errors as $error): ?>
                        <p><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            <form method="POST" id="customer-select-form" style="margin-top: 20px; display: grid; grid-template-columns: 1fr; gap: 20px;">
                <input type="hidden" name="new_customer_id" value="<?php echo isset($_POST['new_customer_id']) ? (int)$_POST['new_customer_id'] : 0; ?>">
                <div class="customer-section" style="position: relative; width: 100%; margin-bottom: 20px;">
                    <label for="customer_search" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Rechercher ou Sélectionner un Client:</label>
                    <div style="position: relative; width: 100%;">
                        <input type="text" id="customer_search" name="customer_search" list="customer_options" placeholder="Commencez à taper le nom du client..." style="width: 100%; padding: 12px 40px 12px 12px; border-radius: 8px; border: 2px solid #4B2F1F; background-color: #FFFFFF; font-size: 16px; color: #4B2F1F; height: 45px; box-sizing: border-box;" autocomplete="off">
                        <span style="position: absolute; right: 12px; top: 50%; transform: translateY(-50%); color: #4B2F1F; font-size: 18px;">🔍</span>
                        <div id="customer_dropdown" style="display: none; position: absolute; top: calc(100% + 5px); left: 0; right: 0; background-color: #FFFFFF; border: 2px solid #4B2F1F; border-radius: 8px; max-height: 300px; overflow-y: auto; z-index: 9999; box-shadow: 0 6px 12px rgba(0,0,0,0.15);">
                        <!-- Dropdown items will be populated by JavaScript -->
                    </div>
                    </div>
                    <datalist id="customer_options">
                        <?php foreach ($customers as $customer): ?>
                            <option value="<?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?>" data-id="<?php echo (int)$customer['id']; ?>">
                                <?php echo htmlspecialchars(trim($customer['full_name']) . ($customer['nif'] ? ' - NIF: ' . $customer['nif'] : ''), ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </datalist>
                    <input type="hidden" id="customer_id" name="customer_id" value="">
                    <div id="selected_customer_display" style="margin-top: 10px; padding: 10px; background-color: #F4F0E4; border-radius: 5px; display: none;">
                        <strong>Sélectionné:</strong> <span id="selected_customer_name"></span>
                    </div>
                </div>
                <div>
                    <label for="new_customer" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Ou Ajouter un Nouveau Client:</label>
                    <input type="checkbox" id="new_customer" name="new_customer" value="1" style="margin-bottom: 10px;">
                </div>
                <div id="new-customer-form" style="display: none; grid-column: span 1;">
                    <label for="full_name" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Nom Complet:</label>
                    <input type="text" id="full_name" name="full_name" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                    <label for="nif" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">NIF:</label>
                    <input type="text" id="nif" name="nif" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                    <label for="tel" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Tél:</label>
                    <input type="text" id="tel" name="tel" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                    <label for="driver_name" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Nom Chauffeur:</label>
                    <input type="text" id="driver_name" name="driver_name" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                    <label for="driver_contacts" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Contacts Chauffeur:</label>
                    <input type="text" id="driver_contacts" name="driver_contacts" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                    <label for="plate_numbers" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Plaques d'Immatriculation:</label>
                    <input type="text" id="plate_numbers" name="plate_numbers" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                    <label for="additional_contacts" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Contacts Additionnels:</label>
                    <textarea id="additional_contacts" name="additional_contacts" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;"></textarea>
                    <label for="user_type" style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Type de Client *</label>
                    <select id="user_type" name="user_type" required style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                        <option value="physical" selected>Personne Physique</option>
                        <option value="moral">Personne Morale</option>
                    </select>
                </div>
                <button type="submit" name="select_customer" style="grid-column: span 1; background-color: #4B2F1F; color: #F4F0E4; padding: 15px; border: none; border-radius: 8px; cursor: pointer; font-size: 18px; font-weight: 600;">Confirmer Client et Continuer</button>
            </form>
        <?php else: ?>
            <!-- Step 2: Invoice Creation -->
            <?php if (!empty($invoice_errors)): ?>
                <div style="background-color: #FFFFFF; color: #FF0000; padding: 10px; border-radius: 5px; border: 1px solid #FF0000; margin-bottom: 15px;">
                    <?php foreach ($invoice_errors as $error): ?>
                        <p><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            <style>
                #invoice-form input[type=number]::-webkit-outer-spin-button,
                #invoice-form input[type=number]::-webkit-inner-spin-button {
                    -webkit-appearance: none;
                    margin: 0;
                }
                #invoice-form input[type=number] {
                    -moz-appearance: textfield;
                }
            </style>
            <form method="POST" id="invoice-form" style="margin-top: 20px; display: grid; grid-template-columns: 1fr; gap: 20px;">
                <div>
                    <label style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Client Sélectionné:</label>
                    <p style="color: #000000; font-size: 16px; padding: 10px; background-color: #FFFFFF; border-radius: 8px; border: 1px solid #4B2F1F;">
                        <?php echo htmlspecialchars($_SESSION['selected_customer']['full_name'], ENT_QUOTES, 'UTF-8'); ?> - NIF: <?php echo htmlspecialchars($_SESSION['selected_customer']['nif'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                        <a href="?clear_customer=1" style="color: #FF0000; text-decoration: underline; margin-left: 10px;">Changer de Client</a>
                    </p>
                </div>
                <div style="display: grid; gap: 12px;">
                    <?php if ($requires_cashier_selection): ?>
                        <div>
                            <label for="assigned_cashier_id" style="color: #4B2F1F; font-weight: 600; font-size: 15px;">Caissier assigné</label>
                            <select name="assigned_cashier_id" id="assigned_cashier_id" required style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background: #FFF;">
                                <option value="">Sélectionner un caissier</option>
                                <?php foreach ($cashiers as $cashier): ?>
                                    <option value="<?php echo (int)$cashier['id']; ?>" <?php echo (isset($assigned_cashier_id) && (int)$assigned_cashier_id === (int)$cashier['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($cashier['full_name'], ENT_QUOTES, 'UTF-8'); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    <?php else: ?>
                        <input type="hidden" name="assigned_cashier_id" value="<?php echo (int)$_SESSION['user_id']; ?>">
                    <?php endif; ?>
                </div>
                <div class="loan-sale-wrapper" style="display: flex; flex-direction: column; gap: 10px; background-color: rgba(255, 255, 255, 0.9); border: 1px solid #D9C4B4; border-radius: 10px; padding: 15px;">
                    <div style="display: flex; align-items: center; gap: 12px; flex-wrap: wrap;">
                        <label for="is_loan_sale" style="color: #4B2F1F; font-weight: 600; font-size: 16px; margin: 0;">Marquer cette commande comme vente à crédit</label>
                        <input type="checkbox" id="is_loan_sale" name="is_loan_sale" value="1" <?php echo $loan_profile['can_pay_by_loan'] ? '' : ' disabled'; ?> style="transform: scale(1.4); cursor: pointer;">
                        <?php if (!$loan_profile['can_pay_by_loan']): ?>
                            <span style="color: #B22222; font-size: 14px;">Option crédit désactivée pour ce client.</span>
                        <?php else: ?>
                            <span id="loan-balance-summary" style="font-size: 14px; color: #2C1810;">
                                Dû: <?php echo number_format($loan_profile['ledger_outstanding'], 0, ',', '.'); ?> BIF /
                                Limite: <?php echo number_format($loan_profile['loan_limit_bif'], 0, ',', '.'); ?> BIF
                                (Restant: <?php echo number_format($loan_remaining_bif, 0, ',', '.'); ?> BIF)
                            </span>
                        <?php endif; ?>
                    </div>
                    <p style="margin: 0; font-size: 13px; color: rgba(75, 47, 31, 0.75);">
                        Les ventes à crédit sont réservées aux clients approuvés par DG/DGA/Admin. Le montant de la vente augmentera le solde crédit de ce client et doit rester dans sa limite de crédit.
                    </p>
                    <div id="loan-sale-additional" style="display: none; gap: 12px;">
                        <div style="display: flex; flex-direction: column; gap: 6px;">
                            <label for="loan_due_date" style="color: #4B2F1F; font-weight: 600; font-size: 14px;">Date d'Échéance Crédit</label>
                            <input type="date" id="loan_due_date" name="loan_due_date" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; color: #4B2F1F;">
                            <small style="color: rgba(75, 47, 31, 0.7); font-size: 12px;">
                                Par défaut, utilise les conditions de crédit du client (<?php echo $loan_profile['loan_terms_days'] > 0 ? $loan_profile['loan_terms_days'] . ' jours' : 'aucune condition définie'; ?>).
                            </small>
                        </div>
                        <div style="display: flex; flex-direction: column; gap: 6px; flex: 1;">
                            <label for="loan_note" style="color: #4B2F1F; font-weight: 600; font-size: 14px;">Note Crédit (optionnel)</label>
                            <textarea id="loan_note" name="loan_note" rows="3" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; color: #4B2F1F; resize: vertical;" placeholder="Ajouter une note interne sur cette vente à crédit..."></textarea>
                        </div>
                    </div>
                </div>
                <div id="product-list" style="grid-column: span 1; margin-bottom: 20px;">
                    <div class="product-row" style="display: flex; flex-wrap: wrap; align-items: flex-start; gap: 15px; margin-bottom: 15px;">
                        <div style="flex: 2; position: relative;">
                            <label style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Produit:</label>
                            <input type="text" id="product_search_0" list="product_options_0" name="products[]" required style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;" placeholder="Tapez pour rechercher produits...">
                            <datalist id="product_options_0">
                                <?php foreach ($products as $product): ?>
                                    <option value="<?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?>" data-id="<?php echo htmlspecialchars($product['id'], ENT_QUOTES, 'UTF-8'); ?>">
                                <?php endforeach; ?>
                            </datalist>
                            <input type="hidden" name="product_ids[]" id="product_id_0">
                        </div>
                        <div style="flex: 1;">
                            <label style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Quantité (Caisses):</label>
                            <input type="number" name="quantities[]" required style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; -moz-appearance: textfield;" min="1" value="1" onwheel="this.blur()" onkeydown="if(event.key==='ArrowUp'||event.key==='ArrowDown') event.preventDefault()">
                        </div>
                        <button type="button" class="remove-product" style="background-color: #FF0000; color: #FFFFFF; padding: 10px; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; margin-top: 28px; display: none;">Supprimer</button>
                    </div>
                </div>
                <button type="button" id="add-product" style="grid-column: span 1; background-color: #4B2F1F; color: #F4F0E4; padding: 10px; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; margin-bottom: 20px;">Ajouter un Autre Produit</button>
                <button type="button" id="preview-invoice" onclick="if(window.handlePreviewButtonClick){window.handlePreviewButtonClick(event);}else{alert('La fonctionnalité de prévisualisation est en cours de chargement. Veuillez patienter un moment et réessayer.');}" style="grid-column: span 1; background-color: #4B2F1F; color: #F4F0E4; padding: 15px; border: none; border-radius: 8px; cursor: pointer; font-size: 18px; font-weight: 600;">Prévisualiser et Soumettre Commande</button>
            </form>

            <!-- Modal for Preview -->
            <div id="invoice-modal" class="invoice-modal" style="display: none;">
                <div class="invoice-modal-container">
                    <div class="invoice-modal-header">
                        <div>
                            <h3>Prévisualisation Commande</h3>
                            <p>Vérifiez les détails ci-dessous avant de soumettre.</p>
                        </div>
                    </div>
                    <div id="modal-content" class="invoice-modal-body"></div>
                    <div class="invoice-modal-footer">
                        <button id="confirm-submit" class="invoice-modal-btn invoice-modal-confirm">Soumettre Commande</button>
                        <button id="cancel-submit" class="invoice-modal-btn invoice-modal-cancel">Retour à l'Édition</button>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
    /* Hide number input spinners for all browsers */
    input[type="number"]::-webkit-inner-spin-button,
    input[type="number"]::-webkit-outer-spin-button {
        -webkit-appearance: none;
        margin: 0;
    }
    input[type="number"] {
        -moz-appearance: textfield;
    }
    
    .invoice-modal {
        position: fixed;
        inset: 0;
        background: rgba(0, 0, 0, 0.55);
        display: none;
        align-items: center;
        justify-content: center;
        padding: 24px;
        z-index: 1100;
    }

    .invoice-modal-container {
        background: linear-gradient(180deg, #FFFFFF 0%, #F4F0E4 100%);
        border-radius: 16px;
        box-shadow: 0 20px 60px rgba(75, 47, 31, 0.35);
        width: min(720px, 95vw);
        max-height: 90vh;
        display: flex;
        flex-direction: column;
        overflow: hidden;
        border: 1px solid #E5D2C3;
    }

    .invoice-modal-header {
        padding: 24px 28px 0;
        border-bottom: 1px solid rgba(75, 47, 31, 0.1);
    }

    .invoice-modal-header h3 {
        margin: 0;
        font-size: 24px;
        color: #4B2F1F;
        font-weight: 700;
    }

    .invoice-modal-header p {
        margin: 6px 0 18px;
        color: rgba(75, 47, 31, 0.75);
        font-size: 14px;
    }

    .invoice-modal-body {
        padding: 24px 28px;
        overflow-y: auto;
    }

    .invoice-preview-meta {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
        gap: 16px;
        margin-bottom: 24px;
    }

    .invoice-preview-meta .meta-label {
        font-size: 12px;
        text-transform: uppercase;
        letter-spacing: 0.06em;
        color: rgba(75, 47, 31, 0.65);
        margin-bottom: 4px;
    }

    .invoice-preview-meta .meta-value {
        font-size: 15px;
        color: #2C1810;
        font-weight: 600;
    }

    .invoice-preview-table {
        width: 100%;
        border-collapse: collapse;
        margin-bottom: 24px;
        background-color: #FFFFFF;
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 8px 20px rgba(75, 47, 31, 0.1);
    }

    .invoice-preview-table th {
        background: #4B2F1F;
        color: #F4F0E4;
        text-align: left;
        padding: 12px 14px;
        font-size: 14px;
        font-weight: 600;
        letter-spacing: 0.02em;
    }

    .invoice-preview-table td {
        padding: 12px 14px;
        border-bottom: 1px solid #F0E4D8;
        color: #2C1810;
        font-size: 14px;
    }

    .invoice-preview-table tr:last-child td {
        border-bottom: none;
    }

    .invoice-preview-table td.text-right {
        text-align: right;
        font-variant-numeric: tabular-nums;
    }

    .invoice-preview-summary {
        display: grid;
        gap: 10px;
        background: rgba(244, 162, 97, 0.12);
        border: 1px solid rgba(244, 162, 97, 0.3);
        border-radius: 12px;
        padding: 18px 20px;
    }

    .invoice-preview-summary .summary-row,
    .invoice-preview-summary .summary-total {
        display: flex;
        justify-content: space-between;
        align-items: center;
        font-size: 15px;
        color: #4B2F1F;
    }

    .invoice-preview-summary .summary-total {
        font-weight: 700;
        font-size: 17px;
    }

    .invoice-preview-note {
        margin-top: 12px;
        font-size: 12px;
        color: rgba(75, 47, 31, 0.7);
    }

    .custom-price-tag {
        display: inline-block;
        margin-left: 6px;
        padding: 2px 6px;
        border-radius: 6px;
        background-color: rgba(244, 162, 97, 0.3);
        color: #4B2F1F;
        font-size: 11px;
        font-weight: 600;
        letter-spacing: 0.03em;
        text-transform: uppercase;
    }

    .custom-pricing-row {
        background-color: rgba(244, 162, 97, 0.08);
    }

    .custom-pricing-summary {
        margin-top: 16px;
        background: rgba(244, 162, 97, 0.12);
        border: 1px solid rgba(244, 162, 97, 0.45);
        border-radius: 10px;
        padding: 14px 16px;
        color: #4B2F1F;
    }

    .custom-pricing-summary ul {
        margin: 8px 0 0;
        padding-left: 18px;
    }

    .custom-pricing-summary li {
        margin-bottom: 4px;
        font-size: 13px;
        line-height: 1.4;
    }

    .custom-pricing-title {
        font-weight: 700;
        font-size: 14px;
    }

    .invoice-modal-footer {
        display: flex;
        justify-content: flex-end;
        gap: 12px;
        padding: 16px 28px 24px;
        border-top: 1px solid rgba(75, 47, 31, 0.1);
        background: linear-gradient(180deg, #F9F6F1 0%, #F3E6D7 100%);
    }

    .invoice-modal-btn {
        padding: 12px 22px;
        border-radius: 8px;
        border: none;
        font-size: 15px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s ease, box-shadow 0.2s ease;
    }

    .invoice-modal-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 6px 16px rgba(0, 0, 0, 0.18);
    }

    .invoice-modal-confirm {
        background: linear-gradient(135deg, #2E7D32 0%, #155724 100%);
        color: #FFFFFF;
    }

    .invoice-modal-cancel {
        background: transparent;
        color: #4B2F1F;
        border: 1px solid rgba(75, 47, 31, 0.35);
    }
</style>

<!-- Searchable Table for Pending Invoices -->
<div style="padding: 20px; background-color: #F4F0E4; margin: 20px 0; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2);">
    <h3 style="color: #4B2F1F; font-size: 20px; margin-bottom: 15px;">Pending Invoices</h3>
    <input type="text" id="invoice_search" placeholder="Search invoices..." style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #FFFFFF; font-size: 14px; color: #4B2F1F; margin-bottom: 15px;">
    <table id="invoices_table" style="width: 100%; border-collapse: collapse;">
        <thead>
            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                <th style="padding: 10px; text-align: left;">Invoice Number</th>
                <th style="padding: 10px; text-align: left;">Customer</th>
                <th style="padding: 10px; text-align: left;">Date of Creation</th>
                <th style="padding: 10px; text-align: left;">Invoice Created By</th>
                <th style="padding: 10px; text-align: left;">Approved By</th>
                <th style="padding: 10px; text-align: left;">Approval Status</th>
                <th style="padding: 10px; text-align: left;">Actions</th>
            </tr>
        </thead>
        <tbody id="invoices_body">
            <?php
            $stmt = $conn->prepare("SELECT ui.id, ui.invoice_number, ui.customer_name, ui.created_at, ui.approval_status, ui.created_by, u.full_name AS created_by_name, ui.approved_by, ua.full_name AS approved_by_name FROM unpaid_invoices ui LEFT JOIN user u ON ui.created_by = u.id LEFT JOIN user ua ON ui.approved_by = ua.id WHERE ui.province_id = ? AND ui.status = 'pending' ORDER BY ui.created_at DESC");
            if ($stmt === false) {
                error_log("Prepare failed: " . $conn->error);
            } else {
                $stmt->bind_param("i", $province_id);
                if ($stmt->execute()) {
                    $result = $stmt->get_result();
                    while ($invoice = $result->fetch_assoc()) {
                        echo "<tr style='border-bottom: 1px solid #D3D3D3;'>";
                        echo "<td style='padding: 10px;'>" . htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8') . "</td>";
                        echo "<td style='padding: 10px;'>" . htmlspecialchars($invoice['customer_name'], ENT_QUOTES, 'UTF-8') . "</td>";
                        echo "<td style='padding: 10px;'>" . date('Y-m-d H:i', strtotime($invoice['created_at'])) . "</td>";
                        echo "<td style='padding: 10px;'>" . htmlspecialchars($invoice['created_by_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8') . "</td>";
                        echo "<td style='padding: 10px;'>" . htmlspecialchars($invoice['approved_by_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8') . "</td>";
                        echo "<td style='padding: 10px;'>" . htmlspecialchars($invoice['approval_status'], ENT_QUOTES, 'UTF-8') . "</td>";
                        echo "<td style='padding: 10px;'>";
                        echo "<a href='edit_invoice.php?id=" . $invoice['id'] . "' style='color: #4CAF50; text-decoration: underline; margin-right: 10px;'>Edit</a>";
                        if ($invoice['approval_status'] === 'pending') {
                            echo "<a href='delete_invoice.php?id=" . $invoice['id'] . "' style='color: #FF0000; text-decoration: underline;' onclick='return confirm(\"Are you sure you want to delete this invoice?\");'>Delete</a>";
                        }
                        echo "<a href='view_invoice.php?id=" . $invoice['id'] . "' style='color: #2196F3; text-decoration: underline; margin-left: 10px;'>View</a>";
                        echo "</td>";
                        echo "</tr>";
                    }
                } else {
                    error_log("Execute failed: " . $stmt->error);
                }
                $stmt->close();
            }
            ?>
        </tbody>
    </table>
</div>

<?php
if (isset($_GET['clear_customer'])) {
    unset($_SESSION['selected_customer']);
}
?>

<script>
    // Simple immediate handler for preview button - works even before full initialization
    // The button now has an inline onclick handler, but we also add event listener as backup
    (function() {
        function attachPreviewButtonHandler() {
            const btn = document.getElementById('preview-invoice');
            if (btn) {
                // Remove any existing handlers by cloning
                if (!btn.dataset.handlerAttached) {
                    btn.dataset.handlerAttached = '1';
                    btn.addEventListener('click', function(e) {
                        e.preventDefault();
                        e.stopPropagation();
                        console.log('🔘 Event listener Preview button clicked!');
                        // Trigger the full handler if it exists
                        if (window.handlePreviewButtonClick) {
                            window.handlePreviewButtonClick(e);
                        } else {
                            console.warn('⚠️ Full preview handler not ready yet, will retry...');
                            setTimeout(function() {
                                if (window.handlePreviewButtonClick) {
                                    window.handlePreviewButtonClick(e);
                                } else {
                                    alert('Preview functionality is still loading. Please wait a moment and try again.');
                                }
                            }, 500);
                        }
                    });
                    console.log('✅ Event listener preview button handler attached');
                }
            }
        }
        
        // Try immediately
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', attachPreviewButtonHandler);
        } else {
            attachPreviewButtonHandler();
        }
        
        // Also try after a short delay
        setTimeout(attachPreviewButtonHandler, 100);
        setTimeout(attachPreviewButtonHandler, 500);
        setTimeout(attachPreviewButtonHandler, 1000);
    })();
    
    document.addEventListener('DOMContentLoaded', function() {
        // Create customer data structure FIRST
        const customerData = <?php echo json_encode(array_map(function($c) { 
            return [
                'id' => (int)$c['id'],
                'full_name' => trim($c['full_name']),
                'nif' => $c['nif'] ? trim($c['nif']) : null,
                'tel' => $c['tel'] ? trim($c['tel']) : null,
                'display' => trim($c['full_name']) . ($c['nif'] ? ' - NIF: ' . trim($c['nif']) : '')
            ]; 
        }, $customers ?? [])); ?>;
        
        console.log('Customer data loaded:', customerData.length, 'customers');
        
        // Product data with available stock
        const productData = <?php echo json_encode(array_map(function($p) { 
            $reserved = (int)($p['reserved_quantity'] ?? 0);
            $pendingReserved = (int)($p['pending_reserved_crates'] ?? 0);
            return [
                'id' => $p['id'],
                'name' => $p['name'],
                'available_quantity' => (int)($p['available_quantity'] ?? 0),
                'stock_quantity' => (int)($p['stock_quantity'] ?? 0),
                'reserved_quantity' => $reserved,
                'pending_reserved_crates' => $pendingReserved,
                'display_reserved_quantity' => max($reserved, $pendingReserved)
            ]; 
        }, $products ?? [])); ?>;
        
        console.log('Product data loaded:', productData.length, 'products');
        
        // Make these globally accessible for the preview handler
        window.productCatalog = <?php echo json_encode(array_column($products, null, 'name'), JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP) ?: '{}'; ?>;
        window.invoicePreviewMeta = {
            customer: <?php echo json_encode($_SESSION['selected_customer']['full_name'] ?? ''); ?>,
            nif: <?php echo json_encode($_SESSION['selected_customer']['nif'] ?? 'N/A'); ?>,
            preparedBy: <?php echo json_encode($_SESSION['full_name'] ?? ''); ?>
        };
        window.customerLoanProfile = <?php echo json_encode([
            'can_pay_by_loan' => $loan_profile['can_pay_by_loan'],
            'loan_limit_bif' => $loan_profile['loan_limit_bif'],
            'loan_balance_bif' => $loan_profile['loan_balance_bif'],
            'ledger_outstanding' => $loan_profile['ledger_outstanding'],
            'loan_terms_days' => $loan_profile['loan_terms_days'],
            'loan_status' => $loan_profile['loan_status']
        ], JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP); ?>;
        
        // Also keep local references for backward compatibility
        const productCatalog = window.productCatalog;
        const invoicePreviewMeta = window.invoicePreviewMeta;
        const customerLoanProfile = window.customerLoanProfile;
        
        // Function to check for duplicate products (case-insensitive) - make globally accessible
        window.checkDuplicateProducts = function() {
            const productInputs = document.querySelectorAll('[name="products[]"]');
            const productNames = [];
            const duplicates = [];
            const seenNames = new Map(); // Track first occurrence index (lowercase)
            
            productInputs.forEach((input, index) => {
                const productName = input.value.trim();
                if (productName) {
                    const productNameLower = productName.toLowerCase();
                    if (seenNames.has(productNameLower)) {
                        // This is a duplicate
                        duplicates.push({index: index, name: productName});
                    } else {
                        productNames.push(productName);
                        seenNames.set(productNameLower, index);
                    }
                }
            });
            
            return duplicates;
        };
        
        // Also keep local reference for backward compatibility
        const checkDuplicateProducts = window.checkDuplicateProducts;
        
        // Function to validate stock availability (case-insensitive) - make globally accessible
        window.validateStockAvailability = function() {
            const productInputs = document.querySelectorAll('[name="products[]"]');
            const quantityInputs = document.querySelectorAll('[name="quantities[]"]');
            const errors = [];
            
            for (let i = 0; i < productInputs.length; i++) {
                const productName = productInputs[i].value.trim();
                const quantity = parseInt(quantityInputs[i].value, 10) || 0;
                
                if (productName && quantity > 0) {
                    const product = productData.find(p => p.name.toLowerCase() === productName.toLowerCase());
                    if (product) {
                        if (quantity > product.available_quantity) {
                            errors.push({
                                index: i,
                                product: productName,
                                requested: quantity,
                                available: product.available_quantity
                            });
                        }
                    }
                }
            }
            
            return errors;
        }
        
        // Get invoice form
        const invoiceForm = document.getElementById('invoice-form');
        
        // Add validation on form submission
        if (invoiceForm) {
            invoiceForm.addEventListener('submit', function(e) {
                console.log('Form submit event triggered');
                
                // Skip validation if this is a confirmed submission from modal (already validated in preview)
                const confirmSubmitInput = invoiceForm.querySelector('input[name="confirm_submit"]');
                if (confirmSubmitInput && confirmSubmitInput.value === '1') {
                    console.log('✅ Form submission confirmed from modal, skipping duplicate/stock validation');
                    return true;
                }
                
                // Check for duplicates
                const duplicates = checkDuplicateProducts();
                if (duplicates.length > 0) {
                    e.preventDefault();
                    const duplicateNames = duplicates.map(d => d.name).join(', ');
                    alert(`Produits en double détectés : ${duplicateNames}. Veuillez supprimer les doublons ou combiner les quantités.`);
                    return false;
                }
                
                // Check stock availability
                const stockErrors = validateStockAvailability();
                if (stockErrors.length > 0) {
                    e.preventDefault();
                    const errorMessages = stockErrors.map(err => 
                        `${err.product} : ${err.requested} caisses demandées, mais seulement ${err.available} caisses disponibles`
                    ).join('\n');
                    alert(`Stock insuffisant :\n${errorMessages}`);
                    return false;
                }
            });
        } else {
            console.error('Invoice form not found!');
        }

        const loanSaleCheckbox = document.getElementById('is_loan_sale');
        const loanSaleAdditional = document.getElementById('loan-sale-additional');
        const loanDueDateInput = document.getElementById('loan_due_date');
        const loanNoteInput = document.getElementById('loan_note');

        const numberFormatter = new Intl.NumberFormat('fr-FR');

        window.computeDefaultLoanDueDate = function() {
            const today = new Date();
            const termsDays = Number(customerLoanProfile.loan_terms_days || 0);
            const offsetDays = termsDays > 0 ? termsDays : 7; // fallback 7 jours si aucune condition définie
            const dueDate = new Date(today);
            dueDate.setDate(dueDate.getDate() + offsetDays);
            const year = dueDate.getFullYear();
            const month = String(dueDate.getMonth() + 1).padStart(2, '0');
            const day = String(dueDate.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        };
        
        const computeDefaultLoanDueDate = window.computeDefaultLoanDueDate;

        function toggleLoanFields(isChecked) {
            if (!loanSaleAdditional) {
                return;
            }
            loanSaleAdditional.style.display = isChecked ? 'grid' : 'none';
            if (isChecked) {
                if (loanDueDateInput && !loanDueDateInput.value) {
                    const defaultDueDate = computeDefaultLoanDueDate();
                    if (defaultDueDate) {
                        loanDueDateInput.value = defaultDueDate;
                    }
                }
            } else {
                if (loanDueDateInput) {
                    loanDueDateInput.value = '';
                }
                if (loanNoteInput) {
                    loanNoteInput.value = '';
                }
            }
        }

        if (loanSaleCheckbox) {
            toggleLoanFields(loanSaleCheckbox.checked);
            loanSaleCheckbox.addEventListener('change', function() {
                toggleLoanFields(this.checked);
            });
        }


        window.escapeHtml = function(value) {
            if (value === null || value === undefined) {
                return '';
            }
            return String(value)
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#039;');
        };

        window.number_format = function(number, decimals = 0, decimalSep = ',', thousandsSep = '.') {
            const num = parseFloat(number);
            if (isNaN(num)) return '0';
            
            const parts = num.toFixed(decimals).split('.');
            const integerPart = parts[0];
            const decimalPart = parts[1] || '';
            
            const integerWithSeparators = integerPart.replace(/\B(?=(\d{3})+(?!\d))/g, thousandsSep);
            
            if (decimals > 0 && decimalPart) {
                return integerWithSeparators + decimalSep + decimalPart;
            }
            return integerWithSeparators;
        };


        const addProductButton = document.getElementById('add-product');
        if (addProductButton) {
            addProductButton.addEventListener('click', function() {
            const productList = document.getElementById('product-list');
            const rowCount = productList.getElementsByClassName('product-row').length;
            const newRow = document.createElement('div');
            newRow.className = 'product-row';
            newRow.style.display = 'flex';
            newRow.style.flexWrap = 'wrap';
            newRow.style.alignItems = 'flex-start';
            newRow.style.gap = '15px';
            newRow.style.marginBottom = '15px';
            newRow.innerHTML = `
                <div style="flex: 2; position: relative;">
                    <label style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Product:</label>
                    <input type="text" id="product_search_${rowCount}" list="product_options_${rowCount}" name="products[]" required style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;" placeholder="Type to search products...">
                    <datalist id="product_options_${rowCount}">
                        <?php foreach ($products as $product): ?>
                            <option value="<?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?>" data-id="<?php echo htmlspecialchars($product['id'], ENT_QUOTES, 'UTF-8'); ?>">
                        <?php endforeach; ?>
                    </datalist>
                    <input type="hidden" name="product_ids[]" id="product_id_${rowCount}">
                </div>
                <div style="flex: 1;">
                        <label style="color: #4B2F1F; font-weight: 600; font-size: 16px; display: block; margin-bottom: 8px;">Quantity (Crates):</label>
                    <input type="number" name="quantities[]" required style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F; -moz-appearance: textfield;" min="1" value="1" onwheel="this.blur()" onkeydown="if(event.key==='ArrowUp'||event.key==='ArrowDown') event.preventDefault()">
                </div>
                <button type="button" class="remove-product" style="background-color: #FF0000; color: #FFFFFF; padding: 10px; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; margin-top: 28px;">Remove</button>
            `;
            productList.appendChild(newRow);
            updateRemoveButtons();
            setupProductSearch(rowCount);
                // Add validation to the new row
                validateProductRow(rowCount);
                // Re-attach validation listeners for all rows
                addRealTimeValidation();
        });
        }

        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('remove-product')) {
                const row = e.target.closest('.product-row');
                if (row) {
                    row.remove();
                    updateRemoveButtons();
                }
            }
        });

        function updateRemoveButtons() {
            const rows = document.querySelectorAll('.product-row');
            const removeButtons = document.querySelectorAll('.remove-product');
            removeButtons.forEach(button => button.style.display = rows.length > 1 ? 'block' : 'none');
        }

        updateRemoveButtons();

        function setupProductSearch(index) {
            const productInput = document.getElementById(`product_search_${index}`);
            const productIdInput = document.getElementById(`product_id_${index}`);
            if (productInput && productIdInput) {
                productInput.addEventListener('input', function() {
                    const datalist = document.getElementById(`product_options_${index}`);
                    const selectedOption = datalist.querySelector(`option[value="${this.value}"]`);
                    if (selectedOption) {
                        productIdInput.value = selectedOption.getAttribute('data-id');
                    } else {
                        productIdInput.value = '';
                    }
                });

                productInput.addEventListener('change', function() {
                    const datalist = document.getElementById(`product_options_${index}`);
                    const selectedOption = datalist.querySelector(`option[value="${this.value}"]`);
                    if (selectedOption) {
                        productIdInput.value = selectedOption.getAttribute('data-id');
                        updateSystemPriceHint(index, this.value.trim());
                        // Trigger validation when product is selected
                        validateProductRow(index);
                    } else {
                        this.value = '';
                        productIdInput.value = '';
                        updateSystemPriceHint(index, null);
                        // Clear validation styling when product is cleared
                        this.style.borderColor = '';
                        this.style.backgroundColor = '';
                        this.title = '';
                        const quantityInput = document.querySelectorAll('[name="quantities[]"]')[index];
                        if (quantityInput) {
                            quantityInput.style.borderColor = '';
                            quantityInput.style.backgroundColor = '';
                            quantityInput.title = '';
                        }
                    }
                });
            } else {
                console.error(`Product search elements for index ${index} not found: productInput=${productInput}, productIdInput=${productIdInput}`);
            }
        }

        setupProductSearch(0);

        const newCustomer = document.getElementById('new_customer');
        if (newCustomer) {
            newCustomer.addEventListener('change', function() {
                const newCustomerForm = document.getElementById('new-customer-form');
                const customerSearch = document.getElementById('customer_search');
                const searchCustomerBtn = document.getElementById('search_customer');
                newCustomerForm.style.display = this.checked ? 'grid' : 'none';
                customerSearch.disabled = this.checked;
                if (searchCustomerBtn) {
                searchCustomerBtn.disabled = this.checked;
                }
                if (this.checked) {
                    customerSearch.value = '';
                    document.getElementById('customer_id').value = '';
                    document.getElementById('customer_dropdown').style.display = 'none';
                }
            });
        } else {
            console.error('Element with ID "new_customer" not found in the DOM.');
        }

        // These will be defined inside DOMContentLoaded
        
        // Add real-time validation on product/quantity change
        function addRealTimeValidation() {
            const productInputs = document.querySelectorAll('[name="products[]"]');
            const quantityInputs = document.querySelectorAll('[name="quantities[]"]');
            
            productInputs.forEach((input, index) => {
                input.addEventListener('change', function() {
                    validateProductRow(index);
                });
            });
            
            quantityInputs.forEach((input, index) => {
                input.addEventListener('input', function() {
                    validateProductRow(index);
                });
            });
        }
        
        function validateProductRow(index) {
            const productInput = document.querySelectorAll('[name="products[]"]')[index];
            const quantityInput = document.querySelectorAll('[name="quantities[]"]')[index];
            
            if (!productInput || !quantityInput) return;
            
            const productName = productInput.value.trim();
            const quantity = parseInt(quantityInput.value, 10) || 0;
            
            // Remove previous error styling
            productInput.style.borderColor = '';
            quantityInput.style.borderColor = '';
            productInput.style.backgroundColor = '';
            quantityInput.style.backgroundColor = '';
            
            if (productName && quantity > 0) {
                // Check for duplicates (case-insensitive)
                const allProductInputs = document.querySelectorAll('[name="products[]"]');
                let duplicateCount = 0;
                allProductInputs.forEach(inp => {
                    if (inp.value.trim().toLowerCase() === productName.toLowerCase()) {
                        duplicateCount++;
                    }
                });
                
                if (duplicateCount > 1) {
                    productInput.style.borderColor = '#FF0000';
                    productInput.style.backgroundColor = '#FFE6E6';
                    productInput.title = 'Ce produit est déjà ajouté. Veuillez supprimer le doublon.';
        } else {
                    productInput.title = '';
                }
                
                // Check stock
                const product = productData.find(p => p.name.toLowerCase() === productName.toLowerCase());
                if (product) {
                    const reservedDisplayQty = (typeof product.display_reserved_quantity === 'number')
                        ? product.display_reserved_quantity
                        : product.reserved_quantity;
                    if (product.available_quantity === 0 && reservedDisplayQty > 0) {
                        // Product is fully reserved
                        productInput.style.borderColor = '#FFC107';
                        productInput.style.backgroundColor = '#FFF3CD';
                        productInput.title = `⚠️ Ce produit est entièrement réservé (${reservedDisplayQty} caisses réservées dans les commandes non payées/non approuvées). Aucun stock disponible.`;
                        quantityInput.style.borderColor = '#FFC107';
                        quantityInput.style.backgroundColor = '#FFF3CD';
                        quantityInput.title = `⚠️ Produit entièrement réservé. Disponible : 0 caisses.`;
                    } else if (quantity > product.available_quantity) {
                        quantityInput.style.borderColor = '#FF0000';
                        quantityInput.style.backgroundColor = '#FFE6E6';
                        quantityInput.title = `Seulement ${product.available_quantity} caisses disponibles.`;
                    } else {
                        quantityInput.title = '';
                    }
                }
            }
        }
        
        // Initialize real-time validation
        addRealTimeValidation();

        // Define the handler function FIRST, before initialization, so it's always available
        window.handlePreviewButtonClick = function(e) {
            if (e) {
                e.preventDefault();
                e.stopPropagation();
            }

            console.log('🔘 Preview button clicked! (Full handler)');

            try {
                // Get elements from global scope or re-fetch if needed
                const invoiceFormEl = window.invoiceFormEl || document.getElementById('invoice-form');
                const invoiceModal = window.invoiceModal || document.getElementById('invoice-modal');
                const modalContent = window.modalContent || document.getElementById('modal-content');
                const loanSaleCheckboxEl = window.loanSaleCheckboxEl || document.getElementById('is_loan_sale');
                const loanDueDateInputEl = window.loanDueDateInputEl || document.getElementById('loan_due_date');
                
                if (!invoiceFormEl || !invoiceModal || !modalContent) {
                    console.error('❌ Required elements not found!', {
                        invoiceFormEl: !!invoiceFormEl,
                        invoiceModal: !!invoiceModal,
                        modalContent: !!modalContent
                    });
                    alert('Error: Required page elements not found. Please refresh the page.');
                    return;
                }

                // Get functions from global scope with fallbacks
                const checkDuplicateProducts = window.checkDuplicateProducts || function() { 
                    console.warn('checkDuplicateProducts not available');
                    return []; 
                };
                const validateStockAvailability = window.validateStockAvailability || function() { 
                    console.warn('validateStockAvailability not available');
                    return []; 
                };
                const escapeHtml = window.escapeHtml || function(v) { return String(v || ''); };
                const number_format = window.number_format || function(n) { return String(n || 0); };
                const computeDefaultLoanDueDate = window.computeDefaultLoanDueDate || function() { return null; };
                
                const duplicates = checkDuplicateProducts();
                console.log('Duplicate check:', duplicates);
                if (duplicates.length > 0) {
                    const duplicateNames = duplicates.map(d => d.name).join(', ');
                    alert(`Produits en double détectés : ${duplicateNames}. Veuillez supprimer les doublons ou combiner les quantités.`);
                    return;
                }

                const stockErrors = validateStockAvailability();
                console.log('Stock validation:', stockErrors);
                if (stockErrors.length > 0) {
                    const errorMessages = stockErrors.map(err =>
                        `${err.product} : ${err.requested} caisses demandées, mais seulement ${err.available} caisses disponibles`
                    ).join('\n');
                    alert(`Stock insuffisant :\n${errorMessages}`);
                    return;
                }

                const products = [];
                const productInputs = invoiceFormEl.querySelectorAll('[name="products[]"]');
                const quantityInputs = invoiceFormEl.querySelectorAll('[name="quantities[]"]');
                
                console.log('Found', productInputs.length, 'product inputs');

                for (let i = 0; i < productInputs.length; i++) {
                    const productName = productInputs[i].value.trim();
                    const quantity = parseInt(quantityInputs[i].value, 10) || 0;

                    if (productName && quantity > 0) {
                        products.push({
                            name: productName,
                            quantity
                        });
                    }
                }

                if (products.length === 0) {
                    alert('Veuillez ajouter au moins un produit pour prévisualiser la commande.');
                    return;
                }

                const isLoanSale = loanSaleCheckboxEl && loanSaleCheckboxEl.checked;
                
                // Get global variables
                const productCatalog = window.productCatalog || {};
                let invoicePreviewMeta = window.invoicePreviewMeta || {};
                const customerLoanProfile = window.customerLoanProfile || {};
                
                // If customer name is not in session, try to get it from the selected customer display or customer search field
                try {
                    if (!invoicePreviewMeta.customer || invoicePreviewMeta.customer.trim() === '') {
                        // First try the selected customer name display (for customer selection step)
                        const selectedCustomerNameEl = document.getElementById('selected_customer_name');
                        if (selectedCustomerNameEl && selectedCustomerNameEl.textContent && selectedCustomerNameEl.textContent.trim() !== '') {
                            invoicePreviewMeta = Object.assign({}, invoicePreviewMeta, {
                                customer: selectedCustomerNameEl.textContent.trim()
                            });
                        } else {
                            // Try the selected customer display in invoice form (when customer is already selected)
                            // Look for the paragraph that contains the selected customer info
                            const customerLabel = invoiceFormEl.querySelector('label[style*="Selected Customer"]');
                            if (customerLabel) {
                                const customerParagraph = customerLabel.nextElementSibling;
                                if (customerParagraph && customerParagraph.tagName === 'P' && customerParagraph.textContent) {
                                    // Extract customer name (before " - NIF:" or "Change Customer")
                                    let customerText = customerParagraph.textContent.trim();
                                    customerText = customerText.split(' - NIF:')[0].trim();
                                    customerText = customerText.split('Change Customer')[0].trim();
                                    if (customerText) {
                                        invoicePreviewMeta = Object.assign({}, invoicePreviewMeta, {
                                            customer: customerText
                                        });
                                    }
                                }
                            }
                            
                            // If still not found, try customer search field
                            if (!invoicePreviewMeta.customer || invoicePreviewMeta.customer.trim() === '') {
                                const customerSearchEl = document.getElementById('customer_search');
                                if (customerSearchEl && customerSearchEl.value && customerSearchEl.value.trim() !== '') {
                                    // Extract just the name (remove NIF part if present)
                                    const searchValue = customerSearchEl.value.trim();
                                    const nameOnly = searchValue.split(' - NIF:')[0].trim();
                                    invoicePreviewMeta = Object.assign({}, invoicePreviewMeta, {
                                        customer: nameOnly
                                    });
                                }
                            }
                        }
                    }
                    
                    // Ensure customer name is set, even if empty
                    if (!invoicePreviewMeta.customer || invoicePreviewMeta.customer.trim() === '') {
                        invoicePreviewMeta = Object.assign({}, invoicePreviewMeta, {
                            customer: 'Nom du client non disponible'
                        });
                    }
                } catch (customerNameError) {
                    console.warn('Error extracting customer name:', customerNameError);
                    // Continue with default customer name
                    if (!invoicePreviewMeta.customer || invoicePreviewMeta.customer.trim() === '') {
                        invoicePreviewMeta = Object.assign({}, invoicePreviewMeta, {
                            customer: 'Nom du client non disponible'
                        });
                    }
                }
                
                if (isLoanSale && !customerLoanProfile.can_pay_by_loan) {
                    alert('Ce client n\'est pas autorisé pour les ventes à crédit. Veuillez décocher l\'option de vente à crédit ou demander à un administrateur d\'activer les crédits pour ce client.');
                    return;
                }

                let rowsHtml = '';
                let total = 0;
                let hasValidationError = false;

                products.forEach((product, index) => {
                    if (hasValidationError) {
                        return;
                    }

                    let catalogEntry = productCatalog[product.name];
                    if (!catalogEntry) {
                        const normalizedName = product.name.trim().toLowerCase();
                        for (const key in productCatalog) {
                            if (key.toLowerCase() === normalizedName) {
                                catalogEntry = productCatalog[key];
                                break;
                            }
                        }
                    }

                    if (!catalogEntry) {
                        alert(`Le produit "${product.name}" n'est pas disponible dans le catalogue actuel. Veuillez vérifier votre sélection.`);
                        hasValidationError = true;
                        return;
                    }

                    // Always use price_per_crate from catalog entry
                    const pricePerCrate = catalogEntry && catalogEntry.price_per_crate ? parseInt(catalogEntry.price_per_crate, 10) : 0;
                    const safeProductName = escapeHtml(product.name);

                    if (!pricePerCrate || pricePerCrate <= 0) {
                        alert(`Le prix système par caisse pour "${product.name}" n'est pas configuré. Veuillez contacter un administrateur.`);
                        hasValidationError = true;
                        return;
                    }

                    // Price is per crate, so itemTotal = price per crate * number of crates
                    const itemTotal = pricePerCrate * product.quantity;
                    total += itemTotal;

                    const quantityDisplay = number_format(product.quantity, 0, ',', '.');
                    const pricePerCrateDisplay = number_format(pricePerCrate, 0, ',', '.');
                    const itemTotalDisplay = number_format(itemTotal, 0, ',', '.');

                    rowsHtml += `
                        <tr>
                            <td>${index + 1}</td>
                            <td>${safeProductName}</td>
                            <td>Crates</td>
                            <td class="text-right">${quantityDisplay}</td>
                            <td class="text-right">${pricePerCrateDisplay} BIF</td>
                            <td class="text-right">${itemTotalDisplay} BIF</td>
                        </tr>
                    `;
                });

                if (hasValidationError) {
                    return;
                }

                const formattedDate = new Date().toLocaleString('en-GB', {
                    day: '2-digit',
                    month: 'short',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
                const totalFormatted = number_format(total, 0, ',', '.');
                const metaNif = invoicePreviewMeta.nif && invoicePreviewMeta.nif.trim() !== '' ? invoicePreviewMeta.nif : 'N/A';
                const loanLimit = Number(customerLoanProfile.loan_limit_bif || 0);
                const loanBalance = Number(customerLoanProfile.ledger_outstanding || customerLoanProfile.loan_balance_bif || 0);
                const loanRemaining = loanLimit > 0 ? loanLimit - loanBalance : Number.POSITIVE_INFINITY;

                if (isLoanSale && loanLimit > 0 && total > loanRemaining) {
                    alert(`Cette vente à crédit dépasse la limite de crédit restante de ${number_format(loanRemaining, 0, ',', '.')} BIF. Réduisez le total de la facture ou ajustez d'abord la limite de crédit du client.`);
                    return;
                }

                if (isLoanSale && loanDueDateInputEl && !loanDueDateInputEl.value) {
                    const defaultDueDate = (typeof computeDefaultLoanDueDate === 'function') ? computeDefaultLoanDueDate() : '';
                    if (defaultDueDate) {
                        loanDueDateInputEl.value = defaultDueDate;
                    } else {
                        alert('Veuillez définir une date d\'échéance crédit avant de soumettre une vente à crédit.');
                        return;
                    }
                }

                const dueDateDisplay = loanDueDateInputEl && loanDueDateInputEl.value
                    ? new Date(loanDueDateInputEl.value + 'T00:00:00').toLocaleDateString('en-GB', {
                        day: '2-digit',
                        month: 'short',
                        year: 'numeric'
                    })
                    : 'N/A';
                const loanMetaHtml = isLoanSale
                    ? `
                        <div>
                            <div class="meta-label">Mode de Vente</div>
                            <div class="meta-value">Crédit</div>
                        </div>
                        <div>
                            <div class="meta-label">Date d'Échéance Crédit</div>
                            <div class="meta-value">${dueDateDisplay}</div>
                        </div>
                        <div>
                            <div class="meta-label">Limite Restante</div>
                            <div class="meta-value">${loanLimit > 0 ? number_format(Math.max(0, loanRemaining - total), 0, ',', '.') + ' BIF' : 'Aucune limite définie'}</div>
                        </div>
                    `
                    : `
                        <div>
                            <div class="meta-label">Mode de Vente</div>
                            <div class="meta-value">Standard</div>
                        </div>
                    `;

                const previewNote = 'Assurez-vous que les détails du client et les quantités de produits sont corrects avant de soumettre.';

                const content = `
                    <div class="invoice-preview-meta">
                        <div>
                            <div class="meta-label">Client</div>
                            <div class="meta-value">${invoicePreviewMeta.customer}</div>
                        </div>
                        <div>
                            <div class="meta-label">NIF</div>
                            <div class="meta-value">${metaNif}</div>
                        </div>
                        <div>
                            <div class="meta-label">Préparé par</div>
                            <div class="meta-value">${invoicePreviewMeta.preparedBy}</div>
                        </div>
                        <div>
                            <div class="meta-label">Préparé le</div>
                            <div class="meta-value">${formattedDate}</div>
                        </div>
                        ${loanMetaHtml}
                    </div>
                    <table class="invoice-preview-table">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Produit</th>
                                <th>Unité</th>
                                <th>Quantité</th>
                                <th>Prix/Caisse (BIF)</th>
                                <th>Total Ligne (BIF)</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${rowsHtml}
                        </tbody>
                    </table>
                    <div class="invoice-preview-summary">
                        <div class="summary-total">
                            <span>Total Général</span>
                            <span>${totalFormatted} BIF</span>
                        </div>
                    </div>
                    <p class="invoice-preview-note">${previewNote}</p>
                `;
                modalContent.innerHTML = content;
                invoiceModal.style.display = 'flex';
                console.log('✅ Invoice preview modal displayed');
            } catch (error) {
                console.error('❌ Error in preview button handler:', error);
                console.error('Error stack:', error.stack);
                alert('An error occurred: ' + error.message);
            }
        };
        
        console.log('✅ Preview button handler function defined and ready');
        
        // Initialize Preview Invoice Button - Use setTimeout to ensure DOM is fully loaded
        function initPreviewButtonHandler() {
            console.log('=== INITIALIZING PREVIEW INVOICE BUTTON ===');
            const previewInvoiceBtn = document.getElementById('preview-invoice');
            const invoiceModal = document.getElementById('invoice-modal');
            const modalContent = document.getElementById('modal-content');
            const confirmSubmitBtn = document.getElementById('confirm-submit');
            const cancelSubmitBtn = document.getElementById('cancel-submit');
            const invoiceFormEl = document.getElementById('invoice-form');

            console.log('Preview button elements check:', {
                previewInvoiceBtn: !!previewInvoiceBtn,
                invoiceModal: !!invoiceModal,
                modalContent: !!modalContent,
                confirmSubmitBtn: !!confirmSubmitBtn,
                cancelSubmitBtn: !!cancelSubmitBtn,
                invoiceFormEl: !!invoiceFormEl,
                checkDuplicateProducts: typeof checkDuplicateProducts,
                validateStockAvailability: typeof validateStockAvailability,
                escapeHtml: typeof escapeHtml,
                number_format: typeof number_format,
                computeDefaultLoanDueDate: typeof computeDefaultLoanDueDate
            });

            if (!previewInvoiceBtn) {
                console.warn('⚠️ Preview invoice button NOT FOUND - customer may not be selected yet');
                return;
            }
            
            if (!invoiceModal || !modalContent || !confirmSubmitBtn || !cancelSubmitBtn || !invoiceFormEl) {
                console.error('❌ Some invoice modal elements not found:', {
                    invoiceModal: !!invoiceModal,
                    modalContent: !!modalContent,
                    confirmSubmitBtn: !!confirmSubmitBtn,
                    cancelSubmitBtn: !!cancelSubmitBtn,
                    invoiceFormEl: !!invoiceFormEl
                });
                return;
            }

            // Check if functions are available
            if (typeof checkDuplicateProducts !== 'function') {
                console.error('❌ checkDuplicateProducts function not found!');
                return;
            }
            if (typeof validateStockAvailability !== 'function') {
                console.error('❌ validateStockAvailability function not found!');
                return;
            }
            if (typeof escapeHtml !== 'function') {
                console.error('❌ escapeHtml function not found!');
                return;
            }
            if (typeof number_format !== 'function') {
                console.error('❌ number_format function not found!');
                return;
            }
            if (typeof computeDefaultLoanDueDate !== 'function') {
                console.warn('⚠️ computeDefaultLoanDueDate function not found - loan due date auto-fill may not work');
            }

            console.log('✅ All preview elements and functions found!');
            
            // Remove any existing event listeners by cloning the button
            if (previewInvoiceBtn.dataset.modalInitialized === '1') {
                console.log('⚠️ Preview button already initialized, re-initializing...');
            }
            previewInvoiceBtn.dataset.modalInitialized = '1';

            // Get loan-related elements
            const loanSaleCheckboxEl = document.getElementById('is_loan_sale');
            const loanDueDateInputEl = document.getElementById('loan_due_date');
            
            console.log('Loan elements:', {
                loanSaleCheckboxEl: !!loanSaleCheckboxEl,
                loanDueDateInputEl: !!loanDueDateInputEl
            });
            
            // Make variables globally accessible for the handler
            window.invoiceFormEl = invoiceFormEl;
            window.invoiceModal = invoiceModal;
            window.modalContent = modalContent;
            window.loanSaleCheckboxEl = loanSaleCheckboxEl;
            window.loanDueDateInputEl = loanDueDateInputEl;
            
            // Handler function is already defined above, just update the global element references
            // No need to redefine it here - it's already available as window.handlePreviewButtonClick
            
            // Remove any existing handlers by cloning (clean slate) - but keep the inline onclick
            const newBtn = previewInvoiceBtn.cloneNode(true);
            // Preserve the inline onclick handler
            newBtn.setAttribute('onclick', previewInvoiceBtn.getAttribute('onclick') || '');
            previewInvoiceBtn.parentNode.replaceChild(newBtn, previewInvoiceBtn);
            const previewInvoiceBtnClean = document.getElementById('preview-invoice');
            
            if (!previewInvoiceBtnClean) {
                console.error('❌ Failed to get preview button after cloning!');
                return;
            }
            
            // Attach click handler using addEventListener as backup
            previewInvoiceBtnClean.addEventListener('click', window.handlePreviewButtonClick, false);
            
            console.log('✅ Preview button handlers attached successfully to:', previewInvoiceBtnClean);

            // Confirm submit button handler
            confirmSubmitBtn.onclick = function() {
                console.log('✅ Confirm submit clicked');
                const hiddenInput = document.createElement('input');
                hiddenInput.type = 'hidden';
                hiddenInput.name = 'confirm_submit';
                hiddenInput.value = '1';
                invoiceFormEl.appendChild(hiddenInput);
                invoiceFormEl.submit();
            };

            // Cancel button handler
            cancelSubmitBtn.onclick = function() {
                console.log('❌ Cancel clicked');
                invoiceModal.style.display = 'none';
                modalContent.innerHTML = '';
            };

            // Close modal when clicking outside
            invoiceModal.addEventListener('click', function(event) {
                if (event.target === invoiceModal) {
                    invoiceModal.style.display = 'none';
                    modalContent.innerHTML = '';
                }
            });

            console.log('✅ Invoice preview modal initialized successfully!');
        }
        
        // Call the initialization function with a small delay
        setTimeout(initPreviewButtonHandler, 100); // Small delay to ensure DOM is ready

        try {
            // Customer search functionality - supports remote search
            const customerSearchEl = document.getElementById('customer_search');
            const customerDropdownEl = document.getElementById('customer_dropdown');
            const customerIdEl = document.getElementById('customer_id');
            const selectedCustomerDisplayEl = document.getElementById('selected_customer_display');
            const selectedCustomerNameEl = document.getElementById('selected_customer_name');
            const customerOptionsDatalist = document.getElementById('customer_options');
            const customerSearchEndpoint = '/masunzu_bar_hotel/modules/cashiers/customer_search_api.php';
            let cachedCustomerSnapshot = Array.isArray(customerData) ? customerData : [];
            let isDropdownOpen = false;
            let activeFetchController = null;
            let searchDebounceTimer = null;
            
            const escapeHtml = window.escapeHtml || function(v) { return String(v || ''); };
            
            if (customerOptionsDatalist && customerSearchEl && customerIdEl) {
                const syncCustomerFromDatalist = () => {
                    const options = customerOptionsDatalist.options;
                    let matchedId = '';
                    for (let i = 0; i < options.length; i++) {
                        if (options[i].value.trim().toLowerCase() === customerSearchEl.value.trim().toLowerCase()) {
                            matchedId = options[i].dataset.id || '';
                            break;
                        }
                    }
                    customerIdEl.value = matchedId;
                };
                customerSearchEl.addEventListener('change', syncCustomerFromDatalist);
                customerSearchEl.addEventListener('blur', syncCustomerFromDatalist);
            }
            const customerSelectForm = document.getElementById('customer-select-form');
            if (customerSelectForm && customerIdEl && customerSearchEl) {
                customerSelectForm.addEventListener('submit', function() {
                    const typedName = customerSearchEl.value.trim().toLowerCase();
                    if (!typedName) {
                        customerIdEl.value = '';
                        return;
                    }
                    const exact = cachedCustomerSnapshot.find(c => c.full_name && c.full_name.toLowerCase() === typedName);
                    if (exact) {
                        customerIdEl.value = exact.id;
                    }
                });
            }
            
            function showDropdownMessage(message) {
                customerDropdownEl.innerHTML = '';
                const info = document.createElement('div');
                info.textContent = message;
                info.style.padding = '12px';
                info.style.color = '#666';
                info.style.textAlign = 'center';
                info.style.fontStyle = 'italic';
                customerDropdownEl.appendChild(info);
            }
            
            function renderCustomerResults(results) {
                customerDropdownEl.innerHTML = '';
                results.forEach(customer => {
                    const item = document.createElement('div');
                    item.className = 'customer-item';
                    item.style.cssText = 'padding: 12px; cursor: pointer; border-bottom: 1px solid #E0E0E0; transition: all 0.2s;';
                    item.innerHTML = `
                        <div style="font-weight: 600; color: #4B2F1F;">${escapeHtml(customer.full_name || '')}</div>
                        ${customer.nif ? `<div style="font-size: 12px; color: #666; margin-top: 4px;">NIF: ${escapeHtml(customer.nif)}</div>` : ''}
                    `;
                    item.addEventListener('mouseenter', function() {
                        this.style.backgroundColor = '#F4F0E4';
                        this.style.paddingLeft = '16px';
                    });
                    item.addEventListener('mouseleave', function() {
                        this.style.backgroundColor = '';
                        this.style.paddingLeft = '12px';
                    });
                    item.addEventListener('click', function(e) {
                        e.stopPropagation();
                        selectCustomer(customer);
                    });
                    customerDropdownEl.appendChild(item);
                });
            }
            
            async function fetchCustomersRemote(term) {
                if (activeFetchController) {
                    activeFetchController.abort();
                }
                activeFetchController = new AbortController();
                const controller = activeFetchController;
                const params = new URLSearchParams();
                if (term) {
                    params.set('q', term);
                }
                const url = `${customerSearchEndpoint}?${params.toString()}`;
                try {
                    const response = await fetch(url, {
                        headers: {'X-Requested-With': 'XMLHttpRequest'},
                        credentials: 'same-origin',
                        signal: controller.signal
                    });
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}`);
                    }
                    const payload = await response.json();
                    if (!payload.success) {
                        throw new Error(payload.message || 'Search failed');
                    }
                    return Array.isArray(payload.customers) ? payload.customers : [];
                } catch (error) {
                    if (error.name === 'AbortError') {
                        return null;
                    }
                    throw error;
                }
            }
            
            async function filterCustomers(searchTerm, options = {}) {
                if (!customerDropdownEl) {
                    return;
                }
                const term = (searchTerm || '').trim();
                const normalizedTerm = term.toLowerCase();
                showDropdownMessage(term ? 'Searching customers...' : 'Loading customers...');
                
                let results = [];
                if (!options.forceRemote && cachedCustomerSnapshot.length > 0) {
                    if (normalizedTerm === '') {
                        results = cachedCustomerSnapshot.slice(0, 20);
        } else {
                        results = cachedCustomerSnapshot.filter(c => {
                            const nameMatch = c.full_name && c.full_name.toLowerCase().includes(normalizedTerm);
                            const nifMatch = c.nif && c.nif.toLowerCase().includes(normalizedTerm);
                            const telMatch = c.tel && c.tel.toLowerCase().includes(normalizedTerm);
                            return nameMatch || nifMatch || telMatch;
                        }).slice(0, 50);
                    }
                }
                
                const shouldFetchRemote = options.forceRemote || !cachedCustomerSnapshot.length || (term !== '' && results.length < 5);
                if (shouldFetchRemote) {
                    try {
                        const remoteResults = await fetchCustomersRemote(term);
                        if (remoteResults === null) {
                            return;
                        }
                        results = remoteResults;
                        if (!term && remoteResults.length) {
                            cachedCustomerSnapshot = results;
                        }
                    } catch (error) {
                        console.error('Customer search failed:', error);
                        showDropdownMessage('Unable to search customers. Please try again.');
                        return;
                    }
                }
                
                if (!Array.isArray(results) || results.length === 0) {
                    showDropdownMessage('No customers found');
                    return;
                }
                
                renderCustomerResults(results);
            }
            
            function selectCustomer(customer) {
                customerSearchEl.value = customer.full_name;
                customerIdEl.value = customer.id;
                customerDropdownEl.style.display = 'none';
                isDropdownOpen = false;
                if (selectedCustomerDisplayEl && selectedCustomerNameEl) {
                    selectedCustomerNameEl.textContent = customer.display;
                    selectedCustomerDisplayEl.style.display = 'block';
                }
            }
            
            function openCustomerDropdown() {
                customerDropdownEl.style.display = 'block';
                isDropdownOpen = true;
            }
            
            function closeCustomerDropdown() {
                customerDropdownEl.style.display = 'none';
                isDropdownOpen = false;
            }
            
            if (!customerSearchEl || !customerDropdownEl || !customerIdEl) {
                console.error('Customer search elements missing. Search cannot initialize.');
            } else {
                customerSearchEl.addEventListener('focus', function() {
                    openCustomerDropdown();
                    filterCustomers(this.value, {forceRemote: !cachedCustomerSnapshot.length});
                });
                
                customerSearchEl.addEventListener('input', function() {
                    if (!isDropdownOpen) {
                        openCustomerDropdown();
                    }
                    const value = this.value;
                    if (searchDebounceTimer) {
                        clearTimeout(searchDebounceTimer);
                    }
                    searchDebounceTimer = setTimeout(() => {
                        filterCustomers(value);
                    }, 200);
                });
                
                document.addEventListener('click', function(e) {
                    if (isDropdownOpen &&
                        !customerDropdownEl.contains(e.target) &&
                        e.target !== customerSearchEl) {
                        closeCustomerDropdown();
                    }
                });
                
                customerDropdownEl.addEventListener('click', function(e) {
                    e.stopPropagation();
                });
                
                customerSearchEl.addEventListener('keydown', function(e) {
                    if (e.key === 'Escape') {
                        closeCustomerDropdown();
                    }
                });
            }
        } catch (error) {
            console.error('Customer search initialization failed:', error);
        }

        // Preview button is initialized above, no need to call initInvoicePreviewModal()
        
        // Fallback: Try to attach preview button handler directly if not already attached
        const previewBtnFallback = document.getElementById('preview-invoice');
        if (previewBtnFallback && !previewBtnFallback.dataset.modalInitialized) {
            console.log('⚠️ Preview button not initialized, attempting fallback initialization...');
            // The button will be initialized by the setTimeout above
        }

        // Search functionality for invoices
        const invoiceSearch = document.getElementById('invoice_search');
        const invoicesBody = document.getElementById('invoices_body');
        if (invoiceSearch && invoicesBody) {
            invoiceSearch.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase();
                const rows = invoicesBody.getElementsByTagName('tr');
                for (let i = 0; i < rows.length; i++) {
                    const text = rows[i].textContent.toLowerCase();
                    rows[i].style.display = text.includes(searchTerm) ? '' : 'none';
                }
            });
        }

        function number_format(number, decimals, dec_point, thousands_sep) {
            number = (number + '').replace(/[^0-9+\-Ee.]/g, '');
            const n = !isFinite(+number) ? 0 : +number;
            const prec = !isFinite(+decimals) ? 0 : Math.abs(decimals);
            const sep = (typeof thousands_sep === 'undefined') ? ',' : thousands_sep;
            const dec = (typeof dec_point === 'undefined') ? '.' : dec_point;
            let s = '';
            const toFixedFix = function(n, prec) {
                const k = Math.pow(10, prec);
                return Math.round(n * k) / k;
            };
            s = (prec ? toFixedFix(n, prec) : Math.round(n)).toString().split('.');
            if (s[0].length > 3) {
                s[0] = s[0].replace(/\B(?=(\d{3})+(?!\d))/g, sep);
            }
            if ((s[1] || '').length < prec) {
                s[1] = s[1] || '';
                s[1] += new Array(prec - s[1].length + 1).join('0');
            }
            return s.join(dec);
        }
    });
</script>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        if (window.masunzuInvoicePreviewInit) {
            setTimeout(function() {
                try {
                    window.masunzuInvoicePreviewInit();
                } catch (error) {
                    console.error('Fallback preview initialization failed:', error);
                }
            }, 0);
        }
    });
</script>

<?php
include __DIR__ . '/../../includes/footer.php';
if ($conn && !$conn->connect_error) {
    $conn->close();
}
?>
