<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../config/database.php';

require_login();

$allowedRoles = ['DG', 'DGA', 'Admin', 'Accountant', 'Stock Manager', 'Caissier_Comptable', 'Board'];
$roleName = $_SESSION['role_name'] ?? '';
if (!in_array($roleName, $allowedRoles, true)) {
    error_log("Unauthorized access to deposit_history.php by {$roleName}");
    header('Location: /masunzu_bar_hotel/index.php');
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    http_response_code(500);
    echo "<p style='color:#b71c1c;text-align:center;padding:20px;'>Connexion base de données indisponible.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

function mbh_table_exists(mysqli $conn, string $table): bool
{
    static $cache = [];
    if (array_key_exists($table, $cache)) {
        return $cache[$table];
    }
    $escaped = $conn->real_escape_string($table);
    $result = $conn->query("SHOW TABLES LIKE '{$escaped}'");
    if ($result) {
        $cache[$table] = $result->num_rows > 0;
        $result->close();
    } else {
        $cache[$table] = false;
    }
    return $cache[$table];
}

$startDate = isset($_GET['start_date']) && $_GET['start_date'] !== '' ? $_GET['start_date'] : date('Y-m-01');
$endDate = isset($_GET['end_date']) && $_GET['end_date'] !== '' ? $_GET['end_date'] : date('Y-m-d');
$provinceFilter = isset($_GET['province_id']) ? (int)$_GET['province_id'] : 0;
$cashierFilter = isset($_GET['cashier_id']) ? (int)$_GET['cashier_id'] : 0;
$bankAccountFilter = isset($_GET['bank_account_id']) ? (int)$_GET['bank_account_id'] : 0;
$slipSearch = trim($_GET['slip'] ?? '');

// Build filters + params for both cashier deposits and direct bank deposits
$cashierFilters = [];
$cashierTypes = '';
$cashierParams = [];
$bankDepositFilters = [];
$bankTypes = '';
$bankParams = [];

if ($startDate !== '') {
    $cashierFilters[] = 'csd.created_at >= ?';
    $cashierTypes .= 's';
    $cashierParams[] = $startDate . ' 00:00:00';

    $bankDepositFilters[] = 'bd.deposit_date >= ?';
    $bankTypes .= 's';
    $bankParams[] = $startDate . ' 00:00:00';
}
if ($endDate !== '') {
    $cashierFilters[] = 'csd.created_at <= ?';
    $cashierTypes .= 's';
    $cashierParams[] = $endDate . ' 23:59:59';

    $bankDepositFilters[] = 'bd.deposit_date <= ?';
    $bankTypes .= 's';
    $bankParams[] = $endDate . ' 23:59:59';
}
if ($provinceFilter > 0) {
    $cashierFilters[] = 'u.province_id = ?';
    $cashierTypes .= 'i';
    $cashierParams[] = $provinceFilter;

    $bankDepositFilters[] = 'bd.province_id = ?';
    $bankTypes .= 'i';
    $bankParams[] = $provinceFilter;
}
if ($cashierFilter > 0) {
    $cashierFilters[] = 'cb.cashier_id = ?';
    $cashierTypes .= 'i';
    $cashierParams[] = $cashierFilter;
}
if ($slipSearch !== '') {
    $cashierFilters[] = '(csd.slip_number LIKE ? OR COALESCE(bdn.slip_number, bd_legacy.slip_number, \'\') LIKE ?)';
    $cashierTypes .= 'ss';
    $cashierParams[] = '%' . $slipSearch . '%';
    $cashierParams[] = '%' . $slipSearch . '%';

    $bankDepositFilters[] = 'bd.slip_number LIKE ?';
    $bankTypes .= 's';
    $bankParams[] = '%' . $slipSearch . '%';
}
if ($bankAccountFilter > 0) {
    $cashierFilters[] = 'csd.bank_account_id = ?';
    $cashierTypes .= 'i';
    $cashierParams[] = $bankAccountFilter;

    $bankDepositFilters[] = 'bd.bank_account_id = ?';
    $bankTypes .= 'i';
    $bankParams[] = $bankAccountFilter;
}

$cashierWhereClause = $cashierFilters ? 'WHERE ' . implode(' AND ', $cashierFilters) : '';
$bankDepositWhereClause = $bankDepositFilters ? 'WHERE ' . implode(' AND ', $bankDepositFilters) : '';

$hasDepositNew = mbh_table_exists($conn, 'bank_deposit_new');
$hasDepositLegacy = mbh_table_exists($conn, 'bank_deposit');

$joins = '';
if ($hasDepositNew) {
    $joins .= 'LEFT JOIN bank_deposit_new bdn ON csd.deposit_id = bdn.id ';
}
if ($hasDepositLegacy) {
    $joins .= 'LEFT JOIN bank_deposit bd_legacy ON csd.deposit_id = bd_legacy.id ';
    if ($hasDepositNew) {
        $joins .= 'AND bdn.id IS NULL ';
    }
}

// Query to combine cashier shift deposits and direct bank deposits
$sql = "
    SELECT
        csd.id,
        csd.shift_id,
        cb.cashier_id,
        u.full_name AS cashier_name,
        pr.name AS province_name,
        csd.amount,
        csd.created_at,
        COALESCE(csd.slip_number, bdn.slip_number, bd_legacy.slip_number) AS slip_number,
        COALESCE(csd.slip_file_path, bdn.slip_file_path, bd_legacy.slip_file_path) AS slip_file_path,
        ba.account_number,
        ba.currency,
        COALESCE(bdn.deposit_date, bd_legacy.deposit_date, csd.created_at) AS deposit_date,
        'cashier' AS deposit_type
    FROM cashier_shift_deposit csd
    JOIN cashier_balance cb ON csd.shift_id = cb.id
    LEFT JOIN user u ON cb.cashier_id = u.id
    LEFT JOIN province pr ON u.province_id = pr.id
    LEFT JOIN bank_account ba ON csd.bank_account_id = ba.id
    $joins
    $cashierWhereClause
    UNION ALL
    SELECT
        bd.id,
        NULL AS shift_id,
        bd.user_id AS cashier_id,
        u2.full_name AS cashier_name,
        pr2.name AS province_name,
        bd.amount,
        bd.deposit_date AS created_at,
        bd.slip_number,
        bd.slip_file_path,
        ba2.account_number,
        ba2.currency,
        bd.deposit_date,
        'accountant' AS deposit_type
    FROM bank_deposit bd
    LEFT JOIN user u2 ON bd.user_id = u2.id
    LEFT JOIN province pr2 ON bd.province_id = pr2.id
    LEFT JOIN bank_account ba2 ON bd.bank_account_id = ba2.id
    $bankDepositWhereClause
    ORDER BY deposit_date DESC, id DESC
    LIMIT 500
";

$stmt = $conn->prepare($sql);
if ($stmt === false) {
    error_log('Failed to prepare deposit history query: ' . $conn->error);
    error_log('SQL: ' . $sql);
    http_response_code(500);
    echo "<p style='color:#b71c1c;text-align:center;padding:20px;'>Impossible de charger l'historique.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// For UNION queries, we need to duplicate parameters for both parts
// Build parameter list: cashier filters first, then bank deposit filters
$unionParams = array_merge($cashierParams, $bankParams);
$unionTypes = $cashierTypes . $bankTypes;

if ($unionTypes && !empty($unionParams)) {
    $refs = [];
    foreach ($unionParams as $key => $value) {
        $refs[$key] = &$unionParams[$key];
    }
    array_unshift($refs, $unionTypes);
    call_user_func_array([$stmt, 'bind_param'], $refs);
}
if (!$stmt->execute()) {
    error_log('Failed to execute deposit history query: ' . $stmt->error);
    error_log('SQL: ' . $sql);
    $stmt->close();
    http_response_code(500);
    echo "<p style='color:#b71c1c;text-align:center;padding:20px;'>Impossible d'exécuter la requête. Veuillez réessayer.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
$history = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$totalDeposits = 0;
foreach ($history as $row) {
    $totalDeposits += (float)($row['amount'] ?? 0);
}

$provinces = [];
$result = $conn->query('SELECT id, name FROM province ORDER BY name');
if ($result) {
    $provinces = $result->fetch_all(MYSQLI_ASSOC);
    $result->close();
}

$cashiers = [];
$result = $conn->query("
    SELECT u.id, u.full_name
    FROM user u
    JOIN role r ON u.role_id = r.id
    WHERE r.name = 'Caissier_Comptable'
    ORDER BY u.full_name
");
if ($result) {
    $cashiers = $result->fetch_all(MYSQLI_ASSOC);
    $result->close();
}

$bankAccounts = [];
$result = $conn->query("SELECT id, account_number, currency FROM bank_account ORDER BY account_number");
if ($result) {
    $bankAccounts = $result->fetch_all(MYSQLI_ASSOC);
    $result->close();
}

$conn->close();

include __DIR__ . '/../../includes/header.php';
?>
<style>
.deposit-history-page {
    padding: 20px;
}
.history-header {
    display: flex;
    flex-wrap: wrap;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 20px;
}
.history-header h2 {
    color: #4B2F1F;
    margin: 0;
}
.filters-card {
    background-color: #FFFFFF;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    margin-bottom: 20px;
}
.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
}
.filters-grid label {
    display: block;
    font-weight: 600;
    color: #4B2F1F;
    margin-bottom: 6px;
}
.filters-grid input,
.filters-grid select {
    width: 100%;
    padding: 10px;
    border-radius: 8px;
    border: 1px solid #D7CCC8;
    font-size: 14px;
    color: #4B2F1F;
}
.filters-actions {
    margin-top: 15px;
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}
.filters-actions button,
.filters-actions a {
    background-color: #4B2F1F;
    color: #F5F5F5;
    border: none;
    border-radius: 8px;
    padding: 10px 20px;
    font-weight: 600;
    cursor: pointer;
    text-decoration: none;
}
.filters-actions a {
    background-color: #B0BEC5;
    color: #263238;
}
.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 15px;
    margin-bottom: 20px;
}
.summary-card {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    border-radius: 12px;
    padding: 18px;
    color: #FFFFFF;
    box-shadow: 0 6px 18px rgba(76, 175, 80, 0.25);
}
.summary-card p {
    margin: 0;
    opacity: 0.8;
}
.summary-card h3 {
    margin: 10px 0 0;
    font-size: 26px;
    font-weight: 700;
}
.history-table-wrapper {
    background-color: #FFFFFF;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    overflow: hidden;
}
.history-table-wrapper table {
    width: 100%;
    border-collapse: collapse;
}
.history-table-wrapper th,
.history-table-wrapper td {
    padding: 12px 14px;
    text-align: left;
    font-size: 14px;
}
.history-table-wrapper th {
    background-color: #4B2F1F;
    color: #F4F0E4;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}
.history-table-wrapper tr:nth-child(even) {
    background-color: #F9F5F0;
}
.history-table-wrapper tr:hover {
    background-color: #FFF3E0;
}
.badge-province {
    background-color: #FF7043;
    color: #FFFFFF;
    padding: 3px 8px;
    border-radius: 999px;
    font-size: 12px;
    display: inline-block;
}
.slip-link {
    color: #1976D2;
    text-decoration: none;
    font-weight: 600;
}
.slip-link:hover {
    text-decoration: underline;
}
.slip-actions {
    display: flex;
    gap: 8px;
    align-items: center;
}
.slip-view-btn {
    background: linear-gradient(135deg, #4B2F1F, #6B4F2F);
    color: #FFFFFF;
    border: none;
    border-radius: 6px;
    padding: 6px 14px;
    cursor: pointer;
    font-size: 13px;
    font-weight: 600;
}
.slip-view-btn:hover {
    opacity: 0.9;
}
.slip-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.65);
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}
.slip-modal.active {
    display: flex;
}
.slip-modal-content {
    width: 90%;
    max-width: 900px;
    background: #fff;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.25);
    display: flex;
    flex-direction: column;
    gap: 16px;
    max-height: 90vh;
}
.slip-modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid #eee;
    padding-bottom: 8px;
}
.slip-modal-header h4 {
    margin: 0;
    color: #4B2F1F;
}
.slip-modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #4B2F1F;
}
.slip-preview-frame,
.slip-preview-image {
    width: 100%;
    border: none;
    border-radius: 10px;
}
.slip-preview-frame {
    height: 70vh;
}
.slip-preview-image {
    max-height: 70vh;
    object-fit: contain;
}
.slip-download-link {
    align-self: flex-end;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    font-weight: 600;
    color: #1976D2;
    text-decoration: none;
}
@media (max-width: 768px) {
    .history-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 10px;
    }
    .filters-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<div class="deposit-history-page">
    <div class="history-header">
        <h2>Historique des Dépôts Bancaires</h2>
        <a href="/masunzu_bar_hotel/modules/banking/deposit.php" style="color:#4B2F1F;font-weight:600;text-decoration:none;">← Retour au dépôt</a>
    </div>

    <div class="filters-card">
        <form method="get">
            <div class="filters-grid">
                <div>
                    <label for="start_date">Date début</label>
                    <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($startDate); ?>">
                </div>
                <div>
                    <label for="end_date">Date fin</label>
                    <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($endDate); ?>">
                </div>
                <div>
                    <label for="province_id">Dépôt</label>
                    <select id="province_id" name="province_id">
                        <option value="0">Toutes</option>
                        <?php foreach ($provinces as $province): ?>
                            <option value="<?php echo (int)$province['id']; ?>" <?php echo $provinceFilter === (int)$province['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($province['name'], ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <label for="cashier_id">Caissier</label>
                    <select id="cashier_id" name="cashier_id">
                        <option value="0">Tous</option>
                        <?php foreach ($cashiers as $cashier): ?>
                            <option value="<?php echo (int)$cashier['id']; ?>" <?php echo $cashierFilter === (int)$cashier['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cashier['full_name'], ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <label for="bank_account_id">Compte bancaire</label>
                    <select id="bank_account_id" name="bank_account_id">
                        <option value="0">Tous</option>
                        <?php foreach ($bankAccounts as $bankAccount): ?>
                            <option value="<?php echo (int)$bankAccount['id']; ?>" <?php echo $bankAccountFilter === (int)$bankAccount['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($bankAccount['account_number'] . ' (' . $bankAccount['currency'] . ')', ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <label for="slip">Bordereau</label>
                    <input type="text" id="slip" name="slip" placeholder="Numéro bordereau" value="<?php echo htmlspecialchars($slipSearch, ENT_QUOTES, 'UTF-8'); ?>">
                </div>
            </div>
            <div class="filters-actions">
                <button type="submit">Filtrer</button>
                <a href="/masunzu_bar_hotel/modules/banking/deposit_history.php">Réinitialiser</a>
            </div>
        </form>
    </div>

    <div class="summary-cards">
        <div class="summary-card">
            <p>Total sur la période</p>
            <h3><?php echo number_format($totalDeposits, 0, ',', '.'); ?> BIF</h3>
        </div>
        <div class="summary-card" style="background:linear-gradient(135deg,#00897B,#004D40);">
            <p>Nombre d'enregistrements</p>
            <h3><?php echo number_format(count($history), 0, ',', '.'); ?></h3>
        </div>
    </div>

    <div class="history-table-wrapper">
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Déposant</th>
                    <th>Dépôt</th>
                    <th>Montant (BIF)</th>
                    <th>Compte Bancaire</th>
                    <th>Bordereau</th>
                    <th>Date Dépôt</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($history)): ?>
                    <tr>
                        <td colspan="7" style="text-align:center;color:#6d4c41;">Aucun dépôt trouvé pour cette période.</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($history as $row): ?>
                        <tr>
                            <td>#<?php echo (int)$row['id']; ?></td>
                            <td><?php echo htmlspecialchars($row['cashier_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <span class="badge-province"><?php echo htmlspecialchars($row['province_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></span>
                            </td>
                            <td><?php echo number_format((float)($row['amount'] ?? 0), 0, ',', '.'); ?></td>
                            <td><?php echo htmlspecialchars($row['account_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <?php if (!empty($row['slip_file_path'])): ?>
                                    <div class="slip-actions">
                                        <button type="button"
                                                class="slip-view-btn"
                                                data-slip-url="<?php echo htmlspecialchars($row['slip_file_path'], ENT_QUOTES, 'UTF-8'); ?>"
                                                data-slip-label="<?php echo htmlspecialchars($row['slip_number'] ?? 'Bordereau', ENT_QUOTES, 'UTF-8'); ?>">
                                            Voir
                                        </button>
                                        <a class="slip-link"
                                           href="<?php echo htmlspecialchars($row['slip_file_path'], ENT_QUOTES, 'UTF-8'); ?>"
                                           target="_blank"
                                           rel="noopener">
                                            Télécharger
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <?php echo htmlspecialchars($row['slip_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($row['deposit_date'] ?? $row['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <div id="slipModal" class="slip-modal">
        <div class="slip-modal-content">
            <div class="slip-modal-header">
                <h4 id="slipModalTitle">Bordereau</h4>
                <button type="button" class="slip-modal-close" aria-label="Fermer">&times;</button>
            </div>
            <iframe class="slip-preview-frame" src="" title="Prévisualisation bordereau" hidden></iframe>
            <img class="slip-preview-image" src="" alt="Prévisualisation bordereau" hidden>
            <a class="slip-download-link" href="#" target="_blank" rel="noopener">⬇ Télécharger le fichier</a>
        </div>
    </div>
</div>

<script>
(function() {
    const modal = document.getElementById('slipModal');
    if (!modal) {
        return;
    }
    const modalTitle = document.getElementById('slipModalTitle');
    const frame = modal.querySelector('.slip-preview-frame');
    const image = modal.querySelector('.slip-preview-image');
    const downloadLink = modal.querySelector('.slip-download-link');
    const closeBtn = modal.querySelector('.slip-modal-close');

    function closeModal() {
        modal.classList.remove('active');
        frame.src = '';
        image.src = '';
        frame.hidden = true;
        image.hidden = true;
    }

    document.querySelectorAll('.slip-view-btn').forEach((btn) => {
        btn.addEventListener('click', () => {
            const url = btn.getAttribute('data-slip-url');
            if (!url) {
                return;
            }
            const label = btn.getAttribute('data-slip-label') || 'Bordereau';
            modalTitle.textContent = label;
            downloadLink.href = url;

            const extension = (url.split('.').pop() || '').toLowerCase();
            if (extension === 'pdf') {
                frame.src = url;
                frame.hidden = false;
                image.hidden = true;
            } else {
                image.src = url;
                image.hidden = false;
                frame.hidden = true;
            }

            modal.classList.add('active');
        });
    });

    closeBtn.addEventListener('click', closeModal);
    modal.addEventListener('click', (event) => {
        if (event.target === modal) {
            closeModal();
        }
    });
    document.addEventListener('keyup', (event) => {
        if (event.key === 'Escape' && modal.classList.contains('active')) {
            closeModal();
        }
    });
})();
</script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
