<?php
include('../../includes/auth.php');
include('../../includes/header.php');
include('../../includes/functions.php');
include('../../config/database.php');
require_once __DIR__ . '/../../includes/cash_helpers.php';

$conn = get_db_connection();
if (!$conn) {
    echo "<div class='alert alert-danger m-3'>Échec de la connexion à la base de données.</div>";
    include('../../includes/footer.php');
    exit;
}

register_shutdown_function(static function () use ($conn) {
    if ($conn instanceof mysqli) {
        @$conn->close();
    }
});

// ============================================
// UTILISER UNE NOUVELLE TABLE bank_deposit_new
// Cette table remplace bank_deposit avec la structure correcte
// ============================================
define('BANK_DEPOSIT_TABLE', 'bank_deposit_new');

// Cette fonction garantit que la nouvelle table existe et est correcte
function ensureBankDepositTableStructure(mysqli $conn): bool {
    $tableName = BANK_DEPOSIT_TABLE;
    try {
        // Vérifier si la table existe déjà
        $tableExists = $conn->query("SHOW TABLES LIKE '" . $tableName . "'");
        $needsCreation = !$tableExists || $tableExists->num_rows === 0;
        if ($tableExists) $tableExists->free();
        
        if ($needsCreation) {
            // Créer directement la nouvelle table avec la structure CORRECTE
            $conn->query("SET FOREIGN_KEY_CHECKS = 0");
            $conn->query("SET SQL_MODE = ''");
            
            // Créer la nouvelle table (supprimer d'abord si elle existe)
            $conn->query("DROP TABLE IF EXISTS " . $tableName);
            
            $createSuccess = $conn->query("CREATE TABLE " . $tableName . " (
                id INT(11) NOT NULL AUTO_INCREMENT,
                user_id INT(11) NOT NULL,
                amount BIGINT(20) NOT NULL,
                slip_number VARCHAR(255) DEFAULT NULL,
                slip_file_path VARCHAR(255) DEFAULT NULL,
                deposit_date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                province_id INT(11) NOT NULL,
                bank_account_id INT(11) DEFAULT NULL,
                PRIMARY KEY (id),
                INDEX idx_user (user_id),
                INDEX idx_province (province_id),
                INDEX idx_bank_account (bank_account_id),
                INDEX idx_deposit_date (deposit_date)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
            
            if (!$createSuccess) {
                error_log("ERROR: Failed to create " . $tableName . " table: " . $conn->error);
                $conn->query("SET FOREIGN_KEY_CHECKS = 1");
                return false;
            }
            error_log("✓ Created " . $tableName . " table");
            
            // Mettre à jour la clé étrangère pour pointer vers la nouvelle table
            $fkCheck = $conn->query("SELECT CONSTRAINT_NAME FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS 
                WHERE CONSTRAINT_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'cashier_shift_deposit' 
                AND CONSTRAINT_NAME = 'fk_shift_deposit_deposit' 
                AND CONSTRAINT_TYPE = 'FOREIGN KEY'");
            if ($fkCheck && $fkCheck->num_rows > 0) {
                // Supprimer l'ancienne clé étrangère
                @$conn->query("ALTER TABLE cashier_shift_deposit DROP FOREIGN KEY fk_shift_deposit_deposit");
                $fkCheck->free();
            }
            
            // Créer la nouvelle clé étrangère pointant vers bank_deposit_new
            $fkResult = @$conn->query("ALTER TABLE cashier_shift_deposit 
                ADD CONSTRAINT fk_shift_deposit_deposit 
                FOREIGN KEY (deposit_id) REFERENCES " . $tableName . "(id) ON DELETE CASCADE");
            if ($fkResult) {
                error_log("✓ Updated foreign key to point to " . $tableName);
            } else {
                error_log("WARNING: Failed to update foreign key (may not exist): " . $conn->error);
            }
            
            $conn->query("SET FOREIGN_KEY_CHECKS = 1");
        }
        
        // Vérifier que ça a fonctionné
        $verify = $conn->query("SHOW COLUMNS FROM " . $tableName . " WHERE Field = 'id'");
        if ($verify) {
            $verifyCol = $verify->fetch_assoc();
            $verify->free();
            $hasAutoInc = stripos($verifyCol['Extra'] ?? '', 'auto_increment') !== false;
            $hasPK = ($verifyCol['Key'] ?? '') === 'PRI';
            
            if ($hasAutoInc && $hasPK) {
                if ($needsCreation) {
                    error_log("✓✓✓ SUCCESS: " . $tableName . ".id has AUTO_INCREMENT and PRIMARY KEY!");
                }
                return true;
            } else {
                error_log("❌ ERROR: " . $tableName . ".id missing AUTO_INCREMENT or PRIMARY KEY!");
                error_log("Current structure: Key=" . ($verifyCol['Key'] ?? '') . ", Extra=" . ($verifyCol['Extra'] ?? ''));
                return false;
            }
        }
        
        return false;
        
    } catch (Exception $e) {
        error_log("ERROR in ensureBankDepositTableStructure: " . $e->getMessage());
        return false;
    }
}

// Appliquer la correction au chargement de la page
ensureBankDepositTableStructure($conn);
// ============================================

require_login();

$user_id = (int)($_SESSION['user_id'] ?? 0);
$roleName = $_SESSION['role_name'] ?? '';

if ($user_id <= 0) {
    echo "<div class='alert alert-danger m-3'>Impossible de déterminer l'utilisateur actuel.</div>";
    include('../../includes/footer.php');
    exit;
}

// This page is for cashiers (Caissier_Comptable) to deposit closed shifts
// Accountants should use bank_accounts.php for direct deposits
if ($roleName === 'Accountant') {
    header('Location: /masunzu_bar_hotel/modules/accounting/bank_accounts.php');
    exit;
}

// For cashiers, province_id is required
if (empty($_SESSION['province_id'])) {
    $stmtProvince = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    if ($stmtProvince) {
        $stmtProvince->bind_param("i", $user_id);
        $stmtProvince->execute();
        $result = $stmtProvince->get_result();
        $row = $result ? $result->fetch_assoc() : null;
        if ($row && isset($row['province_id'])) {
            $_SESSION['province_id'] = (int)$row['province_id'];
        }
        $stmtProvince->close();
    } else {
        error_log("Failed to prepare province lookup in deposit.php: " . $conn->error);
    }
}

$province_id = (int)($_SESSION['province_id'] ?? 0);
if ($province_id <= 0) {
    echo "<div class='alert alert-danger m-3'>Aucune province n'est associée à votre compte. Contactez l'administrateur.</div>";
    include('../../includes/footer.php');
    exit;
}

$activeShift = null;
$stmtActive = $conn->prepare("
    SELECT id, shift_start
    FROM cashier_balance
    WHERE cashier_id = ? AND shift_end IS NULL
    ORDER BY shift_start DESC
    LIMIT 1
");
if ($stmtActive) {
    $stmtActive->bind_param("i", $user_id);
    $stmtActive->execute();
    $activeShift = $stmtActive->get_result()->fetch_assoc() ?: null;
    $stmtActive->close();
}

function fetchTableColumns(mysqli $conn, string $table): array {
    $columns = [];
    $result = $conn->query("SHOW COLUMNS FROM {$table}");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $columns[$row['Field']] = $row;
        }
        $result->close();
    }
    return $columns;
}

function bindParameters(mysqli_stmt $stmt, string $types, array &$values): void {
    $types = normalizeBindTypes($types, $values);

    $params = [];
    $params[] = $types; // First parameter is the types string (not a reference)
    foreach ($values as $key => $value) {
        $params[] = &$values[$key]; // Subsequent parameters must be references
    }
    if (!empty($params)) {
        call_user_func_array([$stmt, 'bind_param'], $params);
    }
}

function computeShiftAmount(array $shift): float {
    if (array_key_exists('end_balance', $shift) && $shift['end_balance'] !== null) {
        return (float)$shift['end_balance'];
    }

    $begin = (float)($shift['beginning_balance'] ?? 0);
    $deposits = (float)($shift['deposits'] ?? 0);
    $withdrawals = (float)($shift['withdrawals'] ?? 0);
    $calculated = $begin + $deposits - $withdrawals;

    return $calculated > 0 ? $calculated : 0.0;
}

/**
 * When type/value counts don't match, auto-correct by padding/truncating types (defaults to 's') and keep errors out of the UI.
 */
function normalizeBindTypes(string $types, array $values): string {
    $lenTypes = strlen($types);
    $countValues = count($values);
    if ($lenTypes === $countValues) {
        return $types;
    }
    if ($lenTypes < $countValues) {
        $types .= str_repeat('s', $countValues - $lenTypes);
    } else {
        $types = substr($types, 0, $countValues);
    }
    error_log("Adjusted bind types to match values. New types='{$types}' (len {$countValues}), values count={$countValues}");
    return $types;
}

$createShiftDepositSql = "
    CREATE TABLE IF NOT EXISTS cashier_shift_deposit (
        id INT AUTO_INCREMENT PRIMARY KEY,
        shift_id INT NOT NULL,
        deposit_id INT NOT NULL,
        bank_account_id INT DEFAULT NULL,
        amount DECIMAL(15,2) NOT NULL,
        slip_number VARCHAR(255) DEFAULT NULL,
        slip_file_path VARCHAR(255) DEFAULT NULL,
        created_by INT NOT NULL,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY uniq_shift (shift_id),
        KEY idx_bank_account (bank_account_id),
        CONSTRAINT fk_shift_deposit_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE CASCADE,
        CONSTRAINT fk_shift_deposit_deposit FOREIGN KEY (deposit_id) REFERENCES bank_deposit(id) ON DELETE CASCADE,
        CONSTRAINT fk_shift_deposit_user FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE CASCADE,
        CONSTRAINT fk_shift_deposit_bank FOREIGN KEY (bank_account_id) REFERENCES bank_account(id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";

if (!$conn->query($createShiftDepositSql)) {
    echo "<div class='alert alert-danger m-3'>Impossible d'initialiser le suivi des dépôts de session. Veuillez contacter l'administrateur.</div>";
    include('../../includes/footer.php');
    exit;
}

$createShiftTransferSql = "
    CREATE TABLE IF NOT EXISTS cashier_shift_transfer (
        id INT AUTO_INCREMENT PRIMARY KEY,
        shift_id INT NOT NULL,
        transfer_id INT NOT NULL,
        amount DECIMAL(15,2) NOT NULL,
        created_by INT NOT NULL,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        UNIQUE KEY uniq_shift (shift_id),
        KEY idx_transfer (transfer_id),
        CONSTRAINT fk_shift_transfer_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE CASCADE,
        CONSTRAINT fk_shift_transfer_transfer FOREIGN KEY (transfer_id) REFERENCES cash_transfer(id) ON DELETE CASCADE,
        CONSTRAINT fk_shift_transfer_user FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";

if (!$conn->query($createShiftTransferSql)) {
    echo "<div class='alert alert-danger m-3'>Impossible d'initialiser le suivi des transferts de session. Veuillez contacter l'administrateur.</div>";
    include('../../includes/footer.php');
    exit;
}

$schemaWarnings = [];
$schemaBlockingErrors = [];

$shiftDepositColumns = fetchTableColumns($conn, 'cashier_shift_deposit');
if (!isset($shiftDepositColumns['slip_number'])) {
    if ($conn->query("ALTER TABLE cashier_shift_deposit ADD COLUMN slip_number VARCHAR(255) DEFAULT NULL AFTER amount")) {
        $shiftDepositColumns = fetchTableColumns($conn, 'cashier_shift_deposit');
    } else {
        error_log("Failed to add slip_number column to cashier_shift_deposit: " . $conn->error);
    }
}
if (!isset($shiftDepositColumns['slip_file_path'])) {
    if ($conn->query("ALTER TABLE cashier_shift_deposit ADD COLUMN slip_file_path VARCHAR(255) DEFAULT NULL AFTER slip_number")) {
        $shiftDepositColumns = fetchTableColumns($conn, 'cashier_shift_deposit');
    } else {
        error_log("Failed to add slip_file_path column to cashier_shift_deposit: " . $conn->error);
    }
}
$hasShiftSlipNumberColumn = isset($shiftDepositColumns['slip_number']);
$hasShiftSlipFilePathColumn = isset($shiftDepositColumns['slip_file_path']);

$bankDepositColumns = fetchTableColumns($conn, BANK_DEPOSIT_TABLE);

// Ensure id column has AUTO_INCREMENT - critical for INSERT operations
// SOLUTION AGGRESSIVE: Drop and recreate if needed
if (!isset($bankDepositColumns['id'])) {
    $schemaBlockingErrors[] = 'La table ' . BANK_DEPOSIT_TABLE . ' n\'existe pas ou n\'a pas de colonne id. Contactez l\'administrateur.';
    error_log("Critical: " . BANK_DEPOSIT_TABLE . " table missing id column");
} else {
    $idColumn = $bankDepositColumns['id'];
    $isAutoIncrement = (stripos($idColumn['Extra'] ?? '', 'auto_increment') !== false);
    
    if (!$isAutoIncrement) {
                    error_log("⚠ " . BANK_DEPOSIT_TABLE . ".id missing AUTO_INCREMENT - attempting fix...");
        
        // Désactiver temporairement les FK
        $conn->query("SET FOREIGN_KEY_CHECKS = 0");
        
        // Supprimer la contrainte FK si elle existe
        $conn->query("ALTER TABLE cashier_shift_deposit DROP FOREIGN KEY IF EXISTS fk_shift_deposit_deposit");
        
        // Essayer plusieurs stratégies en cascade
        $autoIncrementSet = false;
        
        // Stratégie 1: Modifier la colonne directement
        if (!$autoIncrementSet) {
            if ($conn->query("ALTER TABLE " . BANK_DEPOSIT_TABLE . " MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT PRIMARY KEY")) {
                $autoIncrementSet = true;
                error_log("✓ Strategy 1 succeeded: Direct MODIFY");
            }
        }
        
        // Stratégie 2: Supprimer PRIMARY KEY puis modifier
        if (!$autoIncrementSet) {
            @$conn->query("ALTER TABLE " . BANK_DEPOSIT_TABLE . " DROP PRIMARY KEY");
            if ($conn->query("ALTER TABLE " . BANK_DEPOSIT_TABLE . " MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT, ADD PRIMARY KEY (id)")) {
                $autoIncrementSet = true;
                error_log("✓ Strategy 2 succeeded: DROP PK then MODIFY");
            }
        }
        
        // Stratégie 3: Supprimer et recréer la colonne
        if (!$autoIncrementSet) {
            if ($conn->query("ALTER TABLE " . BANK_DEPOSIT_TABLE . " DROP COLUMN id, ADD COLUMN id INT(11) NOT NULL AUTO_INCREMENT PRIMARY KEY FIRST")) {
                $autoIncrementSet = true;
                error_log("✓ Strategy 3 succeeded: DROP and RECREATE column");
            }
        }
        
        // Stratégie 4: Dernière chance - Reconstruire la table complètement (sans données)
        if (!$autoIncrementSet) {
            $countResult = $conn->query("SELECT COUNT(*) as cnt FROM " . BANK_DEPOSIT_TABLE);
            $rowCount = 0;
            if ($countResult) {
                $countRow = $countResult->fetch_assoc();
                $rowCount = (int)($countRow['cnt'] ?? 0);
                $countResult->free();
            }
            
            if ($rowCount === 0) {
                // Table vide, on peut la recréer
                if ($conn->query("DROP TABLE IF EXISTS " . BANK_DEPOSIT_TABLE) &&
                    $conn->query("CREATE TABLE " . BANK_DEPOSIT_TABLE . " (
                        id INT(11) NOT NULL AUTO_INCREMENT PRIMARY KEY,
                        user_id INT(11) NOT NULL,
                        amount BIGINT(20) NOT NULL,
                        slip_number VARCHAR(255) DEFAULT NULL,
                        slip_file_path VARCHAR(255) DEFAULT NULL,
                        deposit_date DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                        province_id INT(11) NOT NULL,
                        bank_account_id INT(11) DEFAULT NULL,
                        INDEX idx_user (user_id),
                        INDEX idx_province (province_id),
                        INDEX idx_bank_account (bank_account_id),
                        INDEX idx_deposit_date (deposit_date)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci AUTO_INCREMENT=1")) {
                    $autoIncrementSet = true;
                    error_log("✓ Strategy 4 succeeded: Recreated entire table");
                }
            } else {
                error_log("⚠ Cannot recreate table - contains {$rowCount} rows. Manual fix required.");
            }
        }
        
        // Réactiver les FK
        $conn->query("SET FOREIGN_KEY_CHECKS = 1");
        
        // Recréer la contrainte FK
        if ($autoIncrementSet) {
            @$conn->query("ALTER TABLE cashier_shift_deposit 
                ADD CONSTRAINT fk_shift_deposit_deposit 
                FOREIGN KEY (deposit_id) REFERENCES " . BANK_DEPOSIT_TABLE . "(id) ON DELETE CASCADE");
            
            // Vérifier que ça a vraiment marché
            $bankDepositColumns = fetchTableColumns($conn, BANK_DEPOSIT_TABLE);
            if (isset($bankDepositColumns['id'])) {
                $verifyExtra = $bankDepositColumns['id']['Extra'] ?? '';
                if (stripos($verifyExtra, 'auto_increment') === false) {
                    error_log("❌ CRITICAL: AUTO_INCREMENT verification FAILED after all attempts!");
                    $schemaBlockingErrors[] = 'ÉCHEC CRITIQUE: Impossible de configurer AUTO_INCREMENT pour ' . BANK_DEPOSIT_TABLE . '.id après toutes les tentatives. Contactez l\'administrateur.';
                } else {
                    error_log("✓✓✓ AUTO_INCREMENT verified and working for " . BANK_DEPOSIT_TABLE . ".id");
                }
            }
        } else {
            error_log("❌ All strategies failed to set AUTO_INCREMENT");
            $schemaBlockingErrors[] = 'ÉCHEC: Toutes les tentatives de configuration AUTO_INCREMENT pour ' . BANK_DEPOSIT_TABLE . '.id ont échoué. Exécutez le script SQL CREATE_NEW_BANK_DEPOSIT_TABLE.sql manuellement.';
        }
    }
}

if (!isset($bankDepositColumns['slip_number']) && isset($bankDepositColumns['bank_slip'])) {
    if ($conn->query("ALTER TABLE " . BANK_DEPOSIT_TABLE . " CHANGE COLUMN bank_slip slip_number VARCHAR(255) DEFAULT NULL")) {
        $bankDepositColumns = fetchTableColumns($conn, BANK_DEPOSIT_TABLE);
    } else {
        error_log("Failed to rename bank_slip column in bank_deposit: " . $conn->error);
    }
}
if (!isset($bankDepositColumns['slip_file_path'])) {
    $afterColumn = isset($bankDepositColumns['slip_number']) ? 'slip_number' : (isset($bankDepositColumns['bank_slip']) ? 'bank_slip' : 'amount');
    if ($conn->query("ALTER TABLE " . BANK_DEPOSIT_TABLE . " ADD COLUMN slip_file_path VARCHAR(255) DEFAULT NULL AFTER {$afterColumn}")) {
        $bankDepositColumns = fetchTableColumns($conn, BANK_DEPOSIT_TABLE);
    } else {
        error_log("Failed to add slip_file_path column to bank_deposit: " . $conn->error);
    }
}
$slipNumberColumnName = isset($bankDepositColumns['slip_number']) ? 'slip_number' : (isset($bankDepositColumns['bank_slip']) ? 'bank_slip' : null);
$hasSlipFilePathColumn = isset($bankDepositColumns['slip_file_path']);
$hasProvinceColumn = isset($bankDepositColumns['province_id']);
$hasBankAccountIdColumn = isset($bankDepositColumns['bank_account_id']);

$createLedgerSql = "
    CREATE TABLE IF NOT EXISTS bank_account_transaction (
        id INT AUTO_INCREMENT PRIMARY KEY,
        bank_account_id INT NOT NULL,
        province_id INT DEFAULT NULL,
        transaction_type ENUM('CASH_TRANSFER_RECEIVED','BANK_DEPOSIT','REVERSAL','ADJUSTMENT') NOT NULL,
        direction ENUM('CREDIT','DEBIT') NOT NULL,
        amount BIGINT(20) NOT NULL,
        reference_type ENUM('CASH_TRANSFER','CASH_DEPOSIT','MANUAL','OTHER') NOT NULL DEFAULT 'OTHER',
        reference_id INT DEFAULT NULL,
        description VARCHAR(255) DEFAULT NULL,
        created_by INT DEFAULT NULL,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_account_created_at (bank_account_id, created_at),
        INDEX idx_reference (reference_type, reference_id),
        CONSTRAINT fk_bat_account FOREIGN KEY (bank_account_id) REFERENCES bank_account(id) ON DELETE CASCADE,
        CONSTRAINT fk_bat_province FOREIGN KEY (province_id) REFERENCES province(id) ON DELETE SET NULL,
        CONSTRAINT fk_bat_user FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createLedgerSql)) {
    $schemaWarnings[] = 'Le grand livre bancaire est indisponible; les dépôts ne seront pas enregistrés dans l\'historique bancaire.';
    error_log('Failed to ensure bank_account_transaction table: ' . $conn->error);
}
if ($slipNumberColumnName === null) {
    $schemaBlockingErrors[] = 'La table ' . BANK_DEPOSIT_TABLE . ' n\'a pas de colonne pour stocker le numéro de bordereau. Veuillez en ajouter une.';
} elseif ($slipNumberColumnName === 'bank_slip') {
    $schemaWarnings[] = 'Envisagez de renommer ' . BANK_DEPOSIT_TABLE . '.bank_slip en slip_number pour refléter son nouveau but.';
}
if (!$hasSlipFilePathColumn) {
    $schemaWarnings[] = 'La table ' . BANK_DEPOSIT_TABLE . ' n\'a pas de colonne slip_file_path; les bordereaux téléchargés ne seront pas liés.';
}
if (!$hasShiftSlipFilePathColumn) {
    $schemaWarnings[] = 'La table cashier_shift_deposit n\'a pas de colonne slip_file_path; les bordereaux téléchargés ne seront pas liés aux enregistrements de session.';
}
if (!$hasSlipFilePathColumn && !$hasShiftSlipFilePathColumn) {
    $schemaBlockingErrors[] = 'Aucune colonne de base de données n\'est disponible pour stocker le chemin du fichier bordereau téléchargé. Veuillez ajouter slip_file_path à bank_deposit ou cashier_shift_deposit.';
}

$projectRoot = realpath(__DIR__ . '/../../');
if ($projectRoot === false) {
    $schemaBlockingErrors[] = 'Impossible de résoudre la racine du projet pour stocker les fichiers bordereaux bancaires.';
}
$uploadDirectory = $projectRoot ? $projectRoot . '/uploads/bank_slips' : null;
$uploadPublicPathPrefix = '/masunzu_bar_hotel/uploads/bank_slips';

$success = '';
$error = '';

// Ensure bank_account table has AUTO_INCREMENT
$bankAccountCheck = $conn->query("SHOW COLUMNS FROM bank_account WHERE Field = 'id'");
if ($bankAccountCheck) {
    $idColumn = $bankAccountCheck->fetch_assoc();
    $bankAccountCheck->free();
    if ($idColumn && stripos($idColumn['Extra'] ?? '', 'auto_increment') === false) {
        // Try to add AUTO_INCREMENT if missing
        $conn->query("SET FOREIGN_KEY_CHECKS = 0");
        $alterResult = @$conn->query("ALTER TABLE bank_account MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT");
        if ($alterResult) {
            error_log("✓ Added AUTO_INCREMENT to bank_account.id");
        } else {
            error_log("⚠ Could not add AUTO_INCREMENT to bank_account.id: " . $conn->error);
        }
        $conn->query("SET FOREIGN_KEY_CHECKS = 1");
    }
}

$bankAccounts = [];
$bankAccountIndex = [];
$stmtAccounts = $conn->prepare("
    SELECT id, account_number, currency
    FROM bank_account
    WHERE province_id IS NULL OR province_id = ?
    ORDER BY currency, account_number
");
if ($stmtAccounts) {
    $stmtAccounts->bind_param("i", $province_id);
    $stmtAccounts->execute();
    $accountsResult = $stmtAccounts->get_result();
    while ($row = $accountsResult->fetch_assoc()) {
        $row['id'] = (int)$row['id'];
        $bankAccounts[] = $row;
        $bankAccountIndex[$row['id']] = $row;
    }
    $stmtAccounts->close();
} else {
    $error = 'Échec du chargement des comptes bancaires. ' . $conn->error;
}

$selectedShiftId = isset($_POST['shift_id']) ? (int)$_POST['shift_id'] : 0;
$selectedBankAccountId = isset($_POST['bank_account_id']) ? (int)$_POST['bank_account_id'] : 0;
$selectedShiftAmount = '';
$slipNumberValue = '';
$duplicateSlipConflict = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $shift_id = $selectedShiftId;
    $bank_account_id = $selectedBankAccountId;
    $slipNumberValue = trim($_POST['slip_number'] ?? '');
    $uploadedFilePublicPath = null;
    $uploadedFileAbsolutePath = null;

    if (!$error && !empty($schemaBlockingErrors)) {
        $error = implode(' ', $schemaBlockingErrors);
    }
    if (!$error && $activeShift) {
        $error = 'Fermez votre session active avant d\'enregistrer un dépôt bancaire.';
    }
    if (!$error && $projectRoot === false) {
        $error = 'Le chemin de base du répertoire de téléchargement n\'est pas disponible. Contactez l\'administrateur.';
    }
    if (!$error && $slipNumberColumnName === null) {
        $error = 'La table bank_deposit n\'est pas configurée pour stocker les numéros de bordereau. Contactez l\'administrateur.';
    }
    if (!$error && $shift_id <= 0) {
        $error = 'Veuillez sélectionner la session que vous souhaitez déposer.';
    } elseif (!$error) {
        // Check if bank_account_id was actually provided in POST (not just default 0)
        $bankAccountIdProvided = isset($_POST['bank_account_id']) && $_POST['bank_account_id'] !== '';
        if (!$bankAccountIdProvided || !isset($bankAccountIndex[$bank_account_id])) {
            $error = 'Veuillez choisir un compte bancaire de destination valide.';
        }
    }
    if (!$error && $slipNumberValue === '') {
        $error = 'Veuillez fournir le numéro du bordereau bancaire.';
    } elseif (!$error && strlen($slipNumberValue) > 255) {
        $error = 'Le numéro du bordereau bancaire doit contenir 255 caractères ou moins.';
    }
    if (!$error && $slipNumberColumnName !== null) {
        $sqlCheckSlip = "SELECT id FROM " . BANK_DEPOSIT_TABLE . " WHERE {$slipNumberColumnName} = ? LIMIT 1";
        $stmtCheckSlip = $conn->prepare($sqlCheckSlip);
        if (!$stmtCheckSlip) {
            $error = 'Impossible de vérifier le numéro du bordereau bancaire. Veuillez réessayer.';
        } else {
            $stmtCheckSlip->bind_param("s", $slipNumberValue);
            $stmtCheckSlip->execute();
            $resultCheckSlip = $stmtCheckSlip->get_result();
            $existingSlip = $resultCheckSlip ? $resultCheckSlip->fetch_assoc() : null;
            if ($existingSlip) {
                $duplicateSlipConflict = true;
                $error = 'Un dépôt bancaire avec ce numéro de bordereau existe déjà.';
            }
            if ($resultCheckSlip) {
                $resultCheckSlip->close();
            }
            $stmtCheckSlip->close();
        }
    }

    if (!$error) {
        $stmtTransferCheck = $conn->prepare("
            SELECT transfer_id
            FROM cashier_shift_transfer
            WHERE shift_id = ?
            LIMIT 1
        ");
        if ($stmtTransferCheck) {
            $stmtTransferCheck->bind_param("i", $shift_id);
            $stmtTransferCheck->execute();
            $transferRow = $stmtTransferCheck->get_result()->fetch_assoc() ?: null;
            $stmtTransferCheck->close();
            if ($transferRow) {
                $error = 'L\'argent de cette session a déjà été remis au comptable.';
            }
        } else {
            $error = 'Impossible de confirmer le statut de remise de la session. Veuillez réessayer.';
        }
    }

    if (!$error) {
        $file = $_FILES['slip_attachment'] ?? null;
        if (!$file || $file['error'] === UPLOAD_ERR_NO_FILE) {
            $error = 'Veuillez télécharger le fichier bordereau bancaire (PDF ou image).';
        } elseif ($file['error'] !== UPLOAD_ERR_OK) {
            $error = 'Échec du téléchargement du bordereau bancaire. Veuillez réessayer.';
        } else {
            if ($file['size'] > 5 * 1024 * 1024) {
                $error = 'Le fichier bordereau bancaire ne doit pas dépasser 5 Mo.';
            }
            $allowedMime = ['application/pdf', 'image/jpeg', 'image/png'];
            $detectedMime = null;
            if (!$error) {
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                if ($finfo) {
                    $detectedMime = finfo_file($finfo, $file['tmp_name']);
                    finfo_close($finfo);
                }
                if ($detectedMime && !in_array($detectedMime, $allowedMime, true)) {
                    $error = 'Le bordereau bancaire doit être un PDF ou une image (JPG/PNG).';
                }
            }
            if (!$error && $uploadDirectory === null) {
                $error = 'Le répertoire de téléchargement n\'est pas configuré.';
            }
            if (!$error) {
                if (!is_dir($uploadDirectory)) {
                    if (!@mkdir($uploadDirectory, 0755, true) && !is_dir($uploadDirectory)) {
                        $mkdirError = error_get_last();
                        $errorDetails = $mkdirError['message'] ?? 'permission refusée';
                        $error = 'Impossible de créer le répertoire de téléchargement des bordereaux bancaires à ' . $uploadDirectory . ' (' . $errorDetails . ').';
                    }
                }
                if (!$error && !is_writable($uploadDirectory)) {
                    $error = 'Le répertoire de téléchargement des bordereaux bancaires n\'est pas accessible en écriture.';
                }
            }
            if (!$error) {
                $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
                $mimeExtensions = [
                    'application/pdf' => 'pdf',
                    'image/jpeg' => 'jpg',
                    'image/png' => 'png',
                ];
                if ($detectedMime && isset($mimeExtensions[$detectedMime])) {
                    $extension = $mimeExtensions[$detectedMime];
                }
                if (!in_array($extension, ['pdf', 'jpg', 'jpeg', 'png'], true)) {
                    $error = 'Type de fichier non pris en charge. Téléchargez un fichier PDF, JPG ou PNG.';
                } else {
                    $extension = $extension === 'jpeg' ? 'jpg' : $extension;
                    try {
                        $randomSuffix = bin2hex(random_bytes(4));
                    } catch (Throwable $th) {
                        $fallback = null;
                        if (function_exists('openssl_random_pseudo_bytes')) {
                            $fallback = openssl_random_pseudo_bytes(4);
                        }
                        if ($fallback !== false && $fallback !== null) {
                            $randomSuffix = bin2hex($fallback);
                        } else {
                            $fallbackInt = function_exists('random_int') ? random_int(0, 0xFFFFFFFF) : mt_rand(0, 0xFFFFFFFF);
                            $randomSuffix = str_pad(dechex($fallbackInt), 8, '0', STR_PAD_LEFT);
                        }
                    }
                    $fileName = sprintf('slip_%d_%s.%s', $shift_id, $randomSuffix, $extension);
                    $destinationPath = $uploadDirectory . '/' . $fileName;
                    if (!move_uploaded_file($file['tmp_name'], $destinationPath)) {
                        $error = 'Échec du stockage du bordereau bancaire téléchargé. Veuillez réessayer.';
                    } else {
                        $uploadedFileAbsolutePath = $destinationPath;
                        $uploadedFilePublicPath = rtrim($uploadPublicPathPrefix, '/') . '/' . $fileName;
                        @chmod($destinationPath, 0644);
                    }
                }
            }
        }
    }

    if (!$error) {
        $maxAttempts = 2;
        $attempt = 0;
        $transactionSucceeded = false;
        $lockWaitEncountered = false;

        while ($attempt < $maxAttempts && !$transactionSucceeded && !$error) {
            $attempt++;
            $conn->begin_transaction();
            try {
                $stmtShift = $conn->prepare("
                SELECT
                    cb.id,
                    cb.shift_start,
                    cb.shift_end,
                    cb.end_balance,
                    cb.beginning_balance,
                    cb.deposits,
                    cb.withdrawals,
                    csc.counted_amount,
                    COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
                    COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant
                FROM cashier_balance cb
                LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
                WHERE cb.id = ? AND cb.cashier_id = ? AND cb.shift_end IS NOT NULL
                FOR UPDATE
            ");
                if (!$stmtShift) {
                    throw new Exception("Failed to prepare shift lookup: " . $conn->error);
                }
                $stmtShift->bind_param("ii", $shift_id, $user_id);
                $stmtShift->execute();
                $shiftResult = $stmtShift->get_result();
                $shiftRow = $shiftResult ? $shiftResult->fetch_assoc() : null;
                $stmtShift->close();

                if (!$shiftRow) {
                    throw new Exception('La session sélectionnée n\'est pas disponible.');
                }
                
                // Check YES/NO flags to ensure shift hasn't been deposited or transferred
                $is_deposited = strtolower($shiftRow['deposited_to_bank'] ?? 'no') === 'yes';
                $is_transferred = strtolower($shiftRow['transferred_to_accountant'] ?? 'no') === 'yes';
                
                if ($is_deposited) {
                    throw new Exception('Cette session a déjà été déposée à la banque.');
                }
                if ($is_transferred) {
                    throw new Exception('Cette session a déjà été transférée au comptable. Vous ne pouvez pas déposer à la banque et transférer au comptable en même temps.');
                }
                $stmtExistingDeposit = $conn->prepare("
                SELECT deposit_id
                FROM cashier_shift_deposit
                WHERE shift_id = ?
                LIMIT 1
            ");
                if (!$stmtExistingDeposit) {
                    throw new Exception("Failed to verify existing deposits: " . $conn->error);
                }
                $stmtExistingDeposit->bind_param("i", $shift_id);
                $stmtExistingDeposit->execute();
                $existingDeposit = $stmtExistingDeposit->get_result()->fetch_assoc() ?: null;
                $stmtExistingDeposit->close();

                if (!empty($existingDeposit)) {
                    throw new Exception('Cette session a déjà été déposée.');
                }

                $stmtExistingTransfer = $conn->prepare("
                SELECT transfer_id
                FROM cashier_shift_transfer
                WHERE shift_id = ?
                LIMIT 1
            ");
                if (!$stmtExistingTransfer) {
                    throw new Exception("Failed to verify existing cash transfers: " . $conn->error);
                }
                $stmtExistingTransfer->bind_param("i", $shift_id);
                $stmtExistingTransfer->execute();
                $existingTransfer = $stmtExistingTransfer->get_result()->fetch_assoc() ?: null;
                $stmtExistingTransfer->close();

                // Note: We allow deposits even if the shift was transferred to accountant
                // The user requested that transferred amounts should not appear in deposit.php
                // This check is removed to allow deposits of non-transferred shifts
                // The WHERE clause in the query already filters out transferred shifts

                if ($slipNumberColumnName !== null) {
                    $sqlDupSlip = "SELECT id FROM " . BANK_DEPOSIT_TABLE . " WHERE {$slipNumberColumnName} = ? LIMIT 1 FOR UPDATE";
                    $stmtDupSlip = $conn->prepare($sqlDupSlip);
                    if (!$stmtDupSlip) {
                        throw new Exception("Failed to verify slip number uniqueness: " . $conn->error);
                    }
                    $stmtDupSlip->bind_param("s", $slipNumberValue);
                    $stmtDupSlip->execute();
                    $resultDupSlip = $stmtDupSlip->get_result();
                    $duplicateSlip = $resultDupSlip ? $resultDupSlip->fetch_assoc() : null;
                    if ($resultDupSlip) {
                        $resultDupSlip->close();
                    }
                    $stmtDupSlip->close();
                    if (!empty($duplicateSlip)) {
                        $duplicateSlipConflict = true;
                        throw new Exception('A bank deposit with this slip number already exists.');
                    }
                }

                // Use counted_amount if available, otherwise calculate from end_balance or beginning+deposits-withdrawals
                $depositAmount = 0;
                if (isset($shiftRow['counted_amount']) && $shiftRow['counted_amount'] !== null && $shiftRow['counted_amount'] > 0) {
                    $depositAmount = (float)$shiftRow['counted_amount'];
                } else {
                    $depositAmount = computeShiftAmount($shiftRow);
                }
                $depositAmount = round($depositAmount, 2);
                if ($depositAmount <= 0) {
                    throw new Exception('La session sélectionnée n\'a pas de solde à déposer.');
                }

                $manualIdFallback = false;
                $manualIdValue = null;
                $depositTimestamp = date('Y-m-d H:i:s');
                $depositFields = ['user_id', 'amount', 'deposit_date'];
                $depositTypes = 'ids';
                $depositValues = [$user_id, $depositAmount, $depositTimestamp];

                if ($slipNumberColumnName !== null) {
                    $depositFields[] = $slipNumberColumnName;
                    $depositTypes .= 's';
                    $depositValues[] = $slipNumberValue;
                }
                if ($hasSlipFilePathColumn && $uploadedFilePublicPath !== null) {
                    $depositFields[] = 'slip_file_path';
                    $depositTypes .= 's';
                    $depositValues[] = $uploadedFilePublicPath;
                }
                if ($hasBankAccountIdColumn) {
                    $depositFields[] = 'bank_account_id';
                    $depositTypes .= 'i';
                    $depositValues[] = $bank_account_id;
                }
                if ($hasProvinceColumn) {
                    $depositFields[] = 'province_id';
                    $depositTypes .= 'i';
                    $depositValues[] = $province_id;
                }

                // GARANTIR QUE id N'EST JAMAIS DANS LES CHAMPS
                // La fonction ensureBankDepositTableStructure() a déjà vérifié/corrigé la structure
                if (in_array('id', $depositFields, true)) {
                    $idIndex = array_search('id', $depositFields);
                    unset($depositFields[$idIndex]);
                    $depositFields = array_values($depositFields); // Réindexer
                    // Retirer aussi les types et valeurs correspondants
                    if (isset($depositTypes[$idIndex])) {
                        $depositTypes = substr_replace($depositTypes, '', $idIndex, 1);
                    }
                    if (isset($depositValues[$idIndex])) {
                        unset($depositValues[$idIndex]);
                        $depositValues = array_values($depositValues);
                    }
                    error_log("Removed 'id' from INSERT fields - AUTO_INCREMENT will handle it");
                }
                
                $depositPlaceholders = implode(', ', array_fill(0, count($depositFields), '?'));
                $sqlDeposit = "INSERT INTO " . BANK_DEPOSIT_TABLE . " (" . implode(', ', $depositFields) . ") VALUES ({$depositPlaceholders})";
                
                // Log pour debug
                error_log("INSERT query: " . $sqlDeposit);
                error_log("Fields: " . implode(', ', $depositFields));
                error_log("Types: " . $depositTypes);
                error_log("Values: " . print_r($depositValues, true));
                
                // Normalize types and values length (avoid UI-breaking errors)
                $depositTypes = normalizeBindTypes($depositTypes, $depositValues);
                
                // Dernière vérification: s'assurer que la table est correcte avant INSERT
                $finalCheck = $conn->query("SHOW COLUMNS FROM " . BANK_DEPOSIT_TABLE . " WHERE Field = 'id'");
                if ($finalCheck) {
                    $finalIdCol = $finalCheck->fetch_assoc();
                    $finalCheck->free();
                    
                    if (stripos($finalIdCol['Extra'] ?? '', 'auto_increment') === false) {
                        error_log("CRITICAL: " . BANK_DEPOSIT_TABLE . ".id still missing AUTO_INCREMENT before INSERT! Re-running fix...");
                        ensureBankDepositTableStructure($conn);
                    }
                }
                
                // Temporarily disable strict mode for this INSERT (STRICT_TRANS_TABLES can cause issues)
                $originalSqlMode = $conn->query("SELECT @@SESSION.SQL_MODE")->fetch_row()[0];
                $conn->query("SET SESSION SQL_MODE = ''");
                
                $stmtDeposit = $conn->prepare($sqlDeposit);
                if (!$stmtDeposit) {
                    $errorMsg = "Failed to prepare bank deposit insert: " . $conn->error;
                    error_log($errorMsg);
                    // Restore SQL mode
                    $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");
                    throw new Exception($errorMsg);
                }
                
                bindParameters($stmtDeposit, $depositTypes, $depositValues);
                if (!$stmtDeposit->execute()) {
                    $errorMsg = "Unable to record the bank deposit: " . $stmtDeposit->error;
                    error_log($errorMsg);
                    // Restore SQL mode
                    $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");
                    
                    // Si l'erreur concerne id/default, réessayer après correction
                    if (stripos($errorMsg, "id") !== false || stripos($errorMsg, "default") !== false) {
                        error_log("Error related to 'id' field - re-running definitive fix and retrying...");
                        if (ensureBankDepositTableStructure($conn)) {
                            // Retry the INSERT after fix
                            $stmtDeposit->close();
                            $stmtDeposit = $conn->prepare($sqlDeposit);
                            if ($stmtDeposit) {
                                bindParameters($stmtDeposit, $depositTypes, $depositValues);
                                if ($stmtDeposit->execute()) {
                                    error_log("✓✓✓ INSERT succeeded after retry with fixed table!");
                                    $bankDepositId = (int)$conn->insert_id;
                                    $stmtDeposit->close();
                                    $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");
                                    goto skip_original_insert_error;
                                }
                                $stmtDeposit->close();
                            }
                        }
                        throw new Exception($errorMsg . " (La table bank_deposit a été corrigée mais l'INSERT a encore échoué. Contactez l'administrateur.)");
                    }
                    throw new Exception($errorMsg);
                }
                
                // Normal success path
                $bankDepositId = (int)$conn->insert_id;
                $stmtDeposit->close();
                // Restore SQL mode after successful INSERT
                $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");
                skip_original_insert_error:

                $stmtUpdateAccount = $conn->prepare("
                UPDATE bank_account
                SET balance = balance + ?
                WHERE id = ? AND (province_id IS NULL OR province_id = ?)
            ");
                if (!$stmtUpdateAccount) {
                    throw new Exception("Failed to prepare bank account update: " . $conn->error);
                }
                $stmtUpdateAccount->bind_param("dii", $depositAmount, $bank_account_id, $province_id);
                $stmtUpdateAccount->execute();
                if ($stmtUpdateAccount->affected_rows === 0) {
                    throw new Exception('Impossible de mettre à jour le compte bancaire sélectionné. Veuillez vérifier les droits d\'accès.');
                }
                $stmtUpdateAccount->close();

                $linkFields = ['shift_id', 'deposit_id', 'bank_account_id', 'amount', 'created_by'];
                $linkTypes = 'iiidi';
                $linkValues = [$shift_id, $bankDepositId, $bank_account_id, $depositAmount, $user_id];
                if ($hasShiftSlipNumberColumn) {
                    $linkFields[] = 'slip_number';
                    $linkTypes .= 's';
                    $linkValues[] = $slipNumberValue;
                }
                if ($hasShiftSlipFilePathColumn && $uploadedFilePublicPath !== null) {
                    $linkFields[] = 'slip_file_path';
                    $linkTypes .= 's';
                    $linkValues[] = $uploadedFilePublicPath;
                }
                // Normalize link types to avoid runtime bind errors
                $linkTypes = normalizeBindTypes($linkTypes, $linkValues);
                
                $linkPlaceholders = implode(', ', array_fill(0, count($linkFields), '?'));
                $stmtLink = $conn->prepare("
                INSERT INTO cashier_shift_deposit (" . implode(', ', $linkFields) . ")
                VALUES ({$linkPlaceholders})
            ");
                if (!$stmtLink) {
                    throw new Exception("Failed to prepare shift deposit linkage: " . $conn->error);
                }
                bindParameters($stmtLink, $linkTypes, $linkValues);
                if (!$stmtLink->execute()) {
                    throw new Exception("Unable to link the deposit to the shift: " . $stmtLink->error);
                }
                $stmtLink->close();
                
                // Update cashier_shift_closure: set deposited_to_bank='yes'
                // Also validate that transferred_to_accountant is not 'yes' (can't deposit and transfer same shift)
                $stmtCheckClosure = $conn->prepare("
                    SELECT transferred_to_accountant 
                    FROM cashier_shift_closure 
                    WHERE shift_id = ?
                ");
                if ($stmtCheckClosure) {
                    $stmtCheckClosure->bind_param("i", $shift_id);
                    $stmtCheckClosure->execute();
                    $closureResult = $stmtCheckClosure->get_result()->fetch_assoc();
                    $stmtCheckClosure->close();
                    
                    if ($closureResult) {
                        $is_transferred = strtolower($closureResult['transferred_to_accountant'] ?? 'no') === 'yes';
                        if ($is_transferred) {
                            throw new Exception('Cette session a déjà été transférée au comptable. Vous ne pouvez pas déposer à la banque et transférer au comptable en même temps.');
                        }
                        
                        // Update deposited_to_bank to 'yes'
                        // WHERE clause ensures transferred_to_accountant is not 'yes' (both cannot be 'yes' at the same time)
                        $stmtUpdateClosure = $conn->prepare("
                            UPDATE cashier_shift_closure 
                            SET deposited_to_bank = 'yes' 
                            WHERE shift_id = ? AND (transferred_to_accountant = 'no' OR transferred_to_accountant IS NULL)
                        ");
                        if ($stmtUpdateClosure) {
                            $stmtUpdateClosure->bind_param("i", $shift_id);
                            if (!$stmtUpdateClosure->execute()) {
                                error_log("Failed to update deposited_to_bank flag: " . $stmtUpdateClosure->error);
                                throw new Exception('Impossible de mettre à jour le statut de dépôt. La session a peut-être déjà été transférée au comptable.');
                            }
                            if ($stmtUpdateClosure->affected_rows === 0) {
                                throw new Exception('Impossible de mettre à jour le statut de dépôt. La session a peut-être déjà été transférée au comptable.');
                            }
                            $stmtUpdateClosure->close();
                        }
                    }
                }
                
                // Enregistrer dans cashier_account
                // Ensure cashier_account table exists
                $createCashierAccountSql = "
                    CREATE TABLE IF NOT EXISTS cashier_account (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        cashier_id INT NOT NULL,
                        shift_id INT NULL,
                        transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                        direction ENUM('CREDIT','DEBIT') NOT NULL,
                        amount BIGINT(20) NOT NULL,
                        reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                        reference_id INT NULL,
                        description VARCHAR(255) DEFAULT NULL,
                        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                        created_by INT NULL,
                        INDEX idx_cashier_created (cashier_id, created_at),
                        INDEX idx_shift (shift_id),
                        INDEX idx_reference (reference_type, reference_id),
                        CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                        CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                        CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                ";
                @$conn->query($createCashierAccountSql);
                
                $accountNumber = $bankAccountIndex[$bank_account_id]['account_number'] ?? ('#' . $bank_account_id);
                $description = sprintf('Dépôt bancaire: Session %d (Bordereau %s) - Compte %s', $shift_id, $slipNumberValue, $accountNumber);
                $stmtAccount = $conn->prepare("
                    INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                    VALUES (?, ?, 'BANK_DEPOSIT', 'DEBIT', ?, 'BANK_DEPOSIT', ?, ?, ?)
                ");
                if ($stmtAccount) {
                    $stmtAccount->bind_param("iiiiiis", $user_id, $shift_id, $depositAmount, $bankDepositId, $user_id, $description);
                    if (!$stmtAccount->execute()) {
                        error_log("Failed to record BANK_DEPOSIT in cashier_account: " . $stmtAccount->error);
                    }
                    $stmtAccount->close();
                }
                
                adjustProvinceCash($conn, $province_id, (float)$depositAmount, 'BANK_DEPOSIT', 'BANK_DEPOSIT', $bankDepositId, $user_id, $description, $depositTimestamp);

                $ledgerStmt = $conn->prepare('INSERT INTO bank_account_transaction (bank_account_id, province_id, transaction_type, direction, amount, reference_type, reference_id, description, created_by, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
                if (!$ledgerStmt) {
                    throw new Exception('Failed to record bank ledger entry: ' . $conn->error);
                }
                $ledgerTxnType = 'BANK_DEPOSIT';
                $ledgerDirection = 'CREDIT';
                $ledgerReferenceType = 'CASH_DEPOSIT';
                $ledgerDescription = sprintf('Shift %d deposit #%d', $shift_id, $bankDepositId);
                $ledgerCreatedAt = $depositTimestamp;
                $ledgerStmt->bind_param(
                    'iissdsisis',
                    $bank_account_id,
                    $province_id,
                    $ledgerTxnType,
                    $ledgerDirection,
                    $depositAmount,
                    $ledgerReferenceType,
                    $bankDepositId,
                    $ledgerDescription,
                    $user_id,
                    $ledgerCreatedAt
                );
                if (!$ledgerStmt->execute()) {
                    $ledgerError = $ledgerStmt->error ?: 'unknown error';
                    $ledgerStmt->close();
                    throw new Exception('Failed to write bank account transaction: ' . $ledgerError);
                }
                $ledgerStmt->close();

                if (function_exists('log_action')) {
                    $shiftEnd = $shiftRow['shift_end'] ?? $shiftRow['shift_start'] ?? '';
                    $shiftLabel = $shiftEnd ? date('Y-m-d H:i', strtotime($shiftEnd)) : 'unknown';
                    log_action(
                        $user_id,
                        'shift_bank_deposit',
                        "Deposited {$depositAmount} BIF from shift {$shift_id} (ended {$shiftLabel}) to bank account {$accountNumber} with slip {$slipNumberValue}"
                    );
                }

                $conn->commit();
                $transactionSucceeded = true;
                $uploadedFileAbsolutePath = null;

                $formattedAmount = number_format($depositAmount, 0, '.', ',');
                $shiftEndDisplay = $shiftRow['shift_end'] ? date('d M Y H:i', strtotime($shiftRow['shift_end'])) : 'heure inconnue';
                $success = "Dépôt de {$formattedAmount} BIF enregistré pour la session se terminant le {$shiftEndDisplay}.";

                $selectedShiftId = 0;
                $selectedBankAccountId = 0;
                $selectedShiftAmount = '';
                $slipNumberValue = '';
            } catch (Throwable $th) {
                $conn->rollback();
                $message = $th->getMessage();
                $isLockWait = stripos($message, 'lock wait timeout exceeded') !== false;
                if ($isLockWait) {
                    $lockWaitEncountered = true;
                }
                if (stripos($message, 'bank deposit with this slip number already exists') !== false) {
                    $duplicateSlipConflict = true;
                    $error = 'Un dépôt bancaire avec ce numéro de bordereau existe déjà.';
                }
                if (stripos($message, 'Duplicate entry') !== false) {
                    $existingDepositId = null;
                    $stmtExistingLink = $conn->prepare("SELECT deposit_id FROM cashier_shift_deposit WHERE shift_id = ? LIMIT 1");
                    if ($stmtExistingLink) {
                        $stmtExistingLink->bind_param("i", $shift_id);
                        $stmtExistingLink->execute();
                        $existingDepositId = $stmtExistingLink->get_result()->fetch_assoc()['deposit_id'] ?? null;
                        $stmtExistingLink->close();
                    }
                    if ($existingDepositId) {
                        if ($uploadedFileAbsolutePath && file_exists($uploadedFileAbsolutePath)) {
                            @unlink($uploadedFileAbsolutePath);
                        }
                        $formattedAmount = number_format($depositAmount, 2, '.', ',');
                        $shiftEndDisplay = $shiftRow['shift_end'] ? date('d M Y H:i', strtotime($shiftRow['shift_end'])) : 'heure inconnue';
                        $success = "Dépôt déjà enregistré (#{$existingDepositId}) pour la session se terminant le {$shiftEndDisplay}. Montant : {$formattedAmount} BIF.";
                        $selectedShiftId = 0;
                        $selectedBankAccountId = 0;
                        $selectedShiftAmount = '';
                        $slipNumberValue = '';
                        $error = '';
                        $transactionSucceeded = true;
                        break;
                    } else {
                        $error = 'La session a été traitée par une autre session. Veuillez actualiser la liste des sessions.';
                    }
                }
                if ($isLockWait && $attempt < $maxAttempts) {
                    usleep(200000);
                    continue;
                }
                if ($uploadedFileAbsolutePath && file_exists($uploadedFileAbsolutePath)) {
                    @unlink($uploadedFileAbsolutePath);
                }
                if ($isLockWait && !$error) {
                    $error = 'La base de données est occupée. Veuillez réessayer dans quelques secondes.';
                    error_log('Lock wait timeout while recording bank deposit: ' . $message);
                } elseif (!$error) {
                    $error = $message;
                }
            }
        }

        if (!$transactionSucceeded && !$error && $lockWaitEncountered) {
            if ($uploadedFileAbsolutePath && file_exists($uploadedFileAbsolutePath)) {
                @unlink($uploadedFileAbsolutePath);
            }
            $error = 'La base de données est occupée. Veuillez réessayer dans quelques secondes.';
        }
    } else {
        if ($uploadedFileAbsolutePath && file_exists($uploadedFileAbsolutePath)) {
            @unlink($uploadedFileAbsolutePath);
        }
    }
}

$availableShifts = [];
$stmtShifts = $conn->prepare("
    SELECT
        cb.id,
        cb.shift_start,
        cb.shift_end,
        cb.end_balance,
        cb.beginning_balance,
        cb.deposits,
        cb.withdrawals,
        COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
        COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
        csc.counted_amount
    FROM cashier_balance cb
    INNER JOIN user u ON cb.cashier_id = u.id
    LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
    WHERE cb.cashier_id = ? AND u.province_id = ? AND cb.shift_end IS NOT NULL 
        AND (COALESCE(csc.deposited_to_bank, 'no') = 'no' OR csc.deposited_to_bank IS NULL)
        AND (COALESCE(csc.transferred_to_accountant, 'no') = 'no' OR csc.transferred_to_accountant IS NULL)
    ORDER BY cb.shift_end DESC
");
if ($stmtShifts) {
    $stmtShifts->bind_param("ii", $user_id, $province_id);
    $stmtShifts->execute();
    $resultShifts = $stmtShifts->get_result();
        while ($row = $resultShifts->fetch_assoc()) {
            // Use counted_amount if available, otherwise calculate from end_balance or beginning+deposits-withdrawals
            $amount = 0;
            if (isset($row['counted_amount']) && $row['counted_amount'] !== null && $row['counted_amount'] > 0) {
                $amount = (float)$row['counted_amount'];
            } else {
                $amount = computeShiftAmount($row);
            }
            if ($amount <= 0) {
                continue;
            }
            $startLabel = $row['shift_start'] ? date('d M Y H:i', strtotime($row['shift_start'])) : 'Début inconnu';
            $endLabel = $row['shift_end'] ? date('d M Y H:i', strtotime($row['shift_end'])) : 'Fin inconnue';
            $availableShifts[] = [
                'id' => (int)$row['id'],
                'amount' => (int)round($amount),
                'label' => sprintf('%s - %s - %s BIF', $startLabel, $endLabel, number_format((int)round($amount), 0, '.', ','))
            ];
        }
    $stmtShifts->close();
} else {
    $error = $error ?: 'Échec du chargement des sessions disponibles.';
}

if ($selectedShiftId > 0) {
    foreach ($availableShifts as $shift) {
        if ($shift['id'] === $selectedShiftId) {
            $selectedShiftAmount = number_format($shift['amount'], 0, '.', '');
            break;
        }
    }
} elseif (!empty($availableShifts)) {
    $selectedShiftId = $availableShifts[0]['id'];
    $selectedShiftAmount = number_format($availableShifts[0]['amount'], 0, '.', '');
}

$formDisabled = empty($availableShifts) || empty($bankAccounts) || $activeShift !== null || !empty($schemaBlockingErrors);
$activeShiftStartLabel = null;
    if ($activeShift) {
        $activeShiftStartLabel = !empty($activeShift['shift_start'])
        ? date('d M Y H:i', strtotime($activeShift['shift_start']))
        : 'Début inconnu';
}
?>
<style>
.deposit-container {
    max-width: 960px;
    margin: 2rem auto;
    padding: 0 1rem;
}

.deposit-header {
    text-align: center;
    margin-bottom: 2rem;
    position: relative;
}

.deposit-header::before {
    content: '🏦';
    font-size: 2.5rem;
    display: block;
    margin-bottom: 0.75rem;
}

.deposit-header h3 {
    color: #4B2F1F;
    font-size: 2rem;
    font-weight: bold;
    margin: 0;
    text-shadow: 2px 2px 4px rgba(0,0,0,0.1);
}

.deposit-card {
    background: linear-gradient(135deg, #F4F0E4 0%, #F4A261 100%);
    border: none;
    border-radius: 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.2);
    overflow: hidden;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.deposit-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 15px 40px rgba(0,0,0,0.3);
}

.deposit-card .card-body {
    padding: 2rem;
    background: rgba(255,255,255,0.9);
    backdrop-filter: blur(10px);
}

.form-group {
    margin-bottom: 1.5rem;
    position: relative;
}

.form-label {
    font-weight: 600;
    color: #4B2F1F;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    font-size: 1rem;
}

.form-label::before {
    content: '';
    width: 24px;
    height: 24px;
    margin-right: 0.5rem;
    background-size: contain;
    background-repeat: no-repeat;
    opacity: 0.8;
}

.label-shift::before { content: '⏰'; }
.label-slip::before { content: '📄'; }
.label-account::before { content: '🏛️'; }
.label-upload::before { content: '📎'; }
.label-amount::before { content: '💰'; }

.form-control, .form-select {
    border: 2px solid #4B2F1F;
    border-radius: 10px;
    padding: 0.75rem 1rem;
    font-size: 1rem;
    transition: all 0.3s ease;
    background-color: #fff;
}

.form-control:focus, .form-select:focus {
    border-color: #F4A261;
    box-shadow: 0 0 0 3px rgba(244, 162, 97, 0.2);
    outline: none;
}

.form-control:hover, .form-select:hover {
    border-color: #F4A261;
}

.btn-deposit {
    background: linear-gradient(135deg, #4B2F1F 0%, #2C1810 100%);
    color: #F4F0E4;
    border: none;
    border-radius: 12px;
    padding: 0.75rem 1.5rem;
    font-size: 1rem;
    font-weight: bold;
    text-transform: uppercase;
    letter-spacing: 1px;
    cursor: pointer;
    transition: all 0.3s ease;
    width: 100%;
    box-shadow: 0 4px 15px rgba(0,0,0,0.2);
}

.btn-deposit:hover:not(:disabled) {
    background: linear-gradient(135deg, #2C1810 0%, #4B2F1F 100%);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(0,0,0,0.3);
}

.btn-deposit:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

.alert {
    border-radius: 10px;
    border: none;
    font-weight: 500;
}

.alert-success {
    background-color: #d4edda;
    color: #155724;
}

.alert-danger {
    background-color: #f8d7da;
    color: #721c24;
}

.alert-duplicate-slip {
    font-size: 1.2rem;
    font-weight: 700;
    padding: 1.5rem;
    border: 2px solid var(--bs-danger-border-subtle, #f5c2c7);
    background-color: var(--bs-danger-bg-subtle, #f8d7da);
    color: var(--bs-danger-text, #842029);
    text-align: center;
}

.alert-warning {
    background-color: #fff3cd;
    color: #856404;
}

.alert-info {
    background-color: #d1ecf1;
    color: #0c5460;
}

@media (max-width: 768px) {
    .deposit-container {
        margin: 1rem auto;
        padding: 0 0.5rem;
    }

    .deposit-header::before {
        font-size: 2rem;
    }

    .deposit-header h3 {
        font-size: 1.5rem;
    }

    .deposit-card .card-body {
        padding: 1.5rem 1rem;
    }

    .form-group {
        margin-bottom: 1.25rem;
    }

    .form-label {
        font-size: 0.9rem;
    }

    .form-control, .form-select {
        padding: 0.6rem 0.8rem;
        font-size: 0.9rem;
    }

    .btn-deposit {
        padding: 0.7rem 1.25rem;
        font-size: 0.9rem;
    }
}

.deposit-actions {
    display: flex;
    justify-content: flex-end;
    margin-bottom: 15px;
}

.deposit-actions a {
    background: #4B2F1F;
    color: #F4F0E4;
    padding: 8px 16px;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: opacity 0.2s ease;
}

.deposit-actions a:hover {
    opacity: 0.85;
}
</style>

<div class="deposit-container">
    <div class="deposit-header">
        <h3>Déposer une Session Fermée à la Banque</h3>
    </div>
    <div class="deposit-actions">
        <a href="/masunzu_bar_hotel/modules/banking/deposit_history.php">Historique Dépôts Banque</a>
    </div>
    <?php if ($success): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
        <?php $errorClass = $duplicateSlipConflict ? 'alert alert-danger alert-duplicate-slip' : 'alert alert-danger'; ?>
        <div class="<?php echo htmlspecialchars($errorClass, ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <?php if (!empty($schemaBlockingErrors)): ?>
        <?php foreach ($schemaBlockingErrors as $blocking): ?>
            <div class="alert alert-danger"><?php echo htmlspecialchars($blocking); ?></div>
        <?php endforeach; ?>
    <?php endif; ?>
    <?php if (!empty($schemaWarnings)): ?>
        <?php foreach ($schemaWarnings as $warning): ?>
            <div class="alert alert-warning"><?php echo htmlspecialchars($warning); ?></div>
        <?php endforeach; ?>
    <?php endif; ?>

    <?php if (empty($availableShifts)): ?>
        <div class="alert alert-info">Toutes vos sessions fermées ont déjà été déposées ou aucune session n'est encore fermée.</div>
    <?php endif; ?>
    <?php if (empty($bankAccounts)): ?>
        <div class="alert alert-warning">Aucun compte bancaire n'est configuré pour votre province. Veuillez contacter les finances.</div>
    <?php endif; ?>
    <?php if ($activeShift): ?>
        <div class="alert alert-warning">
            Vous avez encore une session active (démarrée le <?php echo htmlspecialchars($activeShiftStartLabel, ENT_QUOTES, 'UTF-8'); ?>).
            Fermez la session dans la Gestion du Solde avant d'enregistrer un dépôt bancaire.
        </div>
    <?php endif; ?>

    <form method="POST" enctype="multipart/form-data" class="deposit-card">
        <div class="card-body">
            <div class="form-group">
                <label class="form-label label-shift">Session Fermée</label>
                <select name="shift_id" id="shift_id" class="form-select" <?php echo $formDisabled ? 'disabled' : ''; ?> required>
                    <option value="">Sélectionner une session fermée</option>
                    <?php foreach ($availableShifts as $shift): ?>
                        <option
                            value="<?php echo (int)$shift['id']; ?>"
                            data-amount="<?php echo number_format((int)round($shift['amount']), 0, '.', ''); ?>"
                            <?php echo $selectedShiftId === $shift['id'] ? 'selected' : ''; ?>
                        >
                            <?php echo htmlspecialchars($shift['label']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="form-group">
                <label class="form-label label-slip">Numéro du Bordereau Bancaire</label>
                <input
                    type="text"
                    name="slip_number"
                    class="form-control"
                    value="<?php echo htmlspecialchars($slipNumberValue); ?>"
                    maxlength="255"
                    <?php echo $formDisabled ? 'disabled' : 'required'; ?>
                >
            </div>

            <div class="form-group">
                <label class="form-label label-account">Compte Bancaire</label>
                <select name="bank_account_id" class="form-select" <?php echo $formDisabled ? 'disabled' : ''; ?> required>
                    <option value="">Sélectionner un compte bancaire</option>
                    <?php foreach ($bankAccounts as $account): ?>
                        <option
                            value="<?php echo (int)$account['id']; ?>"
                            <?php echo $selectedBankAccountId === $account['id'] ? 'selected' : ''; ?>
                        >
                            <?php
                                $label = sprintf(
                                    '%s (%s)',
                                    $account['account_number'],
                                    $account['currency']
                                );
                                echo htmlspecialchars($label);
                            ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="form-group">
                <label class="form-label label-upload">Télécharger le Bordereau Bancaire (PDF, JPG, PNG)</label>
                <input
                    type="file"
                    name="slip_attachment"
                    class="form-control"
                    accept=".pdf,.jpg,.jpeg,.png"
                    <?php echo $formDisabled ? 'disabled' : 'required'; ?>
                >
                <small class="form-text text-muted" style="margin-top: 0.5rem; display: block;">Taille maximale du fichier : 5 Mo.</small>
            </div>

            <div class="form-group">
                <label class="form-label label-amount">Montant (BIF)</label>
                <input
                    type="number"
                    id="amount"
                    class="form-control"
                    value="<?php echo htmlspecialchars($selectedShiftAmount); ?>"
                    placeholder="Sélectionner une session pour afficher le montant"
                    readonly
                >
            </div>

            <button class="btn-deposit" <?php echo $formDisabled ? 'disabled' : ''; ?>>
                Enregistrer le Dépôt
            </button>
        </div>
    </form>
</div>
<script>
document.addEventListener('DOMContentLoaded', function () {
    var shiftSelect = document.getElementById('shift_id');
    var amountInput = document.getElementById('amount');
    var depositForm = document.querySelector('form.deposit-card');

    function syncAmount() {
        if (!shiftSelect || !amountInput) {
            return;
        }
        var option = shiftSelect.options[shiftSelect.selectedIndex];
        if (option && option.dataset.amount) {
            var amount = parseInt(option.dataset.amount, 10);
            amountInput.value = isNaN(amount) ? '' : amount.toString();
        } else {
            amountInput.value = '';
        }
    }

    if (shiftSelect && amountInput) {
        shiftSelect.addEventListener('change', syncAmount);
        syncAmount();
    }

    if (depositForm && shiftSelect) {
        var allowSubmit = false;
        var lastMessage = null;

        function handleSubmit(event) {
            if (allowSubmit || !shiftSelect.value) {
                allowSubmit = false;
                return;
            }
            event.preventDefault();

            var payload = new FormData();
            payload.append('shift_id', shiftSelect.value);

            fetch('check_shift_status.php', {
                method: 'POST',
                body: payload,
                credentials: 'same-origin'
            })
                .then(function (response) {
                    return response.json();
                })
                .then(function (data) {
                    if (data && data.available) {
                        allowSubmit = true;
                        depositForm.submit();
                        return;
                    }
                    allowSubmit = false;
                    if (data && data.message && data.message !== lastMessage) {
                        lastMessage = data.message;
                        alert(data.message);
                    }
                    var selectedValue = shiftSelect.value;
                    var option = shiftSelect.querySelector('option[value="' + selectedValue + '"]');
                    if (option) {
                        option.remove();
                    }
                    shiftSelect.selectedIndex = 0;
                    syncAmount();
                })
                .catch(function () {
                    allowSubmit = true;
                    depositForm.submit();
                });
        }

        depositForm.addEventListener('submit', handleSubmit);
    }
});
</script>
<?php include('../../includes/footer.php'); ?>
