<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in view_all_stock.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Restrict to DG, DGA, Board, Admin
if (!in_array(mbh_normalize_role_name($_SESSION['role_name'] ?? ''), ['DG', 'DGA', 'Board', 'Admin'], true)) {
    error_log("Unauthorized access to view_all_stock.php by user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    $conn->close();
    header("Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php");
    exit;
}

$provinceRows = [];
$provinceOptions = [];
$provinceQuery = $conn->query("SELECT id, name FROM province ORDER BY name");
if ($provinceQuery instanceof mysqli_result) {
    while ($row = $provinceQuery->fetch_assoc()) {
        $provinceRows[] = [
            'id' => (int)$row['id'],
            'name' => $row['name'],
        ];
        $slug = strtolower($row['name'] ?? '');
        $provinceOptions[] = [
            'name' => $row['name'],
            'slug' => $slug,
        ];
    }
    $provinceQuery->free();
}

// ==================== MAIN STOCK OVERVIEW ====================
$main_stock_products = [];
$stmt = $conn->prepare("
    SELECT pr.id,
           pr.name,
           pr.crate_quantity,
           pr.price_per_crate,
           pr.low_stock_alert_crates,
           pr.low_stock_alert_units,
           COALESCE(ms.quantity, 0) as total_quantity
    FROM product pr
    LEFT JOIN main_stock ms ON pr.id = ms.product_id
    ORDER BY pr.name
");
$stmt->execute();
$main_stock_products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();
$productCatalog = [];
foreach ($main_stock_products as $productRow) {
    $productCatalog[(int)$productRow['id']] = $productRow;
}

// ==================== STOCK BY PROVINCE (province_stock) ====================
$low_stock_threshold_crates = 10; // crates
$stock_by_province = [];
$low_stock_items = [];
$low_stock_lookup = [];
$stmt = $conn->prepare("
    SELECT p.id AS province_id,
           p.name AS province_name,
           pr.id AS product_id,
           pr.name AS product_name,
           pr.crate_quantity,
           pr.price_per_crate,
           pr.low_stock_alert_crates,
           pr.low_stock_alert_units,
           COALESCE(ps.quantity, 0) AS stock_qty,
           ps.unit_type
    FROM province p
    LEFT JOIN province_stock ps ON ps.province_id = p.id
    LEFT JOIN product pr ON pr.id = ps.product_id
    ORDER BY p.name, pr.name
");
$stmt->execute();
$stock_result = $stmt->get_result();

$province_data = [];
$seenProducts = [];
        foreach ($provinceRows as $provinceRow) {
            $province_data[$provinceRow['id']] = [
                'province_name' => $provinceRow['name'],
                'products' => [],
                'total_crates' => 0,
                'total_value' => 0,
            ];
        }
while ($row = $stock_result->fetch_assoc()) {
    $province_id = (int)$row['province_id'];
    $product_id = (int)($row['product_id'] ?? 0);
    $crateQuantity = max(1, (int)($row['crate_quantity'] ?? 12));
    $storedQuantity = (int)$row['stock_qty'];
    $unitType = strtolower($row['unit_type'] ?? 'crates');
    $pricePerCrate = (float)($row['price_per_crate'] ?? 0);
    $crates = $unitType === 'crates' ? $storedQuantity : ($storedQuantity / $crateQuantity);
    $bottles = $unitType === 'crates' ? ($storedQuantity * $crateQuantity) : $storedQuantity;
    $totalValue = $crates * $pricePerCrate;
    $thresholdCrates = (int)($row['low_stock_alert_crates'] ?? 0);
    $thresholdUnits = (int)($row['low_stock_alert_units'] ?? 0);
    $isLowStock = false;
    if ($thresholdCrates > 0) {
        $isLowStock = $crates <= $thresholdCrates;
    } elseif ($thresholdUnits > 0) {
        $isLowStock = $bottles <= $thresholdUnits;
    } else {
        $isLowStock = $crates <= $low_stock_threshold_crates;
    }
    if (!isset($province_data[$province_id])) {
        $province_data[$province_id] = [
            'province_name' => $row['province_name'] ?? ('Province #' . $province_id),
            'products' => [],
            'total_crates' => 0,
            'total_value' => 0,
        ];
    }

    if ($storedQuantity > 0) {
        $province_data[$province_id]['products'][] = [
            'product_name' => $row['product_name'],
            'product_id' => (int)$row['product_id'],
            'crate_quantity' => $crateQuantity,
            'bottles' => $bottles,
            'crates' => $crates,
            'price_per_crate' => $pricePerCrate,
            'total_value' => $totalValue,
            'threshold_crates' => $thresholdCrates,
            'threshold_units' => $thresholdUnits,
        ];
        $province_data[$province_id]['total_crates'] += $crates;
        $province_data[$province_id]['total_value'] += $totalValue;
        $seenProducts[$province_id][(int)$row['product_id']] = true;

        if ($isLowStock) {
            $key = sprintf('%s-%s', $province_id, $product_id);
            if (!isset($low_stock_lookup[$key])) {
                $low_stock_lookup[$key] = true;
                $low_stock_items[] = [
                    'province' => $row['province_name'],
                    'product' => $row['product_name'],
                    'crates' => $crates,
                    'bottles' => $bottles,
                    'value' => $totalValue,
                    'threshold_crates' => $thresholdCrates,
                    'threshold_units' => $thresholdUnits,
                ];
            }
        }
    }
}

$province_data = array_values($province_data);
foreach ($province_data as $data) {
    $stock_by_province[] = $data;
}
$stmt->close();

$total_main_stock = 0;
$total_main_value = 0;
foreach ($main_stock_products as $product) {
    $total_main_stock += (int)$product['total_quantity'];
    $total_main_value += ((int)$product['total_quantity']) * (float)($product['price_per_crate'] ?? 0);
}

 $total_province_crates = 0;
 $total_province_value = 0;
 $total_product_lines = 0;
 foreach ($stock_by_province as $province) {
     $total_province_crates += $province['total_crates'];
     $total_province_value += $province['total_value'];
     $total_product_lines += count($province['products']);
 }

$provinceOptions = [];
$provinceResult = $conn->query("SELECT name FROM province ORDER BY name");
if ($provinceResult instanceof mysqli_result) {
    while ($row = $provinceResult->fetch_assoc()) {
        $slug = strtolower($row['name'] ?? '');
        $provinceOptions[] = [
            'name' => $row['name'],
            'slug' => $slug,
        ];
    }
    $provinceResult->free();
}

$region_stock_summary = [];
$regionStockStmt = $conn->prepare("
    SELECT r.id AS region_id,
           r.name AS region_name,
           p.id AS product_id,
           p.name AS product_name,
           p.crate_quantity,
           p.price_per_crate,
           COALESCE(SUM(rs.quantity), 0) AS total_quantity
    FROM region_stock rs
    JOIN region r ON rs.region_id = r.id
    JOIN product p ON rs.product_id = p.id
    GROUP BY r.id, r.name, p.id, p.name, p.crate_quantity, p.price_per_crate
    ORDER BY r.name, p.name
");
if ($regionStockStmt) {
    $regionStockStmt->execute();
    $result = $regionStockStmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $regionId = (int)$row['region_id'];
        if (!isset($region_stock_summary[$regionId])) {
            $region_stock_summary[$regionId] = [
                'region_name' => $row['region_name'] ?? 'Région',
                'products' => [],
                'total_crates' => 0,
                'total_value' => 0,
            ];
        }
        $quantity = (int)($row['total_quantity'] ?? 0);
        $price = (int)($row['price_per_crate'] ?? 0);
        $value = $quantity * $price;
        $region_stock_summary[$regionId]['products'][] = [
            'product_name' => $row['product_name'],
            'quantity' => $quantity,
            'value' => $value,
        ];
        $region_stock_summary[$regionId]['total_crates'] += $quantity;
        $region_stock_summary[$regionId]['total_value'] += $value;
    }
    $regionStockStmt->close();
}

$transfer_status_labels = [
    'pending' => 'En attente',
    'confirmed' => 'Reçu',
    'rejected' => 'Rejeté',
];

$transfer_status_summary = [];
foreach ($transfer_status_labels as $code => $label) {
    $transfer_status_summary[$code] = [
        'label' => $label,
        'count' => 0,
        'quantity' => 0,
    ];
}

$stmt = $conn->prepare("
    SELECT status,
           COUNT(*) AS transfer_count,
           COALESCE(SUM(quantity), 0) AS total_quantity
    FROM stock_transfer
    GROUP BY status
");
if ($stmt) {
    $stmt->execute();
    $status_rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    foreach ($status_rows as $row) {
        $statusKey = strtolower($row['status'] ?? '');
        if (isset($transfer_status_summary[$statusKey])) {
            $transfer_status_summary[$statusKey]['count'] = (int)$row['transfer_count'];
            $transfer_status_summary[$statusKey]['quantity'] = (int)$row['total_quantity'];
        }
    }
    $stmt->close();
}

$province_transfer_summary = [];
foreach ($provinceRows as $provinceRow) {
    $province_statuses = [];
    foreach ($transfer_status_labels as $code => $label) {
        $province_statuses[$code] = ['count' => 0, 'quantity' => 0];
    }
    $province_transfer_summary[$provinceRow['id']] = [
        'province_name' => $provinceRow['name'],
        'statuses' => $province_statuses,
        'total_quantity' => 0,
    ];
}

$stmt = $conn->prepare("
    SELECT p.id AS province_id,
           st.status,
           COUNT(*) AS transfer_count,
           COALESCE(SUM(st.quantity), 0) AS total_quantity
    FROM province p
    LEFT JOIN stock_transfer st ON st.province_id = p.id
    GROUP BY p.id, st.status
    ORDER BY p.name, st.status
");
if ($stmt) {
    $stmt->execute();
    $transfer_rows = $stmt->get_result();
    while ($row = $transfer_rows->fetch_assoc()) {
        $provinceId = (int)($row['province_id'] ?? 0);
        $statusKey = strtolower($row['status'] ?? '');
        if (!isset($province_transfer_summary[$provinceId]) || !isset($transfer_status_summary[$statusKey])) {
            continue;
        }
        $province_transfer_summary[$provinceId]['statuses'][$statusKey]['count'] = (int)$row['transfer_count'];
        $province_transfer_summary[$provinceId]['statuses'][$statusKey]['quantity'] = (int)$row['total_quantity'];
        $province_transfer_summary[$provinceId]['total_quantity'] += (int)$row['total_quantity'];
    }
    $stmt->close();
}

$conn->close();

include __DIR__ . '/../../includes/header.php';
?>
<style>
    .stock-container {
        padding: 20px 4vw 60px;
        background-color: #F4F0E4;
        min-height: 80vh;
    }
    .stock-content {
        background-color: #FFFFFF;
        padding: 30px;
        border-radius: 18px;
        box-shadow: 0 12px 30px rgba(75,47,31,0.08);
    }
    @media (max-width: 768px) {
        .stock-content {
            padding: 20px;
        }
    }
    .section-title {
        color: #4B2F1F;
        font-size: 24px;
        font-weight: 600;
        margin: 30px 0 15px 0;
        border-bottom: 2px solid #4B2F1F;
        padding-bottom: 10px;
    }
    .metric-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 18px;
    }
    .metric-card {
        padding: 18px;
        border-radius: 14px;
        background: linear-gradient(135deg, #4CAF50 0%, #81C784 100%);
        color: #FFFFFF;
        box-shadow: 0 10px 20px rgba(76, 175, 80, 0.25);
    }
    .metric-card.alt {
        background: linear-gradient(135deg, #F4A261 0%, #FFB662 100%);
        box-shadow: 0 10px 20px rgba(244, 162, 97, 0.25);
    }
    .metric-card.danger {
        background: linear-gradient(135deg, #D84315 0%, #FF7043 100%);
        box-shadow: 0 10px 20px rgba(216, 67, 21, 0.25);
    }
    .metric-label {
        font-size: 13px;
        text-transform: uppercase;
        letter-spacing: .08em;
        opacity: .9;
    }
    .metric-value {
        font-size: 30px;
        font-weight: 700;
        margin-top: 6px;
    }
    table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 15px;
    }
    th {
        background-color: #4B2F1F;
        color: #F4F0E4;
        padding: 12px;
        text-align: left;
        font-weight: 600;
    }
    td {
        padding: 10px 12px;
        border-bottom: 1px solid #E0E0E0;
    }
    .search-bar {
        margin: 25px 0;
        display: flex;
        flex-wrap: wrap;
        gap: 12px;
    }
    .search-bar input {
        flex: 1 1 280px;
        padding: 12px 16px;
        border: 1px solid #B08968;
        border-radius: 10px;
        font-size: 16px;
        background-color: #FDF8F3;
    }
    .province-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 18px;
    }
    .province-card {
        border-radius: 14px;
        background-color: #FFF;
        border: 1px solid rgba(75,47,31,0.08);
        padding: 18px;
        box-shadow: 0 8px 16px rgba(0,0,0,0.05);
        display: flex;
        flex-direction: column;
        gap: 12px;
    }
    .province-card h3 {
        margin: 0;
        color: #4B2F1F;
        font-size: 18px;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .province-total {
        font-size: 13px;
        color: #6B4F2F;
    }
    .product-entry {
        border-radius: 10px;
        padding: 10px 12px;
        background-color: #FDFBF7;
        border: 1px solid rgba(75,47,31,0.08);
        display: flex;
        flex-direction: column;
    }
    .product-entry.low {
        border-color: #C62828;
        background-color: #FFF6F4;
    }
    .product-name {
        font-weight: 600;
        color: #4B2F1F;
        margin-bottom: 4px;
    }
    .badge {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        font-size: 12px;
        color: #C62828;
    }
    .low-stock-panel {
        background-color: #FFF1F0;
        border: 1px solid #F8BBD0;
        border-radius: 12px;
        padding: 20px;
        margin-top: 25px;
    }
    .low-stock-item {
        display: flex;
        justify-content: space-between;
        font-size: 14px;
        color: #4B2F1F;
        border-bottom: 1px dashed rgba(75,47,31,0.15);
        padding: 6px 0;
    }
</style>

<div class="stock-container">
    <div class="stock-content">
        <div style="display:flex; justify-content:space-between; align-items:center; flex-wrap:wrap; gap:10px;">
            <h2 style="color:#4B2F1F; font-size:28px; margin:0;">Vue d'ensemble des Stocks</h2>
            <button onclick="location.reload();" class="refresh-btn" style="background-color:#4B2F1F; color:#F4F0E4; border:none; padding:12px 20px; border-radius:10px; font-weight:600; cursor:pointer;">🔄 Actualiser</button>
        </div>

<div class="metric-grid" style="margin-top:20px;">
    <div class="metric-card">
        <div class="metric-label">Stock Principal</div>
        <div class="metric-value"><?php echo number_format($total_main_stock); ?> <small>Caisses</small></div>
        <div style="font-size:12px; opacity:0.85;"><?php echo number_format($total_main_value, 0, ',', '.'); ?> BIF</div>
    </div>
    <div class="metric-card alt">
        <div class="metric-label">Stock Provincial</div>
        <div class="metric-value"><?php echo number_format($total_province_crates); ?> <small>Caisses</small></div>
    </div>
        <div class="metric-card">
            <div class="metric-label">Valeur Stock Dépôts</div>
        <div class="metric-value"><?php echo number_format($total_province_value, 0, ',', '.'); ?> <small>BIF</small></div>
        <div style="font-size:12px; opacity:0.85;">Tous produits en provinces</div>
    </div>
        <div class="metric-card danger">
            <div class="metric-label">Alertes stock faible</div>
            <div class="metric-value"><?php echo count($low_stock_items); ?></div>
        </div>
        </div>

        <div class="section-card">
            <div class="section-title">
                <h3>Stock Régional non transféré aux dépôts</h3>
            </div>
            <?php if (empty($region_stock_summary)): ?>
                <p style="color:#4B2F1F;">Aucun stock régional non transféré pour l'instant.</p>
            <?php else: ?>
                <?php foreach ($region_stock_summary as $region): ?>
                    <div style="margin-bottom: 18px; padding-bottom: 14px; border-bottom:1px solid #E0E0E0;">
                        <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:10px;">
                            <strong style="font-size:16px; color:#4B2F1F;"><?php echo htmlspecialchars($region['region_name'], ENT_QUOTES, 'UTF-8'); ?></strong>
                            <span style="color:#4B2F1F;"><?php echo number_format($region['total_crates'], 0, ',', '.'); ?> caisses — <?php echo number_format($region['total_value'], 0, ',', '.'); ?> BIF</span>
                        </div>
                        <div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap:10px;">
                            <?php foreach ($region['products'] as $product): ?>
                                <div style="padding:10px; border-radius:8px; border:1px solid #F0E4D7; background:#FDFBF7;">
                                    <div style="font-weight:600; color:#4B2F1F;"><?php echo htmlspecialchars($product['product_name'], ENT_QUOTES, 'UTF-8'); ?></div>
                                    <div style="font-size:14px; color:#6B5B52;">
                                        <?php echo number_format($product['quantity'], 0, ',', '.'); ?> caisses<br>
                                        <?php echo number_format($product['value'], 0, ',', '.'); ?> BIF
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <h3 class="section-title">Transferts de Stock</h3>
        <div class="metric-grid" style="margin-top:10px;">
            <?php foreach ($transfer_status_labels as $statusCode => $label): ?>
                <?php $status = $transfer_status_summary[$statusCode] ?? ['label' => $label, 'count' => 0, 'quantity' => 0]; ?>
                <?php
                    $cardClass = 'metric-card';
                    if ($statusCode === 'pending') {
                        $cardClass = 'metric-card alt';
                    } elseif ($statusCode === 'rejected') {
                        $cardClass = 'metric-card danger';
                    }
                ?>
                <div class="<?php echo $cardClass; ?>">
                    <div class="metric-label"><?php echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8'); ?></div>
                    <div class="metric-value"><?php echo number_format($status['count'], 0, ',', '.'); ?></div>
                    <div style="font-size:12px; opacity:0.85;"><?php echo number_format($status['quantity'], 0, ',', '.'); ?> caisses</div>
                </div>
            <?php endforeach; ?>
        </div>

        <h3 class="section-title">Transferts par Dépôt</h3>
        <div class="province-card" style="padding:10px 0;">
            <table>
                <thead>
                    <tr>
                        <th>Dépôt</th>
                        <?php foreach ($transfer_status_labels as $label): ?>
                            <th style="text-align:center;"><?php echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8'); ?></th>
                        <?php endforeach; ?>
                        <th style="text-align:center;">Total (caisses)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($province_transfer_summary as $provinceSummary): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($provinceSummary['province_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <?php foreach ($transfer_status_labels as $statusCode => $label): ?>
                                <?php $status = $provinceSummary['statuses'][$statusCode] ?? ['count' => 0, 'quantity' => 0]; ?>
                                <td style="text-align:center;">
                                    <?php echo number_format($status['count'], 0, ',', '.'); ?> /
                                    <?php echo number_format($status['quantity'], 0, ',', '.'); ?>
                                </td>
                            <?php endforeach; ?>
                            <td style="text-align:center;"><?php echo number_format($provinceSummary['total_quantity'], 0, ',', '.'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <h3 class="section-title">Stock Principal (Central)</h3>
        <div class="province-card" style="padding:0;">
            <table>
                <thead>
                    <tr>
                        <th>Produit</th>
                        <th>Caisses</th>
                        <th>Valeur Totale</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($main_stock_products)): ?>
                        <tr>
                            <td colspan="3" style="text-align:center; color:#6B4F2F; padding:20px;">Aucun stock principal enregistré.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($main_stock_products as $row): ?>
                            <?php
                                $crates = (int)($row['total_quantity'] ?? 0);
                                $price = (float)($row['price_per_crate'] ?? 0);
                                $value = $crates * $price;
                            ?>
                            <tr>
                                <td><?php echo htmlspecialchars($row['name']); ?></td>
                                <td><?php echo number_format($crates, 0, ',', '.'); ?></td>
                                <td><?php echo number_format($value, 0, ',', '.'); ?> BIF</td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
                <tfoot>
                    <tr style="background-color:#F4A26120; font-weight:600;">
                        <td colspan="2">Valeur Totale Stock Principal</td>
                        <td><?php echo number_format($total_main_value, 0, ',', '.'); ?> BIF</td>
                    </tr>
                </tfoot>
            </table>
        </div>

<div class="search-bar">
    <div style="flex:1 1 200px;">
        <select id="provinceFilter" onchange="filterProvinces()" style="width:100%; padding:12px 16px; border:1px solid #B08968; border-radius:10px; background-color:#FDF8F3; font-size:16px;">
            <option value="">Tous les dépôts</option>
            <?php foreach ($provinceOptions as $option): ?>
                <option value="<?php echo htmlspecialchars($option['slug']); ?>"><?php echo htmlspecialchars($option['name']); ?></option>
            <?php endforeach; ?>
        </select>
    </div>
    <input type="text" id="provinceSearch" placeholder="Rechercher un produit..." oninput="filterProvinces()">
</div>

<h3 class="section-title">Stock par Dépôt (Détail)</h3>
<p style="color:#4B2F1F; font-weight:600; margin-top:-10px;">Valeur totale des stocks provinciaux : <?php echo number_format($total_province_value, 0, ',', '.'); ?> BIF</p>
<?php if (empty($stock_by_province)): ?>
            <p style="color:#4B2F1F;">Aucun stock provincial enregistré.</p>
        <?php else: ?>
            <div class="province-grid" id="provinceGrid">
                <?php foreach ($stock_by_province as $province): ?>
                    <div class="province-card" data-province="<?php echo htmlspecialchars(strtolower($province['province_name'])); ?>">
                    <h3>
                        <?php echo htmlspecialchars($province['province_name']); ?>
                        <span class="province-total"><?php echo number_format($province['total_crates'], 0); ?> caisses</span>
                    </h3>
                    <?php if (empty($province['products'])): ?>
                        <p style="color:#6B4F2F; font-style:italic; margin:0;">Aucun stock enregistré pour cette province.</p>
                    <?php else: ?>
                        <?php foreach ($province['products'] as $product): ?>
                            <?php $isLow = $product['crates'] > 0 && $product['crates'] <= $low_stock_threshold_crates; ?>
                            <div class="product-entry <?php echo $isLow ? 'low' : ''; ?>"
                                 data-name="<?php echo htmlspecialchars(strtolower($province['province_name'] . ' ' . $product['product_name'])); ?>">
                                <div class="product-name">
                                    <?php echo htmlspecialchars($product['product_name']); ?>
                                    <?php if ($isLow): ?>
                                        <span class="badge">⚠️ Stock Faible</span>
                                    <?php endif; ?>
                                </div>
                                <span><?php echo number_format($product['crates'], 0); ?> caisses</span>
                                <span style="font-size:12px; color:#4B2F1F; font-weight:600;"><?php echo number_format($product['total_value'], 0, ',', '.'); ?> BIF</span>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if (!empty($low_stock_items)): ?>
            <div class="low-stock-panel">
                <h3>Alertes Stock Faible (seuils produits)</h3>
                <div class="low-stock-list">
                    <?php foreach ($low_stock_items as $item): ?>
                        <div class="low-stock-item">
                                <?php
                                    $thresholdLabel = '';
                                    if (!empty($item['threshold_crates'])) {
                                        $thresholdLabel = 'Seuil: ≤ ' . number_format($item['threshold_crates'], 0, ',', '.') . ' caisses';
                                    } elseif (!empty($item['threshold_units'])) {
                                        $thresholdLabel = 'Seuil: ≤ ' . number_format($item['threshold_units'], 0, ',', '.') . ' unités';
                                    }
                                ?>
                                <span><strong><?php echo htmlspecialchars($item['province']); ?></strong> — <?php echo htmlspecialchars($item['product']); ?></span>
                                <span><?php echo number_format($item['crates'], 0, ',', '.'); ?> caisses — <?php echo number_format($item['value'], 0, ',', '.'); ?> BIF <?php echo $thresholdLabel ? '(' . $thresholdLabel . ')' : ''; ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>
<script>
function filterProvinces() {
    const select = document.getElementById('provinceFilter');
    const searchInput = document.getElementById('provinceSearch');
    const selectedProvince = (select.value || '').toLowerCase();
    const query = (searchInput.value || '').toLowerCase();
    document.querySelectorAll('#provinceGrid .province-card').forEach(card => {
        const matchesProvince = !selectedProvince || card.dataset.province === selectedProvince;
        let visibleEntries = 0;
        const entries = card.querySelectorAll('.product-entry');
        entries.forEach(entry => {
            const matchesSearch = entry.dataset.name.includes(query);
            entry.style.display = matchesSearch || query === '' ? 'flex' : 'none';
            if (entry.style.display === 'flex') {
                visibleEntries += 1;
            }
        });
        if (entries.length === 0) {
        card.style.display = matchesProvince ? 'flex' : 'none';
        } else {
            card.style.display = matchesProvince && (visibleEntries > 0 || query === '') ? 'flex' : 'none';
        }
    });
}
</script>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
