<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();
$roleName = mbh_normalize_role_name($_SESSION['role_name'] ?? '');
if ($roleName !== 'Admin') {
    header('Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php');
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    die("<p style='color:#FF0000;text-align:center;padding:20px;'>Connexion BD impossible.</p>");
}

if (!function_exists('mbh_pick_existing_column')) {
    function mbh_pick_existing_column(mysqli $conn, string $table, array $candidates): ?string
    {
        $tableSafe = $conn->real_escape_string($table);
        foreach ($candidates as $column) {
            $columnSafe = $conn->real_escape_string($column);
            $result = $conn->query("SHOW COLUMNS FROM `{$tableSafe}` LIKE '{$columnSafe}'");
            if ($result && $result->num_rows > 0) {
                $result->free();
                return $column;
            }
            if ($result instanceof mysqli_result) {
                $result->free();
            }
        }
        return null;
    }
}

if (!function_exists('mbh_fetch_company_info')) {
    function mbh_fetch_company_info(mysqli $conn): array
    {
        $company = null;
        $columnMap = [
            'name' => null,
            'email' => null,
            'phone' => null,
            'address' => null,
            'tax_number' => null,
        ];
        $tableExists = false;
        $tableResult = $conn->query("SHOW TABLES LIKE 'company_info'");
        if ($tableResult && $tableResult->num_rows > 0) {
            $tableExists = true;
            $columnMap = [
                'name' => mbh_pick_existing_column($conn, 'company_info', ['name', 'company_name']),
                'email' => mbh_pick_existing_column($conn, 'company_info', ['email', 'company_email']),
                'phone' => mbh_pick_existing_column($conn, 'company_info', ['phone', 'company_phone']),
                'address' => mbh_pick_existing_column($conn, 'company_info', ['address', 'company_address']),
                'tax_number' => mbh_pick_existing_column($conn, 'company_info', ['tax_number', 'company_tax_number', 'tin', 'company_tin'])
            ];
            $selectParts = [];
            foreach ($columnMap as $alias => $column) {
                $selectParts[] = $column ? "{$column} AS {$alias}" : "NULL AS {$alias}";
            }
            $query = $conn->query("SELECT id, " . implode(', ', $selectParts) . " FROM company_info LIMIT 1");
            if ($query instanceof mysqli_result) {
                $company = $query->fetch_assoc();
                $query->free();
            }
        }
        if ($tableResult instanceof mysqli_result) {
            $tableResult->free();
        }
        return [$company, $columnMap, $tableExists];
    }
}

$messages = [];
$errors = [];

$roles = [];
$roleNameColumn = mbh_pick_existing_column($conn, 'role', ['name', 'role_name']) ?? 'name';
$roleQuery = $conn->query("SELECT id, {$roleNameColumn} AS name FROM role ORDER BY {$roleNameColumn}");
if ($roleQuery instanceof mysqli_result) {
    $roles = $roleQuery->fetch_all(MYSQLI_ASSOC);
    $roleQuery->free();
}

$regions = [];
$regionNameColumn = mbh_pick_existing_column($conn, 'region', ['name', 'region_name']) ?? 'name';
$regionResult = $conn->query("SELECT id, {$regionNameColumn} AS name FROM region ORDER BY {$regionNameColumn}");
if ($regionResult instanceof mysqli_result) {
    $regions = $regionResult->fetch_all(MYSQLI_ASSOC);
    $regionResult->free();
}

$provinces = [];
$provinceNameColumn = mbh_pick_existing_column($conn, 'province', ['name', 'province_name']) ?? 'name';
$provinceResult = $conn->query("SELECT id, {$provinceNameColumn} AS name, region_id FROM province ORDER BY {$provinceNameColumn}");
if ($provinceResult instanceof mysqli_result) {
    $provinces = $provinceResult->fetch_all(MYSQLI_ASSOC);
    $provinceResult->free();
}

[$company, $companyColumnMap, $companyTableExists] = mbh_fetch_company_info($conn);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($roleName !== 'Admin') {
        $errors[] = "Seul l'administrateur principal peut exécuter des actions sur cette page.";
    }

    if ($action === 'create_user' && empty($errors)) {
        $full_name = trim($_POST['full_name'] ?? '');
        $username = trim($_POST['username'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $role_id = (int)($_POST['role_id'] ?? 0);
        $province_id = $_POST['province_id'] !== '' ? (int)$_POST['province_id'] : null;
        $region_id = $_POST['region_id'] !== '' ? (int)$_POST['region_id'] : null;

        if (!$full_name || !$username || !$email || !$password || !$role_id) {
            $errors[] = "Tous les champs sont requis pour créer un utilisateur.";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = "Adresse email invalide.";
        } else {
            $stmt = $conn->prepare("SELECT {$roleNameColumn} AS name FROM role WHERE id = ?");
            $stmt->bind_param("i", $role_id);
            $stmt->execute();
            $roleRow = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            if (!$roleRow) {
                $errors[] = "Rôle sélectionné invalide.";
            } else {
                $role_name = $roleRow['name'];
                $normalizedRole = mbh_normalize_role_name($role_name);
                $autoDepartments = mbh_get_role_departments($normalizedRole);
                $requiresDepot = mbh_role_requires_depot($role_name);
                $requiresRegion = mbh_role_requires_region($role_name);

                if ($requiresDepot && !$province_id) {
                    $errors[] = "Ce rôle nécessite un dépôt. Veuillez en créer un.";
                }
                if ($requiresRegion && !$region_id) {
                    $errors[] = "Ce rôle nécessite une région assignée.";
                }
                if (!$requiresRegion) {
                    $region_id = null;
                }
                if (!$requiresDepot) {
                    $province_id = null;
                }

                if (empty($errors)) {
                    $stmt = $conn->prepare("SELECT id FROM user WHERE username = ? OR email = ?");
                    $stmt->bind_param("ss", $username, $email);
                    $stmt->execute();
                    if ($stmt->get_result()->num_rows > 0) {
                        $errors[] = "Nom d'utilisateur ou email déjà utilisé.";
                    }
                    $stmt->close();
                }

                if (empty($errors)) {
                    $hashed = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $conn->prepare("
                        INSERT INTO user (username, full_name, email, position, role_id, province_id, region_id, address, password, is_active)
                        VALUES (?, ?, ?, ?, ?, NULLIF(?, -1), NULLIF(?, -1), ?, ?, 1)
                    ");
                    $provinceParam = $province_id ?? -1;
                    $regionParam = $region_id ?? -1;
                    $position = $role_name;
                    $address = trim($_POST['address'] ?? '');
                    $stmt->bind_param("ssssiiiss", $username, $full_name, $email, $position, $role_id, $provinceParam, $regionParam, $address, $hashed);
                    if ($stmt->execute()) {
                        $newUserId = $stmt->insert_id;
                        mbh_sync_user_departments($conn, $newUserId, $autoDepartments);
                        log_action($_SESSION['user_id'], 'create_user_setup', "Utilisateur $username créé via configuration initiale");
                        $messages[] = "Utilisateur \"$full_name\" créé avec succès.";
                    } else {
                        $errors[] = "Échec de la création de l'utilisateur.";
                    }
                    $stmt->close();
                }
            }
        }
    } elseif ($action === 'create_depot' && empty($errors)) {
        $depot_name = trim($_POST['depot_name'] ?? '');
        $region_id = (int)($_POST['depot_region_id'] ?? 0);
        if (!$depot_name || !$region_id) {
            $errors[] = "Le nom du dépôt et la région sont requis.";
        } else {
            $stmt = $conn->prepare("INSERT INTO province (name, region_id) VALUES (?, ?)");
            $stmt->bind_param("si", $depot_name, $region_id);
            if ($stmt->execute()) {
                $messages[] = "Dépôt \"$depot_name\" ajouté.";
            } else {
                $errors[] = "Impossible d'ajouter le dépôt.";
            }
            $stmt->close();
        }
    } elseif ($action === 'update_company' && empty($errors)) {
        if (!$companyTableExists) {
            $errors[] = "La table company_info est introuvable.";
        } else {
            $company_name = trim($_POST['company_name'] ?? '');
            $company_email = trim($_POST['company_email'] ?? '');
            $company_phone = trim($_POST['company_phone'] ?? '');
            $company_address = trim($_POST['company_address'] ?? '');
            $company_tin = trim($_POST['company_tin'] ?? '');
            if (!$company_name) {
                $errors[] = "Le nom de l'entreprise est requis.";
            } else {
                $fieldValues = [
                    'name' => $company_name,
                    'email' => $company_email,
                    'phone' => $company_phone,
                    'address' => $company_address,
                    'tax_number' => $company_tin,
                ];
                $validColumns = [];
                foreach ($fieldValues as $field => $value) {
                    $columnName = $companyColumnMap[$field] ?? null;
                    if ($columnName) {
                        $validColumns[] = ['column' => $columnName, 'value' => $value, 'type' => 's'];
                    }
                }
                $updatedByColumn = mbh_pick_existing_column($conn, 'company_info', ['updated_by', 'updatedBy']);
                $currentUserId = (int)($_SESSION['user_id'] ?? 0);
                if ($updatedByColumn) {
                    if ($currentUserId <= 0) {
                        $errors[] = "Impossible de déterminer l'utilisateur connecté pour la mise à jour.";
                    } else {
                        $validColumns[] = ['column' => $updatedByColumn, 'value' => $currentUserId, 'type' => 'i'];
                    }
                }
                if (empty($validColumns)) {
                    $errors[] = "Impossible de déterminer les colonnes de la table company_info.";
                } elseif (!empty($errors)) {
                    // Skip processing if a validation error (like missing user ID) was added.
                } else {
                    if ($company) {
                        $setClauses = [];
                        $types = '';
                        $params = [];
                        foreach ($validColumns as $entry) {
                            $setClauses[] = "{$entry['column']} = ?";
                            $types .= $entry['type'] ?? 's';
                            $params[] = $entry['value'];
                        }
                        $types .= 'i';
                        $params[] = $company['id'];
                        $stmt = $conn->prepare("UPDATE company_info SET " . implode(', ', $setClauses) . " WHERE id = ?");
                        $stmt->bind_param($types, ...$params);
                    } else {
                        $insertColumns = [];
                        $placeholders = [];
                        $types = '';
                        $params = [];
                        foreach ($validColumns as $entry) {
                            $insertColumns[] = $entry['column'];
                            $placeholders[] = '?';
                            $types .= $entry['type'] ?? 's';
                            $params[] = $entry['value'];
                        }
                        $stmt = $conn->prepare("INSERT INTO company_info (" . implode(', ', $insertColumns) . ") VALUES (" . implode(', ', $placeholders) . ")");
                        $stmt->bind_param($types, ...$params);
                    }
                    if ($stmt->execute()) {
                        $messages[] = "Informations de l'entreprise mises à jour.";
                    } else {
                        $errors[] = "Impossible de sauvegarder les informations.";
                    }
                    $stmt->close();
                }
            }
        }
    }
}

// Refresh dependent lists
$provinceResult = $conn->query("SELECT id, {$provinceNameColumn} AS name, region_id FROM province ORDER BY {$provinceNameColumn}");
$provinces = $provinceResult ? $provinceResult->fetch_all(MYSQLI_ASSOC) : [];
if ($provinceResult instanceof mysqli_result) {
    $provinceResult->free();
}
[$company, $companyColumnMap, $companyTableExists] = mbh_fetch_company_info($conn);

include __DIR__ . '/../../includes/header.php';
?>
<div class="dashboard-container" style="padding: 20px; background-color: #F4F0E4; min-height: 80vh;">
    <div class="dashboard-content" style="background-color: #FFFFFF; padding: 25px; border-radius: 12px; box-shadow: 0 10px 25px rgba(75,47,31,0.1);">
        <h1 style="color:#4B2F1F; font-size:26px; margin-bottom:18px;">Configuration Initiale</h1>
        <?php foreach ($messages as $msg): ?>
            <div style="background:#E6F4EA;color:#1B5E20;padding:12px;border-radius:8px;border:1px solid #A5D6A7;margin-bottom:12px;"><?php echo htmlspecialchars($msg, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endforeach; ?>
        <?php foreach ($errors as $err): ?>
            <div style="background:#FFEBEE;color:#C62828;padding:12px;border-radius:8px;border:1px solid #EF9A9A;margin-bottom:12px;"><?php echo htmlspecialchars($err, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endforeach; ?>

        <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(280px,1fr));gap:20px;">
            <section style="border:1px solid #E0C7AE;border-radius:10px;padding:18px;background:#FFFBF7;">
                <h2 style="color:#4B2F1F;font-size:18px;margin-bottom:12px;">Créer un Utilisateur</h2>
                <form method="POST" style="display:flex;flex-direction:column;gap:10px;">
                    <input type="hidden" name="action" value="create_user">
                    <input type="text" name="full_name" placeholder="Nom complet" required style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;">
                    <input type="text" name="username" placeholder="Nom d'utilisateur" required style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;">
                    <input type="email" name="email" placeholder="Email" required style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;">
                    <input type="password" name="password" placeholder="Mot de passe" required style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;">
                    <select name="role_id" required style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;">
                        <option value="">Sélectionner un rôle</option>
                        <?php foreach ($roles as $role): ?>
                            <option value="<?php echo $role['id']; ?>"><?php echo htmlspecialchars($role['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <select name="region_id" style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;">
                        <option value="">Région (si requis)</option>
                        <?php foreach ($regions as $region): ?>
                            <option value="<?php echo $region['id']; ?>"><?php echo htmlspecialchars($region['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <select name="province_id" style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;">
                        <option value="">Dépôt (si requis)</option>
                        <?php foreach ($provinces as $province): ?>
                            <option value="<?php echo $province['id']; ?>"><?php echo htmlspecialchars($province['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <textarea name="address" placeholder="Adresse (optionnel)" style="padding:10px;border:1px solid #D7CCC8;border-radius:6px;min-height:60px;"></textarea>
                    <button type="submit" style="background:#4B2F1F;color:#FFF;border:none;padding:10px;border-radius:6px;font-weight:600;">Créer l'utilisateur</button>
                </form>
            </section>

            <section style="border:1px solid #E0C7AE;border-radius:10px;padding:18px;background:#F7FFF8;">
                <h2 style="color:#4B2F1F;font-size:18px;margin-bottom:12px;">Ajouter un Dépôt</h2>
                <form method="POST" style="display:flex;flex-direction:column;gap:10px;">
                    <input type="hidden" name="action" value="create_depot">
                    <input type="text" name="depot_name" placeholder="Nom du dépôt" required style="padding:10px;border:1px solid #C8E6C9;border-radius:6px;">
                    <select name="depot_region_id" required style="padding:10px;border:1px solid #C8E6C9;border-radius:6px;">
                        <option value="">Sélectionner une région</option>
                        <?php foreach ($regions as $region): ?>
                            <option value="<?php echo $region['id']; ?>"><?php echo htmlspecialchars($region['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit" style="background:#2E7D32;color:#FFF;border:none;padding:10px;border-radius:6px;font-weight:600;">Ajouter le dépôt</button>
                </form>
                <p style="margin-top:12px;color:#4B2F1F;font-size:13px;">
                    Dépôts existants :
                    <?php if (empty($provinces)): ?>
                        Aucun dépôt enregistré.
                    <?php else: ?>
                        <?php foreach ($provinces as $province): ?>
                            <span style="display:block;"><?php echo htmlspecialchars($province['name'], ENT_QUOTES, 'UTF-8'); ?></span>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </p>
            </section>

            <section style="border:1px solid #E0C7AE;border-radius:10px;padding:18px;background:#F7F3FF;">
                <h2 style="color:#4B2F1F;font-size:18px;margin-bottom:12px;">Informations Société</h2>
                <form method="POST" style="display:flex;flex-direction:column;gap:10px;">
                    <input type="hidden" name="action" value="update_company">
                    <input type="text" name="company_name" placeholder="Nom" value="<?php echo htmlspecialchars($company['name'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" required style="padding:10px;border:1px solid #D1C4E9;border-radius:6px;">
                    <input type="email" name="company_email" placeholder="Email" value="<?php echo htmlspecialchars($company['email'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" style="padding:10px;border:1px solid #D1C4E9;border-radius:6px;">
                    <input type="text" name="company_phone" placeholder="Téléphone" value="<?php echo htmlspecialchars($company['phone'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" style="padding:10px;border:1px solid #D1C4E9;border-radius:6px;">
                    <input type="text" name="company_tin" placeholder="NIF" value="<?php echo htmlspecialchars($company['tax_number'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" style="padding:10px;border:1px solid #D1C4E9;border-radius:6px;">
                    <textarea name="company_address" placeholder="Adresse" style="padding:10px;border:1px solid #D1C4E9;border-radius:6px;min-height:60px;"><?php echo htmlspecialchars($company['address'] ?? '', ENT_QUOTES, 'UTF-8'); ?></textarea>
                    <button type="submit" style="background:#4527A0;color:#FFF;border:none;padding:10px;border-radius:6px;font-weight:600;">Sauvegarder</button>
                </form>
            </section>
        </div>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
