<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in manage_company_info.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Create audit_log table if it doesn't exist
$createAuditTable = "CREATE TABLE IF NOT EXISTS audit_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    action VARCHAR(50) NOT NULL,
    table_name VARCHAR(50) NOT NULL,
    record_id INT DEFAULT NULL,
    timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
    details TEXT
)";
if (!$conn->query($createAuditTable)) {
    error_log("Failed to create audit_log table: " . $conn->error);
}

// Restrict to DG (role_id 2), DGA (role_id 3), and Admin (role_id 1)
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = mbh_normalize_role_name($stmt->get_result()->fetch_assoc()['name'] ?? '');
$stmt->close();
if (!in_array($role, ['DG', 'DGA', 'Admin'])) {
    error_log("Unauthorized access to manage_company_info.php by user {$_SESSION['full_name']} (Role: $role)");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}

// Fetch the single company info record (shared by all depots)
// We'll use the first record or create one if none exists
$company_info = [];
$stmt = $conn->prepare("SELECT * FROM company_info ORDER BY id ASC LIMIT 1");
$stmt->execute();
$company_info = $stmt->get_result()->fetch_assoc() ?: [];
$stmt->close();

// Fetch total depots count for stats
$depotCount = 0;
$depotStmt = $conn->prepare("SELECT COUNT(*) as cnt FROM province");
$depotStmt->execute();
$depotCount = (int)($depotStmt->get_result()->fetch_assoc()['cnt'] ?? 0);
$depotStmt->close();

$errors = [];
$success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_company'])) {
    $company_name = trim($_POST['company_name'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $phone1 = trim($_POST['phone1'] ?? '');
    $phone2 = trim($_POST['phone2'] ?? '');
    $nif = trim($_POST['nif'] ?? '');
    $rc = trim($_POST['rc'] ?? '');
    $website = trim($_POST['website'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $logo_path = isset($company_info['logo_path']) ? $company_info['logo_path'] : null;

    if (empty($company_name)) $errors[] = "Le nom de l'entreprise est requis.";
    if (!filter_var($email, FILTER_VALIDATE_EMAIL) && !empty($email)) $errors[] = "Format d'email invalide.";

    // Handle logo upload
    if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = '/opt/lampp/htdocs/masunzu_bar_hotel/assets/uploads/logos/';
        if (!is_dir($upload_dir)) {
            if (!mkdir($upload_dir, 0755, true)) {
                $errors[] = "Échec de la création du répertoire de téléchargement. Vérifiez les permissions.";
            }
        }
        $logo_name = uniqid() . '_' . basename($_FILES['logo']['name']);
        $target_file = $upload_dir . $logo_name;
        $image_file_type = strtolower(pathinfo($target_file, PATHINFO_EXTENSION));
        $allowed_types = ['jpg', 'jpeg', 'png', 'gif'];
        if (!in_array($image_file_type, $allowed_types)) {
            $errors[] = "Seuls les fichiers JPG, JPEG, PNG et GIF sont autorisés.";
        } elseif ($_FILES['logo']['size'] > 5000000) { // 5MB limit
            $errors[] = "La taille du fichier doit être inférieure à 5 Mo.";
        } else {
            if (move_uploaded_file($_FILES['logo']['tmp_name'], $target_file)) {
                $logo_path = 'assets/uploads/logos/' . $logo_name;
                // Delete old logo if exists
                if (isset($company_info['logo_path']) && $company_info['logo_path'] && file_exists(__DIR__ . '/../../' . $company_info['logo_path'])) {
                    unlink(__DIR__ . '/../../' . $company_info['logo_path']);
                }
            } else {
                $errors[] = "Échec du téléchargement du logo. Vérifiez les permissions du répertoire.";
            }
        }
    }

    if (empty($errors)) {
        $updated_by = $_SESSION['user_id'];
        if (empty($company_info)) {
            // Insert new record (shared by all depots)
            $stmt = $conn->prepare("INSERT INTO company_info (company_name, address, phone1, phone2, nif, rc, website, email, logo_path, updated_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("sssssssssi", $company_name, $address, $phone1, $phone2, $nif, $rc, $website, $email, $logo_path, $updated_by);
        } else {
            // Update existing record (shared by all depots)
            $stmt = $conn->prepare("UPDATE company_info SET company_name = ?, address = ?, phone1 = ?, phone2 = ?, nif = ?, rc = ?, website = ?, email = ?, logo_path = ?, updated_by = ? WHERE id = ?");
            $stmt->bind_param("sssssssssii", $company_name, $address, $phone1, $phone2, $nif, $rc, $website, $email, $logo_path, $updated_by, $company_info['id']);
        }
        if ($stmt->execute()) {
            $success = "Informations de l'entreprise mises à jour avec succès. Ces informations s'appliquent à tous les dépôts.";
            $stmt->close();
            
            // Log the action
            $action = empty($company_info) ? 'INSERT' : 'UPDATE';
            $record_id = empty($company_info) ? $conn->insert_id : $company_info['id'];
            $temp_record_id = $record_id;
            $table_name = 'company_info';
            $user_id = $_SESSION['user_id'];
            $details = json_encode(['company_name' => $company_name, 'address' => $address, 'phone1' => $phone1, 'phone2' => $phone2, 'nif' => $nif, 'rc' => $rc, 'website' => $website, 'email' => $email, 'logo_path' => $logo_path]);
            
            $nextLogIdStmt = $conn->prepare("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM audit_log");
            $nextLogIdStmt->execute();
            $nextLogIdResult = $nextLogIdStmt->get_result()->fetch_assoc();
            $nextLogIdStmt->close();
            $log_id = $nextLogIdResult['next_id'] ?? 1;
            
            $logStmt = $conn->prepare("INSERT INTO audit_log (id, user_id, action, table_name, record_id, details) VALUES (?, ?, ?, ?, ?, ?)");
            $logStmt->bind_param("iissis", $log_id, $user_id, $action, $table_name, $temp_record_id, $details);
            $logStmt->execute();
            $logStmt->close();
            
            // Refresh company_info after save
            $stmt = $conn->prepare("SELECT * FROM company_info ORDER BY id ASC LIMIT 1");
            $stmt->execute();
            $company_info = $stmt->get_result()->fetch_assoc() ?: [];
            $stmt->close();
        } else {
            $errors[] = "Échec de la mise à jour des informations de l'entreprise: " . $conn->error;
            $stmt->close();
        }
    }
}

include __DIR__ . '/../../includes/header.php';
?>
<style>
    @keyframes fadeIn { from { opacity: 0; transform: translateY(-10px); } to { opacity: 1; transform: translateY(0); } }
    .company-form-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 16px; }
    .company-form-grid .full-width { grid-column: span 3; }
    .company-form-grid .half-width { grid-column: span 2; }
    @media (max-width: 900px) {
        .company-form-grid { grid-template-columns: repeat(2, 1fr); }
        .company-form-grid .full-width { grid-column: span 2; }
        .company-form-grid .half-width { grid-column: span 2; }
    }
    @media (max-width: 600px) {
        .company-form-grid { grid-template-columns: 1fr; }
        .company-form-grid .full-width, .company-form-grid .half-width { grid-column: span 1; }
    }
    .info-input {
        width: 100%; 
        padding: 12px 14px; 
        border-radius: 8px; 
        border: 1px solid #d4c4b0; 
        font-size: 14px; 
        box-sizing: border-box;
        transition: border-color 0.2s, box-shadow 0.2s;
        background: #fff;
    }
    .info-input:focus {
        outline: none;
        border-color: #4B2F1F;
        box-shadow: 0 0 0 3px rgba(75, 47, 31, 0.1);
    }
    .info-label {
        display: block; 
        font-size: 13px; 
        font-weight: 600; 
        color: #4B2F1F; 
        margin-bottom: 8px;
    }
    .stat-card {
        background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%);
        color: #F4F0E4;
        padding: 16px 24px;
        border-radius: 12px;
        text-align: center;
        min-width: 140px;
        box-shadow: 0 4px 12px rgba(75, 47, 31, 0.2);
    }
    .stat-card p:first-child {
        margin: 0;
        font-size: 11px;
        text-transform: uppercase;
        letter-spacing: 0.08em;
        opacity: 0.9;
    }
    .stat-card p:last-child {
        margin: 6px 0 0;
        font-size: 24px;
        font-weight: 700;
    }
</style>
<div style="padding: 24px; background: #F4F0E4; min-height: calc(100vh - 70px);">
    <!-- Header -->
    <div style="display: flex; justify-content: space-between; align-items: flex-start; flex-wrap: wrap; gap: 20px; margin-bottom: 24px;">
        <div>
            <h1 style="margin: 0 0 8px; color: #4B2F1F; font-size: 28px; font-weight: 700;">Informations de l'Entreprise</h1>
            <p style="margin: 0; color: #6B4F2F; font-size: 14px;">Ces informations sont partagées par <strong>tous les dépôts</strong> de l'entreprise.</p>
        </div>
        <div style="display: flex; gap: 16px; align-items: center; flex-wrap: wrap;">
            <div class="stat-card">
                <p>Total Dépôts</p>
                <p><?php echo $depotCount; ?></p>
            </div>
            <div class="stat-card" style="background: linear-gradient(135deg, #2E7D32 0%, #388E3C 100%);">
                <p>Statut</p>
                <p style="font-size: 16px;"><?php echo empty($company_info) ? '❌ Non configuré' : '✓ Configuré'; ?></p>
            </div>
        </div>
    </div>

    <!-- Info Banner -->
    <div style="background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%); color: #1565C0; padding: 16px 20px; border-radius: 10px; margin-bottom: 24px; display: flex; align-items: center; gap: 12px; border-left: 4px solid #1565C0;">
        <span style="font-size: 24px;">ℹ️</span>
        <div>
            <strong style="display: block; margin-bottom: 4px;">Information Centralisée</strong>
            <span style="font-size: 13px;">Les informations ci-dessous (nom, adresse, contacts, logo, etc.) seront utilisées pour tous les documents officiels, factures et rapports de tous les dépôts.</span>
        </div>
    </div>

    <!-- Messages -->
    <?php if ($success): ?>
        <div style="background: #e8f5e9; color: #2E7D32; padding: 16px 20px; border-radius: 10px; margin-bottom: 24px; border-left: 4px solid #2E7D32; animation: fadeIn 0.4s ease-out; display: flex; align-items: center; gap: 12px;">
            <span style="font-size: 20px;">✓</span>
            <span><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></span>
        </div>
    <?php endif; ?>
    <?php if (!empty($errors)): ?>
        <div style="background: #ffebee; color: #C62828; padding: 16px 20px; border-radius: 10px; margin-bottom: 24px; border-left: 4px solid #C62828; animation: fadeIn 0.4s ease-out;">
            <strong style="display: block; margin-bottom: 8px;">⚠️ Erreurs détectées:</strong>
            <ul style="margin: 0; padding-left: 20px;">
                <?php foreach ($errors as $error): ?>
                    <li style="margin-bottom: 4px;"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <!-- Company Info Form -->
    <div style="background: #fff; border-radius: 14px; box-shadow: 0 4px 16px rgba(0,0,0,0.08); overflow: hidden;">
        <!-- Form Header -->
        <div style="padding: 20px 24px; border-bottom: 1px solid #F0F0F0; background: linear-gradient(135deg, #FDFBF7 0%, #F8F4EC 100%); display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 12px;">
            <div style="display: flex; align-items: center; gap: 12px;">
                <?php if (isset($company_info['logo_path']) && $company_info['logo_path']): ?>
                    <img src="/masunzu_bar_hotel/<?php echo htmlspecialchars($company_info['logo_path'], ENT_QUOTES, 'UTF-8'); ?>" alt="Logo" style="width: 50px; height: 50px; object-fit: contain; border-radius: 8px; background: #fff; padding: 4px; border: 1px solid #E8E8E8;">
                <?php else: ?>
                    <div style="width: 50px; height: 50px; background: #F0F0F0; border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 24px;">🏢</div>
                <?php endif; ?>
                <div>
                    <h2 style="margin: 0; color: #4B2F1F; font-size: 18px; font-weight: 600;">
                        <?php echo htmlspecialchars($company_info['company_name'] ?? 'Nouvelle Entreprise', ENT_QUOTES, 'UTF-8'); ?>
                    </h2>
                    <?php if (!empty($company_info['updated_at'])): ?>
                        <p style="margin: 4px 0 0; color: #888; font-size: 12px;">
                            Dernière mise à jour: <?php echo date('d/m/Y à H:i', strtotime($company_info['updated_at'])); ?>
                        </p>
                    <?php endif; ?>
                </div>
            </div>
            <span style="background: <?php echo empty($company_info) ? '#FFF3E0' : '#e8f5e9'; ?>; color: <?php echo empty($company_info) ? '#E65100' : '#2E7D32'; ?>; padding: 6px 14px; border-radius: 20px; font-size: 12px; font-weight: 600;">
                <?php echo empty($company_info) ? '⚠️ Non configuré' : '✓ Configuré'; ?>
            </span>
        </div>
        
        <form method="POST" enctype="multipart/form-data" id="companyForm" style="padding: 24px;">
            <div class="company-form-grid">
                <!-- Company Name -->
                <div class="half-width">
                    <label for="company_name" class="info-label">Nom de l'Entreprise <span style="color: #C62828;">*</span></label>
                    <input type="text" id="company_name" name="company_name" class="info-input" value="<?php echo htmlspecialchars($company_info['company_name'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" required placeholder="Ex: Masunzu Bar & Hotel">
                </div>
                
                <!-- Email -->
                <div>
                    <label for="email" class="info-label">Email</label>
                    <input type="email" id="email" name="email" class="info-input" value="<?php echo htmlspecialchars($company_info['email'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" placeholder="contact@entreprise.com">
                </div>
                
                <!-- Address -->
                <div class="full-width">
                    <label for="address" class="info-label">Adresse Complète</label>
                    <input type="text" id="address" name="address" class="info-input" value="<?php echo htmlspecialchars($company_info['address'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" placeholder="Avenue Principale, Quartier Commercial, Bujumbura">
                </div>
                
                <!-- Phone 1 -->
                <div>
                    <label for="phone1" class="info-label">Téléphone Principal</label>
                    <input type="text" id="phone1" name="phone1" class="info-input" value="<?php echo htmlspecialchars($company_info['phone1'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" placeholder="+257 22 XX XX XX">
                </div>
                
                <!-- Phone 2 -->
                <div>
                    <label for="phone2" class="info-label">Téléphone Secondaire</label>
                    <input type="text" id="phone2" name="phone2" class="info-input" value="<?php echo htmlspecialchars($company_info['phone2'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" placeholder="+257 79 XX XX XX">
                </div>
                
                <!-- Website -->
                <div>
                    <label for="website" class="info-label">Site Web</label>
                    <input type="url" id="website" name="website" class="info-input" value="<?php echo htmlspecialchars($company_info['website'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" placeholder="https://www.exemple.com">
                </div>
                
                <!-- NIF -->
                <div>
                    <label for="nif" class="info-label">NIF (Numéro d'Identification Fiscale)</label>
                    <input type="text" id="nif" name="nif" class="info-input" value="<?php echo htmlspecialchars($company_info['nif'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" placeholder="Ex: 4000123456">
                </div>
                
                <!-- RC -->
                <div class="half-width">
                    <label for="rc" class="info-label">RC (Registre de Commerce)</label>
                    <input type="text" id="rc" name="rc" class="info-input" value="<?php echo htmlspecialchars($company_info['rc'] ?? '', ENT_QUOTES, 'UTF-8'); ?>" placeholder="Ex: RC/BJA/2024/1234">
                </div>
                
                <!-- Logo Upload Section -->
                <div class="full-width" style="margin-top: 8px;">
                    <div style="background: #FDFBF7; border: 2px dashed #d4c4b0; border-radius: 12px; padding: 20px;">
                        <div style="display: flex; align-items: flex-start; gap: 24px; flex-wrap: wrap;">
                            <!-- Current Logo Preview -->
                            <div style="text-align: center;">
                                <label class="info-label" style="margin-bottom: 12px;">Logo Actuel</label>
                                <?php if (isset($company_info['logo_path']) && $company_info['logo_path']): ?>
                                    <div style="background: #fff; padding: 16px; border-radius: 10px; border: 1px solid #E8E8E8; display: inline-block;">
                                        <img src="/masunzu_bar_hotel/<?php echo htmlspecialchars($company_info['logo_path'], ENT_QUOTES, 'UTF-8'); ?>" alt="Logo Actuel" style="max-width: 180px; max-height: 100px; display: block;">
                                    </div>
                                <?php else: ?>
                                    <div style="background: #F0F0F0; padding: 30px 40px; border-radius: 10px; color: #888;">
                                        <span style="font-size: 40px; display: block; margin-bottom: 8px;">🖼️</span>
                                        <span style="font-size: 13px;">Aucun logo</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <!-- Upload New Logo -->
                            <div style="flex: 1; min-width: 250px;">
                                <label for="logo" class="info-label">Télécharger un Nouveau Logo</label>
                                <input type="file" id="logo" name="logo" accept=".jpg,.jpeg,.png,.gif" style="width: 100%; padding: 12px; border-radius: 8px; border: 1px solid #d4c4b0; font-size: 13px; box-sizing: border-box; background: #fff; cursor: pointer;">
                                <p style="margin: 10px 0 0; color: #888; font-size: 12px;">
                                    📎 Formats acceptés: JPG, JPEG, PNG, GIF<br>
                                    📏 Taille maximale: 5 Mo<br>
                                    💡 Recommandé: Image carrée ou horizontale avec fond transparent
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Submit Button -->
            <div style="margin-top: 28px; padding-top: 24px; border-top: 1px solid #F0F0F0; display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 16px;">
                <p style="margin: 0; color: #888; font-size: 13px;">
                    <span style="color: #C62828;">*</span> Champs obligatoires
                </p>
                <button type="submit" name="update_company" style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color: #F4F0E4; padding: 14px 32px; border: none; border-radius: 10px; font-size: 15px; font-weight: 600; cursor: pointer; transition: all 0.3s; box-shadow: 0 4px 12px rgba(75, 47, 31, 0.25);" onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 6px 16px rgba(75, 47, 31, 0.35)'" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 4px 12px rgba(75, 47, 31, 0.25)'">
                    ✓ Enregistrer les Informations
                </button>
            </div>
        </form>
    </div>
    
    <!-- Help Section -->
    <div style="margin-top: 24px; background: #fff; border-radius: 12px; padding: 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.04);">
        <h3 style="margin: 0 0 12px; color: #4B2F1F; font-size: 16px; font-weight: 600;">💡 À propos de cette page</h3>
        <ul style="margin: 0; padding-left: 20px; color: #6B4F2F; font-size: 14px; line-height: 1.8;">
            <li>Les informations configurées ici sont <strong>partagées par tous les dépôts</strong> de l'entreprise.</li>
            <li>Le logo et les coordonnées apparaîtront sur les <strong>factures, reçus et rapports</strong> générés par le système.</li>
            <li>Le NIF et RC sont requis pour les documents fiscaux officiels.</li>
            <li>Seuls les utilisateurs avec les rôles <strong>Admin, DG ou DGA</strong> peuvent modifier ces informations.</li>
        </ul>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
