<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

$allowedRoles = ['Admin', 'DG', 'DGA', 'Board'];
$roleName = $_SESSION['role_name'] ?? '';
if (!in_array($roleName, $allowedRoles, true)) {
    header("Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    include __DIR__ . '/../../includes/header.php';
    echo "<div style='padding: 40px; text-align: center;'><p style='color: #C62828;'>Connexion à la base de données impossible.</p></div>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$errors = [];
$success = '';

$accountColumns = [];
$columnsResult = $conn->query('SHOW COLUMNS FROM bank_account');
if ($columnsResult) {
    while ($col = $columnsResult->fetch_assoc()) {
        $accountColumns[$col['Field']] = $col;
    }
    $columnsResult->close();
}
if (!isset($accountColumns['is_main'])) {
    if (!$conn->query("ALTER TABLE bank_account ADD COLUMN is_main TINYINT(1) NOT NULL DEFAULT 0 AFTER currency")) {
        error_log('Failed to add is_main column to bank_account: ' . $conn->error);
    }
}

$provinces = [];
$provinceStmt = $conn->prepare("SELECT id, name FROM province ORDER BY name");
if ($provinceStmt) {
    $provinceStmt->execute();
    $provinces = $provinceStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $provinceStmt->close();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['create_account'])) {
        $accountNumber = trim($_POST['account_number'] ?? '');
        $currency = $_POST['currency'] ?? '';
        $provinceId = isset($_POST['province_id']) && $_POST['province_id'] !== '' ? (int)$_POST['province_id'] : null;
        $isMain = isset($_POST['is_main']) ? 1 : 0;

        if ($accountNumber === '') {
            $errors[] = 'Le numéro du compte est requis.';
        }
        if (!in_array($currency, ['BIF', 'USD'], true)) {
            $errors[] = 'Devise invalide.';
        }

        $stmtCheck = $conn->prepare("SELECT id FROM bank_account WHERE account_number = ? LIMIT 1");
        if ($stmtCheck) {
            $stmtCheck->bind_param('s', $accountNumber);
            $stmtCheck->execute();
            if ($stmtCheck->get_result()->fetch_assoc()) {
                $errors[] = 'Un compte avec ce numéro existe déjà.';
            }
            $stmtCheck->close();
        }

        if (empty($errors)) {
            if ($isMain && $currency === 'BIF') {
                $conn->query("UPDATE bank_account SET is_main = 0 WHERE currency = 'BIF'");
            }
            $stmtInsert = $conn->prepare("INSERT INTO bank_account (account_number, currency, balance, province_id, is_main, created_at) VALUES (?, ?, 0, ?, ?, NOW())");
            if (!$stmtInsert) {
                $errors[] = 'Impossible de créer le compte: ' . $conn->error;
            } else {
                $stmtInsert->bind_param('ssii', $accountNumber, $currency, $provinceId, $isMain);
                if (!$stmtInsert->execute()) {
                    $errors[] = 'Impossible de créer le compte: ' . $stmtInsert->error;
                } else {
                    $success = 'Compte bancaire créé avec succès.';
                    log_action($_SESSION['user_id'], 'create_bank_account', "Created bank account: $accountNumber ($currency)");
                }
                $stmtInsert->close();
            }
        }
    } elseif (isset($_POST['set_main'])) {
        $accountId = isset($_POST['account_id']) ? (int)$_POST['account_id'] : 0;
        if ($accountId > 0) {
            $stmt = $conn->prepare("SELECT currency, account_number FROM bank_account WHERE id = ? LIMIT 1");
            if ($stmt) {
                $stmt->bind_param('i', $accountId);
                $stmt->execute();
                $record = $stmt->get_result()->fetch_assoc();
                $stmt->close();
                if ($record) {
                    $currency = $record['currency'];
                    if ($currency === 'BIF') {
                        $conn->query("UPDATE bank_account SET is_main = 0 WHERE currency = 'BIF'");
                    }
                    $stmtUpdate = $conn->prepare("UPDATE bank_account SET is_main = 1 WHERE id = ?");
                    if ($stmtUpdate) {
                        $stmtUpdate->bind_param('i', $accountId);
                        $stmtUpdate->execute();
                        $stmtUpdate->close();
                        $success = 'Compte principal mis à jour.';
                        log_action($_SESSION['user_id'], 'set_main_account', "Set main account: {$record['account_number']}");
                    }
                }
            }
        }
    }
}

$accounts = [];
$stmt = $conn->prepare("SELECT ba.*, p.name AS province_name FROM bank_account ba LEFT JOIN province p ON ba.province_id = p.id ORDER BY ba.currency, ba.is_main DESC, ba.account_number");
if ($stmt) {
    $stmt->execute();
    $accounts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Calculate totals
$totalBIF = 0;
$totalUSD = 0;
foreach ($accounts as $account) {
    if ($account['currency'] === 'BIF') {
        $totalBIF += (float)($account['balance'] ?? 0);
    } else {
        $totalUSD += (float)($account['balance'] ?? 0);
    }
}

$conn->close();
include __DIR__ . '/../../includes/header.php';
?>
<div style="padding: 20px; background: #F4F0E4; min-height: 70vh;">
    <!-- Header -->
    <div style="display: flex; justify-content: space-between; align-items: flex-start; flex-wrap: wrap; gap: 16px; margin-bottom: 20px;">
        <div>
            <h1 style="margin: 0 0 8px; color: #4B2F1F; font-size: 28px; font-weight: 700;">Gestion Comptes Bancaires</h1>
            <p style="margin: 0; color: #6B4F2F; font-size: 14px;">
                Créez et gérez les comptes bancaires. Le compte principal est référencé sur le tableau de bord.
            </p>
        </div>
        <div style="display: flex; gap: 12px; flex-wrap: wrap;">
            <div style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color: #F4F0E4; padding: 12px 20px; border-radius: 10px; text-align: center; min-width: 130px;">
                <p style="margin: 0; font-size: 11px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.9;">Total BIF</p>
                <p style="margin: 4px 0 0; font-size: 18px; font-weight: 700;"><?php echo number_format($totalBIF, 0, ',', '.'); ?></p>
            </div>
            <div style="background: linear-gradient(135deg, #1565C0 0%, #1976D2 100%); color: #FFF; padding: 12px 20px; border-radius: 10px; text-align: center; min-width: 130px;">
                <p style="margin: 0; font-size: 11px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.9;">Total USD</p>
                <p style="margin: 4px 0 0; font-size: 18px; font-weight: 700;"><?php echo number_format($totalUSD, 2, ',', '.'); ?></p>
            </div>
            <div style="background: #fff; border: 2px solid #4B2F1F; color: #4B2F1F; padding: 12px 20px; border-radius: 10px; text-align: center; min-width: 100px;">
                <p style="margin: 0; font-size: 11px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.7;">Comptes</p>
                <p style="margin: 4px 0 0; font-size: 18px; font-weight: 700;"><?php echo count($accounts); ?></p>
            </div>
        </div>
    </div>

    <!-- Messages -->
    <?php if (!empty($errors)): ?>
        <div style="background: #ffebee; color: #C62828; padding: 14px 18px; border-radius: 8px; margin-bottom: 20px; border-left: 4px solid #C62828;">
            <ul style="margin: 0; padding-left: 18px;">
                <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php elseif ($success): ?>
        <div style="background: #e8f5e9; color: #2E7D32; padding: 14px 18px; border-radius: 8px; margin-bottom: 20px; border-left: 4px solid #2E7D32;">
            ✓ <?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <!-- Create Account Form -->
    <div style="background: #fff; border-radius: 12px; padding: 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.06); margin-bottom: 20px;">
        <h3 style="margin: 0 0 15px; color: #4B2F1F; font-size: 16px; font-weight: 600;">Créer un nouveau compte</h3>
        <form method="POST" style="display: grid; grid-template-columns: repeat(5, 1fr); gap: 15px; align-items: end;">
            <div>
                <label style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Numéro de compte <span style="color: red;">*</span></label>
                <input type="text" name="account_number" required placeholder="Ex: 12345678901" 
                       style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #B08968; font-size: 14px; box-sizing: border-box;">
            </div>
            <div>
                <label style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Devise <span style="color: red;">*</span></label>
                <select name="currency" required style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #B08968; font-size: 14px; box-sizing: border-box;">
                    <option value="BIF">BIF - Franc Burundais</option>
                    <option value="USD">USD - Dollar Américain</option>
                </select>
            </div>
            <div>
                <label style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Province (optionnelle)</label>
                <select name="province_id" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #B08968; font-size: 14px; box-sizing: border-box;">
                    <option value="">Toutes provinces (Global)</option>
                    <?php foreach ($provinces as $province): ?>
                        <option value="<?php echo (int)$province['id']; ?>"><?php echo htmlspecialchars($province['name'], ENT_QUOTES, 'UTF-8'); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div style="display: flex; align-items: center; padding-bottom: 10px;">
                <label style="display: flex; align-items: center; gap: 8px; font-size: 13px; font-weight: 500; color: #4B2F1F; cursor: pointer;">
                    <input type="checkbox" name="is_main" value="1" style="width: 16px; height: 16px;">
                    Compte principal (BIF)
                </label>
            </div>
            <div>
                <button type="submit" name="create_account" 
                        style="width: 100%; background: #4B2F1F; color: #F4F0E4; padding: 11px; border: none; border-radius: 6px; font-weight: 600; cursor: pointer; font-size: 14px; transition: background 0.2s;"
                        onmouseover="this.style.backgroundColor='#6B4F2F'" 
                        onmouseout="this.style.backgroundColor='#4B2F1F'">
                    + Créer le compte
                </button>
            </div>
        </form>
    </div>

    <!-- Accounts Table -->
    <div style="background: #fff; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.06); overflow: hidden;">
        <div style="padding: 16px 20px; border-bottom: 1px solid #F0F0F0; display: flex; justify-content: space-between; align-items: center; background: #FDFBF7;">
            <span style="font-weight: 600; color: #4B2F1F;">
                <?php echo count($accounts); ?> compte<?php echo count($accounts) === 1 ? '' : 's'; ?> bancaire<?php echo count($accounts) === 1 ? '' : 's'; ?>
            </span>
            <span style="color: #6B4F2F; font-size: 13px;">
                Le compte principal est utilisé pour les rapports du tableau de bord
            </span>
        </div>
        <?php if (empty($accounts)): ?>
            <div style="padding: 60px 20px; text-align: center;">
                <div style="font-size: 64px; margin-bottom: 15px;">🏦</div>
                <p style="color: #4B2F1F; font-size: 18px; font-weight: 600; margin: 0 0 8px;">Aucun compte bancaire</p>
                <p style="color: #6B4F2F; font-size: 14px; margin: 0;">Créez votre premier compte bancaire ci-dessus.</p>
            </div>
        <?php else: ?>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse; font-size: 14px; min-width: 800px;">
                    <thead>
                        <tr style="background: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Numéro de compte</th>
                            <th style="padding: 14px 16px; text-align: center; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Devise</th>
                            <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Province</th>
                            <th style="padding: 14px 16px; text-align: right; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Solde</th>
                            <th style="padding: 14px 16px; text-align: center; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Principal</th>
                            <th style="padding: 14px 16px; text-align: center; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($accounts as $account): 
                            $isMain = (int)$account['is_main'] === 1;
                            $isBIF = $account['currency'] === 'BIF';
                        ?>
                            <tr style="border-bottom: 1px solid #F0F0F0; transition: background-color 0.2s; <?php echo $isMain ? 'background-color: #FFF8E1;' : ''; ?>" 
                                onmouseover="this.style.backgroundColor='<?php echo $isMain ? '#FFF3C4' : '#FDFBF7'; ?>'" 
                                onmouseout="this.style.backgroundColor='<?php echo $isMain ? '#FFF8E1' : 'transparent'; ?>'">
                                <td style="padding: 14px 16px;">
                                    <div style="font-weight: 600; color: #2C1810; font-family: monospace; font-size: 15px;">
                                        <?php echo htmlspecialchars($account['account_number'], ENT_QUOTES, 'UTF-8'); ?>
                                    </div>
                                </td>
                                <td style="padding: 14px 16px; text-align: center;">
                                    <span style="display: inline-block; padding: 4px 12px; border-radius: 4px; font-size: 12px; font-weight: 600; 
                                                 background: <?php echo $isBIF ? '#e8f5e9' : '#e3f2fd'; ?>; 
                                                 color: <?php echo $isBIF ? '#2E7D32' : '#1565C0'; ?>;">
                                        <?php echo htmlspecialchars($account['currency'], ENT_QUOTES, 'UTF-8'); ?>
                                    </span>
                                </td>
                                <td style="padding: 14px 16px; color: #4B2F1F;">
                                    <?php echo htmlspecialchars($account['province_name'] ?? 'Global', ENT_QUOTES, 'UTF-8'); ?>
                                </td>
                                <td style="padding: 14px 16px; text-align: right; font-weight: 700; color: #2C1810; font-size: 15px;">
                                    <?php echo number_format((float)($account['balance'] ?? 0), $isBIF ? 0 : 2, ',', '.'); ?>
                                    <span style="font-size: 12px; color: #6B4F2F; font-weight: 500;"><?php echo $account['currency']; ?></span>
                                </td>
                                <td style="padding: 14px 16px; text-align: center;">
                                    <?php if ($isMain): ?>
                                        <span style="display: inline-flex; align-items: center; gap: 4px; padding: 4px 12px; border-radius: 20px; font-size: 12px; font-weight: 600; background: #FFC107; color: #000;">
                                            ⭐ Principal
                                        </span>
                                    <?php else: ?>
                                        <span style="color: #999;">—</span>
                                    <?php endif; ?>
                                </td>
                                <td style="padding: 14px 16px; text-align: center;">
                                    <?php if (!$isMain && $isBIF): ?>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="account_id" value="<?php echo (int)$account['id']; ?>">
                                            <button type="submit" name="set_main" 
                                                    style="background: #F4A261; color: #000; border: none; padding: 8px 14px; border-radius: 6px; cursor: pointer; font-size: 13px; font-weight: 500; transition: background 0.2s;"
                                                    onmouseover="this.style.backgroundColor='#E8924C'" 
                                                    onmouseout="this.style.backgroundColor='#F4A261'">
                                                Définir principal
                                            </button>
                                        </form>
                                    <?php elseif (!$isBIF): ?>
                                        <span style="color: #999; font-size: 12px;">USD uniquement</span>
                                    <?php else: ?>
                                        <span style="color: #2E7D32; font-size: 12px;">✓ Actif</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
