<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

// Restrict to DGA or admin role
$conn = get_db_connection();
if (!$conn) {
    $error_msg = "Échec de la connexion à la base de données dans approve_invoices.php: " . mysqli_connect_error();
    error_log($error_msg);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>$error_msg</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

mbh_ensure_loan_ledger_entry_origin($conn);

// Function to check and expire unpaid invoices after 3 hours
function check_and_expire_unpaid_invoices($conn) {
    try {
        // Find expired invoices (pending approval for more than 3 hours)
        $expiredStmt = $conn->prepare("
            SELECT ui.id, ui.province_id, ui.created_by,
                   ii.product_id, ii.quantity, p.crate_quantity
            FROM unpaid_invoices ui
            INNER JOIN invoice_items ii ON ui.id = ii.invoice_id
            INNER JOIN product p ON ii.product_id = p.id
            WHERE ui.approval_status = 'pending'
              AND ui.expires_at IS NOT NULL
              AND ui.expires_at < NOW()
            ORDER BY ui.id
        ");
        
        if (!$expiredStmt) {
            error_log("Failed to prepare expired invoices query: " . $conn->error);
            return 0;
        }
        
        $expiredStmt->execute();
        $expired_result = $expiredStmt->get_result();
        
        $expired_invoices = [];
        while ($row = $expired_result->fetch_assoc()) {
            $invoice_id = (int)$row['id'];
            if (!isset($expired_invoices[$invoice_id])) {
                $expired_invoices[$invoice_id] = [
                    'id' => $invoice_id,
                    'province_id' => (int)$row['province_id'],
                    'created_by' => (int)$row['created_by'],
                    'items' => []
                ];
            }
            $expired_invoices[$invoice_id]['items'][] = [
                'product_id' => (int)$row['product_id'],
                'quantity' => (int)$row['quantity'],
                'crate_quantity' => max(1, (int)$row['crate_quantity'])
            ];
        }
        $expiredStmt->close();
        
        if (empty($expired_invoices)) {
            return 0;
        }
        
        $expired_count = 0;
        foreach ($expired_invoices as $invoice) {
            $conn->begin_transaction();
            try {
                error_log("⏰ Expiring invoice #{$invoice['id']}: Province {$invoice['province_id']}, {$invoice['created_by']} items");
                // Return reserved stock to available
                foreach ($invoice['items'] as $item) {
                    $product_id = $item['product_id'];
                    $crate_quantity = max(1, (int)$item['crate_quantity']);
                    $quantity_in_crates = max(1, intval($item['quantity']));
                    
                    error_log("Expiring invoice #{$invoice['id']}: Returning {$quantity_in_crates} crates of product {$product_id} to available stock (crate size: {$crate_quantity})");
                    
                    // Check current stock before expiration
                    $checkStmt = $conn->prepare("
                        SELECT quantity, reserved_quantity, unit_type, COALESCE(quantity, 0) - COALESCE(reserved_quantity, 0) as available
                        FROM province_stock 
                        WHERE product_id = ? AND province_id = ? AND unit_type = 'crates'
                    ");
                    if ($checkStmt) {
                        $checkStmt->bind_param("ii", $product_id, $invoice['province_id']);
                        $checkStmt->execute();
                        $stock_result = $checkStmt->get_result();
                        if ($stock_row = $stock_result->fetch_assoc()) {
                            error_log("📊 Stock before expiration: qty={$stock_row['quantity']}, reserved={$stock_row['reserved_quantity']}, unit={$stock_row['unit_type']}, available={$stock_row['available']}");
                        }
                        $checkStmt->close();
                    }
                    
                    $returnStmt = $conn->prepare("
                        UPDATE province_stock 
                        SET reserved_quantity = GREATEST(0, reserved_quantity - ?)
                        WHERE product_id = ? 
                          AND province_id = ?
                          AND unit_type = 'crates'
                    ");
                    if ($returnStmt) {
                        $returnStmt->bind_param("iii", $quantity_in_crates, $product_id, $invoice['province_id']);
                        if ($returnStmt->execute()) {
                            $affected = $returnStmt->affected_rows;
                            error_log("✅ Expired invoice returned {$quantity_in_crates} crates of product {$product_id} to available stock (affected_rows: $affected)");
                            
                            // Check stock after expiration
                            $checkStmt = $conn->prepare("
                                SELECT quantity, reserved_quantity, COALESCE(quantity, 0) - COALESCE(reserved_quantity, 0) as available
                                FROM province_stock 
                                WHERE product_id = ? AND province_id = ? AND unit_type = 'crates'
                            ");
                            if ($checkStmt) {
                                $checkStmt->bind_param("ii", $product_id, $invoice['province_id']);
                                $checkStmt->execute();
                                $stock_result = $checkStmt->get_result();
                                if ($stock_row = $stock_result->fetch_assoc()) {
                                    error_log("📊 Stock after expiration: qty={$stock_row['quantity']}, reserved={$stock_row['reserved_quantity']}, available={$stock_row['available']}");
                                }
                                $checkStmt->close();
                            }
                        } else {
                            error_log("❌ Failed to return stock on expiration: " . $returnStmt->error);
                        }
                        $returnStmt->close();
                    } else {
                        error_log("❌ Failed to prepare stock return query on expiration: " . $conn->error);
                    }
                }
                
                // Mark invoice as rejected/expired
                $updateStmt = $conn->prepare("
                    UPDATE unpaid_invoices 
                    SET approval_status = 'rejected',
                        approved_at = NOW()
                    WHERE id = ? AND approval_status = 'pending'
                ");
                if ($updateStmt) {
                    $updateStmt->bind_param("i", $invoice['id']);
                    $updateStmt->execute();
                    $updateStmt->close();
                }
                
                $conn->commit();
                $expired_count++;
                error_log("Expired invoice #{$invoice['id']} and returned reserved stock");
            } catch (Exception $e) {
                $conn->rollback();
                error_log("Error expiring invoice #{$invoice['id']}: " . $e->getMessage());
            }
        }
        
        if ($expired_count > 0) {
            error_log("Expired {$expired_count} unpaid invoices due to 3-hour timeout");
        }
        
        return $expired_count;
    } catch (Exception $e) {
        error_log("Error checking for expired unpaid invoices: " . $e->getMessage());
        return 0;
    }
}

try {
    error_log("Starting approve_invoices.php for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']}) at " . date('Y-m-d H:i:s'));

    // Check and expire unpaid invoices
    check_and_expire_unpaid_invoices($conn);

    $stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
    if (!$stmt) {
        throw new Exception("Prepare failed for role check: " . $conn->error);
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
    $stmt->close();
    $normalizedRole = mbh_normalize_role_name($role);
    $isStockManager = $normalizedRole === 'Stock Manager';
    if (!in_array($normalizedRole, ['DGA', 'DG', 'Admin', 'Stock Manager'], true)) {
        error_log("Unauthorized access to approve_invoices.php by user {$_SESSION['full_name']} (Role: $role)");
        header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
        exit;
    }
    $loan_admin_roles = ['Board', 'DG', 'DGA', 'Admin'];
    $can_view_loan_invoices = in_array($normalizedRole, $loan_admin_roles, true);

    // Fetch invoice statistics
    $stats = ['pending_count' => 0, 'approved_count' => 0, 'rejected_count' => 0];
    $stats_stmt = $conn->prepare("
        SELECT 
            (SELECT COUNT(*) FROM unpaid_invoices WHERE approval_status = 'pending') as pending_count,
            (SELECT COUNT(*) FROM unpaid_invoices WHERE approval_status = 'approved') as approved_count,
            (SELECT COUNT(*) FROM unpaid_invoices WHERE approval_status = 'rejected') as rejected_count
    ");
    if (!$stats_stmt) {
        throw new Exception("Prepare failed for stats: " . $conn->error);
    }
    $stats_stmt->execute();
    $stats = $stats_stmt->get_result()->fetch_assoc();
    $stats_stmt->close();

    $flashSuccessMessage = $_SESSION['approve_invoices_success'] ?? '';
    if ($flashSuccessMessage) {
        unset($_SESSION['approve_invoices_success']);
    }

    // Pagination setup
    $rows_per_page = 15;
    $page = isset($_GET['page']) && is_numeric($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
    $offset = ($page - 1) * $rows_per_page;

    // Fetch invoices with calculated total amount
    $invoices = [];
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    $status = isset($_GET['status']) ? trim($_GET['status']) : '';
    $province = isset($_GET['province']) ? trim($_GET['province']) : '';

    $query = "
        SELECT i.id, i.customer_name,
               i.invoice_number,
               i.created_at,
               i.has_custom_prices,
               i.custom_price_summary,
               i.is_loan_sale,
               i.paid_amount AS total_amount,
               pr.name AS province_name, i.approval_status
        FROM unpaid_invoices i
        LEFT JOIN province pr ON i.province_id = pr.id
        WHERE 1=1
    ";

    $params = [];
    if ($search) {
        $query .= " AND i.customer_name LIKE ?";
        $params[] = "%$search%";
    }
    if ($status) {
        $query .= " AND i.approval_status = ?";
        $params[] = $status;
    }
    if ($province) {
        $query .= " AND pr.name = ?";
        $params[] = $province;
    } elseif ($province === 'unassigned') {
        $query .= " AND pr.name IS NULL";
    }
    if ($isStockManager && !empty($_SESSION['province_id'])) {
        $query .= " AND i.province_id = ?";
        $params[] = (int)$_SESSION['province_id'];
    }
    if (!$can_view_loan_invoices) {
        $query .= " AND COALESCE(i.is_loan_sale, 0) = 0";
    }
    $total_query = $query; // For counting total rows
    $query .= " ORDER BY i.approval_status = 'pending' DESC, i.created_at DESC LIMIT ? OFFSET ?";
    $params[] = $rows_per_page;
    $params[] = $offset;

    $stmt = $conn->prepare($query);
    if ($stmt === false) {
        throw new Exception("Prepare failed: " . $conn->error);
    }
    if (!empty($params)) {
        $filter_params_count = count($params) - 2; // Exclude LIMIT and OFFSET
        $types = $filter_params_count > 0 ? str_repeat('s', $filter_params_count) : '';
        $types .= 'ii'; // Append 'ii' for LIMIT and OFFSET
        if (!empty($types)) {
            $stmt->bind_param($types, ...$params);
        }
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $invoices = $result->fetch_all(MYSQLI_ASSOC);
    error_log("Fetched " . count($invoices) . " invoices with status='$status' for province='$province' at " . date('Y-m-d H:i:s'));
    $stmt->close();

    // Preload invoice items for display in the modal (system vs custom prices)
    $invoice_items = [];
    if (!empty($invoices)) {
        $ids = array_map(function($row) { return (int)$row['id']; }, $invoices);
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $types_items = str_repeat('i', count($ids));
        $sql_items = "SELECT invoice_id, product_id, quantity, unit_price, price_source, system_unit_price, custom_unit_price, custom_price_basis, custom_price_input, custom_price_note FROM invoice_items WHERE invoice_id IN ($placeholders) ORDER BY invoice_id, product_id";
        $stmt_items = $conn->prepare($sql_items);
        if ($stmt_items) {
            $stmt_items->bind_param($types_items, ...$ids);
            if ($stmt_items->execute()) {
                $res_items = $stmt_items->get_result();
                while ($row = $res_items->fetch_assoc()) {
                    $invId = (int)$row['invoice_id'];
                    if (!isset($invoice_items[$invId])) {
                        $invoice_items[$invId] = [];
                    }
                    $invoice_items[$invId][] = $row;
                }
            }
            $stmt_items->close();
        }
    }

    // Count total rows for pagination
    $total_stmt = $conn->prepare($total_query);
    if ($total_stmt === false) {
        throw new Exception("Prepare failed for total count: " . $conn->error);
    }
    if (!empty($params)) {
        array_pop($params); // Remove OFFSET
        array_pop($params); // Remove LIMIT
        $types = str_repeat('s', count($params));
        if (!empty($types)) {
            $total_stmt->bind_param($types, ...$params);
        }
    }
    $total_stmt->execute();
    $total_result = $total_stmt->get_result();
    $total_rows = $total_result->num_rows;
    $total_pages = ceil($total_rows / $rows_per_page);
    $total_stmt->close();

    // Fetch unique provinces for filter
    $provinces = [];
    $prov_stmt = $conn->prepare("SELECT DISTINCT name FROM province WHERE name IS NOT NULL ORDER BY name");
    if (!$prov_stmt) {
        throw new Exception("Prepare failed for provinces: " . $conn->error);
    }
    $prov_stmt->execute();
    $provinces = $prov_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $prov_stmt->close();

    // Handle form submission for approval/rejection
    $errors = [];
    $success = $flashSuccessMessage;
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        $invoice_id = intval($_POST['invoice_id'] ?? 0);
        $action = $_POST['action'];
        $comment = trim($_POST['comment'] ?? '');

        if ($invoice_id <= 0) {
            $errors[] = "Commande invalide sélectionnée.";
        } elseif (!in_array($action, ['approve', 'reject'])) {
            $errors[] = "Action invalide.";
        } else {
            $conn->begin_transaction();
            try {
                $new_status = ($action === 'approve') ? 'approved' : 'rejected';
                $approved_by = $_SESSION['user_id'];

                $metaStmt = $conn->prepare("SELECT invoice_number, customer_name, created_by FROM unpaid_invoices WHERE id = ? FOR UPDATE");
                if (!$metaStmt) {
                    throw new Exception("Prepare failed for invoice fetch: " . $conn->error);
                }
                $metaStmt->bind_param("i", $invoice_id);
                $metaStmt->execute();
                $invoice_meta = $metaStmt->get_result()->fetch_assoc() ?: [];
                $metaStmt->close();

                // Fetch invoice items and province_id for stock operations
                $itemsStmt = $conn->prepare("
                    SELECT ii.product_id, ii.quantity, p.crate_quantity
                    FROM invoice_items ii
                    INNER JOIN product p ON ii.product_id = p.id
                    WHERE ii.invoice_id = ?
                ");
                if (!$itemsStmt) {
                    throw new Exception("Prepare failed for invoice items: " . $conn->error);
                }
                $itemsStmt->bind_param("i", $invoice_id);
                $itemsStmt->execute();
                $items_result = $itemsStmt->get_result();
                $invoice_items = [];
                while ($row = $items_result->fetch_assoc()) {
                    $invoice_items[] = $row;
                }
                $itemsStmt->close();
                
                // Get province_id from invoice
                $provinceStmt = $conn->prepare("SELECT province_id FROM unpaid_invoices WHERE id = ?");
                if (!$provinceStmt) {
                    throw new Exception("Prepare failed for province fetch: " . $conn->error);
                }
                $provinceStmt->bind_param("i", $invoice_id);
                $provinceStmt->execute();
                $province_result = $provinceStmt->get_result();
                $province_row = $province_result->fetch_assoc();
                $province_id = $province_row ? (int)$province_row['province_id'] : null;
                $provinceStmt->close();
                
                if ($action === 'reject') {
                    // Get invoice details to check if it's a loan sale
                    $invoiceStmt = $conn->prepare("SELECT is_loan_sale, loan_outstanding_bif, customer_id, customer_name, phone_number FROM unpaid_invoices WHERE id = ?");
                    if (!$invoiceStmt) {
                        throw new Exception("Prepare failed for invoice details: " . $conn->error);
                    }
                    $invoiceStmt->bind_param("i", $invoice_id);
                    $invoiceStmt->execute();
                    $invoiceDetails = $invoiceStmt->get_result()->fetch_assoc();
                    $invoiceStmt->close();
                    
                    $is_loan_sale = !empty($invoiceDetails['is_loan_sale']);
                    $loan_outstanding_bif = (int)($invoiceDetails['loan_outstanding_bif'] ?? 0);
                    $customer_id = !empty($invoiceDetails['customer_id']) ? (int)$invoiceDetails['customer_id'] : null;
                    $customer_name = $invoiceDetails['customer_name'] ?? '';
                    $phone_number = $invoiceDetails['phone_number'] ?? '';
                    
                    // CRITICAL: If this is a loan sale, reverse the loan balance that was added when invoice was created
                    if ($is_loan_sale && $loan_outstanding_bif > 0) {
                        // Find customer by ID or name/phone
                        if ($customer_id) {
                            $customerLookupId = $customer_id;
                        } else {
                            // Try to find customer by name and phone
                            $findCustomerStmt = $conn->prepare("SELECT id FROM customer WHERE full_name = ? AND tel = ? LIMIT 1");
                            if ($findCustomerStmt) {
                                $findCustomerStmt->bind_param("ss", $customer_name, $phone_number);
                                $findCustomerStmt->execute();
                                $customerRow = $findCustomerStmt->get_result()->fetch_assoc();
                                $customerLookupId = $customerRow ? (int)$customerRow['id'] : null;
                                $findCustomerStmt->close();
                            } else {
                                $customerLookupId = null;
                            }
                        }
                        
                        if ($customerLookupId) {
                            // Reverse the loan balance (subtract the amount that was added)
                            $reverseStmt = $conn->prepare("
                                UPDATE customer 
                                SET loan_balance_bif = GREATEST(0, loan_balance_bif - ?), 
                                    loan_last_review_at = NOW()
                                WHERE id = ?
                            ");
                            if ($reverseStmt) {
                                $reverseStmt->bind_param("ii", $loan_outstanding_bif, $customerLookupId);
                                if ($reverseStmt->execute()) {
                                    error_log("✅ Reversed loan balance for customer ID {$customerLookupId}: subtracted {$loan_outstanding_bif} BIF (rejected invoice #{$invoice_id})");
                                } else {
                                    error_log("❌ Failed to reverse loan balance: " . $reverseStmt->error);
                                    throw new Exception("Failed to reverse loan balance: " . $reverseStmt->error);
                                }
                                $reverseStmt->close();
                            }
                            
                            // Also create a reversal entry in the loan ledger
                            // Use 'loan_repayment' entry_type with 'credit' direction to reverse the loan
                            $ledgerReversalStmt = $conn->prepare("
                            INSERT INTO customer_loan_ledger (
                                customer_id, entry_type, entry_origin, direction, amount_bif, reference_table, reference_id, notes, created_by
                            ) VALUES (?, 'loan_repayment', 'loan_reversal', 'credit', ?, 'unpaid_invoices', ?, ?, ?)
                            ");
                            if ($ledgerReversalStmt) {
                                $reversalNote = "Reversal: Invoice rejected - Loan balance reversed";
                                $amountParam = (string)$loan_outstanding_bif;
                                $ledgerReversalStmt->bind_param("isisi", $customerLookupId, $amountParam, $invoice_id, $reversalNote, $_SESSION['user_id']);
                                if ($ledgerReversalStmt->execute()) {
                                    error_log("✅ Created loan ledger reversal entry for rejected invoice #{$invoice_id}");
                                } else {
                                    error_log("⚠️ Failed to create loan ledger reversal entry: " . $ledgerReversalStmt->error);
                                    // Don't throw exception - balance reversal is more important than ledger entry
                                }
                                $ledgerReversalStmt->close();
                            }
                        } else {
                            error_log("⚠️ Could not find customer to reverse loan balance for rejected invoice #{$invoice_id} (customer_id: {$customer_id}, name: {$customer_name})");
                        }
                    }
                    
                    // Return reserved stock back to available stock
                    error_log("=== REJECT INVOICE DEBUG ===");
                    error_log("Invoice ID: {$invoice_id}, Province ID: {$province_id}, Items count: " . count($invoice_items));
                    
                    if ($province_id && !empty($invoice_items)) {
                        foreach ($invoice_items as $item) {
                            $product_id = intval($item['product_id']);
                            $quantity_in_crates = max(1, intval($item['quantity']));
                            $crate_quantity = max(1, intval($item['crate_quantity']));

                            error_log("📦 Processing item - Product: {$product_id}, Crates to return: {$quantity_in_crates} (crate size: {$crate_quantity})");
                            
                            // Check current stock state before update - ALL records
                            $checkStmt = $conn->prepare("
                                SELECT id, quantity, reserved_quantity, unit_type, COALESCE(quantity, 0) - COALESCE(reserved_quantity, 0) as available
                                FROM province_stock 
                                WHERE product_id = ? AND province_id = ?
                            ");
                            if ($checkStmt) {
                                $checkStmt->bind_param("ii", $product_id, $province_id);
                                $checkStmt->execute();
                                $stock_result = $checkStmt->get_result();
                                $found_crates = false;
                                while ($stock_row = $stock_result->fetch_assoc()) {
                                    error_log("📊 FOUND stock: id={$stock_row['id']}, qty={$stock_row['quantity']}, reserved={$stock_row['reserved_quantity']}, unit={$stock_row['unit_type']}, available={$stock_row['available']}");
                                    if ($stock_row['unit_type'] === 'crates') {
                                        $found_crates = true;
                                    }
                                }
                                if (!$found_crates) {
                                    error_log("❌ No 'crates' record found for product {$product_id} in province {$province_id}");
                                }
                                $checkStmt->close();
                            }
                            
                            // Decrease reserved_quantity to return stock to available
                            error_log("Rejecting invoice #{$invoice_id}: Returning {$quantity_in_crates} crates of product {$product_id} to available stock");
                            
                                error_log("⚠️ ATTEMPTING UPDATE: reserved_quantity = GREATEST(0, reserved_quantity - {$quantity_in_crates}) WHERE product_id={$product_id} AND province_id={$province_id} AND unit_type='crates'");
                            
                            $returnStmt = $conn->prepare("
                                UPDATE province_stock 
                                SET reserved_quantity = GREATEST(0, reserved_quantity - ?)
                                WHERE product_id = ? 
                                  AND province_id = ?
                                  AND unit_type = 'crates'
                            ");
                            if (!$returnStmt) {
                                throw new Exception("Prepare failed for product return: " . $conn->error);
                            }
                            $returnStmt->bind_param("iii", $quantity_in_crates, $product_id, $province_id);
                            if (!$returnStmt->execute()) {
                                throw new Exception("Failed to return reserved stock: " . $returnStmt->error);
                            }
                            
                            $affected = $returnStmt->affected_rows;
                            error_log("UPDATE result: affected_rows = {$affected}");
                            $returnStmt->close();
                            
                            // Verify that stock was actually returned
                            if ($affected === 0) {
                                error_log("⚠️ WARNING: No rows updated when returning stock for product {$product_id} in province {$province_id}. Stock record may not exist or reserved_quantity may already be 0.");
                                // Don't throw exception - log warning and continue
                            }
                            
                            // Check updated stock state
                            $checkStmt = $conn->prepare("
                                SELECT quantity, reserved_quantity, COALESCE(quantity, 0) - COALESCE(reserved_quantity, 0) as available
                                FROM province_stock 
                                WHERE product_id = ? AND province_id = ? AND unit_type = 'crates'
                            ");
                            if ($checkStmt) {
                                $checkStmt->bind_param("ii", $product_id, $province_id);
                                $checkStmt->execute();
                                $stock_result = $checkStmt->get_result();
                                if ($stock_row = $stock_result->fetch_assoc()) {
                                    error_log("✅ After reject: qty={$stock_row['quantity']}, reserved={$stock_row['reserved_quantity']}, available={$stock_row['available']} (affected_rows: $affected)");
                                } else {
                                    error_log("❌ Stock record missing after update!");
                                }
                                $checkStmt->close();
                            }
                        }
                    } else {
                        error_log("❌ REJECT FAILED: province_id is {$province_id}, items count: " . count($invoice_items));
                    }
                    error_log("=== END REJECT DEBUG ===");
                } elseif ($action === 'approve') {
                    // Finalize stock: deduct from both reserved_quantity and quantity
                    error_log("=== APPROVE INVOICE DEBUG ===");
                    error_log("Invoice ID: {$invoice_id}, Province ID: {$province_id}, Items count: " . count($invoice_items));
                    
                    if ($province_id && !empty($invoice_items)) {
                        foreach ($invoice_items as $item) {
                            $product_id = intval($item['product_id']);
                            $quantity_in_crates = max(1, intval($item['quantity']));
                            $crate_quantity = max(1, intval($item['crate_quantity']));

                            error_log("📦 Processing item - Product: {$product_id}, Crates to finalize: {$quantity_in_crates} (crate size: {$crate_quantity})");
                            
                            // Check current stock before finalization - ALL records
                            $checkStmt = $conn->prepare("
                                SELECT id, quantity, reserved_quantity, unit_type, COALESCE(quantity, 0) - COALESCE(reserved_quantity, 0) as available
                                FROM province_stock 
                                WHERE product_id = ? AND province_id = ?
                            ");
                            if ($checkStmt) {
                                $checkStmt->bind_param("ii", $product_id, $province_id);
                                $checkStmt->execute();
                                $stock_result = $checkStmt->get_result();
                                $found_crates = false;
                                while ($stock_row = $stock_result->fetch_assoc()) {
                                    error_log("📊 FOUND stock (approve): id={$stock_row['id']}, qty={$stock_row['quantity']}, reserved={$stock_row['reserved_quantity']}, unit={$stock_row['unit_type']}, available={$stock_row['available']}");
                                    if ($stock_row['unit_type'] === 'crates') {
                                        $found_crates = true;
                                    }
                                }
                                if (!$found_crates) {
                                    error_log("❌ No 'crates' record found for product {$product_id} in province {$province_id}");
                                }
                                $checkStmt->close();
                            }
                            
                            // Deduct from both reserved_quantity and quantity (permanently remove from stock)
                            error_log("Approving invoice #{$invoice_id}: Finalizing stock deduction of {$quantity_in_crates} crates for product {$product_id}");
                            error_log("⚠️ ATTEMPTING APPROVAL UPDATE: reserved_quantity = GREATEST(0, reserved_quantity - {$quantity_in_crates}), quantity = GREATEST(0, quantity - {$quantity_in_crates}) WHERE product_id={$product_id} AND province_id={$province_id} AND unit_type='crates'");
                            
                            $finalizeStmt = $conn->prepare("
                                UPDATE province_stock 
                                SET reserved_quantity = GREATEST(0, reserved_quantity - ?),
                                    quantity = GREATEST(0, quantity - ?)
                                WHERE product_id = ? 
                                  AND province_id = ?
                                  AND unit_type = 'crates'
                            ");
                            if (!$finalizeStmt) {
                                throw new Exception("Prepare failed for stock finalization: " . $conn->error);
                            }
                            $finalizeStmt->bind_param("iiii", $quantity_in_crates, $quantity_in_crates, $product_id, $province_id);
                            if (!$finalizeStmt->execute()) {
                                throw new Exception("Failed to finalize stock deduction: " . $finalizeStmt->error);
                            }
                            
                            $affected = $finalizeStmt->affected_rows;
                            error_log("APPROVAL UPDATE result: affected_rows = {$affected}");
                            $finalizeStmt->close();
                            
                            // Check stock after finalization
                            $checkStmt = $conn->prepare("
                                SELECT quantity, reserved_quantity, COALESCE(quantity, 0) - COALESCE(reserved_quantity, 0) as available
                                FROM province_stock 
                                WHERE product_id = ? AND province_id = ? AND unit_type = 'crates'
                            ");
                            if ($checkStmt) {
                                $checkStmt->bind_param("ii", $product_id, $province_id);
                                $checkStmt->execute();
                                $stock_result = $checkStmt->get_result();
                                if ($stock_row = $stock_result->fetch_assoc()) {
                                    error_log("✅ After approve: qty={$stock_row['quantity']}, reserved={$stock_row['reserved_quantity']}, available={$stock_row['available']} (affected_rows: $affected)");
                                } else {
                                    error_log("❌ Stock record missing after update!");
                                }
                                $checkStmt->close();
                            }
                            
                            if ($affected === 0) {
                                throw new Exception("Failed to finalize stock deduction for product {$product_id}: No rows updated. Check if stock record exists with matching product_id, province_id, and unit_type='crates'.");
                            }
                        }
                    } else {
                        error_log("❌ APPROVE FAILED: province_id is {$province_id}, items count: " . count($invoice_items));
                    }
                    error_log("=== END APPROVE DEBUG ===");
                }

                $stmt = $conn->prepare("
                    UPDATE unpaid_invoices 
                    SET approval_status = ?, status = ?, approved_by = ?, approved_at = NOW()
                    WHERE id = ? AND approval_status = 'pending'
                ");
                if (!$stmt) {
                    throw new Exception("Prepare failed for update: " . $conn->error);
                }
                $new_payment_status = ($action === 'approve') ? 'unpaid' : 'pending';
                $stmt->bind_param("ssii", $new_status, $new_payment_status, $approved_by, $invoice_id);
                if (!$stmt->execute() || $stmt->affected_rows === 0) {
                    throw new Exception("Failed to update invoice status or invoice is not pending: " . $stmt->error);
                }
                $stmt->close();

                if (!empty($invoice_meta['created_by']) && intval($invoice_meta['created_by']) !== $approved_by) {
                    $notifyMessage = $new_status === 'approved'
                        ? sprintf("Commande %s pour %s a été approuvée.", $invoice_meta['invoice_number'] ?? ("#" . $invoice_id), $invoice_meta['customer_name'] ?? 'client')
                        : sprintf("Commande %s pour %s a été rejetée.", $invoice_meta['invoice_number'] ?? ("#" . $invoice_id), $invoice_meta['customer_name'] ?? 'client');
                    if ($new_status === 'rejected' && $comment !== '') {
                        $reason = substr($comment, 0, 120);
                        if (strlen($comment) > 120) {
                            $reason .= '...';
                        }
                        $notifyMessage .= ' Raison: ' . $reason;
                    }
                    $notifyLink = '/masunzu_bar_hotel/modules/cashiers/view_invoice.php?id=' . $invoice_id;
                    $stmtNotify = $conn->prepare("INSERT INTO user_notification (user_id, message, link) VALUES (?, ?, ?)");
                    if ($stmtNotify) {
                        $stmtNotify->bind_param("iss", $invoice_meta['created_by'], $notifyMessage, $notifyLink);
                        $stmtNotify->execute();
                        $stmtNotify->close();
                    } else {
                        error_log('Failed to prepare notification insert for invoice ' . $invoice_id . ': ' . $conn->error);
                    }
                }
                // Note: assigned_cashier_id column doesn't exist in unpaid_invoices table
                // If cashier assignment is needed in the future, it should be added to the table schema

                log_action($_SESSION['user_id'], 'approve_invoices', "$action invoice $invoice_id with comment: $comment");
                error_log("Invoice $invoice_id $action by user {$_SESSION['user_id']} with comment: $comment");
                $conn->commit();
                $action_fr = ($action === 'approve') ? 'approuvée' : 'rejetée';
                $flashMessage = "Commande $invoice_id $action_fr avec succès. Commentaire: $comment";
                $_SESSION['approve_invoices_success'] = $flashMessage;
                $redirectUrl = $_SERVER['REQUEST_URI'] ?? '/masunzu_bar_hotel/modules/admin/approve_invoices.php';
                $conn->close();
                header("Location: $redirectUrl");
                exit;
            } catch (Exception $e) {
                $conn->rollback();
                $action_fr = ($action === 'approve') ? 'approuver' : 'rejeter';
                $errors[] = "Échec de l'action pour $action_fr la commande: " . $e->getMessage();
                error_log("Error in approve_invoices.php for invoice_id=$invoice_id: " . $e->getMessage());
            }
        }
    }

    include __DIR__ . '/../../includes/header.php';
?>
<style>
    body {
        background-color: #F4F0E4;
        margin: 0;
        padding: 0;
    }
    .approve-invoices-container {
        width: 100%;
        padding: 30px 40px;
        min-height: calc(100vh - 100px);
    }
    .page-header {
        margin-bottom: 30px;
    }
    .page-title {
        color: #4B2F1F;
        font-size: 32px;
        font-weight: 700;
        margin: 0 0 8px 0;
    }
    .page-subtitle {
        color: #6B5B52;
        font-size: 16px;
        margin: 0;
    }
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 20px;
        margin-bottom: 30px;
    }
    .stat-card {
        background: #FFFFFF;
        border-radius: 12px;
        padding: 24px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        text-align: center;
        transition: transform 0.2s, box-shadow 0.2s;
    }
    .stat-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.12);
    }
    .stat-label {
        color: #6B5B52;
        font-size: 14px;
        font-weight: 500;
        margin-bottom: 8px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .stat-value {
        color: #4B2F1F;
        font-size: 36px;
        font-weight: 700;
        margin: 0;
    }
    .stat-value.pending { color: #F4A261; }
    .stat-value.approved { color: #4B2F1F; }
    .stat-value.rejected { color: #DC3545; }
    .filters-bar {
        background: #FFFFFF;
        border-radius: 12px;
        padding: 20px;
        margin-bottom: 30px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        display: flex;
        gap: 15px;
        align-items: center;
        flex-wrap: wrap;
    }
    .filter-input, .filter-select {
        padding: 10px 15px;
        border: 1px solid #D3C2B6;
        border-radius: 8px;
        font-size: 14px;
        color: #4B2F1F;
        background: #F4F0E4;
        transition: border-color 0.2s;
    }
    .filter-input:focus, .filter-select:focus {
        outline: none;
        border-color: #4B2F1F;
        background: #FFFFFF;
    }
    .filter-input {
        flex: 1;
        min-width: 200px;
    }
    .filter-select {
        min-width: 150px;
    }
    .btn-apply {
        padding: 10px 24px;
        background: #4B2F1F;
        color: #F4F0E4;
        border: none;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: background 0.2s;
    }
    .btn-apply:hover {
        background: #3A2418;
    }
    .invoices-table-wrapper {
        background: #FFFFFF;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        overflow: hidden;
    }
    .invoices-table {
        width: 100%;
        border-collapse: collapse;
    }
    .invoices-table thead {
        background: #4B2F1F;
        color: #F4F0E4;
    }
    .invoices-table th {
        padding: 16px;
        text-align: left;
        font-weight: 600;
        font-size: 14px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .invoices-table tbody tr {
        border-bottom: 1px solid #F4F0E4;
        transition: background 0.2s;
    }
    .invoices-table tbody tr:hover {
        background: #FDFBF7;
    }
    .invoices-table td {
        padding: 16px;
        color: #4B2F1F;
        font-size: 14px;
    }
    .badge {
        display: inline-block;
        padding: 4px 10px;
        border-radius: 12px;
        font-size: 11px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-left: 8px;
    }
    .badge-custom {
        background: #FFF3CD;
        color: #856404;
        border: 1px solid #FFE69C;
    }
    .badge-loan {
        background: #D1ECF1;
        color: #0C5460;
        border: 1px solid #BEE5EB;
    }
    @keyframes blinking {
        0%, 49% { opacity: 1; }
        50%, 100% { opacity: 0.3; }
    }
    .loan-sale-highlight {
        color: #FF0000;
        font-weight: bold;
        animation: blinking 1s infinite;
    }
    .badge-status {
        padding: 6px 12px;
        border-radius: 8px;
        font-size: 12px;
    }
    .badge-status.pending {
        background: #FFF3CD;
        color: #856404;
    }
    .badge-status.approved {
        background: #D4EDDA;
        color: #155724;
    }
    .badge-status.rejected {
        background: #F8D7DA;
        color: #721C24;
    }
    .action-buttons {
        display: flex;
        gap: 8px;
    }
    .btn-action {
        padding: 8px 16px;
        border: none;
        border-radius: 6px;
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
    }
    .btn-view {
        background: #2196F3;
        color: #FFFFFF;
    }
    .btn-view:hover {
        background: #1976D2;
    }
    .btn-approve {
        background: #4B2F1F;
        color: #F4F0E4;
    }
    .btn-approve:hover {
        background: #3A2418;
    }
    .btn-reject {
        background: #DC3545;
        color: #FFFFFF;
    }
    .btn-reject:hover {
        background: #C82333;
    }
    .pagination {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 8px;
        margin-top: 30px;
        flex-wrap: wrap;
    }
    .pagination a, .pagination span {
        padding: 10px 16px;
        border-radius: 8px;
        text-decoration: none;
        font-size: 14px;
        font-weight: 600;
        transition: all 0.2s;
    }
    .pagination a {
        background: #F4A261;
        color: #4B2F1F;
    }
    .pagination a:hover {
        background: #E68A4F;
    }
    .pagination span.active {
        background: #4B2F1F;
        color: #F4F0E4;
    }
    .pagination-info {
        color: #6B5B52;
        font-size: 14px;
        margin-left: 15px;
    }
    .footer-actions {
        display: flex;
        justify-content: center;
        gap: 15px;
        margin-top: 30px;
        flex-wrap: wrap;
    }
    .footer-actions a {
        padding: 12px 24px;
        background: #F4A261;
        color: #4B2F1F;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        transition: background 0.2s;
    }
    .footer-actions a:hover {
        background: #E68A4F;
    }
    .alert {
        padding: 16px 20px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-size: 14px;
    }
    .alert-error {
        background: #F8D7DA;
        color: #721C24;
        border: 1px solid #F5C6CB;
    }
    .alert-success {
        background: #D4EDDA;
        color: #155724;
        border: 1px solid #C3E6CB;
    }
    .modal {
        display: none;
        position: fixed;
        z-index: 10000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        overflow: auto;
        background-color: rgba(0,0,0,0.5);
        backdrop-filter: blur(2px);
    }
    .modal.show {
        display: flex;
        align-items: center;
        justify-content: center;
    }
    .modal-content {
        background-color: #F4F0E4;
        margin: auto;
        padding: 0;
        border-radius: 12px;
        width: 90%;
        max-width: 900px;
        max-height: 90vh;
        overflow: hidden;
        box-shadow: 0 8px 32px rgba(0,0,0,0.3);
        display: flex;
        flex-direction: column;
    }
    .modal-header {
        background: #4B2F1F;
        color: #F4F0E4;
        padding: 20px 30px;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .modal-title {
        margin: 0;
        font-size: 24px;
        font-weight: 700;
    }
    .modal-close {
        background: none;
        border: none;
        color: #F4F0E4;
        font-size: 32px;
        cursor: pointer;
        line-height: 1;
        padding: 0;
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        transition: background 0.2s;
    }
    .modal-close:hover {
        background: rgba(255,255,255,0.2);
    }
    .modal-body {
        padding: 30px;
        overflow-y: auto;
        flex: 1;
    }
    .modal-info-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin-bottom: 25px;
        padding: 20px;
        background: #FFFFFF;
        border-radius: 8px;
    }
    .modal-info-item {
        display: flex;
        flex-direction: column;
    }
    .modal-info-label {
        color: #6B5B52;
        font-size: 12px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 4px;
    }
    .modal-info-value {
        color: #4B2F1F;
        font-size: 16px;
        font-weight: 600;
    }
    .modal-items-table {
        width: 100%;
        border-collapse: collapse;
        margin-bottom: 20px;
        background: #FFFFFF;
        border-radius: 8px;
        overflow: hidden;
    }
    .modal-items-table thead {
        background: #F4A261;
        color: #4B2F1F;
    }
    .modal-items-table th {
        padding: 12px;
        text-align: left;
        font-weight: 600;
        font-size: 13px;
        text-transform: uppercase;
    }
    .modal-items-table td {
        padding: 12px;
        border-bottom: 1px solid #F4F0E4;
        color: #4B2F1F;
        font-size: 14px;
    }
    .modal-items-table tbody tr:last-child td {
        border-bottom: none;
    }
    .modal-items-table tbody tr.total-row {
        background: #FDFBF7;
        font-weight: 700;
    }
    .modal-items-table tbody tr.total-row td {
        padding: 16px 12px;
    }
    .custom-price-summary {
        background: #FFF3CD;
        color: #856404;
        border: 1px solid #FFE69C;
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-size: 14px;
    }
    .modal-form {
        margin-top: 20px;
        padding-top: 20px;
        border-top: 2px solid #D3C2B6;
    }
    .modal-form textarea {
        width: 100%;
        padding: 12px;
        border: 1px solid #D3C2B6;
        border-radius: 8px;
        font-size: 14px;
        font-family: inherit;
        resize: vertical;
        min-height: 80px;
        margin-bottom: 15px;
        color: #4B2F1F;
    }
    .modal-form textarea:focus {
        outline: none;
        border-color: #4B2F1F;
    }
    .modal-form-buttons {
        display: flex;
        gap: 12px;
        justify-content: flex-end;
    }
    .empty-state {
        text-align: center;
        padding: 60px 20px;
        color: #6B5B52;
        font-size: 16px;
    }
    @media (max-width: 768px) {
        .approve-invoices-container {
            padding: 20px;
        }
        .stats-grid {
            grid-template-columns: 1fr;
        }
        .filters-bar {
            flex-direction: column;
            align-items: stretch;
        }
        .filter-input, .filter-select {
            width: 100%;
        }
        .invoices-table-wrapper {
            overflow-x: auto;
        }
        .modal-content {
            width: 95%;
            max-height: 95vh;
        }
        .modal-body {
            padding: 20px;
        }
    }
</style>
<div class="approve-invoices-container">
    <div class="page-header">
        <h1 class="page-title">Approbation de Commandes</h1>
        <p class="page-subtitle">Examiner et approuver les commandes en attente</p>
    </div>

    <!-- Statistics Dashboard -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-label">En Attente</div>
            <div class="stat-value pending"><?php echo number_format($stats['pending_count'] ?? 0, 0, ',', '.'); ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Approuvées</div>
            <div class="stat-value approved"><?php echo number_format($stats['approved_count'] ?? 0, 0, ',', '.'); ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Rejetées</div>
            <div class="stat-value rejected"><?php echo number_format($stats['rejected_count'] ?? 0, 0, ',', '.'); ?></div>
        </div>
    </div>

    <?php if (!empty($errors)): ?>
        <div class="alert alert-error">
            <ul style="margin: 0; padding-left: 20px;">
                <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>
    <?php if ($success): ?>
        <div class="alert alert-success">
            <?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-bar">
        <input type="text" id="search" class="filter-input" placeholder="Rechercher par nom client..." value="<?php echo htmlspecialchars($search, ENT_QUOTES, 'UTF-8'); ?>">
        <select id="status" class="filter-select">
            <option value="">Tous les Statuts</option>
            <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>En Attente</option>
            <option value="approved" <?php echo $status === 'approved' ? 'selected' : ''; ?>>Approuvées</option>
            <option value="rejected" <?php echo $status === 'rejected' ? 'selected' : ''; ?>>Rejetées</option>
        </select>
        <select id="province" class="filter-select">
            <option value="">Toutes les Provinces</option>
            <?php foreach ($provinces as $prov): ?>
                <option value="<?php echo htmlspecialchars($prov['name'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $province === $prov['name'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($prov['name'], ENT_QUOTES, 'UTF-8'); ?></option>
            <?php endforeach; ?>
            <option value="unassigned" <?php echo $province === 'unassigned' ? 'selected' : ''; ?>>Non Assignée</option>
        </select>
        <button onclick="applyFilters()" class="btn-apply">Appliquer Filtres</button>
    </div>

    <!-- Invoices Table -->
    <?php if (empty($invoices)): ?>
        <div class="empty-state">
            <p>Aucune commande trouvée avec les filtres actuels.</p>
        </div>
    <?php else: ?>
        <div class="invoices-table-wrapper">
            <table class="invoices-table">
                <thead>
                    <tr>
                        <th>Commande #</th>
                        <th>Client</th>
                        <th>Montant</th>
                        <th>Province</th>
                        <th>Statut</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($invoices as $invoice): ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($invoice['invoice_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></strong>
                                <div style="font-size: 11px; color: #6B5B52; margin-top: 4px;">
                                    <?php echo date('Y-m-d H:i', strtotime($invoice['created_at'])); ?>
                                </div>
                            </td>
                            <td>
                                <?php echo htmlspecialchars($invoice['customer_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                <?php if (!empty($invoice['has_custom_prices'])): ?>
                                    <span class="badge badge-custom">Prix Personnalisé</span>
                                <?php endif; ?>
                                <?php if (!empty($invoice['is_loan_sale'])): ?>
                                    <span class="badge badge-loan">Vente Crédit</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <strong><?php echo number_format($invoice['total_amount'] ?? 0, 0, ',', '.'); ?> BIF</strong>
                            </td>
                            <td><?php echo htmlspecialchars($invoice['province_name'] ?? 'Non Assignée', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <span class="badge badge-status <?php echo htmlspecialchars($invoice['approval_status'] ?? 'pending', ENT_QUOTES, 'UTF-8'); ?>">
                                    <?php 
                                        $status = $invoice['approval_status'] ?? 'pending';
                                        $status_fr = ($status === 'pending') ? 'En Attente' : (($status === 'approved') ? 'Approuvée' : (($status === 'rejected') ? 'Rejetée' : ucfirst($status)));
                                        echo htmlspecialchars($status_fr, ENT_QUOTES, 'UTF-8'); 
                                    ?>
                                </span>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <button onclick="openInvoiceModal(<?php echo $invoice['id']; ?>)" class="btn-action btn-view">Voir</button>
                                    <?php if ($invoice['approval_status'] === 'pending'): ?>
                                        <button onclick="openInvoiceModal(<?php echo $invoice['id']; ?>)" class="btn-action btn-approve">Examiner</button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($status); ?>&province=<?php echo urlencode($province); ?>">Précédent</a>
            <?php endif; ?>
            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                <?php if ($i === $page): ?>
                    <span class="active"><?php echo $i; ?></span>
                <?php else: ?>
                    <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($status); ?>&province=<?php echo urlencode($province); ?>"><?php echo $i; ?></a>
                <?php endif; ?>
            <?php endfor; ?>
            <?php if ($page < $total_pages): ?>
                <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($status); ?>&province=<?php echo urlencode($province); ?>">Suivant</a>
            <?php endif; ?>
            <span class="pagination-info">Page <?php echo $page; ?> sur <?php echo $total_pages; ?></span>
        </div>
    <?php endif; ?>

    <!-- Footer Actions -->
    <div class="footer-actions">
        <a href="/masunzu_bar_hotel/modules/admin/invoice_history.php">Voir Historique</a>
        <a href="/masunzu_bar_hotel/dashboards/admin_dashboard.php">Retour au Tableau de Bord</a>
    </div>
</div>

<!-- Invoice Detail Modals -->
<?php foreach ($invoices as $invoice): ?>
    <div id="modal-<?php echo $invoice['id']; ?>" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 class="modal-title">Commande #<?php echo htmlspecialchars($invoice['invoice_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></h2>
                <button class="modal-close" onclick="closeInvoiceModal(<?php echo $invoice['id']; ?>)">&times;</button>
            </div>
            <div class="modal-body">
                <div class="modal-info-grid">
                    <div class="modal-info-item">
                        <div class="modal-info-label">Client</div>
                        <div class="modal-info-value"><?php echo htmlspecialchars($invoice['customer_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></div>
                    </div>
                    <div class="modal-info-item">
                        <div class="modal-info-label">Numéro Commande</div>
                        <div class="modal-info-value"><?php echo htmlspecialchars($invoice['invoice_number'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></div>
                    </div>
                    <div class="modal-info-item">
                        <div class="modal-info-label">Créée</div>
                        <div class="modal-info-value"><?php echo htmlspecialchars($invoice['created_at'] ?? '-', ENT_QUOTES, 'UTF-8'); ?></div>
                    </div>
                    <div class="modal-info-item">
                        <div class="modal-info-label">Statut</div>
                        <div class="modal-info-value">
                            <span class="badge badge-status <?php echo htmlspecialchars($invoice['approval_status'] ?? 'pending', ENT_QUOTES, 'UTF-8'); ?>">
                                <?php 
                                    $status = $invoice['approval_status'] ?? 'pending';
                                    $status_fr = ($status === 'pending') ? 'En Attente' : (($status === 'approved') ? 'Approuvée' : (($status === 'rejected') ? 'Rejetée' : ucfirst($status)));
                                    echo htmlspecialchars($status_fr, ENT_QUOTES, 'UTF-8'); 
                                ?>
                            </span>
                        </div>
                    </div>
                    <div class="modal-info-item">
                        <div class="modal-info-label">Province</div>
                        <div class="modal-info-value"><?php echo htmlspecialchars($invoice['province_name'] ?? 'Non Assignée', ENT_QUOTES, 'UTF-8'); ?></div>
                    </div>
                    <?php if (!empty($invoice['is_loan_sale'])): ?>
                        <div class="modal-info-item">
                            <div class="modal-info-label">Type</div>
                            <div class="modal-info-value"><span class="loan-sale-highlight">Vente Crédit</span></div>
                        </div>
                    <?php endif; ?>
                </div>

                <?php if (!empty($invoice['has_custom_prices']) && !empty($invoice['custom_price_summary'])): ?>
                    <div class="custom-price-summary">
                        <strong>Résumé Prix Personnalisé:</strong> <?php echo htmlspecialchars($invoice['custom_price_summary'], ENT_QUOTES, 'UTF-8'); ?>
                    </div>
                <?php endif; ?>

                <table class="modal-items-table">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Produit</th>
                            <th style="text-align: right;">Quantité</th>
                            <th style="text-align: right;">Prix Système</th>
                            <th style="text-align: right;">Prix Appliqué</th>
                            <th style="text-align: right;">Total Ligne</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $rows = $invoice_items[$invoice['id']] ?? [];
                        if (!empty($rows)) {
                            $idx = 1; $preview_total = 0;
                            // Get product crate_quantity for conversion
                            $productIds = array_unique(array_column($rows, 'product_id'));
                            $crateQuantities = [];
                            if (!empty($productIds)) {
                                $placeholders = implode(',', array_fill(0, count($productIds), '?'));
                                $types = str_repeat('i', count($productIds));
                                $crateStmt = $conn->prepare("SELECT id, crate_quantity FROM product WHERE id IN ($placeholders)");
                                if ($crateStmt) {
                                    $crateStmt->bind_param($types, ...$productIds);
                                    $crateStmt->execute();
                                    $crateResult = $crateStmt->get_result();
                                    while ($crateRow = $crateResult->fetch_assoc()) {
                                        $crateQuantities[(int)$crateRow['id']] = max(1, (int)$crateRow['crate_quantity']);
                                    }
                                    $crateStmt->close();
                                }
                            }
                            
                            foreach ($rows as $it) {
                                // quantity in invoice_items is stored in crates, not bottles
                                // All calculations use crates as the unit
                                $qty_crates = (int)$it['quantity'];
                                $product_id = (int)$it['product_id'];
                                $system = (int)($it['system_unit_price'] ?? 0);
                                $applied = (int)($it['unit_price'] ?? 0);
                                // unit_price is per crate, multiply by qty_crates
                                $line = $applied * $qty_crates; $preview_total += $line;
                        ?>
                            <tr>
                                <td><?php echo $idx++; ?></td>
                                <td>
                                    Produit #<?php echo htmlspecialchars($it['product_id'], ENT_QUOTES, 'UTF-8'); ?>
                                    <?php if (($it['price_source'] ?? 'system') === 'custom'): ?>
                                        <span class="badge badge-custom">Personnalisé</span>
                                    <?php endif; ?>
                                </td>
                                <td style="text-align: right;"><?php echo number_format($qty_crates, 0, ',', '.'); ?> caisses</td>
                                <td style="text-align: right;"><?php echo $system > 0 ? number_format($system, 0, ',', '.') . ' BIF' : '—'; ?></td>
                                <td style="text-align: right;">
                                    <?php if (($it['price_source'] ?? 'system') === 'custom'): ?>
                                        <div style="display:flex; flex-direction:column; align-items:flex-end;">
                                            <span style="font-weight:600;"><?php echo number_format($applied, 0, ',', '.'); ?> BIF</span>
                                            <span style="color:#999; font-size:11px; text-decoration:line-through;"><?php echo number_format($system, 0, ',', '.'); ?> BIF</span>
                                        </div>
                                    <?php else: ?>
                                        <?php echo number_format($applied, 0, ',', '.'); ?> BIF
                                    <?php endif; ?>
                                </td>
                                <td style="text-align: right; font-weight: 600;"><?php echo number_format($line, 0, ',', '.'); ?> BIF</td>
                            </tr>
                        <?php } ?>
                            <tr class="total-row">
                                <td colspan="5" style="text-align: right;">Montant Total</td>
                                <td style="text-align: right;"><?php echo number_format($preview_total, 0, ',', '.'); ?> BIF</td>
                            </tr>
                        <?php } else { ?>
                            <tr>
                                <td colspan="6" style="text-align: center; padding: 20px; color: #6B5B52;">
                                    Aucun article trouvé. Total: <?php echo number_format($invoice['total_amount'] ?? 0, 0, ',', '.'); ?> BIF
                                </td>
                            </tr>
                        <?php } ?>
                    </tbody>
                </table>

                <?php if ($invoice['approval_status'] === 'pending'): ?>
                    <form method="POST" class="modal-form" id="modal-form-<?php echo $invoice['id']; ?>">
                        <input type="hidden" name="invoice_id" value="<?php echo htmlspecialchars($invoice['id'], ENT_QUOTES, 'UTF-8'); ?>">
                        <textarea name="comment" placeholder="Entrer un commentaire (optionnel)" required></textarea>
                        <div class="modal-form-buttons">
                            <button type="submit" name="action" value="approve" class="btn-action btn-approve">Approuver</button>
                            <button type="submit" name="action" value="reject" class="btn-action btn-reject">Rejeter</button>
                        </div>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    </div>
<?php endforeach; ?>

<script>
    function applyFilters() {
        const search = document.getElementById('search').value;
        const status = document.getElementById('status').value;
        const province = document.getElementById('province').value;
        const url = new URL(window.location);
        url.searchParams.set('search', search);
        url.searchParams.set('status', status);
        url.searchParams.set('province', province);
        url.searchParams.delete('page'); // Reset to first page
        window.location = url.toString();
    }

    function openInvoiceModal(invoiceId) {
        const modal = document.getElementById('modal-' + invoiceId);
        if (modal) {
            modal.classList.add('show');
            document.body.style.overflow = 'hidden';
        }
    }

    function closeInvoiceModal(invoiceId) {
        const modal = document.getElementById('modal-' + invoiceId);
        if (modal) {
            modal.classList.remove('show');
            document.body.style.overflow = '';
        }
    }

    // Close modal when clicking outside
    document.addEventListener('click', function(event) {
        if (event.target.classList.contains('modal')) {
            event.target.classList.remove('show');
            document.body.style.overflow = '';
        }
    });

    // Close modal with Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            const modals = document.querySelectorAll('.modal.show');
            modals.forEach(modal => {
                modal.classList.remove('show');
                document.body.style.overflow = '';
            });
        }
    });

    // Auto-populate filters from URL
    document.addEventListener('DOMContentLoaded', function() {
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.has('search')) {
            document.getElementById('search').value = urlParams.get('search') || '';
        }
        if (urlParams.has('status')) {
            document.getElementById('status').value = urlParams.get('status') || '';
        }
        if (urlParams.has('province')) {
            document.getElementById('province').value = urlParams.get('province') || '';
        }
    });
</script>
<?php
$conn->close();
include __DIR__ . '/../../includes/footer.php';
?>
<?php
} catch (Exception $e) {
    $error_msg = "Critical error in approve_invoices.php: " . $e->getMessage();
    error_log($error_msg);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>$error_msg</p>";
    if ($conn) $conn->close();
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
?>
