<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$base_dir = '/opt/lampp/htdocs/masunzu_bar_hotel';
$auth_path = $base_dir . '/includes/auth.php';
$database_path = $base_dir . '/config/database.php';
$functions_path = $base_dir . '/includes/functions.php';
$footer_path = $base_dir . '/includes/footer.php';
$header_path = $base_dir . '/includes/header.php';

if (!file_exists($auth_path) || !file_exists($database_path) || !file_exists($functions_path)) {
    error_log("ERROR: Missing critical file - auth: " . file_exists($auth_path) . ", database: " . file_exists($database_path) . ", functions: " . file_exists($functions_path));
    die("Erreur: Fichier requis manquant. Contactez l'administrateur.");
}

require_once $auth_path;
require_once $database_path;
require_once $functions_path;

if (!function_exists('mbh_bind_params')) {
    function mbh_bind_params(mysqli_stmt $stmt, string $types, array &$values): bool
    {
        $bindParams = [$types];
        foreach ($values as &$value) {
            $bindParams[] = &$value;
        }
        return call_user_func_array([$stmt, 'bind_param'], $bindParams);
    }
}

try {
    error_log("DEBUG: admin/approve_expenses.php loaded at " . date('Y-m-d H:i:s') . ", user_id=" . ($_SESSION['user_id'] ?? 'unset') . ", method=" . $_SERVER['REQUEST_METHOD'] . ", raw_post=" . json_encode($_POST));

    require_login();

    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception("Échec de la connexion à la base de données: " . mysqli_connect_error());
    }

    // Ensure requisition_amount_history.id has AUTO_INCREMENT
    try {
        $check_id_col = $conn->query("SHOW COLUMNS FROM requisition_amount_history LIKE 'id'");
        if ($check_id_col && $check_id_col->num_rows > 0) {
            $id_col = $check_id_col->fetch_assoc();
            $has_auto_increment = isset($id_col['Extra']) && strpos(strtolower($id_col['Extra']), 'auto_increment') !== false;
            if (!$has_auto_increment) {
                // Check if id is a primary key
                $check_primary = $conn->query("SHOW KEYS FROM requisition_amount_history WHERE Key_name = 'PRIMARY' AND Column_name = 'id'");
                $is_primary = $check_primary && $check_primary->num_rows > 0;
                
                if ($is_primary) {
                    // Add AUTO_INCREMENT to the primary key
                    $conn->query("ALTER TABLE requisition_amount_history MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT");
                    error_log("✅ Added AUTO_INCREMENT to requisition_amount_history.id");
                }
            }
        }
    } catch (Exception $e) {
        error_log("⚠️ Error checking requisition_amount_history.id AUTO_INCREMENT: " . $e->getMessage());
    }

    $user_id = $_SESSION['user_id'];
    $role_id = $_SESSION['role_id'];
    $role_name = $_SESSION['role_name'];
    $province_id = $_SESSION['province_id'];
    $full_name = $_SESSION['full_name'];

    // Restrict access to only DG (role_id=2) or DGA (role_id=3)
    $allowed_roles = [1, 2, 3];
    if (!in_array($role_id, $allowed_roles)) {
        error_log("Unauthorized access attempt to approve_expenses.php by user_id {$user_id}, role_id {$role_id}");
        header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
        exit;
    }

    $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
    if (!$stmt) {
        throw new Exception("Échec de la préparation de la requête pour la province: " . $conn->error);
    }
    $stmt->bind_param("i", $province_id);
    $stmt->execute();
    $province = $stmt->get_result()->fetch_assoc()['name'] ?? 'Unknown';
    $stmt->close();

    $success = isset($_GET['success']) ? htmlspecialchars($_GET['success'], ENT_QUOTES, 'UTF-8') : '';
    $errors = [];

    $search_term = isset($_GET['search']) ? trim($_GET['search']) : '';
    $selected_province_id = filter_input(INPUT_GET, 'province', FILTER_VALIDATE_INT);
    if ($selected_province_id === false) {
        $selected_province_id = null;
    }

    $province_options = [];
    $province_result = $conn->query("SELECT id, name FROM province ORDER BY name");
    if ($province_result instanceof mysqli_result) {
        while ($row = $province_result->fetch_assoc()) {
            $province_options[] = $row;
        }
        $province_result->free();
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
        error_log("DEBUG: Processing POST at " . date('Y-m-d H:i:s') . ", raw_post=" . json_encode($_POST));
        $requisition_id = filter_input(INPUT_POST, 'requisition_id', FILTER_VALIDATE_INT);
        $action = trim(filter_input(INPUT_POST, 'action', FILTER_UNSAFE_RAW) ?? '');
        $comments = trim(filter_input(INPUT_POST, 'comments', FILTER_UNSAFE_RAW) ?? '');
        $approved_amount_input = filter_input(INPUT_POST, 'approved_amount', FILTER_VALIDATE_INT, ['options' => ['default' => null]]);

        if (!$requisition_id) {
            $errors[] = "ID de réquisition invalide.";
            error_log("ERROR: Invalid requisition_id=$requisition_id");
        }
        if (!in_array($action, ['approve', 'reject'])) {
            $errors[] = "Action invalide.";
            error_log("ERROR: Invalid action=$action");
        }

        if (empty($errors)) {
            $stmt = $conn->prepare("SELECT status, province_id, total_amount, original_total_amount, user_id, assigned_cashier_id FROM requisition WHERE id = ?");
            if (!$stmt) {
                throw new Exception("Échec de la préparation de la requête pour la vérification de la réquisition: " . $conn->error);
            }
            $stmt->bind_param("i", $requisition_id);
            $stmt->execute();
            $requisition = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            if (!$requisition) {
                $errors[] = "Réquisition introuvable.";
                error_log("ERROR: Requisition ID $requisition_id not found");
            } elseif ($requisition['province_id'] !== $province_id && $province_id !== null) {
                // DG and DGA can approve requisitions from any province (province_id is null for them)
                // But if they have a province_id set, they should only approve from their province
                $errors[] = "Vous ne pouvez approuver que les réquisitions de votre province.";
                error_log("ERROR: Province mismatch for requisition_id=$requisition_id, user_province=$province_id, requisition_province={$requisition['province_id']}");
            } elseif ($requisition['status'] !== 'pending') {
                $errors[] = "La réquisition n'est pas en attente d'approbation.";
                error_log("ERROR: Requisition ID $requisition_id status is {$requisition['status']}, expected pending");
            } else {
                $current_total_amount = (int)($requisition['total_amount'] ?? 0);
                $final_total_amount = $current_total_amount;
                if ($action === 'approve') {
                    if ($approved_amount_input === null || $approved_amount_input <= 0) {
                        $errors[] = "Le montant approuvé doit être une valeur positive.";
                    } else {
                        $final_total_amount = (int)$approved_amount_input;
                    }
                }

                // Validate comments are required
                if (empty($comments) || trim($comments) === '') {
                    $errors[] = "Les commentaires sont requis pour l'approbation ou le rejet.";
                }

                if (!empty($errors)) {
                    // validation failure; skip processing
                } else {
                    $conn->begin_transaction();
                    try {
                    // Always set original_total_amount if not already set (to track the initial amount)
                    if (empty($requisition['original_total_amount'])) {
                        $stmtSetOriginal = $conn->prepare("UPDATE requisition SET original_total_amount = ? WHERE id = ?");
                        if ($stmtSetOriginal) {
                            $stmtSetOriginal->bind_param("ii", $current_total_amount, $requisition_id);
                            $stmtSetOriginal->execute();
                            $stmtSetOriginal->close();
                        }
                    }
                    
                    $amount_changed = ($action === 'approve') && ($final_total_amount !== $current_total_amount);
                    if ($amount_changed) {
                        $stmtAmount = $conn->prepare("UPDATE requisition SET original_total_amount = COALESCE(original_total_amount, total_amount), total_amount = ?, last_amount_modified_at = NOW(), last_amount_modified_by = ? WHERE id = ?");
                        if (!$stmtAmount) {
                            throw new Exception("Échec de la préparation de la requête pour la mise à jour du montant: " . $conn->error);
                        }
                        $stmtAmount->bind_param("iii", $final_total_amount, $user_id, $requisition_id);
                        $stmtAmount->execute();
                        if ($stmtAmount->affected_rows === 0) {
                            throw new Exception('Échec de la mise à jour du montant de la réquisition.');
                        }
                        $stmtAmount->close();

                        $reason = $comments !== '' ? $comments : 'Amount adjusted during approval';
                        $stmtHistory = $conn->prepare("INSERT INTO requisition_amount_history (requisition_id, old_amount, new_amount, changed_by, reason) VALUES (?, ?, ?, ?, ?)");
                        if (!$stmtHistory) {
                            throw new Exception("Échec de la préparation de la requête pour l'historique du montant de la réquisition: " . $conn->error);
                        }
                        $old_amount = $current_total_amount;
                        $stmtHistory->bind_param("iiiis", $requisition_id, $old_amount, $final_total_amount, $user_id, $reason);
                        $stmtHistory->execute();
                        $stmtHistory->close();

                        if (!empty($requisition['user_id'])) {
                            $notification_stmt = $conn->prepare("INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())");
                            if ($notification_stmt) {
                                $message = sprintf(
                                    'Requisition #%d amount adjusted from %s BIF to %s BIF by %s.',
                                    $requisition_id,
                                    number_format($old_amount, 0, '.', ','),
                                    number_format($final_total_amount, 0, '.', ','),
                                    $full_name
                                );
                                $link = '/masunzu_bar_hotel/modules/expenses/expenses_history.php';
                                $notification_stmt->bind_param('iss', $requisition['user_id'], $message, $link);
                                $notification_stmt->execute();
                                $notification_stmt->close();
                            }
                        }

                        $current_total_amount = $final_total_amount;
                    }

                    $new_status = $action === 'approve' ? 'approved' : 'rejected';
                    $stmt = $conn->prepare("UPDATE requisition SET status = ?, approved_by = ?, comments = ? WHERE id = ?");
                    if (!$stmt) {
                        throw new Exception("Échec de la préparation de la requête pour la mise à jour de la réquisition: " . $conn->error);
                    }
                    $stmt->bind_param("sisi", $new_status, $user_id, $comments, $requisition_id);
                    $stmt->execute();
                    $stmt->close();

                    // Notify cashiers in the same province when requisition is approved (pending payment)
                    // Wrap in try-catch so notification failures don't prevent approval
                    if ($action === 'approve' && !empty($requisition['province_id'])) {
                        try {
                            // Get Caissier_Comptable role_id (typically 11, but we'll look it up to be safe)
                            $roleStmt = $conn->prepare("SELECT id FROM role WHERE name = 'Caissier_Comptable' LIMIT 1");
                            $cashier_role_id = null;
                            if ($roleStmt) {
                                $roleStmt->execute();
                                $roleResult = $roleStmt->get_result();
                                if ($roleRow = $roleResult->fetch_assoc()) {
                                    $cashier_role_id = (int)$roleRow['id'];
                                }
                                $roleStmt->close();
                            }
                            
                            // If we found the Caissier_Comptable role, notify all cashiers in the same province
                            if ($cashier_role_id) {
                                $cashierStmt = $conn->prepare("SELECT id FROM user WHERE role_id = ? AND province_id = ?");
                                if ($cashierStmt) {
                                    $cashierStmt->bind_param("ii", $cashier_role_id, $requisition['province_id']);
                                    $cashierStmt->execute();
                                    $cashierResult = $cashierStmt->get_result();
                                    $cashiers = [];
                                    while ($cashierRow = $cashierResult->fetch_assoc()) {
                                        $cashiers[] = (int)$cashierRow['id'];
                                    }
                                    $cashierStmt->close();
                                    
                                    // Send notification to each cashier
                                    if (!empty($cashiers)) {
                                        $message = sprintf(
                                            'New approved requisition #%d (Amount: %s BIF) is pending payment in your province.',
                                            $requisition_id,
                                            number_format($final_total_amount, 0, '.', ',')
                                        );
                                        $link = '/masunzu_bar_hotel/modules/cashiers/pay_requisitions.php';
                                        
                                        $notifyStmt = $conn->prepare("INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())");
                                        if ($notifyStmt) {
                                            foreach ($cashiers as $cashier_id) {
                                                $notifyStmt->bind_param('iss', $cashier_id, $message, $link);
                                                $notifyStmt->execute();
                                            }
                                            $notifyStmt->close();
                                            error_log("✅ Notified " . count($cashiers) . " cashier(s) about approved requisition #{$requisition_id}");
                                        }
                                    }
                                }
                            }
                            // Notify marketing roles (Chef Marketing / Marketing Agent / Marketing Head) in the same province
                            $marketingRoleNames = ['Chef Marketing', 'Marketing Agent', 'Agent Marketing', 'Marketing Head'];
                            $rolePlaceholders = implode(',', array_fill(0, count($marketingRoleNames), '?'));
                            $marketingRoleIds = [];
                            $marketingRoleStmt = $conn->prepare("SELECT id FROM role WHERE name IN ($rolePlaceholders)");
                            if ($marketingRoleStmt) {
                                mbh_bind_params($marketingRoleStmt, str_repeat('s', count($marketingRoleNames)), $marketingRoleNames);
                                $marketingRoleStmt->execute();
                                $marketingRoleResult = $marketingRoleStmt->get_result();
                                if ($marketingRoleResult) {
                                    while ($roleRow = $marketingRoleResult->fetch_assoc()) {
                                        $marketingRoleIds[] = (int)$roleRow['id'];
                                    }
                                }
                                $marketingRoleStmt->close();
                            }
                            if (!empty($marketingRoleIds)) {
                                $userPlaceholders = implode(',', array_fill(0, count($marketingRoleIds), '?'));
                                $userQuery = "SELECT id FROM user WHERE role_id IN ($userPlaceholders) AND province_id = ?";
                                $userStmt = $conn->prepare($userQuery);
                                if ($userStmt) {
                                    $userParams = array_merge($marketingRoleIds, [(int)$requisition['province_id']]);
                                    mbh_bind_params($userStmt, str_repeat('i', count($marketingRoleIds)) . 'i', $userParams);
                                    $userStmt->execute();
                                    $userResult = $userStmt->get_result();
                                    $marketingUsers = [];
                                    while ($userRow = $userResult->fetch_assoc()) {
                                        $marketingUsers[] = (int)$userRow['id'];
                                    }
                                    $userStmt->close();
                                    if (!empty($marketingUsers)) {
                                        $marketingMessage = sprintf(
                                            'Réquisition #%d approuvée dans votre province.',
                                            $requisition_id
                                        );
                                        $marketingLink = '/masunzu_bar_hotel/modules/expenses/expenses_history.php';
                                        foreach ($marketingUsers as $marketingUserId) {
                                            mbh_notify_user($conn, $marketingUserId, $marketingMessage, $marketingLink);
                                        }
                                    }
                                }
                            }
                        } catch (Exception $notifyError) {
                            // Log but don't fail the approval if notification fails
                            error_log("⚠️ Failed to notify cashiers about approved requisition #{$requisition_id}: " . $notifyError->getMessage());
                        }
            }

            if ($action === 'approve' && !empty($requisition['user_id'])) {
                $initiatorMessage = sprintf(
                    'Votre réquisition #%d a été approuvée par %s.',
                    $requisition_id,
                    $full_name
                );
                        $notifyLink = '/masunzu_bar_hotel/modules/expenses/create.php';
                        if (!mbh_notify_user($conn, (int)$requisition['user_id'], $initiatorMessage, $notifyLink)) {
                    error_log("⚠️ Failed to notify initiator user_id {$requisition['user_id']} about approval for requisition #{$requisition_id}");
                }
            }
            $assignedCashierId = (int)($requisition['assigned_cashier_id'] ?? 0);
            if ($action === 'approve' && $assignedCashierId > 0 && $assignedCashierId !== $user_id) {
                $cashierMessage = sprintf(
                    'Réquisition #%d approuvée et en attente de paiement.',
                    $requisition_id
                );
                $cashierLink = '/masunzu_bar_hotel/modules/cashiers/pay_requisitions.php';
                mbh_notify_user($conn, $assignedCashierId, $cashierMessage, $cashierLink);
            }

                    $conn->commit();
                    log_action($user_id, 'process_requisition', "Requisition ID $requisition_id $new_status by user_id $user_id, comments: $comments");
                    $action_fr = ($new_status === 'approved') ? 'approuvée' : 'rejetée';
                    header("Location: approve_expenses.php?success=" . urlencode("Réquisition $action_fr avec succès."));
                    exit;
                } catch (Exception $e) {
                    $conn->rollback();
                    $errors[] = "Échec du traitement de la réquisition: " . $e->getMessage();
                    error_log("ERROR: Failed to process requisition ID $requisition_id: " . $e->getMessage());
                }
            }
            }
        }
    }

    $query = "SELECT r.id,
                 r.total_amount,
                 r.original_total_amount,
                 r.created_at,
                 r.province_id,
                 r.assigned_cashier_id,
                 r.user_id AS initiator_id,
                 r.recipient_id,
                 r.recipient_type,
                 r.recipient_external_name,
                 r.recipient_external_phone,
                 r.last_amount_modified_at,
                 r.last_amount_modified_by,
                 u.full_name AS initiator_name,
                 recip.full_name AS recipient_name,
                 modifier.full_name AS amount_modifier_name,
                 p.name AS initiator_province,
                 GROUP_CONCAT(
                     CONCAT(
                         ri.category, ': ', ri.description,
                         ' (Qty: ', ri.quantity,
                         ', Unit Price: ', ri.unit_price, ' BIF)'
                     )
                     SEPARATOR '; '
                 ) AS items
          FROM requisition r
          LEFT JOIN user u ON r.user_id = u.id
          LEFT JOIN user recip ON r.recipient_id = recip.id
          LEFT JOIN user modifier ON r.last_amount_modified_by = modifier.id
          LEFT JOIN province p ON u.province_id = p.id
          LEFT JOIN requisition_item ri ON r.id = ri.requisition_id
          WHERE r.status = 'pending'";

    $conditions = [];
    $param_types = '';
    $params = [];

    if (!in_array($role_id, [2, 3])) {
        // Only DG (2) and DGA (3) can see all requisitions
        // Other roles (if any) should filter by province
        $conditions[] = "r.province_id = ?";
        $param_types .= 'i';
        $params[] = $province_id;
    } elseif (!empty($selected_province_id)) {
        $conditions[] = "r.province_id = ?";
        $param_types .= 'i';
        $params[] = $selected_province_id;
    }

    if ($search_term !== '') {
        $conditions[] = "(u.full_name LIKE ? OR recip.full_name LIKE ? OR p.name LIKE ? OR CAST(r.id AS CHAR) LIKE ?)";
        $param_types .= 'ssss';
        $like_term = '%' . $search_term . '%';
        $params[] = $like_term;
        $params[] = $like_term;
        $params[] = $like_term;
        $params[] = $like_term;
    }

    if (!empty($conditions)) {
        $query .= ' AND ' . implode(' AND ', $conditions);
    }

    $query .= " GROUP BY r.id
                ORDER BY r.created_at DESC";

    $stmt = $conn->prepare($query);
    if (!$stmt) {
        throw new Exception("Query preparation failed for requisitions: " . $conn->error);
    }
    if (!empty($params)) {
        $stmt->bind_param($param_types, ...$params);
    }
$stmt->execute();
$result = $stmt->get_result();
$raw_requisitions = $result ? $result->fetch_all(MYSQLI_ASSOC) : [];
$stmt->close();

$requisitions = [];
foreach ($raw_requisitions as $row) {
    $recipientType = strtolower($row['recipient_type'] ?? 'internal');
    $isExternal = $recipientType === 'external';

    $externalName = trim((string)($row['recipient_external_name'] ?? ''));
    $externalPhone = trim((string)($row['recipient_external_phone'] ?? ''));
    $internalName = trim((string)($row['recipient_name'] ?? ''));

    $recipientLabel = $isExternal ? 'External: ' : 'Internal: ';
    if ($isExternal) {
        $displayName = $externalName !== '' ? $externalName : 'Not specified';
        $displayContact = $externalPhone !== '' ? ' (' . $externalPhone . ')' : '';
        $row['recipient_display'] = $recipientLabel . $displayName . $displayContact;
    } else {
        $displayName = $internalName !== '' ? $internalName : 'Not specified';
        $row['recipient_display'] = $recipientLabel . $displayName;
    }

    $row['amount_modifier_name'] = $row['amount_modifier_name'] ?? null;
    $row['amount_modified_at_human'] = !empty($row['last_amount_modified_at'])
        ? date('Y-m-d H:i', strtotime($row['last_amount_modified_at']))
        : null;

    $requisitions[] = $row;
}

    $conn->close();

} catch (Exception $e) {
    error_log("ERROR in admin/approve_expenses.php: " . $e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
    include $footer_path ?: "</body></html>";
    exit;
}

include $header_path ?: die("Error: Header file missing. Contact administrator.");
?>
<style>
    .expenses-container {
        padding: 20px;
        background: #F4F0E4;
        min-height: calc(100vh - 70px);
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    .expenses-content {
        background-color: #fff;
        padding: 24px;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06);
    }

    .header-section {
        margin-bottom: 24px;
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        flex-wrap: wrap;
        gap: 16px;
    }

    .header-section h2 {
        color: #4B2F1F;
        margin: 0 0 8px 0;
        font-size: 28px;
        font-weight: 700;
    }

    .header-section p {
        color: #6B4F2F;
        margin: 0;
        font-size: 14px;
    }
    
    .stats-cards {
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
    }
    
    .stat-card {
        background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%);
        color: #F4F0E4;
        padding: 12px 20px;
        border-radius: 10px;
        text-align: center;
        min-width: 120px;
    }
    
    .stat-card.pending {
        background: linear-gradient(135deg, #F4A261 0%, #E8934F 100%);
        color: #000;
    }
    
    .stat-card p {
        margin: 0;
        font-size: 11px;
        text-transform: uppercase;
        letter-spacing: 0.05em;
        opacity: 0.9;
    }
    
    .stat-card h3 {
        margin: 4px 0 0;
        font-size: 20px;
        font-weight: 700;
    }

    .filter-bar {
        display: flex;
        flex-wrap: wrap;
        gap: 15px;
        align-items: flex-end;
        margin-bottom: 20px;
        background: #FDFBF7;
        padding: 16px;
        border-radius: 8px;
        border: 1px solid #F0F0F0;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
        gap: 6px;
        min-width: 200px;
    }

    .filter-group label {
        font-size: 13px;
        font-weight: 600;
        color: #4B2F1F;
        text-transform: uppercase;
    }

    .filter-group input,
    .filter-group select {
        padding: 9px 12px;
        border: 1px solid #D0C7BE;
        border-radius: 6px;
        font-size: 14px;
        background-color: #FFFFFF;
        color: #333;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
        transition: border-color 0.2s ease, box-shadow 0.2s ease;
    }

    .filter-group input:focus,
    .filter-group select:focus {
        outline: none;
        border-color: #F4A261;
        box-shadow: 0 0 0 3px rgba(244, 162, 97, 0.15);
    }

    .filter-actions {
        display: flex;
        gap: 10px;
        align-items: center;
    }

    .filter-btn {
        background: linear-gradient(135deg, #4B2F1F 0%, #5A3F2F 100%);
        color: #F4F0E4;
        border: none;
        padding: 10px 18px;
        border-radius: 6px;
        font-weight: 600;
        cursor: pointer;
        box-shadow: 0 2px 6px rgba(0, 0, 0, 0.15);
        transition: transform 0.2s ease, box-shadow 0.2s ease;
    }

    .filter-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 10px rgba(75, 47, 31, 0.25);
    }

    .filter-reset {
        text-decoration: none;
        font-weight: 600;
        color: #4B2F1F;
        padding: 10px 12px;
        border-radius: 6px;
        background-color: rgba(75, 47, 31, 0.08);
        transition: background-color 0.2s ease, color 0.2s ease;
    }

    .filter-reset:hover {
        background-color: rgba(75, 47, 31, 0.2);
        color: #2F1F14;
    }

    .success-message {
        background-color: #d4edda;
        border: 1px solid #c3e6cb;
        color: #155724;
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
    }

    .error-message {
        background-color: #f8d7da;
        border: 1px solid #f5c6cb;
        color: #721c24;
        padding: 15px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
    }

    .requisitions-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        border-radius: 8px;
        overflow: hidden;
    }

    .requisitions-table thead {
        background-color: #4B2F1F;
        color: #F4F0E4;
    }

    .requisitions-table thead th {
        padding: 14px 16px;
        text-align: left;
        font-weight: 600;
        font-size: 12px;
        text-transform: uppercase;
        letter-spacing: 0.03em;
        border: none;
    }

    .requisitions-table tbody tr {
        background-color: #FFFFFF;
        border-bottom: 1px solid #F0F0F0;
        transition: background-color 0.2s ease;
    }

    .requisitions-table tbody tr:hover {
        background-color: #FDFBF7;
    }

    .requisitions-table tbody td {
        padding: 14px 16px;
        border: none;
        color: #2C1810;
    }

    .requisitions-table .amount-cell {
        font-weight: 600;
        color: #4B2F1F;
    }

    .action-btn {
        background: linear-gradient(135deg, #F4A261 0%, #E8934F 100%);
        color: #FFFFFF;
        border: none;
        padding: 10px 20px;
        border-radius: 6px;
        cursor: pointer;
        font-weight: 600;
        font-size: 14px;
        transition: all 0.3s ease;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    }

    .action-btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(244, 162, 97, 0.4);
    }

    .action-btn:active {
        transform: translateY(0);
    }

    .no-data {
        text-align: center;
        padding: 40px 20px;
        color: #666;
        font-size: 16px;
    }

    /* Modal Styles */
    .modal {
        display: none;
        position: fixed;
        z-index: 9999;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0, 0, 0, 0.5);
        backdrop-filter: blur(4px);
        animation: fadeIn 0.3s ease;
    }

    @keyframes fadeIn {
        from {
            opacity: 0;
        }
        to {
            opacity: 1;
        }
    }

    .modal.active {
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .modal-content {
        background-color: #F4F0E4;
        padding: 0;
        border-radius: 12px;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
        width: 90%;
        max-width: 600px;
        animation: slideDown 0.3s ease;
    }

    @keyframes slideDown {
        from {
            transform: translateY(-50px);
            opacity: 0;
        }
        to {
            transform: translateY(0);
            opacity: 1;
        }
    }

    .modal-header {
        background: linear-gradient(135deg, #4B2F1F 0%, #5A3F2F 100%);
        color: #F4F0E4;
        padding: 20px;
        border-radius: 12px 12px 0 0;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .modal-header h3 {
        margin: 0;
        font-size: 22px;
        font-weight: 600;
    }

    .modal-close {
        background: none;
        border: none;
        color: #F4F0E4;
        font-size: 28px;
        cursor: pointer;
        padding: 0;
        width: 30px;
        height: 30px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
    }

    .modal-close:hover {
        color: #F4A261;
        transform: rotate(90deg);
    }

    .modal-body {
        padding: 25px;
    }

    .detail-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
        margin-bottom: 15px;
    }

    .detail-item {
        background-color: #FFFFFF;
        padding: 12px;
        border-radius: 6px;
        border-left: 4px solid #F4A261;
    }

    .detail-label {
        color: #999;
        font-size: 12px;
        text-transform: uppercase;
        font-weight: 600;
        margin-bottom: 4px;
    }

    .detail-value {
        color: #4B2F1F;
        font-size: 15px;
        font-weight: 500;
    }

    .detail-row.full {
        grid-template-columns: 1fr;
    }

    .comments-section {
        margin-top: 20px;
    }

    .comments-section label {
        display: block;
        color: #4B2F1F;
        font-weight: 600;
        margin-bottom: 8px;
        font-size: 14px;
    }

    .comments-section textarea {
        width: 100%;
        padding: 12px;
        border: 2px solid #E8E8E8;
        border-radius: 6px;
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        font-size: 14px;
        resize: vertical;
        min-height: 100px;
        transition: all 0.3s ease;
        box-sizing: border-box;
    }

    .comments-section textarea:focus {
        outline: none;
        border-color: #F4A261;
        box-shadow: 0 0 0 3px rgba(244, 162, 97, 0.1);
    }

    .modal-footer {
        padding: 20px 25px;
        border-top: 1px solid #E8E8E8;
        display: flex;
        gap: 10px;
        justify-content: flex-end;
    }

    .btn-approve, .btn-reject {
        padding: 11px 24px;
        border: none;
        border-radius: 6px;
        font-weight: 600;
        font-size: 14px;
        cursor: pointer;
        transition: all 0.3s ease;
        flex: 1;
    }

    .btn-approve {
        background: linear-gradient(135deg, #27ae60 0%, #229954 100%);
        color: white;
        box-shadow: 0 2px 8px rgba(39, 174, 96, 0.3);
    }

    .btn-approve:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 16px rgba(39, 174, 96, 0.4);
    }

    .btn-reject {
        background: linear-gradient(135deg, #e74c3c 0%, #c0392b 100%);
        color: white;
        box-shadow: 0 2px 8px rgba(231, 76, 60, 0.3);
    }

    .btn-reject:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 16px rgba(231, 76, 60, 0.4);
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .expenses-content {
            padding: 16px;
        }

        .header-section {
            flex-direction: column;
        }
        
        .header-section h2 {
            font-size: 24px;
        }
        
        .stats-cards {
            width: 100%;
        }
        
        .stat-card {
            flex: 1;
        }

        .filter-bar {
            flex-direction: column;
            align-items: stretch;
        }

        .filter-group {
            width: 100%;
        }

        .modal-content {
            width: 95%;
            max-width: 100%;
        }

        .detail-row {
            grid-template-columns: 1fr;
            gap: 12px;
        }

        .requisitions-table thead th,
        .requisitions-table tbody td {
            padding: 12px 8px;
            font-size: 13px;
        }

        .action-btn {
            padding: 8px 16px;
            font-size: 13px;
        }

        .modal-footer {
            flex-direction: column;
            gap: 8px;
        }

        .btn-approve, .btn-reject {
            flex: 1;
        }
    }

    @media (max-width: 480px) {
        .expenses-container {
            padding: 10px;
        }

        .expenses-content {
            padding: 12px;
            border-radius: 8px;
        }

        .header-section h2 {
            font-size: 20px;
        }
        
        .stat-card {
            min-width: 90px;
            padding: 10px 14px;
        }
        
        .stat-card h3 {
            font-size: 16px;
        }

        .requisitions-table {
            font-size: 12px;
        }

        .requisitions-table thead th,
        .requisitions-table tbody td {
            padding: 8px 5px;
        }

        .action-btn {
            padding: 6px 12px;
            font-size: 12px;
        }

        .modal-header {
            padding: 15px;
        }

        .modal-header h3 {
            font-size: 18px;
        }

        .modal-body {
            padding: 15px;
        }

        .modal-footer {
            padding: 15px;
        }

        .detail-item {
            padding: 8px;
        }

        .detail-label {
            font-size: 11px;
        }

        .detail-value {
            font-size: 13px;
        }
    }
</style>

<div class="expenses-container">
    <div class="expenses-content">
        <div class="header-section">
            <div>
                <h2>Approuver Réquisitions de Dépenses</h2>
                <p>Bienvenue, <strong><?php echo htmlspecialchars($full_name); ?></strong> (<?php echo htmlspecialchars($role_name); ?>)</p>
            </div>
            <div class="stats-cards">
                <div class="stat-card pending">
                    <p>En attente</p>
                    <h3><?php echo count($requisitions); ?></h3>
                </div>
                <?php
                $total_pending_amount = 0;
                foreach ($requisitions as $req) {
                    $total_pending_amount += (int)($req['total_amount'] ?? 0);
                }
                ?>
                <div class="stat-card">
                    <p>Montant Total</p>
                    <h3><?php echo number_format($total_pending_amount, 0, ',', '.'); ?></h3>
                </div>
            </div>
        </div>

        <?php if ($success): ?>
            <div class="success-message"><?php echo htmlspecialchars($success); ?></div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div class="error-message">
                <strong>Erreur:</strong><br>
                <?php foreach ($errors as $error): ?>
                    <?php echo htmlspecialchars($error) . "<br>"; ?>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <form method="GET" class="filter-bar">
            <div class="filter-group">
                <label for="search">Rechercher</label>
                <input type="text" id="search" name="search" placeholder="Rechercher par nom ou ID" value="<?php echo htmlspecialchars($search_term, ENT_QUOTES, 'UTF-8'); ?>">
            </div>
            <div class="filter-group">
                <label for="province">Province</label>
                <select id="province" name="province">
                    <option value="">Toutes les provinces</option>
                    <?php foreach ($province_options as $option): ?>
                        <option value="<?php echo (int)$option['id']; ?>" <?php echo (!empty($selected_province_id) && (int)$selected_province_id === (int)$option['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($option['name'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-actions">
                <button type="submit" class="filter-btn">Appliquer</button>
                <?php if ($search_term !== '' || !empty($selected_province_id)): ?>
                    <a class="filter-reset" href="approve_expenses.php">Réinitialiser</a>
                <?php endif; ?>
            </div>
        </form>

        <?php if (!empty($requisitions)): ?>
            <table class="requisitions-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Initiateur</th>
                        <th>Province Initiateur</th>
                        <th>Destinataire Espèces</th>
                        <th>Montant (BIF)</th>
                        <th style="text-align: center;">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($requisitions as $req): ?>
                        <tr>
                            <td><strong>#<?php echo htmlspecialchars($req['id']); ?></strong></td>
                            <td><?php echo htmlspecialchars($req['initiator_name'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($req['initiator_province'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($req['recipient_display'] ?? 'N/A'); ?></td>
                            <td class="amount-cell">
                                <?php echo number_format($req['total_amount'], 0, '.', ','); ?> BIF
                                <?php if (!empty($req['original_total_amount']) && (int)$req['original_total_amount'] !== (int)$req['total_amount']): ?>
                                    <div style="font-size: 12px; color: #C62828;">Orig: <?php echo number_format($req['original_total_amount'], 0, '.', ','); ?> BIF</div>
                                <?php endif; ?>
                            </td>
                            <td style="text-align: center;">
                                <button class="action-btn" onclick='openModal(<?php echo json_encode($req, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP); ?>)'>Examiner</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <div class="no-data">
                <p>✓ Aucune réquisition en attente disponible pour approbation.</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Modal -->
<div id="approvalModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Détails Réquisition</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <div class="modal-body">
            <div class="detail-row">
                <div class="detail-item">
                    <div class="detail-label">ID Réquisition</div>
                    <div class="detail-value" id="modal-req-id">#-</div>
                </div>
                <div class="detail-item">
                    <div class="detail-label">Montant Actuel</div>
                    <div class="detail-value" id="modal-req-amount">-</div>
                </div>
            </div>
            <div class="detail-row">
                <div class="detail-item" id="modal-original-amount-wrapper" style="display:none;">
                    <div class="detail-label">Montant Original</div>
                    <div class="detail-value" id="modal-req-original-amount">-</div>
                </div>
                <div class="detail-item">
                    <div class="detail-label">Montant Approuvé (BIF)</div>
                    <input type="number" id="modal-approved-amount" min="1" step="1" value="" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F; background-color: #F4F0E4; font-size: 16px; color: #4B2F1F;">
                </div>
            </div>
            <div class="detail-row">
                <div class="detail-item">
                    <div class="detail-label">Initiateur</div>
                    <div class="detail-value" id="modal-req-initiator">-</div>
                </div>
                <div class="detail-item">
                    <div class="detail-label">Province Initiateur</div>
                    <div class="detail-value" id="modal-req-province">-</div>
                </div>
            </div>
            <div class="detail-row">
                <div class="detail-item">
                    <div class="detail-label">Destinataire Espèces</div>
                    <div class="detail-value" id="modal-req-recipient">-</div>
                </div>
                <div class="detail-item">
                    <div class="detail-label">Date Création</div>
                    <div class="detail-value" id="modal-req-date">-</div>
                </div>
            </div>
            <div class="detail-row" id="modal-amount-review-row" style="display:none;">
                <div class="detail-item">
                    <div class="detail-label">Dernière Révision Montant</div>
                    <div class="detail-value" id="modal-amount-review-detail">-</div>
                </div>
            </div>
            <div class="detail-row full">
                <div class="detail-item">
                    <div class="detail-label">Articles</div>
                    <div class="detail-value" id="modal-req-items">-</div>
                </div>
            </div>
            <div class="comments-section">
                <label for="modal-comments">Ajouter Commentaires <span style="color: #DC3545;">*</span> (Requis)</label>
                <textarea id="modal-comments" placeholder="Entrez vos commentaires..." required></textarea>
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn-reject" onclick="submitAction('reject')">Rejeter</button>
            <button class="btn-approve" onclick="submitAction('approve')">Approuver</button>
        </div>
    </div>
</div>

<script>
    let currentRequisitionId = null;

    function openModal(requisition) {
        currentRequisitionId = requisition.id;
        document.getElementById('modal-req-id').textContent = '#' + requisition.id;
        const formatter = new Intl.NumberFormat('en-US');
        document.getElementById('modal-req-amount').textContent = formatter.format(requisition.total_amount) + ' BIF';

        const approvedAmountField = document.getElementById('modal-approved-amount');
        if (approvedAmountField) {
            approvedAmountField.value = requisition.total_amount || '';
        }

        const originalWrapper = document.getElementById('modal-original-amount-wrapper');
        const originalAmountEl = document.getElementById('modal-req-original-amount');
        if (originalWrapper && originalAmountEl) {
            if (requisition.original_total_amount && Number(requisition.original_total_amount) !== Number(requisition.total_amount)) {
                originalWrapper.style.display = '';
                originalAmountEl.textContent = formatter.format(requisition.original_total_amount) + ' BIF';
            } else {
                originalWrapper.style.display = 'none';
            }
        }
        document.getElementById('modal-req-initiator').textContent = requisition.initiator_name || 'N/A';
        document.getElementById('modal-req-province').textContent = requisition.initiator_province || 'N/A';
        document.getElementById('modal-req-recipient').textContent = requisition.recipient_display || requisition.recipient_name || 'N/A';
        const amountReviewRow = document.getElementById('modal-amount-review-row');
        const amountReviewDetail = document.getElementById('modal-amount-review-detail');
        if (amountReviewRow && amountReviewDetail) {
            if (requisition.amount_modified_at_human) {
                const reviewer = requisition.amount_modifier_name ? ` by ${requisition.amount_modifier_name}` : '';
                amountReviewDetail.textContent = `${requisition.amount_modified_at_human}${reviewer}`;
                amountReviewRow.style.display = '';
            } else {
                amountReviewRow.style.display = 'none';
            }
        }

        const itemsContainer = document.getElementById('modal-req-items');
        itemsContainer.innerHTML = '';
        if (requisition.items) {
            const items = requisition.items.split(';').map(item => item.trim()).filter(item => item.length > 0);
            if (items.length > 0) {
                const list = document.createElement('ul');
                list.style.margin = '0';
                list.style.paddingLeft = '18px';
                items.forEach(item => {
                    const li = document.createElement('li');
                    li.textContent = item;
                    list.appendChild(li);
                });
                itemsContainer.appendChild(list);
            } else {
                itemsContainer.textContent = 'N/A';
            }
        } else {
            itemsContainer.textContent = 'N/A';
        }
        
        const date = new Date(requisition.created_at);
        document.getElementById('modal-req-date').textContent = date.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric', hour: '2-digit', minute: '2-digit' });
        
        document.getElementById('modal-comments').value = '';
        document.getElementById('approvalModal').classList.add('active');
        document.body.style.overflow = 'hidden';
    }

    function closeModal() {
        document.getElementById('approvalModal').classList.remove('active');
        document.body.style.overflow = 'auto';
        currentRequisitionId = null;
    }

    function submitAction(action) {
        const comments = document.getElementById('modal-comments').value.trim();
        const approvedAmountField = document.getElementById('modal-approved-amount');
        let approvedAmountValue = approvedAmountField ? approvedAmountField.value.trim() : '';

        // Validate comments are required
        if (!comments || comments.length === 0) {
            alert('Les commentaires sont requis. Veuillez entrer vos commentaires avant de soumettre.');
            document.getElementById('modal-comments').focus();
            return;
        }

        if (action === 'approve') {
            const parsedAmount = parseInt(approvedAmountValue, 10);
            if (!Number.isFinite(parsedAmount) || parsedAmount <= 0) {
                alert('Veuillez entrer un montant approuvé valide supérieur à zéro.');
                return;
            }
            approvedAmountValue = parsedAmount;
        }

        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = `
            <input type="hidden" name="requisition_id" value="${currentRequisitionId}">
            <input type="hidden" name="action" value="${action}">
            <input type="hidden" name="comments" value="${comments.replace(/"/g, '&quot;')}">
            <input type="hidden" name="approved_amount" value="${approvedAmountValue}">
        `;
        document.body.appendChild(form);
        form.submit();
    }

    // Close modal when clicking outside
    document.getElementById('approvalModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });

    // Close modal with Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeModal();
        }
    });
</script>

<?php include $footer_path ?: "</body></html>"; ?>
