<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/cash_helpers.php';
require_once __DIR__ . '/../../config/database.php';

require_login();

if (($_SESSION['role_name'] ?? '') !== 'Accountant') {
    $targetDashboard = strtolower(str_replace(' ', '_', $_SESSION['role_name'] ?? '')) . '_dashboard.php';
    header('Location: /masunzu_bar_hotel/dashboards/' . $targetDashboard);
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    error_log('receive_cash.php failed to connect to database: ' . mysqli_connect_error());
    include __DIR__ . '/../../includes/header.php';
    echo "<div class='alert alert-danger m-3'>Échec de la connexion à la base de données. Veuillez contacter l'administrateur.</div>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

if (!isset($_SESSION['province_id']) || (int)$_SESSION['province_id'] <= 0) {
    $stmtProvince = $conn->prepare('SELECT province_id FROM user WHERE id = ?');
    if ($stmtProvince) {
        $stmtProvince->bind_param('i', $_SESSION['user_id']);
        $stmtProvince->execute();
        $provinceRow = $stmtProvince->get_result()->fetch_assoc();
        $stmtProvince->close();
        $_SESSION['province_id'] = (int)($provinceRow['province_id'] ?? 0);
    }
}

$schemaErrors = [];

$createNotificationSql = "
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read),
        CONSTRAINT fk_notification_user FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createNotificationSql)) {
    $schemaErrors[] = 'Impossible d\'initialiser le suivi des notifications. Contactez l\'administrateur.';
    error_log('Failed to ensure user_notification table: ' . $conn->error);
}

$success = $_SESSION['receive_cash_success'] ?? '';
$error = $_SESSION['receive_cash_error'] ?? '';
unset($_SESSION['receive_cash_success'], $_SESSION['receive_cash_error']);

if ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($schemaErrors)) {
    $province_id = isset($_POST['province_id']) ? (int)$_POST['province_id'] : (int)($_SESSION['province_id'] ?? 0);
    $user_id = (int)($_SESSION['user_id'] ?? 0);
    $transfer_id = (int)($_POST['transfer_id'] ?? 0);
    $postedAmount = isset($_POST['amount']) ? (int)round((float)$_POST['amount']) : 0;
    $receivedNotes = trim($_POST['received_notes'] ?? '');

    if ($user_id <= 0) {
        $_SESSION['receive_cash_error'] = 'Session invalide. Veuillez vous reconnecter.';
        header('Location: receive_cash.php');
        exit;
    }

    $conn->begin_transaction();
    try {
        $stmt = $conn->prepare('
            SELECT ct.amount, ct.cashier_id, u.province_id
            FROM cash_transfer ct
            INNER JOIN user u ON ct.cashier_id = u.id
            WHERE ct.id = ? AND ct.accountant_id = ? AND ct.status = \'pending\'
            FOR UPDATE
        ');
        if (!$stmt) {
        throw new Exception('Impossible de vérifier le transfert sélectionné: ' . $conn->error);
    }
        $stmt->bind_param('ii', $transfer_id, $user_id);
        $stmt->execute();
        $transferRow = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        if (!$transferRow) {
            throw new Exception('Le transfert sélectionné n\'est plus en attente ou n\'existe pas.');
        }

        $expectedAmount = (int)$transferRow['amount'];
        $cashierId = (int)$transferRow['cashier_id'];
        $cashierProvinceId = (int)($transferRow['province_id'] ?? 0);
        if ($expectedAmount <= 0) {
            throw new Exception('Le montant du transfert est invalide.');
        }

        if ($postedAmount !== $expectedAmount) {
            throw new Exception('Le montant du transfert ne peut pas être modifié manuellement.');
        }

        $stmt = $conn->prepare('UPDATE cash_transfer SET status = \'received\', received_at = NOW() WHERE id = ?');
        if (!$stmt) {
            throw new Exception('Échec de la mise à jour du statut du transfert: ' . $conn->error);
        }
        $stmt->bind_param('i', $transfer_id);
        if (!$stmt->execute()) {
            throw new Exception('Échec de la mise à jour du statut du transfert: ' . $stmt->error);
        }
        $stmt->close();

        unset($receivedNotes);

        $provinceForAdjustment = $province_id > 0 ? $province_id : $cashierProvinceId;
        if ($provinceForAdjustment <= 0) {
            throw new Exception('Impossible de déterminer la province associée à ce transfert.');
        }

        adjustProvinceCash(
            $conn,
            $provinceForAdjustment,
            $expectedAmount,
            'CASH_RECEIVED',
            'OTHER',
            $transfer_id,
            $user_id,
            'Cash transfer #' . $transfer_id . ' received.'
        );

        // Update accountant_balance table
        $stmtCheckBalance = $conn->prepare('SELECT id FROM accountant_balance WHERE accountant_id = ?');
        if (!$stmtCheckBalance) {
            throw new Exception('Failed to check accountant balance: ' . $conn->error);
        }
        $stmtCheckBalance->bind_param('i', $user_id);
        $stmtCheckBalance->execute();
        $balanceExists = $stmtCheckBalance->get_result()->fetch_assoc();
        $stmtCheckBalance->close();

        if ($balanceExists) {
            // Update existing balance
            $stmtUpdateBalance = $conn->prepare('UPDATE accountant_balance SET balance = balance + ?, last_updated = NOW() WHERE accountant_id = ?');
            if (!$stmtUpdateBalance) {
                throw new Exception('Failed to prepare accountant balance update: ' . $conn->error);
            }
            $stmtUpdateBalance->bind_param('di', $expectedAmount, $user_id);
            if (!$stmtUpdateBalance->execute()) {
                throw new Exception('Failed to update accountant balance: ' . $stmtUpdateBalance->error);
            }
            $stmtUpdateBalance->close();
        } else {
            // Insert new balance record
            $stmtInsertBalance = $conn->prepare('INSERT INTO accountant_balance (accountant_id, balance, currency, last_updated) VALUES (?, ?, ?, NOW())');
            if (!$stmtInsertBalance) {
                throw new Exception('Échec de la préparation de l\'insertion du solde comptable: ' . $conn->error);
            }
            $currency = 'BIF';
            $stmtInsertBalance->bind_param('ids', $user_id, $expectedAmount, $currency);
            if (!$stmtInsertBalance->execute()) {
                throw new Exception('Échec de l\'insertion du solde comptable: ' . $stmtInsertBalance->error);
            }
            $stmtInsertBalance->close();
        }

        if (function_exists('log_action')) {
            log_action($user_id, 'receive_cash', "Confirmed transfer #{$transfer_id} for {$expectedAmount} BIF");
        }

        $notificationMessage = sprintf(
            'Transfert d\'espèces #%d reçu par le comptable. Montant: %s BIF.',
            $transfer_id,
            number_format($expectedAmount, 0, ',', '.')
        );
        mbh_notify_user($conn, $cashierId, $notificationMessage, '/masunzu_bar_hotel/modules/cashiers/transfer_cash.php');

        $conn->commit();
        $_SESSION['receive_cash_success'] = 'Espèces reçues et enregistrées avec succès.';
        header('Location: receive_cash.php');
        exit;
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['receive_cash_error'] = $e->getMessage();
        header('Location: receive_cash.php');
        exit;
    }
}

$pendingTransfers = [];
if (empty($schemaErrors)) {
    $stmtPending = $conn->prepare("
        SELECT ct.id, ct.amount, ct.transfer_date, ct.cashier_id, u.full_name AS cashier_name, u.province_id
        FROM cash_transfer ct
        INNER JOIN user u ON ct.cashier_id = u.id
        WHERE ct.accountant_id = ? AND ct.status = 'pending'
        ORDER BY ct.transfer_date DESC
    ");
    if ($stmtPending) {
        $stmtPending->bind_param('i', $_SESSION['user_id']);
        $stmtPending->execute();
        $pendingTransfers = $stmtPending->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtPending->close();
    } else {
        error_log('Failed to load pending transfers in receive_cash.php: ' . $conn->error);
    }
}

$conn->close();

include __DIR__ . '/../../includes/header.php';
?>
<style>
:root {
    --primary-brown: #4B2F1F;
    --accent-orange: #F4A261;
    --cream-bg: #F4F0E4;
    --text-black: #000000;
}

body {
    color: var(--text-black);
    background: var(--cream-bg);
    font-family: system-ui, -apple-system, sans-serif;
    margin: 0;
    padding: 1rem;
}

.receive-cash-container {
    max-width: 800px;
    margin: 0 auto;
}

.page-header {
    text-align: center;
    margin-bottom: 2rem;
    padding: 1rem 0;
}

.page-title {
    color: var(--primary-brown);
    font-weight: 600;
    font-size: 1.75rem;
    margin-bottom: 0.5rem;
}

.page-subtitle {
    color: var(--text-black);
    opacity: 0.7;
    font-size: 0.95rem;
}

.cash-form-card {
    background: var(--cream-bg);
    border: 1px solid var(--primary-brown);
    border-radius: 8px;
    padding: 1.5rem;
    margin-bottom: 1rem;
}

.form-header {
    background: var(--primary-brown);
    color: var(--cream-bg);
    padding: 1rem;
    margin: -1.5rem -1.5rem 1.5rem -1.5rem;
    border-bottom: 1px solid var(--primary-brown);
    border-radius: 8px 8px 0 0;
}

.form-title {
    font-size: 1.25rem;
    font-weight: 600;
    margin: 0;
}

.transfer-selector {
    margin-bottom: 1.5rem;
}

.selector-label {
    font-weight: 500;
    color: var(--text-black);
    margin-bottom: 0.5rem;
    display: block;
}

.transfer-select {
    width: 100%;
    padding: 0.5rem;
    border: 1px solid var(--primary-brown);
    border-radius: 4px;
    background: var(--cream-bg);
    color: var(--text-black);
    font-size: 0.95rem;
}

.transfer-select:focus {
    outline: 2px solid var(--accent-orange);
    outline-offset: 1px;
}

.transfer-info-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

@media (max-width: 640px) {
    .transfer-info-grid {
        grid-template-columns: 1fr;
    }
}

.info-card {
    background: var(--cream-bg);
    border: 1px solid var(--primary-brown);
    border-radius: 6px;
    padding: 0.75rem;
}

.info-label {
    font-size: 0.8rem;
    font-weight: 500;
    color: var(--text-black);
    opacity: 0.8;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 0.25rem;
}

.info-value {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--primary-brown);
}

.denomination-section {
    margin-top: 1.5rem;
    padding-top: 1.5rem;
    border-top: 1px solid var(--primary-brown);
}

.section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--text-black);
    margin-bottom: 1rem;
}


.alert {
    border-radius: 6px;
    border: 1px solid var(--primary-brown);
    padding: 0.75rem;
    margin-bottom: 1rem;
    font-weight: 500;
    background: var(--cream-bg);
    color: var(--text-black);
}

.alert-success {
    border-color: #28a745;
}

.alert-danger {
    border-color: #dc3545;
}

.alert-info {
    border-color: #17a2b8;
}

.empty-state {
    text-align: center;
    padding: 2rem 1rem;
    color: var(--text-black);
    opacity: 0.7;
}

.empty-state h4 {
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.empty-state p {
    font-size: 1rem;
}

/* Animation for denomination count changes */
@keyframes countChange {
    0% {
        background-color: #fef7e6;
        border-color: var(--accent-orange);
        transform: scale(1.05);
    }
    100% {
        background-color: var(--cream-bg);
        border-color: var(--primary-brown);
        transform: scale(1);
    }
}

.denomination-input.count-changed {
    animation: countChange 0.4s ease;
}
</style>

<div class="receive-cash-container">
    <div class="page-header">
        <h1 class="page-title">Recevoir Espèces du Caissier</h1>
        <p class="page-subtitle">Confirmer et enregistrer les transferts d'espèces avec répartition des dénominations</p>
    </div>
    <?php if (!empty($schemaErrors)): ?>
        <div class="alert alert-danger">
            <i class="bi bi-exclamation-triangle-fill me-2"></i>
            <?php echo htmlspecialchars(implode(' ', $schemaErrors), ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <?php if ($success): ?>
        <div class="alert alert-success">
            <i class="bi bi-check-circle-fill me-2"></i>
            <?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">
            <i class="bi bi-x-circle-fill me-2"></i>
            <?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php endif; ?>

    <?php if (empty($schemaErrors) && empty($pendingTransfers)): ?>
        <div class="empty-state">
            <h4>Aucun Transfert en Attente</h4>
            <p>Il n'y a actuellement aucun transfert d'espèces en attente de votre confirmation.</p>
        </div>
    <?php elseif (empty($schemaErrors)): ?>
        <form method="POST" class="cash-form-card" id="receive-cash-form">
            <div class="form-header">
                <h2 class="form-title">Détails Transfert d'Espèces</h2>
            </div>

            <div class="transfer-selector">
                <label class="selector-label">Sélectionner Transfert en Attente</label>
                <select name="transfer_id" id="transfer_id" class="form-select transfer-select" required>
                    <option value="">Choisir un transfert à traiter</option>
                    <?php foreach ($pendingTransfers as $transfer): ?>
                        <option
                            value="<?php echo (int)$transfer['id']; ?>"
                            data-amount="<?php echo (int)$transfer['amount']; ?>"
                            data-cashier-name="<?php echo htmlspecialchars($transfer['cashier_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?>"
                            data-transfer-date="<?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($transfer['transfer_date'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?>"
                            data-province-id="<?php echo isset($transfer['province_id']) ? (int)$transfer['province_id'] : ''; ?>"
                        >
                            <?php
                                echo sprintf(
                                    '#%d · %s · %s · %s BIF',
                                    $transfer['id'],
                                    date('Y-m-d H:i', strtotime($transfer['transfer_date'] ?? 'now')),
                                    htmlspecialchars($transfer['cashier_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'),
                                    number_format((float)$transfer['amount'], 0, ',', '.')
                                );
                            ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="transfer-info-grid">
                <div class="info-card">
                    <div class="info-label">Montant à Recevoir</div>
                    <div class="info-value" id="amount_display">--</div>
                    <input type="hidden" name="amount" id="amount" value="">
                    <input type="hidden" name="province_id" id="province_id_field" value="<?php echo (int)($_SESSION['province_id'] ?? 0); ?>">
                </div>
                <div class="info-card">
                    <div class="info-label">Du Caissier</div>
                    <div class="info-value" id="cashier_display">--</div>
                </div>
            </div>

            <div class="mb-3">
                <label for="received_notes" class="form-label">Notes / Observations (optionnel)</label>
                <textarea
                    class="form-control"
                    id="received_notes"
                    name="received_notes"
                    rows="3"
                    placeholder="Ajouter des commentaires sur la réception si nécessaire"
                ></textarea>
            </div>

            <div class="submit-section">
                <button type="submit" class="submit-btn">Confirmer Réception</button>
            </div>
        </form>
        <script>
        document.addEventListener('DOMContentLoaded', function () {
            var transferSelect = document.getElementById('transfer_id');
            var amountInput = document.getElementById('amount');
            var amountDisplay = document.getElementById('amount_display');
            var cashierDisplay = document.getElementById('cashier_display');
            var provinceField = document.getElementById('province_id_field');

            if (!transferSelect) {
                return;
            }

            function formatAmount(value) {
                return Number(value || 0).toLocaleString('fr-FR', { minimumFractionDigits: 0 });
            }

            transferSelect.addEventListener('change', function () {
                var option = this.options[this.selectedIndex];
                if (!option || !option.value) {
                    amountInput.value = '';
                    amountDisplay.textContent = '--';
                    cashierDisplay.textContent = '--';
                    if (provinceField) {
                        provinceField.value = '';
                    }
                    return;
                }

                var amount = Number(option.dataset.amount || 0);
                var cashierName = option.dataset.cashierName || '';
                var transferDate = option.dataset.transferDate || '';
                var provinceId = option.dataset.provinceId || '';

                amountInput.value = amount > 0 ? String(amount) : '';
                amountDisplay.textContent = amount > 0 ? formatAmount(amount) + ' BIF' : '--';
                cashierDisplay.textContent = cashierName
                    ? cashierName + (transferDate ? ' · ' + transferDate : '')
                    : '--';
                if (provinceField && provinceId) {
                    provinceField.value = provinceId;
                }
            });
        });
        </script>
    <?php endif; ?>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
