<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

// Restrict to Accountant, Admin, DG, DGA, Board roles
$allowed_roles = ['Accountant', 'Admin', 'DG', 'DGA', 'Board'];
if (!in_array($_SESSION['role_name'], $allowed_roles, true)) {
    error_log("Unauthorized access to loan_management.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $_SESSION['role_name'])) . "_dashboard.php");
    exit;
}

$is_admin = in_array($_SESSION['role_name'], ['Admin', 'DG', 'DGA', 'Board'], true);

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in loan_management.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Get province_id (only for accountants, admins see all)
$province_id = null;
if (!$is_admin) {
    $province_id = $_SESSION['province_id'] ?? null;
    if (!$province_id) {
        $stmtProvince = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if ($stmtProvince) {
            $stmtProvince->bind_param("i", $_SESSION['user_id']);
            $stmtProvince->execute();
            $provinceRow = $stmtProvince->get_result()->fetch_assoc();
            $_SESSION['province_id'] = $provinceRow['province_id'] ?? null;
            $province_id = $_SESSION['province_id'];
            $stmtProvince->close();
        }
    }
}

// Handle filters
$filter_unpaid = isset($_GET['filter_unpaid']) ? $_GET['filter_unpaid'] : '';
$filter_late = isset($_GET['filter_late']) ? $_GET['filter_late'] : '';
$filter_status = isset($_GET['filter_status']) ? $_GET['filter_status'] : '';
$search_term = isset($_GET['search']) ? trim($_GET['search']) : '';

// Build WHERE clause for customer query
$where_conditions = ["(c.can_pay_by_loan = 1 OR c.loan_balance_bif > 0)"];
$params = [];
$param_types = '';

// Province filter fragment reused in subqueries
$province_filter_sql = $province_id ? " AND ui.province_id = " . (int)$province_id : '';

// Fetch all customers with loans or loan eligibility with detailed loan information
// Use customer_loan_ledger to calculate outstanding balance (source of truth)
// loan_balance_bif should match the ledger balance, but we calculate from ledger for accuracy
$customers_query = "
    SELECT 
        c.id,
        c.full_name,
        c.nif,
        c.tel,
        c.loan_limit_bif,
        c.loan_balance_bif,
        c.loan_terms_days,
        c.loan_status,
        c.can_pay_by_loan,
        (c.loan_limit_bif - c.loan_balance_bif) AS remaining_limit,
        COALESCE(GREATEST(
            COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
            COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
            0
        ), c.loan_balance_bif, 0) AS ledger_outstanding,
        (
            SELECT COALESCE(SUM(ui.loan_outstanding_bif), 0)
            FROM unpaid_invoices ui
            WHERE ui.is_loan_sale = 1
              AND ui.loan_outstanding_bif > 0
              AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
              AND (ui.approval_status != 'rejected')
              {$province_filter_sql}
              AND (
                ui.customer_id = c.id
                OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel)
              )
        ) AS total_unpaid,
        (
            SELECT COUNT(*)
            FROM unpaid_invoices ui
            WHERE ui.is_loan_sale = 1
              AND ui.loan_outstanding_bif > 0
              AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
              AND (ui.approval_status != 'rejected')
              {$province_filter_sql}
              AND (
                ui.customer_id = c.id
                OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel)
              )
        ) AS unpaid_invoice_count,
        (
            SELECT MIN(ui.loan_due_date)
            FROM unpaid_invoices ui
            WHERE ui.is_loan_sale = 1
              AND ui.loan_outstanding_bif > 0
              AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
              AND (ui.approval_status != 'rejected')
              AND ui.loan_due_date IS NOT NULL
              {$province_filter_sql}
              AND (
                ui.customer_id = c.id
                OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel)
              )
        ) AS earliest_due_date,
        (
            SELECT COUNT(*)
            FROM unpaid_invoices ui
            WHERE ui.is_loan_sale = 1
              AND ui.loan_outstanding_bif > 0
              AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
              AND (ui.approval_status != 'rejected')
              AND ui.loan_due_date IS NOT NULL
              AND ui.loan_due_date < CURDATE()
              {$province_filter_sql}
              AND (
                ui.customer_id = c.id
                OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel)
              )
        ) AS late_invoice_count
    FROM customer c
    LEFT JOIN customer_loan_ledger cll ON (
        cll.customer_id = c.id
        AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
    )
    WHERE " . implode(' AND ', $where_conditions) . "
    GROUP BY c.id, c.full_name, c.nif, c.tel, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status, c.can_pay_by_loan
    ORDER BY c.full_name
";

$all_customers = [];
$stmtCustomers = $conn->prepare($customers_query);
if ($stmtCustomers) {
    if (!empty($params)) {
        $stmtCustomers->bind_param($param_types, ...$params);
    }
    $stmtCustomers->execute();
    $all_customers = $stmtCustomers->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtCustomers->close();
}

// Calculate dashboard metrics
$total_customers_eligible = 0;
$total_unpaid_amount = 0;
$total_outstanding_balance = 0;
$customers_with_unpaid = 0;
$customers_late = 0;
$total_late_amount = 0;

foreach ($all_customers as $customer) {
    $total_customers_eligible++;
    // total_unpaid is the sum of unpaid invoices from unpaid_invoices table
    $unpaid = (int)($customer['total_unpaid'] ?? 0);
    // Use ledger_outstanding (calculated from customer_loan_ledger) as source of truth, fallback to loan_balance_bif
    $ledger_outstanding = (int)($customer['ledger_outstanding'] ?? 0);
    $outstanding = $ledger_outstanding > 0 ? $ledger_outstanding : (int)($customer['loan_balance_bif'] ?? 0);
    $late_count = (int)($customer['late_invoice_count'] ?? 0);
    
    // Use ledger_outstanding for total outstanding balance (source of truth from customer_loan_ledger)
    $total_unpaid_amount += $unpaid; // Sum of unpaid invoices
    $total_outstanding_balance += $outstanding; // Total loan balance from ledger or customer table
    
    // Check if customer has outstanding balance
    if ($outstanding > 0) {
        $customers_with_unpaid++;
    }
    
    if ($late_count > 0) {
        $customers_late++;
        $total_late_amount += $unpaid; // Approximate late amount from unpaid invoices
    }
}

// Apply filters to customers
$filtered_customers = [];
foreach ($all_customers as $customer) {
    // Use ledger_outstanding (from customer_loan_ledger) as source of truth, fallback to loan_balance_bif
    $ledger_outstanding = (int)($customer['ledger_outstanding'] ?? 0);
    $outstanding = $ledger_outstanding > 0 ? $ledger_outstanding : (int)($customer['loan_balance_bif'] ?? 0);
    $unpaid = (int)($customer['total_unpaid'] ?? 0);
    $late_count = (int)($customer['late_invoice_count'] ?? 0);
    $loan_status = strtolower($customer['loan_status'] ?? 'inactive');
    $name = strtolower($customer['full_name'] ?? '');
    
    // Apply filters - use outstanding (from ledger) for unpaid filter
    if ($filter_unpaid === '1' && $outstanding <= 0) continue;
    if ($filter_unpaid === '0' && $outstanding > 0) continue;
    if ($filter_late === '1' && $late_count <= 0) continue;
    if ($filter_late === '0' && $late_count > 0) continue;
    if ($filter_status && $loan_status !== strtolower($filter_status)) continue;
    if ($search_term && strpos($name, strtolower($search_term)) === false) continue;
    
    $filtered_customers[] = $customer;
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="loan_management_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF)); // UTF-8 BOM
    
    fputcsv($output, [
        'Nom Client', 'NIF', 'Tél', 'Limite Crédit (BIF)', 'Solde Impayé (BIF)', 
        'Nombre Commandes Impayées', 'Nombre Commandes Retard', 
        'Date Échéance Prochaine', 'Statut Crédit', 'Limite Restante (BIF)'
    ]);
    
    foreach ($filtered_customers as $customer) {
        fputcsv($output, [
            $customer['full_name'] ?? '',
            $customer['nif'] ?? '',
            $customer['tel'] ?? '',
            $customer['loan_limit_bif'] ?? 0,
            $customer['loan_balance_bif'] ?? 0,
            $customer['unpaid_invoice_count'] ?? 0,
            $customer['late_invoice_count'] ?? 0,
            $customer['earliest_due_date'] ?? 'N/A',
            (($customer['loan_status'] ?? 'inactive') === 'active' ? 'Actif' : (($customer['loan_status'] ?? 'inactive') === 'inactive' ? 'Inactif' : (($customer['loan_status'] ?? 'inactive') === 'suspended' ? 'Suspendu' : ucfirst($customer['loan_status'] ?? 'inactive')))),
            $customer['remaining_limit'] ?? 0
        ]);
    }
    
    fclose($output);
    $conn->close();
    exit;
}

// Get selected customer for detailed view
$selected_customer_id = isset($_GET['customer_id']) ? (int)$_GET['customer_id'] : 0;
$customer_profile = null;
$loan_invoices = [];
$loan_payments = [];
$loan_ledger = [];

if ($selected_customer_id > 0) {
    // Calculate ledger_outstanding as source of truth
    $stmtProfile = $conn->prepare("
        SELECT 
            c.id, 
            c.full_name, 
            c.loan_limit_bif, 
            c.loan_balance_bif, 
            c.loan_terms_days, 
            c.loan_status, 
            c.can_pay_by_loan,
            COALESCE(GREATEST(
                COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
                COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
                0
            ), c.loan_balance_bif, 0) AS ledger_outstanding
        FROM customer c
        LEFT JOIN customer_loan_ledger cll ON (
            cll.customer_id = c.id
            AND cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
        )
        WHERE c.id = ?
        GROUP BY c.id, c.full_name, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status, c.can_pay_by_loan
        LIMIT 1
    ");
    if ($stmtProfile) {
        $stmtProfile->bind_param("i", $selected_customer_id);
        $stmtProfile->execute();
        $customer_profile = $stmtProfile->get_result()->fetch_assoc();
        $stmtProfile->close();
        
        // Use ledger_outstanding as source of truth
        if ($customer_profile) {
            $ledger_outstanding = (int)($customer_profile['ledger_outstanding'] ?? 0);
            $loan_balance = (int)($customer_profile['loan_balance_bif'] ?? 0);
            $customer_profile['loan_balance_bif'] = $ledger_outstanding > 0 ? $ledger_outstanding : $loan_balance;
        }
    }
    
    if ($customer_profile) {
        $customer_name = $customer_profile['full_name'] ?? '';
        $customer_tel = '';
        
        $stmtTel = $conn->prepare("SELECT tel FROM customer WHERE id = ?");
        if ($stmtTel) {
            $stmtTel->bind_param("i", $selected_customer_id);
            $stmtTel->execute();
            $telRow = $stmtTel->get_result()->fetch_assoc();
            $customer_tel = $telRow['tel'] ?? '';
            $stmtTel->close();
        }
        
        $stmtInvoices = $conn->prepare("
            SELECT 
                ui.id,
                ui.invoice_number,
                ui.created_at,
                ui.subtotal_amount,
                ui.loan_outstanding_bif,
                ui.loan_due_date,
                ui.loan_status,
                ui.status,
                ui.province_id,
                p.name AS province_name
            FROM unpaid_invoices ui
            LEFT JOIN province p ON ui.province_id = p.id
            WHERE ui.is_loan_sale = 1 
            AND ui.loan_outstanding_bif > 0
            AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
            AND (ui.approval_status != 'rejected')
            AND (ui.customer_id = ? OR (ui.customer_id IS NULL AND ui.customer_name = ? AND ui.phone_number = ?))
            " . ($province_id ? " AND ui.province_id = " . (int)$province_id : "") . "
            ORDER BY ui.created_at DESC
        ");
        if ($stmtInvoices) {
            $stmtInvoices->bind_param("iss", $selected_customer_id, $customer_name, $customer_tel);
            $stmtInvoices->execute();
            $loan_invoices = $stmtInvoices->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtInvoices->close();
        }
        
        $stmtPayments = $conn->prepare("
            SELECT 
                lp.id,
                lp.payment_amount,
                lp.payment_method,
                lp.payment_date,
                lp.invoice_id,
                ui.invoice_number,
                u.full_name AS created_by_name
            FROM loan_payment lp
            LEFT JOIN unpaid_invoices ui ON lp.invoice_id = ui.id
            LEFT JOIN user u ON lp.created_by = u.id
            WHERE lp.customer_id = ?
            ORDER BY lp.payment_date DESC
        ");
        if ($stmtPayments) {
            $stmtPayments->bind_param("i", $selected_customer_id);
            $stmtPayments->execute();
            $loan_payments = $stmtPayments->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtPayments->close();
        }
        
        $stmtLedger = $conn->prepare("
            SELECT 
                cll.id,
                cll.entry_type,
                cll.direction,
                cll.amount_bif,
                cll.reference_table,
                cll.reference_id,
                cll.notes,
                cll.created_at,
                cll.created_by,
                u.full_name AS created_by_name
            FROM customer_loan_ledger cll
            LEFT JOIN user u ON cll.created_by = u.id
            WHERE cll.customer_id = ?
            ORDER BY cll.created_at DESC
        ");
        if ($stmtLedger) {
            $stmtLedger->bind_param("i", $selected_customer_id);
            $stmtLedger->execute();
            $loan_ledger = $stmtLedger->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtLedger->close();
        }
    }
}

$total_paid = 0;
if (!empty($loan_payments)) {
    foreach ($loan_payments as $payment) {
        $total_paid += (int)($payment['payment_amount'] ?? 0);
    }
}

$conn->close();
include __DIR__ . '/../../includes/header.php';
?>
<style>
    .metric-card {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border-radius: 12px;
        padding: 20px;
        color: white;
        box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        transition: transform 0.2s;
    }
    .metric-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 6px 12px rgba(0,0,0,0.15);
    }
    .metric-card.danger {
        background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    }
    .metric-card.warning {
        background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
    }
    .metric-card.success {
        background: linear-gradient(135deg, #30cfd0 0%, #330867 100%);
    }
    .metric-card.info {
        background: linear-gradient(135deg, #a8edea 0%, #fed6e3 100%);
        color: #333;
    }
    .metric-value {
        font-size: 32px;
        font-weight: 700;
        margin: 10px 0;
    }
    .metric-label {
        font-size: 14px;
        opacity: 0.9;
        text-transform: uppercase;
        letter-spacing: 1px;
    }
    .late-badge {
        background-color: #dc3545;
        color: white;
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 11px;
        font-weight: 600;
        display: inline-block;
    }
    .unpaid-badge {
        background-color: #ffc107;
        color: #333;
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 11px;
        font-weight: 600;
        display: inline-block;
    }
    .customer-row:hover {
        background-color: #f8f9fa;
    }
    .modal-backdrop {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        display: none;
        align-items: center;
        justify-content: center;
        z-index: 1000;
    }
    .modal-content {
        background: #FFFFFF;
        border-radius: 10px;
        padding: 20px;
        width: 95%;
        max-width: 900px;
        max-height: 90vh;
        overflow: auto;
        box-shadow: 0 10px 30px rgba(0,0,0,0.25);
        position: relative;
    }
    .modal-close {
        position: absolute;
        top: 10px;
        right: 12px;
        background: #4B2F1F;
        color: #fff;
        border: none;
        border-radius: 4px;
        padding: 6px 10px;
        cursor: pointer;
    }
    .pill {
        display: inline-block;
        padding: 6px 10px;
        border-radius: 999px;
        font-size: 12px;
        font-weight: 700;
        background: #f0f0f0;
        color: #4B2F1F;
        margin-right: 8px;
    }
</style>
<div class="loan-management-container" style="padding: 20px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 70vh;">
    <div class="loan-management-content" style="background-color: #FFFFFF; padding: 30px; border-radius: 12px; box-shadow: 0 8px 16px rgba(0,0,0,0.2); max-width: 100%; margin: 0;">
        <h2 style="color: #4B2F1F; font-size: 32px; margin-bottom: 10px; font-weight: 700;">Tableau de Bord Gestion Crédits</h2>
        <p style="color: #666; font-size: 16px; margin-bottom: 30px;">Vue complète de tous les clients éligibles aux crédits, crédits impayés et statut des paiements</p>
        
        <!-- Mini Dashboard -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-bottom: 30px;">
            <div class="metric-card info">
                <div class="metric-label">Total Clients Éligibles</div>
                <div class="metric-value"><?php echo number_format($total_customers_eligible, 0, ',', '.'); ?></div>
            </div>
            <div class="metric-card danger">
                <div class="metric-label">Total Montant Impayé</div>
                <div class="metric-value"><?php echo number_format($total_unpaid_amount, 0, ',', '.'); ?> BIF</div>
            </div>
            <div class="metric-card warning">
                <div class="metric-label">Clients avec Impayés</div>
                <div class="metric-value"><?php echo number_format($customers_with_unpaid, 0, ',', '.'); ?></div>
            </div>
            <div class="metric-card danger">
                <div class="metric-label">Paiements Retard</div>
                <div class="metric-value"><?php echo number_format($customers_late, 0, ',', '.'); ?></div>
            </div>
            <div class="metric-card success">
                <div class="metric-label">Total Solde Impayé</div>
                <div class="metric-value"><?php echo number_format($total_outstanding_balance, 0, ',', '.'); ?> BIF</div>
            </div>
        </div>
        
        <!-- Filters -->
        <div style="background-color: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
            <form method="GET" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; align-items: end;">
                <div>
                    <label for="search" style="color: #4B2F1F; font-weight: 600; font-size: 14px; display: block; margin-bottom: 5px;">Rechercher Client</label>
                    <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search_term, ENT_QUOTES, 'UTF-8'); ?>" placeholder="Nom client..." style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #ddd; font-size: 14px;">
                </div>
                <div>
                    <label for="filter_unpaid" style="color: #4B2F1F; font-weight: 600; font-size: 14px; display: block; margin-bottom: 5px;">Crédits Impayés</label>
                    <select id="filter_unpaid" name="filter_unpaid" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #ddd; font-size: 14px;">
                        <option value="">Tous</option>
                        <option value="1" <?php echo $filter_unpaid === '1' ? 'selected' : ''; ?>>A des Impayés</option>
                        <option value="0" <?php echo $filter_unpaid === '0' ? 'selected' : ''; ?>>Aucun Impayé</option>
                    </select>
                </div>
                <div>
                    <label for="filter_late" style="color: #4B2F1F; font-weight: 600; font-size: 14px; display: block; margin-bottom: 5px;">Paiements Retard</label>
                    <select id="filter_late" name="filter_late" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #ddd; font-size: 14px;">
                        <option value="">Tous</option>
                        <option value="1" <?php echo $filter_late === '1' ? 'selected' : ''; ?>>A des Retards</option>
                        <option value="0" <?php echo $filter_late === '0' ? 'selected' : ''; ?>>Aucun Retard</option>
                    </select>
                </div>
                <div>
                    <label for="filter_status" style="color: #4B2F1F; font-weight: 600; font-size: 14px; display: block; margin-bottom: 5px;">Statut Crédit</label>
                    <select id="filter_status" name="filter_status" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #ddd; font-size: 14px;">
                        <option value="">Tous les Statuts</option>
                        <option value="active" <?php echo $filter_status === 'active' ? 'selected' : ''; ?>>Actif</option>
                        <option value="inactive" <?php echo $filter_status === 'inactive' ? 'selected' : ''; ?>>Inactif</option>
                        <option value="suspended" <?php echo $filter_status === 'suspended' ? 'selected' : ''; ?>>Suspendu</option>
                    </select>
                </div>
                <div style="display: flex; gap: 10px;">
                    <button type="submit" style="background-color: #4B2F1F; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 600; flex: 1;">Appliquer Filtres</button>
                    <a href="?" style="background-color: #6c757d; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 600; text-decoration: none; display: inline-block; text-align: center;">Réinitialiser</a>
                </div>
            </form>
            <div style="margin-top: 15px; text-align: right;">
                <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'csv'])); ?>" style="background-color: #28a745; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 600; text-decoration: none; display: inline-block;">Exporter en CSV</a>
            </div>
        </div>
        
        <!-- Customers List -->
        <div style="background-color: #FFFFFF; border-radius: 8px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
            <div style="background-color: #4B2F1F; color: #FFFFFF; padding: 15px;">
                <h3 style="margin: 0; font-size: 20px;">Clients Éligibles aux Crédits (<?php echo count($filtered_customers); ?>)</h3>
            </div>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr style="background-color: #f8f9fa;">
                            <th style="padding: 12px; text-align: left; border-bottom: 2px solid #dee2e6; font-weight: 600;">Client</th>
                            <th style="padding: 12px; text-align: right; border-bottom: 2px solid #dee2e6; font-weight: 600;">Limite Crédit</th>
                            <th style="padding: 12px; text-align: right; border-bottom: 2px solid #dee2e6; font-weight: 600;">Solde Impayé</th>
                            <th style="padding: 12px; text-align: center; border-bottom: 2px solid #dee2e6; font-weight: 600;">Commandes Impayées</th>
                            <th style="padding: 12px; text-align: center; border-bottom: 2px solid #dee2e6; font-weight: 600;">Retard</th>
                            <th style="padding: 12px; text-align: left; border-bottom: 2px solid #dee2e6; font-weight: 600;">Date Échéance Prochaine</th>
                            <th style="padding: 12px; text-align: left; border-bottom: 2px solid #dee2e6; font-weight: 600;">Statut</th>
                            <th style="padding: 12px; text-align: center; border-bottom: 2px solid #dee2e6; font-weight: 600;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($filtered_customers)): ?>
                            <tr>
                                <td colspan="8" style="padding: 30px; text-align: center; color: #666;">Aucun client trouvé correspondant aux filtres.</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($filtered_customers as $customer): ?>
                                <?php
                                // Use ledger_outstanding (from customer_loan_ledger) as source of truth, fallback to loan_balance_bif
                                $ledger_outstanding = (int)($customer['ledger_outstanding'] ?? 0);
                                $outstanding = $ledger_outstanding > 0 ? $ledger_outstanding : (int)($customer['loan_balance_bif'] ?? 0);
                                // total_unpaid is the sum of unpaid invoices from unpaid_invoices table
                                $unpaid = (int)($customer['total_unpaid'] ?? 0);
                                $late_count = (int)($customer['late_invoice_count'] ?? 0);
                                $unpaid_count = (int)($customer['unpaid_invoice_count'] ?? 0);
                                $due_date = $customer['earliest_due_date'] ?? null;
                                $is_late = $due_date && strtotime($due_date) < time();
                                ?>
                                <tr class="customer-row" style="border-bottom: 1px solid #dee2e6;">
                                    <td style="padding: 12px;">
                                        <strong><?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?></strong><br>
                                        <small style="color: #666;"><?php echo htmlspecialchars($customer['tel'] ?: 'N/A', ENT_QUOTES, 'UTF-8'); ?></small>
                                    </td>
                                    <td style="padding: 12px; text-align: right; font-weight: 600;">
                                        <?php echo number_format((int)($customer['loan_limit_bif'] ?? 0), 0, ',', '.'); ?> BIF
                                    </td>
                                    <td style="padding: 12px; text-align: right; font-weight: 600; color: #dc3545;">
                                        <?php echo number_format($outstanding, 0, ',', '.'); ?> BIF
                                    </td>
                                    <td style="padding: 12px; text-align: center;">
                                        <?php if ($unpaid_count > 0): ?>
                                            <span class="unpaid-badge"><?php echo $unpaid_count; ?></span>
                                        <?php else: ?>
                                            <span style="color: #28a745;">0</span>
                                        <?php endif; ?>
                                    </td>
                                    <td style="padding: 12px; text-align: center;">
                                        <?php if ($late_count > 0): ?>
                                            <span class="late-badge"><?php echo $late_count; ?> Retard</span>
                                        <?php else: ?>
                                            <span style="color: #28a745;">-</span>
                                        <?php endif; ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php if ($due_date): ?>
                                            <span style="color: <?php echo $is_late ? '#dc3545' : '#28a745'; ?>; font-weight: 600;">
                                                <?php echo date('Y-m-d', strtotime($due_date)); ?>
                                                <?php if ($is_late): ?>
                                                    <br><small class="late-badge">EN RETARD</small>
                                                <?php endif; ?>
                                            </span>
                                        <?php else: ?>
                                            <span style="color: #666;">N/A</span>
                                        <?php endif; ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php
                                        $loan_status = strtolower($customer['loan_status'] ?? 'inactive');
                                        $loan_status_fr = ($loan_status === 'active') ? 'Actif' : (($loan_status === 'inactive') ? 'Inactif' : (($loan_status === 'suspended') ? 'Suspendu' : ucfirst($loan_status)));
                                        ?>
                                        <span style="padding: 4px 8px; border-radius: 4px; font-size: 12px; font-weight: 600; background-color: <?php echo $loan_status === 'active' ? '#d4edda' : '#fff3cd'; ?>; color: <?php echo $loan_status === 'active' ? '#155724' : '#856404'; ?>;">
                                            <?php echo htmlspecialchars($loan_status_fr, ENT_QUOTES, 'UTF-8'); ?>
                                        </span>
                                    </td>
                                    <td style="padding: 12px; text-align: center;">
                                        <a href="?customer_id=<?php echo $customer['id']; ?><?php echo $search_term ? '&search=' . urlencode($search_term) : ''; ?><?php echo $filter_unpaid ? '&filter_unpaid=' . urlencode($filter_unpaid) : ''; ?><?php echo $filter_late ? '&filter_late=' . urlencode($filter_late) : ''; ?><?php echo $filter_status ? '&filter_status=' . urlencode($filter_status) : ''; ?>#loan-modal" 
                                           style="background-color: #4B2F1F; color: #FFFFFF; padding: 6px 12px; border-radius: 4px; text-decoration: none; font-size: 12px; font-weight: 600;">Voir Détails</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <?php if ($customer_profile): ?>
            <!-- Customer Detail View -->
            <div id="customer-details" style="margin-top: 30px; background-color: #f8f9fa; padding: 20px; border-radius: 8px;">
                <h3 style="color: #4B2F1F; font-size: 24px; margin-bottom: 20px;">Détails Client: <?php echo htmlspecialchars($customer_profile['full_name'], ENT_QUOTES, 'UTF-8'); ?></h3>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
                    <div style="background-color: #FFFFFF; padding: 15px; border-radius: 8px;">
                        <div style="font-size: 12px; color: #666; margin-bottom: 5px;">Solde Impayé</div>
                        <div style="font-size: 24px; font-weight: 700; color: #DC3545;">
                            <?php echo number_format((int)($customer_profile['loan_balance_bif'] ?? 0), 0, ',', '.'); ?> BIF
                        </div>
                    </div>
                    <div style="background-color: #FFFFFF; padding: 15px; border-radius: 8px;">
                        <div style="font-size: 12px; color: #666; margin-bottom: 5px;">Limite Crédit</div>
                        <div style="font-size: 24px; font-weight: 700; color: #4B2F1F;">
                            <?php echo number_format((int)($customer_profile['loan_limit_bif'] ?? 0), 0, ',', '.'); ?> BIF
                        </div>
                    </div>
                    <div style="background-color: #FFFFFF; padding: 15px; border-radius: 8px;">
                        <div style="font-size: 12px; color: #666; margin-bottom: 5px;">Limite Restante</div>
                        <div style="font-size: 24px; font-weight: 700; color: #28a745;">
                            <?php
                                $remainingLimit = max(0, (int)($customer_profile['loan_limit_bif'] ?? 0) - (int)($customer_profile['loan_balance_bif'] ?? 0));
                                echo number_format($remainingLimit, 0, ',', '.');
                            ?> BIF
                        </div>
                    </div>
                    <div style="background-color: #FFFFFF; padding: 15px; border-radius: 8px;">
                        <div style="font-size: 12px; color: #666; margin-bottom: 5px;">Total Payé</div>
                        <div style="font-size: 24px; font-weight: 700; color: #4B2F1F;">
                            <?php echo number_format($total_paid, 0, ',', '.'); ?> BIF
                        </div>
                    </div>
                </div>
                
                <?php /* Section Commandes Crédit retirée sur demande */ ?>
            </div>
        <?php endif; ?>
    </div>
</div>
<?php if ($customer_profile): ?>
    <div class="modal-backdrop" id="loan-modal" data-open="1">
        <div class="modal-content">
            <button class="modal-close" onclick="closeLoanModal()">Fermer</button>
            <h3 style="color:#4B2F1F;margin-top:0;">Détails Client: <?php echo htmlspecialchars($customer_profile['full_name'], ENT_QUOTES, 'UTF-8'); ?></h3>
            <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(180px,1fr));gap:12px;margin-bottom:16px;">
                <div class="pill">Solde Impayé: <?php echo number_format((int)($customer_profile['loan_balance_bif'] ?? 0), 0, ',', '.'); ?> BIF</div>
                <div class="pill">Limite Crédit: <?php echo number_format((int)($customer_profile['loan_limit_bif'] ?? 0), 0, ',', '.'); ?> BIF</div>
                <div class="pill">Limite Restante: <?php echo number_format(max(0, (int)($customer_profile['loan_limit_bif'] ?? 0) - (int)($customer_profile['loan_balance_bif'] ?? 0)), 0, ',', '.'); ?> BIF</div>
                <div class="pill">Total Payé: <?php echo number_format($total_paid, 0, ',', '.'); ?> BIF</div>
            </div>
            <?php if (!empty($loan_invoices)): ?>
                <div style="overflow-x:auto;">
                    <table style="width:100%;border-collapse:collapse;">
                        <thead style="background:#4B2F1F;color:#F4F0E4;">
                            <tr>
                                <th style="padding:8px;text-align:left;">Commande</th>
                                <th style="padding:8px;text-align:right;">Montant</th>
                                <th style="padding:8px;text-align:right;">Impayé</th>
                                <th style="padding:8px;text-align:left;">Date Échéance</th>
                                <th style="padding:8px;text-align:left;">Statut</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($loan_invoices as $invoice): ?>
                                <?php $is_late = $invoice['loan_due_date'] && strtotime($invoice['loan_due_date']) < time(); ?>
                                <tr style="border-bottom:1px solid #eee;">
                                    <td style="padding:8px;"><?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding:8px;text-align:right;"><?php echo number_format((int)$invoice['subtotal_amount'], 0, ',', '.'); ?> BIF</td>
                                    <td style="padding:8px;text-align:right;font-weight:600;color:<?php echo (int)$invoice['loan_outstanding_bif'] > 0 ? '#DC3545' : '#28a745'; ?>;">
                                        <?php echo number_format((int)$invoice['loan_outstanding_bif'], 0, ',', '.'); ?> BIF
                                    </td>
                                    <td style="padding:8px;">
                                        <?php if ($invoice['loan_due_date']): ?>
                                            <span style="color:<?php echo $is_late ? '#dc3545' : '#28a745'; ?>;font-weight:600;">
                                                <?php echo htmlspecialchars($invoice['loan_due_date'], ENT_QUOTES, 'UTF-8'); ?>
                                                <?php if ($is_late): ?><span class="late-badge">EN RETARD</span><?php endif; ?>
                                            </span>
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <td style="padding:8px;">
                                        <?php
                                            $inv_status = strtolower($invoice['loan_status'] ?? 'pending');
                                            $inv_status_fr = ($inv_status === 'settled') ? 'Réglé' : (($inv_status === 'pending') ? 'En Attente' : ucfirst($inv_status));
                                        ?>
                                        <span style="padding:4px 8px;border-radius:4px;font-size:12px;font-weight:600;background-color:<?php echo $inv_status === 'settled' ? '#d4edda' : '#fff3cd'; ?>;color:<?php echo $inv_status === 'settled' ? '#155724' : '#856404'; ?>;">
                                            <?php echo htmlspecialchars($inv_status_fr, ENT_QUOTES, 'UTF-8'); ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p style="color:#666;">Aucune facture crédit trouvée pour ce client.</p>
            <?php endif; ?>
        </div>
    </div>
    <script>
        function closeLoanModal() {
            const modal = document.getElementById('loan-modal');
            if (modal) { modal.style.display = 'none'; }
        }
        document.addEventListener('DOMContentLoaded', function() {
            const modal = document.getElementById('loan-modal');
            if (modal && modal.dataset.open === '1') {
                modal.style.display = 'flex';
            }
        });
    </script>
<?php endif; ?>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
