<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

$allowedRoles = ['DG', 'DGA', 'Board', 'Accountant', 'Admin'];
if (!in_array($_SESSION['role_name'] ?? '', $allowedRoles, true)) {
    header('Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php');
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    include __DIR__ . '/../../includes/header.php';
    echo "<div style='padding: 40px; text-align: center;'><p style='color: #C62828;'>Échec de la connexion à la base de données. Veuillez contacter l'administrateur.</p></div>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Fetch province list for filtering
$provinces = [];
$resultProvinces = $conn->query('SELECT id, name FROM province WHERE name IS NOT NULL ORDER BY name');
if ($resultProvinces) {
    $provinces = $resultProvinces->fetch_all(MYSQLI_ASSOC);
    $resultProvinces->close();
}
$provinceNamesById = [];
if (!empty($provinces)) {
    foreach ($provinces as $provinceRow) {
        $provinceId = (int)($provinceRow['id'] ?? 0);
        if ($provinceId > 0) {
            $provinceNamesById[$provinceId] = $provinceRow['name'] ?? '';
        }
    }
}

$provinceFilter = isset($_GET['province_id']) && $_GET['province_id'] !== '' ? (int)$_GET['province_id'] : null;

$accounts = [];
$sqlAccounts = 'SELECT id, account_number, currency, balance, province_id FROM bank_account';
$params = [];
$types = '';
if ($provinceFilter !== null) {
    $sqlAccounts .= ' WHERE province_id = ? OR province_id IS NULL';
    $params[] = $provinceFilter;
    $types .= 'i';
}
$sqlAccounts .= ' ORDER BY currency, account_number';

$stmtAccounts = $conn->prepare($sqlAccounts);
if ($stmtAccounts) {
    if (!empty($params)) {
        $stmtAccounts->bind_param($types, ...$params);
    }
    $stmtAccounts->execute();
    $accounts = $stmtAccounts->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtAccounts->close();
}

$selectedAccountId = isset($_GET['account_id']) ? (int)$_GET['account_id'] : 0;
if ($selectedAccountId <= 0 && !empty($accounts)) {
    $selectedAccountId = (int)$accounts[0]['id'];
}

$selectedAccount = null;
foreach ($accounts as $account) {
    if ((int)$account['id'] === $selectedAccountId) {
        $selectedAccount = $account;
        break;
    }
}

$today = new DateTimeImmutable('today');
$startDateInput = $_GET['start_date'] ?? $today->format('Y-m-d');
$endDateInput = $_GET['end_date'] ?? $today->format('Y-m-d');

$startDate = DateTimeImmutable::createFromFormat('Y-m-d', $startDateInput) ?: $today;
$endDate = DateTimeImmutable::createFromFormat('Y-m-d', $endDateInput) ?: $today;
if ($startDate > $endDate) {
    [$startDate, $endDate] = [$endDate, $startDate];
}

$rangeStart = $startDate->format('Y-m-d 00:00:00');
$rangeEnd = $endDate->format('Y-m-d 23:59:59');

$transactions = [];
$totalCreditsAllTime = 0;
$totalDebitsAllTime = 0;
$rangeCredits = 0;
$rangeDebits = 0;
$openingBalance = null;
$closingBalance = null;
$netChange = 0;
$currentBalance = null;

if ($selectedAccount) {
    $stmtTotals = $conn->prepare('SELECT SUM(CASE WHEN direction = "CREDIT" THEN amount ELSE 0 END) AS credit_total, SUM(CASE WHEN direction = "DEBIT" THEN amount ELSE 0 END) AS debit_total FROM bank_account_transaction WHERE bank_account_id = ?');
    if ($stmtTotals) {
        $stmtTotals->bind_param('i', $selectedAccountId);
        $stmtTotals->execute();
        $rowTotals = $stmtTotals->get_result()->fetch_assoc();
        $stmtTotals->close();
        $totalCreditsAllTime = (float)($rowTotals['credit_total'] ?? 0);
        $totalDebitsAllTime = (float)($rowTotals['debit_total'] ?? 0);
    }

    $rangeQuery = 'SELECT SUM(CASE WHEN direction = "CREDIT" THEN amount ELSE 0 END) AS credit_total, SUM(CASE WHEN direction = "DEBIT" THEN amount ELSE 0 END) AS debit_total FROM bank_account_transaction WHERE bank_account_id = ? AND created_at BETWEEN ? AND ?';
    $rangeParams = [$selectedAccountId, $rangeStart, $rangeEnd];
    $rangeTypes = 'iss';
    if ($provinceFilter !== null) {
        $rangeQuery .= ' AND (province_id = ? OR province_id IS NULL)';
        $rangeParams[] = $provinceFilter;
        $rangeTypes .= 'i';
    }
    $stmtRange = $conn->prepare($rangeQuery);
    if ($stmtRange) {
        $stmtRange->bind_param($rangeTypes, ...$rangeParams);
        $stmtRange->execute();
        $rowRange = $stmtRange->get_result()->fetch_assoc();
        $stmtRange->close();
        $rangeCredits = (float)($rowRange['credit_total'] ?? 0);
        $rangeDebits = (float)($rowRange['debit_total'] ?? 0);
        $netChange = $rangeCredits - $rangeDebits;
    }

    $afterQuery = 'SELECT SUM(CASE WHEN direction = "CREDIT" THEN amount ELSE -amount END) AS net_amount FROM bank_account_transaction WHERE bank_account_id = ? AND created_at > ?';
    $afterParams = [$selectedAccountId, $rangeEnd];
    $afterTypes = 'is';
    if ($provinceFilter !== null) {
        $afterQuery .= ' AND (province_id = ? OR province_id IS NULL)';
        $afterParams[] = $provinceFilter;
        $afterTypes .= 'i';
    }
    $stmtAfter = $conn->prepare($afterQuery);
    $netAfter = 0.0;
    if ($stmtAfter) {
        $stmtAfter->bind_param($afterTypes, ...$afterParams);
        $stmtAfter->execute();
        $rowAfter = $stmtAfter->get_result()->fetch_assoc();
        $stmtAfter->close();
        $netAfter = (float)($rowAfter['net_amount'] ?? 0);
    }

    $currentBalance = (float)$selectedAccount['balance'];
    $closingBalance = $currentBalance - $netAfter;
    $openingBalance = $closingBalance - $netChange;

    $txnQuery = 'SELECT bat.*, u.full_name AS user_name, p.name AS province_name FROM bank_account_transaction bat LEFT JOIN user u ON bat.created_by = u.id LEFT JOIN province p ON bat.province_id = p.id WHERE bat.bank_account_id = ? AND bat.created_at BETWEEN ? AND ?';
    $txnParams = [$selectedAccountId, $rangeStart, $rangeEnd];
    $txnTypes = 'iss';
    if ($provinceFilter !== null) {
        $txnQuery .= ' AND (bat.province_id = ? OR bat.province_id IS NULL)';
        $txnParams[] = $provinceFilter;
        $txnTypes .= 'i';
    }
    $txnQuery .= ' ORDER BY bat.created_at ASC';
    $stmtTransactions = $conn->prepare($txnQuery);
    if ($stmtTransactions) {
        $stmtTransactions->bind_param($txnTypes, ...$txnParams);
        $stmtTransactions->execute();
        $transactions = $stmtTransactions->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtTransactions->close();
    }

    if (!empty($transactions)) {
        $runningBalanceTracker = (float)($openingBalance ?? 0);
        foreach ($transactions as &$txn) {
            $amountValue = (float)($txn['amount'] ?? 0);
            if (($txn['direction'] ?? '') === 'CREDIT') {
                $runningBalanceTracker += $amountValue;
            } else {
                $runningBalanceTracker -= $amountValue;
            }
            $txn['running_balance'] = $runningBalanceTracker;
        }
        unset($txn);
    }

    $export = $_GET['export'] ?? null;
    if ($export && $selectedAccount) {
        $filenameSafe = preg_replace('/[^a-zA-Z0-9_-]/', '_', $selectedAccount['account_number']);
        if ($export === 'csv') {
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename="bank_ledger_' . $filenameSafe . '_' . $startDate->format('Ymd') . '_' . $endDate->format('Ymd') . '.csv"');
            $out = fopen('php://output', 'w');
            // Add BOM for Excel UTF-8 compatibility
            fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
            fputcsv($out, ['Date', 'Type', 'Direction', 'Montant', 'Référence', 'Description', 'Province', 'Enregistré Par', 'Solde Courant'], ';');
            foreach ($transactions as $txn) {
                $provinceLabel = $txn['province_name'] ?? null;
                if ($provinceLabel === null && !empty($txn['province_id'])) {
                    $provinceId = (int)$txn['province_id'];
                    $provinceLabel = $provinceNamesById[$provinceId] ?? null;
                }
                if ($provinceLabel === null) {
                    $provinceLabel = empty($txn['province_id']) ? 'Toutes les Provinces' : '—';
                }
                $amountValue = (float)($txn['amount'] ?? 0);
                $runningValue = isset($txn['running_balance']) ? (float)$txn['running_balance'] : null;
                fputcsv($out, [
                    date('Y-m-d H:i', strtotime($txn['created_at'] ?? '')), 
                    $txn['transaction_type'] ?? '',
                    $txn['direction'] ?? '',
                    number_format($amountValue, 0, '', ''),
                    ($txn['reference_type'] ?? 'N/A') . ' #' . ($txn['reference_id'] ?? '—'),
                    $txn['description'] ?? '',
                    $provinceLabel,
                    $txn['user_name'] ?? 'System',
                    $runningValue !== null ? number_format($runningValue, 0, '', '') : ''
                ], ';');
            }
            fclose($out);
            exit;
        }
        if ($export === 'pdf') {
            $pdfPath = __DIR__ . '/../../libs/tcpdf/tcpdf.php';
            if (file_exists($pdfPath)) {
                require_once $pdfPath;
                $pdf = new TCPDF('L', 'mm', 'A4');
                $pdf->SetCreator('Masunzu Bank Ledger');
                $pdf->SetAuthor($_SESSION['full_name'] ?? 'System');
                $pdf->SetTitle('Grand Livre Bancaire ' . $selectedAccount['account_number']);
                $pdf->AddPage();
                $html = '<h2>Grand Livre Bancaire</h2>';
                $html .= '<p><strong>Compte:</strong> ' . htmlspecialchars($selectedAccount['account_number'], ENT_QUOTES, 'UTF-8') . '<br>';
                $html .= '<strong>Période:</strong> ' . $startDate->format('d/m/Y') . ' à ' . $endDate->format('d/m/Y') . '</p>';
                $html .= '<table border="1" cellpadding="4"><thead><tr style="background-color:#4B2F1F; color:#FFFFFF;">';
                $html .= '<th>Date</th><th>Type</th><th>Direction</th><th>Montant</th><th>Référence</th><th>Description</th><th>Province</th><th>Enregistré Par</th><th>Solde Courant</th></tr></thead><tbody>';
                foreach ($transactions as $txn) {
                    $provinceLabel = $txn['province_name'] ?? null;
                    if ($provinceLabel === null && !empty($txn['province_id'])) {
                        $provinceId = (int)$txn['province_id'];
                        $provinceLabel = $provinceNamesById[$provinceId] ?? null;
                    }
                    if ($provinceLabel === null) {
                        $provinceLabel = empty($txn['province_id']) ? 'Toutes les Provinces' : '—';
                    }
                    $html .= '<tr>';
                    $html .= '<td>' . htmlspecialchars(date('d/m/Y H:i', strtotime($txn['created_at'] ?? '')), ENT_QUOTES, 'UTF-8') . '</td>';
                    $html .= '<td>' . htmlspecialchars($txn['transaction_type'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>';
                    $html .= '<td>' . htmlspecialchars($txn['direction'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>';
                    $html .= '<td>' . number_format((float)($txn['amount'] ?? 0), 0, ',', '.') . '</td>';
                    $html .= '<td>' . htmlspecialchars(($txn['reference_type'] ?? 'N/A') . ' #' . ($txn['reference_id'] ?? '—'), ENT_QUOTES, 'UTF-8') . '</td>';
                    $html .= '<td>' . htmlspecialchars($txn['description'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>';
                    $html .= '<td>' . htmlspecialchars($provinceLabel, ENT_QUOTES, 'UTF-8') . '</td>';
                    $html .= '<td>' . htmlspecialchars($txn['user_name'] ?? 'System', ENT_QUOTES, 'UTF-8') . '</td>';
                    $html .= '<td>' . number_format((float)($txn['running_balance'] ?? 0), 0, ',', '.') . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</tbody></table>';
                $pdf->writeHTML($html, true, false, true, false, '');
                $pdf->Output('bank_ledger_' . $filenameSafe . '_' . $startDate->format('Ymd') . '_' . $endDate->format('Ymd') . '.pdf', 'D');
                exit;
            }
        }
    }
}

$conn->close();
include __DIR__ . '/../../includes/header.php';
?>
<div style="padding: 20px; background-color: #F4F0E4; min-height: 70vh;">
    <!-- Header -->
    <div style="display: flex; justify-content: space-between; align-items: flex-start; flex-wrap: wrap; gap: 16px; margin-bottom: 20px;">
        <div>
            <h1 style="margin: 0 0 8px; color: #4B2F1F; font-size: 28px; font-weight: 700;">Grand Livre Bancaire</h1>
            <p style="margin: 0; color: #6B4F2F; font-size: 14px;">
                Filtrer par compte, province et période pour examiner les soldes et l'historique des transactions.
            </p>
        </div>
        <?php if ($selectedAccount): ?>
        <div style="display: flex; gap: 12px; flex-wrap: wrap;">
            <div style="background: linear-gradient(135deg, #1565C0 0%, #1976D2 100%); color: #FFF; padding: 12px 20px; border-radius: 10px; text-align: center; min-width: 140px;">
                <p style="margin: 0; font-size: 11px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.9;">Solde Actuel</p>
                <p style="margin: 4px 0 0; font-size: 18px; font-weight: 700;"><?php echo number_format((float)($currentBalance ?? 0), 0, ',', '.'); ?> BIF</p>
            </div>
            <div style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color: #F4F0E4; padding: 12px 20px; border-radius: 10px; text-align: center; min-width: 140px;">
                <p style="margin: 0; font-size: 11px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.9;">Transactions</p>
                <p style="margin: 4px 0 0; font-size: 18px; font-weight: 700;"><?php echo number_format(count($transactions)); ?></p>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <?php if (empty($accounts)): ?>
        <div style="background: #fff; border-radius: 12px; padding: 60px 20px; text-align: center; box-shadow: 0 2px 8px rgba(0,0,0,0.06);">
            <div style="font-size: 64px; margin-bottom: 15px;">🏦</div>
            <p style="color: #4B2F1F; font-size: 18px; font-weight: 600; margin: 0 0 8px;">Aucun compte bancaire configuré</p>
            <p style="color: #6B4F2F; font-size: 14px; margin: 0;">Veuillez créer un compte bancaire pour commencer.</p>
        </div>
    <?php else: ?>
        <!-- Filters -->
        <form method="get" style="display: grid; grid-template-columns: repeat(6, 1fr); gap: 12px; background: #fff; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.06); margin-bottom: 20px;">
            <div>
                <label for="province_id" style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Province</label>
                <select name="province_id" id="province_id" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #B08968; font-size: 14px; box-sizing: border-box;">
                    <option value="">Toutes les Provinces</option>
                    <?php foreach ($provinces as $province): ?>
                        <option value="<?php echo (int)$province['id']; ?>" <?php echo $provinceFilter !== null && (int)$province['id'] === $provinceFilter ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($province['name'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label for="account_id" style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Compte Bancaire</label>
                <select name="account_id" id="account_id" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #B08968; font-size: 14px; box-sizing: border-box;">
                    <?php foreach ($accounts as $account): ?>
                        <?php
                            $accountProvinceLabel = 'Global';
                            if (!empty($account['province_id'])) {
                                $provinceId = (int)$account['province_id'];
                                $accountProvinceLabel = $provinceNamesById[$provinceId] ?? ('Province ' . $provinceId);
                            }
                            $label = sprintf('%s · %s · %s', $account['account_number'], $account['currency'], $accountProvinceLabel);
                        ?>
                        <option value="<?php echo (int)$account['id']; ?>" <?php echo (int)$account['id'] === $selectedAccountId ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label for="start_date" style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Date Début</label>
                <input type="date" name="start_date" id="start_date" value="<?php echo htmlspecialchars($startDate->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>" 
                       style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #B08968; font-size: 14px; box-sizing: border-box;">
            </div>
            <div>
                <label for="end_date" style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Date Fin</label>
                <input type="date" name="end_date" id="end_date" value="<?php echo htmlspecialchars($endDate->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>" 
                       style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #B08968; font-size: 14px; box-sizing: border-box;">
            </div>
            <div style="display: flex; align-items: flex-end;">
                <button type="submit" style="width: 100%; background: #4B2F1F; color: #F4F0E4; padding: 11px; border: none; border-radius: 6px; cursor: pointer; font-weight: 600; font-size: 14px;">
                    Appliquer
                </button>
            </div>
            <div style="display: flex; align-items: flex-end; gap: 8px;">
                <button type="submit" name="export" value="csv" style="flex: 1; background: #2E7D32; color: #FFF; padding: 11px; border: none; border-radius: 6px; cursor: pointer; font-weight: 600; font-size: 13px;">
                    📥 CSV
                </button>
                <button type="submit" name="export" value="pdf" style="flex: 1; background: #C62828; color: #FFF; padding: 11px; border: none; border-radius: 6px; cursor: pointer; font-weight: 600; font-size: 13px;">
                    📄 PDF
                </button>
            </div>
        </form>

        <?php if ($selectedAccount): ?>
            <!-- Balance Cards -->
            <div style="display: grid; grid-template-columns: repeat(5, 1fr); gap: 15px; margin-bottom: 20px;">
                <div style="background: #fff; border-radius: 10px; padding: 16px; border-left: 4px solid #1565C0; box-shadow: 0 2px 8px rgba(0,0,0,0.06);">
                    <p style="margin: 0; color: #6B4F2F; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Solde Actuel</p>
                    <p style="margin: 6px 0 0; color: #1565C0; font-size: 20px; font-weight: 700;">
                        <?php echo number_format((float)($currentBalance ?? 0), 0, ',', '.'); ?> BIF
                    </p>
                </div>
                <div style="background: #fff; border-radius: 10px; padding: 16px; border-left: 4px solid #4B2F1F; box-shadow: 0 2px 8px rgba(0,0,0,0.06);">
                    <p style="margin: 0; color: #6B4F2F; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Solde d'Ouverture</p>
                    <p style="margin: 6px 0 0; color: #4B2F1F; font-size: 20px; font-weight: 700;">
                        <?php echo number_format((float)($openingBalance ?? 0), 0, ',', '.'); ?> BIF
                    </p>
                </div>
                <div style="background: #fff; border-radius: 10px; padding: 16px; border-left: 4px solid #2E7D32; box-shadow: 0 2px 8px rgba(0,0,0,0.06);">
                    <p style="margin: 0; color: #6B4F2F; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Crédits (Période)</p>
                    <p style="margin: 6px 0 0; color: #2E7D32; font-size: 20px; font-weight: 700;">
                        +<?php echo number_format((float)$rangeCredits, 0, ',', '.'); ?> BIF
                    </p>
                </div>
                <div style="background: #fff; border-radius: 10px; padding: 16px; border-left: 4px solid #C62828; box-shadow: 0 2px 8px rgba(0,0,0,0.06);">
                    <p style="margin: 0; color: #6B4F2F; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Débits (Période)</p>
                    <p style="margin: 6px 0 0; color: #C62828; font-size: 20px; font-weight: 700;">
                        -<?php echo number_format((float)$rangeDebits, 0, ',', '.'); ?> BIF
                    </p>
                </div>
                <div style="background: #fff; border-radius: 10px; padding: 16px; border-left: 4px solid #F4A261; box-shadow: 0 2px 8px rgba(0,0,0,0.06);">
                    <p style="margin: 0; color: #6B4F2F; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Solde de Clôture</p>
                    <p style="margin: 6px 0 0; color: #F4A261; font-size: 20px; font-weight: 700;">
                        <?php echo number_format((float)($closingBalance ?? 0), 0, ',', '.'); ?> BIF
                    </p>
                </div>
            </div>

            <!-- Transactions Table -->
            <div style="background: #fff; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.06); overflow: hidden;">
                <div style="padding: 16px 20px; border-bottom: 1px solid #F0F0F0; display: flex; justify-content: space-between; align-items: center; background: #FDFBF7;">
                    <span style="font-weight: 600; color: #4B2F1F;">
                        <?php echo number_format(count($transactions)); ?> transaction<?php echo count($transactions) === 1 ? '' : 's'; ?>
                    </span>
                    <span style="color: #6B4F2F; font-size: 13px;">
                        Compte: <?php echo htmlspecialchars($selectedAccount['account_number'], ENT_QUOTES, 'UTF-8'); ?> · 
                        Période: <?php echo $startDate->format('d/m/Y'); ?> - <?php echo $endDate->format('d/m/Y'); ?>
                    </span>
                </div>
                <?php if (!empty($transactions)): ?>
                    <div style="overflow-x: auto;">
                        <table style="width: 100%; border-collapse: collapse; font-size: 14px; min-width: 1000px;">
                            <thead>
                                <tr style="background: #4B2F1F; color: #F4F0E4;">
                                    <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Date</th>
                                    <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Type</th>
                                    <th style="padding: 14px 16px; text-align: center; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Direction</th>
                                    <th style="padding: 14px 16px; text-align: right; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Montant</th>
                                    <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Référence</th>
                                    <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Description</th>
                                    <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Province</th>
                                    <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Enregistré Par</th>
                                    <th style="padding: 14px 16px; text-align: right; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Solde Courant</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transactions as $txn): ?>
                                    <?php
                                        $amountValue = (float)($txn['amount'] ?? 0);
                                        $provinceLabel = $txn['province_name'] ?? null;
                                        if ($provinceLabel === null && !empty($txn['province_id'])) {
                                            $provinceId = (int)$txn['province_id'];
                                            $provinceLabel = $provinceNamesById[$provinceId] ?? null;
                                        }
                                        if ($provinceLabel === null) {
                                            $provinceLabel = empty($txn['province_id']) ? 'Global' : '—';
                                        }
                                        $runningValue = (float)($txn['running_balance'] ?? 0);
                                        $isCredit = ($txn['direction'] ?? '') === 'CREDIT';
                                    ?>
                                    <tr style="border-bottom: 1px solid #F0F0F0; transition: background-color 0.2s;" 
                                        onmouseover="this.style.backgroundColor='#FDFBF7'" 
                                        onmouseout="this.style.backgroundColor='transparent'">
                                        <td style="padding: 12px 16px; color: #4B2F1F;">
                                            <div style="font-weight: 500;"><?php echo htmlspecialchars(date('d/m/Y', strtotime($txn['created_at'] ?? '')), ENT_QUOTES, 'UTF-8'); ?></div>
                                            <div style="font-size: 12px; color: #6B4F2F;"><?php echo htmlspecialchars(date('H:i', strtotime($txn['created_at'] ?? '')), ENT_QUOTES, 'UTF-8'); ?></div>
                                        </td>
                                        <td style="padding: 12px 16px; color: #4B2F1F;">
                                            <?php echo htmlspecialchars($txn['transaction_type'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?>
                                        </td>
                                        <td style="padding: 12px 16px; text-align: center;">
                                            <span style="display: inline-block; padding: 4px 10px; border-radius: 4px; font-size: 12px; font-weight: 600; background: <?php echo $isCredit ? '#e8f5e9' : '#ffebee'; ?>; color: <?php echo $isCredit ? '#2E7D32' : '#C62828'; ?>;">
                                                <?php echo $isCredit ? '↑ CRÉDIT' : '↓ DÉBIT'; ?>
                                            </span>
                                        </td>
                                        <td style="padding: 12px 16px; text-align: right; font-weight: 700; color: <?php echo $isCredit ? '#2E7D32' : '#C62828'; ?>;">
                                            <?php echo $isCredit ? '+' : '-'; ?><?php echo number_format($amountValue, 0, ',', '.'); ?>
                                        </td>
                                        <td style="padding: 12px 16px; color: #4B2F1F; font-family: monospace; font-size: 13px;">
                                            <?php echo htmlspecialchars(($txn['reference_type'] ?? 'N/A') . ' #' . ($txn['reference_id'] ?? '—'), ENT_QUOTES, 'UTF-8'); ?>
                                        </td>
                                        <td style="padding: 12px 16px; color: #4B2F1F; max-width: 200px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;" title="<?php echo htmlspecialchars($txn['description'] ?? '—', ENT_QUOTES, 'UTF-8'); ?>">
                                            <?php echo htmlspecialchars($txn['description'] ?? '—', ENT_QUOTES, 'UTF-8'); ?>
                                        </td>
                                        <td style="padding: 12px 16px; color: #6B4F2F;">
                                            <?php echo htmlspecialchars($provinceLabel, ENT_QUOTES, 'UTF-8'); ?>
                                        </td>
                                        <td style="padding: 12px 16px; color: #4B2F1F;">
                                            <?php echo htmlspecialchars($txn['user_name'] ?? 'System', ENT_QUOTES, 'UTF-8'); ?>
                                        </td>
                                        <td style="padding: 12px 16px; text-align: right; font-weight: 700; color: #2C1810; font-size: 15px;">
                                            <?php echo number_format($runningValue, 0, ',', '.'); ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div style="padding: 60px 20px; text-align: center;">
                        <div style="font-size: 48px; margin-bottom: 10px;">📋</div>
                        <p style="color: #4B2F1F; font-size: 16px; font-weight: 600; margin: 0 0 5px;">Aucune transaction trouvée</p>
                        <p style="color: #6B4F2F; font-size: 13px; margin: 0;">Aucune transaction enregistrée pour ce compte dans la période sélectionnée.</p>
                    </div>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div style="background: #fff; border-radius: 12px; padding: 40px 20px; text-align: center; box-shadow: 0 2px 8px rgba(0,0,0,0.06);">
                <p style="color: #4B2F1F; font-size: 16px; margin: 0;">Veuillez sélectionner un compte bancaire pour afficher son grand livre.</p>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
