<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$baseDir = __DIR__ . '/../../';
require_once $baseDir . 'includes/auth.php';
require_once $baseDir . 'config/database.php';
require_once $baseDir . 'includes/functions.php';
require_once $baseDir . 'includes/cash_helpers.php';

try {
    require_login();

    if (($_SESSION['role_name'] ?? '') !== 'Accountant') {
        $target = strtolower(str_replace(' ', '_', $_SESSION['role_name'] ?? '')) . '_dashboard.php';
        header('Location: /masunzu_bar_hotel/dashboards/' . $target);
        exit;
    }

    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception('Échec de la connexion à la base de données: ' . mysqli_connect_error());
    }

    if (!array_key_exists('province_id', $_SESSION) || (isset($_SESSION['province_id']) && (int)$_SESSION['province_id'] <= 0)) {
        $stmtProvince = $conn->prepare('SELECT province_id FROM user WHERE id = ?');
        if ($stmtProvince) {
            $stmtProvince->bind_param('i', $_SESSION['user_id']);
            $stmtProvince->execute();
            $row = $stmtProvince->get_result()->fetch_assoc();
            $stmtProvince->close();
            $fetchedProvince = $row['province_id'] ?? null;
            $_SESSION['province_id'] = $fetchedProvince !== null && (int)$fetchedProvince > 0 ? (int)$fetchedProvince : null;
        }
    }
    $provinceId = $_SESSION['province_id'] ?? null;
    if ($provinceId !== null) {
        $provinceId = (int)$provinceId;
        if ($provinceId <= 0) {
            $provinceId = null;
            $_SESSION['province_id'] = null;
        }
    }

    $provinceName = 'Toutes les provinces';
    if ($provinceId !== null) {
        $stmtProvinceName = $conn->prepare('SELECT name FROM province WHERE id = ?');
        if ($stmtProvinceName) {
            $stmtProvinceName->bind_param('i', $provinceId);
            $stmtProvinceName->execute();
            $provinceName = $stmtProvinceName->get_result()->fetch_assoc()['name'] ?? 'Unknown';
            $stmtProvinceName->close();
        }
    }
    $availableCashBalance = function_exists('mbh_get_province_cash_balance') ? mbh_get_province_cash_balance($conn, $provinceId) : 0.0;
    $availableCashLabel = $provinceId !== null ? $provinceName : 'toutes les provinces';

    $createLedgerSql = "
        CREATE TABLE IF NOT EXISTS bank_account_transaction (
            id INT AUTO_INCREMENT PRIMARY KEY,
            bank_account_id INT NOT NULL,
            province_id INT DEFAULT NULL,
            transaction_type ENUM('CASH_TRANSFER_RECEIVED','BANK_DEPOSIT','REVERSAL','ADJUSTMENT') NOT NULL,
            direction ENUM('CREDIT','DEBIT') NOT NULL,
            amount BIGINT(20) NOT NULL,
            reference_type ENUM('CASH_TRANSFER','CASH_DEPOSIT','MANUAL','OTHER') NOT NULL DEFAULT 'OTHER',
            reference_id INT DEFAULT NULL,
            description VARCHAR(255) DEFAULT NULL,
            created_by INT DEFAULT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_account_created_at (bank_account_id, created_at),
            INDEX idx_reference (reference_type, reference_id),
            CONSTRAINT fk_bat_account FOREIGN KEY (bank_account_id) REFERENCES bank_account(id) ON DELETE CASCADE,
            CONSTRAINT fk_bat_province FOREIGN KEY (province_id) REFERENCES province(id) ON DELETE SET NULL,
            CONSTRAINT fk_bat_user FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    if (!$conn->query($createLedgerSql)) {
        $errors[] = 'Impossible d\'initialiser le suivi du grand livre bancaire. Contactez l\'administrateur.';
        error_log('Failed to ensure bank_account_transaction table: ' . $conn->error);
    }

    $bankDepositColumns = [];
    $columnsResult = $conn->query('SHOW COLUMNS FROM bank_deposit');
    if ($columnsResult) {
        while ($col = $columnsResult->fetch_assoc()) {
            $bankDepositColumns[$col['Field']] = $col;
        }
        $columnsResult->close();
    }
    if (!isset($bankDepositColumns['slip_number'])) {
        $conn->query("ALTER TABLE bank_deposit ADD COLUMN slip_number VARCHAR(255) DEFAULT NULL AFTER amount");
    }
    if (!isset($bankDepositColumns['slip_file_path'])) {
        $conn->query("ALTER TABLE bank_deposit ADD COLUMN slip_file_path VARCHAR(255) DEFAULT NULL AFTER slip_number");
    }
    if (!isset($bankDepositColumns['bank_account_id'])) {
        if ($conn->query("ALTER TABLE bank_deposit ADD COLUMN bank_account_id INT NULL AFTER province_id") === true) {
            $conn->query("ALTER TABLE bank_deposit ADD INDEX idx_bank_account_id (bank_account_id)");
            if ($conn->query("ALTER TABLE bank_deposit ADD CONSTRAINT fk_bank_deposit_account FOREIGN KEY (bank_account_id) REFERENCES bank_account(id) ON DELETE SET NULL") === false) {
                error_log('Failed to create FK bank_deposit -> bank_account: ' . $conn->error);
            }
        }
    }

    $bankAccountColumns = [];
    $accountColumnsResult = $conn->query('SHOW COLUMNS FROM bank_account');
    if ($accountColumnsResult) {
        while ($col = $accountColumnsResult->fetch_assoc()) {
            $bankAccountColumns[$col['Field']] = $col;
        }
        $accountColumnsResult->close();
    }
    if (!isset($bankAccountColumns['is_main'])) {
        if (!$conn->query("ALTER TABLE bank_account ADD COLUMN is_main TINYINT(1) NOT NULL DEFAULT 0 AFTER currency")) {
            error_log('Failed to add is_main column to bank_account: ' . $conn->error);
        }
    }

    $uploadDir = $baseDir . 'uploads/bank_slips';
    $errors = $_SESSION['bank_deposit_errors'] ?? [];
    $success = $_SESSION['bank_deposit_success'] ?? '';
    $oldInput = $_SESSION['bank_deposit_old'] ?? [];
    unset($_SESSION['bank_deposit_errors'], $_SESSION['bank_deposit_success'], $_SESSION['bank_deposit_old']);

    $oldInput = array_merge([
        'amount' => '',
        'slip_number' => '',
        'deposit_date' => date('Y-m-d\TH:i'),
        'bank_account_id' => '',
    ], $oldInput);

    $accounts = [];
    $accountIndex = [];
    $usdBalance = 0;
    $bifBalance = 0;

    $result = $conn->query("SHOW TABLES LIKE 'bank_account'");
    if ($result && $result->num_rows > 0) {
        $stmt = $conn->prepare('SELECT currency, SUM(balance) AS total FROM bank_account GROUP BY currency');
        if ($stmt) {
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
            foreach ($rows as $row) {
                if ($row['currency'] === 'USD') {
                    $usdBalance = (float)$row['total'];
                } elseif ($row['currency'] === 'BIF') {
                    $bifBalance = (float)$row['total'];
                }
            }
        }

        $accountSql = 'SELECT id, account_number, currency, balance, province_id FROM bank_account';
        if ($provinceId !== null) {
            $accountSql .= ' WHERE province_id IS NULL OR province_id = ?';
        }
        $accountSql .= ' ORDER BY currency, account_number';
        $stmt = $conn->prepare($accountSql);
        if ($stmt) {
            if ($provinceId !== null) {
                $stmt->bind_param('i', $provinceId);
            }
            $stmt->execute();
            $accounts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
            foreach ($accounts as $account) {
                $accountIndex[(int)$account['id']] = $account;
            }
        }
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_deposit'])) {
        $errors = [];
        $uploadedAbsolute = null;
        $uploadedRelative = null;
        $depositProvinceId = $provinceId;

        $amountInput = trim($_POST['amount'] ?? '');
        $slipNumber = trim($_POST['slip_number'] ?? '');
        $depositDateInput = trim($_POST['deposit_date'] ?? '');
        $bankAccountId = isset($_POST['bank_account_id']) ? (int)$_POST['bank_account_id'] : 0;

        $oldInput = [
            'amount' => $amountInput,
            'slip_number' => $slipNumber,
            'deposit_date' => $depositDateInput !== '' ? $depositDateInput : date('Y-m-d\TH:i'),
            'bank_account_id' => $bankAccountId > 0 ? (string)$bankAccountId : '',
        ];

        $amount = $amountInput !== '' ? filter_var($amountInput, FILTER_VALIDATE_FLOAT) : false;
        if ($amount === false || $amount <= 0) {
            $errors[] = 'Le montant est requis et doit être un nombre positif.';
        } else {
            $amount = round((float)$amount, 2);
        }

        if ($slipNumber === '') {
            $errors[] = 'Le numéro de bordereau est requis.';
        } elseif (strlen($slipNumber) > 255) {
            $errors[] = 'Slip number must be 255 characters or fewer.';
        } else {
            $stmt = $conn->prepare('SELECT id FROM bank_deposit WHERE slip_number = ? LIMIT 1');
            if ($stmt) {
                $stmt->bind_param('s', $slipNumber);
                $stmt->execute();
                if ($stmt->get_result()->fetch_assoc()) {
                    $errors[] = 'A bank deposit with this slip number already exists.';
                }
                $stmt->close();
            }
        }

        $selectedAccount = null;
        if (empty($accountIndex)) {
            $errors[] = 'No bank accounts are configured. Contact the administrator.';
        } elseif ($bankAccountId <= 0 || !isset($accountIndex[$bankAccountId])) {
            $errors[] = 'Please choose a valid bank account.';
        } else {
            $selectedAccount = $accountIndex[$bankAccountId];
            // Determine province_id: use bank account's province, then accountant's province, then find a default
            $accountProvinceId = isset($selectedAccount['province_id']) ? (int)$selectedAccount['province_id'] : 0;
            if ($accountProvinceId > 0) {
                $depositProvinceId = $accountProvinceId;
            } elseif ($provinceId !== null) {
                $depositProvinceId = $provinceId;
            } else {
                // If still null, try to get a default province from any bank account
                $stmtDefaultProvince = $conn->prepare('SELECT province_id FROM bank_account WHERE province_id IS NOT NULL LIMIT 1');
                if ($stmtDefaultProvince) {
                    $stmtDefaultProvince->execute();
                    $defaultProvinceRow = $stmtDefaultProvince->get_result()->fetch_assoc();
                    $stmtDefaultProvince->close();
                    if ($defaultProvinceRow && !empty($defaultProvinceRow['province_id'])) {
                        $depositProvinceId = (int)$defaultProvinceRow['province_id'];
                    }
                }
                // If still null, get the first province from the database
                if ($depositProvinceId === null) {
                    $stmtFirstProvince = $conn->query('SELECT id FROM province ORDER BY id LIMIT 1');
                    if ($stmtFirstProvince && $row = $stmtFirstProvince->fetch_assoc()) {
                        $depositProvinceId = (int)$row['id'];
                    }
                }
            }
            if (($selectedAccount['currency'] ?? '') !== 'BIF') {
                $errors[] = 'Bank deposits in BIF must target a BIF-denominated account.';
            }
        }

        if ($depositDateInput === '') {
            $depositDate = date('Y-m-d H:i:s');
        } else {
            $dateObj = DateTime::createFromFormat('Y-m-d\TH:i', $depositDateInput);
            if (!$dateObj) {
                $errors[] = 'La date de dépôt est requise.';
                $depositDate = date('Y-m-d H:i:s');
            } else {
                $depositDate = $dateObj->format('Y-m-d H:i:s');
            }
        }

        if (!is_dir($uploadDir)) {
            if (!@mkdir($uploadDir, 0755, true)) {
                $errors[] = 'Échec de la création du répertoire de téléchargement. Vérifiez les permissions.';
            }
        }
        if (empty($errors) && (!is_dir($uploadDir) || !is_writable($uploadDir))) {
            $errors[] = 'Le répertoire de téléchargement n\'est pas accessible en écriture.';
        }

        if ($amount !== false && $depositProvinceId) {
            $stmt = $conn->prepare('SELECT balance FROM province_cash_account WHERE province_id = ?');
            if ($stmt) {
                $stmt->bind_param('i', $depositProvinceId);
                $stmt->execute();
                $stmt->bind_result($availableCash);
                $stmt->fetch();
                $stmt->close();
                $availableCash = (float)($availableCash ?? 0);
                if ($amount > $availableCash) {
                    $errors[] = 'Deposit amount exceeds the available cash on hand (' . number_format($availableCash, 2, '.', ',') . ' BIF).';
                }
            }
        }

        if (empty($errors)) {
            if (!isset($_FILES['slip_file']) || $_FILES['slip_file']['error'] === UPLOAD_ERR_NO_FILE) {
                $errors[] = 'Veuillez télécharger le bordereau bancaire (PDF ou image).';
            } elseif ($_FILES['slip_file']['error'] !== UPLOAD_ERR_OK) {
                $errors[] = 'Échec du téléchargement du fichier de bordereau. Vérifiez les permissions du répertoire.';
            } else {
                $file = $_FILES['slip_file'];
                if ($file['size'] > 5 * 1024 * 1024) {
                    $errors[] = 'La taille du fichier doit être inférieure à 5 Mo.';
                } else {
                    $finfo = finfo_open(FILEINFO_MIME_TYPE);
                    $mime = $finfo ? finfo_file($finfo, $file['tmp_name']) : null;
                    if ($finfo) {
                        finfo_close($finfo);
                    }
                    $allowedMimes = [
                        'application/pdf' => 'pdf',
                        'image/jpeg' => 'jpg',
                        'image/png' => 'png',
                    ];
                    if (!$mime || !isset($allowedMimes[$mime])) {
                        $errors[] = 'Seuls les fichiers JPG, JPEG, PNG et GIF sont autorisés.';
                    } else {
                        try {
                            $randomSuffix = bin2hex(random_bytes(4));
                        } catch (Throwable $th) {
                            $fallback = function_exists('openssl_random_pseudo_bytes') ? openssl_random_pseudo_bytes(4) : null;
                            $randomSuffix = $fallback ? bin2hex($fallback) : dechex(mt_rand());
                        }
                        $extension = $allowedMimes[$mime];
                        $filename = sprintf('deposit_%d_%s.%s', $_SESSION['user_id'], $randomSuffix, $extension);
                        $destination = rtrim($uploadDir, '/') . '/' . $filename;
                        if (!move_uploaded_file($file['tmp_name'], $destination)) {
                            $errors[] = 'Échec du téléchargement du fichier de bordereau. Vérifiez les permissions du répertoire.';
                        } else {
                            $uploadedAbsolute = $destination;
                            $uploadedRelative = '/masunzu_bar_hotel/uploads/bank_slips/' . $filename;
                            @chmod($uploadedAbsolute, 0644);
                        }
                    }
                }
            }
        }

        // Final validation: ensure province_id is not null before inserting
        if ($depositProvinceId === null) {
            $errors[] = 'Impossible de déterminer la province pour ce dépôt. Veuillez contacter l\'administrateur pour configurer une province pour votre compte ou pour le compte bancaire sélectionné.';
        }

        if (empty($errors)) {
            try {
                $conn->begin_transaction();

                // Temporarily disable strict mode for this INSERT (STRICT_TRANS_TABLES can cause issues)
                $originalSqlMode = $conn->query("SELECT @@SESSION.SQL_MODE")->fetch_row()[0];
                $conn->query("SET SESSION SQL_MODE = ''");

                $stmt = $conn->prepare('INSERT INTO bank_deposit (user_id, amount, slip_number, slip_file_path, deposit_date, province_id, bank_account_id) VALUES (?, ?, ?, ?, ?, ?, ?)');
                if (!$stmt) {
                    // Restore SQL mode on error
                    $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");
                    throw new Exception('Query preparation failed for deposit: ' . $conn->error);
                }
                $provinceParam = $depositProvinceId;
                $stmt->bind_param('idsssii', $_SESSION['user_id'], $amount, $slipNumber, $uploadedRelative, $depositDate, $provinceParam, $bankAccountId);
                if (!$stmt->execute()) {
                    // Restore SQL mode on error
                    $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");
                    throw new Exception('Échec de l\'enregistrement du dépôt bancaire: ' . $stmt->error);
                }
                $depositId = $stmt->insert_id;
                $stmt->close();

                // Restore SQL mode after successful INSERT
                $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");

                $stmt = $conn->prepare('UPDATE bank_account SET balance = balance + ? WHERE id = ?');
                if (!$stmt) {
                    throw new Exception('Failed to prepare bank account update: ' . $conn->error);
                }
                $stmt->bind_param('di', $amount, $bankAccountId);
                if (!$stmt->execute() || $stmt->affected_rows === 0) {
                    throw new Exception('Failed to update the bank account balance.');
                }
                $stmt->close();

                if ($depositProvinceId) {
                    adjustProvinceCash(
                        $conn,
                        $depositProvinceId,
                        (float)$amount,
                        'BANK_DEPOSIT',
                        'BANK_DEPOSIT',
                        $depositId,
                        $_SESSION['user_id'],
                        'Bank deposit #' . $depositId . ' recorded.',
                        $depositDate
                    );
                }

                $ledgerStmt = $conn->prepare('INSERT INTO bank_account_transaction (bank_account_id, province_id, transaction_type, direction, amount, reference_type, reference_id, description, created_by, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
                if (!$ledgerStmt) {
                    throw new Exception('Failed to record bank ledger entry: ' . $conn->error);
                }
                $transactionType = 'BANK_DEPOSIT';
                $direction = 'CREDIT';
                $referenceType = 'CASH_DEPOSIT';
                $ledgerDescription = sprintf('Bank deposit #%d (%s)', $depositId, $slipNumber);
                $ledgerProvinceId = $depositProvinceId;
                $ledgerStmt->bind_param(
                    'iissdsisis',
                    $bankAccountId,
                    $ledgerProvinceId,
                    $transactionType,
                    $direction,
                    $amount,
                    $referenceType,
                    $depositId,
                    $ledgerDescription,
                    $_SESSION['user_id'],
                    $depositDate
                );
                if (!$ledgerStmt->execute()) {
                    $ledgerError = $ledgerStmt->error ?: 'unknown error';
                    $ledgerStmt->close();
                    throw new Exception('Failed to write bank account transaction: ' . $ledgerError);
                }
                $ledgerStmt->close();

                if (function_exists('log_action')) {
                    log_action($_SESSION['user_id'], 'bank_deposit', "Bank deposit of {$amount} BIF recorded with slip {$slipNumber}");
                }

                $conn->commit();
                $_SESSION['bank_deposit_success'] = 'Dépôt bancaire enregistré avec succès.';
                unset($_SESSION['bank_deposit_old']);
                header('Location: bank_accounts.php');
                exit;
            } catch (Exception $e) {
                // Restore SQL mode if it was changed
                if (isset($originalSqlMode)) {
                    $conn->query("SET SESSION SQL_MODE = '" . $conn->real_escape_string($originalSqlMode) . "'");
                }
                $conn->rollback();
                $errors[] = $e->getMessage();
                if ($uploadedAbsolute && file_exists($uploadedAbsolute)) {
                    @unlink($uploadedAbsolute);
                }
            }
        } else {
            if ($uploadedAbsolute && file_exists($uploadedAbsolute)) {
                @unlink($uploadedAbsolute);
            }
        }

        $_SESSION['bank_deposit_errors'] = $errors;
        $_SESSION['bank_deposit_old'] = $oldInput;
        header('Location: bank_accounts.php');
        exit;
    }

    $showHistory = isset($_POST['show_history']);
    $deposits = [];
    if ($showHistory) {
        $historySql = "
            SELECT bd.id, bd.amount, bd.deposit_date, bd.slip_number, bd.slip_file_path,
                   u.full_name AS user_name, r.name AS role_name
            FROM bank_deposit bd
            LEFT JOIN user u ON bd.user_id = u.id
            LEFT JOIN role r ON u.role_id = r.id
            WHERE r.name IN ('Accountant', 'Caissier_Comptable')
        ";
        if ($provinceId !== null) {
            $historySql .= ' AND bd.province_id = ?';
        }
        $historySql .= ' ORDER BY bd.deposit_date DESC';
        $stmt = $conn->prepare($historySql);
        if ($stmt) {
            if ($provinceId !== null) {
                $stmt->bind_param('i', $provinceId);
            }
            $stmt->execute();
            $deposits = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
        }
    }

    $conn->close();
} catch (Exception $e) {
    error_log('ERROR in bank_accounts.php: ' . $e->getMessage());
    echo "<p style='color:#FF0000;text-align:center;padding:20px;'>" . htmlspecialchars($e->getMessage()) . "</p>";
    include $baseDir . 'includes/footer.php';
    exit;
}

include $baseDir . 'includes/header.php';
?>
<div class="bank-accounts-container" style="padding: 10px 0 30px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 70vh;">
    <div class="bank-accounts-content" style="background-color: #F4F0E4; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 1200px; width: 95%; margin: 0 auto;">
        <h2 style="color: #4B2F1F; margin-bottom: 10px;">Dépôts Bancaires</h2>
        <p style="color: #000000; font-size: 16px;">Bienvenue, <?php echo htmlspecialchars($_SESSION['full_name'], ENT_QUOTES, 'UTF-8'); ?> (Comptable, <?php echo htmlspecialchars($provinceName, ENT_QUOTES, 'UTF-8'); ?>)</p>

        <?php if ($success): ?>
            <div style="background-color: rgba(76, 175, 80, 0.15); border: 1px solid #2E7D32; color: #1B512D; padding: 12px 16px; border-radius: 8px; margin-bottom: 20px; text-align: center;">
                <?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?>
            </div>
        <?php endif; ?>
        <?php if (!empty($errors)): ?>
            <div style="background-color: rgba(244, 67, 54, 0.15); border: 1px solid #C62828; color: #C62828; padding: 12px 16px; border-radius: 8px; margin-bottom: 20px;">
                                <strong>Erreur:</strong>
                <ul style="margin: 10px 0 0 20px; padding: 0;">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Enregistrer Nouveau Dépôt Bancaire</h3>
        <form method="POST" enctype="multipart/form-data" style="margin-bottom: 30px; background-color: #FFFFFF; border-radius: 10px; padding: 20px; box-shadow: 0 3px 10px rgba(0,0,0,0.1);">
            <div style="margin-bottom: 15px;">
                <label for="bank_account_id" style="color: #4B2F1F; font-weight: 600;">Compte Bancaire</label>
                <select name="bank_account_id" id="bank_account_id" required style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F;">
                    <option value="">Sélectionner compte bancaire</option>
                    <?php foreach ($accounts as $account): ?>
                        <option value="<?php echo (int)$account['id']; ?>" <?php echo ($oldInput['bank_account_id'] !== '' && (int)$oldInput['bank_account_id'] === (int)$account['id']) ? 'selected' : ''; ?>>
                            <?php
                                $label = sprintf('%s · %s · %s',
                                    $account['account_number'],
                                    $account['currency'],
                                    $account['province_id'] ? 'Province ' . $account['province_id'] : 'Toutes les Provinces'
                                );
                                echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8');
                            ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div style="margin-bottom: 15px;">
                <label for="amount" style="color: #4B2F1F; font-weight: 600;">Montant (BIF)</label>
                <input type="number" name="amount" id="amount" min="1" step="1" required value="<?php echo htmlspecialchars($oldInput['amount'], ENT_QUOTES, 'UTF-8'); ?>" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="slip_number" style="color: #4B2F1F; font-weight: 600;">Numéro Bordereau</label>
                <input type="text" name="slip_number" id="slip_number" maxlength="255" required value="<?php echo htmlspecialchars($oldInput['slip_number'], ENT_QUOTES, 'UTF-8'); ?>" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="deposit_date" style="color: #4B2F1F; font-weight: 600;">Date Dépôt</label>
                <input type="datetime-local" name="deposit_date" id="deposit_date" value="<?php echo htmlspecialchars($oldInput['deposit_date'], ENT_QUOTES, 'UTF-8'); ?>" style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F;">
            </div>
            <div style="margin-bottom: 15px;">
                <label for="slip_file" style="color: #4B2F1F; font-weight: 600;">Bordereau de Dépôt (Optionnel, JPG, PNG, PDF, max 5Mo)</label>
                <input type="file" name="slip_file" id="slip_file" accept="application/pdf,image/jpeg,image/png" required style="width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #4B2F1F; background-color: #FFFFFF;">
            </div>
            <button type="submit" name="submit_deposit" style="background-color: #4B2F1F; color: #F4F0E4; padding: 12px 24px; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;">Soumettre Dépôt</button>
        </form>

        <h3 style="color: #4B2F1F; margin-bottom: 10px;">Soldes Bancaires Actuels</h3>
        <p style="color: #000000; font-size: 16px; line-height: 1.6;">
            Solde USD: <strong><?php echo number_format($usdBalance, 2, '.', ','); ?> USD</strong><br>
            Solde BIF: <strong><?php echo number_format($bifBalance, 2, '.', ','); ?> BIF</strong><br>
            Encaisse disponible (<?php echo htmlspecialchars($availableCashLabel, ENT_QUOTES, 'UTF-8'); ?>): <strong><?php echo number_format($availableCashBalance, 2, '.', ','); ?> BIF</strong>
        </p>

        <?php if (!empty($accounts)): ?>
            <div style="overflow-x: auto; margin-top: 15px;">
                <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 10px; overflow: hidden;">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 10px; border: 1px solid #4B2F1F;">Numéro Compte</th>
                            <th style="padding: 10px; border: 1px solid #4B2F1F;">Devise</th>
                            <th style="padding: 10px; border: 1px solid #4B2F1F;">Solde</th>
                            <th style="padding: 10px; border: 1px solid #4B2F1F;">Portée</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($accounts as $account): ?>
                            <tr>
                                <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($account['account_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($account['currency'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 10px; border: 1px solid #4B2F1F;">
                                    <?php
                                        $formatted = $account['currency'] === 'USD'
                                            ? number_format($account['balance'], 2, '.', ',') . ' USD'
                                            : number_format($account['balance'], 2, '.', ',') . ' BIF';
                                        echo $formatted;
                                    ?>
                                </td>
                                <td style="padding: 10px; border: 1px solid #4B2F1F;">
                                    <?php echo $account['province_id'] ? 'Province ' . htmlspecialchars($account['province_id'], ENT_QUOTES, 'UTF-8') : 'Toutes les Provinces'; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="color: #000000; font-size: 16px;">Aucun compte bancaire configuré pour cette province.</p>
        <?php endif; ?>

        <h3 style="color: #4B2F1F; margin-top: 30px; margin-bottom: 10px;">Dépôts Récents</h3>
        <form method="POST" style="margin-bottom: 20px;">
            <button type="submit" name="show_history" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer;">Afficher Historique</button>
        </form>

        <?php if ($showHistory && !empty($deposits)): ?>
            <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Date</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Numéro Bordereau</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Montant (BIF)</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Utilisateur</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Rôle</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Bordereau</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($deposits as $deposit): ?>
                        <tr style="background-color: #FFFFFF;">
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($deposit['deposit_date'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($deposit['slip_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo number_format((float)($deposit['amount'] ?? 0), 2, '.', ','); ?> BIF</td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($deposit['user_name'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($deposit['role_name'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;">
                                <?php if (!empty($deposit['slip_file_path'])): ?>
                                    <a href="<?php echo htmlspecialchars($deposit['slip_file_path'], ENT_QUOTES, 'UTF-8'); ?>" target="_blank" style="color: #4B2F1F; text-decoration: underline;">Voir Bordereau</a>
                                <?php else: ?>
                                    N/A
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php elseif ($showHistory): ?>
            <p style="color: #000000; font-size: 16px;">Aucun dépôt récent trouvé.</p>
        <?php endif; ?>
    </div>
</div>
<?php include $baseDir . 'includes/footer.php'; ?>
