<?php

if (!defined('SECURITY_LOADED')) {
    define('SECURITY_LOADED', true);

    require_once __DIR__ . '/../config/database.php';

    /**
     * Initialize audit logging system
     * Ensures audit_log table exists and is properly configured
     */
    function initialize_audit_system() {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $check_table = $conn->query("SHOW TABLES LIKE 'audit_log'");
            if (!$check_table || $check_table->num_rows == 0) {
                $conn->query("
                    CREATE TABLE IF NOT EXISTS `audit_log` (
                        `id` INT(11) NOT NULL AUTO_INCREMENT PRIMARY KEY,
                        `user_id` INT(11) NOT NULL,
                        `action` VARCHAR(100) NOT NULL,
                        `table_name` VARCHAR(100) DEFAULT NULL,
                        `record_id` INT(11) DEFAULT NULL,
                        `old_values` LONGTEXT DEFAULT NULL,
                        `new_values` LONGTEXT DEFAULT NULL,
                        `ip_address` VARCHAR(45) DEFAULT NULL,
                        `user_agent` VARCHAR(255) DEFAULT NULL,
                        `timestamp` DATETIME DEFAULT CURRENT_TIMESTAMP,
                        `details` LONGTEXT DEFAULT NULL,
                        INDEX idx_user (user_id),
                        INDEX idx_action (action),
                        INDEX idx_timestamp (timestamp)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                ");
                error_log("Created audit_log table");
            }

            $check_col = $conn->query("SHOW COLUMNS FROM audit_log LIKE 'old_values'");
            if (!$check_col || $check_col->num_rows == 0) {
                $conn->query("ALTER TABLE audit_log ADD COLUMN old_values LONGTEXT DEFAULT NULL AFTER record_id");
                $conn->query("ALTER TABLE audit_log ADD COLUMN new_values LONGTEXT DEFAULT NULL AFTER old_values");
                $conn->query("ALTER TABLE audit_log ADD COLUMN ip_address VARCHAR(45) DEFAULT NULL AFTER new_values");
                $conn->query("ALTER TABLE audit_log ADD COLUMN user_agent VARCHAR(255) DEFAULT NULL AFTER ip_address");
            }

            $conn->close();
            return true;
        } catch (Exception $e) {
            error_log("Error initializing audit system: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Log an action to audit trail
     * 
     * @param int $user_id User performing the action
     * @param string $action Action name (e.g., 'INVOICE_CREATED', 'PAYMENT_PROCESSED')
     * @param string $table_name Table affected
     * @param int $record_id Record ID affected
     * @param array $old_values Old record values (for updates/deletes)
     * @param array $new_values New record values (for inserts/updates)
     * @param string $details Additional details
     * @return bool
     */
    function audit_log($user_id, $action, $table_name = null, $record_id = null, $old_values = null, $new_values = null, $details = null) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $ip_address = get_client_ip();
            $user_agent = substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 255);
            $old_json = $old_values ? json_encode($old_values) : null;
            $new_json = $new_values ? json_encode($new_values) : null;

            $stmt = $conn->prepare("
                INSERT INTO audit_log (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent, details)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            if (!$stmt) {
                error_log("Failed to prepare audit log: " . $conn->error);
                return false;
            }

            $stmt->bind_param(
                "issisisss",
                $user_id,
                $action,
                $table_name,
                $record_id,
                $old_json,
                $new_json,
                $ip_address,
                $user_agent,
                $details
            );

            $result = $stmt->execute();
            $stmt->close();
            $conn->close();

            return $result;
        } catch (Exception $e) {
            error_log("Error logging audit: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get client IP address (considers proxies)
     * 
     * @return string Client IP address
     */
    function get_client_ip() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
        }

        return trim(filter_var($ip, FILTER_VALIDATE_IP) ? $ip : 'Invalid IP');
    }

    /**
     * Validate and sanitize currency input
     * Prevents negative amounts and unrealistic values
     * 
     * @param mixed $amount Amount to validate
     * @param int $max_allowed Maximum allowed amount (0 = no limit)
     * @return int|bool Sanitized amount or false if invalid
     */
    function validate_currency($amount, $max_allowed = 0) {
        $amount = filter_var($amount, FILTER_VALIDATE_INT);
        
        if ($amount === false || $amount < 0) {
            return false;
        }

        if ($max_allowed > 0 && $amount > $max_allowed) {
            return false;
        }

        return $amount;
    }

    /**
     * Check for suspicious transaction patterns
     * 
     * @param int $user_id User ID to check
     * @param int $amount Transaction amount
     * @param string $action Action type
     * @return bool True if suspicious, false otherwise
     */
    function is_suspicious_activity($user_id, $amount, $action = 'payment') {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $last_hour = date('Y-m-d H:i:s', strtotime('-1 hour'));

            $stmt = $conn->prepare("
                SELECT COUNT(*) as count, SUM(CAST(JSON_EXTRACT(details, '$.amount') AS SIGNED)) as total
                FROM audit_log 
                WHERE user_id = ? AND action LIKE ? AND timestamp > ?
            ");

            if (!$stmt) return false;

            $like_action = '%' . $action . '%';
            $stmt->bind_param("iss", $user_id, $like_action, $last_hour);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            $conn->close();

            $transaction_count = $result['count'] ?? 0;
            $total_amount = $result['total'] ?? 0;

            if ($transaction_count > 50 || $total_amount > 10000000) {
                error_log("Suspicious activity detected: User $user_id has $transaction_count transactions totaling $total_amount in last hour");
                return true;
            }

            return false;
        } catch (Exception $e) {
            error_log("Error checking suspicious activity: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Verify transaction authorization
     * Ensures only authorized users can perform actions on specific records
     * 
     * @param int $user_id User performing action
     * @param string $table_name Table being accessed
     * @param int $record_id Record ID
     * @return bool
     */
    function verify_transaction_authorization($user_id, $table_name, $record_id) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $user_role = get_user_role($user_id);
            
            if (in_array($user_role, ['Admin', 'DG', 'DGA'])) {
                $conn->close();
                return true;
            }

            if ($table_name === 'paid_invoices') {
                $stmt = $conn->prepare("SELECT created_by FROM paid_invoices WHERE id = ?");
                $stmt->bind_param("i", $record_id);
                $stmt->execute();
                $invoice = $stmt->get_result()->fetch_assoc();
                $stmt->close();
                $conn->close();

                return isset($invoice) && $invoice['created_by'] == $user_id;
            }

            if ($table_name === 'cashier_balance') {
                $stmt = $conn->prepare("SELECT cashier_id FROM cashier_balance WHERE id = ?");
                $stmt->bind_param("i", $record_id);
                $stmt->execute();
                $balance = $stmt->get_result()->fetch_assoc();
                $stmt->close();
                $conn->close();

                return isset($balance) && $balance['cashier_id'] == $user_id;
            }

            $conn->close();
            return false;
        } catch (Exception $e) {
            error_log("Error verifying authorization: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get user role
     * 
     * @param int $user_id User ID
     * @return string|null Role name
     */
    function get_user_role($user_id) {
        $conn = get_db_connection();
        if (!$conn) return null;

        try {
            $stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
            if (!$stmt) return null;

            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            $conn->close();

            return $result['name'] ?? null;
        } catch (Exception $e) {
            error_log("Error getting user role: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Prevent double transaction submission
     * Uses token-based approach
     * 
     * @return string Transaction token
     */
    function get_transaction_token() {
        if (empty($_SESSION['transaction_tokens'])) {
            $_SESSION['transaction_tokens'] = [];
        }

        $token = bin2hex(random_bytes(32));
        $_SESSION['transaction_tokens'][$token] = time();

        $_SESSION['current_token'] = $token;
        return $token;
    }

    /**
     * Verify and consume transaction token
     * 
     * @param string $token Token to verify
     * @return bool True if valid and not used, false otherwise
     */
    function verify_transaction_token($token) {
        if (empty($_SESSION['transaction_tokens']) || !isset($_SESSION['transaction_tokens'][$token])) {
            error_log("Invalid transaction token provided");
            return false;
        }

        $timestamp = $_SESSION['transaction_tokens'][$token];

        if (time() - $timestamp > 3600) {
            error_log("Transaction token expired");
            unset($_SESSION['transaction_tokens'][$token]);
            return false;
        }

        unset($_SESSION['transaction_tokens'][$token]);
        return true;
    }

    /**
     * Validate user has permission for action
     * 
     * @param int $user_id User ID
     * @param string $permission Permission string
     * @return bool
     */
    function user_has_permission($user_id, $permission) {
        $role = get_user_role($user_id);

        $permission_map = [
            'approve_invoice' => ['Admin', 'DG', 'DGA'],
            'process_payment' => ['Caissier_Comptable'],
            'manage_stock' => ['Stock Manager', 'Stock Manager Assistant'],
            'manage_users' => ['Admin'],
            'manage_expenses' => ['Admin', 'Finance', 'Accountant'],
            'view_reports' => ['Admin', 'DG', 'DGA', 'Finance'],
        ];

        if (!isset($permission_map[$permission])) {
            return false;
        }

        return in_array($role, $permission_map[$permission]);
    }

    /**
     * Log payment transaction
     * 
     * @param int $user_id User processing payment
     * @param int $invoice_id Invoice ID
     * @param int $amount Amount paid
     * @param string $method Payment method
     * @return bool
     */
    function log_payment_transaction($user_id, $invoice_id, $amount, $method) {
        return audit_log(
            $user_id,
            'PAYMENT_PROCESSED',
            'paid_invoices',
            $invoice_id,
            null,
            ['amount' => $amount, 'method' => $method],
            'Payment processed for invoice #' . $invoice_id
        );
    }

    /**
     * Log cash handling transaction
     * 
     * @param int $user_id User
     * @param int $shift_id Shift ID
     * @param int $amount Amount handled
     * @param string $type Type (deposit/withdrawal)
     * @return bool
     */
    function log_cash_transaction($user_id, $shift_id, $amount, $type) {
        return audit_log(
            $user_id,
            'CASH_' . strtoupper($type),
            'cashier_balance',
            $shift_id,
            null,
            ['amount' => $amount, 'type' => $type],
            'Cash ' . $type . ' of ' . number_format($amount) . ' BIF'
        );
    }

    initialize_audit_system();
}
