<?php
/**
 * ==========================================================
 * Masunzu Bar Hotel - Global Utility & Helper Functions
 * ----------------------------------------------------------
 * Author: Masunzu Dev Team
 * Description: Core reusable PHP functions for database, 
 *              user roles, formatting, and logging.
 * ==========================================================
 */

// ---------------------------------------------------------------------------
//  Prevent multiple inclusions
// ---------------------------------------------------------------------------
if (defined('FUNCTIONS_PHP_LOADED')) {
    error_log("functions.php: Already included, skipping from " . debug_backtrace()[0]['file']);
    return;
}
define('FUNCTIONS_PHP_LOADED', true);
error_log("functions.php: Loaded from " . debug_backtrace()[0]['file']);

// ---------------------------------------------------------------------------
//  Include core dependencies
// ---------------------------------------------------------------------------
require_once __DIR__ . '/../config/database.php';

// ---------------------------------------------------------------------------
//  Language loader (English fallback)
// ---------------------------------------------------------------------------
$lang_file = '/opt/lampp/htdocs/masunzu_bar_hotel/lang/en.php';
if (file_exists($lang_file)) {
    $lang = require_once $lang_file;
} else {
    error_log("Language file $lang_file not found, using default English strings");
    $lang = [
        'login' => 'Login',
        'username' => 'Username',
        'password' => 'Password',
        'submit' => 'Submit',
        'welcome' => 'Welcome',
        'dashboard' => 'Dashboard',
        'logout' => 'Logout',
    ];
}

// ---------------------------------------------------------------------------
//  Redirection Helper
// ---------------------------------------------------------------------------

/**
 * Redirect to a specific URL and terminate execution.
 *
 * @param string $url Target URL for redirection
 * @return void
 */
if (!function_exists('redirect')) {
    function redirect(string $url): void {
        header("Location: $url");
        exit;
    }
}

// ---------------------------------------------------------------------------
//  User Role Utilities
// ---------------------------------------------------------------------------

/**
 * Retrieve the role name of a given user.
 *
 * @param int $user_id User ID
 * @return string|null Role name, or null on failure
 */
if (!function_exists('get_user_role_name')) {
    function get_user_role_name($user_id) {
        $conn = get_db_connection();
        if (!$conn) {
            error_log("Database connection failed in get_user_role_name");
            return null;
        }

        $stmt = $conn->prepare("
            SELECT r.name 
            FROM user u 
            INNER JOIN role r ON u.role_id = r.id 
            WHERE u.id = ?
        ");
        if (!$stmt) {
            error_log("Query preparation failed in get_user_role_name: " . $conn->error);
            $conn->close();
            return null;
        }

        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $role = $result->fetch_assoc();
        $stmt->close();
        $conn->close();

        return $role['name'] ?? null;
    }
}

if (!function_exists('mbh_get_app_config')) {
    function mbh_get_app_config(): array
    {
        static $appConfig = null;
        if ($appConfig === null) {
            $configPath = __DIR__ . '/../config/app.php';
            $appConfig = file_exists($configPath) ? require $configPath : [];
        }
        return $appConfig;
    }
}

if (!function_exists('mbh_get_role_metadata')) {
    function mbh_get_role_metadata(): array
    {
        $config = mbh_get_app_config();
        return $config['roles'] ?? [];
    }
}

if (!function_exists('mbh_get_role_aliases')) {
    function mbh_get_role_aliases(): array
    {
        $config = mbh_get_app_config();
        return $config['role_aliases'] ?? [];
    }
}

if (!function_exists('mbh_normalize_role_name')) {
    function mbh_normalize_role_name(?string $roleName): ?string
    {
        if ($roleName === null) {
            return null;
        }
        $metadata = mbh_get_role_metadata();
        if (isset($metadata[$roleName])) {
            return $roleName;
        }
        $aliases = mbh_get_role_aliases();
        if (isset($aliases[$roleName]) && isset($metadata[$aliases[$roleName]])) {
            return $aliases[$roleName];
        }
        return $roleName;
    }
}

if (!function_exists('mbh_get_role_label')) {
    function mbh_get_role_label(?string $roleName): string
    {
        if ($roleName === null) {
            return '';
        }
        $normalized = mbh_normalize_role_name($roleName);
        $metadata = mbh_get_role_metadata();
        return $metadata[$normalized]['label'] ?? $roleName;
    }
}

if (!function_exists('mbh_get_role_departments')) {
    function mbh_get_role_departments(?string $roleName): array
    {
        if ($roleName === null) {
            return [];
        }
        $normalized = mbh_normalize_role_name($roleName);
        $metadata = mbh_get_role_metadata();
        return $metadata[$normalized]['departments'] ?? [];
    }
}

if (!function_exists('mbh_role_requires_depot')) {
    function mbh_role_requires_depot(?string $roleName): bool
    {
        if ($roleName === null) {
            return false;
        }
        $normalized = mbh_normalize_role_name($roleName);
        $metadata = mbh_get_role_metadata();
        return !empty($metadata[$normalized]['requires_depot']);
    }
}

if (!function_exists('mbh_role_requires_region')) {
    function mbh_role_requires_region(?string $roleName): bool
    {
        if ($roleName === null) {
            return false;
        }
        $normalized = mbh_normalize_role_name($roleName);
        $metadata = mbh_get_role_metadata();
        return !empty($metadata[$normalized]['requires_region']);
    }
}

if (!function_exists('mbh_get_department_metadata')) {
    function mbh_get_department_metadata(): array
    {
        $config = mbh_get_app_config();
        return $config['departments'] ?? [];
    }
}

/**
 * Get the accountant cash balance for a specific province or for all provinces.
 * Returns 0 if the province cash tables are missing.
 */
if (!function_exists('mbh_get_province_cash_balance')) {
    function mbh_get_province_cash_balance(mysqli $conn, ?int $provinceId = null): float
    {
        $tableCheck = $conn->query("SHOW TABLES LIKE 'province_cash_account'");
        if (!$tableCheck || $tableCheck->num_rows === 0) {
            if ($tableCheck instanceof mysqli_result) {
                $tableCheck->free();
            }
            return 0.0;
        }
        $tableCheck->free();

        if ($provinceId !== null) {
            $stmt = $conn->prepare('SELECT balance FROM province_cash_account WHERE province_id = ?');
            if (!$stmt) {
                error_log('mbh_get_province_cash_balance: failed to prepare balance query - ' . $conn->error);
                return 0.0;
            }
            $stmt->bind_param('i', $provinceId);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            return (float)($row['balance'] ?? 0);
        }

        $stmt = $conn->prepare('SELECT IFNULL(SUM(balance), 0) AS total FROM province_cash_account');
        if (!$stmt) {
            error_log('mbh_get_province_cash_balance: failed to prepare sum query - ' . $conn->error);
            return 0.0;
        }
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        return (float)($row['total'] ?? 0);
    }
}

if (!function_exists('mbh_sync_user_departments')) {
    function mbh_sync_user_departments(mysqli $conn, int $userId, array $departmentCodes): void
    {
        $departmentCodes = array_values(array_unique(array_filter($departmentCodes)));
        $departmentMeta = mbh_get_department_metadata();
        $validCodes = array_values(array_intersect($departmentCodes, array_keys($departmentMeta)));

        $stmtDelete = $conn->prepare('DELETE FROM user_department WHERE user_id = ?');
        if ($stmtDelete) {
            $stmtDelete->bind_param('i', $userId);
            $stmtDelete->execute();
            $stmtDelete->close();
        }

        if (empty($validCodes)) {
            return;
        }

        $placeholders = implode(',', array_fill(0, count($validCodes), '?'));
        $codeStmt = $conn->prepare("SELECT id, code FROM department WHERE code IN ($placeholders)");
        if (!$codeStmt) {
            error_log('mbh_sync_user_departments: failed to prepare department lookup - ' . $conn->error);
            return;
        }
        $types = str_repeat('s', count($validCodes));
        $codeStmt->bind_param($types, ...$validCodes);
        $codeStmt->execute();
        $result = $codeStmt->get_result();
        $departmentIds = [];
        while ($row = $result->fetch_assoc()) {
            $departmentIds[] = (int)$row['id'];
        }
        $codeStmt->close();

        if (empty($departmentIds)) {
            return;
        }

        $insertStmt = $conn->prepare('INSERT INTO user_department (user_id, department_id) VALUES (?, ?)');
        if (!$insertStmt) {
            error_log('mbh_sync_user_departments: failed to prepare insert - ' . $conn->error);
            return;
        }
        foreach ($departmentIds as $departmentId) {
            $insertStmt->bind_param('ii', $userId, $departmentId);
            $insertStmt->execute();
        }
        $insertStmt->close();
    }
}

// ---------------------------------------------------------------------------
//  Quantity Formatting Utilities
// ---------------------------------------------------------------------------

/**
 * Format quantities as crates only.
 *
 * @param int $quantity Number of crates being represented
 * @param string $unit_type Legacy parameter retained for backward compatibility
 * @param int $crate_quantity Legacy parameter retained for backward compatibility
 * @return string Formatted string (e.g., "24 crates")
 */
if (!function_exists('format_quantity')) {
    function format_quantity(int $quantity, string $unit_type = 'crates', int $crate_quantity = 12): string {
        unset($unit_type, $crate_quantity); // Parameters kept for backwards compatibility
        return number_format((int)$quantity, 0, ',', '.') . ' crates';
    }
}

// ---------------------------------------------------------------------------
//  Logging Utility
// ---------------------------------------------------------------------------

/**
 * Ensure the `log` table has a primary key and AUTO_INCREMENT on `id`.
 * Prevents "Field 'id' doesn't have a default value" during inserts.
 */
if (!function_exists('ensure_log_table_schema')) {
    function ensure_log_table_schema(mysqli $conn): void {
        static $schema_checked = false;
        if ($schema_checked) {
            return;
        }

        $dbResult = $conn->query("SELECT DATABASE() AS db_name");
        $dbName = $dbResult ? ($dbResult->fetch_assoc()['db_name'] ?? null) : null;
        if (!$dbName) {
            error_log("ensure_log_table_schema: Unable to determine database name");
            return;
        }

        // Make sure there is a primary key on id.
        if ($stmtPk = $conn->prepare("
            SELECT 1 
            FROM information_schema.TABLE_CONSTRAINTS 
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'log' AND CONSTRAINT_TYPE = 'PRIMARY KEY' 
            LIMIT 1
        ")) {
            $stmtPk->bind_param("s", $dbName);
            $stmtPk->execute();
            $hasPk = $stmtPk->get_result()->num_rows > 0;
            $stmtPk->close();
            if (!$hasPk) {
                if (!$conn->query("ALTER TABLE log ADD PRIMARY KEY (id)")) {
                    error_log("ensure_log_table_schema: Failed to add primary key on log.id - " . $conn->error);
                }
            }
        } else {
            error_log("ensure_log_table_schema: Unable to prepare PK check - " . $conn->error);
        }

        // Ensure id is AUTO_INCREMENT so inserts work without explicitly providing it.
        if ($stmtAuto = $conn->prepare("
            SELECT EXTRA 
            FROM information_schema.COLUMNS 
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'log' AND COLUMN_NAME = 'id' 
            LIMIT 1
        ")) {
            $stmtAuto->bind_param("s", $dbName);
            $stmtAuto->execute();
            $extra = $stmtAuto->get_result()->fetch_assoc()['EXTRA'] ?? '';
            $stmtAuto->close();

            if (stripos($extra, 'auto_increment') === false) {
                if (!$conn->query("ALTER TABLE log MODIFY id int(11) NOT NULL AUTO_INCREMENT")) {
                    error_log("ensure_log_table_schema: Failed to set AUTO_INCREMENT on log.id - " . $conn->error);
                }
            }
        } else {
            error_log("ensure_log_table_schema: Unable to prepare AUTO_INCREMENT check - " . $conn->error);
        }

        $schema_checked = true;
    }
}

/**
 * Insert a notification for a specific user, ensuring the table exists.
 */
if (!function_exists('mbh_notify_user')) {
    function mbh_notify_user(mysqli $conn, int $userId, string $message, ?string $link = null): bool {
        static $tableEnsured = false;
        if (!$tableEnsured) {
            $sql = "
                CREATE TABLE IF NOT EXISTS user_notification (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT NOT NULL,
                    message VARCHAR(255) NOT NULL,
                    link VARCHAR(255) DEFAULT NULL,
                    is_read TINYINT(1) NOT NULL DEFAULT 0,
                    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    INDEX idx_user_read (user_id, is_read),
                    CONSTRAINT fk_notification_user_general FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
            ";
            if (!$conn->query($sql)) {
                error_log('mbh_notify_user: failed to ensure user_notification table - ' . $conn->error);
                return false;
            }
            $tableEnsured = true;
        }

        $stmt = $conn->prepare("INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())");
        if (!$stmt) {
            error_log('mbh_notify_user: prepare failed - ' . $conn->error);
            return false;
        }
        $stmt->bind_param('iss', $userId, $message, $link);
        $ok = $stmt->execute();
        if (!$ok) {
            error_log('mbh_notify_user: execute failed - ' . $stmt->error);
        }
        $stmt->close();
        return $ok;
    }
}

if (!function_exists('mbh_ensure_loan_ledger_entry_origin')) {
    function mbh_ensure_loan_ledger_entry_origin(mysqli $conn): void {
        static $ensured = false;
        if ($ensured) {
            return;
        }
        $result = $conn->query("SHOW COLUMNS FROM customer_loan_ledger LIKE 'entry_origin'");
        if ($result && $result->num_rows === 0) {
            $conn->query("ALTER TABLE customer_loan_ledger ADD COLUMN entry_origin VARCHAR(32) NOT NULL DEFAULT 'system' AFTER entry_type");
        }
        $ensured = true;
    }
}

if (!function_exists('mbh_get_loan_outstanding_sql')) {
    function mbh_get_loan_outstanding_sql(string $customerAlias = 'c', string $invoiceAlias = 'ui'): string
    {
        $loanCondition = "{$invoiceAlias}.is_loan_sale = 1 AND {$invoiceAlias}.loan_outstanding_bif > 0 AND ({$invoiceAlias}.approval_status IS NULL OR {$invoiceAlias}.approval_status != 'rejected')";
        $loanSumExpr = "COALESCE(SUM(CASE WHEN {$loanCondition} THEN {$invoiceAlias}.loan_outstanding_bif ELSE 0 END), 0)";
        return "COALESCE(GREATEST(COALESCE({$customerAlias}.loan_balance_bif, 0), {$loanSumExpr}), 0)";
    }
}

// ---------------------------------------------------------------------------
//  Region Stock Utilities
// ---------------------------------------------------------------------------

if (!function_exists('ensure_region_stock_table')) {
    function ensure_region_stock_table(mysqli $conn): void {
        static $ensured = false;
        if ($ensured) {
            return;
        }

        $sql = "
            CREATE TABLE IF NOT EXISTS region_stock (
                id INT AUTO_INCREMENT PRIMARY KEY,
                region_id INT NOT NULL,
                product_id INT NOT NULL,
                quantity INT NOT NULL DEFAULT 0,
                reserved_quantity INT NOT NULL DEFAULT 0,
                unit_type ENUM('crates') NOT NULL DEFAULT 'crates',
                last_updated DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY uniq_region_product (region_id, product_id),
                KEY idx_region_stock_product (product_id),
                CONSTRAINT region_stock_ibfk_1 FOREIGN KEY (region_id) REFERENCES region(id) ON DELETE CASCADE,
                CONSTRAINT region_stock_ibfk_2 FOREIGN KEY (product_id) REFERENCES product(id) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
        ";

        if (!$conn->query($sql)) {
            error_log("ensure_region_stock_table: failed to ensure table - " . $conn->error);
            return;
        }

        $ensured = true;
    }
}

if (!function_exists('mbh_increment_region_stock')) {
    function mbh_increment_region_stock(mysqli $conn, int $regionId, int $productId, int $quantity): bool {
        ensure_region_stock_table($conn);
        if ($quantity <= 0) {
            return true;
        }

        $stmt = $conn->prepare("
            INSERT INTO region_stock (region_id, product_id, quantity, reserved_quantity, unit_type, last_updated)
            VALUES (?, ?, ?, 0, 'crates', NOW())
            ON DUPLICATE KEY UPDATE quantity = quantity + VALUES(quantity), last_updated = NOW()
        ");
        if (!$stmt) {
            error_log("mbh_increment_region_stock: prepare failed - " . $conn->error);
            return false;
        }
        $stmt->bind_param("iii", $regionId, $productId, $quantity);
        $result = $stmt->execute();
        if (!$result) {
            error_log("mbh_increment_region_stock: execute failed - " . $stmt->error);
        }
        $stmt->close();
        return $result;
    }
}

if (!function_exists('mbh_decrement_region_stock')) {
    function mbh_decrement_region_stock(mysqli $conn, int $regionId, int $productId, int $quantity): bool {
        ensure_region_stock_table($conn);
        if ($quantity <= 0) {
            return true;
        }

        $stmt = $conn->prepare("
            UPDATE region_stock
            SET quantity = quantity - ?, last_updated = NOW()
            WHERE region_id = ? AND product_id = ? AND quantity >= ?
        ");
        if (!$stmt) {
            error_log("mbh_decrement_region_stock: prepare failed - " . $conn->error);
            return false;
        }
        $stmt->bind_param("iiii", $quantity, $regionId, $productId, $quantity);
        $stmt->execute();
        $affected = $stmt->affected_rows;
        if ($affected <= 0) {
            error_log("mbh_decrement_region_stock: insufficient stock for region {$regionId}, product {$productId}");
        }
        $stmt->close();
        return $affected > 0;
    }
}

if (!function_exists('mbh_get_region_stock_map')) {
    function mbh_get_region_stock_map(mysqli $conn, int $regionId): array {
        ensure_region_stock_table($conn);
        $stmt = $conn->prepare("SELECT product_id, quantity FROM region_stock WHERE region_id = ?");
        if (!$stmt) {
            error_log("mbh_get_region_stock_map: prepare failed - " . $conn->error);
            return [];
        }
        $stmt->bind_param("i", $regionId);
        $stmt->execute();
        $result = $stmt->get_result();
        $stock = [];
        while ($row = $result->fetch_assoc()) {
            $stock[(int)$row['product_id']] = (int)$row['quantity'];
        }
        $stmt->close();
        return $stock;
    }
}


/**
 * Log a user action into the `log` table with optional product details.
 *
 * @param int $user_id The ID of the user performing the action
 * @param string $action The type of action performed
 * @param string $description Description of the action
 * @param int|null $product_id Optional product ID
 * @param int|null $quantity Optional quantity value
 * @param string|null $unit_type Optional unit type
 * @param int $crate_quantity Bottles per crate (default 12)
 * @return void
 */
if (!function_exists('log_action')) {
    function log_action(int $user_id, string $action, string $description, ?int $product_id = null, ?int $quantity = null, ?string $unit_type = null, int $crate_quantity = 12): void {
        $conn = get_db_connection();
        if (!$conn) {
            error_log("Database connection failed in log_action");
            return;
        }

        ensure_log_table_schema($conn);

        // Append formatted quantity if provided
        if ($product_id !== null && $quantity !== null && $unit_type !== null) {
            $formatted_quantity = format_quantity($quantity, $unit_type, $crate_quantity);
            $description .= " ($formatted_quantity)";
        }

        $stmt = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, ?, ?, NOW())");
        if (!$stmt) {
            error_log("Query preparation failed in log_action: " . $conn->error);
            $conn->close();
            return;
        }

        $stmt->bind_param("iss", $user_id, $action, $description);
        $stmt->execute();
        $stmt->close();
        $conn->close();
    }
}

if (!function_exists('ensure_user_history_table')) {
    function ensure_user_history_table(mysqli $conn): void
    {
        static $historyEnsured = false;
        if ($historyEnsured) {
            return;
        }

        $sql = "
            CREATE TABLE IF NOT EXISTS user_change_history (
                id INT AUTO_INCREMENT PRIMARY KEY,
                target_user_id INT NOT NULL,
                action ENUM('create','edit') NOT NULL,
                changed_by INT NOT NULL,
                change_details LONGTEXT NOT NULL,
                created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_target_user (target_user_id),
                INDEX idx_action (action),
                INDEX idx_changed_by (changed_by),
                INDEX idx_created_at (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
        ";
        if (!$conn->query($sql)) {
            error_log('Failed to ensure user_change_history table: ' . $conn->error);
        }

        $historyEnsured = true;
    }
}

if (!function_exists('record_user_change')) {
    /**
     * Persist a user creation or edit event with serialized change details.
     *
     * @param int   $targetUserId User that was created or modified
     * @param string $action      'create' or 'edit'
     * @param int   $changedBy    User who executed the action
     * @param array $changes      Details about the change (will be JSON encoded)
     */
    function record_user_change(int $targetUserId, string $action, int $changedBy, array $changes): void
    {
        $conn = get_db_connection();
        if (!$conn) {
            error_log('record_user_change: unable to obtain DB connection');
            return;
        }

        ensure_user_history_table($conn);

        $details = json_encode($changes, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($details === false) {
            $details = json_encode(['raw' => $changes]);
        }

        $stmt = $conn->prepare("INSERT INTO user_change_history (target_user_id, action, changed_by, change_details, created_at) VALUES (?, ?, ?, ?, NOW())");
        if ($stmt) {
            $stmt->bind_param('isis', $targetUserId, $action, $changedBy, $details);
            if (!$stmt->execute()) {
                error_log('record_user_change: failed to insert - ' . $stmt->error);
            }
            $stmt->close();
        } else {
            error_log('record_user_change: failed to prepare insert - ' . $conn->error);
        }
        $conn->close();
    }
}

if (!function_exists('mbh_get_stock_manager_ids_by_province')) {
    /**
     * Return active stock manager user IDs for a province.
     *
     * @param mysqli $conn
     * @param int $provinceId
     * @return array<int>
     */
    function mbh_get_stock_manager_ids_by_province(mysqli $conn, int $provinceId): array
    {
        $ids = [];
        if ($stmt = $conn->prepare("
            SELECT u.id
            FROM user u
            JOIN role r ON u.role_id = r.id
            WHERE r.name = 'Stock Manager' AND u.is_active = 1 AND u.province_id = ?
        ")) {
            $stmt->bind_param('i', $provinceId);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $ids[] = (int)$row['id'];
            }
            $stmt->close();
        }
        return $ids;
    }
}

/**
 * Get company info and depot name for receipts/invoices.
 * Company info is now shared across all depots (single record).
 * Depot name is fetched separately based on province_id.
 * 
 * @param mysqli $conn Database connection
 * @param int|null $provinceId Province ID for depot name (null for no depot)
 * @return array Company info with depot_name key added
 */
if (!function_exists('mbh_get_company_info_for_receipt')) {
    function mbh_get_company_info_for_receipt(mysqli $conn, ?int $provinceId = null): array
    {
        // Default company info
        $company_info = [
            'company_name' => 'Masunzu Bar Hotel',
            'address' => '',
            'phone1' => '',
            'phone2' => '',
            'email' => '',
            'nif' => '',
            'rc' => '',
            'website' => '',
            'logo_path' => null,
            'depot_name' => ''
        ];
        
        // Fetch the single shared company info record
        $stmt = $conn->prepare("SELECT company_name, address, phone1, phone2, email, nif, rc, website, logo_path FROM company_info ORDER BY id ASC LIMIT 1");
        if ($stmt) {
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            if ($result) {
                $company_info = array_merge($company_info, $result);
            }
            $stmt->close();
        }
        
        // Fetch depot (province) name if province_id is provided
        if ($provinceId !== null && $provinceId > 0) {
            $stmtDepot = $conn->prepare("SELECT name FROM province WHERE id = ? LIMIT 1");
            if ($stmtDepot) {
                $stmtDepot->bind_param("i", $provinceId);
                $stmtDepot->execute();
                $depotResult = $stmtDepot->get_result()->fetch_assoc();
                if ($depotResult) {
                    $company_info['depot_name'] = $depotResult['name'];
                }
                $stmtDepot->close();
            }
        }
        
        return $company_info;
    }
}
?>
