<?php

if (!defined('EXPORT_HELPERS_LOADED')) {
    define('EXPORT_HELPERS_LOADED', true);

    /**
     * Output PDF headers to open in new tab
     * 
     * @param string $filename Filename for the PDF
     */
    function send_pdf_headers($filename) {
        header('Content-Type: application/pdf');
        header('Content-Disposition: inline; filename="' . basename($filename) . '"');
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
    }

    /**
     * Output CSV headers for download
     * 
     * @param string $filename Filename for the CSV
     */
    function send_csv_headers($filename) {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . basename($filename) . '"');
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
    }

    /**
     * Export table data to CSV
     * 
     * @param array $headers Column headers
     * @param array $rows Data rows
     * @param string $filename Output filename
     * @return bool
     */
    function export_to_csv($headers, $rows, $filename) {
        try {
            send_csv_headers($filename);

            $output = fopen('php://output', 'w');
            if (!$output) return false;

            fputcsv($output, $headers);

            foreach ($rows as $row) {
                fputcsv($output, $row);
            }

            fclose($output);
            return true;
        } catch (Exception $e) {
            error_log("Error exporting CSV: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Safe CSV value - escapes special characters
     * 
     * @param mixed $value Value to escape
     * @return string Escaped value
     */
    function safe_csv_value($value) {
        if ($value === null) return '';
        if (is_bool($value)) return $value ? 'Yes' : 'No';
        if (is_array($value)) return json_encode($value);
        
        $str = (string)$value;
        
        if (strpos($str, '"') !== false || strpos($str, ',') !== false || strpos($str, "\n") !== false) {
            return '"' . str_replace('"', '""', $str) . '"';
        }
        
        return $str;
    }

    /**
     * Export data with numbering
     * 
     * @param array $headers Column headers (with # for row number)
     * @param array $rows Data rows
     * @param string $filename Output filename
     * @return bool
     */
    function export_numbered_data($headers, $rows, $filename) {
        try {
            send_csv_headers($filename);

            $output = fopen('php://output', 'w');
            if (!$output) return false;

            $output_headers = [];
            foreach ($headers as $i => $header) {
                if ($header === '#') {
                    $output_headers[] = 'No.';
                } else {
                    $output_headers[] = $header;
                }
            }

            fputcsv($output, $output_headers);

            foreach ($rows as $index => $row) {
                $row_data = [$index + 1];
                foreach ($row as $value) {
                    $row_data[] = safe_csv_value($value);
                }
                fputcsv($output, $row_data);
            }

            fclose($output);
            return true;
        } catch (Exception $e) {
            error_log("Error exporting numbered data: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Generate JavaScript for opening PDF in new tab
     * Use this in your form/button to open PDF in new tab
     * 
     * @param string $form_action Form action URL
     * @param string $button_id Button element ID
     * @return string JavaScript code
     */
    function get_pdf_new_tab_script($form_action, $button_id) {
        return "
        <script>
            document.getElementById('{$button_id}').addEventListener('click', function(e) {
                e.preventDefault();
                const form = this.closest('form');
                if (!form) return;
                
                const target = form.getAttribute('target') || '_blank';
                const method = form.getAttribute('method') || 'POST';
                const originalAction = form.getAttribute('action');
                const originalTarget = form.getAttribute('target');
                
                form.setAttribute('action', '{$form_action}');
                form.setAttribute('target', '_blank');
                
                form.submit();
                
                form.setAttribute('action', originalAction);
                if (originalTarget) {
                    form.setAttribute('target', originalTarget);
                } else {
                    form.removeAttribute('target');
                }
            });
        </script>
        ";
    }

    /**
     * Get array of commonly needed table exports with filters
     * 
     * @param string $table_name Table name
     * @param array $filters Optional filters
     * @return array Export metadata
     */
    function get_export_info($table_name) {
        $exports = [
            'paid_invoices' => [
                'filename' => 'invoices_' . date('Y-m-d'),
                'headers' => ['#', 'Invoice #', 'Customer', 'Amount', 'Status', 'Date', 'Payment Method'],
                'columns' => ['invoice_number', 'customer_name', 'paid_amount', 'status', 'created_at', 'payment_method'],
            ],
            'loan_payments' => [
                'filename' => 'loan_payments_' . date('Y-m-d'),
                'headers' => ['#', 'Date', 'Customer', 'Amount', 'Method', 'Notes'],
                'columns' => ['payment_date', 'customer_name', 'payment_amount', 'payment_method', 'notes'],
            ],
            'expenses' => [
                'filename' => 'expenses_' . date('Y-m-d'),
                'headers' => ['#', 'ID', 'Amount', 'Method', 'Recipient', 'Caissier_Comptable', 'Date'],
                'columns' => ['id', 'amount', 'payment_method', 'recipient_name', 'cashier_name', 'created_at'],
            ],
            'products' => [
                'filename' => 'products_' . date('Y-m-d'),
                'headers' => ['#', 'Product Name', 'Crate Quantity', 'Price / Crate (BIF)'],
                'columns' => ['name', 'crate_quantity', 'price_per_crate'],
            ],
            'requisitions' => [
                'filename' => 'requisitions_' . date('Y-m-d'),
                'headers' => ['#', 'ID', 'Initiator', 'Recipient', 'Status', 'Amount', 'Date'],
                'columns' => ['id', 'initiator_name', 'recipient_name', 'status', 'total_amount', 'created_at'],
            ],
        ];

        return $exports[$table_name] ?? null;
    }
}
