<?php

if (!defined('DASHBOARD_METRICS_LOADED')) {
    define('DASHBOARD_METRICS_LOADED', true);

    require_once __DIR__ . '/../config/database.php';

    /**
     * Get expired invoice metrics for the period
     * 
     * @param int|null $province_id Optional province filter
     * @param string $date_from Start date (Y-m-d)
     * @param string $date_to End date (Y-m-d)
     * @return array Metrics including count and total amount
     */
    function get_expired_invoice_metrics($province_id = null, $date_from = null, $date_to = null) {
        $conn = get_db_connection();
        if (!$conn) return ['count' => 0, 'total_amount' => 0, 'list' => []];

        try {
            if (!$date_from) $date_from = date('Y-m-d', strtotime('-7 days'));
            if (!$date_to) $date_to = date('Y-m-d');

            $from_dt = $date_from . ' 00:00:00';
            $to_dt = $date_to . ' 23:59:59';

            $query = "
                SELECT COUNT(*) as count, COALESCE(SUM(paid_amount), 0) as total_amount
                FROM paid_invoices
                WHERE expiration_status = 'expired'
                AND created_at BETWEEN ? AND ?
            ";

            if ($province_id) {
                $query .= " AND province_id = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ssi", $from_dt, $to_dt, $province_id);
            } else {
                $stmt = $conn->prepare($query);
                $stmt->bind_param("ss", $from_dt, $to_dt);
            }

            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            $list_query = "
                SELECT id, invoice_number, customer_name, paid_amount, created_at, customer_id
                FROM paid_invoices
                WHERE expiration_status = 'expired'
                AND created_at BETWEEN ? AND ?
            ";

            if ($province_id) {
                $list_query .= " AND province_id = ?";
                $list_stmt = $conn->prepare($list_query);
                $list_stmt->bind_param("ssi", $from_dt, $to_dt, $province_id);
            } else {
                $list_stmt = $conn->prepare($list_query);
                $list_stmt->bind_param("ss", $from_dt, $to_dt);
            }

            $list_stmt->execute();
            $list = $list_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $list_stmt->close();

            $conn->close();

            return [
                'count' => (int)($result['count'] ?? 0),
                'total_amount' => (int)($result['total_amount'] ?? 0),
                'list' => $list
            ];
        } catch (Exception $e) {
            error_log("Error getting expired invoice metrics: " . $e->getMessage());
            return ['count' => 0, 'total_amount' => 0, 'list' => []];
        }
    }

    /**
     * Get loan repayment metrics
     * 
     * @param int|null $province_id Optional province filter
     * @return array Metrics
     */
    function get_loan_metrics($province_id = null) {
        $conn = get_db_connection();
        if (!$conn) return [
            'total_outstanding' => 0,
            'total_due' => 0,
            'active_loans' => 0,
            'defaulted_loans' => 0,
            'on_time_loans' => 0,
            'late_loans' => 0,
            'late_borrowers' => []
        ];

        try {
            $query = "SELECT 
                        COALESCE(SUM(loan_balance_bif), 0) as total_outstanding,
                        COUNT(*) as total_loans,
                        SUM(CASE WHEN loan_status = 'active' THEN 1 ELSE 0 END) as active_loans,
                        SUM(CASE WHEN loan_status = 'defaulted' THEN 1 ELSE 0 END) as defaulted_loans,
                        SUM(CASE WHEN loan_status = 'active' AND next_repayment_date >= CURDATE() THEN 1 ELSE 0 END) as on_time_loans,
                        SUM(CASE WHEN loan_status = 'active' AND next_repayment_date < CURDATE() THEN 1 ELSE 0 END) as late_loans
                      FROM customer
                      WHERE 1=1";

            if ($province_id) {
                $query .= " AND province_id = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("i", $province_id);
            } else {
                $stmt = $conn->prepare($query);
            }

            $stmt->execute();
            $metrics = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            $late_borrowers_query = "
                SELECT id, full_name, loan_balance_bif, next_repayment_date
                FROM customer
                WHERE loan_status = 'active' AND next_repayment_date < CURDATE()
            ";

            if ($province_id) {
                $late_borrowers_query .= " AND province_id = ?";
                $late_stmt = $conn->prepare($late_borrowers_query);
                $late_stmt->bind_param("i", $province_id);
            } else {
                $late_stmt = $conn->prepare($late_borrowers_query);
            }

            $late_stmt->execute();
            $late_borrowers = $late_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $late_stmt->close();

            $conn->close();

            return [
                'total_outstanding' => (int)($metrics['total_outstanding'] ?? 0),
                'total_loans' => (int)($metrics['total_loans'] ?? 0),
                'active_loans' => (int)($metrics['active_loans'] ?? 0),
                'defaulted_loans' => (int)($metrics['defaulted_loans'] ?? 0),
                'on_time_loans' => (int)($metrics['on_time_loans'] ?? 0),
                'late_loans' => (int)($metrics['late_loans'] ?? 0),
                'late_borrowers' => $late_borrowers
            ];
        } catch (Exception $e) {
            error_log("Error getting loan metrics: " . $e->getMessage());
            return [
                'total_outstanding' => 0,
                'total_loans' => 0,
                'active_loans' => 0,
                'defaulted_loans' => 0,
                'on_time_loans' => 0,
                'late_loans' => 0,
                'late_borrowers' => []
            ];
        }
    }

    /**
     * Get eligible loan clients (customers without active loans or with good payment history)
     * 
     * @param int|null $province_id Optional province filter
     * @return int Count of eligible clients
     */
    function get_eligible_loan_clients($province_id = null) {
        $conn = get_db_connection();
        if (!$conn) return 0;

        try {
            $query = "
                SELECT COUNT(*) as count
                FROM customer
                WHERE (loan_status IS NULL OR loan_status = 'inactive')
                AND (credit_limit IS NULL OR credit_limit > 0)
            ";

            if ($province_id) {
                $query .= " AND province_id = ?";
                $stmt = $conn->prepare($query);
                $stmt->bind_param("i", $province_id);
            } else {
                $stmt = $conn->prepare($query);
            }

            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            $conn->close();

            return (int)($result['count'] ?? 0);
        } catch (Exception $e) {
            error_log("Error getting eligible loan clients: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get loan metrics by province
     * 
     * @return array Province-wise loan statistics
     */
    function get_loan_metrics_by_province() {
        $conn = get_db_connection();
        if (!$conn) return [];

        try {
            $query = "
                SELECT 
                    p.id,
                    p.name as province_name,
                    COALESCE(SUM(c.loan_balance_bif), 0) as total_due,
                    COUNT(DISTINCT c.id) as total_borrowers,
                    SUM(CASE WHEN c.loan_status = 'active' THEN 1 ELSE 0 END) as active_borrowers,
                    SUM(CASE WHEN c.next_repayment_date < CURDATE() AND c.loan_status = 'active' THEN 1 ELSE 0 END) as late_borrowers
                FROM province p
                LEFT JOIN customer c ON p.id = c.province_id
                GROUP BY p.id, p.name
                ORDER BY total_due DESC
            ";

            $stmt = $conn->prepare($query);
            if (!$stmt) throw new Exception($conn->error);

            $stmt->execute();
            $results = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
            $conn->close();

            return $results;
        } catch (Exception $e) {
            error_log("Error getting loan metrics by province: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get combined dashboard metrics
     * 
     * @param int|null $province_id Optional province filter
     * @return array All key metrics
     */
    function get_dashboard_metrics($province_id = null) {
        return [
            'expired_invoices' => get_expired_invoice_metrics($province_id),
            'loan_metrics' => get_loan_metrics($province_id),
            'eligible_loan_clients' => get_eligible_loan_clients($province_id),
            'province_loan_metrics' => !$province_id ? get_loan_metrics_by_province() : []
        ];
    }
}
