<?php
/**
 * ==========================================================
 * Masunzu Bar Hotel - Province Cash Helpers
 * ----------------------------------------------------------
 * Provides adjustProvinceCash() to update province balance
 * and write immutable ledger entries into
 * province_cash_transaction with consistent
 * transaction directions.
 * 
 * Usage:
 *   require_once __DIR__ . '/cash_helpers.php';
 *   adjustProvinceCash($conn, $province_id, $amount, 'REQUISITION_PAYMENT', 'REQUISITION', $requisition_id, $user_id, 'desc');
 * ==========================================================
 */

// Prevent multiple inclusions
if (defined('CASH_HELPERS_LOADED')) {
    return;
}
define('CASH_HELPERS_LOADED', true);

// You may include DB config if you want standalone usage, but not required here
// require_once __DIR__ . '/../config/database.php';

/**
 * Adjust province cash balance and log a ledger row.
 *
 * @param mysqli $conn            Active mysqli connection
 * @param int    $province_id     Province ID (wallet owner)
 * @param float  $amount          Positive amount
 * @param string $txn_type        One of: 'CASH_RECEIVED','REQUISITION_PAYMENT','BANK_DEPOSIT','ADJUSTMENT'
 * @param string $reference_type  'INVOICE','REQUISITION','BANK_DEPOSIT','OTHER'
 * @param int|null $reference_id  Reference row id (e.g. requisition id)
 * @param int    $created_by      User id
     * @param string $description     Free text
     * @param string|null $created_at Custom timestamp (Y-m-d H:i:s), defaults to current time
     *
     * @throws Exception if SQL fails or amount invalid
     * @return float Latest running balance after applying the change
 */
if (!function_exists('adjustProvinceCash')) {
    function adjustProvinceCash(
        mysqli $conn,
        int $province_id,
        float $amount,
        string $txn_type,
        string $reference_type = 'OTHER',
        $reference_id = null,
        int $created_by = 0,
        string $description = '',
        ?string $created_at = null
    ): float {

        $amount = round($amount, 2);
        if ($amount <= 0) {
            throw new Exception('Amount must be positive');
        }

        $directionMap = [
            'CASH_RECEIVED' => 'CREDIT',
            'ADJUSTMENT' => 'CREDIT',
            'BANK_DEPOSIT' => 'DEBIT',
            'REQUISITION_PAYMENT' => 'DEBIT',
        ];

        if (!isset($directionMap[$txn_type])) {
            throw new Exception("Unsupported province cash transaction type: {$txn_type}");
        }

        $direction = $directionMap[$txn_type];

        // Ensure province account exists
        $stmt = $conn->prepare("INSERT IGNORE INTO province_cash_account (province_id, balance) VALUES (?, 0.00)");
        if (!$stmt) { throw new Exception("Prepare failed (ensure account): " . $conn->error); }
        $stmt->bind_param("i", $province_id);
        if (!$stmt->execute()) { $stmt->close(); throw new Exception("Ensure account failed: " . $stmt->error); }
        $stmt->close();

        // Update balance
        $sql = ($direction === 'CREDIT')
            ? "UPDATE province_cash_account SET balance = balance + ? WHERE province_id = ?"
            : "UPDATE province_cash_account SET balance = balance - ? WHERE province_id = ?";
        $stmt = $conn->prepare($sql);
        if (!$stmt) { throw new Exception("Prepare failed (update balance): " . $conn->error); }
        $stmt->bind_param("di", $amount, $province_id);
        if (!$stmt->execute()) { $stmt->close(); throw new Exception("Balance update failed: " . $stmt->error); }
        if ($stmt->affected_rows === 0) { $stmt->close(); throw new Exception("No province cash account updated for province_id {$province_id}."); }
        $stmt->close();

        // Fetch new balance
        $stmt = $conn->prepare("SELECT balance FROM province_cash_account WHERE province_id = ?");
        if (!$stmt) { throw new Exception("Prepare failed (select balance): " . $conn->error); }
        $stmt->bind_param("i", $province_id);
        if (!$stmt->execute()) { $stmt->close(); throw new Exception("Select balance failed: " . $stmt->error); }
        $stmt->bind_result($balanceRaw);
        $stmt->fetch();
        $stmt->close();

        $balance = round((float)$balanceRaw, 2);

        $createdAtValue = $created_at ?? (new \DateTimeImmutable())->format('Y-m-d H:i:s');

        // Insert ledger row
        $stmt = $conn->prepare("INSERT INTO province_cash_transaction
            (province_id, txn_type, amount, direction, reference_type, reference_id, description, running_balance, created_by, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        if (!$stmt) { throw new Exception("Prepare failed (insert ledger): " . $conn->error); }

        $referenceIdParam = $reference_id;
        $descriptionParam = $description !== '' ? $description : null;

        $stmt->bind_param(
            "isdsssdiss",
            $province_id,
            $txn_type,
            $amount,
            $direction,
            $reference_type,
            $referenceIdParam,
            $descriptionParam,
            $balance,
            $created_by,
            $createdAtValue
        );

        if (!$stmt->execute()) {
            $stmtError = $stmt->error;
            $stmt->close();

            // Revert balance update to avoid silent mismatches
            $revertSql = ($direction === 'CREDIT')
                ? "UPDATE province_cash_account SET balance = balance - ? WHERE province_id = ?"
                : "UPDATE province_cash_account SET balance = balance + ? WHERE province_id = ?";
            $revertStmt = $conn->prepare($revertSql);
            if ($revertStmt) {
                $revertStmt->bind_param("di", $amount, $province_id);
                $revertStmt->execute();
                $revertStmt->close();
            }

            throw new Exception("Ledger insert failed: " . $stmtError);
        }
        $stmt->close();

        return $balance;
    }
}
