<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/config/database.php';

// Restrict access to Stock Manager Assistant role
if ($_SESSION['role_name'] !== 'Stock Manager Assistant') {
    error_log("Unauthorized access to stock_manager_assistant_dashboard.php by user {$_SESSION['full_name']} (Role: {$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
    exit;
}

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in stock_manager_assistant_dashboard.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}

try {
    // Fetch province_id
    $province_id = $_SESSION['province_id'] ?? null;
    if (!$province_id) {
        $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if (!$stmt) {
            throw new Exception("Failed to prepare query for province_id: " . $conn->error);
        }
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $province_id = $stmt->get_result()->fetch_assoc()['province_id'] ?? null;
        $_SESSION['province_id'] = $province_id;
        $stmt->close();
    }
    if (!$province_id) {
        throw new Exception("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    }

    // Fetch province name
    $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
    if (!$stmt) {
        throw new Exception("Failed to prepare query for province name: " . $conn->error);
    }
    $stmt->bind_param("i", $province_id);
    $stmt->execute();
    $province = $stmt->get_result()->fetch_assoc()['name'] ?? 'Unknown';
    $stmt->close();

    // Fetch pending paid invoices
    $stmt = $conn->prepare("
        SELECT pi.id, pi.invoice_number, pi.customer_name, pi.paid_amount, pi.sale_type, pi.payment_method, pi.payment_details, pi.created_at
        FROM paid_invoices pi
        WHERE pi.province_id = ? AND pi.status = 'paid' AND pi.serve_status = 'Pending'
        ORDER BY pi.created_at DESC
    ");
    if (!$stmt) {
        throw new Exception("Failed to prepare query for pending invoices: " . $conn->error);
    }
    $stmt->bind_param("i", $province_id);
    $stmt->execute();
    $pending_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    foreach ($pending_invoices as &$invoice) {
        if (($invoice['sale_type'] ?? '') === 'retail' && !empty($invoice['invoice_number'])) {
            $invoice['sale_type'] = 'wholesale';
        }
    }
    unset($invoice);

    // Fetch served invoices history
    $stmt = $conn->prepare("
        SELECT pi.id, pi.invoice_number, pi.customer_name, pi.paid_amount, pi.sale_type, pi.paid_at, pi.served_at, u.full_name AS served_by_name
        FROM paid_invoices pi
        LEFT JOIN user u ON pi.served_by = u.id
        WHERE pi.province_id = ? AND pi.status = 'paid' AND pi.serve_status = 'Served'
        ORDER BY pi.served_at DESC
    ");
    if (!$stmt) {
        throw new Exception("Failed to prepare query for served invoices: " . $conn->error);
    }
    $stmt->bind_param("i", $province_id);
    $stmt->execute();
    $served_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    foreach ($served_invoices as &$invoice) {
        if (($invoice['sale_type'] ?? '') === 'retail' && !empty($invoice['invoice_number'])) {
            $invoice['sale_type'] = 'wholesale';
        }
    }
    unset($invoice);

    // Handle serve action
    $success = '';
    $errors = [];
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['serve_invoice'])) {
        $invoice_id = intval($_POST['invoice_id'] ?? 0);
        if ($invoice_id <= 0) {
            $errors[] = "Invalid invoice ID.";
            error_log("Invalid invoice_id in serve action: $invoice_id");
        } else {
            $stmt = $conn->prepare("
                SELECT pi.sale_type, pi.invoice_number, pii.product_id, pii.quantity
                FROM paid_invoices pi
                JOIN paid_invoice_items pii ON pi.id = pii.invoice_id
                WHERE pi.id = ? AND pi.province_id = ? AND pi.status = 'paid' AND pi.serve_status = 'Pending'
            ");
            if (!$stmt) {
                throw new Exception("Failed to prepare query for invoice items: " . $conn->error);
            }
            $stmt->bind_param("ii", $invoice_id, $province_id);
            $stmt->execute();
            $invoice_items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();

            if ($invoice_items) {
                $conn->begin_transaction();
                try {
                    $invoice_number = $invoice_items[0]['invoice_number'] ?? '';
                    $sale_type = $invoice_items[0]['sale_type'] ?? 'retail';
                    if ($sale_type === 'retail' && !empty($invoice_number)) {
                        $sale_type = 'wholesale';
                    }
                    if (!in_array($sale_type, ['wholesale', 'retail'])) {
                        throw new Exception("Invalid sale_type for invoice ID $invoice_id: $sale_type");
                    }
                    $stock_table = $sale_type === 'wholesale' ? 'wholesale_stock' : 'retail_stock';
                    $all_success = true;

                    foreach ($invoice_items as $item) {
                        $product_id = $item['product_id'];
                        $quantity = $item['quantity'];

                        // Check available stock
                        $stmt = $conn->prepare("SELECT quantity FROM $stock_table WHERE province_id = ? AND product_id = ?");
                        if (!$stmt) {
                            throw new Exception("Failed to prepare query for stock check: " . $conn->error);
                        }
                        $stmt->bind_param("ii", $province_id, $product_id);
                        $stmt->execute();
                        $current_quantity = $stmt->get_result()->fetch_assoc()['quantity'] ?? 0;
                        $stmt->close();

                        if ($current_quantity < $quantity) {
                            $all_success = false;
                            $errors[] = "Insufficient $sale_type stock for product ID $product_id: needed $quantity, available $current_quantity";
                            error_log("Insufficient $sale_type stock for product ID $product_id: needed $quantity, available $current_quantity");
                            break;
                        }

                        // Update stock
                        $stmt = $conn->prepare("UPDATE $stock_table SET quantity = quantity - ?, last_updated = NOW() WHERE province_id = ? AND product_id = ? AND quantity >= ?");
                        if (!$stmt) {
                            throw new Exception("Failed to prepare query for stock update: " . $conn->error);
                        }
                        $stmt->bind_param("iiii", $quantity, $province_id, $product_id, $quantity);
                        if (!$stmt->execute() || $stmt->affected_rows === 0) {
                            $all_success = false;
                            $errors[] = "Insufficient $sale_type stock for product ID $product_id while serving invoice ID $invoice_id.";
                            error_log("Failed to update $stock_table for product ID $product_id: insufficient stock or query error.");
                            $stmt->close();
                            break;
                        }
                        $stmt->close();

                        // Log stock adjustment
                        $stmt = $conn->prepare("
                            INSERT INTO stock_adjustment (product_id, province_id, from_stock_type, to_stock_type, quantity_moved, reason, adjusted_at, stock_manager_id)
                            VALUES (?, ?, ?, NULL, ?, ?, NOW(), ?)
                        ");
                        if (!$stmt) {
                            throw new Exception("Failed to prepare query for stock adjustment: " . $conn->error);
                        }
                        $from_stock_type = $sale_type;
                        $reason = "Served paid invoice ID $invoice_id";
                        $stmt->bind_param("iissis", $product_id, $province_id, $from_stock_type, $quantity, $reason, $_SESSION['user_id']);
                        if (!$stmt->execute()) {
                            $all_success = false;
                            $errors[] = "Failed to log stock adjustment for product ID $product_id";
                            error_log("Failed to log stock adjustment for product ID $product_id: " . $stmt->error);
                            $stmt->close();
                            break;
                        }
                        $stmt->close();
                    }

                    if ($all_success) {
                        // Update serve_status and served_by
                        $stmt = $conn->prepare("UPDATE paid_invoices SET serve_status = 'Served', served_by = ?, served_at = NOW() WHERE id = ? AND province_id = ?");
                        if (!$stmt) {
                            throw new Exception("Failed to prepare query for updating paid_invoices: " . $conn->error);
                        }
                        $stmt->bind_param("iii", $_SESSION['user_id'], $invoice_id, $province_id);
                        $stmt->execute();
                        $affected_rows = $stmt->affected_rows;
                        $stmt->close();
                        if ($affected_rows === 0) {
                            throw new Exception("No rows updated for invoice ID $invoice_id in paid_invoices (province_id: $province_id)");
                        }
                        error_log("Updated paid_invoices for invoice ID $invoice_id: serve_status = 'Served', served_by = {$_SESSION['user_id']}, province_id = $province_id");

                        // Log action
                        $stmt = $conn->prepare("INSERT INTO log (user_id, action, description, created_at) VALUES (?, 'serve_invoice', ?, NOW())");
                        if (!$stmt) {
                            throw new Exception("Failed to prepare query for logging action: " . $conn->error);
                        }
                        $description = "Served paid invoice ID $invoice_id in province $province_id";
                        $stmt->bind_param("is", $_SESSION['user_id'], $description);
                        $stmt->execute();
                        $stmt->close();

                        $conn->commit();
                        $success = "Invoice ID $invoice_id served successfully.";
                        error_log("Served paid invoice ID $invoice_id by user {$_SESSION['user_id']} in province $province_id");
                    } else {
                        $conn->rollback();
                    }
                } catch (Exception $e) {
                    $conn->rollback();
                    $errors[] = "Error serving invoice: " . $e->getMessage();
                    error_log("Error serving invoice ID $invoice_id: " . $e->getMessage());
                }
            } else {
                $errors[] = "Invalid invoice or already served.";
                error_log("Invalid or already served invoice ID $invoice_id attempted by user {$_SESSION['user_id']}");
            }
        }

        // Refresh data after serving
        $stmt = $conn->prepare("
            SELECT pi.id, pi.invoice_number, pi.customer_name, pi.paid_amount, pi.sale_type, pi.payment_method, pi.payment_details, pi.created_at
            FROM paid_invoices pi
            WHERE pi.province_id = ? AND pi.status = 'paid' AND pi.serve_status = 'Pending'
            ORDER BY pi.created_at DESC
        ");
        if (!$stmt) {
            throw new Exception("Failed to prepare query for refreshing pending invoices: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $pending_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        foreach ($pending_invoices as &$invoice) {
            if (($invoice['sale_type'] ?? '') === 'retail' && !empty($invoice['invoice_number'])) {
                $invoice['sale_type'] = 'wholesale';
            }
        }
        unset($invoice);

        $stmt = $conn->prepare("
            SELECT pi.id, pi.invoice_number, pi.customer_name, pi.paid_amount, pi.sale_type, pi.paid_at, pi.served_at, u.full_name AS served_by_name
            FROM paid_invoices pi
            LEFT JOIN user u ON pi.served_by = u.id
            WHERE pi.province_id = ? AND pi.status = 'paid' AND pi.serve_status = 'Served'
            ORDER BY pi.served_at DESC
        ");
        if (!$stmt) {
            throw new Exception("Failed to prepare query for refreshing served invoices: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $served_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        foreach ($served_invoices as &$invoice) {
            if (($invoice['sale_type'] ?? '') === 'retail' && !empty($invoice['invoice_number'])) {
                $invoice['sale_type'] = 'wholesale';
            }
        }
        unset($invoice);
    }

    $conn->close();
} catch (Exception $e) {
    error_log("Critical error in stock_manager_assistant_dashboard.php: " . $e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>An unexpected error occurred. Check logs for details. Contact administrator.</p>";
    if ($conn) $conn->close();
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
?>
<div class="container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh;">
    <div class="content" style="background-color: #F4F0E4; padding: 30px; border-radius: 12px; box-shadow: 0 6px 12px rgba(0,0,0,0.15); max-width: 1000px; margin: 0 auto;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px; font-weight: 600; text-align: center;">
            Stock Manager Assistant Dashboard
        </h2>
        <p style="color: #4B2F1F; font-size: 16px; text-align: center;">
            Welcome, <?php echo htmlspecialchars($_SESSION['full_name'], ENT_QUOTES, 'UTF-8'); ?>! (<?php echo htmlspecialchars($province, ENT_QUOTES, 'UTF-8'); ?>)
        </p>
        <?php if ($success): ?>
            <p style="color: #008000; text-align: center; font-size: 16px;"><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></p>
        <?php endif; ?>
        <?php if (!empty($errors)): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 15px; border-radius: 8px; border: 1px solid #FF0000; margin-bottom: 20px;">
                <?php foreach ($errors as $error): ?>
                    <p style="font-size: 14px;"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Pending Paid Invoices to Serve</h3>
        <input type="text" id="pending_search" placeholder="Search pending invoices..." style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #FFFFFF; font-size: 14px; color: #4B2F1F; margin-bottom: 15px;">
        <div style="overflow-x: auto;">
            <table id="pending_table" style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 15px; text-align: left;">Invoice Number</th>
                        <th style="padding: 15px; text-align: left;">Customer</th>
                        <th style="padding: 15px; text-align: left;">Sale Type</th>
                        <th style="padding: 15px; text-align: left;">Payment Method</th>
                        <th style="padding: 15px; text-align: left;">Payment Details</th>
                        <th style="padding: 15px; text-align: left;">Created At</th>
                        <th style="padding: 15px; text-align: left;">Amount (BIF)</th>
                        <th style="padding: 15px; text-align: left;">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($pending_invoices)): ?>
                        <tr>
                            <td colspan="8" style="padding: 15px; text-align: center; color: #4B2F1F;">No pending invoices to serve.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($pending_invoices as $invoice): ?>
                            <tr style="border-bottom: 1px solid #F4F0E4;">
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['customer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars(ucfirst($invoice['sale_type']), ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars(ucfirst($invoice['payment_method'] ?? 'N/A'), ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['payment_details'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo date('Y-m-d H:i', strtotime($invoice['created_at'])); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo number_format($invoice['paid_amount'], 0, ',', '.'); ?></td>
                                <td style="padding: 15px; text-align: left;">
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="invoice_id" value="<?php echo $invoice['id']; ?>">
                                        <button type="submit" name="serve_invoice" style="background-color: #4B2F1F; color: #F4F0E4; padding: 8px 16px; border: none; border-radius: 5px; cursor: pointer; transition: all 0.3s ease;">Serve</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Served Invoices History</h3>
        <input type="text" id="history_search" placeholder="Search served invoices..." style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; background-color: #FFFFFF; font-size: 14px; color: #4B2F1F; margin-bottom: 15px;">
        <div style="overflow-x: auto;">
            <table id="history_table" style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 15px; text-align: left;">Invoice Number</th>
                        <th style="padding: 15px; text-align: left;">Customer</th>
                        <th style="padding: 15px; text-align: left;">Sale Type</th>
                        <th style="padding: 15px; text-align: left;">Served At</th>
                        <th style="padding: 15px; text-align: left;">Served By</th>
                        <th style="padding: 15px; text-align: left;">Amount (BIF)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($served_invoices)): ?>
                        <tr>
                            <td colspan="6" style="padding: 15px; text-align: center; color: #4B2F1F;">No served invoices found.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($served_invoices as $invoice): ?>
                            <tr style="border-bottom: 1px solid #F4F0E4;">
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['customer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars(ucfirst($invoice['sale_type']), ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo $invoice['served_at'] ? date('Y-m-d H:i', strtotime($invoice['served_at'])) : 'N/A'; ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['served_by_name'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo number_format($invoice['paid_amount'], 0, ',', '.'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Search functionality for pending invoices
        const pendingSearch = document.getElementById('pending_search');
        const pendingTable = document.getElementById('pending_table');
        if (pendingSearch && pendingTable) {
            pendingSearch.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase();
                const rows = pendingTable.getElementsByTagName('tr');
                for (let i = 1; i < rows.length; i++) { // Skip header row
                    const text = rows[i].textContent.toLowerCase();
                    rows[i].style.display = text.includes(searchTerm) ? '' : 'none';
                }
            });
        }

        // Search functionality for served invoices history
        const historySearch = document.getElementById('history_search');
        const historyTable = document.getElementById('history_table');
        if (historySearch && historyTable) {
            historySearch.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase();
                const rows = historyTable.getElementsByTagName('tr');
                for (let i = 1; i < rows.length; i++) { // Skip header row
                    const text = rows[i].textContent.toLowerCase();
                    rows[i].style.display = text.includes(searchTerm) ? '' : 'none';
                }
            });
        }
    });
</script>

<style>
    table { min-width: 600px; }
    button:hover {
        background-color: #F4A261;
        color: #4B2F1F;
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.3);
    }
    input:focus {
        outline: none;
        border-color: #F4A261;
        box-shadow: 0 0 5px rgba(244, 162, 97, 0.5);
    }
    @media (max-width: 768px) {
        .container { padding: 15px; }
        .content { width: 100%; max-width: 100%; }
        h2 { font-size: 24px; }
        h3 { font-size: 20px; }
        p, button { font-size: 14px; }
        table { font-size: 14px; }
    }
    @media (max-width: 480px) {
        h2 { font-size: 20px; }
        h3 { font-size: 18px; }
        p, button { font-size: 12px; }
        table { font-size: 12px; }
    }
</style>

<?php
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
?>
