<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/config/database.php';

require_login();

// Restrict to DGA/Leadership roles
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in operation_dashboard.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
if (!$stmt) {
    error_log("Role query preparation failed in operation_dashboard.php: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Internal server error. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
if (!in_array($role, ['DG', 'DGA', 'Board', 'Admin', 'Operation Manager'], true)) {
    error_log("Unauthorized access to operation_dashboard.php by user {$_SESSION['full_name']} (Role: $role)");
    // Use dashboard mapping to redirect correctly
    $dashboard_map = [
        'Board' => 'admin_dashboard.php',
        'DG' => 'admin_dashboard.php',
        'DGA' => 'admin_dashboard.php',
        'Operation Manager' => 'operation_dashboard.php',
        'Procurement Manager' => 'procurement_dashboard.php',
        'Finance Head' => 'finance_dashboard.php',
        'Marketing Head' => 'marketing_dashboard.php',
        'Stock Manager' => 'stock_manager_dashboard.php',
        'Accountant' => 'accountant_dashboard.php',
        'Marketing Agent' => 'marketing_dashboard.php',
        'Cashier' => 'cashier_dashboard.php',
        'Caissier_Comptable' => 'cashier_dashboard.php',
        'Stock Manager Assistant' => 'stock_manager_assistant_dashboard.php'
    ];
    $dashboard = $dashboard_map[$role] ?? 'unauthorized.php';
    header("Location: /masunzu_bar_hotel/dashboards/" . $dashboard);
    exit;
}

// Determine if a region scope should constrain the data
$isOperationManager = $role === 'Operation Manager';
$userRegionId = isset($_SESSION['region_id']) ? (int)$_SESSION['region_id'] : null;
$regionFilterActive = $isOperationManager && $userRegionId > 0;

// Ensure notifications table exists and fetch notifications
$createNotificationSql = "
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read),
        CONSTRAINT fk_notification_user_operation FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createNotificationSql)) {
    error_log('Failed to ensure notification table on operation dashboard: ' . $conn->error);
}

$operationNotifications = [];
$stmtNotif = $conn->prepare("SELECT id, message, link, created_at FROM user_notification WHERE user_id = ? AND is_read = 0 ORDER BY created_at DESC LIMIT 10");
if ($stmtNotif) {
    $stmtNotif->bind_param("i", $_SESSION['user_id']);
    $stmtNotif->execute();
    $operationNotifications = $stmtNotif->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtNotif->close();
    // Don't auto-mark as read - let users hide them manually
    // Debug: log notification count
    if (count($operationNotifications) > 0) {
        error_log("Operation Dashboard: Found " . count($operationNotifications) . " unread notifications for user {$_SESSION['user_id']}");
    }
} else {
    error_log('Failed to fetch notifications for operation dashboard: ' . $conn->error);
}

$receiverWhereSelect = '';
$receiverWhereSelectTypes = '';
$receiverWhereSelectParams = [];

$region_stock_products = [];
$region_stock_total_crates = 0;
$region_stock_total_value = 0;

if ($isOperationManager) {
    // For Operation Manager: show transfers RECEIVED by them (pending) AND transfers sent BY them that were rejected
    // operation_receiver_id = ? means transfers sent TO this Operation Manager (e.g., from DGA) - these are pending to receive
    // operation_manager_id = ? AND status = 'rejected' means transfers sent BY them that were rejected - these need attention
    $receiverWhereSelect = " AND (:alias.operation_receiver_id = ? OR (:alias.operation_manager_id = ? AND :alias.status = 'rejected'))";
    $receiverWhereSelectTypes = 'ii';
    $receiverWhereSelectParams = [$_SESSION['user_id'], $_SESSION['user_id']];
} else {
    $receiverWhereSelect = " AND :alias.operation_manager_id = ?";
    $receiverWhereSelectTypes = 'i';
    $receiverWhereSelectParams = [$_SESSION['user_id']];
}

$lowStockThreshold = 10;
// Fetch province-based low stock
// Determine available provinces for this region (or all when filter inactive)
$provinceRows = [];
$provinceRegionSql = "SELECT id, name FROM province";
$provinceRegionParams = [];
if ($regionFilterActive) {
    $provinceRegionSql .= " WHERE region_id = ?";
    $provinceRegionParams[] = $userRegionId;
}
$provinceRegionSql .= " ORDER BY name";
$provinceStmt = $conn->prepare($provinceRegionSql);
if ($provinceStmt) {
    if ($regionFilterActive) {
        $provinceStmt->bind_param("i", $userRegionId);
    }
    $provinceStmt->execute();
    $resultProvinces = $provinceStmt->get_result();
    while ($provinceRow = $resultProvinces->fetch_assoc()) {
        $provinceRows[] = [
            'id' => (int)$provinceRow['id'],
            'name' => $provinceRow['name'],
        ];
    }
    $provinceStmt->close();
}

$regionName = 'Toutes Régions';
$regionDisplayId = $userRegionId ?: null;
if ($regionDisplayId) {
    $stmtRegionName = $conn->prepare("SELECT name FROM region WHERE id = ?");
    if ($stmtRegionName) {
        $stmtRegionName->bind_param("i", $regionDisplayId);
        $stmtRegionName->execute();
        $regionRow = $stmtRegionName->get_result()->fetch_assoc();
        if (!empty($regionRow['name'])) {
            $regionName = $regionRow['name'];
        }
        $stmtRegionName->close();
    }
}

$provinceStockSql = "
    SELECT pr.name AS province_name,
           pr.id AS province_id,
           p.id AS product_id,
           p.name AS product_name,
           p.crate_quantity,
           p.price_per_crate,
           p.low_stock_alert_crates,
           p.low_stock_alert_units,
           COALESCE(ps.quantity, 0) AS quantity,
           ps.unit_type
    FROM province_stock ps
    JOIN product p ON ps.product_id = p.id
    JOIN province pr ON ps.province_id = pr.id
    WHERE ps.quantity IS NOT NULL AND ps.quantity > 0
";
if ($regionFilterActive) {
    $provinceStockSql .= " AND pr.region_id = ?";
}
$stmt = $conn->prepare($provinceStockSql);
if (!$stmt) {
    error_log("Dépôt stock query preparation failed in operation_dashboard.php: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Internal server error. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
if ($regionFilterActive) {
    $stmt->bind_param("i", $userRegionId);
}
$stmt->execute();
$province_stocks = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$province_inventory = [];
foreach ($province_stocks as $stock) {
    $provinceId = (int)($stock['province_id'] ?? 0);
    $provinceName = $stock['province_name'] ?? ('Province #' . $provinceId);
    $crateQuantity = max(1, (int)($stock['crate_quantity'] ?? 12));
    $quantity = (int)($stock['quantity'] ?? 0);
    $unitType = strtolower($stock['unit_type'] ?? 'crates');
    $crates = $unitType === 'crates' ? $quantity : ($quantity / $crateQuantity);
    $bottles = $unitType === 'crates' ? ($quantity * $crateQuantity) : $quantity;
    $totalValue = $crates * (float)($stock['price_per_crate'] ?? 0);
    $thresholdCrates = (int)($stock['low_stock_alert_crates'] ?? 0);
    $thresholdUnits = (int)($stock['low_stock_alert_units'] ?? 0);
    $isLowStock = false;
    if ($thresholdCrates > 0) {
        $isLowStock = $crates <= $thresholdCrates;
    } elseif ($thresholdUnits > 0) {
        $isLowStock = $bottles <= $thresholdUnits;
    } else {
        $isLowStock = $crates <= $lowStockThreshold;
    }

    if (!isset($province_inventory[$provinceId])) {
        $province_inventory[$provinceId] = [
            'province_name' => $provinceName,
            'items' => [],
            'low_stock' => [],
            'total_crates' => 0,
            'total_bottles' => 0,
            'total_value' => 0,
        ];
    }

    $province_inventory[$provinceId]['items'][] = [
        'product_name' => $stock['product_name'],
        'product_id' => (int)$stock['product_id'],
        'crate_quantity' => $crateQuantity,
        'crates' => $crates,
        'bottles' => $bottles,
        'total_value' => $totalValue,
        'threshold_crates' => $thresholdCrates,
        'threshold_units' => $thresholdUnits,
        'is_low' => $isLowStock,
    ];
    $province_inventory[$provinceId]['total_crates'] += $crates;
    $province_inventory[$provinceId]['total_bottles'] += $bottles;
    $province_inventory[$provinceId]['total_value'] += $totalValue;

    if ($isLowStock) {
        $province_inventory[$provinceId]['low_stock'][] = end($province_inventory[$provinceId]['items']);
    }
}
uasort($province_inventory, static function ($a, $b) {
    return strcasecmp($a['province_name'] ?? '', $b['province_name'] ?? '');
});

$provincePlaceholder = [
    'items' => [],
    'low_stock' => [],
    'total_crates' => 0,
    'total_bottles' => 0,
    'total_value' => 0,
];
foreach ($provinceRows as $provinceRow) {
    $provinceId = $provinceRow['id'];
    if (!isset($province_inventory[$provinceId])) {
        $province_inventory[$provinceId] = array_merge(['province_name' => $provinceRow['name']], $provincePlaceholder);
    }
}

$province_options = [];
foreach ($province_inventory as $provinceId => $provinceData) {
    $province_options[$provinceId] = $provinceData['province_name'] ?? ('Province #' . $provinceId);
}

    $selectedProvinceId = isset($_GET['province_id']) ? (int)$_GET['province_id'] : null;
    if ($selectedProvinceId <= 0 || !isset($province_inventory[$selectedProvinceId])) {
        $selectedProvinceId = array_key_first($province_inventory) ?: null;
    }
    $selectedProvinceData = $selectedProvinceId !== null ? $province_inventory[$selectedProvinceId] : null;

    $province_stock_crates = 0;
    $province_stock_bottles = 0;
    $province_stock_value = 0;
    foreach ($province_inventory as $provinceData) {
        $province_stock_crates += $provinceData['total_crates'];
        $province_stock_bottles += $provinceData['total_bottles'];
        $province_stock_value += $provinceData['total_value'];
    }

// Aggregate region stock table for the region panel
$province_product_stock = [];
$regionStockSql = "
    SELECT rs.region_id,
           p.id AS product_id,
           p.name AS product_name,
           p.crate_quantity,
           p.price_per_crate,
           COALESCE(SUM(rs.quantity), 0) AS total_crates
    FROM region_stock rs
    JOIN product p ON rs.product_id = p.id
";
if ($regionFilterActive) {
    $regionStockSql .= " WHERE rs.region_id = ?";
}
$regionStockSql .= " GROUP BY rs.region_id, p.id, p.name, p.crate_quantity, p.price_per_crate ORDER BY p.name";
$stmt = $conn->prepare($regionStockSql);
if ($stmt) {
    if ($regionFilterActive) {
        $stmt->bind_param("i", $userRegionId);
    }
    $stmt->execute();
    $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    foreach ($rows as $row) {
        $totalCrates = (int)($row['total_crates'] ?? 0);
        $unitCrate = (int)($row['crate_quantity'] ?? 12);
        $value = $totalCrates * (int)($row['price_per_crate'] ?? 0);
        $region_stock_products[] = [
            'product_id' => (int)$row['product_id'],
            'product_name' => $row['product_name'],
            'crate_quantity' => $unitCrate,
            'total_crates' => $totalCrates,
            'total_value' => $value,
        ];
        $region_stock_total_crates += $totalCrates;
        $region_stock_total_value += $value;
    }
    $stmt->close();
}
 $pending_reception_transfers = [];
    // Show both pending transfers (to receive) and rejected transfers (that need attention)
    $pendingTransfersDetailSql = "
    SELECT st.shipment_id, p.name AS product_name, st.quantity, COALESCE(pr.name, CONCAT('Région ', COALESCE(r.name, ''))) AS province_name, st.sent_at, st.estimated_arrival, st.status, st.received_at
    FROM stock_transfer st
    JOIN product p ON st.product_id = p.id
    LEFT JOIN province pr ON st.province_id = pr.id
    LEFT JOIN region r ON st.region_id = r.id
    WHERE st.status IN ('pending', 'rejected')
";
if ($regionFilterActive) {
    // Use COALESCE to check both province.region_id and stock_transfer.region_id
    // This handles cases where DGA sends to Operation Manager (province_id is NULL)
    $pendingTransfersDetailSql .= " AND COALESCE(pr.region_id, st.region_id) = ?";
}
$pendingTransfersDetailSql .= str_replace(':alias', 'st', $receiverWhereSelect);
$pendingTransfersDetailSql .= " ORDER BY st.sent_at DESC LIMIT 6";
$stmt = $conn->prepare($pendingTransfersDetailSql);
if ($stmt) {
    $countTypes = '';
    $countParams = [];
    if ($regionFilterActive) {
        $countTypes .= 'i';
        $countParams[] = $userRegionId;
    }
    $countTypes .= $receiverWhereSelectTypes;
    $countParams = array_merge($countParams, $receiverWhereSelectParams);
    if ($countTypes !== '') {
        $stmt->bind_param($countTypes, ...$countParams);
    }
    $stmt->execute();
    $pending_reception_transfers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    error_log("Operation Dashboard: Failed to prepare pending transfers query: " . ($conn->error ?? 'Unknown error'));
}
$pending_count = count($pending_reception_transfers);

// Fetch recent stock transfers with crate_quantity and unit_type
$recentTransfersSql = "
    SELECT t.id, t.sent_at, t.received_at, p.id AS product_id, p.name AS product_name, t.quantity, t.unit_type, pr.name AS province_name, p.crate_quantity
    FROM stock_transfer t 
    JOIN product p ON t.product_id = p.id 
    LEFT JOIN province pr ON t.province_id = pr.id
    WHERE t.status = 'confirmed'
";
// Use the same receiver logic as pending transfers
$recentTransfersSql .= str_replace(':alias', 't', $receiverWhereSelect);
if ($regionFilterActive) {
    // Use COALESCE to check both province.region_id and stock_transfer.region_id
    // This handles cases where DGA sends to Operation Manager (province_id is NULL)
    $recentTransfersSql .= " AND COALESCE(pr.region_id, t.region_id) = ?";
}
$recentTransfersSql .= "
    ORDER BY t.sent_at DESC LIMIT 5
";
$stmt = $conn->prepare($recentTransfersSql);
if (!$stmt) {
    error_log("Recent transfers query preparation failed in operation_dashboard.php: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Internal server error. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
    $bindTypes = $receiverWhereSelectTypes;
    $bindParams = $receiverWhereSelectParams;
    if ($regionFilterActive) {
        $bindTypes .= 'i';
        $bindParams[] = $userRegionId;
    }
$stmt->bind_param($bindTypes, ...$bindParams);
$stmt->execute();
$recent_transfers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch recent rejected transfers relevant to the logged-in user/region
$rejected_transfers = [];
$rejected_where = ['st.status = \'rejected\''];
$rejected_params = [];
$rejected_types = '';
if ($isOperationManager) {
    $rejected_where[] = 'st.operation_manager_id = ?';
    $rejected_types .= 'i';
    $rejected_params[] = $_SESSION['user_id'];
}
if ($regionFilterActive) {
    $rejected_where[] = 'COALESCE(pr.region_id, st.region_id) = ?';
    $rejected_types .= 'i';
    $rejected_params[] = $userRegionId;
}
$rejected_sql = "
    SELECT st.shipment_id, p.name AS product_name, st.quantity, COALESCE(pr.name, r.name, 'Région') AS destination_name, st.received_at
    FROM stock_transfer st
    JOIN product p ON st.product_id = p.id
    LEFT JOIN province pr ON st.province_id = pr.id
    LEFT JOIN region r ON st.region_id = r.id
    WHERE " . implode(' AND ', $rejected_where) . "
    ORDER BY st.received_at DESC, st.sent_at DESC
    LIMIT 6
";
$stmt = $conn->prepare($rejected_sql);
if ($stmt) {
    if ($rejected_types !== '') {
        $stmt->bind_param($rejected_types, ...$rejected_params);
    }
    $stmt->execute();
    $rejected_transfers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}
$rejected_count = count($rejected_transfers);

foreach ($recent_transfers as &$transfer) {
    $crate_quantity = (int)($transfer['crate_quantity'] ?? 12);
    if ($crate_quantity <= 0) {
        $crate_quantity = 12;
    }
    $transfer['crate_size'] = $crate_quantity;
    $transfer['crates_count'] = (int)$transfer['quantity'];
    $transfer['bottles_count'] = $transfer['crates_count'] * $crate_quantity;
}
unset($transfer);

// Fetch pending orders
$stmt = $conn->prepare("
    SELECT o.id, o.product_id, p.name AS product_name, o.quantity, o.unit_type, o.order_type, o.created_at, o.customer_name, p.crate_quantity
    FROM `order` o 
    JOIN product p ON o.product_id = p.id 
    WHERE o.status = 'pending' AND o.approved_by_operation_manager IS NULL
");
if (!$stmt) {
    error_log("Pending orders query preparation failed in operation_dashboard.php: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Internal server error. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt->execute();
$pending_orders = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

foreach ($pending_orders as &$order) {
    $crate_quantity = (int)($order['crate_quantity'] ?? 12);
    if ($crate_quantity <= 0) {
        $crate_quantity = 12;
    }
    $order['crate_size'] = $crate_quantity;
    $order['crates_count'] = (int)$order['quantity'];
    $order['bottles_count'] = $order['crates_count'] * $crate_quantity;
}
unset($order);

$conn->close();
$pendingOrderCount = count($pending_orders);
$recentTransferCount = count($recent_transfers);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tableau Opérations</title>
    <style>
        * { box-sizing: border-box; }
        body { font-family: 'Segoe UI', Arial, sans-serif; margin: 0; padding: 0; background-color: #F3F1EC; color: #3E2723; }
        .dashboard-container { max-width: 1500px; margin: 0 auto; padding: 25px 20px 60px; }
        .page-head { background-color: #FFFFFF; border-radius: 16px; padding: 20px 24px; box-shadow: 0 8px 24px rgba(0,0,0,0.08); display: flex; flex-wrap: wrap; justify-content: space-between; gap: 15px; align-items: center; margin-bottom: 20px; }
        .page-head h2 { margin: 0; font-size: 28px; color: #4B2F1F; }
        .page-head p { margin: 6px 0 0; color: #6D4C41; }
        .cards-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); gap: 15px; margin-bottom: 25px; }
        .summary-card { background: linear-gradient(135deg, #4B2F1F, #8D6E63); border-radius: 16px; padding: 18px; color: #FFF; box-shadow: 0 12px 30px rgba(75,47,31,0.25); }
        .summary-card:nth-child(2) { background: linear-gradient(135deg, #F9A825, #F57F17); }
        .summary-card:nth-child(3) { background: linear-gradient(135deg, #0288D1, #01579B); }
        .summary-card:nth-child(4) { background: linear-gradient(135deg, #8E24AA, #4A148C); }
        .summary-card p { margin: 0; opacity: 0.8; font-size: 14px; }
        .summary-card h3 { margin: 10px 0 0; font-size: 28px; }
        .section-card { background-color: #FFFFFF; border-radius: 16px; padding: 20px 24px; margin-bottom: 25px; box-shadow: 0 8px 20px rgba(0,0,0,0.06); }
        .section-title { display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 10px; margin-bottom: 10px; }
        .section-title h3 { margin: 0; color: #4B2F1F; font-size: 22px; }
        .section-title span { color: #757575; font-size: 14px; }
        .section-title > div { display: flex; align-items: center; gap: 10px; flex-wrap: wrap; }
        .table-wrapper { overflow-x: auto; border-radius: 12px; }
        table { width: 100%; border-collapse: collapse; min-width: 600px; }
        th, td { padding: 12px 14px; text-align: left; font-size: 14px; border-bottom: 1px solid #E0E0E0; }
        th { background-color: #4B2F1F; color: #F4F0E4; position: sticky; top: 0; }
        tr:nth-child(even) { background-color: #FAF7F2; }
        tr:hover { background-color: #FFF3E0; }
        .province-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); gap: 15px; }
        .province-card { border: 1px solid #F0E4D7; border-radius: 14px; padding: 16px; background-color: #FFF; box-shadow: 0 8px 16px rgba(0,0,0,0.04); }
        .province-card-head { display: flex; justify-content: space-between; align-items: center; margin-bottom: 12px; }
        .province-card-head h4 { margin: 0; font-size: 18px; color: #3E2723; }
        .province-card-head span { font-size: 13px; color: #757575; }
        .low-list { list-style: none; padding: 0; margin: 0; }
        .low-list li { font-size: 13px; margin-bottom: 6px; color: #C62828; font-weight: 600; }
        .empty-state { margin: 0; font-size: 13px; color: #78909C; }
        .province-table-grid { grid-template-columns: repeat(auto-fit, minmax(360px, 1fr)); }
        .mini-table-wrapper { overflow-x: auto; }
        .mini-table-wrapper table { min-width: 320px; }
        .status-badge { padding: 4px 10px; border-radius: 999px; font-size: 11px; font-weight: 700; display: inline-block; }
        .status-badge.danger { background-color: #FFEBEE; color: #C62828; }
        .status-badge.ok { background-color: #E8F5E9; color: #1B5E20; }
        .two-column-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(420px, 1fr)); gap: 20px; }
        .action-buttons { display: flex; gap: 8px; flex-wrap: wrap; }
        .action-buttons button { padding: 6px 10px; border-radius: 6px; border: none; cursor: pointer; font-weight: 600; }
        .action-buttons button[name="approve"] { background-color: #4CAF50; color: #FFF; }
        .action-buttons button[name="reject"] { background-color: #E64A19; color: #FFF; }
        .pending-reception-row { animation: blink 2s infinite; }
        .rejected-transfer-row { background-color: #FFEBEE; border-left: 4px solid #C62828; }
        .rejected-transfer-row:hover { background-color: #FFCDD2; }
        @keyframes blink {
            0%, 100% { background-color: #FFF3E0; }
            50% { background-color: #FFE0B2; }
        }
        .receive-btn { background-color: #4CAF50; color: #FFF; padding: 8px 16px; border: none; border-radius: 6px; cursor: pointer; font-weight: 600; font-size: 13px; text-decoration: none; display: inline-block; transition: all 0.2s; }
        .receive-btn:hover { background-color: #45a049; transform: translateY(-1px); box-shadow: 0 2px 6px rgba(0,0,0,0.2); }
        @media (max-width: 768px) {
            .cards-grid { grid-template-columns: 1fr; }
            .province-table-grid { grid-template-columns: 1fr; }
            table { min-width: 100%; }
        }
    </style>
</head>
<body>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php'; ?>
<div class="dashboard-container">
    <div class="page-head">
        <div>
            <h2>Tableau des Opérations</h2>
            <p>Suivi consolidé des stocks principaux et provinciaux.</p>
        </div>
    </div>

    <div class="cards-grid">
        <div class="summary-card">
            <p>Stock régional non transféré</p>
            <h3><?php echo number_format($region_stock_total_value, 0, ',', '.'); ?> BIF</h3>
            <span><?php echo number_format($region_stock_total_crates, 0, ',', '.'); ?> caisses</span>
        </div>
        <div class="summary-card">
            <p>Stock réparti (dépôts)</p>
            <h3><?php echo number_format($province_stock_value, 0, ',', '.'); ?> BIF</h3>
            <span><?php echo number_format($province_stock_crates, 0, ',', '.'); ?> caisses</span>
        </div>
        <div class="summary-card">
            <p>Transferts en attente</p>
            <h3><?php echo number_format($pending_count, 0, ',', '.'); ?></h3>
            <span><?php echo $pending_count > 0 ? 'Action requise' : 'Aucune attente'; ?></span>
        </div>
        <div class="summary-card">
            <p>Transferts rejetés</p>
            <h3><?php echo number_format($rejected_count, 0, ',', '.'); ?></h3>
            <span><?php echo $rejected_count > 0 ? 'Rejets récents' : 'Aucun rejet'; ?></span>
        </div>
        <div class="summary-card">
            <p>Commandes en attente</p>
            <h3><?php echo number_format($pendingOrderCount, 0, ',', '.'); ?></h3>
            <span><?php echo $pendingOrderCount > 0 ? 'Prioriser la validation' : 'Flux normal'; ?></span>
        </div>
        <div class="summary-card">
            <p>Transferts confirmés</p>
            <h3><?php echo number_format($recentTransferCount, 0, ',', '.'); ?></h3>
            <span>Derniers mouvements reçus</span>
        </div>
    </div>

    <div class="section-card">
        <div class="section-title">
            <h3>Notifications de Transferts</h3>
            <div style="display: flex; align-items: center; gap: 10px; flex-wrap: wrap;">
                <span>Transferts nécessitant une action</span>
                <?php if (!empty($operationNotifications)): ?>
                    <button id="toggle-notifications-btn" onclick="toggleNotificationsPanel()" style="background-color: #F57C00 !important; color: white !important; border: 2px solid #E65100 !important; border-radius: 6px; padding: 8px 16px; cursor: pointer; font-size: 14px; font-weight: 600; transition: all 0.2s; box-shadow: 0 2px 4px rgba(0,0,0,0.2); min-width: 100px; display: inline-block !important; visibility: visible !important; opacity: 1 !important;" title="Masquer/Afficher les notifications" onmouseover="this.style.backgroundColor='#E65100'; this.style.transform='scale(1.05)'" onmouseout="this.style.backgroundColor='#F57C00'; this.style.transform='scale(1)'">
                        <span id="toggle-notifications-text">Masquer</span>
                    </button>
                <?php else: ?>
                    <span style="color: #757575; font-size: 13px;">Aucune notification (<?php echo count($operationNotifications); ?>)</span>
                <?php endif; ?>
            </div>
        </div>
        <?php if (!empty($operationNotifications)): ?>
            <div id="notifications-panel" style="background-color: #FFF3E0; border-left: 4px solid #F57C00; padding: 12px 16px; margin-bottom: 20px; border-radius: 6px; position: relative;">
                <h4 style="margin: 0 0 10px 0; color: #E65100; font-size: 16px;">Notifications Récentes (<?php echo count($operationNotifications); ?>)</h4>
                <ul style="list-style: none; margin: 0; padding: 0;">
                    <?php foreach ($operationNotifications as $note): ?>
                        <li id="notification-<?php echo (int)$note['id']; ?>" style="padding: 8px 0; border-bottom: 1px solid #FFE0B2; position: relative; padding-right: 30px;">
                            <button onclick="hideNotification(<?php echo (int)$note['id']; ?>)" style="position: absolute; right: 0; top: 8px; background: #F57C00; color: white; border: none; border-radius: 50%; width: 24px; height: 24px; cursor: pointer; font-size: 18px; line-height: 1; display: flex; align-items: center; justify-content: center; padding: 0; font-weight: bold; z-index: 10;" title="Masquer cette notification">×</button>
                            <span style="color: #3E2723; font-size: 14px;"><?php echo htmlspecialchars($note['message'], ENT_QUOTES, 'UTF-8'); ?></span><br>
                            <small style="color: #6D4C41;"><?php echo date('Y-m-d H:i', strtotime($note['created_at'])); ?></small>
                            <?php if (!empty($note['link'])): ?>
                                <div><a href="<?php echo htmlspecialchars($note['link'], ENT_QUOTES, 'UTF-8'); ?>" style="color: #F57C00; font-size: 13px; text-decoration: underline; font-weight: 600;">Voir détails →</a></div>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
        <div class="table-wrapper">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Produit</th>
                        <th>Quantité (caisses)</th>
                        <th>Dépôt</th>
                        <th>Statut</th>
                        <th>Envoyé le</th>
                        <th>Arrivée prévue</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($pending_reception_transfers)): ?>
                        <tr>
                            <td colspan="8" style="text-align:center; color:#6B5B52;">Aucun transfert nécessitant une action.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($pending_reception_transfers as $transfer): ?>
                            <?php 
                                $transferStatus = strtolower($transfer['status'] ?? 'pending');
                                $isRejected = $transferStatus === 'rejected';
                                $rowClass = $isRejected ? 'rejected-transfer-row' : 'pending-reception-row';
                            ?>
                            <tr class="<?php echo $rowClass; ?>">
                                <td><?php echo htmlspecialchars($transfer['shipment_id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($transfer['product_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo number_format((int)$transfer['quantity'], 0, ',', '.'); ?></td>
                                <td><?php echo htmlspecialchars($transfer['province_name'] ?? 'Région', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td>
                                    <?php if ($isRejected): ?>
                                        <span class="status-badge danger">Rejeté</span>
                                    <?php else: ?>
                                        <span class="status-badge ok">En attente</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo $transfer['sent_at'] ? htmlspecialchars(date('Y-m-d H:i', strtotime($transfer['sent_at'])), ENT_QUOTES, 'UTF-8') : '—'; ?></td>
                                <td><?php echo $transfer['estimated_arrival'] ? htmlspecialchars(date('Y-m-d H:i', strtotime($transfer['estimated_arrival'])), ENT_QUOTES, 'UTF-8') : '—'; ?></td>
                                <td>
                                    <?php if ($isRejected): ?>
                                        <span style="color: #C62828; font-size: 13px;">Rejeté le <?php echo $transfer['received_at'] ? htmlspecialchars(date('Y-m-d H:i', strtotime($transfer['received_at'])), ENT_QUOTES, 'UTF-8') : '—'; ?></span>
                                    <?php else: ?>
                                        <a href="/masunzu_bar_hotel/modules/operation/receive_stock.php?shipment_id=<?php echo (int)$transfer['shipment_id']; ?>" class="receive-btn">
                                            Confirmer/Rejeter
                                        </a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <div class="section-card">
        <div class="section-title">
            <h3>Transferts rejetés</h3>
            <span>Rejets récents liés à vos envois</span>
        </div>
        <?php if (empty($rejected_transfers)): ?>
            <p class="empty-state">Aucun rejet de transfert récent.</p>
        <?php else: ?>
            <div class="table-wrapper">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Produit</th>
                            <th>Quantité (caisses)</th>
                            <th>Destination</th>
                            <th>Rejeté le</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($rejected_transfers as $rejected): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($rejected['shipment_id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($rejected['product_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo number_format((int)$rejected['quantity'], 0, ',', '.'); ?></td>
                                <td><?php echo htmlspecialchars($rejected['destination_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($rejected['received_at'] ? date('Y-m-d H:i', strtotime($rejected['received_at'])) : '—', ENT_QUOTES, 'UTF-8'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <div class="section-card">
        <div class="section-title">
            <h3>Stock régional non transféré – <?php echo htmlspecialchars($regionName, ENT_QUOTES, 'UTF-8'); ?></h3>
        </div>
        <div style="display:flex; align-items:center; gap:12px; flex-wrap:wrap; margin-bottom:12px;">
            <label for="region-stock-search" style="font-weight:600; color:#4B2F1F;">Filtrer par produit</label>
            <input id="region-stock-search" type="text" placeholder="Nom du produit..." style="flex:1 1 220px; padding:8px 12px; border-radius:8px; border:1px solid #B08968;">
        </div>
        <?php if (!empty($region_stock_products)): ?>
            <div class="table-wrapper">
                <table id="region-stock-table">
                    <thead>
                        <tr>
                            <th>Produit</th>
                            <th>Caisses</th>
                            <th>Valeur</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($region_stock_products as $product): ?>
                            <?php
                                $name = $product['product_name'] ?? 'Produit inconnu';
                                $crates = (int)($product['total_crates'] ?? 0);
                                $value = (int)($product['total_value'] ?? 0);
                            ?>
                            <tr class="region-stock-row" data-product="<?php echo htmlspecialchars(strtolower($name), ENT_QUOTES, 'UTF-8'); ?>">
                                <td><?php echo htmlspecialchars($name, ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo number_format($crates, 0, ',', '.'); ?></td>
                                <td><?php echo number_format($value, 0, ',', '.'); ?> BIF</td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p class="empty-state">Aucun stock régional non transféré à afficher.</p>
        <?php endif; ?>
    </div>

    <div class="section-card">
        <div class="section-title">
            <h3>Stock régional réparti par dépôt</h3>
            <?php if (!empty($province_options)): ?>
                <form method="get" style="display:flex; gap:10px; align-items:center; flex-wrap:wrap; margin:0;">
                    <label style="font-size:14px; font-weight:600; color:#4B2F1F;">Dépôt</label>
                    <select name="province_id" onchange="this.form.submit()" style="padding:8px 12px; border-radius:8px; border:1px solid #B08968; background:#fff;">
                        <?php foreach ($province_options as $provinceId => $provinceName): ?>
                            <option value="<?php echo $provinceId; ?>" <?php echo $provinceId === $selectedProvinceId ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($provinceName, ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </form>
            <?php else: ?>
                <span style="font-size:14px; color:#C62828;">Aucun dépôt disponible dans la région.</span>
            <?php endif; ?>
        </div>
        <?php if ($selectedProvinceData && !empty($selectedProvinceData['items'])): ?>
            <div class="table-wrapper">
                <table>
                    <thead>
                        <tr>
                            <th>Produit</th>
                            <th>Caisses</th>
                            <th>Valeur</th>
                            <th>Alerte</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($selectedProvinceData['items'] as $item): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($item['product_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo number_format($item['crates'], 0, ',', '.'); ?></td>
                                <td><?php echo number_format((int)$item['total_value'], 0, ',', '.'); ?> BIF</td>
                                <td><?php echo $item['is_low'] ? '<span class="status-badge status-rejected">Alerte</span>' : 'OK'; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p class="empty-state">Aucun stock disponible pour ce dépôt.</p>
        <?php endif; ?>
    </div>

    <div class="section-card">
        <div class="section-title">
            <h3>Alertes Stock</h3>
        </div>
        <div class="province-grid">
            <?php foreach ($province_inventory as $provinceId => $provinceData): ?>
                <?php $alerts = $provinceData['low_stock'] ?? []; ?>
                <div class="province-card">
                    <div class="province-card-head">
                        <h4><?php echo htmlspecialchars($provinceData['province_name'], ENT_QUOTES, 'UTF-8'); ?></h4>
                        <span><?php echo count($alerts); ?> alertes</span>
                    </div>
                    <?php if (!empty($alerts)): ?>
                        <ul class="low-list">
                            <?php foreach ($alerts as $alert): ?>
                                <li><?php echo htmlspecialchars($alert['product_name'], ENT_QUOTES, 'UTF-8'); ?> — <?php echo number_format((float)$alert['crates'], 1, ',', '.'); ?> caisses</li>
                            <?php endforeach; ?>
                        </ul>
                    <?php else: ?>
                        <p class="empty-state">RAS</p>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>

    <div class="two-column-grid">
        <div class="section-card">
            <div class="section-title">
                <h3>Transferts Récents</h3>
            </div>
            <div class="table-wrapper">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Produit</th>
                            <th>Caisses</th>
                            <th>Dépôt</th>
                            <th>Envoyé</th>
                            <th>Reçu</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_transfers as $transfer): ?>
                            <?php
                                $productName = $transfer['product_name'] ?? 'Produit inconnu';
                                $provinceName = $transfer['province_name'] ?? '—';
                                $crates = (int)($transfer['crates_count'] ?? 0);
                                $sentAt = $transfer['sent_at'] ? date('Y-m-d H:i', strtotime($transfer['sent_at'])) : '—';
                                $receivedAt = $transfer['received_at'] ? date('Y-m-d H:i', strtotime($transfer['received_at'])) : 'N/A';
                            ?>
                            <tr>
                                <td><?php echo htmlspecialchars($transfer['id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($productName, ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo number_format($crates, 0, ',', '.'); ?></td>
                                <td><?php echo htmlspecialchars($provinceName, ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($sentAt, ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($receivedAt, ENT_QUOTES, 'UTF-8'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <div class="section-card">
            <div class="section-title">
                <h3>Commandes en Attente</h3>
            </div>
            <div class="table-wrapper">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Produit</th>
                            <th>Caisses</th>
                            <th>Client</th>
                            <th>Créé le</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pending_orders as $order): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($order['id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars($order['product_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo number_format((float)$order['crates_count'], 1, ',', '.'); ?></td>
                                <td><?php echo htmlspecialchars($order['customer_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($order['created_at'])), ENT_QUOTES, 'UTF-8'); ?></td>
                                <td>
                                    <form method="POST" action="/masunzu_bar_hotel/modules/operation/approve_order.php" class="action-buttons">
                                        <input type="hidden" name="order_id" value="<?php echo (int)$order['id']; ?>">
                                        <button type="submit" name="approve">Approuver</button>
                                        <button type="submit" name="reject">Rejeter</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
<script>
    document.addEventListener('DOMContentLoaded', function () {
        const searchInput = document.getElementById('region-stock-search');
        if (searchInput) {
            const rows = Array.from(document.querySelectorAll('.region-stock-row'));
            searchInput.addEventListener('input', function () {
                const term = searchInput.value.trim().toLowerCase();
                rows.forEach(function (row) {
                    const product = row.getAttribute('data-product') ?? '';
                    row.style.display = term === '' || product.includes(term) ? '' : 'none';
                });
            });
        }

        // Load hidden notifications from localStorage and hide them
        const hiddenNotifications = JSON.parse(localStorage.getItem('hiddenNotifications') || '[]');
        const notificationsPanel = document.getElementById('notifications-panel');
        let visibleCount = 0;
        
        hiddenNotifications.forEach(function(notifId) {
            const notifElement = document.getElementById('notification-' + notifId);
            if (notifElement) {
                notifElement.style.display = 'none';
            }
        });
        
        // Check if all notifications are hidden on page load, hide the panel if so
        if (notificationsPanel) {
            const allNotifications = notificationsPanel.querySelectorAll('li[id^="notification-"]');
            allNotifications.forEach(function(li) {
                if (li.style.display !== 'none') {
                    visibleCount++;
                }
            });
            if (visibleCount === 0 && allNotifications.length > 0) {
                notificationsPanel.style.display = 'none';
            }
            
            // Check if panel was previously hidden by user
            const panelHidden = localStorage.getItem('notificationsPanelHidden');
            const toggleText = document.getElementById('toggle-notifications-text');
            if (panelHidden === 'true') {
                notificationsPanel.style.display = 'none';
                if (toggleText) {
                    toggleText.textContent = 'Afficher';
                }
            }
        }
    });

    function toggleNotificationsPanel() {
        const panel = document.getElementById('notifications-panel');
        const toggleBtn = document.getElementById('toggle-notifications-btn');
        const toggleText = document.getElementById('toggle-notifications-text');
        
        if (panel && toggleBtn && toggleText) {
            if (panel.style.display === 'none') {
                panel.style.display = 'block';
                toggleText.textContent = 'Masquer';
                localStorage.setItem('notificationsPanelHidden', 'false');
            } else {
                panel.style.display = 'none';
                toggleText.textContent = 'Afficher';
                localStorage.setItem('notificationsPanelHidden', 'true');
            }
        }
    }

    function hideNotification(notificationId) {
        // Hide the notification element
        const notifElement = document.getElementById('notification-' + notificationId);
        if (notifElement) {
            notifElement.style.display = 'none';
        }

        // Save to localStorage so it stays hidden
        const hiddenNotifications = JSON.parse(localStorage.getItem('hiddenNotifications') || '[]');
        if (!hiddenNotifications.includes(notificationId)) {
            hiddenNotifications.push(notificationId);
            localStorage.setItem('hiddenNotifications', JSON.stringify(hiddenNotifications));
        }

        // Mark as read in database via AJAX
        fetch('/masunzu_bar_hotel/modules/operation/mark_notification_read.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'notification_id=' + notificationId
        }).catch(function(error) {
            console.error('Error marking notification as read:', error);
        });

        // Check if all notifications are hidden, hide the panel
        const notificationsPanel = document.getElementById('notifications-panel');
        if (notificationsPanel) {
            const allNotifications = notificationsPanel.querySelectorAll('li[id^="notification-"]');
            let visibleCount = 0;
            allNotifications.forEach(function(li) {
                if (li.style.display !== 'none') {
                    visibleCount++;
                }
            });
            if (visibleCount === 0 && allNotifications.length > 0) {
                notificationsPanel.style.display = 'none';
            }
        }
    }
</script>
</body>
</html>
