<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$base_dir = '/opt/lampp/htdocs/masunzu_bar_hotel';
require_once $base_dir . '/includes/auth.php';
require_once $base_dir . '/config/database.php';
require_once $base_dir . '/includes/functions.php';

try {
    error_log("DEBUG: admin_dashboard.php loaded at " . date('Y-m-d H:i:s') . ", user_id=" . ($_SESSION['user_id'] ?? 'unset') . ", role_name=" . ($_SESSION['role_name'] ?? 'unset'));

    require_login();

    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception("Database connection failed: " . mysqli_connect_error());
    }

    $colResult = $conn->query("SHOW COLUMNS FROM bank_account LIKE 'is_main'");
    if ($colResult && $colResult->num_rows === 0) {
        $conn->query("ALTER TABLE bank_account ADD COLUMN is_main TINYINT(1) NOT NULL DEFAULT 0 AFTER currency");
    }

    // Restrict to leadership roles
    if (!in_array($_SESSION['role_name'], ['DG', 'DGA', 'Board', 'Admin'])) {
        error_log("ERROR: Unauthorized access attempt by user_id={$_SESSION['user_id']}, role={$_SESSION['role_name']}");
        header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
        exit;
    }

    $user_id = $_SESSION['user_id'];
    $role_name = $_SESSION['role_name'];
    $province_id = $_SESSION['province_id'];
    $full_name = $_SESSION['full_name'];

    // Fetch dépôt name
    $province = 'Tous les Dépôts';
    if ($province_id) {
        $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $province = $stmt->get_result()->fetch_assoc()['name'] ?? 'Inconnu';
        $stmt->close();
    }

    $today = date('Y-m-d');
    
    // ==================== FETCH NOTIFICATIONS FOR DGA/DG ====================
    // Ensure notifications table exists
    $createNotificationSql = "
        CREATE TABLE IF NOT EXISTS user_notification (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            message VARCHAR(255) NOT NULL,
            link VARCHAR(255) DEFAULT NULL,
            is_read TINYINT(1) NOT NULL DEFAULT 0,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_user_read (user_id, is_read),
            CONSTRAINT fk_notification_user_admin FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    if (!$conn->query($createNotificationSql)) {
        error_log('Failed to ensure notification table on admin dashboard: ' . $conn->error);
    }
    
    // Fetch unread notifications for DGA/DG
    $adminNotifications = [];
    $stmtNotif = $conn->prepare("SELECT id, message, link, created_at FROM user_notification WHERE user_id = ? AND is_read = 0 ORDER BY created_at DESC LIMIT 10");
    if ($stmtNotif) {
        $stmtNotif->bind_param("i", $user_id);
        $stmtNotif->execute();
        $adminNotifications = $stmtNotif->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtNotif->close();
        
        // Mark notifications as read when displayed
        if (!empty($adminNotifications)) {
            $ids = array_map('intval', array_column($adminNotifications, 'id'));
            $idList = implode(',', $ids);
            $stmtMark = $conn->prepare("UPDATE user_notification SET is_read = 1 WHERE user_id = ? AND id IN ($idList)");
            if ($stmtMark) {
                $stmtMark->bind_param('i', $user_id);
                $stmtMark->execute();
                $stmtMark->close();
            }
        }
    } else {
        error_log('Failed to fetch notifications for admin dashboard: ' . $conn->error);
    }
    
    // ==================== BIF ACCOUNT BALANCE ====================
    // Get balance of the single BIF account
    $bif_balance = 0.0;
    $bifBalanceSql = "
        SELECT 
            COALESCE(SUM(CASE WHEN bat.direction = 'CREDIT' THEN bat.amount ELSE 0 END), 0) -
            COALESCE(SUM(CASE WHEN bat.direction = 'DEBIT' THEN bat.amount ELSE 0 END), 0) AS total
        FROM bank_account_transaction bat
        INNER JOIN bank_account ba ON bat.bank_account_id = ba.id
        WHERE ba.currency = 'BIF'
    ";
    if ($province_id) {
        $bifBalanceSql .= " AND (bat.province_id = ? OR (bat.province_id IS NULL AND ba.province_id = ?))";
        $stmt = $conn->prepare($bifBalanceSql);
        if ($stmt) {
            $stmt->bind_param("ii", $province_id, $province_id);
        }
    } else {
        $stmt = $conn->prepare($bifBalanceSql);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $bif_balance = (float)($result['total'] ?? 0);
        $stmt->close();
    }
    if ($bif_balance === 0.0) {
        $fallbackSql = "SELECT COALESCE(SUM(balance), 0) AS total FROM bank_account WHERE currency = 'BIF'";
        if ($province_id) {
            $fallbackSql .= " AND (province_id = ? OR province_id IS NULL)";
            $stmt = $conn->prepare($fallbackSql);
            if ($stmt) {
                $stmt->bind_param("i", $province_id);
            }
        } else {
            $stmt = $conn->prepare($fallbackSql);
        }
        if ($stmt) {
            $stmt->execute();
            $bif_balance = (float)($stmt->get_result()->fetch_assoc()['total'] ?? 0);
            $stmt->close();
        }
    }

    $main_account_info = null;
    $stmt = $conn->prepare("SELECT id, account_number, balance FROM bank_account WHERE currency = 'BIF' AND is_main = 1 LIMIT 1");
    if ($stmt) {
        $stmt->execute();
        $main_account_info = $stmt->get_result()->fetch_assoc() ?: null;
        $stmt->close();
    }
    if (!$main_account_info) {
        $stmt = $conn->prepare("SELECT id, account_number, balance FROM bank_account WHERE currency = 'BIF' ORDER BY id ASC LIMIT 1");
        if ($stmt) {
            $stmt->execute();
            $main_account_info = $stmt->get_result()->fetch_assoc() ?: null;
            $stmt->close();
        }
    }
    $main_account_balance = $main_account_info ? (float)$main_account_info['balance'] : $bif_balance;
    $main_account_label = $main_account_info ? 'Compte Principale ' . htmlspecialchars($main_account_info['account_number'] ?? '', ENT_QUOTES, 'UTF-8') : 'Solde Compte BIF';

    // Track deposits that landed on the main bank account today
    $main_account_deposits_today = 0.0;
    if ($main_account_info && !empty($main_account_info['id'])) {
        $depositSql = "
            SELECT COALESCE(SUM(amount), 0) AS total
            FROM bank_deposit
            WHERE bank_account_id = ?
              AND deposit_date >= CURDATE()
              AND deposit_date <= NOW()
        ";
        if ($province_id) {
            $depositSql .= " AND province_id = ?";
            $stmt = $conn->prepare($depositSql);
            if ($stmt) {
                $stmt->bind_param("ii", $main_account_info['id'], $province_id);
            }
        } else {
            $stmt = $conn->prepare($depositSql);
            if ($stmt) {
                $stmt->bind_param("i", $main_account_info['id']);
            }
        }
        if ($stmt) {
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $main_account_deposits_today = (float)($row['total'] ?? 0);
            $stmt->close();
        }
    }

    // ==================== TOTAL LOANS DUE ====================
    // Combine ledger activity with cached customer.loan_balance_bif for resilience.
    $total_loans_due = 0;
    $total_loans_from_customers = 0;
    $total_loans_from_ledger = 0;

    if (!$province_id) {
        $stmt = $conn->prepare("SELECT COALESCE(SUM(loan_balance_bif), 0) AS total FROM customer WHERE loan_balance_bif > 0");
        if ($stmt) {
            $stmt->execute();
            $total_loans_from_customers = (int)($stmt->get_result()->fetch_assoc()['total'] ?? 0);
            $stmt->close();
        }
    }

    $loanLedgerSql = "
        SELECT COALESCE(SUM(outstanding), 0) AS total_outstanding
        FROM (
            SELECT GREATEST(
                COALESCE(SUM(CASE WHEN cll.direction = 'debit' THEN cll.amount_bif ELSE 0 END), 0) -
                COALESCE(SUM(CASE WHEN cll.direction = 'credit' THEN cll.amount_bif ELSE 0 END), 0),
                0
            ) AS outstanding
            FROM customer_loan_ledger cll
            LEFT JOIN unpaid_invoices ui ON cll.reference_table = 'unpaid_invoices' AND cll.reference_id = ui.id
            WHERE cll.entry_type IN ('loan_sale', 'loan_repayment', 'adjustment')
    ";
    if ($province_id) {
        $loanLedgerSql .= " AND (ui.province_id = ? OR ui.province_id IS NULL)";
    }
    $loanLedgerSql .= " GROUP BY cll.customer_id) AS balances";
    $stmt = $conn->prepare($loanLedgerSql);
    if ($stmt) {
        if ($province_id) {
            $stmt->bind_param("i", $province_id);
        }
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $total_loans_from_ledger = (int)($result['total_outstanding'] ?? 0);
        $stmt->close();
    } else {
        error_log("Failed to compute outstanding loans from ledger: " . $conn->error);
    }

    if ($province_id) {
        $total_loans_due = $total_loans_from_ledger;
        if ($total_loans_due <= 0) {
            $fallbackSql = "
                SELECT COALESCE(SUM(loan_outstanding_bif), 0) AS total
                FROM unpaid_invoices
                WHERE is_loan_sale = 1
                  AND loan_outstanding_bif > 0
                  AND (approval_status = 'approved' OR approval_status IS NULL)
                  AND (approval_status != 'rejected')
                  AND province_id = ?
            ";
            $stmt = $conn->prepare($fallbackSql);
            if ($stmt) {
                $stmt->bind_param("i", $province_id);
                $stmt->execute();
                $total_loans_due = (int)($stmt->get_result()->fetch_assoc()['total'] ?? 0);
                $stmt->close();
            }
        }
    } else {
        $total_loans_due = max($total_loans_from_customers, $total_loans_from_ledger);
        if ($total_loans_due <= 0) {
            $stmt = $conn->prepare("
                SELECT COALESCE(SUM(loan_outstanding_bif), 0) AS total
                FROM unpaid_invoices
                WHERE is_loan_sale = 1
                  AND loan_outstanding_bif > 0
                  AND (approval_status = 'approved' OR approval_status IS NULL)
                  AND (approval_status != 'rejected')
            ");
            if ($stmt) {
                $stmt->execute();
                $total_loans_due = (int)($stmt->get_result()->fetch_assoc()['total'] ?? 0);
                $stmt->close();
            }
        }
    }

    // ==================== SALES TODAY (UP TO NOW) ====================
    $sales_today = 0;
    if ($province_id) {
        $stmt = $conn->prepare("SELECT SUM(paid_amount) as total FROM paid_invoices WHERE province_id = ? AND created_at >= CURDATE() AND created_at <= NOW()");
        if ($stmt) {
            $stmt->bind_param("i", $province_id);
            $stmt->execute();
            $sales_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
            $stmt->close();
        }
    } else {
        $stmt = $conn->prepare("SELECT SUM(paid_amount) as total FROM paid_invoices WHERE created_at >= CURDATE() AND created_at <= NOW()");
        if ($stmt) {
            $stmt->execute();
            $sales_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
            $stmt->close();
        }
    }

    // ==================== REQUISITIONS PAID TODAY ====================
    // Use expense.created_at (payment date) instead of requisition.approved_at
    $requisitions_paid_today = 0;
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(r.total_amount), 0) as total
            FROM requisition r
            INNER JOIN expense e ON e.requisition_id = r.id
            WHERE r.status = 'paid' AND DATE(e.created_at) = ? AND r.province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(r.total_amount), 0) as total
            FROM requisition r
            INNER JOIN expense e ON e.requisition_id = r.id
            WHERE r.status = 'paid' AND DATE(e.created_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $requisitions_paid_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
    $stmt->close();
    
    // ==================== COMPREHENSIVE REQUISITION METRICS ====================
    // All statuses breakdown
    $requisition_all_statuses = [
        'pending' => ['count' => 0, 'amount' => 0],
        'approved' => ['count' => 0, 'amount' => 0],
        'paid' => ['count' => 0, 'amount' => 0],
        'rejected' => ['count' => 0, 'amount' => 0],
    ];
    
    $requisitionStatusQuery = "SELECT status, COUNT(*) AS cnt, SUM(total_amount) AS total FROM requisition WHERE 1=1";
    if ($province_id && $role_name !== 'Board') {
        $requisitionStatusQuery .= " AND province_id = ?";
    }
    $requisitionStatusQuery .= " GROUP BY status";
    
    $stmt = $conn->prepare($requisitionStatusQuery);
    if ($province_id && $role_name !== 'Board') {
        $stmt->bind_param("i", $province_id);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $status = strtolower($row['status'] ?? '');
            if (isset($requisition_all_statuses[$status])) {
                $requisition_all_statuses[$status]['count'] = (int)($row['cnt'] ?? 0);
                $requisition_all_statuses[$status]['amount'] = (int)($row['total'] ?? 0);
            }
        }
        $stmt->close();
    }
    
    // Requisitions this week
    $week_start = date('Y-m-d', strtotime('monday this week'));
    $requisitions_this_week = ['count' => 0, 'amount' => 0];
    $requisitionWeekQuery = "SELECT COUNT(*) AS cnt, SUM(total_amount) AS total FROM requisition WHERE DATE(created_at) >= ?";
    if ($province_id && $role_name !== 'Board') {
        $requisitionWeekQuery .= " AND province_id = ?";
    }
    $stmt = $conn->prepare($requisitionWeekQuery);
    if ($province_id && $role_name !== 'Board') {
        $stmt->bind_param("si", $week_start, $province_id);
    } else {
        $stmt->bind_param("s", $week_start);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $requisitions_this_week['count'] = (int)($result['cnt'] ?? 0);
        $requisitions_this_week['amount'] = (int)($result['total'] ?? 0);
        $stmt->close();
    }
    
    // Requisitions this month
    $month_start = date('Y-m-01');
    $requisitions_this_month = ['count' => 0, 'amount' => 0];
    $requisitionMonthQuery = "SELECT COUNT(*) AS cnt, SUM(total_amount) AS total FROM requisition WHERE DATE(created_at) >= ?";
    if ($province_id && $role_name !== 'Board') {
        $requisitionMonthQuery .= " AND province_id = ?";
    }
    $stmt = $conn->prepare($requisitionMonthQuery);
    if ($province_id && $role_name !== 'Board') {
        $stmt->bind_param("si", $month_start, $province_id);
    } else {
        $stmt->bind_param("s", $month_start);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $requisitions_this_month['count'] = (int)($result['cnt'] ?? 0);
        $requisitions_this_month['amount'] = (int)($result['total'] ?? 0);
        $stmt->close();
    }
    
    // Recent requisition payments (last 10)
    $recent_requisition_payments = [];
    $recentPaymentsQuery = "
        SELECT r.id, r.total_amount, r.created_at,
               u.full_name AS initiator_name, p.name AS province_name,
               e.receipt_number, e.created_at AS payment_date,
               cashier.full_name AS cashier_name
        FROM requisition r
        LEFT JOIN user u ON r.user_id = u.id
        LEFT JOIN province p ON r.province_id = p.id
        LEFT JOIN expense e ON e.requisition_id = r.id
        LEFT JOIN user cashier ON e.cashier_id = cashier.id
        WHERE r.status = 'paid'";
    if ($province_id && $role_name !== 'Board') {
        $recentPaymentsQuery .= " AND r.province_id = ?";
    }
    $recentPaymentsQuery .= " ORDER BY e.created_at DESC LIMIT 10";
    $stmt = $conn->prepare($recentPaymentsQuery);
    if ($province_id && $role_name !== 'Board') {
        $stmt->bind_param("i", $province_id);
    }
    if ($stmt) {
        $stmt->execute();
        $recent_requisition_payments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }
    
    // Top requisition categories (from requisition_item)
    $top_requisition_categories = [];
    $categoryQuery = "
        SELECT ri.category, COUNT(DISTINCT ri.requisition_id) AS req_count, SUM(ri.quantity * ri.unit_price) AS total_amount
        FROM requisition_item ri
        JOIN requisition r ON ri.requisition_id = r.id
        WHERE 1=1";
    if ($province_id && $role_name !== 'Board') {
        $categoryQuery .= " AND r.province_id = ?";
    }
    $categoryQuery .= " GROUP BY ri.category ORDER BY total_amount DESC LIMIT 10";
    $stmt = $conn->prepare($categoryQuery);
    if ($province_id && $role_name !== 'Board') {
        $stmt->bind_param("i", $province_id);
    }
    if ($stmt) {
        $stmt->execute();
        $top_requisition_categories = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }

    // ==================== LOANS SALES TODAY ====================
    $loans_sales_today = 0;
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT SUM(cll.amount_bif) as total
            FROM customer_loan_ledger cll
            JOIN unpaid_invoices ui ON cll.reference_id = ui.id AND cll.reference_table = 'unpaid_invoices'
            WHERE cll.entry_type = 'loan_sale' 
              AND cll.direction = 'debit' 
              AND DATE(cll.created_at) = ? 
              AND ui.province_id = ?
              AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
              AND (ui.approval_status != 'rejected')
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT SUM(cll.amount_bif) as total
            FROM customer_loan_ledger cll
            LEFT JOIN unpaid_invoices ui ON cll.reference_id = ui.id AND cll.reference_table = 'unpaid_invoices'
            WHERE cll.entry_type = 'loan_sale' 
              AND cll.direction = 'debit' 
              AND DATE(cll.created_at) = ?
              AND (ui.approval_status IS NULL OR ui.approval_status = 'approved' OR ui.approval_status != 'rejected')
        ");
        $stmt->bind_param("s", $today);
    }
    if ($stmt) {
        $stmt->execute();
        $loans_sales_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
        $stmt->close();
    }

    // ==================== PENDING APPROVAL INVOICES TODAY ====================
    $pending_approval_invoices_today = 0;
    $pending_approval_invoices_amount = 0;
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'pending' AND is_loan_sale = 1 AND DATE(created_at) = ? AND province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'pending' AND is_loan_sale = 1 AND DATE(created_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $pending_approval_invoices_today = (int)($result['cnt'] ?? 0);
    $pending_approval_invoices_amount = (int)($result['total'] ?? 0);
    $stmt->close();

    // ==================== PENDING PAYMENT APPROVED INVOICES TODAY ====================
    $pending_payment_invoices_today = 0;
    $pending_payment_invoices_amount = 0;
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, SUM(paid_amount) as total
            FROM unpaid_invoices
            WHERE approval_status = 'approved' AND is_loan_sale = 1 AND DATE(created_at) = ? AND province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, SUM(paid_amount) as total
            FROM unpaid_invoices
            WHERE approval_status = 'approved' AND is_loan_sale = 1 AND DATE(created_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $pending_payment_invoices_today = $result['cnt'] ?? 0;
    $pending_payment_invoices_amount = $result['total'] ?? 0;
    $stmt->close();

    // ==================== EXPIRED INVOICES TODAY ====================
    $expired_invoices_today = 0;
    $expired_invoices_amount = 0;
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'rejected' AND expires_at IS NOT NULL AND DATE(expires_at) = ? AND province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'rejected' AND expires_at IS NOT NULL AND DATE(expires_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $expired_invoices_today = (int)($result['cnt'] ?? 0);
    $expired_invoices_amount = (int)($result['total'] ?? 0);
    $stmt->close();

    // ==================== INVOICES STATISTICS TODAY ====================
    // Count all unpaid invoices created today (pending, approved, rejected)
    $invoices_created_today = 0;
    $invoices_created_amount = 0;
    $invoices_pending_approval_today = 0;
    $invoices_pending_approval_amount = 0;
    $invoices_approved_today = 0;
    $invoices_approved_amount = 0;
    $invoices_paid_today = 0;
    $invoices_paid_amount = 0;

    // All unpaid invoices created today
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE DATE(created_at) = ? AND province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE DATE(created_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $invoices_created_today = (int)($result['cnt'] ?? 0);
    $invoices_created_amount = (int)($result['total'] ?? 0);
    $stmt->close();

    // Pending approval invoices created today
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'pending' AND is_loan_sale = 1 AND DATE(created_at) = ? AND province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'pending' AND is_loan_sale = 1 AND DATE(created_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $invoices_pending_approval_today = (int)($result['cnt'] ?? 0);
    $invoices_pending_approval_amount = (int)($result['total'] ?? 0);
    $stmt->close();

    // Approved invoices created today
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'approved' AND DATE(created_at) = ? AND province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM unpaid_invoices
            WHERE approval_status = 'approved' AND DATE(created_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $invoices_approved_today = (int)($result['cnt'] ?? 0);
    $invoices_approved_amount = (int)($result['total'] ?? 0);
    $stmt->close();

    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM paid_invoices
            WHERE DATE(created_at) = ? AND province_id = ?
        ");
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(paid_amount), 0) as total
            FROM paid_invoices
            WHERE DATE(created_at) = ?
        ");
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $invoices_paid_today = (int)($result['cnt'] ?? 0);
    $invoices_paid_amount = (int)($result['total'] ?? 0);
    $stmt->close();

    // ==================== INVOICES BY PROVINCE ====================
    $invoices_by_province = [];
    $stmt = $conn->prepare("
        SELECT p.name as province_name,
               COUNT(ui.id) as created_today,
               SUM(CASE WHEN ui.approval_status = 'pending' THEN 1 ELSE 0 END) as pending_approval,
               SUM(CASE WHEN ui.approval_status = 'approved' THEN 1 ELSE 0 END) as approved,
               SUM(CASE WHEN ui.approval_status = 'pending' THEN ui.paid_amount ELSE 0 END) as pending_amount,
               SUM(CASE WHEN ui.approval_status = 'approved' THEN ui.paid_amount ELSE 0 END) as approved_amount,
               COALESCE(SUM(pi.paid_amount), 0) as paid_today
        FROM province p
        LEFT JOIN unpaid_invoices ui ON p.id = ui.province_id AND DATE(ui.created_at) = ? AND ui.is_loan_sale = 1
        LEFT JOIN paid_invoices pi ON p.id = pi.province_id AND DATE(pi.created_at) = ?
        GROUP BY p.id, p.name
        ORDER BY p.name
    ");
    $stmt->bind_param("ss", $today, $today);
    $stmt->execute();
    $invoices_by_province = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== CURRENT CASH BALANCE ====================
    $current_cash_balance = 0;
    $stmt = $conn->prepare("SELECT SUM(balance) as total FROM accountant_balance");
    if ($stmt) {
        $stmt->execute();
        $current_cash_balance = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
        $stmt->close();
    }

    // ==================== TOTAL DEPOSITED TO BANK TODAY ====================
    $total_deposited_today = 0;
    if ($province_id) {
        $stmt = $conn->prepare("SELECT SUM(amount) as total FROM bank_deposit WHERE province_id = ? AND deposit_date >= CURDATE() AND deposit_date <= NOW()");
        if ($stmt) {
            $stmt->bind_param("i", $province_id);
            $stmt->execute();
            $total_deposited_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
            $stmt->close();
        }
    } else {
        $stmt = $conn->prepare("SELECT SUM(amount) as total FROM bank_deposit WHERE deposit_date >= CURDATE() AND deposit_date <= NOW()");
        if ($stmt) {
            $stmt->execute();
            $total_deposited_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
            $stmt->close();
        }
    }

    // ==================== TOTAL EXPENSES TODAY ====================
    $total_expenses_today = 0;
    if ($province_id) {
        $stmt = $conn->prepare("SELECT SUM(amount) as total FROM expense WHERE province_id = ? AND created_at >= CURDATE() AND created_at <= NOW()");
        if ($stmt) {
            $stmt->bind_param("i", $province_id);
            $stmt->execute();
            $total_expenses_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
            $stmt->close();
        }
    } else {
        $stmt = $conn->prepare("SELECT SUM(amount) as total FROM expense WHERE created_at >= CURDATE() AND created_at <= NOW()");
        if ($stmt) {
            $stmt->execute();
            $total_expenses_today = $stmt->get_result()->fetch_assoc()['total'] ?? 0;
            $stmt->close();
        }
    }

    // ==================== SALES BY PROVINCE TODAY ====================
    $sales_by_province = [];
$stmt = $conn->prepare("
    SELECT p.name as province_name,
           COUNT(pi.id) as sales_count,
           SUM(pi.paid_amount) as total_sales,
           SUM(r.total_amount) as total_requisitions,
           SUM(CASE WHEN ui.province_id = p.id 
                    AND (ui.approval_status = 'approved' OR ui.approval_status IS NULL)
                    AND (ui.approval_status != 'rejected')
                    THEN cll.amount_bif ELSE 0 END) as total_loans
        FROM province p
        LEFT JOIN paid_invoices pi ON p.id = pi.province_id AND DATE(pi.created_at) = ?
        LEFT JOIN requisition r ON p.id = r.province_id AND r.status = 'paid' AND DATE(r.created_at) = ?
        LEFT JOIN customer_loan_ledger cll ON cll.entry_type = 'loan_sale' AND cll.direction = 'debit' AND DATE(cll.created_at) = ? AND cll.reference_table = 'unpaid_invoices'
        LEFT JOIN unpaid_invoices ui ON cll.reference_id = ui.id
        GROUP BY p.id, p.name
        ORDER BY p.name
    ");
    if ($stmt) {
        $stmt->bind_param("sss", $today, $today, $today);
        $stmt->execute();
        $sales_by_province = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    } else {
        $sales_by_province = [];
        error_log("Failed to prepare sales by province query: " . $conn->error);
    }

    // ==================== STOCK TRANSFERS TODAY BY PROVINCE ====================
    $stock_transfers_today = [];
    $stmt = $conn->prepare("
        SELECT p.name as province_name,
               COUNT(st.id) as transfer_count,
               SUM(st.quantity) as total_quantity,
               GROUP_CONCAT(DISTINCT st.status ORDER BY st.status SEPARATOR ', ') as status_summary
        FROM province p
        LEFT JOIN stock_transfer st ON p.id = st.province_id AND DATE(st.sent_at) = ?
        GROUP BY p.id, p.name
        ORDER BY p.name
    ");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $stock_transfers_today = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== CASHIERS BALANCES BY PROVINCE ====================
    // Ensure cashier_account table exists
    $createCashierAccountSql = "
        CREATE TABLE IF NOT EXISTS cashier_account (
            id INT AUTO_INCREMENT PRIMARY KEY,
            cashier_id INT NOT NULL,
            shift_id INT NULL,
            transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
            direction ENUM('CREDIT','DEBIT') NOT NULL,
            amount BIGINT(20) NOT NULL,
            reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
            reference_id INT NULL,
            description VARCHAR(255) DEFAULT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            created_by INT NULL,
            INDEX idx_cashier_created (cashier_id, created_at),
            INDEX idx_shift (shift_id),
            INDEX idx_reference (reference_type, reference_id),
            CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
            CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
            CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    @$conn->query($createCashierAccountSql);

    $cashiers_balances = [];
    
    $has_yesno_columns = false;
    $checkColumns = $conn->query("
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
          AND TABLE_NAME = 'cashier_shift_closure' 
          AND COLUMN_NAME = 'deposited_to_bank'
    ");
    if ($checkColumns && $checkColumns->num_rows > 0) {
        $has_yesno_columns = true;
        $checkColumns->close();
    }

    $cashierListQuery = "
        SELECT u.id, u.full_name AS cashier_name, COALESCE(p.name, 'N/A') AS province_name
        FROM user u
        INNER JOIN role r ON u.role_id = r.id AND r.name = 'Caissier_Comptable'
        LEFT JOIN province p ON u.province_id = p.id
        WHERE COALESCE(u.is_active, 1) = 1
    ";
    
    if ($province_id && $role_name !== 'Board') {
        $cashierListQuery .= " AND u.province_id = ?";
    }
    
    $cashierListQuery .= " ORDER BY province_name, cashier_name";
    
    $cashierListStmt = $conn->prepare($cashierListQuery);
    $cashiers = [];
    if ($cashierListStmt) {
        if ($province_id && $role_name !== 'Board') {
            $cashierListStmt->bind_param("i", $province_id);
        }
        $cashierListStmt->execute();
        $cashiers = $cashierListStmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $cashierListStmt->close();
    }

    foreach ($cashiers as $cashier) {
        $cid = (int)($cashier['id'] ?? 0);
        
        $available_from_closed_shifts = 0.0;
        $current_balance = 0.0;
        $argent_disponible = 0.0;

        if ($has_yesno_columns) {
            $stmtClosed = $conn->prepare("
                SELECT 
                    cb.id,
                    cb.shift_start,
                    cb.shift_end,
                    cb.beginning_balance,
                    cb.end_balance,
                    cb.deposits,
                    cb.withdrawals,
                    COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
                    COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
                    csc.counted_amount
                FROM cashier_balance cb
                LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
                WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
                ORDER BY cb.shift_end DESC
            ");
        } else {
            $stmtClosed = $conn->prepare("
                SELECT 
                    cb.id,
                    cb.shift_start,
                    cb.shift_end,
                    cb.beginning_balance,
                    cb.end_balance,
                    cb.deposits,
                    cb.withdrawals
                FROM cashier_balance cb
                WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
                ORDER BY cb.shift_end DESC
            ");
        }

        if ($stmtClosed) {
            $stmtClosed->bind_param("i", $cid);
            $stmtClosed->execute();
            $closedShifts = $stmtClosed->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtClosed->close();

            foreach ($closedShifts as $shift) {
                $shift_id = (int)$shift['id'];
                $end_balance = null;
                if (isset($shift['end_balance']) && $shift['end_balance'] !== null) {
                    $end_balance = (float)$shift['end_balance'];
                } else {
                    $beginning = (float)($shift['beginning_balance'] ?? 0);
                    $deposits = (float)($shift['deposits'] ?? 0);
                    $withdrawals = (float)($shift['withdrawals'] ?? 0);
                    $end_balance = $beginning + $deposits - $withdrawals;
                }
                
                $is_deposited = false;
                $is_transferred = false;

                if ($has_yesno_columns) {
                    $deposited_flag = $shift['deposited_to_bank'] ?? 'no';
                    $transferred_flag = $shift['transferred_to_accountant'] ?? 'no';
                    $is_deposited = strtolower($deposited_flag) === 'yes';
                    $is_transferred = strtolower($transferred_flag) === 'yes';
                } else {
                    $stmtDeposit = $conn->prepare("
                        SELECT COUNT(*) AS has_deposit
                        FROM cashier_shift_deposit csd
                        INNER JOIN bank_deposit bd ON csd.deposit_id = bd.id
                        WHERE csd.shift_id = ?
                    ");
                    if ($stmtDeposit) {
                        $stmtDeposit->bind_param("i", $shift_id);
                        $stmtDeposit->execute();
                        $depositResult = $stmtDeposit->get_result()->fetch_assoc();
                        $is_deposited = (int)($depositResult['has_deposit'] ?? 0) > 0;
                        $stmtDeposit->close();
                    }

                    $stmtTransfer = $conn->prepare("
                        SELECT COUNT(*) AS has_transfer
                        FROM cashier_shift_transfer cst
                        INNER JOIN cash_transfer ct ON cst.transfer_id = ct.id
                        WHERE cst.shift_id = ? AND ct.status = 'received'
                    ");
                    if ($stmtTransfer) {
                        $stmtTransfer->bind_param("i", $shift_id);
                        $stmtTransfer->execute();
                        $transferResult = $stmtTransfer->get_result()->fetch_assoc();
                        $is_transferred = (int)($transferResult['has_transfer'] ?? 0) > 0;
                        $stmtTransfer->close();
                    }
                }
                
                $available_amount = $end_balance;
                
                if ($has_yesno_columns && isset($shift['counted_amount']) && $shift['counted_amount'] !== null) {
                    $available_amount = (float)$shift['counted_amount'];
                }
                
                if ($available_amount > 0 && !$is_deposited && !$is_transferred) {
                    $available_from_closed_shifts += $available_amount;
                }
            }
        }

        $stmtActive = $conn->prepare("SELECT id, shift_start, beginning_balance, deposits FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
        if ($stmtActive) {
            $stmtActive->bind_param("i", $cid);
            $stmtActive->execute();
            $activeShift = $stmtActive->get_result()->fetch_assoc();
            $stmtActive->close();
            
            if ($activeShift && !empty($activeShift['shift_start'])) {
                $shift_start = $activeShift['shift_start'];
                $beginning_balance = (float)($activeShift['beginning_balance'] ?? 0);
                $shift_deposits = (float)($activeShift['deposits'] ?? 0);

                $shift_expenses = 0.0;
                $stmtExp = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
                if ($stmtExp) {
                    $stmtExp->bind_param("is", $cid, $shift_start);
                    $stmtExp->execute();
                    $shift_expenses = (float)($stmtExp->get_result()->fetch_assoc()['total_expenses'] ?? 0);
                    $stmtExp->close();
                }

                $shift_transfers = 0.0;
                $stmtTrans = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transfers FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
                if ($stmtTrans) {
                    $stmtTrans->bind_param("is", $cid, $shift_start);
                    $stmtTrans->execute();
                    $shift_transfers = (float)($stmtTrans->get_result()->fetch_assoc()['total_transfers'] ?? 0);
                    $stmtTrans->close();
                }
                
                $current_balance = $beginning_balance + $shift_deposits - $shift_expenses - $shift_transfers;
                if ($current_balance < 0) {
                    $current_balance = 0;
                }
            }
        }

        $argent_disponible = $available_from_closed_shifts + $current_balance;
        if ($argent_disponible < 0) {
            $argent_disponible = 0;
        }

        $cashiers_balances[] = [
            'province_name' => $cashier['province_name'],
            'cashier_name' => $cashier['cashier_name'],
            'balance' => $argent_disponible
        ];
    }

    // ==================== LOAN REPAYMENTS TODAY ====================
    // Calculate loan repayments from loan_payment table (more accurate)
    $loan_repayments_today_count = 0;
    $loan_repayments_today_amount = 0;
    $loan_repayments_by_province = [];
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(payment_amount), 0) as total
            FROM loan_payment
            WHERE DATE(payment_date) = ? AND province_id = ?
        ");
        if ($stmt) {
            $stmt->bind_param("si", $today, $province_id);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $loan_repayments_today_count = (int)($result['cnt'] ?? 0);
            $loan_repayments_today_amount = (int)($result['total'] ?? 0);
            $stmt->close();
        }
    } else {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as cnt, COALESCE(SUM(payment_amount), 0) as total
            FROM loan_payment
            WHERE DATE(payment_date) = ?
        ");
        if ($stmt) {
            $stmt->bind_param("s", $today);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $loan_repayments_today_count = (int)($result['cnt'] ?? 0);
            $loan_repayments_today_amount = (int)($result['total'] ?? 0);
            $stmt->close();
        }
    }

    // Loan repayments by province (today)
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT p.name AS province_name,
                   COUNT(lp.id) AS repayment_count,
                   COALESCE(SUM(lp.payment_amount), 0) AS total_amount
            FROM province p
            LEFT JOIN loan_payment lp ON lp.province_id = p.id AND DATE(lp.payment_date) = ?
            WHERE p.id = ?
            GROUP BY p.id, p.name
        ");
        if ($stmt) {
            $stmt->bind_param("si", $today, $province_id);
            $stmt->execute();
            $loan_repayments_by_province = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
        }
    } else {
        $stmt = $conn->prepare("
            SELECT p.name AS province_name,
                   COUNT(lp.id) AS repayment_count,
                   COALESCE(SUM(lp.payment_amount), 0) AS total_amount
            FROM province p
            LEFT JOIN loan_payment lp ON lp.province_id = p.id AND DATE(lp.payment_date) = ?
            GROUP BY p.id, p.name
            ORDER BY p.name
        ");
        if ($stmt) {
            $stmt->bind_param("s", $today);
            $stmt->execute();
            $loan_repayments_by_province = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
        }
    }

    // ==================== REQUISITIONS BY PROVINCE ====================
    $requisitions_by_province = [];
    $stmt = $conn->prepare("
        SELECT p.name as province_name,
               COUNT(r.id) as requisition_count,
               SUM(r.total_amount) as total_amount
        FROM province p
        LEFT JOIN requisition r ON p.id = r.province_id AND DATE(r.created_at) = ?
        GROUP BY p.id, p.name
        ORDER BY p.name
    ");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $requisitions_by_province = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== REQUISITIONS PAID TODAY (UPDATED) ====================
    // Use expense.created_at (payment date) for accurate payment tracking
    $requisitions_paid_count = 0;
    $requisitions_paid_amount = 0;
    $requisitionsPaidTodayQuery = "
        SELECT COUNT(DISTINCT r.id) as cnt, COALESCE(SUM(r.total_amount), 0) as total
        FROM requisition r
        INNER JOIN expense e ON e.requisition_id = r.id
        WHERE r.status = 'paid' AND DATE(e.created_at) = ?";
    if ($province_id && $role_name !== 'Board') {
        $requisitionsPaidTodayQuery .= " AND r.province_id = ?";
    }
    $stmt = $conn->prepare($requisitionsPaidTodayQuery);
    if ($province_id && $role_name !== 'Board') {
        $stmt->bind_param("si", $today, $province_id);
    } else {
        $stmt->bind_param("s", $today);
    }
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $requisitions_paid_count = $result['cnt'] ?? 0;
    $requisitions_paid_amount = $result['total'] ?? 0;
    $stmt->close();

    // ==================== ACTIVE USERS ====================
    $active_users = [];
    $stmt = $conn->prepare("
        SELECT u.id, u.full_name, u.username, p.name as province_name, r.name as role_name,
               MAX(CASE WHEN l.action = 'login' THEN l.created_at END) as login_time
        FROM user u
        LEFT JOIN province p ON u.province_id = p.id
        LEFT JOIN role r ON u.role_id = r.id
        LEFT JOIN log l ON u.id = l.user_id
        WHERE l.action = 'login' AND DATE(l.created_at) = ?
        GROUP BY u.id, u.full_name, u.username, p.name, r.name
        ORDER BY login_time DESC
    ");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $active_users = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== ACTIVE CASHIERS ====================
    $active_cashiers = [];
    $stmt = $conn->prepare("
        SELECT u.full_name as cashier_name,
               p.name as province_name,
               cb.shift_start,
               cb.shift_end
        FROM cashier_balance cb
        JOIN user u ON cb.cashier_id = u.id
        LEFT JOIN province p ON u.province_id = p.id
        WHERE cb.shift_end IS NULL
        ORDER BY cb.shift_start DESC
    ");
    $stmt->execute();
    $active_cashiers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== TOP 10 PRODUCTS BY SALES TODAY ====================
    $top_products_sales = [];
    $stmt = $conn->prepare("
        SELECT pr.name, SUM(pii.quantity) as total_quantity
        FROM paid_invoices pi
        JOIN paid_invoice_items pii ON pi.id = pii.invoice_id
        JOIN product pr ON pii.product_id = pr.id
        WHERE DATE(pi.created_at) = ?
        GROUP BY pii.product_id
        ORDER BY total_quantity DESC
        LIMIT 10
    ");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $top_products_sales = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== TOP 10 PRODUCTS BY AMOUNT RECEIVED TODAY ====================
    $top_products_amount = [];
    $stmt = $conn->prepare("
        SELECT pr.name, SUM(pi.paid_amount) as total_amount
        FROM paid_invoices pi
        JOIN paid_invoice_items pii ON pi.id = pii.invoice_id
        JOIN product pr ON pii.product_id = pr.id
        WHERE DATE(pi.created_at) = ?
        GROUP BY pii.product_id
        ORDER BY total_amount DESC
        LIMIT 10
    ");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $top_products_amount = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== TOP 3 CASHIERS TODAY ====================
    $top_cashiers_today = [];
    $stmt = $conn->prepare("
        SELECT u.full_name as cashier_name,
               p.name as province_name,
               SUM(pi.paid_amount) as total_amount
        FROM paid_invoices pi
        JOIN user u ON pi.created_by = u.id
        LEFT JOIN province p ON u.province_id = p.id
        WHERE DATE(pi.created_at) = ?
        GROUP BY u.id, u.full_name, p.name
        ORDER BY total_amount DESC
        LIMIT 3
    ");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $top_cashiers_today = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== MAIN STOCK OVERVIEW ====================
    $main_stock_products = [];
    $stmt = $conn->prepare("
        SELECT pr.id,
               pr.name,
               pr.crate_quantity,
               COALESCE(ms.quantity, 0) as total_quantity
        FROM product pr
        LEFT JOIN main_stock ms ON pr.id = ms.product_id
        ORDER BY pr.name
    ");
    $stmt->execute();
    $main_stock_products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== STOCK BY PROVINCE (SPLIT STOCK ONLY) WITH PRODUCTS ====================
    $stock_by_province = [];
    $stmt = $conn->prepare("
        SELECT p.id, p.name as province_name,
               pr.id as product_id,
               pr.name as product_name,
               pr.crate_quantity,
               COALESCE(ws.quantity, 0) as wholesale_qty,
               COALESCE(rs.quantity, 0) as retail_qty
        FROM province p
        LEFT JOIN (
            SELECT DISTINCT product_id, province_id FROM wholesale_stock
            UNION
            SELECT DISTINCT product_id, province_id FROM retail_stock
        ) AS products ON p.id = products.province_id
        LEFT JOIN product pr ON products.product_id = pr.id
        LEFT JOIN wholesale_stock ws ON p.id = ws.province_id AND pr.id = ws.product_id
        LEFT JOIN retail_stock rs ON p.id = rs.province_id AND pr.id = rs.product_id
        WHERE COALESCE(ws.quantity, 0) > 0 OR COALESCE(rs.quantity, 0) > 0
        ORDER BY p.name, pr.name
    ");
    $stmt->execute();
    $stock_result = $stmt->get_result();
    
    $province_data = [];
    while ($row = $stock_result->fetch_assoc()) {
        $province_id = $row['id'];
        $province_name = $row['province_name'];
        
        if (!isset($province_data[$province_id])) {
            $province_data[$province_id] = [
                'province_name' => $province_name,
                'products' => [],
                'total_wholesale' => 0,
                'total_retail' => 0
            ];
        }
        
        $wholesale_qty = (int)$row['wholesale_qty'];
        $retail_qty = (int)$row['retail_qty'];
        
        if ($wholesale_qty > 0 || $retail_qty > 0) {
            $province_data[$province_id]['products'][] = [
                'product_name' => $row['product_name'],
                'product_id' => $row['product_id'],
                'crate_quantity' => $row['crate_quantity'],
                'wholesale_qty' => $wholesale_qty,
                'retail_qty' => $retail_qty
            ];
            
            $province_data[$province_id]['total_wholesale'] += $wholesale_qty;
            $province_data[$province_id]['total_retail'] += $retail_qty;
        }
    }
    
    foreach ($province_data as $data) {
        $stock_by_province[] = $data;
    }
    $stmt->close();

    // ==================== ACTIVE USERS ====================
    $active_users = [];
    $stmt = $conn->prepare("
        SELECT u.id, u.full_name, u.username, r.name as role_name, p.name as province_name,
               us.login_time, us.session_id,
               TIMESTAMPDIFF(MINUTE, us.login_time, NOW()) as minutes_logged_in
        FROM user_session us
        JOIN user u ON us.user_id = u.id
        LEFT JOIN role r ON u.role_id = r.id
        LEFT JOIN province p ON u.province_id = p.id
        WHERE us.logout_time IS NULL
        ORDER BY us.login_time DESC
    ");
    $stmt->execute();
    $active_users = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // ==================== ACTIVE CASHIERS ====================
    $active_cashiers = [];
    $stmt = $conn->prepare("
        SELECT u.full_name as cashier_name, p.name as province_name,
               cb.shift_start, cb.shift_end,
               TIMESTAMPDIFF(MINUTE, cb.shift_start, NOW()) as minutes_in_shift
        FROM cashier_balance cb
        JOIN user u ON cb.cashier_id = u.id
        LEFT JOIN province p ON u.province_id = p.id
        WHERE cb.shift_end IS NULL
        ORDER BY cb.shift_start ASC
    ");
    $stmt->execute();
    $active_cashiers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Fetch pending invoices from unpaid_invoices table
    $invoices = [];
    $invoiceQuery = "
        SELECT
            ui.id,
            ui.status,
            ui.approval_status,
            ui.created_at,
            COALESCE(
                NULLIF(ui.subtotal_amount + ui.tax_amount, 0),
                NULLIF(SUM(ii.quantity * ii.unit_price), 0),
                ui.paid_amount,
                0
            ) AS total_amount,
            u.full_name AS initiator_name,
            p.name AS province_name
        FROM unpaid_invoices ui
        LEFT JOIN invoice_items ii ON ui.id = ii.invoice_id
        JOIN user u ON ui.created_by = u.id
        JOIN province p ON ui.province_id = p.id
        WHERE ui.approval_status = 'pending'
          AND ui.is_loan_sale = 1
    ";
    $invoiceTypes = '';
    $invoiceParams = [];
    if ($province_id && $role_name !== 'Board') {
        $invoiceQuery .= " AND ui.province_id = ?";
        $invoiceTypes .= 'i';
        $invoiceParams[] = $province_id;
    }
    $invoiceQuery .= "
        GROUP BY ui.id, ui.status, ui.approval_status, ui.created_at, ui.subtotal_amount, ui.tax_amount, ui.paid_amount, u.full_name, p.name
        ORDER BY ui.created_at DESC
    ";
    $stmt = $conn->prepare($invoiceQuery);
    if ($stmt === false) {
        throw new Exception('Impossible de préparer la requête des commandes en attente: ' . $conn->error);
    }
    if ($invoiceTypes !== '') {
        $stmt->bind_param($invoiceTypes, ...$invoiceParams);
    }
    $stmt->execute();
    $invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    $transferMetrics = [
        'pending' => ['count' => 0, 'amount' => 0],
        'approved' => ['count' => 0, 'amount' => 0],
    ];
    $transferQuery = "SELECT ct.status, COUNT(*) AS cnt, SUM(ct.amount) AS total
                      FROM cash_transfer ct
                      INNER JOIN user cu ON cu.id = ct.cashier_id";
    if ($province_id && $role_name !== 'Board') {
        $transferQuery .= " WHERE cu.province_id = ? GROUP BY ct.status";
        $stmt = $conn->prepare($transferQuery);
        $stmt->bind_param("i", $province_id);
    } else {
        $transferQuery .= " GROUP BY ct.status";
        $stmt = $conn->prepare($transferQuery);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $status = strtolower($row['status'] ?? '');
            $count = (int)($row['cnt'] ?? 0);
            $amount = (int)($row['total'] ?? 0);
            if ($status === 'pending') {
                $transferMetrics['pending']['count'] = $count;
                $transferMetrics['pending']['amount'] = $amount;
            } elseif ($status === 'received' || $status === 'approved') {
                $transferMetrics['approved']['count'] = $count;
                $transferMetrics['approved']['amount'] = $amount;
            }
        }
        $stmt->close();
    }

    $invoiceStatusMetrics = [
        'pending'  => ['count' => 0, 'amount' => 0],
        'approved' => ['count' => 0, 'amount' => 0],
    ];

    // 1) Pending / approved invoices that are still in unpaid_invoices
    $invoiceQuery = "SELECT approval_status, COUNT(*) AS cnt, SUM(paid_amount) AS total
                     FROM unpaid_invoices
                     WHERE approval_status IN ('pending','approved')
                       AND is_loan_sale = 1";
    if ($province_id && $role_name !== 'Board') {
        $invoiceQuery .= " AND province_id = ? GROUP BY approval_status";
        $stmt = $conn->prepare($invoiceQuery);
        $stmt->bind_param("i", $province_id);
    } else {
        $invoiceQuery .= " GROUP BY approval_status";
        $stmt = $conn->prepare($invoiceQuery);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $status = strtolower($row['approval_status'] ?? '');
            $count  = (int)($row['cnt'] ?? 0);
            $amount = (int)($row['total'] ?? 0);
            if ($status === 'pending' || $status === 'approved') {
                $invoiceStatusMetrics[$status]['count']  = $count;
                $invoiceStatusMetrics[$status]['amount'] = $amount;
            }
        }
        $stmt->close();
    }

    /**
     * 2) Approved invoices that have already been moved to paid_invoices.
     * When an invoice is paid, it is removed from unpaid_invoices and inserted
     * into paid_invoices with approval_status = 'approved'. To keep the admin
     * dashboard metrics accurate, we need to include those as well in the
     * "approved" bucket.
     */
    $paidInvoiceQuery = "SELECT COUNT(*) AS cnt, SUM(paid_amount) AS total
                         FROM paid_invoices
                         WHERE approval_status = 'approved'
                           AND is_loan_sale = 1";
    if ($province_id && $role_name !== 'Board') {
        $paidInvoiceQuery .= " AND province_id = ?";
        $stmt = $conn->prepare($paidInvoiceQuery);
        $stmt->bind_param("i", $province_id);
    } else {
        $stmt = $conn->prepare($paidInvoiceQuery);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $approvedCount  = (int)($row['cnt'] ?? 0);
            $approvedAmount = (int)($row['total'] ?? 0);
            // Add paid_invoices numbers on top of the approved unpaid_invoices
            $invoiceStatusMetrics['approved']['count']  += $approvedCount;
            $invoiceStatusMetrics['approved']['amount'] += $approvedAmount;
        }
        $stmt->close();
    }

    if ($invoiceStatusMetrics['pending']['count'] > 0) {
        $adminNotifications[] = [
            'id' => 0,
            'message' => sprintf(
                "Il y a %s commandes en attente d'approbation.",
                number_format($invoiceStatusMetrics['pending']['count'], 0, '.', ',')
            ),
            'link' => '/masunzu_bar_hotel/modules/admin/approve_invoices.php',
            'created_at' => date('Y-m-d H:i:s'),
        ];
    }

    // Legacy requisitionStatusMetrics for backward compatibility
    $requisitionStatusMetrics = [
        'pending' => ['count' => $requisition_all_statuses['pending']['count'], 'amount' => $requisition_all_statuses['pending']['amount']],
        'approved' => ['count' => $requisition_all_statuses['approved']['count'], 'amount' => $requisition_all_statuses['approved']['amount']],
    ];
    
    // Fetch province-based low stock alerts
    $province_low_stock = [];
    $stmt = $conn->prepare("
        SELECT 
            pr.name AS province_name,
            p.id AS product_id,
            p.name AS product_name,
            ps.quantity,
            ps.reserved_quantity,
            ps.unit_type,
            p.crate_quantity,
            p.low_stock_alert_crates,
            p.low_stock_alert_units
        FROM province_stock ps
        JOIN product p ON ps.product_id = p.id
        JOIN province pr ON ps.province_id = pr.id
        WHERE ps.quantity IS NOT NULL
    ");
    if ($stmt) {
        $stmt->execute();
        $province_stocks = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();

        foreach ($province_stocks as $stock) {
            $crateQuantity = max(1, (int)($stock['crate_quantity'] ?? 12));
            $rawQuantity = (int)($stock['quantity'] ?? 0);
            $reserved = (int)($stock['reserved_quantity'] ?? 0);
            $available = max(0, $rawQuantity - $reserved);

            $availableUnits = $stock['unit_type'] === 'crates'
                ? $available * $crateQuantity
                : $available;
            $availableCrates = $availableUnits / $crateQuantity;

            $thresholdUnits = (int)($stock['low_stock_alert_units'] ?? 0);
            if ($thresholdUnits <= 0) {
                $thresholdCrates = (int)($stock['low_stock_alert_crates'] ?? 0);
                if ($thresholdCrates > 0) {
                    $thresholdUnits = $thresholdCrates * $crateQuantity;
                }
            }
            if ($thresholdUnits <= 0) {
                continue; // no alert configured for this product
            }

            if ($availableUnits <= $thresholdUnits) {
                $thresholdCrates = $thresholdUnits / $crateQuantity;
                $stock['available_crates'] = $availableCrates;
                $stock['available_units'] = $availableUnits;
                $stock['threshold_crates'] = $thresholdCrates;
                $stock['threshold_units'] = $thresholdUnits;
                $stock['crate_size'] = $crateQuantity;
                $stock['reserved_quantity'] = $reserved;
                $province_low_stock[$stock['province_name']][] = $stock;
            }
        }
    }
    
    // Fetch all accountant balances
    $accountant_balances = [];
    $stmtAccountants = $conn->prepare("
        SELECT 
            u.id AS accountant_id,
            u.full_name AS accountant_name,
            p.name AS province_name,
            COALESCE(ab.balance, 0) AS balance,
            COALESCE(SUM(e.amount), 0) AS total_expenses,
            COALESCE(ab.balance, 0) - COALESCE(SUM(e.amount), 0) AS available_balance,
            ab.last_updated
        FROM user u
        JOIN role r ON u.role_id = r.id
        LEFT JOIN province p ON u.province_id = p.id
        LEFT JOIN accountant_balance ab ON u.id = ab.accountant_id
        LEFT JOIN expense e ON u.id = e.accountant_id
        WHERE r.name = 'Accountant'
        GROUP BY u.id, u.full_name, p.name, ab.balance, ab.last_updated
        ORDER BY p.name, u.full_name
    ");
    if ($stmtAccountants) {
        $stmtAccountants->execute();
        $accountant_balances = $stmtAccountants->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtAccountants->close();
    }
    
    $conn->close();

} catch (Exception $e) {
    error_log("ERROR in admin_dashboard.php: " . $e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
    include $base_dir . '/includes/footer.php';
    exit;
}

include $base_dir . '/includes/header.php';
?>
<style>
    .card { background: #FFFFFF; border-radius: 8px; padding: 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin-bottom: 20px; }
    .metric-card { background: #FFFFFF; padding: 15px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); text-align: center; }
    .metric-label { color: #4B2F1F; font-size: 12px; text-transform: uppercase; font-weight: 600; opacity: 0.7; }
    .metric-value { color: #4B2F1F; font-size: 24px; font-weight: bold; margin-top: 8px; }
    .section-title { color: #4B2F1F; font-size: 18px; font-weight: 600; margin: 30px 0 15px 0; border-bottom: 2px solid #4B2F1F; padding-bottom: 10px; }
    .subsection-title { color: #4B2F1F; font-size: 14px; font-weight: 600; margin: 15px 0 10px 0; }
    button[onclick*="location.reload"]:hover {
        background-color: #F4A261 !important;
        color: #4B2F1F !important;
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.2);
    }
    table { width: 100%; border-collapse: collapse; }
    th { background-color: #4B2F1F; color: #F4F0E4; padding: 12px; text-align: left; }
    td { padding: 10px 12px; border-bottom: 1px solid #E0E0E0; }
    tr:hover { background-color: #F9F9F9; }
    .dashboard-container { display: flex; gap: 20px; }
    .nav-sidebar { position: sticky; top: 100px; width: 220px; flex-shrink: 0; height: fit-content; background: #FFFFFF; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); overflow-y: auto; max-height: calc(100vh - 120px); }
    .nav-sidebar a { display: block; padding: 12px 16px; color: #4B2F1F; text-decoration: none; font-size: 13px; font-weight: 500; border-left: 3px solid transparent; transition: all 0.2s; }
    .nav-sidebar a:hover { background-color: #F4F0E4; border-left-color: #F4A261; }
    .nav-sidebar a.active { background-color: #F4F0E4; border-left-color: #4B2F1F; color: #4B2F1F; font-weight: 600; }
    .main-content { flex: 1; padding-bottom: 40px; }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const navLinks = document.querySelectorAll('.nav-sidebar a');
        const sections = document.querySelectorAll('[id^="section-"]');
        
        window.addEventListener('scroll', function() {
            let current = '';
            sections.forEach(section => {
                const sectionTop = section.offsetTop;
                const sectionHeight = section.clientHeight;
                if (pageYOffset >= sectionTop - 200) {
                    current = section.getAttribute('id');
                }
            });
            
            navLinks.forEach(link => {
                link.classList.remove('active');
                if (link.getAttribute('href') === '#' + current) {
                    link.classList.add('active');
                }
            });
        });
    });
</script>

<div style="padding: 0; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh;">
    <div style="background-color: #F4F0E4; padding: 30px 15px; border-radius: 0; box-shadow: none; min-height: 100vh;">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px;">
<div>
                <h2 style="color: #4B2F1F; margin: 0 0 5px 0;">Tableau de Bord Admin</h2>
<p style="color: #666; font-size: 14px; margin: 0;">Bienvenue, <?php echo htmlspecialchars($full_name); ?> • <?php echo htmlspecialchars($role_name); ?> • Dépôt : <?php echo htmlspecialchars($province); ?></p>
            </div>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 10px 20px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 14px; transition: background-color 0.3s;">
                🔄 Actualiser
            </button>
        </div>

        <?php if (!empty($adminNotifications)): ?>
        <div style="background-color: #E3F2FD; border-left: 4px solid #2196F3; padding: 20px; border-radius: 8px; margin-bottom: 30px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
            <div style="display: flex; justify-content: space-between; align-items: flex-start;">
                <div style="flex: 1;">
                    <h3 style="color: #1565C0; margin: 0 0 15px 0; font-size: 18px; font-weight: 600;">
                        🔔 Notifications (<?php echo count($adminNotifications); ?>)
                    </h3>
                    <ul style="list-style: none; margin: 0; padding: 0;">
                        <?php foreach ($adminNotifications as $note): ?>
                            <li style="padding: 12px 0; border-bottom: 1px solid #BBDEFB;">
                                <div style="display: flex; justify-content: space-between; align-items: flex-start;">
                                    <div style="flex: 1;">
                                        <span style="color: #000000; font-size: 14px; line-height: 1.5;">
                                            <?php echo htmlspecialchars($note['message'], ENT_QUOTES, 'UTF-8'); ?>
                                        </span><br>
                                        <small style="color: #666; font-size: 12px;">
                                            <?php echo date('Y-m-d H:i', strtotime($note['created_at'])); ?>
                                        </small>
                                        <?php if (!empty($note['link'])): ?>
                                            <?php
                                            // Validate and fix notification links
                                            $notificationLink = trim($note['link']);
                                            // If link doesn't start with /, make it absolute
                                            if (!empty($notificationLink) && $notificationLink[0] !== '/') {
                                                $notificationLink = '/masunzu_bar_hotel/' . ltrim($notificationLink, '/');
                                            }
                                            // Ensure link is not empty and doesn't point to non-existent files
                                            $invalidLinks = ['dga_dashboard.php', 'dga_dashboard', 'DGA_dashboard.php', 'DGA_dashboard'];
                                            if (in_array(basename($notificationLink), $invalidLinks)) {
                                                // Redirect to admin dashboard instead
                                                $notificationLink = '/masunzu_bar_hotel/dashboards/admin_dashboard.php';
                                            }
                                            ?>
                                            <div style="margin-top: 8px;">
                                                <a href="<?php echo htmlspecialchars($notificationLink, ENT_QUOTES, 'UTF-8'); ?>" 
                                                   style="background-color: #2196F3; color: #FFFFFF; padding: 6px 12px; border-radius: 4px; text-decoration: none; font-size: 12px; font-weight: 500; display: inline-block;">
                                                    Voir Détails →
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if ($invoiceStatusMetrics['pending']['count'] > 0): ?>
        <div style="background-color: #FFF3CD; border-left: 4px solid #FF9800; padding: 20px; border-radius: 8px; margin-bottom: 30px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <h3 style="color: #856404; margin: 0 0 8px 0; font-size: 18px; font-weight: 600;">⚠️ Approbations de Commandes en Attente</h3>
                    <p style="color: #856404; margin: 0; font-size: 14px;">
                        <strong><?php echo $invoiceStatusMetrics['pending']['count']; ?></strong> commandes en attente d'approbation
                        <strong>(<?php echo number_format($invoiceStatusMetrics['pending']['amount'], 0, '.', ','); ?> BIF)</strong>
                    </p>
                </div>
                <a href="/masunzu_bar_hotel/modules/admin/approve_invoices.php" style="background-color: #FF9800; color: #FFFFFF; padding: 10px 20px; border-radius: 5px; text-decoration: none; font-weight: 600; font-size: 14px; display: inline-block; transition: background-color 0.3s;">
                    Examiner les Commandes →
                </a>
            </div>
        </div>
        <?php endif; ?>

        <div class="dashboard-container">
            <!-- NAVIGATION SIDEBAR -->
            <div class="nav-sidebar">
                <a href="#section-account-balance">Solde Compte</a>
                <a href="#section-sales-today">Ventes Aujourd'hui</a>
                <a href="#section-sales-depot">Ventes par Dépôt</a>
                <a href="#section-pending-approvals">Approbations en Attente</a>
                <a href="#section-loan-repayments">Remboursements Crédits</a>
                <a href="#section-invoices">Commandes</a>
                <a href="#section-stock-transfers">Transfers de Stock</a>
                <a href="#section-stock-overview">Vue d'Ensemble Stock</a>
                <a href="#section-operational">Aperçu Opérationnel</a>
                <a href="#section-cashier-balances">Soldes Caissiers</a>
                <a href="#section-accountant-balances">Soldes Comptables</a>
                <a href="#section-requisitions">Réquisitions</a>
                <a href="#section-active-cashiers">Sessions Actives</a>
                <a href="#section-top-performers">Top Performeurs</a>
                <a href="/masunzu_bar_hotel/modules/admin/low_stock_alerts.php">Alertes Stock Faible</a>
            </div>

            <!-- MAIN CONTENT -->
            <div class="main-content">
        <!-- Loan repayments moved below account balance -->

        <!-- SECTION 1: ACCOUNT BALANCE -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" style="margin: 0; flex: 1;">Solde Compte</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div id="section-account-balance" style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 30px;">
            <div class="metric-card" style="background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%);">
            <div class="metric-label" style="color: #FFFFFF; opacity: 0.9;">Dépôts Aujourd'hui • <?php echo htmlspecialchars($main_account_label, ENT_QUOTES, 'UTF-8'); ?></div>
            <div class="metric-value" style="color: #FFFFFF;"><?php echo number_format($main_account_deposits_today, 0, '.', ','); ?> BIF</div>
            <div style="color: #E8F5E9; font-size: 12px; margin-top: 6px; font-weight: 600;">Solde actuel: <?php echo number_format($main_account_balance, 0, '.', ','); ?> BIF</div>
            </div>
            <div class="metric-card" style="background: linear-gradient(135deg, #E91E63 0%, #F06292 100%);">
                <div class="metric-label" style="color: #FFFFFF; opacity: 0.9;">Total Crédits Dûs</div>
                <div class="metric-value" style="color: #FFFFFF;"><?php echo number_format($total_loans_due, 0, '.', ','); ?> BIF</div>
            </div>
        </div>

        <div id="section-loan-repayments" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); gap: 15px; margin-bottom: 30px;">
            <div class="metric-card">
                <div class="metric-label">Remboursements Crédits (Aujourd'hui)</div>
                <div class="metric-value" style="color: #357A38;"><?php echo number_format($loan_repayments_today_amount, 0, '.', ','); ?> BIF</div>
                <p style="color: #666; font-size: 12px; margin-top: 6px;"><?php echo number_format($loan_repayments_today_count, 0, '.', ','); ?> remboursement(s)</p>
            </div>
            <div class="card" style="margin: 0;">
                <table>
                    <thead>
                        <tr>
                            <th>Dépôt</th>
                            <th>Nombre</th>
                            <th>Montant Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($loan_repayments_by_province as $row): ?>
                        <tr>
                            <td style="font-weight: 500;"><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                            <td><?php echo number_format((int)($row['repayment_count'] ?? 0), 0, '.', ','); ?></td>
                            <td><?php echo number_format((int)($row['total_amount'] ?? 0), 0, '.', ','); ?> BIF</td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if (empty($loan_repayments_by_province)): ?>
                        <tr><td colspan="3" style="text-align:center; padding: 12px; color: #666;">Aucun remboursement enregistré aujourd'hui.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- SECTION 2: SALES TODAY -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-sales-today" style="margin: 0; flex: 1;">Ventes Aujourd'hui</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 30px;">
            <div class="metric-card">
                <div class="metric-label">Ventes Totales</div>
                <div class="metric-value" style="color: #4CAF50;"><?php echo number_format($sales_today, 0, '.', ','); ?> BIF</div>
            </div>
            <div class="metric-card">
                <div class="metric-label">Réquisitions Payées</div>
                <div class="metric-value" style="color: #2196F3;"><?php echo number_format($requisitions_paid_today, 0, '.', ','); ?> BIF</div>
            </div>
            <div class="metric-card">
                <div class="metric-label">Ventes de Crédits</div>
                <div class="metric-value" style="color: #FF9800;"><?php echo number_format($loans_sales_today, 0, '.', ','); ?> BIF</div>
            </div>
        </div>

        <!-- SECTION 3: SALES BY DÉPÔT TODAY -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-sales-depot" style="margin: 0; flex: 1;">Ventes par Dépôt</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div class="card">
            <table>
                <thead>
                    <tr>
                        <th>Dépôt</th>
                        <th>Nombre Ventes</th>
                        <th>Ventes Totales</th>
                        <th>Réquisitions</th>
                        <th>Crédits</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($sales_by_province as $row): ?>
                    <tr>
                        <td style="font-weight: 500;"><?php echo htmlspecialchars($row['province_name']); ?></td>
                        <td><?php echo $row['sales_count'] ?? 0; ?></td>
                        <td><?php echo number_format($row['total_sales'] ?? 0, 0, '.', ','); ?> BIF</td>
                        <td><?php echo number_format($row['total_requisitions'] ?? 0, 0, '.', ','); ?> BIF</td>
                        <td><?php echo number_format($row['total_loans'] ?? 0, 0, '.', ','); ?> BIF</td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>


        <!-- SECTION 5: INVOICES STATISTICS -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-invoices" style="margin: 0; flex: 1;">Statistiques Commandes</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 30px;">
            <div class="metric-card">
                <div class="metric-label">Créées Aujourd'hui</div>
                <div class="metric-value" style="color: #2196F3;"><?php echo $invoices_created_today; ?></div>
                <p style="color: #666; font-size: 12px; margin-top: 8px;"><?php echo number_format($invoices_created_amount, 0, '.', ','); ?> BIF</p>
            </div>
            <div class="metric-card">
                <div class="metric-label">En Attente d'Approbation</div>
                <div class="metric-value" style="color: #FF9800;"><?php echo $invoiceStatusMetrics['pending']['count']; ?></div>
                <p style="color: #666; font-size: 12px; margin-top: 8px;"><?php echo number_format($invoiceStatusMetrics['pending']['amount'], 0, '.', ','); ?> BIF</p>
            </div>
            <div class="metric-card">
                <div class="metric-label">Approuvées Aujourd'hui</div>
                <div class="metric-value" style="color: #4CAF50;"><?php echo $invoices_approved_today; ?></div>
                <p style="color: #666; font-size: 12px; margin-top: 8px;"><?php echo number_format($invoices_approved_amount, 0, '.', ','); ?> BIF</p>
            </div>
            <div class="metric-card">
                <div class="metric-label">Payées Aujourd'hui</div>
                <div class="metric-value" style="color: #357A38;"><?php echo $invoices_paid_today; ?></div>
                <p style="color: #666; font-size: 12px; margin-top: 8px;"><?php echo number_format($invoices_paid_amount, 0, '.', ','); ?> BIF</p>
            </div>
            <div class="metric-card">
                <div class="metric-label">Expirées Aujourd'hui</div>
                <div class="metric-value" style="color: #D32F2F;"><?php echo $expired_invoices_today; ?></div>
                <p style="color: #666; font-size: 12px; margin-top: 8px;"><?php echo number_format($expired_invoices_amount, 0, '.', ','); ?> BIF</p>
            </div>
        </div>

        <h4 class="subsection-title">Commandes par Dépôt Aujourd'hui</h4>
        <div class="card">
            <table>
                <thead>
                    <tr>
                        <th>Dépôt</th>
                        <th>Créées</th>
                        <th>En Attente</th>
                        <th>Approuvées</th>
                        <th>Montant En Attente</th>
                        <th>Montant Approuvé</th>
                        <th>Payé</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($invoices_by_province as $row): ?>
                    <tr>
                        <td style="font-weight: 500;"><?php echo htmlspecialchars($row['province_name']); ?></td>
                        <td><?php echo $row['created_today'] ?? 0; ?></td>
                        <td><?php echo $row['pending_approval'] ?? 0; ?></td>
                        <td><?php echo $row['approved'] ?? 0; ?></td>
                        <td><?php echo number_format($row['pending_amount'] ?? 0, 0, '.', ','); ?> BIF</td>
                        <td><?php echo number_format($row['approved_amount'] ?? 0, 0, '.', ','); ?> BIF</td>
                        <td><?php echo number_format($row['paid_today'] ?? 0, 0, '.', ','); ?> BIF</td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- SECTION 6: STOCK TRANSFERS TODAY -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-stock-transfers" style="margin: 0; flex: 1;">Transferts de Stock reçu Aujourd'hui par Dépôt</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div class="card">
            <table>
                <thead>
                    <tr>
                        <th>Destination</th>
                        <th>Transferts</th>
                        <th>Quantité</th>
                        <th>Statut</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($stock_transfers_today as $row): ?>
                    <tr>
                        <td style="font-weight: 500;"><?php echo htmlspecialchars($row['province_name']); ?></td>
                        <td><?php echo $row['transfer_count'] ?? 0; ?></td>
                        <td><?php echo number_format($row['total_quantity'] ?? 0, 0, '.', ','); ?></td>
                        <td><?php echo htmlspecialchars($row['status_summary'] ?? 'Aucun'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- SECTION 6: STOCK PRODUCTS OVERVIEW -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-stock-overview" style="margin: 0; flex: 1;">Vue d'Ensemble Stock</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        
        <!-- Main Stock Overview -->
        <h4 class="subsection-title">Vue d'Ensemble Stock Principal (Central - Non Réparti)</h4>
        <div style="margin-bottom: 30px;">
            <div class="card">
                <table>
                    <thead>
                        <tr>
                            <th>Produit</th>
                            <th>Caisses</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($main_stock_products as $row): ?>
                        <?php
                            $total_qty = (int)($row['total_quantity'] ?? 0);
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                            <td><?php echo number_format($total_qty, 0, ',', '.'); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Dépôts Stock Overview -->
        <h4 class="subsection-title">Vue d'Ensemble Stock Dépôts (Stock Réparti Détail + Gros)</h4>
        <div style="margin-bottom: 30px;">
            <?php foreach ($stock_by_province as $dépôt): ?>
                <div class="card" style="margin-bottom: 20px; background-color: #FFFFFF; padding: 20px; border-radius: 8px;">
                    <h5 style="color: #4B2F1F; font-size: 18px; margin: 0 0 15px 0; border-bottom: 2px solid #F4A261; padding-bottom: 10px;">
                        <?php echo htmlspecialchars($dépôt['province_name']); ?>
                    </h5>
                    
                    <table style="width: 100%; border-collapse: collapse;">
                        <thead>
                            <tr style="background-color: #F4F0E4;">
                                <th style="padding: 12px; text-align: left; width: 70%; color: #4B2F1F; font-weight: 600;">Produit</th>
                                <th style="padding: 12px; text-align: center; color: #FF9800; font-weight: 600; background-color: #FFF3E0; width: 30%;">Caisses totales</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                                $province_total_crates = 0;
                            ?>
                            <?php foreach ($dépôt['products'] as $product): ?>
                                <?php
                                    $crate_qty = max(1, (int)($product['crate_quantity'] ?? 12));
                                    $product_units = ($product['wholesale_qty'] ?? 0) + ($product['retail_qty'] ?? 0);
                                    $product_crates = $product_units / $crate_qty;
                                    $province_total_crates += $product_crates;
                                ?>
                                <tr style="border-bottom: 1px solid #E0E0E0;">
                                    <td style="padding: 12px; text-align: left; color: #4B2F1F;"><?php echo htmlspecialchars($product['product_name']); ?></td>
                                    <td style="padding: 12px; text-align: center; color: #FF9800; background-color: #FFF3E0;"><?php echo number_format($product_crates, 2, ',', '.'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                            <tr style="background-color: #F4A26120; font-weight: 600;">
                                    <td style="padding: 12px; text-align: left; color: #4B2F1F;">Total Dépôt</td>
                                <td style="padding: 12px; text-align: center; color: #FF9800; background-color: #FFF3E0;"><?php echo number_format($province_total_crates, 2, ',', '.'); ?></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- SECTION 7: OPERATIONAL SNAPSHOT -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-operational" style="margin: 0; flex: 1;">Aperçu Opérationnel</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 15px; margin-bottom: 30px;">
            <div class="card">
                <p style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 8px;">Transferts d'Espèces</p>
                <p style="color: #4B2F1F; margin: 0;">En Attente: <strong><?php echo number_format($transferMetrics['pending']['count'], 0, ',', '.'); ?></strong> transfert(s)</p>
                <p style="color: #357A38; margin: 6px 0 0;">Reçu: <strong><?php echo number_format($transferMetrics['approved']['count'], 0, ',', '.'); ?></strong> transfert(s)</p>
            </div>
            <div class="card">
                <p style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 8px;">Commandes</p>
                <p style="color: #4B2F1F; margin: 0;">En Attente: <strong><?php echo number_format($invoiceStatusMetrics['pending']['count'], 0, ',', '.'); ?></strong></p>
                <p style="color: #357A38; margin: 6px 0 0;">Approuvées: <strong><?php echo number_format($invoiceStatusMetrics['approved']['count'], 0, ',', '.'); ?></strong></p>
            </div>
            <div class="card">
                <p style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 8px;">Remboursements de Crédits (Aujourd'hui)</p>
                <p style="color: #4B2F1F; margin: 0;">Nombre: <strong><?php echo number_format($loan_repayments_today_count, 0, ',', '.'); ?></strong></p>
                <p style="color: #357A38; margin: 6px 0 0;">Montant: <strong><?php echo number_format($loan_repayments_today_amount, 0, '.', ','); ?> BIF</strong></p>
            </div>
            <div class="card">
                <p style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 8px;">Réquisitions Payées (Aujourd'hui)</p>
                <p style="color: #4B2F1F; margin: 0;">Nombre: <strong><?php echo number_format($requisitions_paid_count, 0, ',', '.'); ?></strong></p>
                <p style="color: #357A38; margin: 6px 0 0;">Montant: <strong><?php echo number_format($requisitions_paid_amount, 0, '.', ','); ?> BIF</strong></p>
            </div>
        </div>

        <!-- SECTION 8: CASHIERS BALANCES BY DÉPÔT -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-cashier-balances" style="margin: 0; flex: 1;">Soldes Caissiers par Dépôt</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div class="card">
            <table>
                <thead>
                    <tr>
                        <th>Dépôt</th>
                        <th>Caissier</th>
                        <th>Solde</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($cashiers_balances as $row): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                        <td><?php echo htmlspecialchars($row['cashier_name'] ?? 'N/A'); ?></td>
                        <td><?php echo number_format((int)round($row['balance'] ?? 0), 0, ',', '.'); ?> BIF</td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- SECTION 9: REQUISITIONS OVERVIEW -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-requisitions" style="margin: 0; flex: 1;">Vue d'Ensemble Réquisitions</h3>
            <div style="display: flex; gap: 10px;">
                <a href="/masunzu_bar_hotel/modules/expenses/expenses_history.php" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; text-decoration: none; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                    Voir Historique →
                </a>
                <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                    ↻ Actualiser
                </button>
            </div>
        </div>
        
        <!-- Requisition Status Summary Cards -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
            <div class="metric-card" style="background: linear-gradient(135deg, #FF9800 0%, #FFB74D 100%);">
                <div class="metric-label" style="color: #FFFFFF; opacity: 0.9;">En Attente</div>
                <div class="metric-value" style="color: #FFFFFF;"><?php echo number_format($requisition_all_statuses['pending']['count'], 0, '.', ','); ?></div>
                <div style="color: #FFFFFF; font-size: 12px; margin-top: 5px; opacity: 0.9;"><?php echo number_format($requisition_all_statuses['pending']['amount'], 0, '.', ','); ?> BIF</div>
            </div>
            <div class="metric-card" style="background: linear-gradient(135deg, #2196F3 0%, #64B5F6 100%);">
                <div class="metric-label" style="color: #FFFFFF; opacity: 0.9;">Approuvées</div>
                <div class="metric-value" style="color: #FFFFFF;"><?php echo number_format($requisition_all_statuses['approved']['count'], 0, '.', ','); ?></div>
                <div style="color: #FFFFFF; font-size: 12px; margin-top: 5px; opacity: 0.9;"><?php echo number_format($requisition_all_statuses['approved']['amount'], 0, '.', ','); ?> BIF</div>
            </div>
            <div class="metric-card" style="background: linear-gradient(135deg, #4CAF50 0%, #81C784 100%);">
                <div class="metric-label" style="color: #FFFFFF; opacity: 0.9;">Payées</div>
                <div class="metric-value" style="color: #FFFFFF;"><?php echo number_format($requisition_all_statuses['paid']['count'], 0, '.', ','); ?></div>
                <div style="color: #FFFFFF; font-size: 12px; margin-top: 5px; opacity: 0.9;"><?php echo number_format($requisition_all_statuses['paid']['amount'], 0, '.', ','); ?> BIF</div>
            </div>
            <div class="metric-card" style="background: linear-gradient(135deg, #F44336 0%, #E57373 100%);">
                <div class="metric-label" style="color: #FFFFFF; opacity: 0.9;">Rejetées</div>
                <div class="metric-value" style="color: #FFFFFF;"><?php echo number_format($requisition_all_statuses['rejected']['count'], 0, '.', ','); ?></div>
                <div style="color: #FFFFFF; font-size: 12px; margin-top: 5px; opacity: 0.9;"><?php echo number_format($requisition_all_statuses['rejected']['amount'], 0, '.', ','); ?> BIF</div>
            </div>
        </div>
        
        <!-- Requisition Time Period Summary -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; margin-bottom: 20px;">
            <div class="card">
                <p style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 8px;">Cette Semaine</p>
                <p style="color: #4B2F1F; margin: 0;">Nombre: <strong><?php echo number_format($requisitions_this_week['count'], 0, ',', '.'); ?></strong></p>
                <p style="color: #357A38; margin: 6px 0 0;">Montant: <strong><?php echo number_format($requisitions_this_week['amount'], 0, '.', ','); ?> BIF</strong></p>
            </div>
            <div class="card">
                <p style="color: #4B2F1F; font-size: 14px; font-weight: 600; margin-bottom: 8px;">Ce Mois</p>
                <p style="color: #4B2F1F; margin: 0;">Nombre: <strong><?php echo number_format($requisitions_this_month['count'], 0, ',', '.'); ?></strong></p>
                <p style="color: #357A38; margin: 6px 0 0;">Montant: <strong><?php echo number_format($requisitions_this_month['amount'], 0, '.', ','); ?> BIF</strong></p>
            </div>
        </div>
        
        <!-- Requisitions by Dépôt (Today) -->
        <div class="card" style="margin-bottom: 20px;">
        <h4 class="subsection-title">Réquisitions par Dépôt (Aujourd'hui)</h4>
            <table>
                <thead>
                    <tr>
                        <th>Dépôt</th>
                        <th>Nombre</th>
                        <th>Montant Total</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($requisitions_by_province)): ?>
                        <?php foreach ($requisitions_by_province as $row): ?>
                        <tr>
                            <td style="font-weight: 500;"><?php echo htmlspecialchars($row['province_name']); ?></td>
                            <td><?php echo $row['requisition_count'] ?? 0; ?></td>
                            <td><?php echo number_format($row['total_amount'] ?? 0, 0, '.', ','); ?> BIF</td>
                        </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="3" style="text-align: center; color: #666; padding: 20px;">Aucune réquisition aujourd'hui</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Recent Requisition Payments -->
        <?php if (!empty($recent_requisition_payments)): ?>
        <div class="card" style="margin-bottom: 20px;">
            <h4 class="subsection-title">Paiements de Réquisitions Récents</h4>
            <table>
                <thead>
                    <tr>
                        <th>ID Réquisition</th>
                        <th>Initiateur</th>
                        <th>Dépôt</th>
                        <th>Montant</th>
                        <th>Caissier</th>
                        <th>Reçu</th>
                        <th>Date Paiement</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recent_requisition_payments as $payment): ?>
                    <tr>
                        <td>#<?php echo htmlspecialchars($payment['id']); ?></td>
                        <td><?php echo htmlspecialchars($payment['initiator_name'] ?? 'N/A'); ?></td>
                        <td><?php echo htmlspecialchars($payment['province_name'] ?? 'N/A'); ?></td>
                        <td><?php echo number_format($payment['total_amount'] ?? 0, 0, '.', ','); ?> BIF</td>
                        <td><?php echo htmlspecialchars($payment['cashier_name'] ?? 'N/A'); ?></td>
                        <td><?php echo htmlspecialchars($payment['receipt_number'] ?? 'N/A'); ?></td>
                        <td><?php echo $payment['payment_date'] ? date('Y-m-d H:i', strtotime($payment['payment_date'])) : 'N/A'; ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
        
        <!-- Top Requisition Categories removed -->

        <!-- SECTION 10: ACTIVE USERS removed -->

        <!-- SECTION 11: ACTIVE CASHIERS -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-active-cashiers" style="margin: 0; flex: 1;">Sessions Caissiers Actives</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div class="card">
            <?php if (!empty($active_cashiers)): ?>
            <table>
                <thead>
                    <tr>
                        <th>Caissier</th>
                        <th>Dépôt</th>
                        <th>Session Débutée</th>
                        <th>Temps en Session</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($active_cashiers as $cashier): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($cashier['cashier_name']); ?></td>
                        <td><?php echo htmlspecialchars($cashier['province_name'] ?? 'N/A'); ?></td>
                        <td><?php echo date('H:i:s', strtotime($cashier['shift_start'])); ?></td>
                        <td><?php 
                            $hours = intdiv($cashier['minutes_in_shift'], 60);
                            $mins = $cashier['minutes_in_shift'] % 60;
                            if ($hours > 0) {
                                echo $hours . 'h ' . $mins . 'm';
                            } else {
                                echo $mins . 'm';
                            }
                        ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php else: ?>
            <p style="color: #666; text-align: center; padding: 20px;">Aucune session caissier active en ce moment</p>
            <?php endif; ?>
        </div>

        <!-- SECTION 12: TOP PRODUCTS & CASHIERS -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
            <h3 class="section-title" id="section-top-performers" style="margin: 0; flex: 1;">Top Performeurs Aujourd'hui</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-bottom: 30px;">
            <!-- Top Products by Sales -->
            <div class="card">
                <h4 class="subsection-title">Top 10 Produits par Ventes</h4>
                <table>
                    <thead>
                        <tr>
                            <th>Produit</th>
                            <th>Quantité</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($top_products_sales as $row): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                            <td><?php echo number_format($row['total_quantity'], 0, '.', ','); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Top Products by Amount -->
            <div class="card">
                <h4 class="subsection-title">Top 10 Produits par Montant</h4>
                <table>
                    <thead>
                        <tr>
                            <th>Produit</th>
                            <th>Montant</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($top_products_amount as $row): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                            <td><?php echo number_format($row['total_amount'], 0, '.', ','); ?> BIF</td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Top 3 Cashiers -->
            <div class="card">
                <h4 class="subsection-title">Top 3 Caissiers Aujourd'hui</h4>
                <table>
                    <thead>
                        <tr>
                            <th>Caissier</th>
                            <th>Dépôt</th>
                            <th>Montant</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($top_cashiers_today as $row): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['cashier_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                            <td><?php echo number_format($row['total_amount'], 0, '.', ','); ?> BIF</td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Pending Invoices -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 20px; margin-bottom: 15px;">
            <h3 style="color: #4B2F1F; margin: 0; flex: 1;">Commandes en Attente</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <?php if (empty($invoices)): ?>
            <p style="color: #000000; font-size: 16px; text-align: center;">Aucune commande en attente.</p>
        <?php else: ?>
            <table style="width: 100%; border-collapse: collapse; margin-top: 10px; margin-bottom: 20px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">ID</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Initiateur</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Dépôt</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Montant Total (BIF)</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Statut</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($invoices as $inv): ?>
                        <tr style="background-color: #FFFFFF;">
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($inv['id']); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($inv['initiator_name']); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($inv['province_name']); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;">
                                <?php echo number_format($inv['total_amount'], 2, '.', ','); ?> BIF
                            </td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($inv['approval_status']); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;">
                                <a href="/masunzu_bar_hotel/modules/admin/approve_invoices.php?invoice_id=<?php echo $inv['id']; ?>" style="color: #4CAF50; text-decoration: none;">Voir Détails</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <a href="/masunzu_bar_hotel/modules/admin/approve_invoices.php" style="display: inline-block; background-color: #4CAF50; color: #FFFFFF; padding: 10px 20px; border-radius: 5px; text-decoration: none; margin-top: 10px;">Approuver/Rejeter les Commandes</a>
        <?php endif; ?>
        
        <!-- Low Stock Alerts removed (moved to dedicated page) -->
        
        <!-- Accountant Balances -->
        <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 30px; margin-bottom: 15px;">
            <h3 id="section-accountant-balances" style="color: #4B2F1F; margin: 0; flex: 1;">Soldes Comptables</h3>
            <button onclick="location.reload();" style="background-color: #4B2F1F; color: #F4F0E4; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-weight: 600; font-size: 12px; transition: background-color 0.3s; white-space: nowrap;">
                ↻ Actualiser
            </button>
        </div>
        <?php if (!empty($accountant_balances)): ?>
            <div style="overflow-x: auto; margin-bottom: 20px;">
                <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden;">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 12px; text-align: left;">Comptable</th>
                            <th style="padding: 12px; text-align: left;">Dépôt</th>
                            <th style="padding: 12px; text-align: right;">Total Reçu</th>
                            <th style="padding: 12px; text-align: right;">Total Dépenses</th>
                            <th style="padding: 12px; text-align: right;">Solde Disponible</th>
                            <th style="padding: 12px; text-align: left;">Dernière Mise à Jour</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($accountant_balances as $acct): ?>
                            <?php
                                $availableBalance = max(0, (float)($acct['available_balance'] ?? 0));
                                $totalReceived = (float)($acct['balance'] ?? 0);
                                $totalExpenses = (float)($acct['total_expenses'] ?? 0);
                            ?>
                            <tr style="border-bottom: 1px solid #F4F0E4;">
                                <td style="padding: 12px; color: #4B2F1F;"><?php echo htmlspecialchars($acct['accountant_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; color: #4B2F1F;"><?php echo htmlspecialchars($acct['province_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; text-align: right; color: #4B2F1F; font-weight: 600;"><?php echo number_format($totalReceived, 0, ',', '.'); ?> BIF</td>
                                <td style="padding: 12px; text-align: right; color: #DC3545;"><?php echo number_format($totalExpenses, 0, ',', '.'); ?> BIF</td>
                                <td style="padding: 12px; text-align: right; color: <?php echo $availableBalance > 0 ? '#28a745' : '#4B2F1F'; ?>; font-weight: 600;"><?php echo number_format($availableBalance, 0, ',', '.'); ?> BIF</td>
                                <td style="padding: 12px; color: #4B2F1F; font-size: 12px;"><?php echo $acct['last_updated'] ? date('Y-m-d H:i', strtotime($acct['last_updated'])) : 'Jamais'; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="color: #000000; font-size: 16px; text-align: center;">Aucun comptable trouvé.</p>
        <?php endif; ?>
            </div>
            <!-- END MAIN CONTENT -->
        </div>
        <!-- END DASHBOARD CONTAINER -->
    </div>
</div>
<?php include $base_dir . '/includes/footer.php'; ?>
