<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/functions.php';

function accountant_table_exists(mysqli $conn, string $table): bool {
    static $cache = [];
    if (array_key_exists($table, $cache)) {
        return $cache[$table];
    }
    $safe = $conn->real_escape_string($table);
    $query = "SHOW TABLES LIKE '{$safe}'";
    $result = $conn->query($query);
    if ($result === false) {
        error_log('accountant_table_exists failed: ' . $conn->error);
        return $cache[$table] = false;
    }
    $exists = $result->num_rows > 0;
    $result->close();
    return $cache[$table] = $exists;
}

try {
    error_log("DEBUG: Entering accountant_dashboard.php");
    error_log("DEBUG: role_name=" . ($_SESSION['role_name'] ?? 'unset'));
    error_log("DEBUG: user_id=" . ($_SESSION['user_id'] ?? 'unset'));

    require_login();

    // Restrict to Accountant role
    if ($_SESSION['role_name'] !== 'Accountant') {
        error_log("Unauthorized access to accountant_dashboard.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
        // Use dashboard mapping to redirect correctly
        $dashboard_map = [
            'Board' => 'admin_dashboard.php',
            'DG' => 'admin_dashboard.php',
            'DGA' => 'admin_dashboard.php',
            'Admin' => 'admin_dashboard.php',
            'Procurement Manager' => 'procurement_dashboard.php',
            'Operation Manager' => 'operation_dashboard.php',
            'Finance Head' => 'finance_dashboard.php',
            'Marketing Head' => 'marketing_dashboard.php',
            'Stock Manager' => 'stock_manager_dashboard.php',
            'Marketing Agent' => 'marketing_dashboard.php',
            'Cashier' => 'cashier_dashboard.php',
            'Caissier_Comptable' => 'cashier_dashboard.php',
            'Stock Manager Assistant' => 'stock_manager_assistant_dashboard.php'
        ];
        $dashboard = $dashboard_map[$_SESSION['role_name']] ?? 'unauthorized.php';
        header("Location: /masunzu_bar_hotel/dashboards/" . $dashboard);
        exit;
    }

    // Fetch province_id if not set (Accountant may not have province_id)
    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception("Database connection failed: " . mysqli_connect_error());
    }

    if (!isset($_SESSION['province_id'])) {
        $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if (!$stmt) {
            throw new Exception("Query preparation failed for province_id: " . $conn->error);
        }
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        $_SESSION['province_id'] = $user['province_id'] ?? null;
        $stmt->close();
    }
    $province_id = $_SESSION['province_id'];

    // Fetch dépôt name (if province_id exists)
    $province = 'Tous les Dépôts';
    if ($province_id) {
        $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
        if (!$stmt) {
            throw new Exception("Query preparation failed for province name: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $province_result = $stmt->get_result()->fetch_assoc();
        $province = $province_result['name'] ?? 'Inconnu';
        $stmt->close();
    }
    $province_scope_label = $province_id ? 'dépôt ' . $province : 'toutes les provinces';
    $global_scope_label = 'toutes les provinces';

    $todayStart = date('Y-m-d 00:00:00');
    $todayEnd = date('Y-m-d 23:59:59');

    // Totaux du jour (toutes provinces/utilisateurs)
    $stmt = $conn->prepare("SELECT IFNULL(SUM(amount),0) AS total_received FROM cash_transfer WHERE status = 'received' AND received_at BETWEEN ? AND ?");
    if (!$stmt) {
        throw new Exception("Query preparation failed for total_received today: " . $conn->error);
    }
    $stmt->bind_param("ss", $todayStart, $todayEnd);
    $stmt->execute();
    $total_received = $stmt->get_result()->fetch_assoc()['total_received'] ?? 0.00;
    $stmt->close();

    $total_deposits = 0.0;
    if (accountant_table_exists($conn, 'bank_deposit_new')) {
        $stmt = $conn->prepare("SELECT IFNULL(SUM(amount),0) AS total_deposits FROM bank_deposit_new WHERE deposit_date BETWEEN ? AND ?");
        if ($stmt) {
            $stmt->bind_param("ss", $todayStart, $todayEnd);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $total_deposits += (float)($row['total_deposits'] ?? 0);
            $stmt->close();
        }
    }
    if (accountant_table_exists($conn, 'bank_deposit')) {
        $stmt = $conn->prepare("SELECT IFNULL(SUM(amount),0) AS total_deposits FROM bank_deposit WHERE deposit_date BETWEEN ? AND ?");
        if ($stmt) {
            $stmt->bind_param("ss", $todayStart, $todayEnd);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $total_deposits += (float)($row['total_deposits'] ?? 0);
            $stmt->close();
        }
    }

    $stmt = $conn->prepare("
        SELECT IFNULL(SUM(amount),0) AS total_expenses 
        FROM expense 
        WHERE payment_status IN ('processed','completed') 
          AND created_at BETWEEN ? AND ?
    ");
    if (!$stmt) {
        throw new Exception("Query preparation failed for total_expenses today: " . $conn->error);
    }
    $stmt->bind_param("ss", $todayStart, $todayEnd);
    $stmt->execute();
    $total_expenses = $stmt->get_result()->fetch_assoc()['total_expenses'] ?? 0.00;
    $stmt->close();

    // Ventes encaissées aujourd'hui (factures payées)
    $sales_today = 0.0;
    if ($province_id) {
        $stmt = $conn->prepare("SELECT IFNULL(SUM(paid_amount), 0) AS total FROM paid_invoices WHERE province_id = ? AND created_at BETWEEN ? AND ?");
        if ($stmt) {
            $stmt->bind_param("iss", $province_id, $todayStart, $todayEnd);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $sales_today = (float)($row['total'] ?? 0);
            $stmt->close();
        }
    } else {
        $stmt = $conn->prepare("SELECT IFNULL(SUM(paid_amount), 0) AS total FROM paid_invoices WHERE created_at BETWEEN ? AND ?");
        if ($stmt) {
            $stmt->bind_param("ss", $todayStart, $todayEnd);
            $stmt->execute();
            $row = $stmt->get_result()->fetch_assoc();
            $sales_today = (float)($row['total'] ?? 0);
            $stmt->close();
        }
    }

    $main_bif_balance = 0.0;
    $cashier_cash_positions = [];
    $cashier_cash_total = 0.0;
    $cashier_cash_by_province = [];
    $cashier_cash_province_summaries = [];

    // Mirror admin dashboard logic for cash held by cashiers
    $has_yesno_columns = false;
    $checkColumns = $conn->query("
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = DATABASE() 
          AND TABLE_NAME = 'cashier_shift_closure' 
          AND COLUMN_NAME = 'deposited_to_bank'
    ");
    if ($checkColumns && $checkColumns->num_rows > 0) {
        $has_yesno_columns = true;
        $checkColumns->close();
    }

    $cashierListQuery = "
        SELECT u.id, u.full_name AS cashier_name, COALESCE(p.name, 'N/A') AS province_name
        FROM user u
        INNER JOIN role r ON u.role_id = r.id AND r.name = 'Caissier_Comptable'
        LEFT JOIN province p ON u.province_id = p.id
        WHERE COALESCE(u.is_active, 1) = 1
    ";
    if ($province_id) {
        $cashierListQuery .= " AND u.province_id = ?";
    }
    $cashierListQuery .= " ORDER BY province_name, cashier_name";

    $cashiers = [];
    $cashierListStmt = $conn->prepare($cashierListQuery);
    if ($cashierListStmt) {
        if ($province_id) {
            $cashierListStmt->bind_param("i", $province_id);
        }
        $cashierListStmt->execute();
        $cashiers = $cashierListStmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $cashierListStmt->close();
    }

    foreach ($cashiers as $cashier) {
        $cashierId = (int)($cashier['id'] ?? 0);
        $available_from_closed_shifts = 0.0;
        $closed_pending_shifts = 0;

        // Closed shifts
        if ($has_yesno_columns) {
            $stmtClosed = $conn->prepare("
                SELECT 
                    cb.id,
                    cb.shift_start,
                    cb.shift_end,
                    cb.beginning_balance,
                    cb.end_balance,
                    cb.deposits,
                    cb.withdrawals,
                    COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
                    COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
                    csc.counted_amount
                FROM cashier_balance cb
                LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
                WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
                ORDER BY cb.shift_end DESC
            ");
        } else {
            $stmtClosed = $conn->prepare("
                SELECT 
                    cb.id,
                    cb.shift_start,
                    cb.shift_end,
                    cb.beginning_balance,
                    cb.end_balance,
                    cb.deposits,
                    cb.withdrawals
                FROM cashier_balance cb
                WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
                ORDER BY cb.shift_end DESC
            ");
        }

        if ($stmtClosed) {
            $stmtClosed->bind_param("i", $cashierId);
            $stmtClosed->execute();
            $closedShifts = $stmtClosed->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmtClosed->close();

            foreach ($closedShifts as $shift) {
                $shiftId = (int)($shift['id'] ?? 0);
                $end_balance = null;
                if (isset($shift['end_balance']) && $shift['end_balance'] !== null) {
                    $end_balance = (float)$shift['end_balance'];
                } else {
                    $beginning = (float)($shift['beginning_balance'] ?? 0);
                    $deposits = (float)($shift['deposits'] ?? 0);
                    $withdrawals = (float)($shift['withdrawals'] ?? 0);
                    $end_balance = $beginning + $deposits - $withdrawals;
                }

                $is_deposited = false;
                $is_transferred = false;

                if ($has_yesno_columns) {
                    $deposited_flag = $shift['deposited_to_bank'] ?? 'no';
                    $transferred_flag = $shift['transferred_to_accountant'] ?? 'no';
                    $is_deposited = strtolower($deposited_flag) === 'yes';
                    $is_transferred = strtolower($transferred_flag) === 'yes';
                } else {
                    $stmtDeposit = $conn->prepare("
                        SELECT COUNT(*) AS has_deposit
                        FROM cashier_shift_deposit csd
                        INNER JOIN bank_deposit bd ON csd.deposit_id = bd.id
                        WHERE csd.shift_id = ?
                    ");
                    if ($stmtDeposit) {
                        $stmtDeposit->bind_param("i", $shiftId);
                        $stmtDeposit->execute();
                        $depositResult = $stmtDeposit->get_result()->fetch_assoc();
                        $is_deposited = (int)($depositResult['has_deposit'] ?? 0) > 0;
                        $stmtDeposit->close();
                    }

                    $stmtTransfer = $conn->prepare("
                        SELECT COUNT(*) AS has_transfer
                        FROM cashier_shift_transfer cst
                        INNER JOIN cash_transfer ct ON cst.transfer_id = ct.id
                        WHERE cst.shift_id = ? AND ct.status = 'received'
                    ");
                    if ($stmtTransfer) {
                        $stmtTransfer->bind_param("i", $shiftId);
                        $stmtTransfer->execute();
                        $transferResult = $stmtTransfer->get_result()->fetch_assoc();
                        $is_transferred = (int)($transferResult['has_transfer'] ?? 0) > 0;
                        $stmtTransfer->close();
                    }
                }

                $available_amount = $end_balance;
                if ($has_yesno_columns && isset($shift['counted_amount']) && $shift['counted_amount'] !== null) {
                    $available_amount = (float)$shift['counted_amount'];
                }

                if ($available_amount > 0 && !$is_deposited && !$is_transferred) {
                    $available_from_closed_shifts += $available_amount;
                    $closed_pending_shifts++;
                }
            }
        }

        $active_shifts = 0;
        $active_amount = 0.0;
        $stmtActive = $conn->prepare("SELECT id, shift_start, beginning_balance, deposits FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
        if ($stmtActive) {
            $stmtActive->bind_param("i", $cashierId);
            $stmtActive->execute();
            $activeShift = $stmtActive->get_result()->fetch_assoc();
            $stmtActive->close();

            if ($activeShift && !empty($activeShift['shift_start'])) {
                $active_shifts = 1;
                $shift_start = $activeShift['shift_start'];
                $beginning_balance = (float)($activeShift['beginning_balance'] ?? 0);
                $shift_deposits = (float)($activeShift['deposits'] ?? 0);

                $shift_expenses = 0.0;
                $stmtExp = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
                if ($stmtExp) {
                    $stmtExp->bind_param("is", $cashierId, $shift_start);
                    $stmtExp->execute();
                    $shift_expenses = (float)($stmtExp->get_result()->fetch_assoc()['total_expenses'] ?? 0);
                    $stmtExp->close();
                }

                $shift_transfers = 0.0;
                $stmtTrans = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transfers FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
                if ($stmtTrans) {
                    $stmtTrans->bind_param("is", $cashierId, $shift_start);
                    $stmtTrans->execute();
                    $shift_transfers = (float)($stmtTrans->get_result()->fetch_assoc()['total_transfers'] ?? 0);
                    $stmtTrans->close();
                }

                $active_amount = $beginning_balance + $shift_deposits - $shift_expenses - $shift_transfers;
                if ($active_amount < 0) {
                    $active_amount = 0;
                }
            }
        }

        $total_outstanding = $available_from_closed_shifts + $active_amount;
        if ($total_outstanding <= 0) {
            continue;
        }

        $position = [
            'cashier_id' => $cashierId,
            'cashier_name' => $cashier['cashier_name'] ?? 'N/A',
            'province_name' => $cashier['province_name'] ?? 'Sans province',
            'active_amount' => $active_amount,
            'active_shifts' => $active_shifts,
            'closed_pending_amount' => $available_from_closed_shifts,
            'closed_pending_shifts' => $closed_pending_shifts,
            'pending_transfers' => 0.0,
            'total_outstanding' => $total_outstanding
        ];
        $cashier_cash_positions[] = $position;
        $cashier_cash_total += $total_outstanding;
        $provinceKey = $position['province_name'];
        if (!isset($cashier_cash_by_province[$provinceKey])) {
            $cashier_cash_by_province[$provinceKey] = [
                'province_name' => $provinceKey,
                'total' => 0.0,
                'cashiers' => 0
            ];
        }
        $cashier_cash_by_province[$provinceKey]['total'] += $total_outstanding;
        $cashier_cash_by_province[$provinceKey]['cashiers']++;
    }

    if (!empty($cashier_cash_by_province)) {
        $cashier_cash_province_summaries = array_values($cashier_cash_by_province);
        usort($cashier_cash_province_summaries, static function ($a, $b) {
            return $b['total'] <=> $a['total'];
        });
    }

    $pending_count = 0;
    $pending_total = 0.0;
    $approved_count = 0;
    $approved_total = 0.0;
    $paid_count = 0;
    $paid_total = 0.0;
    $recent_requisitions = [];

    // Fetch requisitions - if accountant has province_id, filter by it; otherwise show all
    if ($province_id) {
        $stmt = $conn->prepare("SELECT COUNT(*) AS pending_count, IFNULL(SUM(total_amount),0) AS pending_total FROM requisition WHERE province_id = ? AND status = 'pending'");
        if (!$stmt) {
            throw new Exception("Query preparation failed for pending requisitions: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $pending_row = $stmt->get_result()->fetch_assoc();
        $pending_count = (int)($pending_row['pending_count'] ?? 0);
        $pending_total = (float)($pending_row['pending_total'] ?? 0);
        $stmt->close();

        $stmt = $conn->prepare("SELECT COUNT(*) AS approved_count, IFNULL(SUM(total_amount),0) AS approved_total FROM requisition WHERE province_id = ? AND status = 'approved'");
        if (!$stmt) {
            throw new Exception("Query preparation failed for approved requisitions: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $approved_row = $stmt->get_result()->fetch_assoc();
        $approved_count = (int)($approved_row['approved_count'] ?? 0);
        $approved_total = (float)($approved_row['approved_total'] ?? 0);
        $stmt->close();

        $stmt = $conn->prepare("SELECT COUNT(*) AS paid_count, IFNULL(SUM(total_amount),0) AS paid_total FROM requisition WHERE province_id = ? AND status = 'paid'");
        if (!$stmt) {
            throw new Exception("Query preparation failed for paid requisitions: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $paid_row = $stmt->get_result()->fetch_assoc();
        $paid_count = (int)($paid_row['paid_count'] ?? 0);
        $paid_total = (float)($paid_row['paid_total'] ?? 0);
        $stmt->close();
    } else {
        // Accountant without province - show all requisitions
        $stmt = $conn->prepare("SELECT COUNT(*) AS pending_count, IFNULL(SUM(total_amount),0) AS pending_total FROM requisition WHERE status = 'pending'");
        if (!$stmt) {
            throw new Exception("Query preparation failed for pending requisitions: " . $conn->error);
        }
        $stmt->execute();
        $pending_row = $stmt->get_result()->fetch_assoc();
        $pending_count = (int)($pending_row['pending_count'] ?? 0);
        $pending_total = (float)($pending_row['pending_total'] ?? 0);
        $stmt->close();

        $stmt = $conn->prepare("SELECT COUNT(*) AS approved_count, IFNULL(SUM(total_amount),0) AS approved_total FROM requisition WHERE status = 'approved'");
        if (!$stmt) {
            throw new Exception("Query preparation failed for approved requisitions: " . $conn->error);
        }
        $stmt->execute();
        $approved_row = $stmt->get_result()->fetch_assoc();
        $approved_count = (int)($approved_row['approved_count'] ?? 0);
        $approved_total = (float)($approved_row['approved_total'] ?? 0);
        $stmt->close();

        $stmt = $conn->prepare("SELECT COUNT(*) AS paid_count, IFNULL(SUM(total_amount),0) AS paid_total FROM requisition WHERE status = 'paid'");
        if (!$stmt) {
            throw new Exception("Query preparation failed for paid requisitions: " . $conn->error);
        }
        $stmt->execute();
        $paid_row = $stmt->get_result()->fetch_assoc();
        $paid_count = (int)($paid_row['paid_count'] ?? 0);
        $paid_total = (float)($paid_row['paid_total'] ?? 0);
        $stmt->close();
    }

    $total_requisitions = $pending_count + $approved_count + $paid_count;
    $outstanding_total = $pending_total + $approved_total;

    $pending_cash_transfers = 0;
    $pending_cash_total = 0.0;
    $stmt = $conn->prepare("
        SELECT COUNT(*) AS transfer_count, IFNULL(SUM(amount), 0) AS total_amount
        FROM cash_transfer
        WHERE status = 'pending'
    ");
    if (!$stmt) {
        throw new Exception("Query preparation failed for pending cash transfers: " . $conn->error);
    }
    $stmt->execute();
    $pending_cash_row = $stmt->get_result()->fetch_assoc();
    $pending_cash_transfers = (int)($pending_cash_row['transfer_count'] ?? 0);
    $pending_cash_total = (float)($pending_cash_row['total_amount'] ?? 0);
    $stmt->close();

    $stmt = $conn->prepare("
        SELECT 'Received' as transaction_type, amount, transfer_date as created_at, u.full_name as user_name
        FROM cash_transfer ct
        LEFT JOIN user u ON ct.cashier_id = u.id
        WHERE ct.accountant_id = ? AND ct.status = 'received'
        UNION
        SELECT 'Deposit' as transaction_type, amount, deposit_date as created_at, 'Self' as user_name
        FROM bank_deposit bd
        WHERE bd.user_id = ?
        UNION
        SELECT 'Expense' as transaction_type, amount, created_at, 'Self' as user_name
        FROM expense e
        WHERE e.accountant_id = ?
        ORDER BY created_at DESC
        LIMIT 5
    ");
    if (!$stmt) {
        throw new Exception("Query preparation failed for recent transactions: " . $conn->error);
    }
    $stmt->bind_param("iii", $_SESSION['user_id'], $_SESSION['user_id'], $_SESSION['user_id']);
    $stmt->execute();
    $recent_transactions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Enhanced recent requisitions query with payment information
    if ($province_id) {
        $stmt = $conn->prepare("
            SELECT r.id, r.total_amount, r.status, r.created_at,
                   u.full_name AS initiator_name, recipient.full_name AS recipient_name,
                   e.receipt_number, e.created_at AS payment_date, cashier.full_name AS cashier_name
            FROM requisition r
            LEFT JOIN user u ON r.user_id = u.id
            LEFT JOIN user recipient ON r.recipient_id = recipient.id
            LEFT JOIN expense e ON e.requisition_id = r.id
            LEFT JOIN user cashier ON e.cashier_id = cashier.id
            WHERE r.province_id = ?
            ORDER BY r.created_at DESC
            LIMIT 6
        ");
        if (!$stmt) {
            throw new Exception("Query preparation failed for recent requisitions: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $recent_requisitions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
        
        // Requisitions this week
        $week_start = date('Y-m-d', strtotime('monday this week'));
        $requisitions_this_week = ['count' => 0, 'amount' => 0];
        $stmt = $conn->prepare("SELECT COUNT(*) AS cnt, IFNULL(SUM(total_amount), 0) AS total FROM requisition WHERE province_id = ? AND DATE(created_at) >= ?");
        if ($stmt) {
            $stmt->bind_param("is", $province_id, $week_start);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $requisitions_this_week['count'] = (int)($result['cnt'] ?? 0);
            $requisitions_this_week['amount'] = (float)($result['total'] ?? 0);
            $stmt->close();
        }
        
        // Requisitions this month
        $month_start = date('Y-m-01');
        $requisitions_this_month = ['count' => 0, 'amount' => 0];
        $stmt = $conn->prepare("SELECT COUNT(*) AS cnt, IFNULL(SUM(total_amount), 0) AS total FROM requisition WHERE province_id = ? AND DATE(created_at) >= ?");
        if ($stmt) {
            $stmt->bind_param("is", $province_id, $month_start);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $requisitions_this_month['count'] = (int)($result['cnt'] ?? 0);
            $requisitions_this_month['amount'] = (float)($result['total'] ?? 0);
            $stmt->close();
        }
        
        // Requisitions paid today (using expense.created_at for payment date)
        $requisitions_paid_today_count = 0;
        $requisitions_paid_today_amount = 0;
        $today = date('Y-m-d');
        $stmt = $conn->prepare("
            SELECT COUNT(DISTINCT r.id) AS cnt, IFNULL(SUM(r.total_amount), 0) AS total
            FROM requisition r
            INNER JOIN expense e ON e.requisition_id = r.id
            WHERE r.province_id = ? AND r.status = 'paid' AND DATE(e.created_at) = ?
        ");
        if ($stmt) {
            $stmt->bind_param("is", $province_id, $today);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $requisitions_paid_today_count = (int)($result['cnt'] ?? 0);
            $requisitions_paid_today_amount = (float)($result['total'] ?? 0);
            $stmt->close();
        }
    } else {
        // Accountant without province - show all requisitions
        $stmt = $conn->prepare("
            SELECT r.id, r.total_amount, r.status, r.created_at,
                   u.full_name AS initiator_name, recipient.full_name AS recipient_name,
                   e.receipt_number, e.created_at AS payment_date, cashier.full_name AS cashier_name
            FROM requisition r
            LEFT JOIN user u ON r.user_id = u.id
            LEFT JOIN user recipient ON r.recipient_id = recipient.id
            LEFT JOIN expense e ON e.requisition_id = r.id
            LEFT JOIN user cashier ON e.cashier_id = cashier.id
            ORDER BY r.created_at DESC
            LIMIT 6
        ");
        if (!$stmt) {
            throw new Exception("Query preparation failed for recent requisitions: " . $conn->error);
        }
        $stmt->execute();
        $recent_requisitions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
        
        // Requisitions this week
        $week_start = date('Y-m-d', strtotime('monday this week'));
        $requisitions_this_week = ['count' => 0, 'amount' => 0];
        $stmt = $conn->prepare("SELECT COUNT(*) AS cnt, IFNULL(SUM(total_amount), 0) AS total FROM requisition WHERE DATE(created_at) >= ?");
        if ($stmt) {
            $stmt->bind_param("s", $week_start);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $requisitions_this_week['count'] = (int)($result['cnt'] ?? 0);
            $requisitions_this_week['amount'] = (float)($result['total'] ?? 0);
            $stmt->close();
        }
        
        // Requisitions this month
        $month_start = date('Y-m-01');
        $requisitions_this_month = ['count' => 0, 'amount' => 0];
        $stmt = $conn->prepare("SELECT COUNT(*) AS cnt, IFNULL(SUM(total_amount), 0) AS total FROM requisition WHERE DATE(created_at) >= ?");
        if ($stmt) {
            $stmt->bind_param("s", $month_start);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $requisitions_this_month['count'] = (int)($result['cnt'] ?? 0);
            $requisitions_this_month['amount'] = (float)($result['total'] ?? 0);
            $stmt->close();
        }
        
        // Requisitions paid today (using expense.created_at for payment date)
        $requisitions_paid_today_count = 0;
        $requisitions_paid_today_amount = 0;
        $today = date('Y-m-d');
        $stmt = $conn->prepare("
            SELECT COUNT(DISTINCT r.id) AS cnt, IFNULL(SUM(r.total_amount), 0) AS total
            FROM requisition r
            INNER JOIN expense e ON e.requisition_id = r.id
            WHERE r.status = 'paid' AND DATE(e.created_at) = ?
        ");
        if ($stmt) {
            $stmt->bind_param("s", $today);
            $stmt->execute();
            $result = $stmt->get_result()->fetch_assoc();
            $requisitions_paid_today_count = (int)($result['cnt'] ?? 0);
            $requisitions_paid_today_amount = (float)($result['total'] ?? 0);
            $stmt->close();
        }
    }

    $conn->close();

    $approval_rate = $total_requisitions > 0 ? (($approved_count + $paid_count) / $total_requisitions) * 100 : 0;
    $pending_ratio = $total_requisitions > 0 ? ($pending_count / $total_requisitions) * 100 : 0;
    $cash_utilization = $total_received > 0 ? min(($total_expenses / $total_received) * 100, 100) : 0;
    $outstanding_total = $pending_total + $approved_total;
    $last_refresh = date('Y-m-d H:i');

} catch (Exception $e) {
    error_log("ERROR in accountant_dashboard.php: " . $e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur : " . htmlspecialchars($e->getMessage()) . "</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}

include __DIR__ . '/../includes/header.php';
?>
<style>
    .accountant-dashboard { padding: 32px 36px 64px; background: #f5f7fb; min-height: calc(100vh - 70px); }
    .dashboard-container { width: 100%; display: flex; flex-direction: column; gap: 28px; }
    .page-header { display: flex; flex-wrap: wrap; justify-content: space-between; gap: 16px; }
    .page-header h2 { margin: 0; font-size: 30px; color: #1f2a37; }
    .page-header p { margin: 4px 0 0; color: #5c6576; font-size: 15px; }
    .chip-row { display: flex; flex-wrap: wrap; gap: 12px; }
    .chip { padding: 8px 16px; border-radius: 999px; background: #e3e7ef; color: #1f2a37; font-weight: 600; font-size: 13px; text-transform: uppercase; letter-spacing: 0.08em; }
    .kpi-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 12px; }
    .kpi-card { background: #ffffff; border-radius: 12px; padding: 14px 16px; border: 1px solid #e1e4ec; box-shadow: 0 8px 18px rgba(15,23,42,0.08); display: flex; flex-direction: column; gap: 4px; }
    .kpi-card strong { font-size: 22px; color: #1f2937; }
    .kpi-card span { font-size: 11px; text-transform: uppercase; letter-spacing: 0.06em; color: #6c7687; }
    .kpi-card small { color: #7b8295; font-size: 12px; line-height: 1.4; }
    .callout { padding: 18px 24px; border-radius: 14px; display: flex; align-items: center; gap: 16px; border: 1px solid; background: #fff; box-shadow: 0 12px 24px rgba(15,23,42,0.06); }
    .callout.pending { border-color: #f8d6a3; background: #fffaf3; }
    .callout.clean { border-color: #c6f6d5; background: #f0fdf4; }
    .callout-icon { width: 44px; height: 44px; border-radius: 12px; display: flex; align-items: center; justify-content: center; font-size: 22px; }
    .callout.pending .callout-icon { background: #f59f00; color: #fff; }
    .callout.clean .callout-icon { background: #34a853; color: #fff; }
    .callout h4 { margin: 0; font-size: 16px; color: #1f2937; }
    .callout p { margin: 2px 0 0; color: #4b5563; font-size: 14px; }
    .status-layout { display: grid; grid-template-columns: repeat(auto-fit, minmax(320px, 1fr)); gap: 20px; }
    .panel { background: #ffffff; border-radius: 14px; border: 1px solid #e1e4ec; padding: 22px; box-shadow: 0 10px 24px rgba(15,23,42,0.08); display: flex; flex-direction: column; gap: 18px; }
    .panel h3 { margin: 0; font-size: 18px; color: #1f2937; }
    .status-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 16px; }
    .status-card { border: 1px solid #edf0f7; border-radius: 10px; padding: 16px; display: flex; flex-direction: column; gap: 4px; background: #f9fafc; }
    .status-card strong { font-size: 24px; color: #111827; }
    .status-card span { color: #6c7687; font-size: 13px; }
    .insight-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 14px; }
    .insight-card { border: 1px solid #edf0f7; border-radius: 10px; padding: 14px; background: #f9fafc; display: flex; flex-direction: column; gap: 4px; }
    .insight-card strong { font-size: 18px; color: #111827; }
    .progress { width: 100%; height: 10px; border-radius: 999px; background: #e5e7eb; overflow: hidden; margin-top: 6px; }
    .progress-bar { height: 100%; background: linear-gradient(135deg, #2563eb 0%, #06b6d4 100%); }
    .section-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(380px, 1fr)); gap: 24px; }
    .table-panel table { width: 100%; border-collapse: collapse; }
    .table-panel th, .table-panel td { padding: 12px; border-bottom: 1px solid #edf0f7; font-size: 14px; text-align: left; }
    .table-panel thead { background: #111827; color: #f9fafc; }
    .badge { padding: 4px 10px; border-radius: 999px; font-size: 12px; font-weight: 600; text-transform: capitalize; }
    .badge.pending { background: #fff1e6; color: #b45309; }
    .badge.approved { background: #ecfdf5; color: #047857; }
    .badge.paid { background: #e0f2fe; color: #0369a1; }
    .empty { text-align: center; color: #6b7280; padding: 24px 0; }
    .panel-note { margin: 0 0 8px; color: #6b7280; font-size: 14px; }
    .province-summary { display: flex; flex-wrap: wrap; gap: 10px; margin-bottom: 14px; }
    .province-chip { background: #eef2ff; color: #3730a3; border-radius: 12px; padding: 8px 14px; display: flex; flex-direction: column; min-width: 160px; }
    .province-chip strong { font-size: 16px; color: #1e1b4b; }
    .province-chip span { font-size: 12px; color: #4c1d95; }
    .cashier-badge { display: block; color: #6b7280; font-size: 12px; margin-top: 4px; }
    .cashier-amount { font-weight: 600; color: #111827; font-size: 15px; }
    @media (max-width: 768px) {
        .accountant-dashboard { padding: 20px 18px 48px; }
    }
</style>
<div class="accountant-dashboard">
    <div class="dashboard-container">
        <div class="page-header">
            <div>
                <h2>Tableau de Bord Comptable</h2>
                <p>Bienvenue, <?php echo htmlspecialchars($_SESSION['full_name'], ENT_QUOTES, 'UTF-8'); ?> • Dépôt : <?php echo htmlspecialchars($province, ENT_QUOTES, 'UTF-8'); ?></p>
            </div>
        </div>
        <div class="chip-row">
            <span class="chip">Rôle : <?php echo htmlspecialchars($_SESSION['role_name'], ENT_QUOTES, 'UTF-8'); ?></span>
            <span class="chip">Mise à jour <?php echo htmlspecialchars($last_refresh, ENT_QUOTES, 'UTF-8'); ?></span>
        </div>
        <div class="kpi-grid">
            <div class="kpi-card">
                <span>Ventes encaissées (aujourd'hui · <?php echo htmlspecialchars($province_scope_label, ENT_QUOTES, 'UTF-8'); ?>)</span>
                <strong><?php echo number_format($sales_today, 0, '.', ','); ?> BIF</strong>
                <small>Factures payées aujourd'hui sur ce périmètre</small>
            </div>
            <div class="kpi-card">
                <span>Espèces reçues (aujourd'hui · <?php echo htmlspecialchars($global_scope_label, ENT_QUOTES, 'UTF-8'); ?>)</span>
                <strong><?php echo number_format($total_received, 0, '.', ','); ?> BIF</strong>
                <small>Transferts confirmés ce jour</small>
            </div>
            <div class="kpi-card">
                <span>Dépôts bancaires (aujourd'hui · <?php echo htmlspecialchars($global_scope_label, ENT_QUOTES, 'UTF-8'); ?>)</span>
                <strong><?php echo number_format($total_deposits, 0, '.', ','); ?> BIF</strong>
                <small>Dépôts bancaires saisis aujourd'hui</small>
            </div>
            <div class="kpi-card">
                <span>Dépenses payées (aujourd'hui · <?php echo htmlspecialchars($global_scope_label, ENT_QUOTES, 'UTF-8'); ?>)</span>
                <strong><?php echo number_format($total_expenses, 0, '.', ','); ?> BIF</strong>
                <small>Réquisitions payées aujourd'hui (tous dépôts)</small>
            </div>
            <div class="kpi-card">
                <span>Réquisitions en suspens (<?php echo htmlspecialchars($province_scope_label, ENT_QUOTES, 'UTF-8'); ?>)</span>
                <strong><?php echo number_format($outstanding_total, 0, '.', ','); ?> BIF</strong>
                <small><?php echo number_format($pending_count + $approved_count, 0, '.', ','); ?> dossiers</small>
            </div>
            <div class="kpi-card">
                <span>Transferts en attente (<?php echo htmlspecialchars($global_scope_label, ENT_QUOTES, 'UTF-8'); ?>)</span>
                <strong><?php echo number_format($pending_cash_total, 0, '.', ','); ?> BIF</strong>
                <small><?php echo number_format($pending_cash_transfers, 0, '.', ','); ?> transferts à confirmer</small>
            </div>
            <div class="kpi-card">
                <span>Espèces chez caissiers (<?php echo htmlspecialchars($province_scope_label, ENT_QUOTES, 'UTF-8'); ?>)</span>
                <strong><?php echo number_format($cashier_cash_total, 0, '.', ','); ?> BIF</strong>
                <small><?php echo number_format(count($cashier_cash_positions), 0, '.', ','); ?> caissiers concernés</small>
            </div>
        </div>
        <div class="callout <?php echo $pending_cash_transfers > 0 ? 'pending' : 'clean'; ?>">
            <div class="callout-icon"><?php echo $pending_cash_transfers > 0 ? '⚠️' : '✅'; ?></div>
            <div>
                <h4><?php echo $pending_cash_transfers > 0 ? 'Transferts de cash en attente' : 'Transferts à jour'; ?></h4>
                <p>
                    <?php if ($pending_cash_transfers > 0): ?>
                        <?php echo number_format($pending_cash_transfers, 0, '.', ','); ?> transferts à valider représentant <?php echo number_format($pending_cash_total, 0, '.', ','); ?> BIF.
                    <?php else: ?>
                        Tous les transferts ont été confirmés. Dernière vérification <?php echo htmlspecialchars($last_refresh, ENT_QUOTES, 'UTF-8'); ?>.
                    <?php endif; ?>
                </p>
            </div>
        </div>
        <div class="status-layout">
            <div class="panel">
                <h3>Statut des réquisitions</h3>
                <div class="status-grid">
                    <div class="status-card">
                        <span>En attente</span>
                        <strong><?php echo number_format($pending_count, 0, '.', ','); ?></strong>
                        <span><?php echo number_format($pending_total, 0, '.', ','); ?> BIF</span>
                    </div>
                    <div class="status-card">
                        <span>Approuvées</span>
                        <strong><?php echo number_format($approved_count, 0, '.', ','); ?></strong>
                        <span><?php echo number_format($approved_total, 0, '.', ','); ?> BIF</span>
                    </div>
                    <div class="status-card">
                        <span>Payées</span>
                        <strong><?php echo number_format($paid_count, 0, '.', ','); ?></strong>
                        <span><?php echo number_format($paid_total, 0, '.', ','); ?> BIF</span>
                    </div>
                </div>
            </div>
            <div class="panel">
                <h3>Indicateurs rapides</h3>
                <div class="insight-grid">
                    <div class="insight-card">
                        <span>Cette semaine</span>
                        <strong><?php echo number_format($requisitions_this_week['count'] ?? 0, 0, '.', ','); ?></strong>
                        <small><?php echo number_format($requisitions_this_week['amount'] ?? 0, 0, '.', ','); ?> BIF</small>
                    </div>
                    <div class="insight-card">
                        <span>Ce mois</span>
                        <strong><?php echo number_format($requisitions_this_month['count'] ?? 0, 0, '.', ','); ?></strong>
                        <small><?php echo number_format($requisitions_this_month['amount'] ?? 0, 0, '.', ','); ?> BIF</small>
                    </div>
                    <div class="insight-card">
                        <span>Payées aujourd'hui</span>
                        <strong><?php echo number_format($requisitions_paid_today_count, 0, '.', ','); ?></strong>
                        <small><?php echo number_format($requisitions_paid_today_amount, 0, '.', ','); ?> BIF</small>
                    </div>
                </div>
                <div>
                    <div style="display:flex; justify-content:space-between; font-size:13px; color:#6c7687;">
                        <span>Taux d'approbation</span>
                        <span><?php echo number_format($approval_rate, 1, '.', ','); ?>%</span>
                    </div>
                    <div class="progress">
                        <div class="progress-bar" style="width: <?php echo min(100, max(0, $approval_rate)); ?>%;"></div>
                    </div>
                </div>
                <div>
                    <div style="display:flex; justify-content:space-between; font-size:13px; color:#6c7687;">
                        <span>Réquisitions en attente</span>
                        <span><?php echo number_format($pending_ratio, 1, '.', ','); ?>%</span>
                    </div>
                    <div class="progress">
                        <div class="progress-bar" style="width: <?php echo min(100, max(0, $pending_ratio)); ?>%; background: linear-gradient(135deg,#f97316,#fb7185);"></div>
                    </div>
                </div>
                <div>
                    <div style="display:flex; justify-content:space-between; font-size:13px; color:#6c7687;">
                        <span>Utilisation des espèces reçues</span>
                        <span><?php echo number_format($cash_utilization, 1, '.', ','); ?>%</span>
                    </div>
                    <div class="progress">
                        <div class="progress-bar" style="width: <?php echo min(100, max(0, $cash_utilization)); ?>%; background: linear-gradient(135deg,#16a34a,#4ade80);"></div>
                    </div>
                </div>
            </div>
        </div>
        <div class="section-grid">
            <div class="panel table-panel">
                <h3>Réquisitions récentes</h3>
                <?php if (empty($recent_requisitions)): ?>
                    <div class="empty">Aucune réquisition récente.</div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Initiateur</th>
                                    <th>Destinataire</th>
                                    <th>Montant</th>
                                    <th>Statut</th>
                                    <th>Paiement</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recent_requisitions as $requisition): ?>
                                    <tr>
                                        <td>#<?php echo htmlspecialchars($requisition['id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td><?php echo htmlspecialchars($requisition['initiator_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td><?php echo htmlspecialchars($requisition['recipient_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td><?php echo number_format((float)($requisition['total_amount'] ?? 0), 0, '.', ','); ?> BIF<br><small><?php echo htmlspecialchars(date('Y-m-d', strtotime($requisition['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></small></td>
                                        <td>
                                            <?php $status = strtolower($requisition['status'] ?? 'pending'); ?>
                                            <span class="badge <?php echo htmlspecialchars($status, ENT_QUOTES, 'UTF-8'); ?>">
                                                <?php echo htmlspecialchars($requisition['status'] ?? 'pending', ENT_QUOTES, 'UTF-8'); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if (!empty($requisition['receipt_number'])): ?>
                                                <div>Reçu <?php echo htmlspecialchars($requisition['receipt_number'], ENT_QUOTES, 'UTF-8'); ?></div>
                                                <small><?php echo htmlspecialchars($requisition['cashier_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></small>
                                            <?php else: ?>
                                                <div class="badge pending">En attente</div>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
            <div class="panel table-panel">
                <h3>Transactions récentes</h3>
                <?php if (empty($recent_transactions)): ?>
                    <div class="empty">Aucune transaction récente.</div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Type</th>
                                    <th>Utilisateur</th>
                                    <th>Montant</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recent_transactions as $transaction): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($transaction['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td>
                                            <?php
                                                $type_map = [
                                                    'Received' => 'Transfert reçu',
                                                    'Deposit' => 'Dépôt bancaire',
                                                    'Expense' => 'Dépense'
                                                ];
                                                $type = $transaction['transaction_type'] ?? 'Unknown';
                                                echo htmlspecialchars($type_map[$type] ?? $type, ENT_QUOTES, 'UTF-8');
                                            ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($transaction['user_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td><?php echo number_format((float)($transaction['amount'] ?? 0), 2, ',', '.'); ?> BIF</td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
            <div class="panel table-panel">
                <h3>Encaisse caissiers & provinces</h3>
                <p class="panel-note">Total à sécuriser : <?php echo number_format($cashier_cash_total, 0, '.', ','); ?> BIF</p>
                <?php if (!empty($cashier_cash_province_summaries)): ?>
                    <div class="province-summary">
                        <?php foreach ($cashier_cash_province_summaries as $summary): ?>
                            <div class="province-chip">
                                <strong><?php echo htmlspecialchars($summary['province_name'], ENT_QUOTES, 'UTF-8'); ?></strong>
                                <span><?php echo number_format($summary['total'], 0, '.', ','); ?> BIF</span>
                                <span><?php echo number_format($summary['cashiers'], 0, '.', ','); ?> caissier(s)</span>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
                <?php if (empty($cashier_cash_positions)): ?>
                    <div class="empty">Aucun montant en attente chez les caissiers.</div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>Caissier</th>
                                    <th>Dépôt</th>
                                    <th>Sessions actives</th>
                                    <th>Sessions clôturées non transférées</th>
                                    <th>Total à remettre</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($cashier_cash_positions as $position): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($position['cashier_name'], ENT_QUOTES, 'UTF-8'); ?></strong>
                                        </td>
                                        <td><?php echo htmlspecialchars($position['province_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td>
                                            <div class="cashier-amount"><?php echo number_format($position['active_amount'], 0, '.', ','); ?> BIF</div>
                                            <span class="cashier-badge"><?php echo number_format($position['active_shifts'], 0, '.', ','); ?> session(s)</span>
                                        </td>
                                        <td>
                                            <div class="cashier-amount"><?php echo number_format($position['closed_pending_amount'], 0, '.', ','); ?> BIF</div>
                                            <span class="cashier-badge"><?php echo number_format($position['closed_pending_shifts'], 0, '.', ','); ?> session(s)</span>
                                        </td>
                                        <td>
                                            <div class="cashier-amount"><?php echo number_format($position['total_outstanding'], 0, '.', ','); ?> BIF</div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php include __DIR__ . '/../includes/footer.php'; ?>
