<?php
/**
 * ==========================================================
 * Masunzu Bar Hotel - Database Connection Handler
 * ----------------------------------------------------------
 * Multi-attempt MySQLi connection with socket fallback
 * Wrapped in function_exists() to prevent redeclaration errors.
 * ==========================================================
 */

// Load environment variables from .env file if it exists
if (file_exists(__DIR__ . '/../.env')) {
    $envFile = file(__DIR__ . '/../.env', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($envFile as $line) {
        if (strpos(trim($line), '#') === 0) continue; // Skip comments
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim($value);
            if (!empty($key) && !isset($_ENV[$key])) {
                $_ENV[$key] = $value;
            }
        }
    }
}

// Database configuration with environment variable support
// Priority: Environment variable > .env file > default value
if (!defined('DB_HOST')) {
    define('DB_HOST', getenv('DB_HOST') ?: $_ENV['DB_HOST'] ?? 'localhost:3306');
}
if (!defined('DB_USER')) {
    define('DB_USER', getenv('DB_USER') ?: $_ENV['DB_USER'] ?? 'root');
}
if (!defined('DB_PASS')) {
    define('DB_PASS', getenv('DB_PASSWORD') ?: $_ENV['DB_PASSWORD'] ?? '@Mysql-x45#');
}
if (!defined('DB_NAME')) {
    define('DB_NAME', getenv('DB_NAME') ?: $_ENV['DB_NAME'] ?? 'masunzu_bar_hotel');
}
if (!defined('DB_SOCKET')) {
    define('DB_SOCKET', getenv('DB_SOCKET') ?: $_ENV['DB_SOCKET'] ?? '/opt/lampp/var/mysql/mysql.sock');
}

if (!function_exists('ensure_bank_deposit_auto_increment')) {
    /**
     * Hotfix to guarantee bank_deposit.id is AUTO_INCREMENT.
     * Prevents "Field 'id' doesn't have a default value" crashes.
     */
    function ensure_bank_deposit_auto_increment(mysqli $conn): void
    {
        static $checked = false;
        if ($checked) {
            return;
        }

        $dbName = defined('DB_NAME') ? DB_NAME : null;
        if (!$dbName) {
            return;
        }

        $tableResult = $conn->query("SHOW TABLES LIKE 'bank_deposit'");
        if (!$tableResult || $tableResult->num_rows === 0) {
            if ($tableResult) {
                $tableResult->free();
            }
            $checked = true;
            return;
        }
        $tableResult->free();

        $hasPk = false;
        $hasAuto = false;
        $stmt = $conn->prepare("
            SELECT COLUMN_KEY, EXTRA 
            FROM information_schema.COLUMNS 
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'bank_deposit' AND COLUMN_NAME = 'id'
            LIMIT 1
        ");
        if ($stmt) {
            $stmt->bind_param("s", $dbName);
            $stmt->execute();
            $colInfo = $stmt->get_result()->fetch_assoc() ?: [];
            $stmt->close();

            $hasPk = ($colInfo['COLUMN_KEY'] ?? '') === 'PRI';
            $hasAuto = stripos($colInfo['EXTRA'] ?? '', 'auto_increment') !== false;
            if ($hasPk && $hasAuto) {
                $checked = true;
                return;
            }
        }

        $conn->query("SET FOREIGN_KEY_CHECKS = 0");
        if (empty($hasPk)) {
            $conn->query("ALTER TABLE bank_deposit ADD PRIMARY KEY (id)");
        }
        $conn->query("ALTER TABLE bank_deposit MODIFY id INT(11) NOT NULL AUTO_INCREMENT");
        $conn->query("SET FOREIGN_KEY_CHECKS = 1");
        $checked = true;
    }
}

if (!function_exists('get_db_connection')) {

    /**
     * Create and return a MySQLi database connection.
     * Attempt order:
     * 1. Normal connection using password
     * 2. Normal connection with empty password (for XAMPP fallback)
     * 3. Socket connection
     *
     * @return mysqli|false
     */
    function get_db_connection()
    {
        // Attempt #1 — normal login
        $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);

        if ($conn->connect_error) {
            error_log("DB Connect failed (with password): " . $conn->connect_error);

            // Attempt #2 — try empty password (XAMPP default)
            $conn = new mysqli(DB_HOST, DB_USER, '', DB_NAME);

            if ($conn->connect_error) {
                error_log("DB Connect failed (empty password): " . $conn->connect_error);

                // Attempt #3 — socket connection
                $conn = new mysqli('localhost', DB_USER, DB_PASS, DB_NAME, null, DB_SOCKET);

                if ($conn->connect_error) {
                    error_log("DB Connect failed (socket): " . $conn->connect_error);
                    return false;
                }
            }
        }

        $conn->set_charset('utf8mb4');
        $conn->query("SET sql_mode=''");
        ensure_bank_deposit_auto_increment($conn);
        return $conn;
    }
}
?>
