<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// ---------- ACCESS CONTROL ----------
$allowed_roles = ['DG', 'Board', 'DGA', 'Admin'];
$currentUserRole = $_SESSION['role_name'] ?? '';
if (!in_array($_SESSION['role_name'], $allowed_roles)) {
    header("Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php");
    exit;
}

// ---------- DATABASE ----------
$conn = get_db_connection();
$hiddenRoles = ['Stock Manager Assistant'];
$adminOnlyRoles = ['Admin', 'DG', 'DGA'];

// Fetch roles excluding hidden ones
$escapedHiddenRoles = array_map([$conn, 'real_escape_string'], $hiddenRoles);
$hiddenRoleList = "'" . implode("','", $escapedHiddenRoles) . "'";
$roles_query = "SELECT id, name FROM role WHERE name NOT IN ($hiddenRoleList) ORDER BY name";
$roles_result = $conn->query($roles_query);
$roles = $roles_result ? $roles_result->fetch_all(MYSQLI_ASSOC) : [];

// Fetch provinces
$provinces_result = $conn->query("SELECT id, name FROM province ORDER BY name");
$provinces = $provinces_result ? $provinces_result->fetch_all(MYSQLI_ASSOC) : [];

$regions_result = $conn->query("SELECT id, name FROM region ORDER BY name");
$regions = $regions_result ? $regions_result->fetch_all(MYSQLI_ASSOC) : [];

$departments_result = $conn->query("SELECT id, code, label FROM department ORDER BY label");
$departments = $departments_result ? $departments_result->fetch_all(MYSQLI_ASSOC) : [];
$departmentCodesAvailable = $departments ? array_column($departments, 'code') : [];

$conn->close();

// ---------- FORM PROCESSING ----------
$error = '';
$success = '';
$selected_role_id = 0;
$input = ['department_codes' => []];

if (isset($_SESSION['user_success'])) {
    $success = $_SESSION['user_success'];
    unset($_SESSION['user_success']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $provinceValue = $_POST['province_id'] ?? '';
    $regionValue = $_POST['region_id'] ?? '';
    $departmentInput = array_map('strval', (array)($_POST['department_codes'] ?? []));
    $input = [
        'username'    => trim($_POST['username'] ?? ''),
        'full_name'   => trim($_POST['full_name'] ?? ''),
        'email'       => trim($_POST['email'] ?? ''),
        'mobile1'     => trim($_POST['mobile1'] ?? ''),
        'mobile2'     => trim($_POST['mobile2'] ?? ''),
        'address'     => trim($_POST['address'] ?? ''),
        'role_id'     => (int)($_POST['role_id'] ?? 0),
        'province_id' => $provinceValue === '' ? null : (int)$provinceValue,
        'region_id'   => $regionValue === '' ? null : (int)$regionValue,
        'password'    => $_POST['password'] ?? '',
        'is_active'   => isset($_POST['is_active']) ? 1 : 0,
        'department_codes' => array_values(array_unique($departmentInput)),
    ];

    $selected_role_id = $input['role_id'];

    // Get role name
    $role_name = '';
    foreach ($roles as $r) {
        if ($r['id'] == $input['role_id']) {
            $role_name = $r['name'];
            break;
        }
    }

    $normalizedRole = mbh_normalize_role_name($role_name);
    $autoDepartments = mbh_get_role_departments($normalizedRole);
    $hasAutoDepartments = !empty($autoDepartments);
    if ($hasAutoDepartments) {
        $input['department_codes'] = $autoDepartments;
    }

    $requires_province = mbh_role_requires_depot($role_name);
    $requires_region = mbh_role_requires_region($role_name);

    // ---------- VALIDATION ----------
    $required_fields = ['username', 'full_name', 'email', 'role_id', 'password'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            $error = "Tous les champs obligatoires doivent être remplis.";
            break;
        }
    }

    if (!$error && $role_name === '') {
        $error = "Rôle sélectionné invalide.";
    }

    if (!$error && in_array($role_name, $hiddenRoles, true)) {
        $error = "Le rôle sélectionné n'est plus disponible.";
    }

    if (
        !$error
        && in_array($role_name, $adminOnlyRoles, true)
        && $currentUserRole !== 'Admin'
    ) {
        $error = "Seul un utilisateur avec le profil Admin peut créer des comptes $role_name.";
    }

    if (!$error && $input['province_id'] !== null) {
        $provinceIds = array_map('intval', array_column($provinces, 'id'));
        if (!in_array($input['province_id'], $provinceIds, true)) {
            $error = "Dépôt sélectionné invalide.";
        }
    }

    if (!$requires_province) {
        $input['province_id'] = null;
    }
    if (!$requires_region) {
        $input['region_id'] = null;
    }

    if (!$error && $requires_province && $input['province_id'] === null) {
        $error = "La province est obligatoire pour le rôle : <strong>$role_name</strong>.";
    }

    if (!$error && $input['region_id'] !== null) {
        $regionIds = array_map('intval', array_column($regions, 'id'));
        if (!in_array($input['region_id'], $regionIds, true)) {
            $error = "Région sélectionnée invalide.";
        }
    }

    if (!$error && $requires_region && $input['region_id'] === null) {
        $error = "La région est obligatoire pour le rôle : <strong>$role_name</strong>.";
    }

    // Only validate manually selected departments against database
    // Auto-assigned departments are trusted (they come from role metadata)
    if (!$error && !empty($input['department_codes']) && !$hasAutoDepartments) {
        foreach ($input['department_codes'] as $code) {
            if (!in_array($code, $departmentCodesAvailable, true)) {
                $error = "Département sélectionné invalide.";
                break;
            }
        }
    }

    $departmentOptionalRoles = ['DG', 'DGA', 'Admin', 'Board'];
    if (
        !$error
        && empty($input['department_codes'])
        && !in_array($normalizedRole, $departmentOptionalRoles, true)
    ) {
        $error = "Veuillez sélectionner au moins un département.";
    }

    if (!$error && !filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
        $error = "Adresse email invalide.";
    }

    if (!$error && strlen($input['password']) < 6) {
        $error = "Le mot de passe doit contenir au moins 6 caractères.";
    }

    // ---------- CHECK UNIQUENESS ----------
    if (!$error) {
        $conn = get_db_connection();
        $stmt = $conn->prepare("SELECT id FROM user WHERE username = ? OR email = ?");
        $stmt->bind_param("ss", $input['username'], $input['email']);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $error = "Nom d'utilisateur ou email déjà existant.";
        }
        $stmt->close();
    }

    // ---------- INSERT USER ----------
    if (!$error) {
        $hashed_password = password_hash($input['password'], PASSWORD_DEFAULT);
        $provinceParam = $input['province_id'] ?? -1;
        $regionParam = $input['region_id'] ?? -1;

        $sql = "INSERT INTO user 
                (username, full_name, email, mobile1, mobile2, position, role_id, province_id, region_id, address, password, is_active) 
                VALUES (?, ?, ?, ?, ?, ?, ?, NULLIF(?, -1), NULLIF(?, -1), ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param(
            "ssssssiiissi",
            $input['username'],
            $input['full_name'],
            $input['email'],
            $input['mobile1'],
            $input['mobile2'],
            $role_name,
            $input['role_id'],
            $provinceParam,
            $regionParam,
            $input['address'],
            $hashed_password,
            $input['is_active']
        );

        if ($stmt->execute()) {
            $new_user_id = $conn->insert_id;
            $province_name = $input['province_id'] !== null ? get_province_name($input['province_id']) : 'N/A';
            $region_name = $input['region_id'] !== null ? get_region_name($input['region_id']) : 'N/A';
            mbh_sync_user_departments($conn, $new_user_id, $input['department_codes']);
            log_action(
                $_SESSION['user_id'],
                'add_user',
                "Added user '{$input['username']}' (ID: $new_user_id, Role: $role_name, Depot: $province_name, Region: $region_name, Mobile1: {$input['mobile1']}, Mobile2: {$input['mobile2']}, Address: {$input['address']})"
            );
            record_user_change($new_user_id, 'create', (int)$_SESSION['user_id'], [
                'username' => $input['username'],
                'full_name' => $input['full_name'],
                'email' => $input['email'],
                'mobile1' => $input['mobile1'],
                'mobile2' => $input['mobile2'],
                'role' => $role_name,
                'province' => $province_name,
                'region' => $region_name,
                'address' => $input['address'],
                'departments' => implode(', ', $input['department_codes'] ?? []),
                'is_active' => $input['is_active'] ? 'Oui' : 'Non',
            ]);
            $_SESSION['user_success'] = "Utilisateur '{$input['username']}' créé avec succès.";
            $conn->close();
            header("Location: /masunzu_bar_hotel/modules/users/add_user.php");
            exit;
        } else {
            $error = "La création de l'utilisateur a échoué. Veuillez réessayer.";
        }
        $stmt->close();
        $conn->close();
    }
}

// ---------- HELPER: GET PROVINCE NAME ----------
function get_province_name($id) {
    static $cache = [];
    if (isset($cache[$id])) return $cache[$id];

    $conn = get_db_connection();
    $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $name = $row ? $row['name'] : 'Unknown';
    $cache[$id] = $name;

    $stmt->close();
    $conn->close();
    return $name;
}

function get_region_name($id) {
    static $cache = [];
    if (isset($cache[$id])) {
        return $cache[$id];
    }

    $conn = get_db_connection();
    $stmt = $conn->prepare("SELECT name FROM region WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $name = $row ? $row['name'] : 'Unknown';
    $cache[$id] = $name;

    $stmt->close();
    $conn->close();
    return $name;
}

include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>

<div class="form-container" style="padding: 20px; background-color: #F4F0E4; min-height: 70vh;">
    <h1 style="color: #4B2F1F; font-size: 28px; font-weight: bold; margin-bottom: 20px;">Ajouter un nouvel utilisateur</h1>

    <div class="form-content" style="background-color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.1); max-width: 500px; margin: 0 auto;">

        <?php if ($error): ?>
            <div style="background: #ffe6e6; color: red; padding: 12px; border-radius: 5px; margin-bottom: 15px; font-size: 0.95em;">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div style="background: #e6ffe6; color: #008000; padding: 12px; border-radius: 5px; margin-bottom: 15px; font-size: 0.95em;">
                <?php echo htmlspecialchars($success); ?>
            </div>
        <?php endif; ?>

        <form method="POST" id="addUserForm">
            <!-- Username -->
            <input type="text" name="username" placeholder="Nom d'utilisateur" value="<?php echo htmlspecialchars($input['username'] ?? ''); ?>" required
                   style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">

            <!-- Full Name -->
            <input type="text" name="full_name" placeholder="Nom complet" value="<?php echo htmlspecialchars($input['full_name'] ?? ''); ?>" required
                   style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">

            <!-- Email -->
            <input type="email" name="email" placeholder="Email" value="<?php echo htmlspecialchars($input['email'] ?? ''); ?>" required
                   style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">

            <!-- Mobile 1 -->
            <input type="tel" name="mobile1" placeholder="Téléphone principal" value="<?php echo htmlspecialchars($input['mobile1'] ?? ''); ?>"
                   style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">

            <!-- Mobile 2 -->
            <input type="tel" name="mobile2" placeholder="Téléphone secondaire (optionnel)" value="<?php echo htmlspecialchars($input['mobile2'] ?? ''); ?>"
                   style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">

         <!-- Address -->
         <input type="text" name="address" placeholder="Adresse (optionnel)" value="<?php echo htmlspecialchars($input['address'] ?? ''); ?>"
             style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">

            <!-- Role -->
            <?php if ($currentUserRole !== 'Admin'): ?>
                <p style="color:#6B4F2F; font-size:0.9em; margin:5px 0 0;">
                    Les rôles Admin, DG et DGA sont réservés au profil Admin.
                </p>
            <?php endif; ?>
            <select name="role_id" id="role_id" required onchange="toggleAssignments()"
                    style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">
                <option value=""><?php echo htmlspecialchars('Sélectionnez un rôle'); ?></option>
                <?php foreach ($roles as $role): ?>
                    <?php
                    $roleName = $role['name'];
                    $isRestrictedRole = in_array($roleName, $adminOnlyRoles, true);
                    $disabledAttr = ($isRestrictedRole && $currentUserRole !== 'Admin') ? 'disabled' : '';
                    $requiresProvinceAttr = mbh_role_requires_depot($roleName) ? '1' : '0';
                    $requiresRegionAttr = mbh_role_requires_region($roleName) ? '1' : '0';
                    $autoDepartmentsAttr = implode(',', mbh_get_role_departments(mbh_normalize_role_name($roleName)));
                    ?>
                    <option value="<?php echo $role['id']; ?>"
                            data-requires-province="<?php echo $requiresProvinceAttr; ?>"
                            data-requires-region="<?php echo $requiresRegionAttr; ?>"
                            data-auto-departments="<?php echo htmlspecialchars($autoDepartmentsAttr, ENT_QUOTES, 'UTF-8'); ?>"
                            <?php echo ($selected_role_id == $role['id']) ? 'selected' : ''; ?> <?php echo $disabledAttr; ?>>
                        <?php echo htmlspecialchars($roleName); ?>
                    </option>
                <?php endforeach; ?>
            </select>

            <!-- Dépôt (Conditional) -->
            <div id="province_section" style="display: none; margin: 10px 0;">
                <label for="depot_search" style="display:block; margin-bottom:6px; font-weight:600; color:#4B2F1F;">Dépôt assigné</label>
                <?php if (empty($provinces)): ?>
                    <p style="color:#B00020; font-size:0.9em;">Aucun dépôt n'est disponible. Créez un dépôt avant d'assigner ce rôle.</p>
                <?php else: ?>
                    <input type="text" id="depot_search" placeholder="Tapez pour filtrer les dépôts"
                           style="width:100%; padding:10px; border:2px solid #4B2F1F; border-radius:8px; margin-bottom:8px;">
                    <select name="province_id" id="province_id"
                            style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">
                        <option value=""><?php echo htmlspecialchars('Sélectionnez un dépôt'); ?></option>
                        <?php foreach ($provinces as $prov): ?>
                            <option value="<?php echo $prov['id']; ?>" <?php echo (($input['province_id'] ?? '') == $prov['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($prov['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                <?php endif; ?>
            </div>

            <!-- Region (Conditional) -->
            <div id="region_section" style="display: none; margin: 10px 0;">
                <label for="region_id" style="display:block; margin-bottom:6px; font-weight:600; color:#4B2F1F;">Région</label>
                <select name="region_id" id="region_id"
                        style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">
                    <option value=""><?php echo htmlspecialchars('Sélectionnez une région'); ?></option>
                    <?php foreach ($regions as $region): ?>
                        <option value="<?php echo $region['id']; ?>" <?php echo (($input['region_id'] ?? '') == $region['id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($region['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <!-- Departments -->
            <div id="department_section" style="margin: 15px 0;">
                <p style="margin:0 0 8px; font-weight:600; color:#4B2F1F;">Départements assignés</p>
                <?php foreach ($departments as $department): ?>
                    <?php
                        $code = $department['code'];
                        $checked = in_array($code, $input['department_codes'] ?? [], true);
                    ?>
                    <label style="display:block; margin-bottom:6px; color:#4B2F1F;">
                        <input type="checkbox" class="department-checkbox" name="department_codes[]" value="<?php echo htmlspecialchars($code); ?>"
                               <?php echo $checked ? 'checked' : ''; ?>>
                        <?php echo htmlspecialchars($department['label']); ?>
                    </label>
                <?php endforeach; ?>
                <div id="department_hidden_inputs"></div>
            </div>

            <!-- Password -->
            <input type="password" name="password" placeholder="Mot de passe" required
                   style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;">

            <!-- Active -->
            <label style="display: block; margin: 15px 0; color: #000;">
                <input type="checkbox" name="is_active" checked> Utilisateur actif
            </label>

            <!-- Submit -->
            <button type="submit"
                    style="background-color: #F4A261; color: #000; padding: 14px; border: none; border-radius: 8px; cursor: pointer; width: 100%; font-size: 16px; font-weight: bold;">
                Créer l'utilisateur
            </button>
        </form>
    </div>
</div>

<script>
function toggleAssignments() {
    const roleSelect = document.getElementById('role_id');
    const provinceSection = document.getElementById('province_section');
    const provinceSelect = document.getElementById('province_id');
    const regionSection = document.getElementById('region_section');
    const regionSelect = document.getElementById('region_id');
    const selectedOption = roleSelect.options[roleSelect.selectedIndex];

    const needsProvince = selectedOption && selectedOption.dataset.requiresProvince === '1';
    const needsRegion = selectedOption && selectedOption.dataset.requiresRegion === '1';

    if (provinceSection) {
        provinceSection.style.display = needsProvince ? 'block' : 'none';
        if (provinceSelect) {
            if (needsProvince) {
                provinceSelect.setAttribute('required', 'required');
            } else {
                provinceSelect.removeAttribute('required');
                provinceSelect.value = '';
            }
        }
    }

    if (regionSection) {
        regionSection.style.display = needsRegion ? 'block' : 'none';
        if (regionSelect) {
            if (needsRegion) {
                regionSelect.setAttribute('required', 'required');
            } else {
                regionSelect.removeAttribute('required');
                regionSelect.value = '';
            }
        }
    }

    const autoDepartments = selectedOption && selectedOption.dataset.autoDepartments
        ? selectedOption.dataset.autoDepartments.split(',').filter(Boolean)
        : [];
    updateDepartmentCheckboxes(autoDepartments);
}

function updateDepartmentCheckboxes(autoDepartments) {
    const checkboxes = document.querySelectorAll('.department-checkbox');
    const hiddenContainer = document.getElementById('department_hidden_inputs');
    if (hiddenContainer) {
        hiddenContainer.innerHTML = '';
    }
    if (checkboxes.length === 0) {
        return;
    }
    if (autoDepartments.length > 0) {
        checkboxes.forEach(cb => {
            const shouldCheck = autoDepartments.includes(cb.value);
            cb.checked = shouldCheck;
            cb.disabled = true;
        });
        if (hiddenContainer) {
            autoDepartments.forEach(code => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'department_codes[]';
                input.value = code;
                hiddenContainer.appendChild(input);
            });
        }
    } else {
        checkboxes.forEach(cb => {
            cb.disabled = false;
        });
    }
}

function filterDepots() {
    const searchInput = document.getElementById('depot_search');
    if (!searchInput) {
        return;
    }
    const filterValue = (searchInput.value || '').toLowerCase();
    const provinceSelect = document.getElementById('province_id');
    if (!provinceSelect) {
        return;
    }
    const options = provinceSelect.querySelectorAll('option');
    let firstVisibleValue = null;

    options.forEach(option => {
        if (option.value === '') {
            option.hidden = false;
            return;
        }
        const text = (option.text || '').toLowerCase();
        const isMatch = filterValue === '' || text.startsWith(filterValue);
        option.hidden = !isMatch;
        if (isMatch && firstVisibleValue === null) {
            firstVisibleValue = option.value;
        }
    });

    if (provinceSelect.value && provinceSelect.querySelector(`option[value="${provinceSelect.value}"]`)?.hidden) {
        provinceSelect.value = '';
    }

    if (filterValue !== '' && firstVisibleValue !== null) {
        provinceSelect.value = firstVisibleValue;
    }
}

document.addEventListener('DOMContentLoaded', () => {
    toggleAssignments();
    const depotSearch = document.getElementById('depot_search');
    if (depotSearch) {
        depotSearch.addEventListener('input', filterDepots);
    }
});
</script>

<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
