<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Stock Manager or Operation Manager roles
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in receive_stock.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
$allowedRoles = ['Stock Manager', 'Operation Manager'];
if (!in_array($role, $allowedRoles, true)) {
    error_log("Unauthorized access to receive_stock.php by user {$_SESSION['full_name']} (Role: $role)");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}
$userId = (int)($_SESSION['user_id'] ?? 0);
$isStockManager = ($role === 'Stock Manager');
$isOperationManager = ($role === 'Operation Manager');

// Build WHERE fragments so Operation Managers can also act as receivers when assigned
    $receiverWhereSelect = '';
    $receiverWhereSelectTypes = '';
    $receiverWhereSelectParams = [];
    $receiverWhereUpdate = '';
    $receiverWhereUpdateTypes = '';
    $receiverWhereUpdateParams = [];
    if ($isStockManager) {
        $receiverWhereSelect = " AND st.stock_manager_id = ?";
        $receiverWhereSelectTypes = 'i';
        $receiverWhereSelectParams[] = $userId;

        $receiverWhereUpdate = " AND stock_manager_id = ?";
        $receiverWhereUpdateTypes = 'i';
        $receiverWhereUpdateParams[] = $userId;
    } elseif ($isOperationManager) {
        $receiverWhereSelect = " AND (st.stock_manager_id = ? OR st.operation_receiver_id = ? OR (st.operation_receiver_id IS NULL AND st.operation_manager_id = ?))";
        $receiverWhereSelectTypes = 'iii';
        $receiverWhereSelectParams[] = $userId;
        $receiverWhereSelectParams[] = $userId;
        $receiverWhereSelectParams[] = $userId;

        $receiverWhereUpdate = " AND (stock_manager_id = ? OR operation_receiver_id = ? OR (operation_receiver_id IS NULL AND operation_manager_id = ?))";
        $receiverWhereUpdateTypes = 'iii';
        $receiverWhereUpdateParams[] = $userId;
        $receiverWhereUpdateParams[] = $userId;
        $receiverWhereUpdateParams[] = $userId;
    }

ensure_region_stock_table($conn);

// Ensure province_stock.id has AUTO_INCREMENT
$check_stmt = $conn->query("SHOW COLUMNS FROM province_stock WHERE Field = 'id'");
$id_column = $check_stmt->fetch_assoc();
$check_stmt->close();
if ($id_column && strpos($id_column['Extra'], 'auto_increment') === false) {
    // Add AUTO_INCREMENT if missing
    $conn->query("ALTER TABLE province_stock MODIFY COLUMN id int(11) NOT NULL AUTO_INCREMENT");
}

// Handle confirm reception action
$notificationTargets = null;
$errors = [];
$success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $shipment_id = intval($_POST['shipment_id'] ?? 0);

    if ($shipment_id <= 0) {
        $errors[] = "Invalid transfert ID.";
    } else {
        $conn->begin_transaction();
        try {
            // Fetch transfer details
            $stmt = $conn->prepare("
                SELECT st.product_id, st.quantity, st.province_id, st.region_id, st.operation_manager_id, st.operation_receiver_id, st.stock_manager_id, st.initiator_id,
                       u.region_id AS operation_region_id, initiator.region_id AS initiator_region_id, COALESCE(initiator.full_name, u.full_name) AS sender_name
                FROM stock_transfer st
                LEFT JOIN user u ON st.operation_manager_id = u.id
                LEFT JOIN user initiator ON st.initiator_id = initiator.id
                WHERE st.shipment_id = ? AND st.status = 'pending' {$receiverWhereSelect}
            ");
            $selectTypes = 'i' . $receiverWhereSelectTypes;
            $selectParams = array_merge([$shipment_id], $receiverWhereSelectParams);
            $stmt->bind_param($selectTypes, ...$selectParams);
            $stmt->execute();
            $transfers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();

            if (empty($transfers)) {
                throw new Exception("No pending transfers found for transfert ID $shipment_id.");
            }

            if ($action === 'confirm') {
                // Confirm reception - add all products to province_stock
                // province_stock is where received stock is stored for the province
                foreach ($transfers as $transfer) {
                    $product_id = $transfer['product_id'];
                    $quantity = $transfer['quantity'];
                    $province_id = (int)($transfer['province_id'] ?? 0);
                    $region_id = (int)($transfer['region_id'] ?? 0);
                    $province_stock_id = null;

                    if ($province_id > 0) {
                        $stmt = $conn->prepare("SELECT id FROM province_stock WHERE product_id = ? AND province_id = ?");
                        $stmt->bind_param("ii", $product_id, $province_id);
                        $stmt->execute();
                        $result = $stmt->get_result();
                        $existing = $result->fetch_assoc();
                        $stmt->close();

                        if ($existing) {
                            $province_stock_id = $existing['id'];
                            $stmt = $conn->prepare("
                                UPDATE province_stock 
                                SET quantity = quantity + ?, unit_type = 'crates'
                                WHERE id = ?
                            ");
                            $stmt->bind_param("ii", $quantity, $province_stock_id);
                            if (!$stmt->execute()) {
                                throw new Exception("Failed to update province stock for product ID $product_id: " . $stmt->error);
                            }
                            $stmt->close();
                        } else {
                            $stmt = $conn->prepare("
                                INSERT INTO province_stock (product_id, province_id, quantity, unit_type)
                                VALUES (?, ?, ?, 'crates')
                            ");
                            $stmt->bind_param("iii", $product_id, $province_id, $quantity);
                            if (!$stmt->execute()) {
                                throw new Exception("Failed to insert province stock for product ID $product_id: " . $stmt->error);
                            }
                            $province_stock_id = $conn->insert_id;
                            $stmt->close();
                        }
                    } elseif ($region_id > 0) {
                        if (!mbh_increment_region_stock($conn, $region_id, $product_id, $quantity)) {
                            throw new Exception("Failed to update region stock for product ID $product_id.");
                        }
                    } else {
                        throw new Exception("Transfer $shipment_id lacks a valid destination for product ID $product_id.");
                    }

                    $stmt = $conn->prepare("
                        UPDATE stock_transfer
                        SET status = 'confirmed', stock_manager_id = ?, received_at = NOW(), province_stock_id = ?
                        WHERE shipment_id = ? AND product_id = ? AND status = 'pending' {$receiverWhereUpdate}
                    ");
                    $updateTypes = "iiii{$receiverWhereUpdateTypes}";
                    $updateParams = array_merge(
                        [$_SESSION['user_id'], $province_stock_id, $shipment_id, $product_id],
                        $receiverWhereUpdateParams
                    );
                    $stmt->bind_param($updateTypes, ...$updateParams);
                    if (!$stmt->execute() || $stmt->affected_rows === 0) {
                        throw new Exception("Failed to confirm transfer for product ID $product_id.");
                    }
                    $stmt->close();
                }
                log_action($_SESSION['user_id'], 'confirm_transfer', "Confirmed stock transfer reception for transfert ID $shipment_id");

                $operationManagerIds = [];
                $receiverIds = [];
                foreach ($transfers as $transfer) {
                    $senderId = (int)($transfer['initiator_id'] ?? $transfer['operation_manager_id'] ?? 0);
                    if ($senderId > 0) {
                        $operationManagerIds[$senderId] = true;
                    }
                    $receiverId = (int)($transfer['stock_manager_id'] ?? 0);
                    if ($receiverId <= 0) {
                        $receiverId = (int)($transfer['operation_receiver_id'] ?? 0);
                    }
                    if ($receiverId <= 0) {
                        $receiverId = $userId;
                    }
                    if ($receiverId > 0) {
                        $receiverIds[$receiverId] = true;
                    }
                }

                // Collect all sender IDs (Operation Managers, DGA, etc. who initiated the transfer)
                $allSenderIds = [];
                foreach ($transfers as $transfer) {
                    // Get initiator (who started the transfer)
                    $initiatorId = (int)($transfer['initiator_id'] ?? 0);
                    if ($initiatorId > 0) {
                        $allSenderIds[$initiatorId] = true;
                    }
                    // Also get operation_manager_id if different (for backward compatibility)
                    $opMgrId = (int)($transfer['operation_manager_id'] ?? 0);
                    if ($opMgrId > 0 && $opMgrId !== $initiatorId) {
                        $allSenderIds[$opMgrId] = true;
                    }
                }

                $notificationTargets = [
                    'type' => 'confirm',
                    'shipment_id' => $shipment_id,
                    'senders' => array_keys($allSenderIds),
                    'receivers' => array_keys($receiverIds),
                ];

                $success = "Transfert $shipment_id confirmed successfully. All products have been added to province stock. Initiator has been notified.";
            } elseif ($action === 'reject') {
                foreach ($transfers as $transfer) {
                    $product_id = $transfer['product_id'];
                    $quantity = $transfer['quantity'];
                    $origin_operation_manager = (int)($transfer['initiator_id'] ?? $transfer['operation_manager_id'] ?? 0);
                    $origin_region_id = (int)($transfer['initiator_region_id'] ?? $transfer['operation_region_id'] ?? 0);

                    if ($origin_region_id > 0) {
                        if (!mbh_increment_region_stock($conn, $origin_region_id, $product_id, $quantity)) {
                            throw new Exception("Failed to restore regional stock for product ID $product_id.");
                        }
                    } else {
                        $stmt = $conn->prepare("
                            INSERT INTO main_stock (product_id, quantity, unit_type)
                            VALUES (?, ?, 'crates')
                            ON DUPLICATE KEY UPDATE quantity = quantity + ?, unit_type = 'crates'
                        ");
                        $stmt->bind_param("iii", $product_id, $quantity, $quantity);
                        if (!$stmt->execute()) {
                            throw new Exception("Failed to restore main stock for product ID $product_id: " . $stmt->error);
                        }
                        $stmt->close();
                    }

                    // Update stock_transfer status to rejected
                    $stmt = $conn->prepare("
                        UPDATE stock_transfer
                        SET status = 'rejected', stock_manager_id = ?, received_at = NOW()
                        WHERE shipment_id = ? AND product_id = ? AND status = 'pending' {$receiverWhereUpdate}
                    ");
                    $rejectTypes = "iii{$receiverWhereUpdateTypes}";
                    $rejectParams = array_merge(
                        [$_SESSION['user_id'], $shipment_id, $product_id],
                        $receiverWhereUpdateParams
                    );
                    $stmt->bind_param($rejectTypes, ...$rejectParams);
                    if (!$stmt->execute() || $stmt->affected_rows === 0) {
                        throw new Exception("Failed to reject transfer for product ID $product_id.");
                    }
                    $stmt->close();
                }
                log_action($_SESSION['user_id'], 'reject_transfer', "Rejected stock transfer for transfert ID $shipment_id");
                
                // Collect all sender IDs (Operation Managers, DGA, etc. who initiated the transfer)
                $allSenderIds = [];
                $receiverIds = [];
                foreach ($transfers as $transfer) {
                    // Get initiator (who started the transfer)
                    $initiatorId = (int)($transfer['initiator_id'] ?? 0);
                    if ($initiatorId > 0) {
                        $allSenderIds[$initiatorId] = true;
                    }
                    // Also get operation_manager_id if different (for backward compatibility)
                    $opMgrId = (int)($transfer['operation_manager_id'] ?? 0);
                    if ($opMgrId > 0 && $opMgrId !== $initiatorId) {
                        $allSenderIds[$opMgrId] = true;
                    }
                    // Get receiver ID
                    $receiverId = (int)($transfer['stock_manager_id'] ?? 0);
                    if ($receiverId <= 0) {
                        $receiverId = (int)($transfer['operation_receiver_id'] ?? 0);
                    }
                    if ($receiverId <= 0) {
                        $receiverId = $userId;
                    }
                    if ($receiverId > 0) {
                        $receiverIds[$receiverId] = true;
                    }
                }

                $notificationTargets = [
                    'type' => 'reject',
                    'shipment_id' => $shipment_id,
                    'senders' => array_keys($allSenderIds),
                    'receivers' => array_keys($receiverIds),
                ];

                $success = "Transfert $shipment_id rejected successfully. Products restored to main stock. Initiator has been notified.";
            } else {
                throw new Exception("Invalid action specified.");
            }

            $conn->commit();

            if ($notificationTargets && isset($notificationTargets['shipment_id'])) {
                $link = "/masunzu_bar_hotel/modules/operation/receive_stock.php";
                $shipmentArrived = (int)$notificationTargets['shipment_id'];
                $type = $notificationTargets['type'] ?? '';
                $senderMessage = $type === 'confirm'
                    ? "Le transfert #$shipmentArrived a été reçu et confirmé. Les produits ont été ajoutés au stock de la province."
                    : "Le transfert #$shipmentArrived a été rejeté. Les produits ont été restaurés au stock principal.";
                $receiverMessage = $type === 'confirm'
                    ? "Réception du transfert #$shipmentArrived confirmée. Stock de la province mis à jour."
                    : "La réception du transfert #$shipmentArrived a été rejetée.";
                $senderIds = array_filter($notificationTargets['senders'] ?? [], function ($id) { return (int)$id > 0; });
                $receiverIds = array_filter($notificationTargets['receivers'] ?? [], function ($id) { return (int)$id > 0; });
                $senderIds = array_values(array_unique(array_map('intval', $senderIds)));
                $receiverIds = array_values(array_unique(array_map('intval', $receiverIds)));
                foreach ($senderIds as $recipient) {
                    mbh_notify_user($conn, (int)$recipient, $senderMessage, $link);
                }
                foreach ($receiverIds as $recipient) {
                    if (in_array($recipient, $senderIds, true)) {
                        continue;
                    }
                    mbh_notify_user($conn, (int)$recipient, $receiverMessage, $link);
                }
            }
        } catch (Exception $e) {
            $conn->rollback();
            $errors[] = "Failed to $action transfert. Please try again or contact the administrator.";
            error_log("Action '$action' error for transfert ID $shipment_id: " . $e->getMessage());
        }
    }
}

// Fetch pending transfers grouped by shipment_id
$shipmentQuery = "
    SELECT 
        st.shipment_id, 
        st.province_id, 
        st.region_id,
        COALESCE(p.name, '') AS province_name,
        COALESCE(r.name, '') AS region_name,
        st.operation_manager_id,
        st.operation_receiver_id,
        st.initiator_id,
        COALESCE(initiator.full_name, u.full_name) AS sender_name,
        COALESCE(initiator.region_id, u.region_id) AS initiator_region_id,
        st.sent_at,
        st.estimated_arrival,
        st.driver_name,
        st.driver_mobile,
        st.plate_number,
        GROUP_CONCAT(
            CONCAT(st.product_id, ':', pr.name, ':', st.quantity, ':', pr.price_per_crate, ':', pr.crate_quantity)
            SEPARATOR '|'
        ) AS product_details
    FROM stock_transfer st
    LEFT JOIN province p ON st.province_id = p.id
    LEFT JOIN region r ON st.region_id = r.id
    LEFT JOIN user u ON st.operation_manager_id = u.id
    LEFT JOIN user initiator ON st.initiator_id = initiator.id
    JOIN product pr ON st.product_id = pr.id
    WHERE st.status = 'pending' {$receiverWhereSelect}
    GROUP BY st.shipment_id
";
$stmt = $conn->prepare($shipmentQuery);
if ($receiverWhereSelectTypes) {
    $stmt->bind_param($receiverWhereSelectTypes, ...$receiverWhereSelectParams);
}
$stmt->execute();
$shipments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();
$conn->close();

include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<div class="receive-page-shell">
    <div class="receive-page-inner">
        <div class="receive-header">
            <h2>Réception des transferts</h2>
            <p>Les transferts en attente de réception (dans votre région) sont listés ci-dessous.</p>
        </div>

        <?php if ($success): ?>
            <div class="notice success"><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div class="notice danger">
                <ul>
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <?php if (empty($shipments)): ?>
            <div class="empty-state">
                <p>Aucun transfert en attente pour le moment.</p>
            </div>
        <?php else: ?>
            <div class="shipments-grid">
                <?php foreach ($shipments as $shipment): ?>
                    <?php
                    $products = [];
                    $total_amount = 0;
                    foreach (explode('|', $shipment['product_details']) as $detail) {
                        list($product_id, $product_name, $quantity, $price_per_crate, $crate_quantity) = explode(':', $detail);
                        $crate_qty = (int)$crate_quantity;
                        $qty_crates = (int)$quantity;
                        $products[] = [
                            'product_id' => $product_id,
                            'name' => $product_name,
                            'quantity_crates' => $qty_crates,
                            'price_per_crate' => $price_per_crate,
                            'amount' => $qty_crates * $price_per_crate
                        ];
                        $total_amount += $qty_crates * $price_per_crate;
                    }
                    ?>
                    <?php $destinationName = $shipment['province_name'] ?: $shipment['region_name'] ?: 'Destination non assignée'; ?>
                    <article class="shipment-card">
                        <div class="shipment-card__header">
                            <div>
                                <span class="muted">Transfert</span>
                                <strong><?php echo htmlspecialchars($shipment['shipment_id'], ENT_QUOTES, 'UTF-8'); ?></strong>
                            </div>
                            <span class="badge"><?php echo htmlspecialchars($destinationName, ENT_QUOTES, 'UTF-8'); ?></span>
                        </div>
                        <div class="shipment-card__details">
                            <div>
                                <p class="label">Expéditeur</p>
                                <p><?php echo htmlspecialchars($shipment['sender_name'], ENT_QUOTES, 'UTF-8'); ?></p>
                            </div>
                            <div>
                                <p class="label">Destination</p>
                                <p><?php echo htmlspecialchars($destinationName, ENT_QUOTES, 'UTF-8'); ?></p>
                            </div>
                            <div>
                                <p class="label">Envoyé le</p>
                                <p><?php echo htmlspecialchars($shipment['sent_at'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                            </div>
                            <div>
                                <p class="label">Arrivée prévue</p>
                                <p><?php echo htmlspecialchars($shipment['estimated_arrival'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                            </div>
                            <div>
                                <p class="label">Chauffeur</p>
                                <p><?php echo htmlspecialchars($shipment['driver_name'], ENT_QUOTES, 'UTF-8'); ?></p>
                            </div>
                            <div>
                                <p class="label">Téléphone</p>
                                <p><?php echo htmlspecialchars($shipment['driver_mobile'], ENT_QUOTES, 'UTF-8'); ?></p>
                            </div>
                            <div>
                                <p class="label">Plaque</p>
                                <p><?php echo htmlspecialchars($shipment['plate_number'], ENT_QUOTES, 'UTF-8'); ?></p>
                            </div>
                        </div>

                        <div class="product-table-wrapper">
                            <table>
                                <thead>
                                <tr>
                                    <th>Produit</th>
                                    <th class="center">Caisses</th>
                                    <th class="right">Prix / caisse</th>
                                    <th class="right">Total</th>
                                </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($products as $product): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                            <td class="center"><?php echo number_format($product['quantity_crates'], 0, '.', ','); ?></td>
                                            <td class="right"><?php echo number_format($product['price_per_crate'], 0, '.', ','); ?> BIF</td>
                                            <td class="right"><?php echo number_format($product['amount'], 0, '.', ','); ?> BIF</td>
                                        </tr>
                                    <?php endforeach; ?>
                                <tr class="total-row">
                                    <td><strong>Total</strong></td>
                                    <td></td>
                                    <td></td>
                                    <td class="right"><strong><?php echo number_format($total_amount, 0, '.', ','); ?> BIF</strong></td>
                                </tr>
                                </tbody>
                            </table>
                        </div>

                        <form method="POST" class="shipment-actions">
                            <input type="hidden" name="shipment_id" value="<?php echo $shipment['shipment_id']; ?>">
                            <button type="submit" name="action" value="confirm" class="btn confirm">Confirmer la réception</button>
                            <button type="submit" name="action" value="reject" class="btn reject">Rejeter</button>
                        </form>
                        <p class="note">Note : Confirmer l'arrivée met les produits à jour dans le stock provincial.</p>
                    </article>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>
<style>
    body {
        background: #F4F0E4;
    }
    .receive-page-shell {
        min-height: 100vh;
        background: #F4F0E4;
        display: flex;
        justify-content: center;
        padding: 40px 20px 80px;
        box-sizing: border-box;
    }
    .receive-page-inner {
        width: min(1300px, 100%);
        background: #fff;
        border-radius: 24px;
        padding: 32px;
        box-shadow: 0 20px 45px rgba(15, 15, 15, 0.15);
        display: flex;
        flex-direction: column;
        gap: 20px;
    }
    .receive-header h2 {
        margin: 0;
        font-size: 30px;
        color: #2F1F12;
    }
    .receive-header p {
        margin: 4px 0 0;
        color: #4B2F1F;
    }
    .notice {
        padding: 16px;
        border-radius: 12px;
        font-size: 15px;
        border: 1px solid transparent;
    }
    .notice.success {
        background: #E8F6E4;
        color: #23562F;
        border-color: #94D7A7;
    }
    .notice.danger {
        background: #FFEFEF;
        color: #9C1B1B;
        border-color: #F2A0A0;
    }
    .notice ul {
        margin: 0;
        padding-left: 18px;
    }
    .empty-state {
        text-align: center;
        color: #5A3924;
        padding: 32px;
        border-radius: 16px;
        background: #fff6ed;
        border: 1px solid #f0c9a0;
    }
    .shipments-grid {
        display: flex;
        flex-direction: column;
        gap: 24px;
    }
    .shipment-card {
        border-radius: 18px;
        background: #F4F0E4;
        padding: 24px;
        display: flex;
        flex-direction: column;
        gap: 16px;
        border: 1px solid rgba(244, 240, 228, 0.7);
    }
    .shipment-card__header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        gap: 12px;
    }
    .shipment-card__header strong {
        font-size: 18px;
    }
    .badge {
        padding: 6px 16px;
        background: #4B2F1F;
        color: #F4F0E4;
        border-radius: 999px;
        font-size: 14px;
        text-transform: uppercase;
        letter-spacing: 0.4px;
    }
    .shipment-card__details {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
        gap: 16px;
        font-size: 13px;
        color: #3E2721;
    }
    .shipment-card__details .label {
        text-transform: uppercase;
        letter-spacing: 0.5px;
        font-weight: 600;
        color: #4B2F1F;
        margin: 0 0 4px;
        font-size: 11px;
    }
    .product-table-wrapper {
        overflow-x: auto;
        background: #fff;
        border-radius: 12px;
        padding: 12px;
        box-shadow: inset 0 0 0 1px rgba(75, 47, 31, 0.1);
    }
    table {
        width: 100%;
        border-collapse: collapse;
        font-size: 14px;
    }
    thead th {
        text-align: left;
        font-size: 12px;
        color: #4B2F1F;
        text-transform: uppercase;
        letter-spacing: 0.6px;
    }
    tbody tr {
        border-bottom: 1px solid #EADDDB;
    }
    tbody tr:last-child {
        border-bottom: none;
    }
    td, th {
        padding: 10px;
    }
    .center {
        text-align: center;
    }
    .right {
        text-align: right;
    }
    .total-row {
        background: rgba(244, 162, 97, 0.15);
        font-weight: 600;
    }
    .shipment-actions {
        margin-top: 4px;
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
    }
    .btn {
        border: none;
        border-radius: 10px;
        padding: 12px 24px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s ease, box-shadow 0.2s ease;
    }
    .btn.confirm {
        background: #4B2F1F;
        color: #F4F0E4;
    }
    .btn.reject {
        background: #dc3545;
        color: #ffffff;
    }
    .btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px rgba(15, 15, 15, 0.15);
    }
    .note {
        margin: 0;
        font-size: 12px;
        color: #2F1F12;
    }
    @media (max-width: 768px) {
        .receive-page-inner {
            padding: 24px;
        }
        .shipment-card__details {
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        }
    }
</style>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
