<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../config/database.php';

require_login();

// Restrict to DG, DGA, Board, Admin, Operation Manager, Stock Manager, Senior Accountant, Junior Accountant
$allowed_roles = ['DG', 'DGA', 'Board', 'Admin', 'Operation Manager', 'Stock Manager', 'Senior Accountant', 'Junior Accountant', 'Accountant'];
if (!in_array($_SESSION['role_name'] ?? '', $allowed_roles)) {
    error_log("Unauthorized access to product_price_history.php by user {$_SESSION['full_name']} (Role: {$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $_SESSION['role_name'])) . "_dashboard.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in product_price_history.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Ensure product_price_history table exists
// First check if table exists and has correct structure
$tableCheck = $conn->query("SHOW TABLES LIKE 'product_price_history'");
if ($tableCheck && $tableCheck->num_rows > 0) {
    // Table exists, check columns
    $columnCheck = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'old_price'");
    if (!$columnCheck || $columnCheck->num_rows === 0) {
        // Table exists but missing columns - drop and recreate
        $conn->query("DROP TABLE IF EXISTS product_price_history");
        $tableCheck->free();
        if ($columnCheck) $columnCheck->free();
    } else {
        if ($columnCheck) $columnCheck->free();
        $tableCheck->free();
    }
}

$createTableSql = "
    CREATE TABLE IF NOT EXISTS product_price_history (
        id INT AUTO_INCREMENT PRIMARY KEY,
        product_id INT NOT NULL,
        old_price INT NOT NULL,
        new_price INT NOT NULL,
        changed_by INT NOT NULL,
        changed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_product (product_id),
        INDEX idx_changed_at (changed_at),
        FOREIGN KEY (product_id) REFERENCES product(id) ON DELETE CASCADE,
        FOREIGN KEY (changed_by) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createTableSql)) {
    error_log("Failed to ensure product_price_history table: " . $conn->error);
    // If foreign key constraint fails, try without foreign keys
    $createTableSqlSimple = "
        CREATE TABLE IF NOT EXISTS product_price_history (
            id INT AUTO_INCREMENT PRIMARY KEY,
            product_id INT NOT NULL,
            old_price INT NOT NULL,
            new_price INT NOT NULL,
            changed_by INT NOT NULL,
            changed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_product (product_id),
            INDEX idx_changed_at (changed_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    if (!$conn->query($createTableSqlSimple)) {
        error_log("Failed to create product_price_history table even without foreign keys: " . $conn->error);
    }
}

// Handle filters
$product_id = isset($_GET['product_id']) ? (int)$_GET['product_id'] : 0;

$start_date_input = $_GET['start_date'] ?? '';
$end_date_input = $_GET['end_date'] ?? '';

$start_date = '';
if ($start_date_input) {
    $start_dt = DateTime::createFromFormat('Y-m-d', $start_date_input);
    if ($start_dt && $start_dt->format('Y-m-d') === $start_date_input) {
        $start_date = $start_date_input;
    }
}

$end_date = '';
if ($end_date_input) {
    $end_dt = DateTime::createFromFormat('Y-m-d', $end_date_input);
    if ($end_dt && $end_dt->format('Y-m-d') === $end_date_input) {
        $end_date = $end_date_input;
    }
}

$where_clauses = [];
$params = [];
$param_types = '';

if ($product_id > 0) {
    $where_clauses[] = "pph.product_id = ?";
    $params[] = $product_id;
    $param_types .= 'i';
}
if ($start_date) {
    $where_clauses[] = "DATE(pph.changed_at) >= ?";
    $params[] = $start_date;
    $param_types .= 's';
}
if ($end_date) {
    $where_clauses[] = "DATE(pph.changed_at) <= ?";
    $params[] = $end_date;
    $param_types .= 's';
}

$query = "SELECT pph.id, pph.product_id, pph.old_price, pph.new_price, pph.changed_at,
                 p.name AS product_name, p.created_at AS product_created_at,
                 u_changed.full_name AS changed_by_name,
                 u_created.full_name AS created_by_name
          FROM product_price_history pph
          LEFT JOIN product p ON pph.product_id = p.id
          LEFT JOIN user u_changed ON pph.changed_by = u_changed.id
          LEFT JOIN user u_created ON p.created_by = u_created.id";

if (!empty($where_clauses)) {
    $query .= " WHERE " . implode(' AND ', $where_clauses);
}

$query .= " ORDER BY pph.changed_at DESC";

$stmt = $conn->prepare($query);
if ($stmt && $param_types && !empty($params)) {
    $bind_params = [$param_types];
    foreach ($params as $key => $value) {
        $bind_params[] = &$params[$key];
    }
    if (!call_user_func_array([$stmt, 'bind_param'], $bind_params)) {
        error_log("Failed to bind parameters in product_price_history.php: " . $stmt->error);
    }
} elseif (!$stmt) {
    error_log("Query preparation failed in product_price_history.php: " . ($conn->error ?? 'Unknown error'));
    $price_history = [];
}

if ($stmt) {
    $stmt->execute();
    $price_history = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    $price_history = [];
}

// Get all products for filter dropdown
$products_query = $conn->query("SELECT id, name FROM product ORDER BY name");
$all_products = $products_query->fetch_all(MYSQLI_ASSOC);
$products_query->free();

$conn->close();
include __DIR__ . '/../../includes/header.php';
?>
<style>
    body {
        margin: 0;
        padding: 0;
        overflow-x: hidden;
    }
    .history-container {
        padding: 20px;
        background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%);
        min-height: 70vh;
        width: 100%;
        margin: 0;
        box-sizing: border-box;
    }
    .history-content {
        width: 100%;
        max-width: 100%;
        margin: 0 auto;
        padding: 0;
    }
    .filters-card, .history-card {
        background-color: #FFFFFF;
        padding: 20px;
        border-radius: 8px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        margin-bottom: 30px;
        border: none;
    }
    table {
        width: 100%;
        border-collapse: collapse;
    }
    table th, table td {
        border: none;
        padding: 12px;
        text-align: left;
    }
    table tr {
        border-bottom: 1px solid #E0E0E0;
    }
    table tr:hover {
        background-color: #F4A26120;
    }
    .price-increase {
        color: #4CAF50;
        font-weight: 600;
    }
    .price-decrease {
        color: #F44336;
        font-weight: 600;
    }
</style>

<div class="history-container">
    <div class="history-content">
        <h1 style="color: #4B2F1F; font-size: 28px; font-weight: bold; text-align: center; margin-bottom: 20px;">Historique des Changements de Prix des Produits</h1>
        
        <!-- Filters -->
        <div class="filters-card">
            <form method="GET" style="display: flex; flex-wrap: wrap; gap: 15px;">
                <div style="flex: 1; min-width: 200px;">
                    <label for="product_id" style="color: #4B2F1F; font-size: 14px; font-weight: 500;">Produit</label>
                    <select id="product_id" name="product_id" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                        <option value="">Tous les produits</option>
                        <?php foreach ($all_products as $prod): ?>
                            <option value="<?php echo $prod['id']; ?>" <?php echo ($product_id == $prod['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($prod['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div style="flex: 1; min-width: 150px;">
                    <label for="start_date" style="color: #4B2F1F; font-size: 14px; font-weight: 500;">Date de Début</label>
                    <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                </div>
                <div style="flex: 1; min-width: 150px;">
                    <label for="end_date" style="color: #4B2F1F; font-size: 14px; font-weight: 500;">Date de Fin</label>
                    <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
                </div>
                <div style="flex: 1; min-width: 150px; display: flex; align-items: flex-end;">
                    <button type="submit" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px; border: none; border-radius: 8px; font-size: 14px; cursor: pointer; width: 100%;">Appliquer les Filtres</button>
                </div>
            </form>
        </div>

        <!-- Price History Table -->
        <div class="history-card">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                <h2 style="color: #4B2F1F; font-size: 22px; margin: 0;">Historique des Changements de Prix</h2>
                <?php if (!empty($price_history)): ?>
                    <a href="product_catalog.php" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 20px; border-radius: 8px; text-decoration: none; font-size: 14px; display: inline-block;">← Retour au Catalogue</a>
                <?php endif; ?>
            </div>
            
            <?php if (empty($price_history)): ?>
                <p style="color: #4B2F1F; text-align: center; padding: 20px;">Aucun changement de prix trouvé pour les filtres sélectionnés.</p>
            <?php else: ?>
                <div style="overflow-x: auto;">
                    <table>
                        <thead>
                            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                                <th style="border: none;">ID</th>
                                <th style="border: none;">Produit</th>
                                <th style="border: none;">Ancien Prix (BIF)</th>
                                <th style="border: none;">Nouveau Prix (BIF)</th>
                                <th style="border: none;">Différence</th>
                                <th style="border: none;">Changé Par</th>
                                <th style="border: none;">Date du Changement</th>
                                <th style="border: none;">Créé Par</th>
                                <th style="border: none;">Date de Création Produit</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($price_history as $history): ?>
                                <?php
                                    $old_price = (int)$history['old_price'];
                                    $new_price = (int)$history['new_price'];
                                    $difference = $new_price - $old_price;
                                    $diff_class = $difference > 0 ? 'price-increase' : ($difference < 0 ? 'price-decrease' : '');
                                    $diff_symbol = $difference > 0 ? '+' : '';
                                ?>
                                <tr>
                                    <td style="border: none;"><?php echo htmlspecialchars($history['id']); ?></td>
                                    <td style="border: none; font-weight: 600;"><?php echo htmlspecialchars($history['product_name'] ?: 'N/A'); ?></td>
                                    <td style="border: none;"><?php echo number_format($old_price, 0, ',', '.'); ?> BIF</td>
                                    <td style="border: none; font-weight: 600;"><?php echo number_format($new_price, 0, ',', '.'); ?> BIF</td>
                                    <td style="border: none;" class="<?php echo $diff_class; ?>">
                                        <?php echo $diff_symbol . number_format($difference, 0, ',', '.'); ?> BIF
                                        <?php if ($old_price > 0): ?>
                                            <?php 
                                                $percent_change = ($difference / $old_price) * 100;
                                                echo '(' . number_format(abs($percent_change), 2, ',', '.') . '%)';
                                            ?>
                                        <?php endif; ?>
                                    </td>
                                    <td style="border: none;"><?php echo htmlspecialchars($history['changed_by_name'] ?: 'Inconnu'); ?></td>
                                    <td style="border: none;"><?php echo date('d/m/Y à H:i:s', strtotime($history['changed_at'])); ?></td>
                                    <td style="border: none;"><?php echo htmlspecialchars($history['created_by_name'] ?: 'Inconnu'); ?></td>
                                    <td style="border: none;"><?php echo $history['product_created_at'] ? date('d/m/Y à H:i', strtotime($history['product_created_at'])) : 'N/A'; ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
