<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../config/database.php';

require_login();

$allowedRoles = ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant', 'Caissier_Comptable'];
$roleName = $_SESSION['role_name'] ?? '';
if (!in_array($roleName, $allowedRoles, true)) {
    header('Location: /masunzu_bar_hotel/dashboards/unauthorized.php');
    exit;
}

$customerId = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
$customerId = $customerId > 0 ? $customerId : 0;
$error = '';
$customer = null;

$conn = get_db_connection();
if (!$conn) {
    $error = 'Impossible de se connecter à la base de données. Contactez l\'administrateur.';
} elseif ($customerId === 0) {
    $error = 'Identifiant client invalide.';
} else {
    $query = "
        SELECT
            c.id,
            c.full_name,
            c.nif,
            c.tel,
            c.driver_name,
            c.driver_contacts,
            c.plate_numbers,
            c.additional_contacts,
            c.address,
            c.user_type,
            c.can_pay_by_loan,
            c.loan_limit_bif,
            c.loan_balance_bif,
            c.loan_terms_days,
            c.loan_status,
            c.loan_last_review_at,
            c.created_at,
            u.full_name AS created_by_name,
            u.username AS created_by_username,
            r.name AS created_by_role,
            COALESCE(SUM(CASE
                WHEN ui.is_loan_sale = 1
                 AND ui.loan_outstanding_bif > 0
                 AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected')
                THEN ui.loan_outstanding_bif
                ELSE 0
            END), 0) AS loan_due
        FROM customer c
        LEFT JOIN user u ON c.created_by = u.id
        LEFT JOIN role r ON u.role_id = r.id
        LEFT JOIN unpaid_invoices ui ON ui.customer_id = c.id
        WHERE c.id = ?
        GROUP BY c.id, c.full_name, c.nif, c.tel, c.driver_name, c.driver_contacts, c.plate_numbers, c.additional_contacts, c.address,
                 c.user_type, c.can_pay_by_loan, c.loan_limit_bif, c.loan_balance_bif, c.loan_terms_days, c.loan_status, c.loan_last_review_at,
                 c.created_at, u.full_name, u.username, r.name
        LIMIT 1
    ";
    $stmt = $conn->prepare($query);
    if (!$stmt) {
        $error = 'Erreur lors de la récupération du client.';
    } else {
        $stmt->bind_param('i', $customerId);
        $stmt->execute();
        $customer = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        if (!$customer) {
            $error = 'Client introuvable ou supprimé.';
        }
    }
}

if ($conn) {
    $conn->close();
}

include __DIR__ . '/../../includes/header.php';
?>
<style>
    .customer-view-page {
        padding: 20px;
        background: #F4F0E4;
        min-height: calc(100vh - 140px);
    }
    .customer-card {
        max-width: 1100px;
        margin: 0 auto;
        background: #FFF;
        border-radius: 16px;
        padding: 30px;
        box-shadow: 0 18px 45px rgba(0,0,0,0.12);
        border: 1px solid #ECE7DA;
    }
    .customer-card__header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        flex-wrap: wrap;
        gap: 20px;
        margin-bottom: 25px;
    }
    .eyebrow {
        text-transform: uppercase;
        letter-spacing: 0.2em;
        font-size: 12px;
        color: #7C4A33;
        margin: 0 0 4px;
    }
    .customer-title {
        font-size: 32px;
        margin: 0;
        color: #331E10;
    }
    .customer-card__header p {
        margin: 0;
        color: #4B2F1F;
    }
    .type-pill {
        padding: 12px 20px;
        background: #4B2F1F;
        color: #fff;
        border-radius: 999px;
        font-weight: 600;
        text-transform: uppercase;
        font-size: 13px;
    }
    .detail-grid,
    .loan-grid,
    .meta-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
        gap: 16px;
        margin-bottom: 20px;
    }
    .detail-card,
    .detail-grid div,
    .loan-grid div,
    .meta-grid div {
        background: #F9F7F4;
        border-radius: 12px;
        padding: 14px 16px;
        border: 1px solid #EFE5D8;
    }
    .detail-grid .label,
    .loan-grid .label,
    .meta-grid .label {
        font-size: 12px;
        text-transform: uppercase;
        letter-spacing: 0.2em;
        color: #7C4A33;
        margin-bottom: 6px;
    }
    .detail-grid .value,
    .loan-grid .value,
    .meta-grid .value {
        font-size: 16px;
        color: #2C1810;
        margin: 0;
        line-height: 1.4;
    }
    .actions {
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
        margin-top: 10px;
    }
    .btn-primary,
    .btn-secondary {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 12px 24px;
        border-radius: 8px;
        font-weight: 600;
        border: none;
        cursor: pointer;
        text-decoration: none;
    }
    .btn-primary {
        background: #4B2F1F;
        color: #fff;
    }
    .btn-secondary {
        background: #EDE4D8;
        color: #4B2F1F;
    }
    .error-card {
        max-width: 720px;
        margin: 0 auto;
        background: #fff3f3;
        color: #8B1E3B;
        border-radius: 12px;
        padding: 24px;
        border: 1px solid #F5C6CB;
        box-shadow: 0 10px 24px rgba(139,30,59,0.15);
    }
    @media (max-width: 640px) {
        .customer-card {
            padding: 22px;
        }
    }
</style>
<div class="customer-view-page">
    <?php if ($error): ?>
        <div class="error-card">
            <strong>Erreur :</strong> <?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?>
        </div>
    <?php else: ?>
        <?php
            $detailType = isset($customer['user_type']) && $customer['user_type'] === 'moral' ? 'Personne Morale' : 'Personne Physique';
            $loanEligible = isset($customer['can_pay_by_loan']) && (int)$customer['can_pay_by_loan'] === 1 ? 'Oui' : 'Non';
        ?>
        <div class="customer-card">
            <div class="customer-card__header">
                <div>
                    <p class="eyebrow">Fiche client</p>
                    <h1 class="customer-title"><?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?></h1>
                    <p>NIF: <?php echo htmlspecialchars($customer['nif'] ?? 'Non renseigné', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div class="type-pill"><?php echo htmlspecialchars($detailType, ENT_QUOTES, 'UTF-8'); ?></div>
            </div>
            <div class="detail-grid">
                <div>
                    <p class="label">Téléphone</p>
                    <p class="value"><?php echo htmlspecialchars($customer['tel'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Contacts additionnels</p>
                    <p class="value"><?php echo htmlspecialchars($customer['additional_contacts'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Nom chauffeur</p>
                    <p class="value"><?php echo htmlspecialchars($customer['driver_name'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Contacts chauffeur</p>
                    <p class="value"><?php echo htmlspecialchars($customer['driver_contacts'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Plaques</p>
                    <p class="value"><?php echo htmlspecialchars($customer['plate_numbers'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Adresse</p>
                    <p class="value"><?php echo htmlspecialchars($customer['address'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
            </div>
            <div class="meta-grid">
                <div>
                    <p class="label">Éligible Crédit</p>
                    <p class="value"><?php echo htmlspecialchars($loanEligible, ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Limite Crédit</p>
                    <p class="value"><?php echo htmlspecialchars(number_format((int)($customer['loan_limit_bif'] ?? 0), 0, ',', '.'), ENT_QUOTES, 'UTF-8'); ?> BIF</p>
                </div>
                <div>
                    <p class="label">Crédit Impayé</p>
                    <p class="value"><?php echo htmlspecialchars(number_format((int)($customer['loan_balance_bif'] ?? 0), 0, ',', '.'), ENT_QUOTES, 'UTF-8'); ?> BIF</p>
                </div>
                <div>
                    <p class="label">Conditions Crédit</p>
                    <p class="value"><?php echo isset($customer['loan_terms_days']) && $customer['loan_terms_days'] !== null ? (int)$customer['loan_terms_days'] . ' jours' : '—'; ?></p>
                </div>
                <div>
                    <p class="label">Statut Crédit</p>
                    <p class="value"><?php echo htmlspecialchars(ucfirst((string)($customer['loan_status'] ?? 'inactive')), ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Dernière révision</p>
                    <p class="value"><?php echo $customer['loan_last_review_at'] ? htmlspecialchars(date('d/m/Y H:i', strtotime($customer['loan_last_review_at'])), ENT_QUOTES, 'UTF-8') : '—'; ?></p>
                </div>
            </div>
            <div class="loan-grid">
                <div>
                    <p class="label">Crédit dû actuel</p>
                    <p class="value"><?php echo htmlspecialchars(number_format((int)($customer['loan_due'] ?? 0), 0, ',', '.'), ENT_QUOTES, 'UTF-8'); ?> BIF</p>
                </div>
                <div>
                    <p class="label">Créé par</p>
                    <p class="value"><?php echo htmlspecialchars($customer['created_by_name'] ?? '—', ENT_QUOTES, 'UTF-8'); ?> <br><small><?php echo htmlspecialchars($customer['created_by_username'] ?? '', ENT_QUOTES, 'UTF-8'); ?></small></p>
                </div>
                <div>
                    <p class="label">Rôle</p>
                    <p class="value"><?php echo htmlspecialchars($customer['created_by_role'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
                <div>
                    <p class="label">Créé le</p>
                    <p class="value"><?php echo htmlspecialchars(date('d/m/Y H:i', strtotime($customer['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></p>
                </div>
            </div>
            <div class="actions">
                <a href="/masunzu_bar_hotel/modules/customers/history.php" class="btn-secondary">← Retour à l'historique</a>
            </div>
        </div>
    <?php endif; ?>
</div>
<?php include __DIR__ . '/../../includes/footer.php';
