<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

$conn = get_db_connection();
if (!$conn) {
    $error_msg = "Échec de la connexion à la base de données dans cash_transfer_history.php: " . mysqli_connect_error();
    error_log($error_msg);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>$error_msg</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Restrict to DG, DGA, Admin, Board
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
if (!$stmt) {
    throw new Exception("Prepare failed for role check: " . $conn->error);
}
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
$normalizedRole = mbh_normalize_role_name($role);

if (!in_array($normalizedRole, ['DG', 'DGA', 'Admin', 'Board', 'Stock Manager', 'Operation Manager', 'Accountant'], true)) {
    error_log("Unauthorized access to cash_transfer_history.php by user {$_SESSION['full_name']} (Role: $role)");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="historique_transferts_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    fputcsv($output, ['ID', 'Date Transfert', 'Expéditeur', 'Dépôt/Région', 'Destinataire', 'Montant (BIF)', 'Statut', 'Date Réception'], ';');
    
    // Build query for export
    $dateFrom = isset($_GET['date_from']) ? trim($_GET['date_from']) : '';
    $dateTo = isset($_GET['date_to']) ? trim($_GET['date_to']) : '';
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    $status = isset($_GET['status']) ? trim($_GET['status']) : '';
    
    $exportQuery = "
        SELECT 
            ct.id,
            ct.transfer_date,
            cashier.full_name AS cashier_name,
            p_cashier.name AS cashier_province,
            accountant.full_name AS accountant_name,
            ct.amount,
            ct.status,
            ct.received_at
        FROM cash_transfer ct
        INNER JOIN user cashier ON ct.cashier_id = cashier.id
        LEFT JOIN province p_cashier ON cashier.province_id = p_cashier.id
        INNER JOIN user accountant ON ct.accountant_id = accountant.id
        WHERE 1=1
    ";
    
    $params = [];
    $types = '';
    
    if ($dateFrom) {
        $exportQuery .= " AND DATE(ct.transfer_date) >= ?";
        $params[] = $dateFrom;
        $types .= 's';
    }
    if ($dateTo) {
        $exportQuery .= " AND DATE(ct.transfer_date) <= ?";
        $params[] = $dateTo;
        $types .= 's';
    }
    if ($search) {
        $exportQuery .= " AND (cashier.full_name LIKE ? OR accountant.full_name LIKE ? OR p_cashier.name LIKE ?)";
        $searchParam = "%$search%";
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
        $types .= 'sss';
    }
    if ($status) {
        $exportQuery .= " AND ct.status = ?";
        $params[] = $status;
        $types .= 's';
    }
    
    $exportQuery .= " ORDER BY ct.transfer_date DESC";
    
    $stmt = $conn->prepare($exportQuery);
    if ($stmt && !empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $statusFr = match($row['status']) {
                'pending' => 'En Attente',
                'received' => 'Reçu',
                'rejected' => 'Rejeté',
                default => $row['status']
            };
            fputcsv($output, [
                $row['id'],
                $row['transfer_date'],
                $row['cashier_name'],
                $row['cashier_province'] ?? 'N/A',
                $row['accountant_name'],
                number_format((int)$row['amount'], 0, ',', '.'),
                $statusFr,
                $row['received_at'] ?? 'N/A'
            ], ';');
        }
        $stmt->close();
    }
    
    fclose($output);
    exit;
}

// Fetch dashboard statistics
$stats = [
    'total_transfers' => 0,
    'total_amount' => 0,
    'pending_count' => 0,
    'pending_amount' => 0,
    'received_count' => 0,
    'received_amount' => 0,
    'rejected_count' => 0,
    'rejected_amount' => 0
];

$statsQuery = "
    SELECT 
        COUNT(*) AS total_transfers,
        COALESCE(SUM(ct.amount), 0) AS total_amount,
        SUM(CASE WHEN ct.status = 'pending' THEN 1 ELSE 0 END) AS pending_count,
        COALESCE(SUM(CASE WHEN ct.status = 'pending' THEN ct.amount ELSE 0 END), 0) AS pending_amount,
        SUM(CASE WHEN ct.status = 'received' THEN 1 ELSE 0 END) AS received_count,
        COALESCE(SUM(CASE WHEN ct.status = 'received' THEN ct.amount ELSE 0 END), 0) AS received_amount,
        SUM(CASE WHEN ct.status = 'rejected' THEN 1 ELSE 0 END) AS rejected_count,
        COALESCE(SUM(CASE WHEN ct.status = 'rejected' THEN ct.amount ELSE 0 END), 0) AS rejected_amount
    FROM cash_transfer ct
";
$statsResult = $conn->query($statsQuery);
if ($statsResult) {
    $stats = $statsResult->fetch_assoc();
    $statsResult->close();
}

// Fetch transfers with filters
$dateFrom = isset($_GET['date_from']) ? trim($_GET['date_from']) : '';
$dateTo = isset($_GET['date_to']) ? trim($_GET['date_to']) : '';
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$status = isset($_GET['status']) ? trim($_GET['status']) : '';

$query = "
    SELECT 
        ct.id,
        ct.transfer_date,
        ct.amount,
        ct.status,
        ct.received_at,
        cashier.id AS cashier_id,
        cashier.full_name AS cashier_name,
        p_cashier.id AS cashier_province_id,
        p_cashier.name AS cashier_province,
        accountant.id AS accountant_id,
        accountant.full_name AS accountant_name,
        cb.shift_start,
        cb.shift_end
    FROM cash_transfer ct
    INNER JOIN user cashier ON ct.cashier_id = cashier.id
    LEFT JOIN province p_cashier ON cashier.province_id = p_cashier.id
    INNER JOIN user accountant ON ct.accountant_id = accountant.id
    LEFT JOIN cashier_shift_transfer cst ON ct.id = cst.transfer_id
    LEFT JOIN cashier_balance cb ON cst.shift_id = cb.id
    WHERE 1=1
";

$params = [];
$types = '';

if ($dateFrom) {
    $query .= " AND DATE(ct.transfer_date) >= ?";
    $params[] = $dateFrom;
    $types .= 's';
}
if ($dateTo) {
    $query .= " AND DATE(ct.transfer_date) <= ?";
    $params[] = $dateTo;
    $types .= 's';
}
if ($search) {
    $query .= " AND (cashier.full_name LIKE ? OR accountant.full_name LIKE ? OR p_cashier.name LIKE ?)";
    $searchParam = "%$search%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
    $types .= 'sss';
}
if ($status) {
    $query .= " AND ct.status = ?";
    $params[] = $status;
    $types .= 's';
}

$query .= " ORDER BY ct.transfer_date DESC LIMIT 500";

$transfers = [];
$stmt = $conn->prepare($query);
if ($stmt) {
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $transfers[] = $row;
    }
    $stmt->close();
}

include __DIR__ . '/../../includes/header.php';
?>
<style>
    body {
        background-color: #F4F0E4;
        margin: 0;
        padding: 0;
    }
    .transfer-history-container {
        width: 100%;
        padding: 30px 40px;
        min-height: calc(100vh - 100px);
    }
    .page-header {
        margin-bottom: 30px;
    }
    .page-title {
        color: #4B2F1F;
        font-size: 32px;
        font-weight: 700;
        margin: 0 0 8px 0;
    }
    .page-subtitle {
        color: #6B5B52;
        font-size: 16px;
        margin: 0;
    }
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }
    .stat-card {
        background: #FFFFFF;
        border-radius: 12px;
        padding: 24px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        text-align: center;
        transition: transform 0.2s, box-shadow 0.2s;
    }
    .stat-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.12);
    }
    .stat-label {
        color: #6B5B52;
        font-size: 14px;
        font-weight: 500;
        margin-bottom: 8px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .stat-value {
        color: #4B2F1F;
        font-size: 28px;
        font-weight: 700;
        margin: 0;
    }
    .stat-value.pending { color: #F4A261; }
    .stat-value.received { color: #2A9D8F; }
    .stat-value.rejected { color: #DC3545; }
    .filters-bar {
        background: #FFFFFF;
        border-radius: 12px;
        padding: 20px;
        margin-bottom: 30px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        display: flex;
        gap: 15px;
        align-items: center;
        flex-wrap: wrap;
    }
    .filter-input, .filter-select {
        padding: 10px 15px;
        border: 1px solid #D3C2B6;
        border-radius: 8px;
        font-size: 14px;
        color: #4B2F1F;
        background: #F4F0E4;
        transition: border-color 0.2s;
    }
    .filter-input:focus, .filter-select:focus {
        outline: none;
        border-color: #4B2F1F;
        background: #FFFFFF;
    }
    .filter-input {
        flex: 1;
        min-width: 200px;
    }
    .filter-select {
        min-width: 150px;
    }
    .btn-apply, .btn-export {
        padding: 10px 24px;
        background: #4B2F1F;
        color: #F4F0E4;
        border: none;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: background 0.2s;
        text-decoration: none;
        display: inline-block;
    }
    .btn-apply:hover, .btn-export:hover {
        background: #3A2418;
    }
    .btn-export {
        background: #2A9D8F;
    }
    .btn-export:hover {
        background: #21867A;
    }
    .transfers-table-wrapper {
        background: #FFFFFF;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        overflow: hidden;
    }
    .transfers-table {
        width: 100%;
        border-collapse: collapse;
    }
    .transfers-table thead {
        background: #4B2F1F;
        color: #F4F0E4;
    }
    .transfers-table th {
        padding: 16px;
        text-align: left;
        font-weight: 600;
        font-size: 14px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .transfers-table tbody tr {
        border-bottom: 1px solid #F4F0E4;
        transition: background 0.2s;
    }
    .transfers-table tbody tr:hover {
        background: #FDFBF7;
    }
    .transfers-table td {
        padding: 16px;
        color: #4B2F1F;
        font-size: 14px;
    }
    .badge-status {
        padding: 6px 12px;
        border-radius: 8px;
        font-size: 12px;
        font-weight: 600;
    }
    .badge-status.pending {
        background: #FFF3CD;
        color: #856404;
    }
    .badge-status.received {
        background: #D4EDDA;
        color: #155724;
    }
    .badge-status.rejected {
        background: #F8D7DA;
        color: #721C24;
    }
    .empty-state {
        text-align: center;
        padding: 60px 20px;
        color: #6B5B52;
        font-size: 16px;
    }
    @media (max-width: 768px) {
        .transfer-history-container {
            padding: 20px;
        }
        .stats-grid {
            grid-template-columns: 1fr;
        }
        .filters-bar {
            flex-direction: column;
            align-items: stretch;
        }
        .filter-input, .filter-select {
            width: 100%;
        }
        .transfers-table-wrapper {
            overflow-x: auto;
        }
    }
</style>
<div class="transfer-history-container">
    <div class="page-header">
        <h1 class="page-title">Historique des Transferts d'Espèces</h1>
        <p class="page-subtitle">Suivi des transferts de caissiers-comptables vers comptables</p>
    </div>

    <!-- Statistics Dashboard -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-label">Total Transferts</div>
            <div class="stat-value"><?php echo number_format($stats['total_transfers'] ?? 0, 0, ',', '.'); ?></div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Montant Total</div>
            <div class="stat-value"><?php echo number_format($stats['total_amount'] ?? 0, 0, ',', '.'); ?> BIF</div>
        </div>
        <div class="stat-card">
            <div class="stat-label">En Attente</div>
            <div class="stat-value pending"><?php echo number_format($stats['pending_count'] ?? 0, 0, ',', '.'); ?></div>
            <div style="font-size: 12px; color: #6B5B52; margin-top: 4px;">
                <?php echo number_format($stats['pending_amount'] ?? 0, 0, ',', '.'); ?> BIF
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Reçus</div>
            <div class="stat-value received"><?php echo number_format($stats['received_count'] ?? 0, 0, ',', '.'); ?></div>
            <div style="font-size: 12px; color: #6B5B52; margin-top: 4px;">
                <?php echo number_format($stats['received_amount'] ?? 0, 0, ',', '.'); ?> BIF
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-label">Rejetés</div>
            <div class="stat-value rejected"><?php echo number_format($stats['rejected_count'] ?? 0, 0, ',', '.'); ?></div>
            <div style="font-size: 12px; color: #6B5B52; margin-top: 4px;">
                <?php echo number_format($stats['rejected_amount'] ?? 0, 0, ',', '.'); ?> BIF
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="filters-bar">
        <input type="text" id="search" class="filter-input" placeholder="Rechercher par nom, dépôt..." value="<?php echo htmlspecialchars($search, ENT_QUOTES, 'UTF-8'); ?>">
        <input type="date" id="date_from" class="filter-input" placeholder="Date début" value="<?php echo htmlspecialchars($dateFrom, ENT_QUOTES, 'UTF-8'); ?>" style="max-width: 180px;">
        <input type="date" id="date_to" class="filter-input" placeholder="Date fin" value="<?php echo htmlspecialchars($dateTo, ENT_QUOTES, 'UTF-8'); ?>" style="max-width: 180px;">
        <select id="status" class="filter-select">
            <option value="">Tous les Statuts</option>
            <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>En Attente</option>
            <option value="received" <?php echo $status === 'received' ? 'selected' : ''; ?>>Reçu</option>
            <option value="rejected" <?php echo $status === 'rejected' ? 'selected' : ''; ?>>Rejeté</option>
        </select>
        <button onclick="applyFilters()" class="btn-apply">Appliquer Filtres</button>
        <button onclick="resetFilters()" class="btn-export" style="background: #6B5B52;">🔄 Réinitialiser</button>
        <a href="?" onclick="return exportCSV()" class="btn-export">📥 Exporter CSV</a>
    </div>

    <!-- Transfers Table -->
    <?php if (empty($transfers)): ?>
        <div class="empty-state">
            <p>Aucun transfert trouvé avec les filtres actuels.</p>
        </div>
    <?php else: ?>
        <div class="transfers-table-wrapper">
            <table class="transfers-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Date Transfert</th>
                        <th>Expéditeur</th>
                        <th>Dépôt/Région</th>
                        <th>Destinataire</th>
                        <th>Montant</th>
                        <th>Statut</th>
                        <th>Date Réception</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($transfers as $transfer): ?>
                        <tr>
                            <td><strong>#<?php echo htmlspecialchars($transfer['id'], ENT_QUOTES, 'UTF-8'); ?></strong></td>
                            <td><?php echo htmlspecialchars($transfer['transfer_date'] ? date('d/m/Y H:i', strtotime($transfer['transfer_date'])) : 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($transfer['cashier_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($transfer['cashier_province'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($transfer['accountant_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><strong><?php echo number_format((int)$transfer['amount'], 0, ',', '.'); ?> BIF</strong></td>
                            <td>
                                <span class="badge-status <?php echo htmlspecialchars($transfer['status'] ?? 'pending', ENT_QUOTES, 'UTF-8'); ?>">
                                    <?php 
                                        $statusFr = match($transfer['status']) {
                                            'pending' => 'En Attente',
                                            'received' => 'Reçu',
                                            'rejected' => 'Rejeté',
                                            default => $transfer['status']
                                        };
                                        echo htmlspecialchars($statusFr, ENT_QUOTES, 'UTF-8'); 
                                    ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($transfer['received_at'] ? date('d/m/Y H:i', strtotime($transfer['received_at'])) : '—', ENT_QUOTES, 'UTF-8'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<script>
    function applyFilters() {
        const search = document.getElementById('search').value;
        const dateFrom = document.getElementById('date_from').value;
        const dateTo = document.getElementById('date_to').value;
        const status = document.getElementById('status').value;
        const url = new URL(window.location);
        if (search) url.searchParams.set('search', search);
        else url.searchParams.delete('search');
        if (dateFrom) url.searchParams.set('date_from', dateFrom);
        else url.searchParams.delete('date_from');
        if (dateTo) url.searchParams.set('date_to', dateTo);
        else url.searchParams.delete('date_to');
        if (status) url.searchParams.set('status', status);
        else url.searchParams.delete('status');
        window.location = url.toString();
    }

    function resetFilters() {
        // Clear all filter inputs
        document.getElementById('search').value = '';
        document.getElementById('date_from').value = '';
        document.getElementById('date_to').value = '';
        document.getElementById('status').value = '';
        // Reload page without any filter parameters
        window.location = window.location.pathname;
    }

    function exportCSV() {
        const search = document.getElementById('search').value;
        const dateFrom = document.getElementById('date_from').value;
        const dateTo = document.getElementById('date_to').value;
        const status = document.getElementById('status').value;
        const url = new URL(window.location);
        url.searchParams.set('export', 'csv');
        if (search) url.searchParams.set('search', search);
        if (dateFrom) url.searchParams.set('date_from', dateFrom);
        if (dateTo) url.searchParams.set('date_to', dateTo);
        if (status) url.searchParams.set('status', status);
        window.location = url.toString();
        return false;
    }

    // Auto-populate filters from URL
    document.addEventListener('DOMContentLoaded', function() {
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.has('search')) {
            document.getElementById('search').value = urlParams.get('search') || '';
        }
        if (urlParams.has('date_from')) {
            document.getElementById('date_from').value = urlParams.get('date_from') || '';
        }
        if (urlParams.has('date_to')) {
            document.getElementById('date_to').value = urlParams.get('date_to') || '';
        }
        if (urlParams.has('status')) {
            document.getElementById('status').value = urlParams.get('status') || '';
        }
    });
</script>
<?php
$conn->close();
include __DIR__ . '/../../includes/footer.php';
?>

