<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

try {
    require_login();

    // Restrict to Accountant role
    if ($_SESSION['role_name'] !== 'Accountant') {
        error_log("Unauthorized access to accountant_dashboard.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
        header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $_SESSION['role_name'])) . "_dashboard.php");
        exit;
    }

    // Fetch province_id if not set
    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception("Échec de la connexion à la base de données: " . mysqli_connect_error());
    }

    if (!isset($_SESSION['province_id'])) {
        $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if (!$stmt) {
            throw new Exception("Échec de la préparation de la requête pour province_id: " . $conn->error);
        }
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        $_SESSION['province_id'] = $user['province_id'] ?? null;
        $stmt->close();
    }
    $province_id = $_SESSION['province_id'] ?? null;

    if ($province_id) {
        // Fetch province name
        $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
        if (!$stmt) {
            throw new Exception("Échec de la préparation de la requête pour le nom de la province: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $province = $stmt->get_result()->fetch_assoc()['name'] ?? 'Unknown';
        $stmt->close();
    } else {
        $province = 'Toutes les provinces';
    }

    // Fetch total received cash from cash_transfer
    $stmt = $conn->prepare("SELECT SUM(amount) as total_received FROM cash_transfer WHERE accountant_id = ? AND status = 'received'");
    if (!$stmt) {
        throw new Exception("Échec de la préparation de la requête pour le total reçu: " . $conn->error);
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $total_received = $stmt->get_result()->fetch_assoc()['total_received'] ?? 0.00;
    $stmt->close();

    // Fetch total deposits from bank_deposit
    $stmt = $conn->prepare("SELECT SUM(amount) as total_deposits FROM bank_deposit WHERE user_id = ?");
    if (!$stmt) {
        throw new Exception("Échec de la préparation de la requête pour le total des dépôts: " . $conn->error);
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $total_deposits = $stmt->get_result()->fetch_assoc()['total_deposits'] ?? 0.00;
    $stmt->close();

    // Fetch total expenses from expense
    $stmt = $conn->prepare("SELECT SUM(amount) as total_expenses FROM expense WHERE accountant_id = ?");
    if (!$stmt) {
        throw new Exception("Échec de la préparation de la requête pour le total des dépenses: " . $conn->error);
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $total_expenses = $stmt->get_result()->fetch_assoc()['total_expenses'] ?? 0.00;
    $stmt->close();

    // Fetch recent transactions from cash_transfer, bank_deposit, and expense
    $stmt = $conn->prepare("
        SELECT 'Received' as transaction_type, amount, transfer_date as created_at, u.full_name as user_name
        FROM cash_transfer ct
        LEFT JOIN user u ON ct.cashier_id = u.id
        WHERE ct.accountant_id = ? AND ct.status = 'received'
        UNION
        SELECT 'Deposit' as transaction_type, amount, deposit_date as created_at, 'Self' as user_name
        FROM bank_deposit bd
        WHERE bd.user_id = ?
        UNION
        SELECT 'Expense' as transaction_type, amount, created_at, 'Self' as user_name
        FROM expense e
        WHERE e.accountant_id = ?
        ORDER BY created_at DESC
        LIMIT 5
    ");
    if (!$stmt) {
        throw new Exception("Échec de la préparation de la requête pour les transactions récentes: " . $conn->error);
    }
    $stmt->bind_param("iii", $_SESSION['user_id'], $_SESSION['user_id'], $_SESSION['user_id']);
    $stmt->execute();
    $recent_transactions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    $conn->close();

} catch (Exception $e) {
    error_log($e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: " . htmlspecialchars($e->getMessage()) . "</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

include __DIR__ . '/../../includes/header.php';
?>
<div class="dashboard-container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 70vh;">
    <div class="dashboard-content" style="background-color: #F4F0E4; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 800px; margin: 20px auto;">
        <h2 style="color: #4B2F1F; margin-bottom: 10px;">Tableau de Bord Comptable</h2>
        <p style="color: #000000; font-size: 16px;">Bienvenue, <?php echo htmlspecialchars($_SESSION['full_name'], ENT_QUOTES, 'UTF-8'); ?> (Comptable, <?php echo htmlspecialchars($province, ENT_QUOTES, 'UTF-8'); ?>)</p>
        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Métriques Actuelles</h3>
        <p style="color: #000000; font-size: 16px;">
            Total Espèces Reçues: <?php echo number_format($total_received, 2, ',', '.') . ' BIF'; ?><br>
            Total Dépôts: <?php echo number_format($total_deposits, 2, ',', '.') . ' BIF'; ?><br>
            Total Dépenses: <?php echo number_format($total_expenses, 2, ',', '.') . ' BIF'; ?>
        </p>
        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Transactions Récentes</h3>
        <?php if (!empty($recent_transactions)): ?>
            <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Date</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Utilisateur</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Montant</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Type</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recent_transactions as $transaction): ?>
                        <tr style="background-color: #FFFFFF;">
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($transaction['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($transaction['user_name'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo number_format($transaction['amount'] ?? 0, 2, ',', '.') . ' BIF'; ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($transaction['transaction_type'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p style="color: #000000; font-size: 16px;">Aucune transaction récente trouvée.</p>
        <?php endif; ?>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
