<?php

require_once __DIR__ . '/../config/database.php';

if (!defined('INVOICE_MGMT_LOADED')) {
    define('INVOICE_MGMT_LOADED', true);

    /**
     * Initialize invoice expiration system columns if they don't exist
     * This function checks and adds necessary columns to track invoice expiration
     */
    function initialize_invoice_expiration_columns() {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $check_created = $conn->query("SHOW COLUMNS FROM `paid_invoices` LIKE 'expires_at'");
            if (!$check_created || $check_created->num_rows == 0) {
                $conn->query("ALTER TABLE `paid_invoices` ADD COLUMN `expires_at` DATETIME DEFAULT NULL AFTER `created_at`");
                error_log("Added expires_at column to paid_invoices table");
            }

            $check_status = $conn->query("SHOW COLUMNS FROM `paid_invoices` LIKE 'expiration_status'");
            if (!$check_status || $check_status->num_rows == 0) {
                $conn->query("ALTER TABLE `paid_invoices` ADD COLUMN `expiration_status` ENUM('active','expired','approved') NOT NULL DEFAULT 'active' AFTER `approval_status`");
                error_log("Added expiration_status column to paid_invoices table");
            }

            $check_invoice = $conn->query("SHOW COLUMNS FROM `invoice` LIKE 'expires_at'");
            if (!$check_invoice || $check_invoice->num_rows == 0) {
                $conn->query("ALTER TABLE `invoice` ADD COLUMN `expires_at` DATETIME DEFAULT NULL AFTER `created_at`");
                error_log("Added expires_at column to invoice table");
            }

            $check_invoice_status = $conn->query("SHOW COLUMNS FROM `invoice` LIKE 'expiration_status'");
            if (!$check_invoice_status || $check_invoice_status->num_rows == 0) {
                $conn->query("ALTER TABLE `invoice` ADD COLUMN `expiration_status` ENUM('active','expired','approved') NOT NULL DEFAULT 'active' AFTER `approval_status`");
                error_log("Added expiration_status column to invoice table");
            }

            $check_reserved = $conn->query("SHOW COLUMNS FROM `main_stock` LIKE 'reserved_quantity'");
            if (!$check_reserved || $check_reserved->num_rows == 0) {
                $conn->query("ALTER TABLE `main_stock` ADD COLUMN `reserved_quantity` INT(11) NOT NULL DEFAULT 0 AFTER `quantity`");
                error_log("Added reserved_quantity column to main_stock table");
            }

            $conn->close();
            return true;
        } catch (Exception $e) {
            error_log("Error initializing invoice expiration columns: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Create invoice with automatic expiration in 3 hours and stock reservation
     * 
     * @param array $invoice_data Invoice data to insert
     * @param array $items Invoice items with quantities
     * @return int|bool Invoice ID on success, false on failure
     */
    function create_invoice_with_expiration($invoice_data, $items) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $conn->begin_transaction();

            $expires_at = date('Y-m-d H:i:s', strtotime('+3 hours'));

            $stmt = $conn->prepare("
                INSERT INTO paid_invoices (
                    stock_manager_id, status, customer_name, phone_number, 
                    province_id, nif, rc, invoice_number, serve_status, paid_amount,
                    subtotal_amount, tax_amount, tax_rate, tax_mode, approval_status,
                    created_by, payment_details, sale_type,
                    is_loan_sale, loan_amount_bif, customer_id, expires_at, expiration_status
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            if (!$stmt) throw new Exception("Prepare failed: " . $conn->error);

            $stmt->bind_param(
                "issssissisiiisiisssiisi",
                $invoice_data['stock_manager_id'],
                $invoice_data['status'] ?? 'pending',
                $invoice_data['customer_name'],
                $invoice_data['phone_number'],
                $invoice_data['province_id'],
                $invoice_data['nif'],
                $invoice_data['rc'],
                $invoice_data['invoice_number'],
                $invoice_data['serve_status'] ?? 'Pending',
                $invoice_data['paid_amount'],
                $invoice_data['subtotal_amount'] ?? 0,
                $invoice_data['tax_amount'] ?? 0,
                $invoice_data['tax_rate'] ?? 18.00,
                $invoice_data['tax_mode'] ?? 'HTVA',
                $invoice_data['approval_status'] ?? 'pending',
                $invoice_data['created_by'],
                $invoice_data['payment_details'],
                $invoice_data['sale_type'] ?? 'retail',
                $invoice_data['is_loan_sale'] ?? 0,
                $invoice_data['loan_amount_bif'] ?? 0,
                $invoice_data['customer_id'],
                $expires_at,
                'active'
            );

            if (!$stmt->execute()) throw new Exception("Execute failed: " . $stmt->error);

            $invoice_id = $conn->insert_id;
            $stmt->close();

            reserve_stock_for_invoice($invoice_id, $items, $invoice_data['province_id']);

            $conn->commit();
            error_log("Invoice #{$invoice_id} created with expiration at {$expires_at}");

            return $invoice_id;
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Error creating invoice with expiration: " . $e->getMessage());
            return false;
        } finally {
            $conn->close();
        }
    }

    /**
     * Reserve stock for a pending invoice
     * 
     * @param int $invoice_id Invoice ID
     * @param array $items Items to reserve with quantities
     * @param int $province_id Province ID
     * @return bool
     */
    function reserve_stock_for_invoice($invoice_id, $items, $province_id) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            foreach ($items as $item) {
                $product_id = $item['product_id'] ?? null;
                $quantity = $item['quantity'] ?? 0;

                if (!$product_id || $quantity <= 0) continue;

                $update_stmt = $conn->prepare("
                    UPDATE main_stock 
                    SET reserved_quantity = reserved_quantity + ?, 
                        last_updated = NOW()
                    WHERE product_id = ?
                ");

                if (!$update_stmt) {
                    error_log("Failed to prepare stock reservation: " . $conn->error);
                    continue;
                }

                $update_stmt->bind_param("ii", $quantity, $product_id);
                if (!$update_stmt->execute()) {
                    error_log("Failed to reserve stock for product {$product_id}: " . $update_stmt->error);
                }
                $update_stmt->close();

                error_log("Reserved {$quantity} units of product {$product_id} for invoice {$invoice_id}");
            }

            $conn->close();
            return true;
        } catch (Exception $e) {
            error_log("Error reserving stock: " . $e->getMessage());
            $conn->close();
            return false;
        }
    }

    /**
     * Check for expired invoices and revert stock
     * Should be called periodically (e.g., every few minutes via cron or strategically)
     * 
     * @return int Number of invoices expired
     */
    function check_and_expire_invoices() {
        $conn = get_db_connection();
        if (!$conn) return 0;

        try {
            $expired_count = 0;

            $stmt = $conn->prepare("
                SELECT id, created_by, province_id 
                FROM paid_invoices 
                WHERE expiration_status = 'active' 
                AND approval_status = 'pending'
                AND expires_at < NOW()
                AND expires_at IS NOT NULL
                LIMIT 100
            ");

            if (!$stmt) {
                error_log("Failed to prepare expiration check: " . $conn->error);
                $conn->close();
                return 0;
            }

            $stmt->execute();
            $result = $stmt->get_result();

            while ($invoice = $result->fetch_assoc()) {
                if (expire_invoice($invoice['id'], $invoice['created_by'], $invoice['province_id'])) {
                    $expired_count++;
                }
            }

            $stmt->close();
            $conn->close();

            if ($expired_count > 0) {
                error_log("Expired {$expired_count} invoices due to approval timeout");
            }

            return $expired_count;
        } catch (Exception $e) {
            error_log("Error checking for expired invoices: " . $e->getMessage());
            $conn->close();
            return 0;
        }
    }

    /**
     * Expire a single invoice and revert its stock reservation
     * 
     * @param int $invoice_id Invoice ID to expire
     * @param int $created_by User ID who created the invoice
     * @param int $province_id Province ID
     * @return bool
     */
    function expire_invoice($invoice_id, $created_by, $province_id) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $conn->begin_transaction();

            $item_stmt = $conn->prepare("
                SELECT product_id, quantity 
                FROM paid_invoice_items 
                WHERE invoice_id = ?
            ");

            if (!$item_stmt) throw new Exception("Failed to prepare items query: " . $conn->error);

            $item_stmt->bind_param("i", $invoice_id);
            $item_stmt->execute();
            $items_result = $item_stmt->get_result();

            while ($item = $items_result->fetch_assoc()) {
                revert_stock_reservation($item['product_id'], $item['quantity']);
            }

            $item_stmt->close();

            $update_stmt = $conn->prepare("
                UPDATE paid_invoices 
                SET expiration_status = 'expired', status = 'expired'
                WHERE id = ?
            ");

            if (!$update_stmt) throw new Exception("Failed to prepare update: " . $conn->error);

            $update_stmt->bind_param("i", $invoice_id);
            if (!$update_stmt->execute()) {
                throw new Exception("Failed to update invoice status: " . $update_stmt->error);
            }

            $update_stmt->close();

            send_expiration_notification($invoice_id, $created_by, $province_id);

            $conn->commit();
            error_log("Invoice #{$invoice_id} expired successfully");

            return true;
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Error expiring invoice: " . $e->getMessage());
            return false;
        } finally {
            $conn->close();
        }
    }

    /**
     * Revert stock reservation for expired invoice
     * 
     * @param int $product_id Product ID
     * @param int $quantity Quantity to revert
     * @return bool
     */
    function revert_stock_reservation($product_id, $quantity) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $stmt = $conn->prepare("
                UPDATE main_stock 
                SET reserved_quantity = GREATEST(0, reserved_quantity - ?),
                    last_updated = NOW()
                WHERE product_id = ?
            ");

            if (!$stmt) {
                error_log("Failed to prepare stock revert: " . $conn->error);
                return false;
            }

            $stmt->bind_param("ii", $quantity, $product_id);
            if (!$stmt->execute()) {
                error_log("Failed to revert stock for product {$product_id}: " . $stmt->error);
                $stmt->close();
                return false;
            }

            $stmt->close();
            error_log("Reverted {$quantity} units of product {$product_id}");

            $conn->close();
            return true;
        } catch (Exception $e) {
            error_log("Error reverting stock: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to admin and invoice creator about expiration
     * 
     * @param int $invoice_id Invoice ID
     * @param int $created_by User ID who created invoice
     * @param int $province_id Province ID
     * @return bool
     */
    function send_expiration_notification($invoice_id, $created_by, $province_id) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $invoice = $conn->query("SELECT invoice_number FROM paid_invoices WHERE id = {$invoice_id}")->fetch_assoc();
            $invoice_num = $invoice['invoice_number'] ?? 'Unknown';

            $admin_ids = $conn->query("
                SELECT DISTINCT u.id FROM user u 
                JOIN role r ON u.role_id = r.id
                WHERE r.name IN ('Admin', 'DG', 'DGA')
            ")->fetch_all(MYSQLI_ASSOC);

            $notification_data = [
                'type' => 'invoice_expired',
                'title' => 'Invoice Expired',
                'message' => "Invoice #{$invoice_num} has expired due to approval timeout (3 hours)",
                'invoice_id' => $invoice_id,
                'province_id' => $province_id,
                'created_at' => date('Y-m-d H:i:s')
            ];

            $insert_stmt = $conn->prepare("
                INSERT INTO notifications (user_id, type, title, message, reference_id, is_read, created_at) 
                VALUES (?, ?, ?, ?, ?, 0, NOW())
            ");

            if ($insert_stmt) {
                foreach ($admin_ids as $admin) {
                    $insert_stmt->bind_param(
                        "isssi",
                        $admin['id'],
                        $notification_data['type'],
                        $notification_data['title'],
                        $notification_data['message'],
                        $invoice_id
                    );
                    $insert_stmt->execute();
                }

                $insert_stmt->bind_param(
                    "isssi",
                    $created_by,
                    $notification_data['type'],
                    $notification_data['title'],
                    $notification_data['message'],
                    $invoice_id
                );
                $insert_stmt->execute();
                $insert_stmt->close();

                error_log("Expiration notifications sent for invoice #{$invoice_id}");
            }

            $conn->close();
            return true;
        } catch (Exception $e) {
            error_log("Error sending expiration notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Approve invoice and finalize stock deduction
     * 
     * @param int $invoice_id Invoice ID to approve
     * @param int $approved_by User ID approving the invoice
     * @return bool
     */
    function approve_invoice($invoice_id, $approved_by) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $conn->begin_transaction();

            $approve_stmt = $conn->prepare("
                UPDATE paid_invoices 
                SET approval_status = 'approved', 
                    approved_by = ?,
                    approved_at = NOW(),
                    expiration_status = 'approved'
                WHERE id = ?
            ");

            if (!$approve_stmt) throw new Exception("Prepare failed: " . $conn->error);

            $approve_stmt->bind_param("ii", $approved_by, $invoice_id);
            if (!$approve_stmt->execute()) {
                throw new Exception("Approval update failed: " . $approve_stmt->error);
            }
            $approve_stmt->close();

            finalize_stock_deduction($invoice_id);

            $conn->commit();
            error_log("Invoice #{$invoice_id} approved and stock finalized");

            return true;
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Error approving invoice: " . $e->getMessage());
            return false;
        } finally {
            $conn->close();
        }
    }

    /**
     * Finalize stock deduction (move from reserved to actual deduction)
     * 
     * @param int $invoice_id Invoice ID
     * @return bool
     */
    function finalize_stock_deduction($invoice_id) {
        $conn = get_db_connection();
        if (!$conn) return false;

        try {
            $items_stmt = $conn->prepare("
                SELECT product_id, quantity 
                FROM paid_invoice_items 
                WHERE invoice_id = ?
            ");

            if (!$items_stmt) return false;

            $items_stmt->bind_param("i", $invoice_id);
            $items_stmt->execute();
            $items = $items_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $items_stmt->close();

            foreach ($items as $item) {
                $update_stmt = $conn->prepare("
                    UPDATE main_stock 
                    SET quantity = GREATEST(0, quantity - ?),
                        reserved_quantity = GREATEST(0, reserved_quantity - ?),
                        last_updated = NOW()
                    WHERE product_id = ?
                ");

                if (!$update_stmt) continue;

                $qty = $item['quantity'];
                $update_stmt->bind_param("iii", $qty, $qty, $item['product_id']);
                $update_stmt->execute();
                $update_stmt->close();
            }

            $conn->close();
            return true;
        } catch (Exception $e) {
            error_log("Error finalizing stock deduction: " . $e->getMessage());
            return false;
        }
    }

    initialize_invoice_expiration_columns();
}
