<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Stock Manager role
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in stock_manager_dashboard.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
if ($role !== 'Stock Manager') {
    error_log("Unauthorized access to stock_manager_dashboard.php by user {$_SESSION['full_name']} (Role: $role)");
    // Use dashboard mapping to redirect correctly
    $dashboard_map = [
        'Board' => 'admin_dashboard.php',
        'DG' => 'admin_dashboard.php',
        'DGA' => 'admin_dashboard.php',
        'Admin' => 'admin_dashboard.php',
        'Procurement Manager' => 'procurement_dashboard.php',
        'Operation Manager' => 'operation_dashboard.php',
        'Finance Head' => 'finance_dashboard.php',
        'Marketing Head' => 'marketing_dashboard.php',
        'Accountant' => 'accountant_dashboard.php',
        'Marketing Agent' => 'marketing_dashboard.php',
        'Cashier' => 'cashier_dashboard.php',
        'Caissier_Comptable' => 'cashier_dashboard.php',
        'Stock Manager Assistant' => 'stock_manager_assistant_dashboard.php'
    ];
    $dashboard = $dashboard_map[$role] ?? 'unauthorized.php';
    header("Location: /masunzu_bar_hotel/dashboards/" . $dashboard);
    exit;
}

// Fetch province_id
$province_id = $_SESSION['province_id'] ?? null;
if ($province_id === null || $province_id === '') {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $province_id = $stmt->get_result()->fetch_assoc()['province_id'] ?? null;
    $_SESSION['province_id'] = $province_id;
    $stmt->close();
}
$province_id = $_SESSION['province_id'] ?? null;
if ($province_id === null || $province_id === '') {
    error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur : aucune province assignée. Contactez l'administrateur.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$province_id = (int)$province_id;

// Debug: Log session variables
error_log("Stock Manager Dashboard - User ID: {$_SESSION['user_id']}, Province ID: " . ($province_id ?? 'NULL') . ", Role: $role");

// Fetch province name
$stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
$stmt->bind_param("i", $province_id);
$stmt->execute();
$province = $stmt->get_result()->fetch_assoc()['name'] ?? 'Unknown';
$stmt->close();


// Ensure notification table exists
$conn->query("
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
");

$hideFormSubmitted = $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['hide_notifications']);
if ($hideFormSubmitted) {
    $hideStmt = $conn->prepare("UPDATE user_notification SET is_read = 1 WHERE user_id = ?");
    if ($hideStmt) {
        $hideStmt->bind_param("i", $_SESSION['user_id']);
        $hideStmt->execute();
        $hideStmt->close();
    }
    header("Location: /masunzu_bar_hotel/dashboards/stock_manager_dashboard.php");
    exit;
}

// Fetch user notifications (shipments etc.)
$shipmentNotifications = [];
$notificationIds = [];
$notifStmt = $conn->prepare("SELECT id, message, link, created_at FROM user_notification WHERE user_id = ? AND is_read = 0 ORDER BY created_at DESC LIMIT 5");
if ($notifStmt) {
    $notifStmt->bind_param("i", $_SESSION['user_id']);
    $notifStmt->execute();
    $shipmentNotifications = $notifStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $notifStmt->close();
    if (!empty($shipmentNotifications)) {
        $notificationIds = array_column($shipmentNotifications, 'id');
    }
}
if (!empty($notificationIds)) {
    $ids = implode(',', array_map('intval', $notificationIds));
    $conn->query("UPDATE user_notification SET is_read = 1 WHERE id IN ($ids)");
}

// Fetch pending stock shipments
$stmt = $conn->prepare("
    SELECT COUNT(DISTINCT shipment_id) AS pending_count
    FROM stock_transfer
    WHERE (stock_manager_id = ? OR (stock_manager_id IS NULL AND province_id = ?))
    AND status = 'pending'
");
$stmt->bind_param("ii", $_SESSION['user_id'], $province_id);
$stmt->execute();
$pending_count = $stmt->get_result()->fetch_assoc()['pending_count'] ?? 0;
$stmt->close();
error_log("Pending shipments count: $pending_count for user {$_SESSION['user_id']}, province $province_id");

// Fetch unassigned shipments
$stmt = $conn->prepare("
    SELECT COUNT(DISTINCT shipment_id) AS unassigned_count
    FROM stock_transfer
    WHERE stock_manager_id IS NULL AND province_id = ? AND status = 'pending'
");
$stmt->bind_param("i", $province_id);
$stmt->execute();
$unassigned_count = $stmt->get_result()->fetch_assoc()['unassigned_count'] ?? 0;
$stmt->close();
error_log("Unassigned shipments count: $unassigned_count for province $province_id");

// Fetch province stock metrics for main content
$stmt = $conn->prepare("
    SELECT 
        p.name AS product_name, 
        ps.quantity, 
        ps.reserved_quantity,
        ps.unit_type,
        p.expiration_date, 
        p.price_per_crate AS unit_price, 
        p.crate_quantity 
    FROM province_stock ps
    JOIN product p ON ps.product_id = p.id
    WHERE ps.province_id = ? AND ps.quantity > 0
    ORDER BY p.name
");
$stmt->bind_param("i", $province_id);
$stmt->execute();
$stocks = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();
error_log("Stock count: " . count($stocks) . " for province $province_id");

// Calculate total available value and track low stock alerts
$total_value = 0;
$total_available_crates = 0;
$total_reserved_crates = 0;
$low_stock = [];
foreach ($stocks as &$stock) {
    $quantity_crates = (int)($stock['quantity'] ?? 0);
    $reserved_crates = max(0, (int)($stock['reserved_quantity'] ?? 0));
    $available_crates = max(0, $quantity_crates - $reserved_crates);
    $price_per_crate = (float)($stock['unit_price'] ?? 0);
    $available_value = $available_crates * $price_per_crate;

    $stock['reserved_quantity'] = $reserved_crates;
    $stock['available_quantity'] = $available_crates;
    $stock['available_value'] = $available_value;
    $stock['price_per_crate'] = $price_per_crate;

    $total_value += $available_value;
    $total_available_crates += $available_crates;
    $total_reserved_crates += $reserved_crates;

    if ($available_crates < 10) {
        $low_stock[] = $stock;
    }
}
unset($stock);

// Fetch recent receptions
$stmt = $conn->prepare("
    SELECT t.id, t.sent_at, t.received_at, p.name AS product_name, t.quantity, p.crate_quantity 
    FROM stock_transfer t 
    JOIN product p ON t.product_id = p.id 
    WHERE t.stock_manager_id = ? AND t.status = 'confirmed' 
    ORDER BY t.received_at DESC LIMIT 5
");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$recent_receptions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch sold products from paid_invoices only
$stmt = $conn->prepare("
    SELECT p.name AS product_name, SUM(ii.quantity) AS sold_quantity, p.crate_quantity
    FROM paid_invoices pi
    JOIN invoice_items ii ON pi.id = ii.invoice_id
    JOIN product p ON ii.product_id = p.id 
    WHERE pi.province_id = ? AND pi.status = 'paid'
    GROUP BY p.id
");
$stmt->bind_param("i", $province_id);
$stmt->execute();
$sold_products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch recent paid invoices
$stmt = $conn->prepare("
    SELECT pi.id, pi.created_at, u.full_name AS cashier_name, SUM(ii.quantity) AS total_quantity, pi.paid_amount
    FROM paid_invoices pi
    JOIN user u ON pi.created_by = u.id
    JOIN paid_invoice_items ii ON pi.id = ii.invoice_id
    WHERE pi.province_id = ?
    GROUP BY pi.id
    ORDER BY pi.created_at DESC LIMIT 10
");
$stmt->bind_param("i", $province_id);
$stmt->execute();
$paid_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Pending invoice approvals in this province
$pending_invoice_approvals = [];
    $stmt = $conn->prepare("
    SELECT ui.id, ui.invoice_number, ui.customer_name, ui.paid_amount, ui.created_at, u.full_name AS initiator_name
    FROM unpaid_invoices ui
    LEFT JOIN user u ON ui.created_by = u.id
    WHERE ui.province_id = ? AND ui.approval_status = 'pending' AND COALESCE(ui.is_loan_sale, 0) = 0
    ORDER BY ui.created_at DESC
    LIMIT 6
");
$stmt->bind_param("i", $province_id);
$stmt->execute();
$pending_invoice_approvals = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch requisition notifications (approved or rejected requisitions created by this user)
// Show requisitions that have been approved or rejected in the last 30 days
$stmt = $conn->prepare("
    SELECT r.id, r.total_amount, r.status, r.comments, r.created_at, r.approved_by,
           u.full_name AS approved_by_name,
           r.recipient_type, r.recipient_external_name, r.recipient_external_phone,
           u2.full_name AS recipient_name
    FROM requisition r
    LEFT JOIN user u ON r.approved_by = u.id
    LEFT JOIN user u2 ON r.recipient_id = u2.id
    WHERE r.user_id = ? 
    AND r.status IN ('approved', 'rejected')
    AND r.approved_by IS NOT NULL
    AND r.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    ORDER BY r.created_at DESC
    LIMIT 10
");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$requisition_notifications = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<style>
    @keyframes blink {
        0%, 100% { opacity: 1; }
        50% { opacity: 0; }
    }
    .pending-invoice-row {
        background: linear-gradient(90deg, #FFF7EE, #FFE082);
        border-left: 4px solid #FF9800;
        transition: transform 0.3s ease;
    }
    .pending-invoice-row:hover {
        transform: translateX(2px);
    }
    .pending-invoice-row .blink-text {
        animation: blink 1.2s ease-in-out infinite;
        font-weight: 700;
    }
</style>
<div class="container" style="padding: 0; margin: 0; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: calc(100vh - 140px); width: 100%; max-width: none;">
    <!-- Main Content -->
    <div class="content" style="width: 100%; max-width: none; background-color: #F4F0E4; padding: 25px; border-radius: 0; box-shadow: none; overflow-y: auto; min-height: calc(100vh - 140px); box-sizing: border-box;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px; font-weight: 600; text-align: center;">Tableau de bord Gestionnaire de Stock</h2>
        <p style="color: #4B2F1F; font-size: 16px; text-align: center;">Bienvenue, <?php echo htmlspecialchars($_SESSION['full_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?> (Gestionnaire, <?php echo htmlspecialchars($province ?? 'Inconnue', ENT_QUOTES, 'UTF-8'); ?>)</p>
        
        <!-- SECTION 1: REQUISITION NOTIFICATIONS -->
        <?php if (!empty($requisition_notifications)): ?>
            <div style="background-color: #FFFFFF; border-radius: 8px; padding: 20px; margin-top: 20px; margin-bottom: 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                <h3 style="color: #4B2F1F; font-size: 20px; font-weight: 600; margin-bottom: 15px; display: flex; align-items: center; gap: 10px;">
                    <span style="background-color: #F4A261; color: #4B2F1F; padding: 4px 10px; border-radius: 50%; font-size: 14px; font-weight: bold;"><?php echo count($requisition_notifications); ?></span>
                    Notifications de réquisitions
                </h3>
                <div style="display: flex; flex-direction: column; gap: 12px;">
                    <?php foreach ($requisition_notifications as $notification): ?>
                        <?php
                            $is_approved = $notification['status'] === 'approved';
                            $bg_color = $is_approved ? '#D4EDDA' : '#F8D7DA';
                            $border_color = $is_approved ? '#28A745' : '#DC3545';
                            $text_color = $is_approved ? '#155724' : '#721C24';
                            $icon = $is_approved ? '✓' : '✗';
                            $status_text = $is_approved ? 'Approuvée' : 'Rejetée';
                            
                            // Format recipient name
                            $recipient_display = '';
                            if ($notification['recipient_type'] === 'external') {
                                $recipient_display = htmlspecialchars($notification['recipient_external_name'] ?? 'Bénéficiaire externe', ENT_QUOTES, 'UTF-8');
                                if (!empty($notification['recipient_external_phone'])) {
                                    $recipient_display .= ' (' . htmlspecialchars($notification['recipient_external_phone'], ENT_QUOTES, 'UTF-8') . ')';
                                }
                            } else {
                                $recipient_display = htmlspecialchars($notification['recipient_name'] ?? 'Bénéficiaire interne', ENT_QUOTES, 'UTF-8');
                            }
                            
                            // Calculate time ago
                            $created_time = strtotime($notification['created_at']);
                            $time_ago = '';
                            $diff = time() - $created_time;
                            if ($diff < 3600) {
                                $time_ago = 'Il y a ' . floor($diff / 60) . ' minute(s)';
                            } elseif ($diff < 86400) {
                                $time_ago = 'Il y a ' . floor($diff / 3600) . ' heure(s)';
                            } else {
                                $time_ago = 'Il y a ' . floor($diff / 86400) . ' jour(s)';
                            }
                        ?>
                        <div style="background-color: <?php echo $bg_color; ?>; border-left: 4px solid <?php echo $border_color; ?>; padding: 15px; border-radius: 6px; position: relative;">
                            <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 8px;">
                                <div style="flex: 1;">
                                    <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 6px;">
                                        <span style="background-color: <?php echo $border_color; ?>; color: #FFFFFF; padding: 4px 8px; border-radius: 4px; font-weight: bold; font-size: 14px;">
                                            <?php echo $icon; ?> <?php echo $status_text; ?>
                                        </span>
                                        <span style="color: #4B2F1F; font-weight: 600; font-size: 16px;">
                                            Réquisition n°<?php echo $notification['id']; ?>
                                        </span>
                                    </div>
                                    <div style="color: #4B2F1F; font-size: 14px; margin-bottom: 4px;">
                                        <strong>Montant :</strong> <?php echo number_format($notification['total_amount'], 0, ',', '.'); ?> BIF
                                    </div>
                                    <div style="color: #4B2F1F; font-size: 14px; margin-bottom: 4px;">
                                        <strong>Bénéficiaire :</strong> <?php echo $recipient_display; ?>
                                    </div>
                                    <?php if ($notification['approved_by_name']): ?>
                                        <div style="color: #4B2F1F; font-size: 14px; margin-bottom: 4px;">
                                            <strong>Validée par :</strong> <?php echo htmlspecialchars($notification['approved_by_name'], ENT_QUOTES, 'UTF-8'); ?>
                                        </div>
                                    <?php endif; ?>
                                    <?php if (!empty($notification['comments'])): ?>
                                        <div style="color: <?php echo $text_color; ?>; font-size: 13px; margin-top: 8px; padding: 8px; background-color: rgba(255,255,255,0.6); border-radius: 4px;">
                                            <strong>Commentaires :</strong> <?php echo htmlspecialchars($notification['comments'], ENT_QUOTES, 'UTF-8'); ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div style="text-align: right; color: #666; font-size: 12px;">
                                    <?php echo $time_ago; ?>
                                </div>
                            </div>
                            <div style="margin-top: 10px;">
                                <a href="/masunzu_bar_hotel/modules/expenses/expenses_history.php?requisition_id=<?php echo $notification['id']; ?>" 
                                   style="display: inline-block; background-color: <?php echo $border_color; ?>; color: #FFFFFF; padding: 6px 12px; text-decoration: none; border-radius: 4px; font-size: 13px; font-weight: 500;">
                                    Voir les détails
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div style="margin-top: 15px; text-align: center;">
                    <a href="/masunzu_bar_hotel/modules/expenses/expenses_history.php" 
                       style="color: #4B2F1F; text-decoration: underline; font-size: 14px; font-weight: 500;">
                        Voir toutes les réquisitions
                    </a>
                </div>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($shipmentNotifications)): ?>
            <div style="background-color:#FFF; border-radius:12px; padding:20px; box-shadow:0 4px 12px rgba(0,0,0,0.1); margin-bottom:20px;">
                <h3 style="margin-top:0; color:#4B2F1F;">Notifications</h3>
                <form method="post" style="margin-bottom:10px;">
                    <button name="hide_notifications" type="submit" style="background:#4B2F1F;color:#fff;border:none;padding:6px 12px;border-radius:6px;cursor:pointer;">Masquer</button>
                </form>
                <div style="display:flex; flex-direction:column; gap:12px;">
                    <?php foreach ($shipmentNotifications as $notif): ?>
                        <div style="border-left:4px solid #F4A261; padding:10px 14px; background-color:#FFF7EE;">
                            <div style="color:#4B2F1F; font-weight:600;"><?php echo htmlspecialchars($notif['message'], ENT_QUOTES, 'UTF-8'); ?></div>
                            <div style="font-size:12px; color:#6B5B52;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($notif['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></div>
                            <?php if (!empty($notif['link'])): ?>
                                <a href="<?php echo htmlspecialchars($notif['link'], ENT_QUOTES, 'UTF-8'); ?>" style="font-size:13px; color:#4B2F1F; font-weight:600;">Ouvrir</a>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <div style="background-color:#FFFFFF; border-radius:12px; padding:20px; box-shadow:0 4px 12px rgba(0,0,0,0.1); margin-bottom:20px;">
            <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:15px;">
                <h3 style="margin:0; color:#4B2F1F;">Commandes en attente d'approbation</h3>
                <a href="/masunzu_bar_hotel/modules/admin/approve_invoices.php" style="background:#4B2F1F; color:#FFF; padding:8px 14px; border-radius:6px; text-decoration:none; font-weight:600;">Voir toutes</a>
            </div>
            <?php if (empty($pending_invoice_approvals)): ?>
                <p style="margin:0; color:#6D4C41;">Aucune commande en attente pour votre dépôt.</p>
            <?php else: ?>
                <div style="overflow-x:auto;">
                    <table style="width:100%; border-collapse:collapse; font-size:14px;">
                        <thead>
                            <tr style="background:#F4F0E4; color:#4B2F1F;">
                                <th style="padding:10px; text-align:left;">Commande</th>
                                <th style="padding:10px; text-align:left;">Client</th>
                                <th style="padding:10px; text-align:right;">Montant</th>
                                <th style="padding:10px; text-align:left;">Initiateur</th>
                                <th style="padding:10px; text-align:left;">Créée le</th>
                                <th style="padding:10px; text-align:center;">Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pending_invoice_approvals as $invoice): ?>
                                <tr class="pending-invoice-row" style="border-bottom:1px solid #E0E0E0;">
                                    <td style="padding:10px;">
                                        <span class="blink-text"><?php echo htmlspecialchars($invoice['invoice_number'], ENT_QUOTES, 'UTF-8'); ?></span>
                                    </td>
                                    <td style="padding:10px;"><?php echo htmlspecialchars($invoice['customer_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding:10px; text-align:right;"><?php echo number_format($invoice['paid_amount'] ?? 0, 0, ',', '.'); ?> BIF</td>
                                    <td style="padding:10px;"><?php echo htmlspecialchars($invoice['initiator_name'] ?? '—', ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding:10px;"><?php echo htmlspecialchars(date('d/m/Y H:i', strtotime($invoice['created_at'])), ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding:10px; text-align:center;">
                                        <a href="/masunzu_bar_hotel/modules/admin/approve_invoices.php?invoice_id=<?php echo (int)$invoice['id']; ?>" style="background:#4B2F1F; color:#FFF; padding:6px 10px; border-radius:6px; text-decoration:none; font-size:12px;">Examiner</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- SECTION 2: STOCK SHIPMENT NOTIFICATIONS -->
        <?php if ($pending_count > 0): ?>
            <div style="background-color: #F4A261; color: #000000; padding: 15px; border-radius: 8px; margin-top: 10px; margin-bottom: 20px;">
                <p style="margin: 0; font-size: 16px;">
                    Vous avez <?php echo $pending_count; ?> transfert(s) de stock en attente de réception.
                    <?php if ($unassigned_count > 0): ?>
                        (<?php echo $unassigned_count; ?> transfert(s) ne sont attribués à aucun gestionnaire.)
                    <?php endif; ?>
                    <a href="/masunzu_bar_hotel/modules/operation/receive_stock.php" style="color: #4B2F1F; text-decoration: underline;">Voir maintenant</a>
                </p>
            </div>
        <?php else: ?>
            <p style="color: #4B2F1F; font-size: 16px; margin-top: 10px; text-align: center;">Aucun transfert de stock en attente.</p>
        <?php endif; ?>
        
        <!-- SECTION 3: PROVINCE STOCK DETAILS -->
        <h3>Détails du stock provincial</h3>
        <?php if (empty($stocks)): ?>
            <p style="color: #4B2F1F; font-size: 16px; text-align: center;">Aucun stock disponible dans votre province.</p>
        <?php else: ?>
            <div style="margin-bottom: 20px; display: flex; justify-content: center;">
                <input type="text" id="province_stock_search" placeholder="Rechercher un produit..." style="width: 100%; max-width: 400px; padding: 12px; border-radius: 8px; border: 2px solid #4B2F1F; background-color: #FFFFFF; font-size: 14px; color: #4B2F1F; margin-bottom: 15px;">
            </div>
            <div style="overflow-x: auto;">
                <table id="province_stock_table" style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 15px; text-align: left; width: 28%;">Produit</th>
                            <th style="padding: 15px; text-align: center; width: 18%;">Disponible (caisses)</th>
                            <th style="padding: 15px; text-align: center; width: 18%;">Réservé (caisses)</th>
                            <th style="padding: 15px; text-align: right; width: 18%;">Prix / caisse</th>
                            <th style="padding: 15px; text-align: right; width: 18%;">Valeur disponible</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($stocks as $stock): ?>
                            <?php
                                $available_quantity = max(0, (int)($stock['available_quantity'] ?? 0));
                                $reserved_quantity = max(0, (int)($stock['reserved_quantity'] ?? 0));
                                $price_per_crate = (float)($stock['price_per_crate'] ?? $stock['unit_price'] ?? 0);
                                $available_value_item = (float)($stock['available_value'] ?? ($available_quantity * $price_per_crate));
                            ?>
                            <tr class="stock-row" data-product-name="<?php echo htmlspecialchars(strtolower($stock['product_name']), ENT_QUOTES, 'UTF-8'); ?>" data-total-value="<?php echo $available_value_item; ?>">
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($stock['product_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: center; font-weight: 600;"><?php echo number_format($available_quantity, 0, ',', '.'); ?></td>
                                <td style="padding: 15px; text-align: center; color: #6B5B52;"><?php echo number_format($reserved_quantity, 0, ',', '.'); ?></td>
                                <td style="padding: 15px; text-align: right;"><?php echo number_format($price_per_crate, 0, ',', '.'); ?> BIF</td>
                                <td style="padding: 15px; text-align: right; font-weight: 600;"><?php echo number_format($available_value_item, 0, ',', '.'); ?> BIF</td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr style="background-color: #F4A261; color: #4B2F1F;">
                            <td style="padding: 15px; text-align: left; font-weight: 600;" colspan="4">Valeur totale du stock disponible :</td>
                            <td id="total_stock_value" style="padding: 15px; text-align: right; font-weight: 700; font-size: 16px;"><?php echo number_format($total_value, 0, ',', '.'); ?> BIF</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        <?php endif; ?>
        
        <!-- SECTION 4: LOW STOCK ALERTS -->
        <?php if (!empty($low_stock)): ?>
        <h3>Alertes de stock faible</h3>
            <div style="background-color: #FFF3E0; border-left: 4px solid #FF9800; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                <ul style="margin: 0; padding-left: 20px;">
                    <?php foreach ($low_stock as $item): ?>
                        <li style="color: #E65100; font-size: 14px; margin-bottom: 8px;">
                            <strong><?php echo htmlspecialchars($item['product_name'], ENT_QUOTES, 'UTF-8'); ?></strong> - 
                            Il ne reste que <?php echo number_format((int)($item['available_quantity'] ?? 0), 0, ',', '.'); ?> caisses disponibles
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
        
        <!-- SECTION 7: RECENT PAID INVOICES -->
        <h3>Recent Paid Invoices</h3>
        <?php if (empty($paid_invoices)): ?>
            <p style="color: #4B2F1F; font-size: 16px; text-align: center;">Aucune facture payée pour le moment.</p>
        <?php else: ?>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 15px; text-align: left; width: 12%;">Facture</th>
                            <th style="padding: 15px; text-align: left; width: 25%;">Caissier</th>
                            <th style="padding: 15px; text-align: center; width: 18%;">Total bouteilles</th>
                            <th style="padding: 15px; text-align: right; width: 20%;">Montant</th>
                            <th style="padding: 15px; text-align: left; width: 25%;">Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($paid_invoices as $invoice): ?>
                            <tr>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($invoice['cashier_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: center;"><?php echo number_format($invoice['total_quantity'], 0, ',', '.'); ?></td>
                                <td style="padding: 15px; text-align: right;"><?php echo number_format($invoice['paid_amount'], 0, ',', '.'); ?> BIF</td>
                                <td style="padding: 15px; text-align: left;"><?php echo date('Y-m-d H:i', strtotime($invoice['created_at'])); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
        
        <div style="margin-top: 25px; display: flex; gap: 15px; justify-content: center; flex-wrap: wrap;">
            <a href="/masunzu_bar_hotel/modules/operation/receive_stock.php" style="display: inline-block; background-color: #4B2F1F; color: #F4F0E4; padding: 12px 24px; text-decoration: none; font-size: 16px; font-weight: 500; border-radius: 8px; transition: all 0.3s ease;">Réceptionner le stock</a>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Search functionality for Dépôt Stock Details table
        const provinceStockSearch = document.getElementById('province_stock_search');
        const provinceStockTable = document.getElementById('province_stock_table');
        const totalStockValueCell = document.getElementById('total_stock_value');
        
        if (provinceStockSearch && provinceStockTable) {
            // Store original total value
            const originalTotalValue = totalStockValueCell ? totalStockValueCell.textContent.trim() : '';
            
            provinceStockSearch.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase().trim();
                const rows = provinceStockTable.querySelectorAll('tbody .stock-row');
                let visibleRows = 0;
                let filteredTotalValue = 0;
                
                rows.forEach(row => {
                    const productName = row.getAttribute('data-product-name') || '';
                    const shouldShow = productName.includes(searchTerm);
                    row.style.display = shouldShow ? '' : 'none';
                    
                    if (shouldShow) {
                        visibleRows++;
                        // Get total value from data attribute
                        const rowValue = parseFloat(row.getAttribute('data-total-value')) || 0;
                        filteredTotalValue += rowValue;
                    }
                });
                
                // Update total value display
                if (totalStockValueCell) {
                    if (searchTerm === '') {
                        totalStockValueCell.textContent = originalTotalValue;
                    } else {
                        // Format the filtered total value (using same format as PHP: dots as thousands separator)
                        const formattedValue = Math.round(filteredTotalValue).toLocaleString('en-US').replace(/,/g, '.');
                        totalStockValueCell.textContent = formattedValue + ' BIF';
                    }
                }
                
                // Show message if no results
                const tbody = provinceStockTable.querySelector('tbody');
                let noResultsMsg = tbody.querySelector('.no-results-msg');
                if (visibleRows === 0 && searchTerm !== '') {
                    if (!noResultsMsg) {
                        noResultsMsg = document.createElement('tr');
                        noResultsMsg.className = 'no-results-msg';
                        noResultsMsg.innerHTML = '<td colspan="5" style="padding: 20px; text-align: center; color: #666;">Aucun produit ne correspond à "' + searchTerm + '"</td>';
                        tbody.appendChild(noResultsMsg);
                    }
                } else if (noResultsMsg) {
                    noResultsMsg.remove();
                }
            });
        }
    });
</script>

<style>
    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
    }
    
    html {
        margin: 0;
        padding: 0;
        width: 100%;
        overflow-x: hidden;
    }
    
    body {
        margin: 0 !important;
        padding: 0 !important;
        width: 100% !important;
        overflow-x: hidden !important;
        background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%) !important;
    }
    
    .container {
        width: 100% !important;
        max-width: none !important;
        margin: 0 !important;
        padding: 0 !important;
    }
    
    .content {
        width: 100% !important;
        max-width: none !important;
        box-sizing: border-box !important;
    }
    
    table { 
        min-width: 600px;
        border-collapse: collapse;
    }
    
    table tbody tr {
        border: none;
        background-color: #FFFFFF;
    }
    
    table tbody tr:hover {
        background-color: #F9F7F4;
    }
    
    table tbody tr td {
        border: none;
    }
    
    button:hover {
        background-color: #F4A261;
        color: #4B2F1F;
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.3);
    }
    
    a:hover {
        color: #FF8C42;
    }
    
    .content a {
        transition: all 0.3s ease;
    }
    
    .content a[href*="split_stock"]:hover, 
    .content a[href*="receive_stock"]:hover,
    .content a[style*="background-color"]:hover {
        background-color: #F4A261 !important;
        color: #4B2F1F !important;
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.3);
    }
    
    input:focus, select:focus {
        outline: none;
        border-color: #F4A261;
        box-shadow: 0 0 5px rgba(244, 162, 97, 0.5);
    }
    
    .content h3 {
        color: #4B2F1F;
        font-weight: 600;
        margin-top: 25px;
        margin-bottom: 15px;
        border-bottom: 2px solid #F4A261;
        padding-bottom: 8px;
    }
    
    #province_stock_search {
        transition: all 0.3s ease;
    }
    
    #province_stock_search:focus {
        outline: none;
        border-color: #F4A261;
        box-shadow: 0 0 8px rgba(244, 162, 97, 0.5);
    }
    
    @media (max-width: 768px) {
        .container { padding: 0; width: 100%; margin: 0; }
        .content { width: 100%; max-width: none; padding: 15px; }
        h2 { font-size: 24px; }
        h3 { font-size: 18px; }
        p, a { font-size: 14px; }
        table { font-size: 14px; }
        #province_stock_search { max-width: 100%; }
    }
    
    @media (max-width: 480px) {
        h2 { font-size: 20px; }
        h3 { font-size: 16px; }
        p, a { font-size: 12px; }
        table { font-size: 12px; }
        .content { padding: 15px; }
        #province_stock_search { font-size: 12px; padding: 10px; }
    }
</style>

<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
<?php
if ($conn && !$conn->connect_error) {
    $conn->close();
}
?>
